-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Streams;
with Ada.Real_Time;
with Buffer_Stream;  -- Parent Class

-------------------------------------------------------------------------------
-- This package provides a child class of buffer_stream to allow access to
-- simulation history data in a task-safe and structured manner.
--
-- A buffer stream is created with a certian max size of data it may hold. When
-- a write is attempted and there isn't enough room, old *series* of writes
-- (delimited by Open_Writes and Close_Writes pairs) will be deleted to make
-- room if possible. If this occurs, the deletion will start with the oldest
-- series of writes and move forward.
-------------------------------------------------------------------------------
package Buffer_Stream.Simulation is

   type Instance is new Buffer_Stream.Instance with private;

   type Handle is access all Instance'Class;

   type Read_Location is private;

   -- The amount of stream space taken up as overhead for each Open_Write/Close_Writes pair.
   Series_Overhead : constant Natural;

   --
   -- Overrides of inherited routines
   --

   -------------------------------------------------------------------------------
   -- Prepare for a series of reads from the stream. This routine should be called
   -- before a stream is read from.
   -- This implementation saves off the starting read location, for use in the
   -- close operation (see close_reads).
   -------------------------------------------------------------------------------
   procedure Open_Reads (Stream  : in out Instance);

   -------------------------------------------------------------------------------
   -- Complete a series of reads from the stream. This routine will be called
   -- after a series of reads has completed.
   -- This implementation rewinds the stream head to where it was before the reads
   -- started. Thus reads from a simulation stream will not permanently delete
   -- the read items.
   -------------------------------------------------------------------------------
   procedure Close_Reads (Stream : in out Instance);


   -------------------------------------------------------------------------------
   -- Complete a series of writes to the stream. This routine will be called
   -- after a series of writes has completed.
   -- This implementation places a perforation marker, and unlocks the stream.
   -------------------------------------------------------------------------------
   procedure Close_Writes (Stream : in out Instance);

   --
   -- Unique routines
   --

   -------------------------------------------------------------------------------
   -- Skip over all entries in the input stream older than the given time. The
   -- time stamp of the first entry in the stream will be returned.
   --
   -- For this routine to operate correctly, the data in the stream has to conform
   -- to a certain expected format. In particular:
   --   o  An identifing string must have been written to the front of each entry
   --      with String'Output.
   --   o  The rest of the entry must be filled with data that was originally
   --      written into a Buffer_Stream.Save stream.
   -------------------------------------------------------------------------------
   procedure Skip (Stream : in out Instance;
                   Past   : in     Ada.Real_Time.Time;
                   Time   :    out Ada.Real_Time.Time
                  );

   -------------------------------------------------------------------------------
   -- Clean out the given stream. The stream should have been opened for writes.
   -------------------------------------------------------------------------------
   procedure Flush (Stream : in out Instance);

   -------------------------------------------------------------------------------
   -- Return a marker to the current Read_Location. The value returned can be
   -- passed into Rewind, at least until the stream is closed. After that it is
   -- invalid.
   -------------------------------------------------------------------------------
   function Location (Stream : in Instance) return Read_Location;

   -------------------------------------------------------------------------------
   -- Rewind the read head of the stream back to the given Read_Location. The
   -- value given should have been retrieved sometime since the last Open_Reads.
   -------------------------------------------------------------------------------
   procedure Rewind (Stream   : in out Instance;
                     Location : in     Read_Location);

private
   type Instance is new Buffer_Stream.Instance with record
      Start : Buffer_Stream.Stream_Buffer.Read_Location;
   end record;

   type Read_Location is new Buffer_Stream.Stream_Buffer.Read_Location;

   Series_Overhead : constant Natural :=
     Natural(Stream_Buffer.Data_Block_Overhead) * Bytes_Per_Element;

end Buffer_Stream.Simulation;
