-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Real_Time;

with Saved_Data_Header;
with Log;

-------------------------------------------------------------------------------
-- This package provides a child class of buffer_stream to allow access to
-- simulation history data in a task-safe and structured manner.
--
-- A buffer stream is created with a certian max size of data it may hold. When
-- a write is attempted and there isn't enough room, old *series* of writes
-- (delimited by Open_Writes and Close_Writes pairs) will be deleted to make
-- room if possible. If this occurs, the deletion will start with the oldest
-- series of writes and move forward.
-------------------------------------------------------------------------------
package body Buffer_Stream.Simulation is


   -------------------------------------------------------------------------------
   -- Prepare for a series of reads from the stream. This routine should be called
   -- before a stream is read from.
   -- This implementation saves off the starting read location, for use in the
   -- close operation (see close_reads).
   -------------------------------------------------------------------------------
   procedure Open_Reads (Stream  : in out Instance) is
   begin

      Stream.Start := Stream_Buffer.Location (Stream.Buffer.all);

   end Open_Reads;

   -------------------------------------------------------------------------------
   -- Complete a series of reads from the stream. This routine will be called
   -- after a series of reads has completed.
   -- This implementation rewinds the stream head to where it was before the reads
   -- started. Thus reads from a simulation stream will not permanently delete
   -- the read items.
   -------------------------------------------------------------------------------
   procedure Close_Reads (Stream : in out Instance) is
   begin

      Stream_Buffer.Rewind( Buffer   => Stream.Buffer.all,
                            Location => Stream.Start
                            );
   end Close_Reads;

   -------------------------------------------------------------------------------
   -- Complete a series of writes to the stream. This routine will be called
   -- after a series of writes has completed.
   -- This implementation places a perforation marker, and unlocks the stream.
   -------------------------------------------------------------------------------
   procedure Close_Writes (Stream : in out Instance) is
   begin

      -- Write a perforation at the end
      Stream_Buffer.Perforate (Stream.Buffer.all);

      -- Unlock the stream
      Stream.Use_Lock.Release;

   end Close_Writes;

   -------------------------------------------------------------------------------
   -- Skip over all entires in the input stream older than the given time. The
   -- time stamp of the first entry in the stream will be returned.
   --
   -- For this routine to operate correctly, the stream must be filled will data
   -- placed into a Buffer_Stream.Save stream. The stream should also be opened
   -- for reads.
   -------------------------------------------------------------------------------
   procedure Skip (Stream : in out Instance;
                   Past   : in     Ada.Real_Time.Time;
                   Time   :    out Ada.Real_Time.Time
                   ) is

     use type Ada.Real_Time.Time;
     use type Ada.Real_Time.Time_Span;

      Entry_Start  : Stream_Buffer.Read_Location;
      Header       : Ada.Streams.Stream_Element_Array(1..Saved_Data_Header.Elements);
   begin

      -- Make sure there is data to examine
      if Stream_Buffer.Size(Stream.Buffer.all) = 0 then
         Log.Report ("The buffer to skip through is empty", Log.Warning);
         return;
      end if;

      Entry_Start := Stream_Buffer.Location(Stream.Buffer.all);

      -- Read in the source (which we don't care about)
      declare
         Trash : constant String := String'Input (Stream'access);
      begin
         null;
      end;

      -- Read in the header.
      Stream_Buffer.Read
        (Buffer   => Stream.Buffer.all,
         New_Data => Header
         );

      -- Report the amount of time stored in the buffer.
      Log.Report ("The oldest entry in the playback buffer is from " &
                  Duration'Image
                  (Ada.Real_Time.To_Duration
                   (Ada.Real_Time.Clock - Saved_Data_Header.Time(Header))
                   / 60.0) & " minutes ago.");

      -- Quit looping when we find an entry later than the given delta
      while Saved_Data_Header.Time(Header) < Past loop

         -- Skip over the rest of the data in the entry
         declare
            Trash : Ada.Streams.Stream_Element_Array(1..
                    Ada.Streams.Stream_Element_Offset(Saved_Data_Header.Size(Header)));
         begin
            Stream_Buffer.Read
              (Buffer   => Stream.Buffer.all,
               New_Data => Trash
               );
         end;

         -- If we have read all the data, quit.
         if Stream_Buffer.Size(Stream.Buffer.all) = 0 then
            return;
         end if;

         Entry_Start := Stream_Buffer.Location(Stream.Buffer.all);

         -- Read in the source (which we don't care about)
         declare
            Trash : constant String := String'Input (Stream'access);
         begin
            null;
         end;

         -- Read in the header.
         Stream_Buffer.Read
           (Buffer   => Stream.Buffer.all,
            New_Data => Header
           );

      end loop;

      Time := Saved_Data_Header.Time(Header);

      Stream_Buffer.Rewind
        (Buffer   => Stream.Buffer.all,
         Location => Entry_Start
        );

   end Skip;

   -------------------------------------------------------------------------------
   -- Clean out the given stream. The stream should have been opened for writes.
   -------------------------------------------------------------------------------
   procedure Flush (Stream : in out Instance) is
   begin
      Stream_Buffer.Flush (Stream.Buffer.all);
   end Flush;

   -------------------------------------------------------------------------------
   -- Return a marker to the current Read_Location. The value returned can be
   -- passed into Rewind, at least until the stream is closed. After that it is
   -- invalid.
   -------------------------------------------------------------------------------
   function Location (Stream : in Instance) return Read_Location is
   begin
      return Read_Location
        (Buffer_Stream.Stream_Buffer.Read_Location'
         (Buffer_Stream.Stream_Buffer.Location (Stream.Buffer.all))
         );
   end Location;

   -------------------------------------------------------------------------------
   -- Rewind the read head of the stream back to the given Read_Location. The
   -- value given should have been retrieved sometime since the last Open_Reads.
   -------------------------------------------------------------------------------
   procedure Rewind (Stream   : in out Instance;
                     Location : in     Read_Location) is
   begin
      Buffer_Stream.Stream_Buffer.Rewind
        (Buffer   => Stream.Buffer.all,
         Location => Buffer_Stream.Stream_Buffer.Read_Location(Location)
         );
   end Rewind;
end Buffer_Stream.Simulation;

