-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Saved_Data_Header;

-------------------------------------------------------------------------------
-- This package provides a child class of buffer_stream to allow saving of
-- data in a task-safe and structured manner.
--
-- A buffer stream is created with a certian max size of data it may hold. When
-- a write is attempted and there isn't enough room, old *series* of writes
-- (delimited by Open_Writes and Close_Writes pairs) will be deleted to make
-- room if possible. If this occurs, the deletion will start with the oldest
-- series of writes and move forward.
-------------------------------------------------------------------------------
package body Buffer_Stream.Save is

   -------------------------------------------------------------------------------
   -- Complete a series of writes to the stream. This routine will be called
   -- after a series of writes has completed.
   -- This implementation fills in the size header, places a perforation marker,
   -- and unlocks the stream.
   -------------------------------------------------------------------------------
   procedure Close_Writes (Stream : in out Instance) is
   begin
      -- Write a data size value into the stream (its just a placeholder for later)
      Stream_Buffer.Rewrite
        (Buffer   => Stream.Buffer.all,
         New_data => Saved_Data_Header.Create
         (Size => Stream.Amount_Written,
          Time => Ada.Real_Time.Clock),
         Location => Stream.Header_Location
         );

      -- Write a perforation at the end
      Stream_Buffer.Perforate (Stream.Buffer.all);

      -- Unlock the stream
      Stream.Use_Lock.Release;
   end Close_Writes;

   -------------------------------------------------------------------------------
   -- Return True if data is available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Boolean is
   begin
      return Stream_Buffer.Size (Stream.Buffer.all) > 0;
   end Data_available;

   -------------------------------------------------------------------------------
   -- Return the amount of data available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Natural is
   begin
      return Stream_Buffer.Size (Stream.Buffer.all) * Bytes_Per_Element;
   end Data_available;

   -------------------------------------------------------------------------------
   -- Prepare for a series of writes to the stream. This routine should be called
   -- before a stream is written to.
   -- This implementation puts a size header in the stream and locks it.
   -------------------------------------------------------------------------------
   procedure Open_Writes (Stream  : in out Instance;
                          Success :    out Boolean) is
   begin

      -- See if we can lock the stream
      select
         Stream.Use_Lock.Acquire;

         -- Write a data size value into the stream (its just a placeholder for later)
         Stream_Buffer.Write
           (Buffer   => Stream.Buffer.all,
            New_data => Saved_Data_Header.Create
            (Size => 0,
             Time => Ada.Real_Time.Time_First),
            Location => Stream.Header_Location
            );

         Stream.Amount_Written := 0;

         Success := True;
      else
         Success := False;
      end select;

   end Open_Writes;

   -------------------------------------------------------------------------------
   -- Reimplementation of the inherited Write routine from Root_Stream_Type.
   -- This routine must keep track of how many elements are being written, so that
   -- the header can be filled in at close time.
   -------------------------------------------------------------------------------
   procedure Write
     (Stream : in out Instance;
      Item   : in     Ada.Streams.Stream_Element_Array
     ) is
   begin

      -- Redispatch to the default write routine
      Write
        (Stream => Buffer_Stream.Instance(Stream),
         Item   => Item
         );

      -- Add the amount we just wrote to the running total
      Stream.Amount_Written := Stream.Amount_Written + Item'Length;
   end Write;

end Buffer_Stream.Save;

