-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Streams;
with Ada.Real_Time;
with Buffer_Stream;  -- Parent Class

-------------------------------------------------------------------------------
-- This package provides a child class of buffer_stream to allow saving of
-- data in a task-safe and structured manner.
--
-- A buffer stream is created with a certian max size of data it may hold. When
-- a write is attempted and there isn't enough room, old *series* of writes
-- (delimited by Open_Writes and Close_Writes pairs) will be deleted to make
-- room if possible. If this occurs, the deletion will start with the oldest
-- series of writes and move forward.
-------------------------------------------------------------------------------
package Buffer_Stream.Restore is

   type Instance is new Buffer_Stream.Instance with private;

   type Handle is access all Instance'Class;


   --
   -- Overrides of inherited routines
   --

   -------------------------------------------------------------------------------
   -- Return True if data is available in the stream.
   -------------------------------------------------------------------------------
   function Data_Available (Stream : access Instance) return Boolean;

   -------------------------------------------------------------------------------
   -- Return the amount of data available in the stream.
   -------------------------------------------------------------------------------
   function Data_Available (Stream : access Instance) return Natural;

   -------------------------------------------------------------------------------
   -- Return the amount of storage space (in stream elements) available in the
   -- stream.
   -------------------------------------------------------------------------------
   function Space_Available (Stream : access Instance) return Natural;

   -------------------------------------------------------------------------------
   -- Return only when the requested amount of space is avilable in the stream
   -- for writing.
   -------------------------------------------------------------------------------
   procedure Wait_For_Space (Stream : access Instance;
                             Amount : in     Natural
                            );

   -------------------------------------------------------------------------------
   -- Prepare for a series of reads from the stream. This routine should be called
   -- before a stream is read from.
   -- This implementation attempts to lock the stream. If it can't get the lock
   -- immdediately, it returns with success of FALSE.
   -- Next it reads the header from the stream. If there isn't as much data in
   -- in the stream as the header indicates, the header is put back on the stream,
   -- the stream is unlocked, and success is set to FALSE.
   -------------------------------------------------------------------------------
   procedure Open_Reads (Stream  : in out Instance;
                         Success :    out Boolean);

   -------------------------------------------------------------------------------
   -- Implementation of the inherited Write routine from Root_Stream_Type.
   -- This implementation will wait for space to free up in the stream, rather
   -- than raising an exception when there is no space available.
   -------------------------------------------------------------------------------
   procedure Write
     (Stream : in out Instance;
      Item   : in     Ada.Streams.Stream_Element_Array
     );

   --
   -- Routines unique to this class
   --

   -------------------------------------------------------------------------------
   -- Set the systemwide playback offset to the given time span. Opens will fail
   -- if the next item is not due yet, based on the offset from the current time.
   -------------------------------------------------------------------------------
   procedure Set_Playback_Offset (Offset : in Ada.Real_Time.Time_Span);

   -------------------------------------------------------------------------------
   -- Clean out the given stream. The stream should have been opened for writes.
   -------------------------------------------------------------------------------
   procedure Flush (Stream : in out Instance);

private

   type Instance is new Buffer_Stream.Instance with record
      Header_Location : Buffer_Stream.Stream_Buffer.Read_Location;
   end record;



end Buffer_Stream.Restore;















