-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Streams;
with Buffer_Stream;  -- Parent Class
with Saved_Data_Header;

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Log;

-------------------------------------------------------------------------------
-- This package provides a child class of buffer_stream to allow restoring of
-- data in a task-safe and structured manner.
-------------------------------------------------------------------------------
package body Buffer_Stream.Restore is

   -- An offset to apply to the current RT clock to arrive at the current
   -- "replay time".
   Time_Offset : Ada.Real_Time.Time_Span := Ada.Real_Time.Time_Span_Zero;


   -------------------------------------------------------------------------------
   -- Return True if data is available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Boolean is

   begin

      return Stream_Buffer.Size (Stream.Buffer.all) > 0;

   exception
      when Error : others =>
         Log.Report (Event => "Exception in Buffer_Stream.Restore.Retrieve_Handle: " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);
         return False;

   end Data_available;


   -------------------------------------------------------------------------------
   -- Return the amount of data available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Natural is

   begin

      return Stream_Buffer.Size (Stream.Buffer.all) * Bytes_Per_Element;

   end Data_available;

   -------------------------------------------------------------------------------
   -- Return the amount of storage space (in stream elements) available in the
   -- stream.
   -------------------------------------------------------------------------------
   function Space_Available (Stream : access Instance) return Natural is
   begin
      return Stream_Buffer.Available (Stream.Buffer.all);
   end Space_Available;

   -------------------------------------------------------------------------------
   -- Return only when the requested amount of space is avilable in the stream
   -- for writing.
   -------------------------------------------------------------------------------
   procedure Wait_For_Space (Stream : access Instance;
                             Amount : in     Natural
                            ) is
   begin
      Stream_Buffer.Wait_For_Space
        (Buffer => Stream.Buffer.all,
         Amount => Amount
         );
   end Wait_For_Space;

   -------------------------------------------------------------------------------
   -- Prepare for a series of reads from the stream. This routine should be called
   -- before a stream is read from.
   -- This implementation attempts to lock the stream. If it can't get the lock
   -- immdediately, it returns with success of FALSE.
   -- Next it reads the header from the stream. If there isn't as much data in
   -- in the stream as the header indicates, the header is put back on the stream,
   -- the stream is unlocked, and success is set to FALSE.
   -------------------------------------------------------------------------------
   procedure Open_Reads (Stream  : in out Instance;
                         Success :    out Boolean) is

      -- Allow primitive ops on the following types;
      use type Ada.Real_Time.Time;
      use type Ada.Real_Time.Time_Span;

      Header_Array_Read : Ada.Streams.Stream_Element_Array (1..Saved_Data_Header.Elements);
      -- Array containing the header data.

      Size_Of_Callers_Stream             : Natural;
      -- Size of the callers stream to be written.

      Size_Of_Stream_Indicated_By_Header : Natural;
      -- Size of the input stream indicated by the header.

   begin

      select

         Stream.Use_Lock.Acquire;

         -- Read the header.
         Stream_Buffer.Read
           (Buffer   => Stream.Buffer.all,
            New_data => Header_Array_Read,
            Location => Stream.Header_Location );

         -- Convert Header size to Data_Header
         Size_Of_Stream_Indicated_By_Header :=
           Saved_Data_Header.Size (Header_Array_Read);

         -- Compute the size of callers steam.
         Size_Of_Callers_Stream := Stream_Buffer.Size (Stream.Buffer.all) * Bytes_Per_Element;

         -- Compare size of callers stream with that of written stream.
         -- Compare timestamp with current replay delta.
         if ( Size_Of_Callers_Stream < Size_Of_Stream_Indicated_By_Header ) or
            Saved_Data_Header.Time (Header_Array_Read) > Ada.Real_Time.Clock + Time_Offset
         then
            Stream_Buffer.Rewind( Buffer   => Stream.Buffer.all,
                                  Location => Stream.Header_Location );

            Stream.Use_Lock.Release;

            Success := False;

         else

            -- Callers stream is big enough, and the time is right.
            Success := True;
         end if;

      else

         Success := False;

      end select;

   end Open_Reads;


   -------------------------------------------------------------------------------
   -- Implementation of the inherited Write routine from Root_Stream_Type.
   -- This implementation will wait for space to free up in the stream, rather
   -- than raising an exception when there is no space available.
   -------------------------------------------------------------------------------
   procedure Write
     (Stream : in out Instance;
      Item   : in     Ada.Streams.Stream_Element_Array
     ) is
   begin

      while Stream_Buffer.Available(Stream.Buffer.all) < Item'Length loop
         Stream.Use_Lock.Loan_Wait;
      end loop;

      Stream_Buffer.Write
        (Buffer   => Stream.Buffer.all,
         New_Data => Item
         );

   end Write;

   -------------------------------------------------------------------------------
   -- Set the playback offset to the given time span. Opens will fail if the
   -- next item is not due yet, based on the offset from the current time.
   -------------------------------------------------------------------------------
   procedure Set_Playback_Offset (Offset : in Ada.Real_Time.Time_Span) is
   begin

      Time_Offset := Offset;

   end Set_Playback_Offset;

   -------------------------------------------------------------------------------
   -- Clean out the given stream. The stream should have been opened for writes.
   -------------------------------------------------------------------------------
   procedure Flush (Stream : in out Instance) is
   begin
      Stream_Buffer.Flush (Stream.Buffer.all);
   end Flush;


end Buffer_Stream.Restore;


