-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with JPATS_Radio_DB_IF.Container,
  Jpats_Radio_Db_If_Types,
  Jpats_Radio_Db_If.Ios_If,
  Jpats_Simulated_Aircraft,
  Jpats_Avionics;
with Jpats_Ios_Interface;
with Ada.Text_Io,Log;
with System;

use Jpats_Radio_Db_If_Types;
use Jpats_Radio_Db_If.Ios_If;
use Ada.Text_Io,Log;

package body JPATS_Radio_DB_IF.Controller is

   package Sim_Ac renames Jpats_Simulated_Aircraft;
   package Jrt    renames Jpats_Radio_Db_If_Types;

   Ios : Container.Ios_Interface_Instance renames Container.This_Ios_Interface;

   Position : Jrt.Position_Type := (45.0,-45.0,1000.0);

   N_Rdb_Init_Value : Jrt.Rdb_Init_Type;

   Rcvr_Request     : Jrt.RCVR_REQUEST_Type;
   Ac               : Jrt.Ac_Struct_Type;
   Pointer_Ac       : Jrt.Pac_Struct_Type;
   Iosac            : Jrt.Ac_Struct_Type;
   Pointer_Iosac    : Jrt.Ac_Struct_Type;
   Nav_Freq         : Integer;

   Max_Kill          : constant := 16;
   Kill_Table_Ident  : String (1 .. 4*Max_Kill);
   Kill_Table_Option : Jrt.Mask_Array_Type;
   New_Kill          : Boolean := True;
   Kill_Count        : Integer := 0;

   function N_Rdb_Init return Jrt.Rdb_Init_Type;
   pragma Import(C, N_Rdb_Init, "n_rdb_init");

   procedure N_Stn_Srch(Rcvr : in out Jrt.Rcvr_Request_Type;
                        Stn  : in out Jrt.Stn_Struct_Type);
   pragma Import(C, N_Stn_Srch, "n_stn_srch");

   procedure N_Mkr_Srch(Rcvr : in out Jrt.Rcvr_Request_Type;
                        Stn  : in out Jrt.Mrkr_Stn_Struct);
   pragma Import(C, N_Mkr_Srch, "n_stn_srch");

   procedure N_Magvar(Ac  : in out Jrt.Ac_Struct_Type;
                      Var : in out Float);
   pragma Import(C, N_Magvar, "n_magvar");

   procedure N_Elev(Ac  : in out Jrt.Ac_Struct_Type;
                    Elev : in out Float);
   pragma Import (C, N_Elev, "n_elev");

   procedure N_Stn_Kill(Stn_Kill_Request : in System.Address);
   pragma Import(C, N_Stn_Kill, "n_stn_kill");

   Pstn_Kill_Request : Jrt.Pstn_Kill_Request_Type;

   procedure Initialize is
   begin

      --Initialize magnetic variation:
      Ac.Lat   := Position.Latitude;
      Ac.Lon   := Position.Longitude;
      Ac.Alt   := Position.Altitude;
      Ac.Thdg  := 0.0;
      Ac.Pitch := 0.0;
      Ac.Roll  := 0.0;
      N_Magvar(Ac,Container.This_Subsystem.The_Magnetic_Variation);

      for I in 1 .. Max_Kill loop
         Kill_Table_Ident(I*4-3 .. I*4) := "    ";
         Kill_Table_Option(I) := 0;
         Kill_Count := 0;
      end loop;

      --Register IOS variables
      JPATS_Radio_Db_If.Ios_If.Register_Ios_Variables;

      --Register IOS Array variables
      JPATS_IOS_Interface.Register
        (Name               => "Station_Kill_Stn_Idnt",
         Var_Array          => Container.This_Ios_Interface.Station_Kill_Stn_Idnt'address,
         Set_Routine        => JPATS_Radio_Db_If.Ios_If.Set_Station_Kill_Stn_Idnt'Access,
         Dimension_1_Length => 4*Max_Kill
         );

      JPATS_IOS_Interface.Register
        (Name               => "Station_Kill_Kill_Optn",
         Var_Array          => Container.This_Ios_Interface.Station_Kill_Kill_Optn'address,
         Set_Routine        => JPATS_Radio_Db_If.Ios_If.Set_Station_Kill_Kill_Optn'Access,
         Dimension_1_Length => Max_Kill
         );

      --Initialize radio database:
      N_Rdb_Init_Value := N_Rdb_Init;

   end Initialize;

   procedure Update
     (Integration_Constant : in Float) is
   begin
      --get aircraft position from simulated aircraft
      position.latitude  := Sim_Ac.Get_North;
      position.longitude := Sim_Ac.Get_East;
      position.altitude  := abs(Sim_Ac.Get_Aircraft_Geometric_Altitude);

      --Set station kill before searches:
      if Ios.Station_Kill_Was_Set then

--         Pstn_Kill_Request := Ios.Station_Kill'address;
--         N_Stn_Kill(Ios.Station_Kill);
         N_Stn_Kill(Ios.Station_Kill'address);

         if Ios.Station_Kill.Stn_Idnt = "    " then
            Ios.Station_Kill.Stn_Idnt := "n/a!";
         end if;

      --Set Local Station Kill Table for IOS
--*********************************************************************
--*># STATION KILL SUPPORT FOR AIOS
--*********************************************************************
--*>
--*> The AIOS system initiates the kill function by setting the
--*> ident/option of the facility to be killed, and the mode to
--*> the appropriate value.  The facility ident is stored in the
--*> label Ios.Station_Kill.Stn_Idnt, and the option is stored in
--*> the label Ios.Station_Kill.Kill_Optn.  All idents
--*> are in ascii, left justified, with trailing blanks.  The station
--*> kill tables contain a list of the stations selected for kill.
--*> This table is maintained for AIOS use and is not used by any
--*> other host modules.  The kill tables are Kill_Table_Ident for the
--*> identifier, and Kill_Table_Option for the option.  The kill mode
--*> label can contain values of from 0 to 3, with the following
--*> actions:
--*>
--*>     0 - no action taken
--*>     1 - enter killed ident/option into kill table
--*>     2 - reset killed ident/option from kill table
--*>     3 - clear all ident/option in kill table
--*>
--*> The kill mode enters the facility ident/option into the kill
--*> table.  The table is first checked for an existing entry, and
--*> if so doesn't modify the table.  If the ident/option is not
--*> found currently in the table, the table is searched for a
--*> blank entry and the ident/option is entered.  If the table
--*> is full, no action is taken and the table is not modified.
--*>
--*> The freezes/resets page on the instructor's station contains
--*> a button to reset all killed stations. The button is labeled
--*> 'failed stations'.  When this button is pressed the IOS sets
--*> the label zkillsr to '.true.'.  This module then sets nkmode
--*> to 3 and the systems modules resets zkillsr back to '.false.'.
--*>
--*>
         if Ios.Station_Kill.Kill_Mode = Jrt.Nul then
            null;

         elsif Ios.Station_Kill.Kill_Mode = Jrt.Kill then
            New_Kill := True;
            for I in 1 .. Max_Kill loop
               if Ios.Station_Kill.Stn_Idnt = Kill_Table_Ident(I*4-3 .. I*4) then
                  New_Kill := False;
                  Kill_Table_Option(I) := Ios.Station_Kill.Kill_Optn;
               end if;
            end loop;

            if New_Kill then
               for I in 1 .. Max_Kill loop
                  if Kill_Table_Ident(I*4-3 .. I*4) = "    "
                    and New_Kill then
                     Kill_Table_Ident(I*4-3 .. I*4) := Ios.Station_Kill.Stn_Idnt;
                     Kill_Table_Option(I) := Ios.Station_Kill.Kill_Optn;
                     New_Kill := False;
                     Kill_Count := Kill_Count + 1;
                  end if;
               end loop;
            end if;

--*> The reset mode removes an existing entry from the kill table.
--*> The table is searched for a matching ident/option, and if
--*> found, the table entry is set to ascii blanks.  If the entry
--*> is not found, no action is taken and the table is not modified.
--*>

         elsif Ios.Station_Kill.Kill_Mode = Jrt.Reset
           and Ios.Station_Kill.Stn_Idnt /= "    " then
            for I in 1 .. Max_Kill loop
               if Kill_Table_Ident(I*4-3 .. I*4) = Ios.Station_Kill.Stn_Idnt then
                  Kill_Table_Ident(I*4-3 .. I*4) := "    ";
                  Kill_Table_Option(I) := 0;
                  Kill_Count := Kill_Count - 1;
               end if;
            end loop;

--*> The reset/clear all mode clears all entries from the kill
--*> table.  The table is looped through setting the ident and
--*> option entries to ascii blanks.
--*>

         elsif Ios.Station_Kill.Kill_Mode = Jrt.Reset_All then
            for I in 1 .. Max_Kill loop
               Kill_Table_Ident(I*4-3 .. I*4) := "    ";
               Kill_Table_Option(I) := 0;
               Kill_Count := 0;
            end loop;
         end if;

      -- pack IOS station kill table
         Ios.Station_Kill_Kill_Optn := Kill_Table_Option;
         Ios.Station_Kill_Stn_Idnt  := Kill_Table_Ident;
         Ios.Station_Kill_Count     := Kill_Count;

         Ios.Station_Kill_Was_Set := False;
         Ios.Station_Kill.Kill_Mode := Kill;

      end if;

      --Search for vhf nav station:
      RCVR_REQUEST.Ac_Lat    := Position.Latitude;
      RCVR_REQUEST.Ac_Lon    := Position.longitude;
      RCVR_REQUEST.Freq      := Jpats_Avionics.Nav_Tuning.Frequency;
      RCVR_REQUEST.Rcvr      := Jrt.Vor_Ils;
      RCVR_REQUEST.Bandwidth := 0;
      RCVR_REQUEST.Apt_Idnt  := "     ";
      RCVR_REQUEST.Rwy_Idnt  := "     ";

      N_Stn_Srch(Rcvr_Request,Container.This_Subsystem.The_Nav_Station);

      --Marker Beacon Search:
      RCVR_REQUEST.Rcvr      := Jrt.Mkr;

      N_Mkr_Srch(Rcvr_Request,Container.This_Subsystem.The_Marker_Station);

      --Search for DME station:
--      if Jpats_Avionics.Dme_Tuning.Hold then

      RCVR_REQUEST.Freq      := Jpats_Avionics.Dme_Tuning.Frequency;
      RCVR_REQUEST.Rcvr      := Jrt.Dme;

      N_Stn_Srch(Rcvr_Request,Container.This_Subsystem.The_Dme_Station);

--      else
--         Container.This_Subsystem.The_Dme_Station :=
--           Container.This_Subsystem.The_Nav_Station;
--
--      end if;

      --Search for UHF Comm Station
      RCVR_REQUEST.Freq      := Jpats_Avionics.Uhf_Tuning.Frequency / 10;
      RCVR_REQUEST.Rcvr      := Jrt.Com;

      N_Stn_Srch(Rcvr_Request,Container.This_Subsystem.The_Uhf_Comm_Station);

      --Search for VHF Comm Station:
      RCVR_REQUEST.Freq      := Jpats_Avionics.Vhf_Tuning.Frequency / 10;
      RCVR_REQUEST.Rcvr      := Jrt.Com;

      N_Stn_Srch(Rcvr_Request,Container.This_Subsystem.The_Vhf_Comm_Station);

      --Obtain magnetic variation:
      Ac.Lat   := Position.Latitude;
      Ac.Lon   := Position.Longitude;
      Ac.Alt   := Position.Altitude;
      Ac.Thdg  := 0.0;
      Ac.Pitch := 0.0;
      Ac.Roll  := 0.0;
      N_Magvar(Ac,Container.This_Subsystem.The_Magnetic_Variation);

      --IOS Spider Web magnetic variation:
      if Ios.Magvar_Latitude_Was_Set
        and Ios.Magvar_Longitude_Was_Set then
         IosAc.Lat   := Ios.Magvar_Latitude;
         IosAc.Lon   := Ios.Magvar_Longitude;
         IosAc.Alt   := 0.0;
         IosAc.Thdg  := 0.0;
         IosAc.Pitch := 0.0;
         IosAc.Roll  := 0.0;
         N_Magvar(IosAc,Ios.Magnetic_Variation);
         Ios.Magvar_Latitude_Was_Set  := False;
         Ios.Magvar_Longitude_Was_Set := False;
      end if;

      Ios.Local_Magvar := Container.This_Subsystem.The_Magnetic_Variation;

      --Obtain terrain elevation from radio database:
      N_Elev(Ac,Container.This_Subsystem.The_Terrain_Elevation);

   end Update;

end JPATS_Radio_DB_IF.Controller;



