-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random,
  Radio_Utilities,
  Ada.Text_IO,Log;
use  ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random,
  Ada.Text_IO,Log;

package body VOR is

   package RU renames Radio_Utilities;

   Tuning_Delay     : Float;
   Temp             : Float;
   True_Brg_Deg     : Radio_Types.Bearing_Type;
   Cone_Tangent     : Float;
   Cone_Error       : Float;
   Cone_Timer       : Radio_Types.Timer_Type;
   Line_Of_Sight    : Radio_Types.Range_Type;
   Course_Error     : Float;
   XGAS             : Float;
   YGAS             : Float;
   RGAS             : Float;
   ZGAS             : Float;
   Brg_To_Stn       : Radio_Types.Bearing_Type;
   G                : ada.Numerics.Float_Random.Generator;
   True_Bearing     : Float;
   TF_Direction     : Float;
   DOT_Number       : Float;
   Max_Altitude     : Constant := 20_000.0;
   Altitude_Fade    : Float;
   Max_Range        : Float;
   Scaled_RGAS      : Float;
   Station_Strength : Float;
   Signal_Strength  : Float;

   procedure Set_VOR_Inputs
     (An_Instance        : in out Instance;
      Power              : in     Boolean;
      Mode_On            : in     Boolean;
      Aircraft_Position  : in     Radio_Types.Position_Type;
      Course_Selected    : in     Radio_Types.Bearing_Type;
      Station            : in     Jpats_Radio_Db_If_Types.Stn_Struct_Type;
      Frequency_Changed  : in     Boolean) is
   begin
      An_Instance.Power              := Power;
      An_Instance.Mode_On            := Mode_On;
      An_Instance.Aircraft_Position  := Aircraft_Position;
      An_Instance.Course_Selected    := Course_Selected;
      An_Instance.Station            := Station;
      An_Instance.Frequency_Changed  := Frequency_Changed;
   end;


   function Audio_On
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Audio_On;
   end;

   function To_From_Flag
     (An_Instance : in Instance)
      return Radio_Types.Normalized_Type is
   begin
      return An_Instance.To_From_Flag;
   end;

   function Dev_Dots
     (An_Instance : in Instance)
      return Radio_Types.Dots_Type is
   begin
      return An_Instance.Dev_Dots;
   end;

    function Valid
      (An_Instance : in Instance)
     return Boolean is
   begin
      return An_Instance.Valid;
   end;

   function VCA
     (An_Instance : in Instance)
      return Radio_Types.Normalized_Type is
   begin
      return An_Instance.VCA;
   end;

   function Receiving
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Receiving;
   end;

   function Mag_Brg_Deg
     (An_Instance : in Instance)
      return Radio_Types.Bearing_Type is
   begin
      return An_Instance.Mag_Brg_Deg;
   end;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =

   procedure INIT(An_Instance : in out Instance) is

      Variable_Registered : Boolean;
   begin

      -- Initialize instance components:
      An_Instance.Power              := False;
      An_Instance.Mode_On            := False;
      An_Instance.Aircraft_Position  := (Latitude  => 45.0,
                                         Longitude => -45.0,
                                         Altitude  => 1000.0);
      An_Instance.Course_Selected    := 0.0;
      An_Instance.Frequency_Changed  := False;
      An_Instance.Audio_On           := False;
      An_Instance.To_From_Flag       := 0.0;
      An_Instance.Dev_Dots           := 0.0;
      An_Instance.Valid              := False;
      An_Instance.VCA                := 0.0;
      An_Instance.Receiving          := False;
      An_Instance.Mag_Brg_Deg        := 0.0;
      Tuning_Delay                   := 0.0;
      Temp                           := 0.0;
      True_Brg_Deg                   := 0.0;
      Cone_Tangent                   := 0.0;
      Cone_Error                     := 0.0;
      Cone_Timer                     := 0.0;
      Line_Of_Sight                  := 0.0;
      Course_Error                   := 0.0;
      XGAS                           := 0.0;
      YGAS                           := 0.0;
      RGAS                           := 0.0;
      ZGAS                           := 0.0;
      Brg_To_Stn                     := 0.0;
      True_Bearing                   := 0.0;
      TF_Direction                   := 0.0;
      DOT_Number                     := 0.0;
      Altitude_Fade                  := 0.0;
      Max_Range                      := 0.0;
      Scaled_RGAS                    := 0.0;
      Station_Strength               := 0.0;
      Signal_Strength                := 0.0;

   end Init;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
   procedure Update(Integration_Constant : in     Float;
                    An_Instance          : in out Instance) is

   begin

--+---------------------------------------------------------------------------+
--| NAV Geometry:                                                             |
--|                                                                           |
--|  This section calculates for the VOR:                                     |
--|    Z-Distance to station (ZGAS), in Feet.                                 |
--|    Ground range to station (RGAS), in Nautical Miles.                     |
--|    X-Distance to localizer (AGAS), in Nautical Miles.                     |
--|    Y-Distance to localizer (BGAS), in Nautical Miles.                     |
--|    X-Distance to touchdown (AGAS_TO_TD), in Nautical Miles.               |
--|    True Bearing to station (BRG_TO_STN), in Degrees.                      |
--|                                                                           |
--+---------------------------------------------------------------------------+

--*********************************************************************
-- XGAS TO STATION, IN NAUTICAL MILES ( N/S DISTANCE )
--*********************************************************************
--
-- The xgas term is the north-south distance from the aircraft
-- to the selected ground station, in nautical miles.  The value
-- is protected from a zero result to prevent overflow problems
-- in other modules.  The equation for xgas is:
--
--     xgas=(stn lat - a/c lat) * 60.0  (nm/deg)
--
      XGAS := float(An_Instance.Station.Lat -
                    An_Instance.Aircraft_Position.Latitude ) * 60.0;
      -- Prevent zero divide errors:
      if Xgas = 0.0 then
         Xgas := 0.00001;
      end if;

--*******************************************************************
-- YGAS TO STATION, IN NAUTICAL MILES ( E/W DISTANCE )
--*******************************************************************
--
-- The ygas term is the east-west distance from the aircraft
-- to the selected ground station, in nautical miles.  The
-- distance is corrected for meridian convergence by use of the
-- cosine of latitude term.  This value is zero protected.
-- The equation for ygas is:
--
--   ygas = (stn lon - a/c lon) * 60.0nm/deg * cosine(stn lat)
--
      YGAS := float(RU.Xn180(An_Instance.Station.Lon -
                             An_Instance.Aircraft_Position.Longitude))*60.0 *
                   cos(float(An_Instance.Station.Lat),360.0);

      -- Prevent zero divide errors:
      if YGAS = 0.0 then
         YGAS := 0.00001;
      end if;

--*********************************************************************
-- RANGE TO STATION, IN NAUTICAL MILES ( GROUND RANGE )
--*********************************************************************
--
-- The range to station is the range measured along the ground
-- from the aircraft to the ground station in nautical miles.
-- The pythagorean theorem is used in conjunction with the xgas
-- and ygas terms to produce the ground range rgas.  This
-- is determined by the equation:
--
--     rgas = sqrt( xgas**2 + ygas**2 )
--
      RGAS := sqrt( XGAS ** 2 + YGAS ** 2 );

--*********************************************************************
-- A/C ALTITUDE ABOVE STATION, IN FEET
--*********************************************************************
--
-- The aircraft altitude above the ground station is determined
-- by the difference between the aircraft geometric altitude
-- (feet msl) and the ground station elevation (feet msl).
-- Positive indicates the aircraft is above the station.  The
-- altitude above the station term (nzgas) is determined by
-- the equation:
--
--     zgas= a/c geometric altitude - station elevation
--
      ZGAS := An_Instance.Aircraft_Position.Altitude -
                  An_Instance.Station.Elv ;

      if ZGAS < 1.0 then ZGAS := 1.0; end if;

--*********************************************************************
-- BEARING TO THE STATION, IN +/- 180 DEGREES TRUE
--*********************************************************************
--
-- The bearing to station term (nbrgstn) is computed by taking
-- the inverse tangent of the xgas and ygas terms.  The terms
-- are passed to the routine in such a way to produce an angle
-- that is relative to true north.  In the case of a VOT type
-- of facility, the bearing is set to 180 degrees with the
-- magnetic variation added.  This allows the VOR module to
-- operate in a normal manner without specific VOT code.
-- The bearing to station is computed by the equation:
--
--                          nxgas (nm.)
--     nbrgstn = arctan ( --------------- )
--                          nygas (nm.)
--
      if An_Instance.Station.Typ2(22) then
         Brg_To_Stn := Ru.Xn180(180.0 + An_Instance.Station.Hdg);
      else
         Brg_To_Stn := arctan(YGAS, XGAS, 360.0 );
      end if;

-- =  =  end of NAV Geometry section.  =  =  =

      if An_Instance.Power and then An_Instance.Mode_On then
         An_Instance.Valid        := True;   -- Valid data.
         An_Instance.Audio_On     := True;   -- Audio available.
      else
         An_Instance.Valid        := False;  -- No valid data.
         An_Instance.Audio_On     := False;  -- Audio not avail.
         An_Instance.Receiving    := False;  -- Not receiving signal.
         An_Instance.VCA          := 0.0;    -- No VCA.
         An_Instance.To_From_Flag := 0.0;    -- Stow to/from flag.
      end if;


      if An_Instance.Valid then

         --*******************************************************************
         -- NAV RECEIVING CHECK                                             *
         --*******************************************************************
         --
         -- A check is made to see if the aircraft is within line of sight
         -- recieving distance for the ground station to be recieved.
         -- The formula for calculating the aircrafts maxium distance
         -- from the station at any given altitude above the station is:
         --
         --   DIST = 1.225 X SQUARE ROOT OF THE TRANSMITTING ANTENNA HEIGHT
         --    MINUS THE GROUND ELEVATION PLUS THE HEIGHT OF THE A/C ABOVE
         --    THE STATION.
         --
         -- Note:  The beginning of this section provides a means for setting
         -- the NAV flag invalid.  The primary use of this control is for
         -- disallowing signal reception due to mountain interference, or
         -- loss of line-of-sight receiving.
         --

         --Limit station range to 40 nm
         if An_Instance.Station.Rng < 40.0 then
            An_Instance.Station.Rng := 40.0;
         end if;

         --Tuning delay
         if An_Instance.Frequency_Changed then
            Tuning_Delay := 0.0;
         elsif Tuning_Delay < 0.4 then
            Tuning_Delay := Tuning_Delay + Integration_Constant;
         end if;

         --0.4 second delay and with station range
         if Tuning_Delay >= 0.4 and
            An_Instance.Station.Rng > RGAS then

            Line_Of_Sight := RU.Flimit((1.23*Sqrt(ZGAS)),
                                                    20.0, 300.0);

            An_Instance.Receiving := Line_Of_Sight > RGAS;

         else

            An_Instance.Receiving := false;
            Line_Of_Sight         := 0.0;

         end if;

         -- VOR receiving calculations
         if An_Instance.Receiving then

            --Altitude fade calculation
            if ZGAS > 0.0 then
               Altitude_Fade := Max_Altitude / ZGAS;
            else
               Altitude_Fade := 1.0;
            end if;

            --The maximum range is the lesser of the station range or
            --the line of sight distance.
            Max_Range :=  An_Instance.Station.Rng;
            if Max_Range > Line_Of_Sight then
               Max_Range := Line_Of_Sight;
            end if;

            --Audio signal strength is function of altitude and
            --range to station.
            --Audio noise is inverse function of audio signal strength.
            Scaled_RGAS := RGAS / Max_Range;
            if Scaled_RGAS >= 1.0 then
               Scaled_RGAS := 1.0;
            elsif Scaled_RGAS <= -1.0 then
               Scaled_RGAS :=  -1.0;
            end if;

            Station_Strength := cos(Scaled_RGAS * 90.0, 360.0) * Altitude_Fade;
            if Station_Strength <= 0.05 then
               Station_Strength := 0.05;
            elsif Station_Strength >= 0.95 then
               Station_Strength := 0.95;
            end if;
            An_Instance.VCA := Station_Strength;

            -- Cone of confusion error calculations:
            if RGAS > 4.5 then
               Cone_Error := 0.0;
            else
               if ZGAS > 0.0 then
                  Cone_Tangent := RGAS * 6080.0 / ZGAS;
               else
                  Cone_Tangent := 1.0;
               end if;

               if Cone_Tangent <= 0.57735 then -- Within 30 degs.

                  if Cone_Timer >= 1.0 then
                     Cone_Timer := 0.0;
                     Cone_Error :=(0.57735 - Cone_Tangent) * 155.0 * Random(G);
                  else
                     Cone_Timer := Cone_Timer + Integration_Constant;
                  end if;

               else
                  Cone_Error := 0.0;
               end if;

            end if;

            --bearing computations
            True_Bearing := Brg_To_Stn + Cone_Error;

            Temp := RU.xn180(True_Bearing - True_Brg_Deg);
            Temp := RU.Flimit(Temp, -112.0*Integration_Constant,
                                     112.0*Integration_Constant);

            True_Brg_Deg := RU.Xn180(True_Brg_Deg + Temp);

            An_Instance.Mag_Brg_Deg := RU.Xn180(True_Brg_Deg -
                                       An_Instance.Station.Hdg);

            --VOR deviations
            Course_Error := RU.Xn180(Course_Error +
                   RU.Xn180(RU.Xn180(An_Instance.Mag_Brg_Deg -
                                     An_Instance.Course_Selected) -
                                     Course_Error)*Integration_Constant);

            An_Instance.Dev_Dots := RU.Flimit(Sin(Course_Error, 360.0) *
                                            (-11.52),-8.0,8.0);

            An_Instance.To_From_Flag := RU.Flimit((90.0 - abs(Course_Error))
                                                  * 0.2,-1.0,1.0);

         else

           An_Instance.VCA := 0.0;
           An_Instance.To_From_Flag := 0.0;

         end if;

      end if;               -- DC power not valid.

   end Update;

end VOR;
