-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------


package body Radio_Utilities is

   Agas : Float;
   Bgas : Float;
   Zgas : Float;
   Vca  : Float;

   Nm_To_Ft : constant := 6076.1;

   function xn180(BRG : Long_Float) return Long_Float is

      Answer : Long_Float;

   begin

      Answer := Brg;

      while Answer > 180.0 loop
         Answer := Answer - 360.0;
      end loop;

      while Answer <= -180.0 loop
         Answer := Answer + 360.0;
      end loop;

      return Answer;

   end Xn180;

-------------------------------------------------------------------------------

   function Flimit(VALUE : Float;
                   MIN   : Float;
                   MAX   : Float) return Float is

      Answer : Float;

   begin

      if VALUE < MIN then
         Answer := MIN;
      elsif VALUE > MAX then
         Answer := MAX;
      else
         Answer := VALUE;
      end if;

      return Answer;

   end Flimit;
-------------------------------------------------------------------------------

   function Flimit(VALUE : Integer;
                   MIN   : Integer;
                   MAX   : Integer) return Integer is

      Answer : Integer;

   begin

      if VALUE < MIN then
         Answer := MIN;
      elsif VALUE > MAX then
         Answer := MAX;
      else
         Answer := VALUE;
      end if;

      return Answer;

   end Flimit;
-------------------------------------------------------------------------------

   function Om_Model(Agas_In : Float;
                     Bgas_In : Float;
                     Zgas_In : Float) return Float is

      Om_Gain  : constant := 1_500.0;
      Om_Ka    : constant :=     4.0;
      Om_Kb    : constant :=     1.0;
      Om_Kz    : constant :=     1.0;

   begin

      Agas := Agas_In * Nm_To_Ft;
      Bgas := Bgas_In * Nm_To_Ft;
      Zgas := Zgas_In;
      if Zgas < 1.0 then Zgas := 1.0; end if;

      Vca := (Om_Gain * Zgas) /
        (Om_Ka*Agas**2 + Om_Kb*Bgas**2 + Om_Kz*Zgas**2);

      if Vca > 1.0 then Vca := 1.0; end if;

      return Vca;

   end Om_Model;

------------------------------------------------------------------------------

   function Mm_Model(Agas_In : Float;
                     Bgas_In : Float;
                     Zgas_In : Float) return Float is

      Mm_Gain  : constant := 1_500.0;
      Mm_Ka    : constant :=     4.0;
      Mm_Kb    : constant :=     1.0;
      Mm_Kz    : constant :=     1.0;

   begin

      --Convert to feet:
      Agas := Agas_In * Nm_To_Ft;
      Bgas := Bgas_In * Nm_To_Ft;
      Zgas := Zgas_In;
      if Zgas < 1.0 then Zgas := 1.0; end if;

      Vca := (Mm_Gain * Zgas) /
        (Mm_Ka*Agas**2 + Mm_Kb*Bgas**2 + Mm_Kz*Zgas**2);

      if Vca > 1.0 then Vca := 1.0; end if;

      return Vca;

   end Mm_Model;

------------------------------------------------------------------------------

   function Im_Model(Agas_In       : Float;
                     Bgas_In       : Float;
                     Zgas_In       : Float;
                     Station_Type2 : Dbt.Bitfield) return Float is

      type Im_Awm_Type is (None,Bone_Marker,Z_Marker,Fan_Marker,
                           Bc_Marker,Inner_Marker);
      Im_Awm       : Im_Awm_Type;

      Temp      : Float;
      Temp_Agas : Float;
      Temp_Bgas : Float;

      --Inner marker shaping constants.
      Im_Gain : constant := 1_500.0;
      Im_Ka   : constant :=     4.0;
      Im_Kb   : constant :=     1.0;
      Im_Kz   : constant :=     1.0;

      --Bone marker shaping constants.
      Bm_Gain : constant := 8_000.0;
      Bm_Ka   : constant :=     8.0;
      Bm_Kb   : constant :=     1.0;
      Bm_Kz   : constant :=     4.0;

      --Fan Marker shaping constants.
      Fm_Gain : constant := 8_000.0;
      Fm_Ka   : constant :=     8.0;
      Fm_Kb   : constant :=     1.0;
      Fm_Kz   : constant :=     2.0;

      --Z Marker shaping constants.
      Zm_Gain : constant := 5_000.0;
      Zm_Ka   : constant :=     8.0;
      Zm_Kb   : constant :=     8.0;
      Zm_Kz   : constant :=     1.0;

      --Back Course Marker shaping constants.
      Bc_Gain : constant := 3_000.0;
      Bc_Ka   : constant :=     4.0;
      Bc_Kb   : constant :=     1.0;
      Bc_Kz   : constant :=     1.0;

   begin

--*********************************************************************
--*># INNER MARKER / AIRWAYS MARKERS
--*********************************************************************
--*>
--*> The inner and airways markers are combined into the same
--*> channel. A type check is done to see which type of marker is
--*> being received to simulate the correct shape and power output.
--*>
      if Station_Type2(24)    then
         Im_Awm := Bone_Marker;
      elsif Station_Type2(23) then
         Im_Awm := Z_Marker;
      elsif Station_Type2(22) then
         Im_Awm := Fan_Marker;
      elsif Station_Type2(21) then
         Im_Awm := Bc_Marker;
      elsif Station_Type2(29) then
         Im_Awm := Inner_Marker;
      else
         Im_Awm := None;
      end if;

      --Convert to feet:
      Agas := Agas_In * Nm_To_Ft;
      Bgas := Bgas_In * Nm_To_Ft;
      Zgas := Zgas_In;
      if Zgas < 1.0 then Zgas := 1.0; end if;

      --Selection based upon type of inner marker.
      case Im_Awm is

         when Inner_Marker =>

            Vca := (Im_Gain * Zgas)/
              (Im_Ka*Agas**2 + Im_Kb*Bgas**2 + Im_Kz*Zgas**2);

--*********************************************************************
--*>## AIRWAYS MARKERS
--*********************************************************************
--*>
--*> The airways markers are generally a high power marker beacon
--*> located along an airway. There are four seperate types that
--*> differ in power output and shape.
--*>
--*********************************************************************
--*>##.1 BONE MARKER
--*********************************************************************
--*>
--*> Bone markers are generally located along an airway and has a
--*> distinctive bone shape where the ends of the radiated pattern
--*> covers a larger area than the center of the marker. They have
--*> a high power output to enable them to be received at altitudes
--*> up to 20000 feet.
--*>
         when Bone_Marker =>

            Temp_Agas := abs(Agas);
            if Temp_Agas < 2_000.0 then Temp_Agas := 2_000.0; end if;

            Temp_Bgas := abs(Bgas);
            if Temp_Bgas < 500.0 then Temp_Bgas := 500.0; end if;

            Temp := Temp_Bgas/Temp_Agas + 0.5;

            Vca := (Bm_Gain*Zgas)/
              (Bm_Ka*Agas**2 + Bm_Kb*Bgas**2 + Bm_Kz*Zgas**2) * Temp;

--*********************************************************************
--*>##.2 FAN MARKER
--*********************************************************************
--*>
--*> Fan markers are generally located along an airway and are
--*> oval in shape much like an ils marker beacon. They have a
--*> high power output to enable them to be received at altitudes
--*> upto 20000 feet.
--*>
         when Fan_Marker =>

            Vca := (Fm_Gain*Zgas)/
              (Fm_Ka*Agas**2 + Fm_Kb*Bgas**2 + Fm_Kz*Zgas**2);

--*********************************************************************
--*>##.3 Z MARKER
--*********************************************************************
--*>
--*> Z markers are generally co-located with a vor station and are
--*> cone shaped. They are used to mark a position directly above
--*> the associated vor station.
--*>
         when Z_Marker =>

            Vca := (Zm_Gain*Zgas)/
              (Zm_Ka*Agas**2 + Zm_Kb*Bgas**2 + Zm_Kz*Zgas**2);

--*********************************************************************
--*>##.4 BACK COURSE MARKER
--*********************************************************************
--*>
--*> Back course markers are generally located along a back course
--*> ils approach to provide a distance fix for aircraft flying the
--*> back course approach. They are shaped like the ils marker
--*> beacons but have a higher power output.
--*>
         when Bc_Marker =>

            Vca := (Bc_Gain*Zgas)/
              (Bc_Ka*Agas**2 + Bc_Kb*Bgas**2 + Bc_Kz*Zgas**2);

--*****************************************************************
--*>##.5 INVALID STATION TYPE
--*****************************************************************
--*>
--*>  If the station type does not match the types for an inner or
--*>  airway marker, then the vca output is set to zero.
--*>

         when others =>

            Vca := 0.0;

      end case;

      if Vca > 1.0 then Vca := 1.0; end if;

      return Vca;

   end Im_Model;



end Radio_Utilities;
