-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random,
  Radio_Utilities;

use  ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random;

package body Radio.Audio.DME is

   package RU renames Radio_Utilities;

   Power_Timer          : Float;
   Rras_N1              : Float;
   Table_Timer          : Float;
   type Spd_Table_Type is array (1 .. 20) of Float;
   Spd_Table            : Spd_Table_Type;
   Index                : Integer;
   Rec_Timer            : Float;
   Closure_Rate         : Float;
   Spd_Table_N1         : Float;
   Table_Sum            : Float;
   Average_Closure_Rate : Float;
   Delay_Timer          : Float;
   Tts_Timer            : Float;
   XGAS                 : Float;
   YGAS                 : Float;
   RGAS                 : Float;
   ZGAS                 : Float;
   Rras                 : Float;
   Dme_Range            : Float;
   Alt_Fade             : Float;
   Max_Alt              : constant := 20_000.0;
   Hr                   : constant := 3_600.0;
   Line_Of_Sight        : Float;

   procedure Set_ILS_Freq_Tuned
     (An_Instance    : in out Instance;
      ILS_Freq_Tuned : in     Boolean) is
   begin
      An_Instance.ILS_Freq_Tuned := ILS_Freq_Tuned;
   end;

   procedure Set_Frequency_Hold_Selected
     (An_Instance             : in out Instance;
      Frequency_Hold_Selected : in     Boolean) is
   begin
      An_Instance.Frequency_Hold_Selected := Frequency_Hold_Selected;
   end;

   procedure Set_Range_To_Station
     (An_Instance      : in out Instance;
      Range_To_Station : in     Radio_Types.Range_Type) is
   begin
      An_Instance.Range_To_Station := Range_To_Station;
   end;

   procedure Set_Knots_Calculated
     (An_Instance      : in out Instance;
      Knots_Calculated : in     Radio_Types.Knots_Type) is
   begin
      An_Instance.Knots_Calculated := Knots_Calculated;
   end;

   procedure Set_TTS_Calculated
     (An_Instance    : in out Instance;
      TTS_Calculated : in     Radio_Types.Timer_Type) is
   begin
      An_Instance.TTS_Calculated := TTS_Calculated;
   end;

   procedure Set_Kts_Tts_Valid
     (An_Instance   : in out Instance;
      Kts_Tts_Valid : in     Boolean) is
   begin
      An_Instance.Kts_Tts_Valid := Kts_Tts_Valid;
   end;

   procedure Set_Dme_Hold
     (An_Instance : in out Instance;
      Dme_Hold    : in     Boolean) is
   begin
      An_Instance.Dme_Hold := Dme_Hold;
   end;

   procedure Set_No_Computed_Data
     (An_Instance      : in out Instance;
      No_Computed_Data : in     Boolean) is
   begin
      An_Instance.No_Computed_Data := No_Computed_Data;
   end;

   function ILS_Freq_Tuned
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.ILS_Freq_Tuned;
   end;

   function Frequency_Hold_Selected
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Frequency_Hold_Selected;
   end;

   function Range_To_Station
     (An_Instance : in Instance)
      return Radio_Types.Range_Type is
   begin
      return An_Instance.Range_To_Station;
   end;

    function Knots_Calculated
      (An_Instance : in Instance)
     return Radio_Types.Knots_Type is
   begin
      return An_Instance.Knots_Calculated;
   end;

   function TTS_Calculated
     (An_Instance : in Instance)
      return Radio_Types.Timer_Type is
   begin
      return An_Instance.TTS_Calculated;
   end;

   function Kts_Tts_Valid
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Kts_Tts_Valid;
   end;

   function Dme_Hold
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Dme_Hold;
   end;

   function No_Computed_Data
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.No_Computed_Data;
   end;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =

   procedure INIT(An_Instance : in out Instance) is

      Variable_Registered : Boolean;
   begin

      -- Initialize instance components:
      An_Instance.ILS_Freq_Tuned          := false;
      An_Instance.Frequency_Hold_Selected := false;
      An_Instance.Range_To_Station        := 0.0;
      An_Instance.Knots_Calculated        := 0.0;
      An_Instance.TTS_Calculated          := 0.0;
      An_Instance.Kts_Tts_Valid           := false;
      An_Instance.Dme_Hold                := False;
      An_Instance.No_Computed_Data        := False;

      --Local values
      Power_Timer  := 0.0;
      Rras_N1      := 0.0;
      Table_Timer  := 0.0;
      for I in Spd_Table_Type'Range loop
         Spd_Table(I) := 0.0;
      end loop;
      Index        := 0;
      Rec_Timer     := 0.0;
      Closure_Rate  := 0.0;
      Spd_Table_N1  := 0.0;
      Table_Sum     := 0.0;
      Average_Closure_Rate := 0.0;
      Delay_Timer := 0.0;
      Tts_Timer   := 0.0;

   end Init;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
   procedure Update(Integration_Constant : in     Float;
                    An_Instance          : in out Instance) is

   begin

--*********************************************************************
-- XGAS TO STATION, IN NAUTICAL MILES ( N/S DISTANCE )
--*********************************************************************
--
-- The xgas term is the north-south distance from the aircraft
-- to the selected ground station, in nautical miles.  The value
-- is protected from a zero result to prevent overflow problems
-- in other modules.  The equation for xgas is:
--
--     xgas=(stn lat - a/c lat) * 60.0  (nm/deg)
--
      XGAS := float(An_Instance.Station.Lat -
                    An_Instance.Aircraft_Position.Latitude ) * 60.0;
      -- Prevent zero divide errors:
      if Xgas = 0.0 then
         Xgas := 0.00001;
      end if;

--*******************************************************************
-- YGAS TO STATION, IN NAUTICAL MILES ( E/W DISTANCE )
--*******************************************************************
--
-- The ygas term is the east-west distance from the aircraft
-- to the selected ground station, in nautical miles.  The
-- distance is corrected for meridian convergence by use of the
-- cosine of latitude term.  This value is zero protected.
-- The equation for ygas is:
--
--   ygas = (stn lon - a/c lon) * 60.0nm/deg * cosine(a/c lat)
--
      YGAS := float(RU.Xn180(An_Instance.Station.Lon -
                             An_Instance.Aircraft_Position.Longitude))*60.0 *
        cos(float(An_Instance.Aircraft_Position.Latitude),360.0);

      -- Prevent zero divide errors:
      if YGAS = 0.0 then
         YGAS := 0.00001;
      end if;

--*********************************************************************
-- RANGE TO STATION, IN NAUTICAL MILES ( GROUND RANGE )
--*********************************************************************
--
-- The range to station is the range measured along the ground
-- from the aircraft to the ground station in nautical miles.
-- The pythagorean theorem is used in conjunction with the xgas
-- and ygas terms to produce the ground range rgas.  This
-- is determined by the equation:
--
--     rgas = sqrt( xgas**2 + ygas**2 )
--
      RGAS := sqrt( XGAS ** 2 + YGAS ** 2 );

--*********************************************************************
-- A/C ALTITUDE ABOVE STATION, IN FEET
--*********************************************************************
--
-- The aircraft altitude above the ground station is determined
-- by the difference between the aircraft geometric altitude
-- (feet msl) and the ground station elevation (feet msl).
-- Positive indicates the aircraft is above the station.  The
-- altitude above the station term (nzgas) is determined by
-- the equation:
--
--     zgas= a/c geometric altitude - station elevation
--
      ZGAS := An_Instance.Aircraft_Position.Altitude -
        An_Instance.Station.Elv ;

      if ZGAS < 1.0 then ZGAS := 1.0; end if;

--*********************************************************************
-- SLANT RANGE TO STATION, NAUTICAL MILES
--*********************************************************************
--
-- The slant range to the station only applies for DME and TACAN
-- facility types.  The slant range is computed using the ground
-- range to the station, the aircraft height above the station,
-- and the pythagorean theroem.  The slant range (nrras) is
-- determined by the equation:
--
--                                         nzgas (feet)
--     rras = sqrt(  rgas (feet) ** 2  +  --------------- ** 2 )
--                                         6076.1 feet/nm
--
      Rras := Sqrt(Rgas ** 2 + (Zgas / 6076.1) ** 2);

-- =  =  end of NAV Geometry section.  =  =  =

--*********************************************************************
--*># DME RECEIVER VALID CHECK                                        *
--*********************************************************************
--*>
--*> The DME systems main DC power is checked to see if the DME system
--*> is powered up for operation. If power is available, then a short
--*> power up timer is run to simulate the warmup time. When this
--*> timer reaches the maximum time for warmup, a valid flag is set
--*> to indicate that the DME is ready for operation.  If the valid
--*> flag is not set, then all output data is set to a inoperational
--*> state and a jump is made to the end of the module.
--*>
      if An_Instance.Power then
         if Power_Timer >= 1.0 then
            An_Instance.Valid := True;
            An_Instance.Audio_On := True;
         else
            Power_Timer := Power_Timer + Integration_Constant;
            An_Instance.Valid := False;
            An_Instance.Audio_On := False;
         end if;

         if not An_Instance.Valid then
            An_Instance.Vca := 0.0;
            An_Instance.Receiving := False;
            An_Instance.Kts_Tts_Valid := False;
            An_Instance.Dme_Hold := False;
            An_Instance.Range_To_Station := 0.0;
            An_Instance.Knots_Calculated := 0.0;
            An_Instance.TTS_Calculated := 0.0;
         else

            --Dme receiver valid
--******************************************************************
-- DME RECEIVING                                                   *
--******************************************************************
--
-- A check is made to see if the aircraft is within line of sight
-- recieving distance for the ground station to be recieved,
-- and that the station is either a dme or tacan station.
-- The formula for calculating the aircrafts maxium distance
-- from the station at any given altitude above the station is:
--
--      DIST = 1.23 X SQUARE ROOT (A/C'S ALTITUDE ABOVE STATION
--
            if An_Instance.ILS_Freq_Tuned then
               Dme_Range := An_Instance.Station.Rng;
            else
               Dme_Range := (An_Instance.Station.Rng - 10.0) * 2.5;
            end if;

            if Dme_Range > RGAS then
               Line_Of_Sight := RU.Flimit((1.23*Sqrt(ZGAS)),20.0, 300.0);
               An_Instance.Receiving := Line_Of_Sight > Rgas;

            else
               An_Instance.Receiving := False;
               Line_Of_Sight := 0.0;

            end if;

--*> Once the DME transceiver is receiving a station, a delay timer is
--*> triggered to allow some tracking calculations to occur before the
--*> DME output data is enabled. For the ARINC 429 output, this enable
--*> flag is "ndncd" ( No Computed Data ). This logic accounts for the
--*> delay time that normally occurs between DME Search & Track modes.
--*>
            if An_Instance.Receiving then
               if Rec_Timer > 0.0 then
                  Rec_Timer := Rec_Timer - Integration_Constant;
                  An_Instance.No_Computed_Data := True;
               else
                  An_Instance.No_Computed_Data := False;
               end if;
            else
               Rec_Timer := 2.0;
               An_Instance.No_Computed_Data := True;
            end if;

--*********************************************************************
--*># DME SIGNAL STRENGTH                                             *
--*********************************************************************
--*>
--*> The audio and noise volume levels are a function of distance
--*> from the station and the stations maxium transmitting range.
--*>
--*>                                   DISTANCE FROM STATION
--*>    AUDIO VOLUME = 1.02 -    ----------------------------------
--*>                             STATIONS MAXIUM TRANSMITTING RANGE
--*>
--*> The noise is calculated as the inverse of the audio.
--*>
            if An_Instance.Receiving then
               if Zgas > 0.0 then
                  Alt_Fade := Max_Alt/Zgas;
               else
                  Alt_Fade := 1.0;
               end if;

               Alt_Fade := Ru.Flimit(Alt_Fade,0.0,1.0);

               An_Instance.Vca := Ru.Flimit
                 ((1.2 - (Rgas/Line_Of_Sight)) * Alt_Fade,0.0,1.0);
            else
               An_Instance.Vca := 0.0;
            end if;

--*********************************************************************
--*># SLANT RANGE DISTANCE FOR DME                                    *
--*********************************************************************
--*>
--*> Label NRRAS which IS the slant range distance from the A/C to the
--*> DME station (or any other station).  The slant range distance is
--*> calculated using the equation:
--*>
--*>     RRAS = SQRT( RGAS**2 + ( ZGAS/6080.0 ) **2 )
--*>
            An_Instance.Range_To_Station := Ru.Flimit(Rras,0.0,300.0);

--*********************************************************************
--*># KTS/TTS CALCULATIONS                                            *
--*********************************************************************
--*>
--*> This section computes the closure rate of the aircraft towards
--*> the station.  The speed data supplied by the DME is sometimes
--*> mistaken for true ground speed.  This is only true at long dist-
--*> ances and when flying directly towards or away from the station.
--*> The Time To Station is also calculated as a function of distance
--*> from the station and closure rate.
--*>

--*********************************************************************
--*>## CLOSURE RATE (KTS)                                             *
--*********************************************************************
--*>
--*> NGSPD calculates an instantaneous closure rate in KTS each pass
--*> of this module.
--*>
            Closure_Rate := abs(Rras - Rras_N1) * (Hr/Integration_Constant);
            Rras_N1 := Rras;

--*> There exists a table of 20 elements for each DME channel.  The
--*> table saves the last 20 values of closure rate which will be used
--*> to calculate an average closure rate in knots.  Every time the
--*> table timer NTBLTMR counts down to 0, the next element in line is
--*> updated with the current closure rate value from NGSPD in knots.
--*> Please note that if the DME closure rate is updating too quickly,
--*> the rate at which the averaging occurs can be slowed by changing
--*> the table timer to a higher value.
--*>
            Table_Timer := Table_Timer - Integration_Constant;

            if Table_Timer <= 0.0 then
               Table_Timer := 1.0;

               --Increment table index
               if Index < 20 then
                  Index := Index + 1;
               else
                  Index := 1;
               end if;

               --Save old speed table value
               Spd_Table_N1 := Spd_Table(Index);
               --Store new speed table value
               Spd_Table(Index) := Closure_Rate;

--*> Once the table has been updated with the new element value, the
--*> sum of the 20 table elements is once again calculated.
--*>
               Table_Sum := Table_Sum - Spd_Table_N1 + Spd_Table(Index);

--*> Finally the average closure rate in kts "navgkts" is calculated
--*> by simply dividing the table sum "ndktsum" by 20 (# of elements).
--*>
               Average_Closure_Rate := Table_Sum/20.0;
            end if;

--*> It may be necessary to put some delay on the DME closure rate
--*> value which is output to the respective display via NDKTS.  So,
--*> a second timer, "nktstmr" is added to provide some flexibility
--*> for updating the DME display.
--*>
            if Delay_Timer <= 0.0 then
               Delay_Timer := 2.0;
               An_Instance.Knots_Calculated := Average_Closure_Rate;
            else
               Delay_Timer := Delay_Timer - Integration_Constant;
            end if;

            --Limit to 999 kts
            if An_Instance.Knots_Calculated > 999.0 then
               An_Instance.Knots_Calculated := 999.0;
            end if;

--*> There is a valid flag for the closure rate (KTS) which does not
--*> become valid until KTS is 50 knots or greater.
--*>
            An_Instance.Kts_Tts_Valid := An_Instance.Knots_Calculated >= 50.0;

--*********************************************************************
--*>## TIME TO STATION (TTS)                                          *
--*********************************************************************
--*>
--*> Time To Station is calculated by dividing slant range distance by
--*> the closure rate to obtain the time in hours.  A constant value
--*> of 60 is multiplyed with the closure rate to convert the time
--*> into minutes.  A check is made to ensure closure rate is greater
--*> than zero to prevent divide errors at run-time.
--*>
            if Tts_Timer <= 0.0 then
               Tts_Timer := 2.0;
               if An_Instance.Knots_Calculated > 1.0 then
                  An_Instance.TTS_Calculated :=
                    Rras * 60.0 /An_Instance.Knots_Calculated;
               else
                  An_Instance.TTS_Calculated := 0.0;
               end if;

               --Upper limit to 120 minutes
               if An_Instance.TTS_Calculated > 120.0 then
                  An_Instance.TTS_Calculated := 120.0;
               end if;

            else
               Tts_Timer := Tts_Timer - Integration_Constant;
            end if;

--*********************************************************************
--*># IDENT DISPLAY                                                   *
--*********************************************************************
--*>
--*> DME station identification ( or ident ) is transmitted from the
--*> ground station every 30 to 35 seconds.  The ident is in Inter-
--*> national Morse code and can have from two to four alpha ( or
--*> numeric ) characters.  There are presently no numeric characters
--*> used in idents, but the possibility exists in the future.  To
--*> reduce the complexity of station ident determination in the code,
--*> a simple timer of approximately 30 seconds exists to represent an
--*> average time for the DME transceiver to detect an ident.  If a
--*> station is not in tune or is not being received, then the ident
--*> is replaced with dashes (-).  If a new station is tuned from an
--*> existing valid station, the 30 second delay begins again.
--*>
--      if ( .not.nd1rec(i) ) then                !not receiving
--        nidttmr(i) = 30.0                       !  reset for 30 sec
--        noldstn(i) = 0                          !  clr old stn ident
--        niddisp(i) = z'2d2d2d2d'                !  clear ident
--      elseif ( nstnidnt(l) .ne. noldstn(i) ) then !old & new stn same
--        nidttmr(i) = 30.0                       !  reset for 30 sec
--        noldstn(i) = nstnidnt(l)                !  save old nstnidnt
--        niddisp(i) = z'2d2d2d2d'                !  clear ident
--      else                                      !rec & same stn
--        if ( nidttmr(i) .le. 0.0 ) then         !  if 30 sec expired
--          niddisp(i) = nstnidnt(l)              !    display ident
--        else                                    !  still counting
--          nidttmr(i) = nidttmr(i) - iconst      !    decr. ident tmr
--        endif                                   !
--      endif                                     !

--*> The station ident from the radio database is left justified.  The
--*> ARINC 429 data spec requires the ident characters to be right
--*> justified and that any ident with fewer than 4 characters have
--*> the blank code bits set to zero.  The code below provides for two
--*> blank character checks since there are at least two characters in
--*> any ident.  If idents are less than four characters, then one or
--*> both of the shifts will occur.  Zeros will replace the leading
--*> bytes by default when the shift is done.
--*>
--      itemp = iand( niddisp(i), z'000000ff' )   !mask 4th character
--      if ( itemp .eq. z'00000020' )             !if 4th char = space,
--     >  niddisp(i) = ishft( niddisp(i), -8 )    !  shft right 1 char.

--      itemp = iand( niddisp(i), z'000000ff' )   !mask 3rd character
--      if ( itemp .eq. z'00000020' )             !if 3rd char = space,
--     >  niddisp(i) = ishft( niddisp(i), -8 )    !  shft right 1 char.


         end if;
-- DME not powered
      else
         An_Instance.Valid := False;
         An_Instance.Vca := 0.0;
         An_Instance.Receiving := False;
         An_Instance.Kts_Tts_Valid := False;
         An_Instance.Dme_Hold := False;
         An_Instance.Range_To_Station := 0.0;
         An_Instance.Knots_Calculated := 0.0;
         An_Instance.TTS_Calculated := 0.0;
      end if;
   end Update;

end Radio.Audio.Dme;
