-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  Radio_Utilities,
  Jpats_Avionics;

use  ada.Numerics.Elementary_Functions;

package body Radio.Audio.Communication.UHF_Comm is

   package RU renames Radio_Utilities;

   Power_Timer : Float;

   procedure Set_Frequency
     (An_Instance : in out Instance;
      Frequency   : in     Integer) is
   begin
      An_Instance.Frequency := Frequency;
   end;

   procedure Set_Elt_Transmit
     (An_Instance  : in out Instance;
      Elt_Transmit : in     Boolean) is
   begin
      An_Instance.Elt_Transmit := Elt_Transmit;
   end;

   function Frequency
     (An_Instance : in Instance)
      return Integer is
   begin
      return An_Instance.Frequency;
   end;

   function Elt_Transmit
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Elt_Transmit;
   end;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =

   procedure INIT(An_Instance : in out Instance) is

   begin

      Power_Timer := 0.0;
      An_Instance.Frequency := 245000;
      An_Instance.Elt_Transmit := False;

   end Init;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
   procedure Update(Integration_Constant : in     Float;
                    An_Instance          : in out Instance) is

      XGAS          : Float;
      YGAS          : Float;
      RGAS          : Float;
      ZGAS          : Float;
      Stn_Range     : Float;
      Line_Of_Sight : Float := 10.0;

   begin

--*********************************************************************
-- XGAS TO STATION, IN NAUTICAL MILES ( N/S DISTANCE )
--*********************************************************************
--
-- The xgas term is the north-south distance from the aircraft
-- to the selected ground station, in nautical miles.  The value
-- is protected from a zero result to prevent overflow problems
-- in other modules.  The equation for xgas is:
--
--     xgas=(stn lat - a/c lat) * 60.0  (nm/deg)
--
      XGAS := float(An_Instance.Station.Lat -
                    An_Instance.Aircraft_Position.Latitude ) * 60.0;
      -- Prevent zero divide errors:
      if Xgas = 0.0 then
         Xgas := 0.00001;
      end if;

--*******************************************************************
-- YGAS TO STATION, IN NAUTICAL MILES ( E/W DISTANCE )
--*******************************************************************
--
-- The ygas term is the east-west distance from the aircraft
-- to the selected ground station, in nautical miles.  The
-- distance is corrected for meridian convergence by use of the
-- cosine of latitude term.  This value is zero protected.
-- The equation for ygas is:
--
--   ygas = (stn lon - a/c lon) * 60.0nm/deg * cosine(a/c lat)
--
      YGAS := float(RU.Xn180(An_Instance.Station.Lon -
                             An_Instance.Aircraft_Position.Longitude))*60.0 *
        cos(float(An_Instance.Aircraft_Position.Latitude),360.0);

      -- Prevent zero divide errors:
      if YGAS = 0.0 then
         YGAS := 0.00001;
      end if;

--*********************************************************************
-- RANGE TO STATION, IN NAUTICAL MILES ( GROUND RANGE )
--*********************************************************************
--
-- The range to station is the range measured along the ground
-- from the aircraft to the ground station in nautical miles.
-- The pythagorean theorem is used in conjunction with the xgas
-- and ygas terms to produce the ground range rgas.  This
-- is determined by the equation:
--
--     rgas = sqrt( xgas**2 + ygas**2 )
--
      RGAS := sqrt( XGAS ** 2 + YGAS ** 2 );

--*********************************************************************
-- A/C ALTITUDE ABOVE STATION, IN FEET
--*********************************************************************
--
-- The aircraft altitude above the ground station is determined
-- by the difference between the aircraft geometric altitude
-- (feet msl) and the ground station elevation (feet msl).
-- Positive indicates the aircraft is above the station.  The
-- altitude above the station term (nzgas) is determined by
-- the equation:
--
--     zgas= a/c geometric altitude - station elevation
--
      ZGAS := An_Instance.Aircraft_Position.Altitude -
                  An_Instance.Station.Elv ;

      if ZGAS < 1.0 then ZGAS := 1.0; end if;

-- =  =  end of NAV Geometry section.  =  =  =

--*********************************************************************
--*># RECEIVER POWER CHECK                                          *
--*********************************************************************
--*>
--*> The reciever power logic is checked to see if the reciever
--*> is powered up and operational. If not, then all flags are set
--*> invalid and the rest of the code is bypassed.
--*>
      if An_Instance.Power then
         An_Instance.Valid := True;
         An_Instance.Audio_On := True;
      else
         An_Instance.Valid := False;
         An_Instance.Audio_On := False;
         An_Instance.Receiving := False;
         Power_Timer := 0.0;
         An_Instance.Vca := 0.0;
      end if;

      --Power available.
      if An_Instance.Valid then
--*********************************************************************
--*># COMMUNICATIONS RECIEVER POWER UP TEST TONE                    *
--*********************************************************************
--*>
--*> Upon initial power application to the COMM receiver, an 800 Hz
--*> tone is heard through the audio system signaling the power up
--*> self-test of the internal microprocessors memory.
--*>
         if Power_Timer < 6.0 then
            Power_Timer := Power_Timer + Integration_Constant;
         end if;

--:Note provision for power-up test tone
--*********************************************************************
--*># COMM RECEIVING CHECK
--*********************************************************************
--*>
--*> A check is made to see if the aircraft is within line of sight
--*> recieving distance for the ground station to be recieved.
--*> The formula for calculating the aircrafts maxium distance
--*> from the station at any given altitude above the station is:
--*>
--*>  DIST = 1.225 X SQUARE ROOT OF THE TRANSMITTING ANTENNA HEIGHT
--*>   MINUS THE GROUND ELEVATION PLUS THE HEIGHT OF THE A/C ABOVE
--*>   THE STATION.
--*>
         --If an atis station
         if An_Instance.Station.Typ2(25) then
            Stn_Range := 80.0;
         else
            Stn_Range := An_Instance.Station.Rng;
         end if;

         if Stn_Range < 0.00001 then
            Stn_Range := 0.00001;
         end if;

         if Stn_Range * 1.5 > Rgas then
            Line_Of_Sight := RU.Flimit((1.23*Sqrt(ZGAS)),
                                       10.0, 255.0);

            An_Instance.Receiving := Line_Of_Sight > RGAS;
         else
            An_Instance.Receiving := False;
         end if;

--*********************************************************************
--*># COMM SIGNAL STRENGTH
--*********************************************************************
--*>
--*> The audio and noise volume levels are a function of distance
--*> from the station and the stations maxium transmitting range.
--*>
--*>                                   DISTANCE FROM STATION
--*>    AUDIO VOLUME = 1.02  -   ----------------------------------
--*>                             STATIONS MAXIUM TRANSMITTING RANGE
--*>
--*> The noise is calculated as the inverse of the audio.
--*>
         if not An_Instance.Transmitting then
            if An_Instance.Receiving then
               if Stn_Range > Rgas then
                  An_Instance.Vca := 1.0;
               else
                  An_Instance.Vca :=
                    Ru.Flimit((1.0 - ((Rgas - Stn_Range)/10.0)**2),0.0,1.0);
               end if;
            elsif Stn_Range > Line_Of_Sight then
               An_Instance.Vca :=
                 1.0 - Ru.Flimit((Rgas - Line_Of_Sight) / 5.0, 0.0, 1.0);
            else
               An_Instance.Vca := 0.0;
            end if;
         else
            An_Instance.Vca := 0.0;
         end if;
      end if;
--*********************************************************************
--*># SELF TEST
--*********************************************************************
--*>
--*>  The VHF-422 transceiver has an internal self test which can be
--*>  selected from the RTU controller.  While the self test is in
--*>  progress, two audio tones can be heard through the audio system.
--*>
--      if ( nutstmr(i) .gt. 0.0 ) then             !test in progress ?
--        nutstmr(i) = nutstmr(i) - iconst          !decrement tst timer
--        nutest(i) = ( ( nutstmr(i) .gt. 2.0 ) .and.   !sound tone
--     >                ( nutstmr(i) .lt. 4.0 ) )   !other time in test
--      else                                        !
--        if ( nutsti(i) .and. .not. nutstn1(i) ) then  !test state change
--          nutstmr(i) = 5.0                        !5 second test
--        end if
--      end if

--      nutstn1(i) = nutsti(i)                      !save for n-1


   end Update;


end Radio.Audio.Communication.UHF_Comm;

