-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random,
  Radio_Utilities;

use  ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random;

package body Marker_Beacon is

   package RU renames Radio_Utilities;
   package Db renames Jpats_Radio_Db_If_Types;

   Xgas         : Jpats_Radio_Db_If_Types.Marker_Float;
   YGAS         : Jpats_Radio_Db_If_Types.Marker_Float;
   RGAS         : Jpats_Radio_Db_If_Types.Marker_Float;
   ZGAS         : Jpats_Radio_Db_If_Types.Marker_Float;
   Sinhdg       : Jpats_Radio_Db_If_Types.Marker_Float;
   Coshdg       : Jpats_Radio_Db_If_Types.Marker_Float;
   Agas         : Jpats_Radio_Db_If_Types.Marker_Float;
   Bgas         : Jpats_Radio_Db_If_Types.Marker_Float;

   function Power
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Power;
   end;

   function High_Sense
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.High_Sense;
   end;

   function Aircraft_Position
     (An_Instance : in Instance)
      return Radio_Types.Position_Type is
   begin
      return An_Instance.Aircraft_Position;
   end;

   function Station
     (An_Instance : in Instance)
      return Jpats_Radio_Db_If_Types.Mrkr_Stn_Struct is
   begin
      return An_Instance.Station;
   end;

   function Inner_Vca
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Inner_Vca;
   end;

   function Middle_Vca
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Middle_Vca;
   end;

   function Outer_Vca
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Outer_Vca;
   end;

   function Inner_Receiving
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Inner_Receiving;
   end;

   function Middle_Receiving
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Middle_Receiving;
   end;

   function Outer_Receiving
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Outer_Receiving;
   end;

   procedure Set_Power
     (An_Instance : in out Instance;
      Power       : in     Boolean) is
   begin
      An_Instance.Power := Power;
   end;

   procedure Set_High_Sense
     (An_Instance    : in out Instance;
      High_Sense : in     Boolean) is
   begin
      An_Instance.High_Sense := High_Sense;
   end;

   procedure Set_Aircraft_Position
     (An_Instance       : in out Instance;
      Aircraft_Position : in     Radio_Types.Position_Type) is
   begin
      An_Instance.Aircraft_Position := Aircraft_Position;
   end;

   procedure Set_Station
     (An_Instance : in out Instance;
      Station     : in     Jpats_Radio_Db_If_Types.Mrkr_Stn_Struct) is
   begin
      An_Instance.Station := Station;
   end;

   procedure Set_Inner_Vca
     (An_Instance : in out Instance;
      Inner_Vca   : in     Float) is
   begin
      An_Instance.Inner_Vca := Inner_Vca;
   end;

   procedure Set_Middle_Vca
     (An_Instance : in out Instance;
      Middle_Vca  : in     Float) is
   begin
      An_Instance.Middle_Vca := Middle_Vca;
   end;

   procedure Set_Outer_Vca
     (An_Instance : in out Instance;
      Outer_Vca   : in     Float) is
   begin
      An_Instance.Outer_Vca := Outer_Vca;
   end;

   procedure Set_Inner_Receiving
     (An_Instance     : in out Instance;
      Inner_Receiving : in     Boolean) is
   begin
      An_Instance.Inner_Receiving := Inner_Receiving;
   end;

   procedure Set_Middle_Receiving
     (An_Instance      : in out Instance;
      Middle_Receiving : in     Boolean) is
   begin
      An_Instance.Middle_Receiving := Middle_Receiving;
   end;

   procedure Set_Outer_Receiving
     (An_Instance     : in out Instance;
      Outer_Receiving : in     Boolean) is
   begin
      An_Instance.Outer_Receiving := Outer_Receiving;
   end;


-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =

   procedure Init(An_Instance : in out Instance) is

      Variable_Registered : Boolean;
   begin

      -- Initialize instance components:
      An_Instance.Power              := False;
      An_Instance.High_Sense         := False;
      An_Instance.Aircraft_Position  := (Latitude  => 45.0,
                                         Longitude => -45.0,
                                         Altitude  => 1000.0);
      An_Instance.Inner_Vca          := 0.0;
      An_Instance.Inner_Receiving    := False;
      An_Instance.Middle_Vca         := 0.0;
      An_Instance.Middle_Receiving   := False;
      An_Instance.Outer_Vca          := 0.0;
      An_Instance.Outer_Receiving    := False;

      Xgas         := (0.0,0.0,0.0);
      YGAS         := (0.0,0.0,0.0);
      RGAS         := (0.0,0.0,0.0);
      ZGAS         := (0.0,0.0,0.0);
      Sinhdg       := (0.0,0.0,0.0);
      Coshdg       := (0.0,0.0,0.0);
      Agas         := (0.0,0.0,0.0);
      Bgas         := (0.0,0.0,0.0);

   end Init;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
   procedure Update(Integration_Constant : in     Float;
                    An_Instance          : in out Instance) is

      -- MARKER SHAPING CONSTANTS
      Om_Ratio : constant := 1.75;
      Mm_Ratio : constant := 2.50;
      Im_Ratio : constant := 2.50;

      Om_Alt_Limit : constant := 1500.0;
      Mm_Alt_Limit : constant := 1500.0;
      Im_Alt_Limit : constant := 1500.0;


   begin

      for I in Jpats_Radio_Db_If_Types.Marker_Type'Range loop

--*********************************************************************
-- XGAS TO STATION, IN NAUTICAL MILES ( N/S DISTANCE )
--*********************************************************************
--
-- The xgas term is the north-south distance from the aircraft
-- to the selected ground station, in nautical miles.  The value
-- is protected from a zero result to prevent overflow problems
-- in other modules.  The equation for xgas is:
--
--     xgas=(stn lat - a/c lat) * 60.0  (nm/deg)
--
         Xgas(I) := float(An_Instance.Station(I).Lat -
                          An_Instance.Aircraft_Position.Latitude ) * 60.0;
         -- Prevent zero divide errors:
         if Xgas(I) = 0.0 then
            Xgas(I) := 0.00001;
         end if;

--*******************************************************************
-- YGAS TO STATION, IN NAUTICAL MILES ( E/W DISTANCE )
--*******************************************************************
--
-- The ygas term is the east-west distance from the aircraft
-- to the selected ground station, in nautical miles.  The
-- distance is corrected for meridian convergence by use of the
-- cosine of latitude term.  This value is zero protected.
-- The equation for ygas is:
--
--   ygas = (stn lon - a/c lon) * 60.0nm/deg * cosine(a/c lat)
--
         Ygas(I) :=
           float(RU.Xn180(An_Instance.Station(I).Lon -
                          An_Instance.Aircraft_Position.Longitude))*60.0 *
           cos(float(An_Instance.Aircraft_Position.Latitude),360.0);

         -- Prevent zero divide errors:
         if Ygas(I) = 0.0 then
            Ygas(I) := 0.00001;
         end if;

--*********************************************************************
-- RANGE TO STATION, IN NAUTICAL MILES ( GROUND RANGE )
--*********************************************************************
--
-- The range to station is the range measured along the ground
-- from the aircraft to the ground station in nautical miles.
-- The pythagorean theorem is used in conjunction with the xgas
-- and ygas terms to produce the ground range rgas.  This
-- is determined by the equation:
--
--     rgas = sqrt( xgas**2 + ygas**2 )
--
         Rgas(I) := sqrt( Xgas(I) ** 2 + Ygas(I) ** 2 );

--*********************************************************************
-- A/C ALTITUDE ABOVE STATION, IN FEET
--*********************************************************************
--
-- The aircraft altitude above the ground station is determined
-- by the difference between the aircraft geometric altitude
-- (feet msl) and the ground station elevation (feet msl).
-- Positive indicates the aircraft is above the station.  The
-- altitude above the station term (nzgas) is determined by
-- the equation:
--
--     zgas= a/c geometric altitude - station elevation
--
         Zgas(I) := An_Instance.Aircraft_Position.Altitude -
           An_Instance.Station(I).Elv;

         if Zgas(I) < 1.0 then Zgas(I) := 1.0; end if;
--*********************************************************************
--*># ILS AND MKR MODES
--*********************************************************************
--*>
--*> ILS and marker beacon facilities require additional information
--*> due to the facilities' orientation.  Common terms are computed
--*> first to optimize the program execution.  Note that the
--*> equations for rotation have been rearranged to take into
--*> account zero degrees is true north.
--*>

         Sinhdg(I) := Sin(An_Instance.Station(I).Hdg,360.0);
         Coshdg(I) := Cos(An_Instance.Station(I).Hdg,360.0);

--*********************************************************************
--*>## AGAS TO LOCALIZER, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The down range term (nagas) is the distance from the aircraft
--*> to the facility measured along the axis of the facility
--*> heading.  Positive values indicate the aircraft is on the
--*> front course, with negative values indicate the aircraft is
--*> on the back course.  This term is in nautical miles, and is
--*> computed using the equation:
--*>
--*>     agas = ygas * sin(rwy hdg) + xgas * cos(rwy hdg)
--*>
--*>
         Agas(I) := Ygas(I) * Sinhdg(I) + Xgas(I) * Coshdg(I);

--*********************************************************************
--*>## BGAS TO LOCALIZER, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The cross range term (nbgas) is the distance from the
--*> aircraft to the facility centerline, measured perpendicular
--*> to the facility centerline.  Positive values indicate the
--*> aircraft is to the right of the facility centerline, and
--*> negative values indicate the aircraft is to the left of the
--*> facility centerline.  This term is in nautical miles, and
--*> is computed using the equation:
--*>
--*>     bgas = xgas * sin(rwy hdg) - ygas * cos(rwy hdg)
--*>
--*>
         Bgas(I) := Xgas(I) * Sinhdg(I) - Ygas(I) * Coshdg(I);

      end loop;

--*********************************************************************
-- OUTER MARKER
--*********************************************************************
--
--*> The outer marker is located along the localizer centerline
--*> approximately 4 to 10 miles from the runway threshold. The
--*> marker shape is oval with a power output for reception up to
--*> an altitude of approximately 3000 feet above the station.
--*>
      if An_Instance.Station(Db.Outer).Lat /= 0.0 then  --any outer markers?

         An_Instance.Outer_Vca := Ru.Om_Model(Agas(Db.Outer),
                                              Bgas(Db.Outer),
                                              Zgas(Db.Outer));
      else
         An_Instance.Outer_Vca := 0.0;
      end if;

--*****************************************************************
--*># MIDDLE MARKER
--*****************************************************************
--*>
--*> The middle marker is located along the localizer centerline
--*> approximately 1/2 to 2 miles from the runway threshold. The
--*> marker shape is oval with a power output for reception up to
--*> an altitude of about 2000 feet above the station.
--*>
      if An_Instance.Station(Db.Middle).Lat /= 0.0 then  --any middle markers?

         An_Instance.Middle_Vca := Ru.Mm_Model(Agas(Db.Middle),
                                               Bgas(Db.Middle),
                                               Zgas(Db.Middle));
      else
         An_Instance.Middle_Vca := 0.0;
      end if;

--*****************************************************************
--*>## INNER MARKER
--*****************************************************************
--*>
--*> The inner marker is located along the localizer centerline
--*> at a distance that will give a glideslope intercept altitude
--*> of approximately 100 feet above the landing runways touch-down
--*> zone elevation. The marker shape is oval with a power output
--*> for reception upto an altitude of 1000 feet above the station.
--*>
      if An_Instance.Station(Db.Inner).Lat /= 0.0 then  --any inner markers?

         An_Instance.Inner_Vca := Ru.Im_Model
           (Agas(Db.Inner),
            Bgas(Db.Inner),
            Zgas(Db.Inner),
            An_Instance.Station(Db.Inner).Typ2);
      else
         An_Instance.Inner_Vca := 0.0;
      end if;

--*****************************************************************
--*># SYSTEM LOGIC & OUTPUT TO D/A'S
--*****************************************************************
--*>
--*> The receiver outputs are done here according to power
--*> availability and marker receiver sense. A high/low sense is
--*> done by increasing the output if the receiver is in the high
--*> sense mode.
--*>
      if not An_Instance.Power then
         An_Instance.Outer_Vca  := 0.0;
         An_Instance.Middle_Vca := 0.0;
         An_Instance.Inner_Vca  := 0.0;
      elsif An_Instance.High_Sense then
         An_Instance.Outer_Vca  :=
           Ru.Flimit(An_Instance.Outer_Vca  * 2.0,0.0,1.0);
         An_Instance.Middle_Vca :=
           Ru.Flimit(An_Instance.Middle_Vca * 2.0,0.0,1.0);
         An_Instance.Inner_Vca  :=
           Ru.Flimit(An_Instance.Inner_Vca  * 2.0,0.0,1.0);
      end if;

--*********************************************************************
--*># MARKER BEACON ON FLAG
--*********************************************************************
--*>
--*>
--*>
--*>
      An_Instance.Inner_Receiving  := An_Instance.Inner_Vca  > 0.05;
      An_Instance.Middle_Receiving := An_Instance.Middle_Vca > 0.05;
      An_Instance.Outer_Receiving  := An_Instance.Outer_Vca  > 0.05;

   end Update;

end Marker_Beacon;
