-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random,
  Radio_Utilities;

use  ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random;

package body Localizer is

   package RU renames Radio_Utilities;

   Temp              : Float;
   Xgas              : Float;
   Ygas              : Float;
   Rgas              : Float;
   Zgas              : Float;
   Agas              : Float;
   Bgas              : Float;
   Agas_Td           : Float;
   Brg_To_Stn        : Radio_Types.Bearing_Type;
   Sinhdg            : Float;
   Coshdg            : Float;
   Ft_To_Nm          : Float := 1.0/6076.0;
   Loc_Antenna_Dist  : Float := 13.33;  --distance from cg to antenna
   Localizer_Cg_Dist : Float;
   Localizer_Range   : Float := 32.0;
   Line_Of_Sight     : Float;
   In_Range          : Boolean;
   Ils_Station       : Boolean;
   Back_Course_Avail : Boolean;
   Max_Agas          : Float;
   Tmp_Agas          : Float;
   Tmp_Bgas          : Float;
   Ac_To_Stn_Angle   : Float;
   Ac_To_Loc_Delta   : Float;
   Threshold_To_Rwy  : Float;
   Loc_Beam_Width    : Float;
   Max_Altitude      : Constant := 20_000.0;
   Altitude_Fade     : Float;
   Max_Range         : Float;
   Scaled_RGAS       : Float;
   Station_Strength  : Float;

   procedure Set_Localizer_Inputs
     (An_Instance         : in out Instance;
      Power               : in     Boolean;
      Mode_On             : in     Boolean;
      Aircraft_Position   : in     Radio_Types.Position_Type;
      Aircraft_True_Hdg   : in     Radio_Types.Heading_Type;
      Station             : in     Jpats_Radio_Db_If_Types.Stn_Struct_Type;
      Center_Of_Gravity_X : in     Float;
      Reference_Airfield  : in     Jpats_Radio_Db_If_Types.Apt_Struct_Type;
      Reference_Runway    : in     Jpats_Radio_Db_If_Types.Rwy_Struct_Type) is
   begin
      An_Instance.Power               := Power;
      An_Instance.Mode_On             := Mode_On;
      An_Instance.Aircraft_Position   := Aircraft_Position;
      An_Instance.Station             := Station;
      An_Instance.Center_Of_Gravity_X := Center_Of_Gravity_X;
      An_Instance.Reference_Airfield  := Reference_Airfield;
      An_Instance.Reference_Runway    := Reference_Runway;
   end;


   function Audio_On
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Audio_On;
   end;

   function To_From_Flag
     (An_Instance : in Instance)
      return Radio_Types.Normalized_Type is
   begin
      return An_Instance.To_From_Flag;
   end;

   function Valid
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Valid;
   end;

   function VCA
     (An_Instance : in Instance)
      return Radio_Types.Normalized_Type is
   begin
      return An_Instance.VCA;
   end;

   function Receiving
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Receiving;
   end;

   function Dev_Dots
     (An_Instance : in Instance)
      return Radio_Types.Dots_Type is
   begin
      return An_Instance.Dev_Dots;
   end;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =

   procedure INIT(An_Instance : in out Instance) is

      Variable_Registered : Boolean;
   begin

      -- Initialize instance components:
      An_Instance.Power               := False;
      An_Instance.Mode_On             := False;
      An_Instance.Aircraft_Position   := (Latitude  => 45.0,
                                          Longitude => -45.0,
                                          Altitude  => 1000.0);
      An_Instance.Aircraft_True_Hdg   := 0.0;
      An_Instance.Center_Of_Gravity_X := 0.0;
      An_Instance.Audio_On            := False;
      An_Instance.To_From_Flag        := 0.0;
      An_Instance.Valid               := False;
      An_Instance.VCA                 := 0.0;
      An_Instance.Receiving           := False;
      An_Instance.Dev_Dots            := 0.0;

      Temp              := 0.0;
      Xgas              := 0.0;
      Ygas              := 0.0;
      Rgas              := 0.0;
      Zgas              := 0.0;
      Agas              := 0.0;
      Bgas              := 0.0;
      Agas_Td           := 0.0;
      Brg_To_Stn        := 0.0;
      Sinhdg            := 0.0;
      Coshdg            := 0.0;
      Localizer_Cg_Dist := 0.0;
      Line_Of_Sight     := 0.0;
      In_Range          := False;
      Ils_Station       := False;
      Back_Course_Avail := False;
      Max_Agas          := 0.0;
      Tmp_Agas          := 0.0;
      Tmp_Bgas          := 0.0;
      Ac_To_Stn_Angle   := 0.0;
      Ac_To_Loc_Delta   := 0.0;
      Threshold_To_Rwy  := 0.0;
      Loc_Beam_Width    := 0.0;
      Altitude_Fade     := 0.0;
      Max_Range         := 0.0;
      Scaled_RGAS       := 0.0;
      Station_Strength  := 0.0;

   end Init;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
   procedure Update_Localizer(Integration_Constant : in     Float;
                              An_Instance          : in out Instance) is

   begin

--*********************************************************************
-- XGAS TO STATION, IN NAUTICAL MILES ( N/S DISTANCE )
--*********************************************************************
--
-- The xgas term is the north-south distance from the aircraft
-- to the selected ground station, in nautical miles.  The value
-- is protected from a zero result to prevent overflow problems
-- in other modules.  The equation for xgas is:
--
--     xgas=(stn lat - a/c lat) * 60.0  (nm/deg)
--
      XGAS := float(An_Instance.Station.Lat -
                    An_Instance.Aircraft_Position.Latitude ) * 60.0;
      -- Prevent zero divide errors:
      if XGAS = 0.0 then
         XGAS := 0.00001;
      end if;

--*******************************************************************
-- YGAS TO STATION, IN NAUTICAL MILES ( E/W DISTANCE )
--*******************************************************************
--
-- The ygas term is the east-west distance from the aircraft
-- to the selected ground station, in nautical miles.  The
-- distance is corrected for meridian convergence by use of the
-- cosine of latitude term.  This value is zero protected.
-- The equation for ygas is:
--
--   ygas = (stn lon - a/c lon) * 60.0nm/deg * cosine(a/c lat)
--
      YGAS := float(RU.Xn180(An_Instance.Station.Lon -
                             An_Instance.Aircraft_Position.Longitude))*60.0 *
        cos(float(An_Instance.Aircraft_Position.Latitude),360.0);

      -- Prevent zero divide errors:
      if YGAS = 0.0 then
         YGAS := 0.00001;
      end if;

--*********************************************************************
-- RANGE TO STATION, IN NAUTICAL MILES ( GROUND RANGE )
--*********************************************************************
--
-- The range to station is the range measured along the ground
-- from the aircraft to the ground station in nautical miles.
-- The pythagorean theorem is used in conjunction with the xgas
-- and ygas terms to produce the ground range rgas.  This
-- is determined by the equation:
--
--     rgas = sqrt( xgas**2 + ygas**2 )
--
      RGAS := sqrt( XGAS ** 2 + YGAS ** 2 );

--*********************************************************************
-- A/C ALTITUDE ABOVE STATION, IN FEET
--*********************************************************************
--
-- The aircraft altitude above the ground station is determined
-- by the difference between the aircraft geometric altitude
-- (feet msl) and the ground station elevation (feet msl).
-- Positive indicates the aircraft is above the station.  The
-- altitude above the station term (nzgas) is determined by
-- the equation:
--
--     zgas= a/c geometric altitude - station elevation
--
      ZGAS := An_Instance.Aircraft_Position.Altitude -
        An_Instance.Station.Elv ;

      if ZGAS < 1.0 then ZGAS := 1.0; end if;

--*********************************************************************
-- BEARING TO THE STATION, IN +/- 180 DEGREES TRUE
--*********************************************************************
--
-- The bearing to station term (nbrgstn) is computed by taking
-- the inverse tangent of the xgas and ygas terms.  The terms
-- are passed to the routine in such a way to produce an angle
-- that is relative to true north.  In the case of a VOT type
-- of facility, the bearing is set to 180 degrees with the
-- magnetic variation added.  This allows the VOR module to
-- operate in a normal manner without specific VOT code.
-- The bearing to station is computed by the equation:
--
--                          nxgas (nm.)
--     nbrgstn = arctan ( --------------- )
--                          nygas (nm.)
--
      Brg_To_Stn := arctan(YGAS, XGAS, 360.0 );

--*********************************************************************
--*># ILS AND MKR MODES
--*********************************************************************
--*>
--*> ILS and marker beacon facilities require additional information
--*> due to the facilities' orientation.  Common terms are computed
--*> first to optimize the program execution.  Note that the
--*> equations for rotation have been rearranged to take into
--*> account zero degrees is true north.
--*>

      Sinhdg := Sin(An_Instance.Station.Hdg,360.0);
      Coshdg := Cos(An_Instance.Station.Hdg,360.0);

--*********************************************************************
--*>## AGAS TO LOCALAIZER, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The down range term (nagas) is the distance from the aircraft
--*> to the facility measured along the axis of the facility
--*> heading.  Positive values indicate the aircraft is on the
--*> front course, with negative values indicate the aircraft is
--*> on the back course.  This term is in nautical miles, and is
--*> computed using the equation:
--*>
--*>     agas = ygas * sin(rwy hdg) + xgas * cos(rwy hdg)
--*>
--*>
      Agas := Ygas * Sinhdg + Xgas * Coshdg;

--*********************************************************************
--*>## BGAS TO LOCALIZER, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The cross range term (nbgas) is the distance from the
--*> aircraft to the facility centerline, measured perpendicular
--*> to the facility centerline.  Positive values indicate the
--*> aircraft is to the right of the facility centerline, and
--*> negative values indicate the aircraft is to the left of the
--*> facility centerline.  This term is in nautical miles, and
--*> is computed using the equation:
--*>
--*>     bgas = xgas * sin(rwy hdg) - ygas * cos(rwy hdg)
--*>
--*>
       Bgas := Xgas * Sinhdg - Ygas * Coshdg;

--*********************************************************************
--*>## AGAS TO TOUCHDOWN, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The distance to touchdown term (nagastd) is the distance
--*> from the aircraft to the glidepath position, measured along
--*> the facility centerline.  The term is computed as a function
--*> of the distance to the station (nagas) and the localizer to
--*> glidepath distance (nstnlgd).  The distance is in nautical
--*> miles, and is determined by the equation:
--*>
--*>     nagastd = nagas + loc to gs distance

       Agas_Td := Agas - An_Instance.Station.Lgd * Ft_To_Nm;

--*******************************************************************
--ANTENNA DISTANCES                                                 *
--*******************************************************************
--
-- Antenna distances are calculated here based on the location
-- of the CG sent by flight.  This is done so that the loc
-- deviations are accurate for the HGS.
--

       Localizer_Cg_Dist :=
         An_Instance.Center_Of_Gravity_X - Loc_Antenna_Dist;

--******************************************************************
-- STATION TYPE DECODE                                             *
--******************************************************************
--
-- The two station type variables, typ1 and typ2, contain information
-- about the vor/ils.
--

       Ils_Station :=  An_Instance.Station.Typ1(30) and   --ils station?
                       An_Instance.Station.Typ2(20);      --secondary loc
       Back_Course_Avail := An_Instance.Station.Typ2(29); --back course avail?

--
--******************************************************************
-- ILS MODE SELECTIONS                                             *
--******************************************************************
--
-- Checking for validity and mode selection.
--

       if An_Instance.Power and then An_Instance.Mode_On then
          An_Instance.Valid        := True;   -- Valid data.
          An_Instance.Audio_On     := True;   -- Audio available.
       else
          An_Instance.Valid        := False;  -- No valid data.
          An_Instance.Audio_On     := False;  -- Audio not avail.
          An_Instance.Receiving    := False;  -- Not receiving signal.
          An_Instance.VCA          := 0.0;    -- No VCA.
          An_Instance.Dev_Dots     := 0.0;    -- Center needle.
          An_Instance.To_From_Flag := 0.0;    -- Stow to/from flag.
       end if;


       if An_Instance.Valid then

--*******************************************************************
--# NAV RECEIVING CHECK                                             *
--*******************************************************************
--
-- A check is made to see if the aircraft is within line of sight
-- recieving distance for the ground station to be recieved.
-- The formula for calculating the aircrafts maxium distance
-- from the station at any given altitude above the station is:
--
--   DIST = 1.225 X SQUARE ROOT OF THE TRANSMITTING ANTENNA HEIGHT
--    MINUS THE GROUND ELEVATION PLUS THE HEIGHT OF THE A/C ABOVE
--    THE STATION.
--

          if Localizer_Range > RGAS then
             Line_Of_Sight := RU.Flimit((1.23*Sqrt(ZGAS)),20.0, 300.0);
             In_Range      := Line_Of_Sight > RGAS;

          else
             In_Range      := false;
             Line_Of_Sight := 0.0;

          end if;

--********************************************************************
--**># GLIDESLOPE AND LOCALIZER TESTS                                 *
--********************************************************************
--*>
--*> The ILS test function is initiated from the NAV control head.
--*> When activated, the test will put the VIR32 NAV receiver into a
--*> self-test mode with the appropriate NAV indicators displaying
--*> the test results.
--*>
--      if ( nitstmr(i) .gt. 0.0                   !vor test
--     >  .and. nvtst(i) ) then
--        nitstmr(i) = nitstmr(i) - iconst         !  count down timer
--        if ( nitstmr(i) .lt. 7.0 ) then          !  1st 3 sec = inval
--          nilflg(i) = .not.nlocfail(i)           !  loc valid
--          if ( .not.nlocfail(i) ) nildv(i) = 0.10 !* 2/3 scale down
--        end if                                   !
--        goto 800                                 !
--      elseif ( .not.nvtst(i) ) then              !test not set
--        nitstmr(i) = 0.0                         !  reset test timer
--      else                                       !no test in progress
--        if (nvtst(i) .and. .not.nvtstn1(i))      !  if test pushed
--     >    nitstmr(i) = 10.0                      !    start 10
--      end if                                     !

--      nvtstn1(i) = nvtst(i)                      !save for n-1 value
--*********************************************************************
--*># LOCALIZER VALIDITY                                              *
--*********************************************************************
--*>
--*> The Localizer becomes valid if the following conditions are met.
--*> The NAV station in tune is an ILS frequency, the station in tune
--*> is being received, there are no malfunctions, and the aircraft
--*> is downrange of the localizer antenna or the localizer has back-
--*> course availability.
--*>
          An_Instance.Receiving := Ils_Station and In_Range and
            (Agas > 0.0 or Back_Course_Avail);

--*********************************************************************
--*># LOCALIZER DEVIATION                                             *
--*********************************************************************
--*>
--*> Check for localizer valid and for backcourse availability
--*> Calculate the angle between the aircraft and the localizers
--*> center line. Calculate the amount of deviation from the
--*> localizer in dots, then scale and smooth the output for the
--*> instruments.
--*>
--*>     LOCALIZER         |       |       |       |       |
--*>     SENSITIVITY      2.5     1.25     X      1.25    2.5
--*>                                       X
--*>       VOR             |       |       |       |       |
--*>    SENSITIVITY       10       5       X       5       10
--*>                                       X
--*> For information purposes, the beam width for a new localizer can
--*> be determined as follows.  The localizer position, if unknown can
--*> usually be positioned from 500 to 1000 feet beyond the far end of
--*> the runway.  With this informaion, the localizer beam width is
--*> calculated such that full scale deflection (5 degrees) equates to
--*> 700 feet at the runway threshold.  For example: given a 8000 foot
--*> runway, localizer distance beyond the runway end is 1000 feet,
--*> and there is no displaced threshold.  Localizer beam width is:
--*>
--*>  ARCTAN = ( 0.5 * 700 ) / ( THRESHOLD TO LOC DIST ) * 2
--*>  ARCTAN = ( 350 / ( 8000 + 1000 ) ) * 2
--*>  ARCTAN = 4.45 DEGREES
--*>
          if An_Instance.Receiving then

            --Altitude fade calculation
            if ZGAS > 0.0 then
               Altitude_Fade := Max_Altitude / ZGAS;
            else
               Altitude_Fade := 1.0;
            end if;

            --The maximum range is the lesser of the station range or
            --the line of sight distance.
            Max_Range :=  An_Instance.Station.Rng;
            if Max_Range > Line_Of_Sight then
               Max_Range := Line_Of_Sight;
            end if;

            --Audio signal strength is function of altitude and
            --range to station.
            --Audio noise is inverse function of audio signal strength.
            Scaled_RGAS := RGAS / Max_Range;
            if Scaled_RGAS >= 1.0 then
               Scaled_RGAS := 1.0;
            elsif Scaled_RGAS <= -1.0 then
               Scaled_RGAS :=  -1.0;
            end if;

            Station_Strength := cos(Scaled_RGAS * 90.0, 360.0) * Altitude_Fade;
            if Station_Strength <= 0.05 then
               Station_Strength := 0.05;
            elsif Station_Strength >= 0.95 then
               Station_Strength := 0.95;
            end if;
            An_Instance.VCA := Station_Strength;

             Max_Agas := (An_Instance.Station.Lgd + 1200.0) * Ft_To_Nm;
             Tmp_agas := abs(Agas);
             if Tmp_Agas = 0.0 then Tmp_agas := 0.00001; end if;
--             if Tmp_agas < Max_Agas then Tmp_agas := Max_Agas; end if;

             Ac_To_Stn_Angle := An_Instance.Aircraft_True_Hdg -
               An_Instance.Station.Hdg;

             Tmp_Bgas := ((Sin(Ac_To_Stn_Angle,360.0)*Localizer_Cg_Dist)*
                          Ft_To_Nm) + Bgas;

             Ac_To_Loc_Delta := Arctan(Tmp_Bgas,Tmp_Agas,360.0);
             if Ac_To_Loc_Delta > 180.0 then
                Ac_To_Loc_Delta := Ac_To_Loc_Delta - 360.0;
             end if;

             if An_Instance.Station.Lbw > 0.0 and
               An_Instance.Reference_Airfield.Idnt /= "KTST" then
                Temp := Ac_To_Loc_Delta / (An_Instance.Station.Lbw/4.0);
             else
                Threshold_To_Rwy := An_Instance.Reference_Runway.Len -
                  An_Instance.Reference_Runway.Dtr + 1000.0;

                if Threshold_To_Rwy <= 0.0 then
                   Threshold_To_Rwy := 8000.0;
                end if;

                Loc_Beam_Width := 2.0 * Arctan(350.0,Threshold_To_Rwy,360.0);

                if Loc_Beam_Width /= 0.0 then
                   Temp := Ac_To_Loc_Delta / (Loc_Beam_Width/4.0);
                end if;
             end if;
             An_Instance.Dev_Dots := Ru.Flimit(Temp,-4.0,4.0);
             An_Instance.To_From_Flag := 0.0;    -- Stow to/from flag.
          else
             An_Instance.Dev_Dots := 0.0;
             An_Instance.To_From_Flag := 0.0;    -- Stow to/from flag.
          end if;
       end if;

   end Update_Localizer;

   end Localizer;
