-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-- DATA SOURCE:  USAF TO 1T-6A-1
--               Raytheon Wiring Diagrams-Avionics and Electrical Systems
--               Flight Safety Simulation Models
--
-------------------------------------------------------------------------------

with Jpats_Simulated_Aircraft,
  Jpats_Aircraft_Body,
  Jpats_Radio_Db_If,
  JPATS_Radio_Db_If_Types,
  Jpats_Radio.Ios_If,
  Jpats_Avionics,
  Jpats_Avionics_Types,
  Jpats_Electrical,
  Jpats_Electrical_Types,
  JPATS_Radio.Container,
  Jpats_Reposition,
  JPATS_Simphonics,
  Vhf_Nav,
  Radio.Audio.Dme,
  Radio.Elt,
  Radio_Types,
  Radio.Audio.Communication.Vhf_Comm,
  Radio.Audio.Communication.Uhf_Comm,
  Radio.Transponder,
  Vor,
  Ils,
  Localizer,
  Glideslope,
  Marker_Beacon,
  Radio_Utilities;
with Jpats_Io_Types, Jpats_Io, Io_Types;
with Arinc_429_Bcd,Arinc_429_Bnr,Arinc_429_Types;
with Ada.Text_IO,Log;
with Ada.Io_Exceptions;

use Jpats_Electrical_Types,Arinc_429_Types;

package body JPATS_Radio.Controller is

   package Types renames JPATS_Radio_Types;
   package Dme renames Radio.Audio.Dme;
   package Elt renames Radio.Elt;
   package Vhf_Comm renames Radio.Audio.Communication.Vhf_Comm;
   package Uhf_Comm renames Radio.Audio.Communication.Uhf_Comm;
   package Xpdr renames Radio.Transponder;
   package Sim_Ac renames Jpats_Simulated_Aircraft;
   package Ac_Body renames Jpats_Aircraft_Body;
   package Db renames Jpats_Radio_Db_If;
   package Db_Types renames Jpats_Radio_Db_If_Types;
   package Jiot renames Jpats_Io_Types;
   package Iot renames Io_Types;
   package Ru renames Radio_Utilities;
   package A429t renames Arinc_429_Types;
   package Audio renames Radio.Audio;

   Ios : Container.Ios_Interface_Instance renames Container.This_Ios_Interface;
   Io : Container.Io_Interface_Instance renames Container.This_Io_Interface;

   Station             : Db_Types.Stn_Struct_Type;
   Vhf_Station         : Db_Types.Stn_Struct_Type;
   Uhf_Station         : Db_Types.Stn_Struct_Type;
   Position            : Radio_Types.Position_Type := (45.0,-45.0,1000.0);
   Frequency_Changed   : Boolean := False;
   Nav_Frequency_N1    : Integer;
   Local_Vor           : Vor.Instance;
   Local_Ils           : Ils.Instance;
   Local_Localizer     : Localizer.Instance;
   Local_Glideslope    : Glideslope.Instance;
   Local_Mkr           : Marker_Beacon.Instance;
   True_Heading        : Radio_Types.Heading_Type := 270.0;
   Aircraft_Pitch      : Float := 0.0;
   Reference_Airfield  : Jpats_Radio_Db_If_Types.Apt_Struct_Type;
   Reference_Runway    : Jpats_Radio_Db_If_Types.Rwy_Struct_Type;
   Crash               : Boolean := False;
   Elt_Timer           : Float := 0.0;
   Elt_Flash           : Boolean := False;

   Debug               : Boolean := False;

   --types packages for arinc 429 pack/unpack
   type Spare_Type is new Integer range 0 .. 1;
   for Spare_Type'Size use 1;
   Spare : constant Spare_Type := 0;

   --ARINC 429 PACKAGES:
   type Discretes_Tx_Nav_034_Type is
      record
         Spare_11      : Spare_Type := Spare;
         Spare_12      : Spare_Type := Spare;
         Vor_Filtering : Boolean := False;
         Ils_Mode      : Boolean := False;
      end record;
   for Discretes_Tx_Nav_034_Type use
      record
         Spare_11      at 0 range 0 .. 0;
         Spare_12      at 0 range 1 .. 1;
         Vor_Filtering at 0 range 2 .. 2;
         Ils_Mode      at 0 range 3 .. 3;
      end record;
   for Discretes_Tx_Nav_034_Type'Size use 4;

   type Discretes_Tx_Nav_173_Type is
      record
         Tune_Inhibit : Boolean := False;
         Spare_12     : Spare_Type := Spare;
         Spare_13     : Spare_Type := Spare;
         Spare_14     : Spare_Type := Spare;
         Spare_15     : Spare_Type := Spare;
         Spare_16     : Spare_Type := Spare;
      end record;
   for Discretes_Tx_Nav_173_Type use
      record
         Tune_Inhibit at 0 range 0 .. 0;
         Spare_12     at 0 range 1 .. 1;
         Spare_13     at 0 range 2 .. 2;
         Spare_14     at 0 range 3 .. 3;
         Spare_15     at 0 range 4 .. 4;
         Spare_16     at 0 range 5 .. 5;
      end record;
   for Discretes_Tx_Nav_173_Type'Size use 6;

   type Discretes_Tx_Nav_174_Type is
      record
         Tune_Inhibit : Boolean := False;
         Spare_12     : Spare_Type := Spare;
         Spare_13     : Spare_Type := Spare;
         Spare_14     : Spare_Type := Spare;
         Spare_15     : Spare_Type := Spare;
         Spare_16     : Spare_Type := Spare;
      end record;
   for Discretes_Tx_Nav_174_Type use
      record
         Tune_Inhibit at 0 range 0 .. 0;
         Spare_12     at 0 range 1 .. 1;
         Spare_13     at 0 range 2 .. 2;
         Spare_14     at 0 range 3 .. 3;
         Spare_15     at 0 range 4 .. 4;
         Spare_16     at 0 range 5 .. 5;
      end record;
   for Discretes_Tx_Nav_174_Type'Size use 6;

   type Discretes_Tx_Nav_222_Type is
      record
         Outer_Marker_Beacon  : Boolean := False;
         Middle_Marker_Beacon : Boolean := False;
         Inner_Marker_Beacon  : Boolean := False;
         Spare_14             : Spare_Type := Spare;
         Spare_15             : Spare_Type := Spare;
         Spare_16             : Spare_Type := Spare;
      end record;
   for Discretes_Tx_Nav_222_Type use
      record
         Outer_Marker_Beacon  at 0 range 0 .. 0;
         Middle_Marker_Beacon at 0 range 1 .. 1;
         Inner_Marker_Beacon  at 0 range 2 .. 2;
         Spare_14             at 0 range 3 .. 3;
         Spare_15             at 0 range 4 .. 4;
         Spare_16             at 0 range 5 .. 5;
      end record;
   for Discretes_Tx_Nav_222_Type'Size use 6;

   type Dme_Mode_Type is
      (Standby,Directed_Freq_1,Directed_Freq_2,Directed_Freq_3,Hold_Freq_1,
       Hold_Freq_2,Free_Scan,None);
   for Dme_Mode_Type'Size use 3;
   type Nav_Mode_Type is (Vor_Mode,Ils_Mode,Mls_Mode,Fms_Mode);
   for Nav_Mode_Type'Size use 2;

   type Discretes_Tx_Dme_035_Type is
      record
         Dme_Mode       : Dme_Mode_Type := Standby;
         Nav_Mode       : Nav_Mode_Type := Vor_Mode;
         Display_Enable : Boolean := False;
         Ident_On       : Boolean := False;
         Mhzp05         : Boolean := False;
      end record;
   for Discretes_Tx_Dme_035_Type use
      record
         Dme_Mode       at 0 range 0 .. 2;
         Nav_Mode       at 0 range 3 .. 4;
         Display_Enable at 0 range 5 .. 5;
         Ident_On       at 0 range 6 .. 6;
         Mhzp05         at 0 range 7 .. 7;
      end record;
   for Discretes_Tx_Dme_035_Type'Size use 8;

   type Discretes_Tx_Dme_202_Type is
      record
         Memory_On  : Boolean := False;
         Foreground : Boolean := False;
      end record;
   for Discretes_Tx_Dme_202_Type use
      record
         Memory_On  at 0 range 0 .. 0;
         Foreground at 0 range 1 .. 1;
      end record;
   for Discretes_Tx_Dme_202_Type'Size use 2;

   Discretes_Tx_Nav_034 : Discretes_Tx_Nav_034_Type;
   Discretes_Tx_Nav_173 : Discretes_Tx_Nav_173_Type;
   Discretes_Tx_Nav_174 : Discretes_Tx_Nav_174_Type;
   Discretes_Tx_Nav_222 : Discretes_Tx_Nav_222_Type;
   Discretes_Tx_Dme_035 : Discretes_Tx_Dme_035_Type;
   Discretes_Tx_Dme_202 : Discretes_Tx_Dme_202_Type;
   No_Discretes         : A429t.No_Discretes_Type;
   Vor_Bnr_Ssm          : A429t.Bnr_Ssm_Type := A429t.No_Computed_Data;
   Mkr_Bnr_Ssm          : A429t.Bnr_Ssm_Type := A429t.No_Computed_Data;
   Loc_Bnr_Ssm          : A429t.Bnr_Ssm_Type := A429t.No_Computed_Data;
   Gs_Ssm               : A429t.Bnr_Ssm_Type := A429t.No_Computed_Data;
   Dme_Bcd_Ssm          : A429t.Bcd_Ssm_Type := A429t.No_Computed_Data;
   Dme_Bnr_Ssm          : A429t.Bnr_Ssm_Type := A429t.No_Computed_Data;


   --   VN411B VOR Label Definitions

   --  LABEL - 034   HEX ID - 11   VOR/ILS FREQUENCY   (BCD)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11-12    SPARES
   --  13       VOR DIGITAL OMNIBEARING FILTERING     No Filtering  Filtering
   --  14       ILS MODE        ILS Mode        VOR Mode
   --  15-18    HUNDREDTHS OF MHz
   --  19-22    TENTHS OF MHz
   --  23-26    UNITS OF MHz
   --  27-29    TENS OF MHz
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Nav_034 is new
     Arinc_429_Bcd(Label              => 8#034#,
                   Value_Type         => Integer,
                   Significant_Digits => 4,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bcd_Ssm_Type,
                   Discretes_Type     => Discretes_Tx_Nav_034_Type);

   --  LABEL - 173   HEX ID - 10   LOCALIZER DEVIATION (BNR)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11       TUNE INHIBIT    Inhibit Allow
   --  12-16    SPARES          Pad Zeros
   --  17-28    BINARY DDM (0.4/4,096)
   --  29       SIGN            Fly Left        Fly Right
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Nav_173 is new
     Arinc_429_Bnr(Label              => 8#173#,
                   Value_Type         => Float,
                   Scale              => 0.4,
                   Significant_Digits => 12,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bnr_Ssm_Type,
                   Discretes_Type     => Discretes_Tx_Nav_173_Type);

   --  LABEL - 174   HEX ID - 10   GLIDESLOPE DEVIATION (BNR)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11       TUNE INHIBIT    Inhibit Allow
   --  12-16    SPARES          Pad Zeros
   --  17-28    BINARY DDM (0.8/4,096)
   --  29       SIGN            Fly Up  Fly Down
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Nav_174 is new
     Arinc_429_Bnr(Label              => 8#174#,
                   Value_Type         => Float,
                   Scale              => 0.8,
                   Significant_Digits => 12,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bnr_Ssm_Type,
                   Discretes_Type     => Discretes_Tx_Nav_174_Type);

   --  LABEL - 222   HEX ID - 11   VOR OMNIBEARING (BNR)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11       OUTER MARKER BEACON 400HZ       Present Absent
   --  12       MIDDLE MARKER BEACON 1300 HZ    Present Absent
   --  13       INNER MARKER BEACON 3000 HZ     Present Absent
   --  14-16    SPARES          Pad Zeros
   --  17-28    BINARY ANGLE (180/4096)
   --  29       SIGN            180-360 Deg     0-180 Deg
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Nav_222 is new
     Arinc_429_Bnr(Label              => 8#222#,
                   Value_Type         => Float,
                   Scale              => 180.0,
                   Significant_Digits => 12,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bnr_Ssm_Type,
                   Discretes_Type     => Discretes_Tx_Nav_222_Type);

   --  6.2      DM441B DME Label Definitions

   --  LABEL - 012   HEX ID - 09   GROUND SPEED  (BCD)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11-14    SPARES                  Pad Zeros
   --  15-18    UNITS OF KNOTS
   --  19-22    TENS OF KNOTS
   --  23-26    HUNDREDS OF KNOTS
   --  27       THOUSANDS OF KNOTS
   --  28-29    SPARES          Pad Zeros
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Dme_012 is new
     Arinc_429_Bcd(Label              => 8#012#,
                   Value_Type         => Integer,
                   Significant_Digits => 4,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bcd_Ssm_Type,
                   Discretes_Type     => Arinc_429_Types.No_Discretes_Type);


   --  LABEL - 035   HEX ID - 09   DME FREQUENCY  (BCD)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11-13    DME MODE
   --    13(0) & 12(0) & 11(0) - Standby
   --    13(0) & 12(0) & 11(1) - Directed Freq. 1
   --    13(0) & 12(1) & 11(0) - Directed Freq. 2
   --    13(0) & 12(1) & 11(1) - Directed Freq. 3
   --    13(1) & 12(0) & 11(0) - Hold Freq. 1
   --    13(1) & 12(0) & 11(1) - Hold Freq. 2
   --    13(1) & 12(1) & 11(0) - Free Scan (Not Used)
   --    13(1) & 12(1) & 11(1) - Spare
   --  14-15    NAV MODE
   --    15(0) & 14(0) - VOR
   --    15(0) & 14(1) - ILS
   --    15(1) & 14(0) - MLS
   --    15(1) & 14(1) - FMS
   --  16       DISPLAY Enable  Disable
   --  17       IDENT   Audio On        Audio Off
   --  18       0.05 MHz        0.05 MHz        0.00 MHz
   --  19-22    TENTHS OF MHz
   --  23-26    UNITS OF MHz
   --  27-29    TENS OF MHz
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Dme_035 is new
     Arinc_429_Bcd(Label              => 8#035#,
                   Value_Type         => Integer,
                   Significant_Digits => 3,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bcd_Ssm_Type,
                   Discretes_Type     => Discretes_Tx_Dme_035_Type);

   --  LABEL - 202   HEX ID - 09   DME DISTANCE  (BNR)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11       MEMORY  Memory On       Memory Off
   --  12       FOREGROUND      Foreground      Not Foreground
   --  13-28    BINARY NAUT. MI (512/65536)
   --  29       SIGN                    Always Zero
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Dme_202 is new
     Arinc_429_Bnr(Label              => 8#202#,
                   Value_Type         => Float,
                   Scale              => 512.0,
                   Significant_Digits => 16,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bnr_Ssm_Type,
                   Discretes_Type     => Discretes_Tx_Dme_202_Type);

   --  LABEL - 252   HEX ID - 09   TIME TO GO  (BNR)
   --                           BIT STATUS
   --  BIT NO.  FUNCTION        ONE     ZERO
   --  01-08    LABEL
   --  09-10    SDI
   --  11-19    SPARES          Pad Zeros
   --  20-28    BINARY MINUTES (512/512)
   --  29       SIGN
   --  30-31    SSM
   --  32       PARITY (ODD)
   package Tx_Dme_252 is new
     Arinc_429_Bnr(Label              => 8#252#,
                   Value_Type         => Float,
                   Scale              => 512.0,
                   Significant_Digits => 9,
                   Sdi_Type           => Arinc_429_Types.Sdi_Type,
                   Ssm_Type           => Arinc_429_Types.Bnr_Ssm_Type,
                   Discretes_Type     => Arinc_429_Types.No_Discretes_Type);

   procedure Register_Io_Variables is
   begin

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "ELT_XMT_LT",
         A_Direction     => Iot.Output,
         An_Address      => Io.Elt_Xmt_Lt'Address,
         A_Size          => Io.Elt_Xmt_Lt'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "ELT_ARM_SW",
         A_Direction     => Iot.Input,
         An_Address      => Io.Elt_Arm_Sw'Address,
         A_Size          => Io.Elt_Arm_Sw'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_vor_l034",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Nav_034'Address,
         A_Size          => Io.Tx_Nav_034'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_vor_l173",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Nav_173'Address,
         A_Size          => Io.Tx_Nav_173'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_vor_l174",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Nav_174'Address,
         A_Size          => Io.Tx_Nav_174'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_vor_l222",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Nav_222'Address,
         A_Size          => Io.Tx_Nav_222'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_dme_l252",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Dme_252'Address,
         A_Size          => Io.Tx_Dme_252'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_dme_l012",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Dme_012'Address,
         A_Size          => Io.Tx_Dme_012'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_dme_l035",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Dme_035'Address,
         A_Size          => Io.Tx_Dme_035'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "aico_a429_dme_l202",
         A_Direction     => Iot.Output,
         An_Address      => Io.Tx_Dme_202'Address,
         A_Size          => Io.Tx_Dme_202'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "AUDIO_MKR_HI",
         A_Direction     => Iot.Input,
         An_Address      => Io.Mkr_Hi'Address,
         A_Size          => Io.Mkr_Hi'Size);

   end;

   procedure Initialize is
   begin

      --local labels

      --register io variables:
      Register_Io_Variables;

      --register Ios variables:
      JPATS_Radio.Ios_If.Register_Ios_Variables;

      --Initialize VHF Nav attributes:
      VHF_Nav.Set_Power(Container.This_Subsystem.The_VHF_Nav,false);
      VHF_Nav.Set_Mode_On(Container.This_Subsystem.The_VHF_Nav,false);
      VHF_Nav.Set_Aircraft_Position(Container.This_Subsystem.The_VHF_Nav,
                                    (Latitude  => 0.0,
                                     Longitude => 0.0,
                                     Altitude  => 0.0));
      VHF_Nav.Set_Ground_Elevation(Container.This_Subsystem.The_VHF_Nav,0.0);
      VHF_Nav.Set_Valid(Container.This_Subsystem.The_VHF_Nav,false);

      VHF_Nav.Set_VCA(Container.This_Subsystem.The_VHF_Nav,0.0);
      VHF_Nav.Set_VCN(Container.This_Subsystem.The_VHF_Nav,0.0);
      VHF_Nav.Set_Ident(Container.This_Subsystem.The_VHF_Nav,"    ");
      VHF_Nav.Set_Receiving(Container.This_Subsystem.The_VHF_Nav,
                            false);
      VHF_Nav.Set_Test(Container.This_Subsystem.The_VHF_Nav,false);
      VHF_Nav.Set_Audio_On(Container.This_Subsystem.The_VHF_Nav,false);
      VHF_Nav.Set_Dev_Dots(Container.This_Subsystem.The_VHF_Nav,0.0);
      VHF_Nav.Set_Frequency_Changed(Container.This_Subsystem.The_VHF_Nav,
                                    false);
      VHF_Nav.Init(Container.This_Subsystem.The_VHF_Nav);

      --Initialize DME attributes
      DME.Set_Power(Container.This_Subsystem.The_DME,false);
      DME.Set_Mode_On(Container.This_Subsystem.The_DME,false);
      DME.Set_Aircraft_Position(Container.This_Subsystem.The_DME,
                                (Latitude  => 0.0,
                                 Longitude => 0.0,
                                 Altitude  => 0.0));
      DME.Set_Ground_Elevation(Container.This_Subsystem.The_DME,0.0);
      DME.Set_Valid(Container.This_Subsystem.The_DME,false);

      DME.Set_VCA(Container.This_Subsystem.The_DME,0.0);
      DME.Set_VCN(Container.This_Subsystem.The_DME,0.0);
      DME.Set_Ident(Container.This_Subsystem.The_DME,"    ");
      DME.Set_Receiving(Container.This_Subsystem.The_DME,false);
      DME.Set_Test(Container.This_Subsystem.The_DME,false);
      DME.Set_Audio_On(Container.This_Subsystem.The_DME,false);
      DME.Set_Dev_Dots(Container.This_Subsystem.The_DME,0.0);
      DME.Set_Frequency_Changed(Container.This_Subsystem.The_DME,false);
      DME.Init(Container.This_Subsystem.The_DME);

      --initialize ELT attributes
      ELT.Set_Power(Container.This_Subsystem.The_ELT,false);
      ELT.Set_Mode_On(Container.This_Subsystem.The_ELT,false);
      ELT.Set_Aircraft_Position(Container.This_Subsystem.The_ELT,
                                (Latitude  => 0.0,
                                 Longitude => 0.0,
                                 Altitude  => 0.0));
      ELT.Set_Ground_Elevation(Container.This_Subsystem.The_ELT,0.0);
      ELT.Set_Valid(Container.This_Subsystem.The_ELT,false);
      ELT.Init(Container.This_Subsystem.The_ELT);

      --Continue with transponder, uhf & vhf comm

      --Initialize Transponder attributes
      Xpdr.Set_Power(Container.This_Subsystem.The_Transponder,false);
      Xpdr.Set_Mode_On(Container.This_Subsystem.The_Transponder,false);
      Xpdr.Set_Aircraft_Position(Container.This_Subsystem.The_Transponder,
                                (Latitude  => 0.0,
                                 Longitude => 0.0,
                                 Altitude  => 0.0));
      Xpdr.Set_Ground_Elevation(Container.This_Subsystem.The_Transponder,0.0);
      Xpdr.Set_Valid(Container.This_Subsystem.The_Transponder,false);
      Xpdr.Init(Container.This_Subsystem.The_Transponder);

      --Initialize VHF Comm attributes
      Vhf_Comm.Set_Power(Container.This_Subsystem.The_VHF_Comm,false);
      VHF_Comm.Set_Mode_On(Container.This_Subsystem.The_VHF_Comm,false);
      VHF_Comm.Set_Aircraft_Position(Container.This_Subsystem.The_VHF_Comm,
                                     (Latitude  => 0.0,
                                      Longitude => 0.0,
                                      Altitude  => 0.0));
      VHF_Comm.Set_Ground_Elevation(Container.This_Subsystem.The_VHF_Comm,0.0);
      VHF_Comm.Set_Valid(Container.This_Subsystem.The_VHF_Comm,false);

      VHF_Comm.Set_VCA(Container.This_Subsystem.The_VHF_Comm,0.0);
      VHF_Comm.Set_VCN(Container.This_Subsystem.The_VHF_Comm,0.0);
      VHF_Comm.Set_Ident(Container.This_Subsystem.The_VHF_Comm,"    ");
      VHF_Comm.Set_Receiving(Container.This_Subsystem.The_VHF_Comm,
                             false);
      VHF_Comm.Set_Test(Container.This_Subsystem.The_VHF_Comm,false);
      VHF_Comm.Set_Audio_On(Container.This_Subsystem.The_VHF_Comm,false);
      VHF_Comm.Set_Dev_Dots(Container.This_Subsystem.The_VHF_Comm,0.0);
      VHF_Comm.Set_Frequency_Changed(Container.This_Subsystem.The_VHF_Comm,
                                     false);
      VHF_Comm.Init(Container.This_Subsystem.The_VHF_Comm);

      --Initialize UHF Comm attributes
      Uhf_Comm.Set_Power(Container.This_Subsystem.The_UHF_Comm,false);
      UHF_Comm.Set_Mode_On(Container.This_Subsystem.The_UHF_Comm,
                           JPATS_Avionics.Uhf_Tuning.Mode_On);
      UHF_Comm.Set_Aircraft_Position(Container.This_Subsystem.The_UHF_Comm,
                                     (Latitude  => 0.0,
                                      Longitude => 0.0,
                                      Altitude  => 0.0));
      UHF_Comm.Set_Ground_Elevation(Container.This_Subsystem.The_UHF_Comm,0.0);
      UHF_Comm.Set_Valid(Container.This_Subsystem.The_UHF_Comm,false);

      UHF_Comm.Set_VCA(Container.This_Subsystem.The_UHF_Comm,0.0);
      UHF_Comm.Set_VCN(Container.This_Subsystem.The_UHF_Comm,0.0);
      UHF_Comm.Set_Ident(Container.This_Subsystem.The_UHF_Comm,"    ");
      UHF_Comm.Set_Receiving(Container.This_Subsystem.The_UHF_Comm,
                             false);
      UHF_Comm.Set_Test(Container.This_Subsystem.The_UHF_Comm,false);
      UHF_Comm.Set_Audio_On(Container.This_Subsystem.The_UHF_Comm,false);
      UHF_Comm.Set_Dev_Dots(Container.This_Subsystem.The_UHF_Comm,0.0);
      UHF_Comm.Set_Frequency_Changed(Container.This_Subsystem.The_UHF_Comm,
                                     false);
      UHF_Comm.Init(Container.This_Subsystem.The_UHF_Comm);

   end Initialize;

   procedure Update
     (Integration_Constant : in Float) is
   begin

      -- Debug := JPATS_Avionics.Nacws_First_Detent;

      --get aircraft position from simulated aircraft
      position.latitude  := Sim_Ac.Get_North;
      position.longitude := Sim_Ac.Get_East;
      position.altitude  := Sim_Ac.Get_Aircraft_Geometric_Altitude;
      True_Heading       := Sim_Ac.Get_Hdg_Angle * 57.3;
      Aircraft_Pitch     := Sim_Ac.Get_Pitch_Angle * 57.3;

--VHF NAV section---------------------------------------------------------
      --get station vhf nav station data
      Station := Jpats_Radio_Db_If.Nav_Station;

      Frequency_Changed :=
        JPATS_Avionics.Nav_Tuning.Frequency /= Nav_Frequency_N1;
      Nav_Frequency_N1 :=
        JPATS_Avionics.Nav_Tuning.Frequency;

      --Inputs to VHF Nav from other categories
      VHF_Nav.Set_Power(Container.This_Subsystem.The_VHF_Nav,
                        Jpats_Electrical.Is_Powered(Vhf_Nav_Cb));
      VHF_Nav.Set_Mode_On(Container.This_Subsystem.The_VHF_Nav,true);
      VHF_Nav.Set_Aircraft_Position(Container.This_Subsystem.The_VHF_Nav,
                                    position);
      VHF_Nav.Set_Station(Container.This_Subsystem.The_VHF_Nav,Station);
      Vhf_Nav.Set_Frequency_Changed(Container.This_Subsystem.The_VHF_Nav,
                                    Frequency_Changed);
      VHF_Nav.Set_ILS_Freq_Tuned(Container.This_Subsystem.The_VHF_Nav,
         Container.This_Subsystem.The_Nav_Status.Ils_Freq_Selected);
      VHF_Nav.Set_Aircraft_True_Hdg(Container.This_Subsystem.The_VHF_Nav,
                                    True_Heading);
      VHF_Nav.Set_Course_Selected(Container.This_Subsystem.The_VHF_Nav,
                                  Jpats_Avionics.Course_Selected);
      Vhf_Nav.Set_High_Sense(Container.This_Subsystem.The_VHF_Nav,Io.Mkr_Hi);
      VHF_Nav.Set_Marker_Station(Container.This_Subsystem.The_VHF_Nav,
                                 Jpats_Radio_Db_If.Marker_Station);
      VHF_Nav.Set_Aircraft_Pitch(Container.This_Subsystem.The_VHF_Nav,
                                 Aircraft_Pitch);
      VHF_Nav.Set_Reference_Airfield(Container.This_Subsystem.The_VHF_Nav,
                                     Jpats_Reposition.Reference_Airfield);
      VHF_Nav.Set_Reference_Runway(Container.This_Subsystem.The_VHF_Nav,
                                   Jpats_Reposition.Reference_Runway);

      --VHF Nav update routine
      VHF_Nav.Update(Integration_Constant,
                     Container.This_Subsystem.The_VHF_Nav);

      --Set Nav Status to simphonics
      Container.This_Subsystem.The_Nav_Status.Ident := Station.Idnt;
      Container.This_Subsystem.The_Nav_Status.Atis := Station.Typ2(25);
      Container.This_Subsystem.The_Nav_Status.Position.Latitude :=
        Station.Lat;
      Container.This_Subsystem.The_Nav_Status.Position.Longitude :=
        Station.Lon;
      Container.This_Subsystem.The_Nav_Status.Frequency :=
        Station.Frq;
      Container.This_Subsystem.The_Nav_Status.Typ1 := Station.Typ1;

      --Outputs from vor
      Local_Vor := Vhf_Nav.The_Vor(Container.This_Subsystem.The_VHF_Nav);

      --Outputs from ILS--
      Local_Ils := Vhf_Nav.The_Ils(Container.This_Subsystem.The_VHF_Nav);
      Local_Localizer := Ils.The_Localizer(Local_Ils);
      Local_Glideslope := Ils.The_Glideslope(Local_Ils);

      --Outputs from Marker Beacon
      Local_Mkr := Vhf_Nav.The_Marker_Beacon
        (Container.This_Subsystem.The_Vhf_Nav);

      --audio output to simphonics
      --vor
      if Container.This_Subsystem.The_Nav_Status.Ils_Freq_Selected then
         --ils
         if Localizer.Audio_On(Local_Localizer) then
            Container.This_Subsystem.The_Nav_Status.Vca :=
              Localizer.VCA(Local_Localizer);
         else
            Container.This_Subsystem.The_Nav_Status.Vca := 0.0;
         end if;
      else
         --vor
         if Vor.Audio_On(Local_Vor) then
            Container.This_Subsystem.The_Nav_Status.Vca := Vor.VCA(Local_Vor);
         else
            Container.This_Subsystem.The_Nav_Status.Vca := 0.0;
         end if;
      end if;

       --marker beacon
       if Marker_Beacon.Inner_Receiving(Local_Mkr) then
          Container.This_Subsystem.The_Nav_Status.Im_Vca :=
            Marker_Beacon.Inner_Vca(Local_Mkr);
       else
          Container.This_Subsystem.The_Nav_Status.Im_Vca := 0.0;
       end if;

       if Marker_Beacon.Middle_Receiving(Local_Mkr) then
          Container.This_Subsystem.The_Nav_Status.Mm_Vca :=
            Marker_Beacon.Middle_Vca(Local_Mkr);
       else
          Container.This_Subsystem.The_Nav_Status.Mm_Vca := 0.0;
       end if;

       if Marker_Beacon.Outer_Receiving(Local_Mkr) then
          Container.This_Subsystem.The_Nav_Status.Om_Vca :=
            Marker_Beacon.Outer_Vca(Local_Mkr);
       else
          Container.This_Subsystem.The_Nav_Status.Om_Vca := 0.0;
       end if;

       --Output vhf nav 429 information
       if Vhf_Nav.Power(Container.This_Subsystem.The_Vhf_Nav) then

          Container.This_Subsystem.The_Nav_Status.Nav_Power_Fail := False;

          Discretes_Tx_Nav_034.Ils_Mode :=
             Jpats_Avionics.Nav_Tuning.Ils_Frequency_Selected;

          if Discretes_Tx_Nav_034.Ils_Mode then
             Vor_Bnr_Ssm := Arinc_429_Types.No_Computed_Data;
             Mkr_Bnr_Ssm := Arinc_429_Types.Normal;

             if Localizer.Receiving(Local_Localizer) then
                Loc_Bnr_Ssm := Arinc_429_Types.Normal;
             else
                Loc_Bnr_Ssm := Arinc_429_Types.No_Computed_Data;
             end if;
             if Glideslope.Receiving(Local_Glideslope) then
                if Ios.Gs_Fail then
                   Gs_Ssm := Arinc_429_Types.Failure;
                else
                   Gs_Ssm := Arinc_429_Types.Normal;
                end if;
             else
                Gs_Ssm := Arinc_429_Types.No_Computed_Data;
             end if;
          else
             Loc_Bnr_Ssm := Arinc_429_Types.No_Computed_Data;
             Gs_Ssm      := Arinc_429_Types.No_Computed_Data;
             Mkr_Bnr_Ssm := Arinc_429_Types.No_Computed_Data;

             if Vor.Receiving(Local_Vor) then
                Vor_Bnr_Ssm := Arinc_429_Types.Normal;
             else
                Vor_Bnr_Ssm := Arinc_429_Types.No_Computed_Data;
             end if;
          end if;
       else
          Container.This_Subsystem.The_Nav_Status.Nav_Power_Fail := True;
          Discretes_Tx_Nav_034.Ils_Mode := False;
          Vor_Bnr_Ssm := Arinc_429_Types.Failure;
          Loc_Bnr_Ssm := Arinc_429_Types.Failure;
          Gs_Ssm      := Arinc_429_Types.Failure;
          Mkr_Bnr_Ssm := Arinc_429_Types.Failure;
       end if;

       Container.This_Subsystem.The_Nav_Status.Ils_Freq_Selected :=
          Discretes_Tx_Nav_034.Ils_Mode;
       Container.This_Subsystem.The_Nav_Status.Valid :=
         (Vor_Bnr_Ssm = Arinc_429_Types.Normal);
       Container.This_Subsystem.The_Nav_Status.Vor_Ncd :=
         (Vor_Bnr_Ssm = Arinc_429_Types.No_Computed_Data);

       Discretes_Tx_Nav_034.Vor_Filtering := False;
      -- LABEL - 034   HEX ID - 11   VOR/ILS FREQUENCY   (BCD)
      if Jpats_Avionics.Nav_Tuning.Frequency >= 10800 then

         Io.Tx_Nav_034 := Tx_Nav_034.Pack
           (A_Value           =>
              Ru.Flimit(Jpats_Avionics.Nav_Tuning.Frequency - 10_000,0,9999),
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => A429t.Positive,
            A_Discretes_Value => Discretes_Tx_Nav_034);
      else
         Io.Tx_Nav_034 := Tx_Nav_034.Pack
           (A_Value           =>
              Ru.Flimit(Jpats_Avionics.Nav_Tuning.Frequency - 10_000,0,9999),
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => A429t.No_Computed_Data,
            A_Discretes_Value => Discretes_Tx_Nav_034);
      end if;

      Discretes_Tx_Nav_173.Tune_Inhibit := False;
      --LABEL - 173   HEX ID - 10   LOCALIZER DEVIATION (BNR)
      Io.Tx_Nav_173 := Tx_Nav_173.Pack
        (A_Value           =>
           --convert to DDM:  0.155DDM = 2.0 dots
           Localizer.Dev_Dots(Local_Localizer)* (-0.155)/2.0,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Loc_Bnr_Ssm,
         A_Discretes_Value => Discretes_Tx_Nav_173);
      Container.This_Subsystem.The_Nav_Status.Localizer_Valid :=
        (Loc_Bnr_Ssm = Arinc_429_Types.Normal);
      Container.This_Subsystem.The_Nav_Status.Localizer_Deviation :=
        -Localizer.Dev_Dots(Local_Localizer);
      Container.This_Subsystem.The_Nav_Status.Vor_Deviation :=
        Vor.Dev_Dots(Local_Vor);
      Container.This_Subsystem.The_Nav_Status.To_From_Flag :=
        VHF_Nav.To_From_Flag(Container.This_Subsystem.The_VHF_Nav);

      Discretes_Tx_Nav_174.Tune_Inhibit := False;
      --LABEL - 174   HEX ID - 10   GLIDESLOPE DEVIATION (BNR)
      Io.Tx_Nav_174 := Tx_Nav_174.Pack
        (A_Value           =>
           --convert to DDM:  0.175DDM = 2.0 dots
           Glideslope.Dev_Dots(Local_Glideslope)*0.175/2.0,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Gs_Ssm,
         A_Discretes_Value => Discretes_Tx_Nav_174);
      Container.This_Subsystem.The_Nav_Status.Glideslope_Valid :=
        (Gs_Ssm = Arinc_429_Types.Normal);
      Container.This_Subsystem.The_Nav_Status.Glideslope_Deviation :=
        -Glideslope.Dev_Dots(Local_Glideslope);
      Container.This_Subsystem.The_Nav_Status.Glideslope_Rec_Fail :=
        (Gs_Ssm = Arinc_429_Types.Failure);

      --Set Marker receiving bits for label 222
      Discretes_Tx_Nav_222.Outer_Marker_Beacon :=
        Marker_Beacon.Outer_Receiving(Local_Mkr);

      Discretes_Tx_Nav_222.Middle_Marker_Beacon :=
        Marker_Beacon.Middle_Receiving(Local_Mkr)
        and Marker_Beacon.Middle_Vca(Local_Mkr)
            > Marker_Beacon.Inner_Vca(Local_Mkr);

      Discretes_Tx_Nav_222.Inner_Marker_Beacon :=
        Marker_Beacon.Inner_Receiving(Local_Mkr)
        and Marker_Beacon.Inner_Vca(Local_Mkr)
            > Marker_Beacon.Middle_Vca(Local_Mkr);

      -- Test Case, set all inputs
      if JPATS_Simphonics.Marker_Test then
         Discretes_Tx_Nav_222.Outer_Marker_Beacon := True;
         Discretes_Tx_Nav_222.Middle_Marker_Beacon := True;
         Discretes_Tx_Nav_222.Inner_Marker_Beacon := True;
         Vor_Bnr_Ssm := Arinc_429_Types.Normal;
         Loc_Bnr_Ssm := Arinc_429_Types.Normal;
      end if;

      -- LABEL - 222   HEX ID - 11   VOR OMNIBEARING (BNR)
      Io.Tx_Nav_222 := Tx_Nav_222.Pack
        (A_Value           => Ru.Xn180(Vor.Mag_Brg_Deg(Local_Vor) + 180.0),
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Vor_Bnr_Ssm,
         A_Discretes_Value => Discretes_Tx_Nav_222);
      Container.This_Subsystem.The_Nav_Status.Bearing :=
        Vor.Mag_Brg_Deg(Local_Vor);


--DME section---------------------------------------------------------------

      --obatain dme station data.
      Station := Jpats_Radio_Db_If.Dme_Station;

      --Inputs to DME from other categories
      DME.Set_Power(Container.This_Subsystem.The_DME,
                    Jpats_Electrical.Is_Powered(Dme_Cb));
      DME.Set_Aircraft_Position(Container.This_Subsystem.The_DME,position);
      DME.Set_Station(Container.This_Subsystem.The_DME,Station);
      Dme.Set_ILS_Freq_Tuned(Container.This_Subsystem.The_Dme,
                             Container.This_Subsystem.The_Nav_Status.Ils_Freq_Selected);
      Dme.Set_Frequency_Hold_Selected(Container.This_Subsystem.The_DME,
                                      Jpats_Avionics.Dme_Tuning.Hold);

      --DME update routine
      DME.Update(Integration_Constant,Container.This_Subsystem.The_DME);

      --DME audio output to simphonics
      Container.This_Subsystem.The_Dme_Status.Ident := Station.Idnt;

      if DME.Audio_On(Container.This_Subsystem.The_DME) then
         Container.This_Subsystem.The_Dme_Status.Vca :=
           Dme.Vca(Container.This_Subsystem.The_Dme);
      else
         Container.This_Subsystem.The_Dme_Status.Vca := 0.0;
      end if;

      --Dme receiver arinc 429 outputs
      if DME.Valid(Container.This_Subsystem.The_DME) then

         if DME.Receiving(Container.This_Subsystem.The_DME) then
            Dme_Bcd_Ssm := Arinc_429_Types.Positive;
            Dme_Bnr_Ssm := Arinc_429_Types.Normal;
            Container.This_Subsystem.The_Dme_Status.Valid := True;
         else
            Container.This_Subsystem.The_Dme_Status.Valid := False;
            Dme_Bcd_Ssm := Arinc_429_Types.No_Computed_Data;
            Dme_Bnr_Ssm := Arinc_429_Types.No_Computed_Data;
         end if;
      else
         Container.This_Subsystem.The_Dme_Status.Valid := False;
         Dme_Bcd_Ssm := Arinc_429_Types.No_Computed_Data;
         Dme_Bnr_Ssm := Arinc_429_Types.No_Computed_Data;
      end if;

      -- LABEL - 012   HEX ID - 09   GROUND SPEED  (BCD)
      Container.This_Subsystem.The_Dme_Status.Groundspeed :=
        DME.Knots_Calculated(Container.This_Subsystem.The_DME);
      Io.Tx_Dme_012 := Tx_Dme_012.Pack
        (A_Value           =>
           Integer(DME.Knots_Calculated(Container.This_Subsystem.The_DME)),
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Dme_Bcd_Ssm,
         A_Discretes_Value => No_Discretes);

      if Jpats_Avionics.Dme_Tuning.Hold then
         Discretes_Tx_Dme_035.Dme_Mode := Hold_Freq_1;
      else
         Discretes_Tx_Dme_035.Dme_Mode := Directed_Freq_1;
      end if;

      if Container.This_Subsystem.The_Nav_Status.Ils_Freq_Selected then
         Discretes_Tx_Dme_035.Nav_Mode := Ils_Mode;
      else
         Discretes_Tx_Dme_035.Nav_Mode := Vor_Mode;
      end if;

      Discretes_Tx_Dme_035.Display_Enable := True;

      Discretes_Tx_Dme_035.Ident_On :=
        DME.Audio_On(Container.This_Subsystem.The_DME);

      Discretes_Tx_Dme_035.Mhzp05 :=
        Jpats_Avionics.Dme_Tuning.Frequency rem 10 > 0;

      -- LABEL - 035   HEX ID - 09   DME FREQUENCY  (BCD)
      Io.Tx_Dme_035 := Tx_Dme_035.Pack
        (A_Value           =>
           Ru.Flimit(Jpats_Avionics.Dme_Tuning.Frequency/10 - 1000,80,179),
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Dme_Bcd_Ssm,
         A_Discretes_Value => Discretes_Tx_Dme_035);

      Discretes_Tx_Dme_202.Memory_On := False;
      Discretes_Tx_Dme_202.Foreground := False;

      -- LABEL - 202   HEX ID - 09   DME DISTANCE  (BNR)
      Container.This_Subsystem.The_Dme_Status.Distance :=
        DME.Range_To_Station(Container.This_Subsystem.The_DME);
      Io.Tx_Dme_202 := Tx_Dme_202.Pack
        (A_Value           =>
           DME.Range_To_Station(Container.This_Subsystem.The_DME),
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Dme_Bnr_Ssm,
         A_Discretes_Value => Discretes_Tx_Dme_202);

      --  LABEL - 252   HEX ID - 09   TIME TO GO  (BNR)
      Container.This_Subsystem.The_Dme_Status.Ttg :=
        DME.Tts_Calculated(Container.This_Subsystem.The_DME);
      Io.Tx_Dme_252 := Tx_Dme_252.Pack
        (A_Value           =>
           DME.Tts_Calculated(Container.This_Subsystem.The_DME),
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Dme_Bnr_Ssm,
         A_Discretes_Value => No_Discretes);

--ELT section---------------------------------------------------------------

      --Inputs to ELT from other categories
      ELT.Set_Power(Container.This_Subsystem.The_ELT,
                    Jpats_Electrical.Is_Powered(Elt_Cb));
      ELT.Set_Mode_On(Container.This_Subsystem.The_ELT,true);
      ELT.Set_Aircraft_Position(Container.This_Subsystem.The_ELT,position);
      ELT.Set_Ground_Elevation(Container.This_Subsystem.The_ELT,0.0);
      ELT.Set_Arm_Sw(Container.This_Subsystem.The_ELT,Io.Elt_Arm_Sw);
      ELT.Set_On_Sw(Container.This_Subsystem.The_ELT,not Io.Elt_Arm_Sw);
      Crash := Sim_Ac.Get_Crash_Mountain_Impact;

      ELT.Set_Crash(Container.This_Subsystem.The_ELT,Crash);

      --ELT update routine
      ELT.Update(Integration_Constant,
                 Container.This_Subsystem.The_ELT);

      --Outputs from ELT
      Elt_Timer := Elt_Timer + Integration_Constant;
      if Elt_Timer > 1.0 then Elt_Timer := 0.0; end if;
      Elt_Flash := Elt_Timer > 0.5;

      Io.Elt_Xmt_Lt := ELT.Transmit(Container.This_Subsystem.The_ELT)
        and Elt_Flash;
      Ios.Elt_Arm_Sw := Io.Elt_Arm_Sw;

--Transponder section--------------------------------------------------------

      --Inputs to Transponder from other categories
      Xpdr.Set_Power(Container.This_Subsystem.The_Transponder,
                     Jpats_Electrical.Is_Powered(Xpdr_Cb));
      Xpdr.Set_Aircraft_Position(Container.This_Subsystem.The_Transponder,
                                    position);
      Xpdr.Set_Ground_Elevation(Container.This_Subsystem.The_Transponder,0.0);
      Xpdr.Set_Mode_Sel(Container.This_Subsystem.The_Transponder,
                        Jpats_Avionics.Atc_Tuning.Mode);
      Xpdr.Set_Ident_Sel(Container.This_Subsystem.The_Transponder,
                         Jpats_Avionics.Atc_Tuning.Ident);
      Xpdr.Set_Test_Sel(Container.This_Subsystem.The_Transponder,
                        Jpats_Avionics.Atc_Tuning.Test);

      --Transponder update routine
      Xpdr.Update(Integration_Constant,
                  Container.This_Subsystem.The_Transponder);

      --Outputs from Transponder
      Xpdr.Set_Valid(Container.This_Subsystem.The_Transponder,
         Xpdr.Valid(Container.This_Subsystem.The_Transponder));

--VHF Communication section--------------------------------------------------

      Vhf_Station := Jpats_Radio_Db_If.Vhf_Comm_Station;

      --Inputs to VHF Nav from other categories
      VHF_Comm.Set_Power(Container.This_Subsystem.The_VHF_Comm,
                         Jpats_Electrical.Is_Powered(Vhf_Comm_Cb));
      VHF_Comm.Set_Aircraft_Position(Container.This_Subsystem.The_VHF_Comm,
                                    position);
      VHF_Comm.Set_Ground_Elevation(Container.This_Subsystem.The_VHF_Comm,0.0);
      VHF_Comm.Set_Station(Container.This_Subsystem.The_VHF_Comm,Vhf_Station);
      VHF_Comm.Set_Frequency(Container.This_Subsystem.The_VHF_Comm,
                             Jpats_Avionics.Vhf_Tuning.Frequency);
      VHF_Comm.Set_Squelch(Container.This_Subsystem.The_VHF_Comm,
                             Jpats_Avionics.Vhf_Tuning.Squelch);
      VHF_Comm.Set_Elt_Transmit(Container.This_Subsystem.The_VHF_Comm,
                             ELT.Transmit(Container.This_Subsystem.The_ELT));

      --VHF Comm update routine
      VHF_Comm.Update(Integration_Constant,
                      Container.This_Subsystem.The_VHF_Comm);

      --Outputs from Vhf_Comm
      VHF_Comm.Set_Valid(Container.This_Subsystem.The_VHF_Comm,
         VHF_Comm.Valid(Container.This_Subsystem.The_VHF_Comm));

      --Audio outputs from vhf comm to simphonics
      if VHF_Comm.Audio_On(Container.This_Subsystem.The_VHF_Comm) then
         Container.This_Subsystem.The_Vhf_Status.Vca :=
           VHF_Comm.VCA(Container.This_Subsystem.The_VHF_Comm);
      else
         Container.This_Subsystem.The_Vhf_Status.Vca := 0.0;
      end if;
      Container.This_Subsystem.The_Vhf_Status.Valid :=
        VHF_Comm.Valid(Container.This_Subsystem.The_VHF_Comm);
      Container.This_Subsystem.The_Vhf_Status.Atis := Vhf_Station.Typ2(25);
      Container.This_Subsystem.The_Vhf_Status.Position.Latitude :=
        Vhf_Station.Lat;
      Container.This_Subsystem.The_Vhf_Status.Position.Longitude :=
        Vhf_Station.Lon;
      Container.This_Subsystem.The_Vhf_Status.Frequency :=
        Jpats_Avionics.Vhf_Tuning.Frequency;
      Container.This_Subsystem.The_Vhf_Status.Typ2 := Vhf_Station.Typ2;
      Container.This_Subsystem.The_Vhf_Status.Elt_Xmit :=
        Vhf_Comm.Elt_Transmit(Container.This_Subsystem.The_Vhf_Comm);
      Container.This_Subsystem.The_Vhf_Status.Squelch :=
        Vhf_Comm.Squelch(Container.This_Subsystem.The_Vhf_Comm);
      Container.This_Subsystem.The_Vhf_Status.Icao :=
        Vhf_Station.Icao;
      Container.This_Subsystem.The_Vhf_Status.Aptid :=
        Vhf_Station.Aid;
      Container.This_Subsystem.The_Vhf_Status.Stnid :=
        Vhf_Station.Idnt;

--UHF Comm section-----------------------------------------------------------

      Uhf_Station := Jpats_Radio_Db_If.Uhf_Comm_Station;

      --Inputs to UHF Comm from other categories
      UHF_Comm.Set_Power(Container.This_Subsystem.The_UHF_Comm,
                         Jpats_Electrical.Is_Powered(Uhf_Comm_Cb));
      UHF_Comm.Set_Aircraft_Position(Container.This_Subsystem.The_UHF_Comm,
                                    position);
      UHF_Comm.Set_Ground_Elevation(Container.This_Subsystem.The_UHF_Comm,0.0);

      UHF_Comm.Set_Station(Container.This_Subsystem.The_UHF_Comm,
                           Uhf_Station);
      UHF_Comm.Set_Frequency(Container.This_Subsystem.The_UHF_Comm,
                             Jpats_Avionics.Uhf_Tuning.Frequency);
      UHF_Comm.Set_Squelch(Container.This_Subsystem.The_UHF_Comm,
                             Jpats_Avionics.Uhf_Tuning.Squelch);
      UHF_Comm.Set_Elt_Transmit(Container.This_Subsystem.The_UHF_Comm,
                           ELT.Transmit(Container.This_Subsystem.The_ELT));

      --UHF Comm update routine
      UHF_Comm.Update(Integration_Constant,
                      Container.This_Subsystem.The_UHF_Comm);

      --Outputs from UHF Comm to simphonics
      if UHF_Comm.Audio_On(Container.This_Subsystem.The_UHF_Comm) then
         Container.This_Subsystem.The_Uhf_Status.Vca :=
           UHF_Comm.VCA(Container.This_Subsystem.The_UHF_Comm);
      else
         Container.This_Subsystem.The_Uhf_Status.Vca := 0.0;
      end if;

      Container.This_Subsystem.The_Uhf_Status.Valid :=
        UHF_Comm.Valid(Container.This_Subsystem.The_UHF_Comm);
      Container.This_Subsystem.The_Uhf_Status.Atis :=
        Uhf_Station.Typ2(25);
      Container.This_Subsystem.The_Uhf_Status.Elt_Xmit :=
        Uhf_Comm.Elt_Transmit(Container.This_Subsystem.The_UHF_Comm);
      Container.This_Subsystem.The_Uhf_Status.Frequency :=
        Jpats_Avionics.Uhf_Tuning.Frequency;
      Container.This_Subsystem.The_Uhf_Status.Typ2 :=
        Uhf_Station.Typ2;
      Container.This_Subsystem.The_Uhf_Status.Squelch :=
        Uhf_Comm.Squelch(Container.This_Subsystem.The_Uhf_Comm);
      Container.This_Subsystem.The_Uhf_Status.Tone :=
        JPATS_Avionics.Uhf_Tuning.Tone;
      Container.This_Subsystem.The_Uhf_Status.Icao :=
        Uhf_Station.Icao;
      Container.This_Subsystem.The_Uhf_Status.Aptid :=
        Uhf_Station.Aid;
      Container.This_Subsystem.The_Uhf_Status.Stnid :=
        Uhf_Station.Idnt;

--      if Debug then
--         Log.Report("UHF Freq " & Integer'Image(Jpats_Avionics.Uhf_Tuning.Frequency));
--         Log.Report("UHF Typ2(25) " & Boolean'Image(Uhf_Station.Typ2(25)));
--         Log.Report("VHF Freq " & Integer'Image(Jpats_Avionics.Vhf_Tuning.Frequency));
--         Log.Report("VHF Typ2(25) " & Boolean'Image(Vhf_Station.Typ2(25)));
--      end if;

   end Update;

end JPATS_Radio.Controller;


