-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random,
  Radio_Utilities;

use  ada.Numerics.Elementary_Functions,
  ada.Numerics.Float_Random;

package body Glideslope is

   package RU renames Radio_Utilities;

   Temp              : Float;
   XGAS              : Float;
   YGAS              : Float;
   RGAS              : Float;
   ZGAS              : Float;
   Agas              : Float;
   Bgas              : Float;
   Agas_Td           : Float;
   Brg_To_Stn        : Radio_Types.Bearing_Type;
   Sinhdg            : Float;
   Coshdg            : Float;
   Ft_To_Nm          : constant := 1.0/6076.0;
   Gs_Cg_Dist        : Float;
   Gs_Antenna_Dist_X : constant := 13.3;
   Gs_Antenna_Dist_Z : constant := 5.58;
   Gs_Avail          : Boolean;
   Gs_Range          : Float;
   Line_Of_Sight     : Float;
   In_Range          : Boolean;
   Gs_Cg_Hgt         : Float;
   Ac_To_Stn_Angle   : Float;
   Gs_Dist_Forward   : Float;
   Gs_Dist_Above     : Float;
   Gs_Antenna_Height : Float;
   Local_Gs_Dist     : Float;
   Gs_Angle          : Float;
   Degs_To_Dots      : Float;

   procedure Set_Glideslope_Inputs
     (An_Instance         : in out Instance;
      Power               : in     Boolean;
      Mode_On             : in     Boolean;
      Aircraft_Position   : in     Radio_Types.Position_Type;
      Aircraft_True_Hdg   : in     Radio_Types.Heading_Type;
      Station             : in     Jpats_Radio_Db_If_Types.Stn_Struct_Type;
      Center_Of_Gravity_X : in     Float;
      Center_Of_Gravity_Z : in     Float;
      Aircraft_Pitch      : in     Float) is
   begin
      An_Instance.Power               := Power;
      An_Instance.Mode_On             := Mode_On;
      An_Instance.Aircraft_Position   := Aircraft_Position;
      An_Instance.Station             := Station;
      An_Instance.Center_Of_Gravity_X := Center_Of_Gravity_X;
      An_Instance.Center_Of_Gravity_Z := Center_Of_Gravity_Z;
      An_Instance.Aircraft_Pitch      := Aircraft_Pitch;
   end;


    function Valid
      (An_Instance : in Instance)
     return Boolean is
   begin
      return An_Instance.Valid;
   end;

   function Receiving
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Receiving;
   end;

   function Dev_Dots
     (An_Instance : in Instance)
      return Radio_Types.Dots_Type is
   begin
      return An_Instance.Dev_Dots;
   end;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =

   procedure INIT(An_Instance : in out Instance) is

      Variable_Registered : Boolean;
   begin

      -- Initialize instance components:
      An_Instance.Power              := False;
      An_Instance.Mode_On            := False;
      An_Instance.Aircraft_Position  := (Latitude  => 45.0,
                                         Longitude => -45.0,
                                         Altitude  => 1000.0);
      An_Instance.Aircraft_True_Hdg  := 0.0;
      An_Instance.Center_Of_Gravity_X := 0.0;
      An_Instance.Center_Of_Gravity_Z := 0.0;
      An_Instance.Aircraft_Pitch := 0.0;
      An_Instance.Valid              := False;
      An_Instance.Receiving          := False;
      An_Instance.Dev_Dots           := 0.0;

      Temp              := 0.0;
      XGAS              := 0.0;
      YGAS              := 0.0;
      RGAS              := 0.0;
      ZGAS              := 0.0;
      Agas              := 0.0;
      Bgas              := 0.0;
      Agas_Td           := 0.0;
      Brg_To_Stn        := 0.0;
      Sinhdg            := 0.0;
      Coshdg            := 0.0;
      Gs_Cg_Dist        := 0.0;
      Gs_Avail          := False;
      Gs_Range          := 0.0;
      Line_Of_Sight     := 0.0;
      In_Range          := False;
      Gs_Cg_Hgt         := 0.0;
      Ac_To_Stn_Angle   := 0.0;
      Gs_Dist_Forward   := 0.0;
      Gs_Dist_Above     := 0.0;
      Gs_Antenna_Height := 0.0;
      Local_Gs_Dist     := 0.0;
      Gs_Angle          := 0.0;
      Degs_To_Dots      := 0.0;

   end Init;

-- =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =  =
   procedure Update(Integration_Constant : in     Float;
                    An_Instance          : in out Instance) is

   begin

--*********************************************************************
-- XGAS TO STATION, IN NAUTICAL MILES ( N/S DISTANCE )
--*********************************************************************
--
-- The xgas term is the north-south distance from the aircraft
-- to the selected ground station, in nautical miles.  The value
-- is protected from a zero result to prevent overflow problems
-- in other modules.  The equation for xgas is:
--
--     xgas=(stn lat - a/c lat) * 60.0  (nm/deg)
--
      XGAS := float(An_Instance.Station.Lat -
                    An_Instance.Aircraft_Position.Latitude ) * 60.0;
      -- Prevent zero divide errors:
      if XGAS = 0.0 then
         XGAS := 0.00001;
      end if;

--*******************************************************************
-- YGAS TO STATION, IN NAUTICAL MILES ( E/W DISTANCE )
--*******************************************************************
--
-- The ygas term is the east-west distance from the aircraft
-- to the selected ground station, in nautical miles.  The
-- distance is corrected for meridian convergence by use of the
-- cosine of latitude term.  This value is zero protected.
-- The equation for ygas is:
--
--   ygas = (stn lon - a/c lon) * 60.0nm/deg * cosine(a/c lat)
--
      YGAS :=
        float(RU.Xn180(An_Instance.Station.Lon -
                       An_Instance.Aircraft_Position.Longitude))*60.0 *
        cos(float(An_Instance.Aircraft_Position.Latitude),360.0);

      -- Prevent zero divide errors:
      if YGAS = 0.0 then
         YGAS := 0.00001;
      end if;

--*********************************************************************
-- RANGE TO STATION, IN NAUTICAL MILES ( GROUND RANGE )
--*********************************************************************
--
-- The range to station is the range measured along the ground
-- from the aircraft to the ground station in nautical miles.
-- The pythagorean theorem is used in conjunction with the xgas
-- and ygas terms to produce the ground range rgas.  This
-- is determined by the equation:
--
--     rgas = sqrt( xgas**2 + ygas**2 )
--
      RGAS := sqrt( XGAS ** 2 + YGAS ** 2 );

--*********************************************************************
-- A/C ALTITUDE ABOVE STATION, IN FEET
--*********************************************************************
--
-- The aircraft altitude above the ground station is determined
-- by the difference between the aircraft geometric altitude
-- (feet msl) and the ground station elevation (feet msl).
-- Positive indicates the aircraft is above the station.  The
-- altitude above the station term (nzgas) is determined by
-- the equation:
--
--     zgas= a/c geometric altitude - station elevation
--
      ZGAS := An_Instance.Aircraft_Position.Altitude -
        An_Instance.Station.Elv ;

      if ZGAS < 1.0 then ZGAS := 1.0; end if;

--*********************************************************************
-- BEARING TO THE STATION, IN +/- 180 DEGREES TRUE
--*********************************************************************
--
-- The bearing to station term (nbrgstn) is computed by taking
-- the inverse tangent of the xgas and ygas terms.  The terms
-- are passed to the routine in such a way to produce an angle
-- that is relative to true north.
-- The bearing to station is computed by the equation:
--
--                          nxgas (nm.)
--     nbrgstn = arctan ( --------------- )
--                          nygas (nm.)
--
      Brg_To_Stn := arctan(YGAS, XGAS, 360.0 );

--*********************************************************************
--*># ILS AND MKR MODES
--*********************************************************************
--*>
--*> ILS and marker beacon facilities require additional information
--*> due to the facilities' orientation.  Common terms are computed
--*> first to optimize the program execution.  Note that the
--*> equations for rotation have been rearranged to take into
--*> account zero degrees is true north.
--*>

      Sinhdg := Sin(An_Instance.Station.Hdg,360.0);
      Coshdg := Cos(An_Instance.Station.Hdg,360.0);

--*********************************************************************
--*>## AGAS TO LOCALIZER, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The down range term (nagas) is the distance from the aircraft
--*> to the facility measured along the axis of the facility
--*> heading.  Positive values indicate the aircraft is on the
--*> front course, with negative values indicate the aircraft is
--*> on the back course.  This term is in nautical miles, and is
--*> computed using the equation:
--*>
--*>     agas = ygas * sin(rwy hdg) + xgas * cos(rwy hdg)
--*>
--*>
      Agas := Ygas * Sinhdg + Xgas * Coshdg;

--*********************************************************************
--*>## BGAS TO LOCALIZER, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The cross range term (nbgas) is the distance from the
--*> aircraft to the facility centerline, measured perpendicular
--*> to the facility centerline.  Positive values indicate the
--*> aircraft is to the right of the facility centerline, and
--*> negative values indicate the aircraft is to the left of the
--*> facility centerline.  This term is in nautical miles, and
--*> is computed using the equation:
--*>
--*>     bgas = xgas * sin(rwy hdg) - ygas * cos(rwy hdg)
--*>
--*>
       Bgas := Xgas * Sinhdg - Ygas * Coshdg;

--*********************************************************************
--*>## AGAS TO TOUCHDOWN, IN NAUTICAL MILES
--*********************************************************************
--*>
--*> The distance to touchdown term (nagastd) is the distance
--*> from the aircraft to the glidepath position, measured along
--*> the facility centerline.  The term is computed as a function
--*> of the distance to the station (nagas) and the localizer to
--*> glidepath distance (nstnlgd).  The distance is in nautical
--*> miles, and is determined by the equation:
--*>
--*>     nagastd = nagas + loc to gs distance

       Agas_Td := Agas - An_Instance.Station.Lgd * Ft_To_Nm;

--******************************************************************
-- ANTENNA DISTANCES                                               *
--******************************************************************
--*>
--*> Antenna distances are calculated here based on the location
--*> of the CG sent by flight.  This is done so that the loc and G/S
--*> deviations are accurate for the HGS.
--*>

       Gs_Cg_Dist := An_Instance.Center_Of_Gravity_X - Gs_Antenna_Dist_X;
       Gs_Cg_Hgt  := An_Instance.Center_Of_Gravity_Z - Gs_Antenna_Dist_Z;

--******************************************************************
-- STATION TYPE DECODE                                             *
--******************************************************************
--
-- The two station type variables, typ1 and typ2, contain information
-- about the vor/ils.
--

       Gs_Avail := An_Instance.Station.Typ1(30) and   --ils station?
         An_Instance.Station.Typ2(21);

--*****************************************************************
-- GLIDESLOPE RANGE OVERRIDES                                     *
--*****************************************************************
--*>
--*> This section allows the programmer to manually set a default
--*> glideslope range to values other than values
--*> set in radio database.  If the glideslope is available, then
--*> the default is assigned.
--*>
       if Gs_avail then
          Gs_range := 25.0;
       else
            Gs_range := 0.0;
       end if;

--*****************************************************************
-- ILS MODE SELECTIONS                                            *
--*****************************************************************
--*>
--*> A check is made to see if the respective NAV receiver is in the
--*> ILS mode of operation.  A discrete bit is set whenever an ILS
--*> frequency is in tune.  If the ILS mode is not valid, then skip
--*> the rest of this module.
--*>

       if An_Instance.Power then
          An_Instance.Valid := True;
       else
          An_Instance.Valid := False;
          An_Instance.Receiving    := False;
          An_Instance.Dev_Dots     := 0.0;
       end if;


       if An_Instance.Valid then

--*****************************************************************
-- GLIDESLOPE AND LOCALIZER TESTS                                 *
--*****************************************************************
--*>
--*> The ILS test function is initiated from the NAV control head.
--*> When activated, the test will put the VIR32 NAV receiver into a
--*> self-test mode with the appropriate NAV indicators displaying
--*> the test results.
--*>
--      if ( nitstmr(i) .gt. 0.0                   !vor test
--     >  .and. nvtst(i) ) then
--        nitstmr(i) = nitstmr(i) - iconst         !  count down timer
--        if ( nitstmr(i) .lt. 7.0 ) then          !  1st 3 sec = inval
--          nigflg(i) = .not.ngsfail(i)            !  g/s valid
--          if ( .not.ngsfail(i)  ) nigdv(i) = 0.10 !* 2/3 scale right
--        end if                                   !
--        goto 800                                 !
--      elseif ( .not.nvtst(i) ) then              !test not set
--        nitstmr(i) = 0.0                         !  reset test timer
--      else                                       !no test in progress
--        if (nvtst(i) .and. .not.nvtstn1(i))      !  if test pushed
--     >    nitstmr(i) = 10.0                      !    start 10
--      end if                                     !

--     nvtstn1(i) = nvtst(i)                      !save for n-1 value

--*******************************************************************
--# NAV RECEIVING CHECK                                             *
--*******************************************************************
--
-- A check is made to see if the aircraft is within line of sight
-- recieving distance for the ground station to be recieved.
-- The formula for calculating the aircrafts maxium distance
-- from the station at any given altitude above the station is:
--
--   DIST = 1.225 X SQUARE ROOT OF THE TRANSMITTING ANTENNA HEIGHT
--    MINUS THE GROUND ELEVATION PLUS THE HEIGHT OF THE A/C ABOVE
--    THE STATION.
--

          if Gs_Range > Rgas then
             Line_Of_Sight := RU.Flimit((1.23*Sqrt(ZGAS)),20.0, 300.0);
             In_Range      := Line_Of_Sight > RGAS;
          else
             In_Range      := False;
             Line_Of_Sight := 0.0;
          end if;

--*********************************************************************
--*># GLIDESLOPE VALIDITY                                             *
--*********************************************************************
--*>
--*> The Glideslope becomes valid if the following conditions are met.
--*> The NAV station in tune is an ILS frequency, the station in tune
--*> is being received, the aircraft is downrange of the glideslope
--*> antenna, there are no malfunctions, the glideslope signal is in
--*> range, the glideslope angle is other than zero, and the glide-
--*> slope available flag is set.
--*>

          An_Instance.Receiving :=
            Gs_Avail and        --g/s avail
            In_Range and        --in line of sight
            Agas_Td > 0.0 and   --not past shack
            Rgas < Gs_range and --in g/s range
            An_Instance.Station.Gsa >= 0.0;

--*********************************************************************
--*># G/S DEVIATIONS                                                  *
--*********************************************************************
--*>
--*> COMPUTE GS DEVIATION IN DOTS
--*> LIMIT AGAS TO 440 FT (0.07237 NM) TO FLARE OUT G/S BEAM
--*>  0.0094227 = ( RAD TO DEG CONV ) / ( NM TO FEET CONV )
--*>  0.0094227 = 57.29 / 6080.0
--*>  2.857     = DEG TO DOTS CONV ( FOR 1.4 DEG G/S WIDTH )
--*>
--*>      *   +.75 DEGREES
--*>
--*>      *
--*>
--*>  ----*   ON GLIDEPATH         ** GLIDEPATH SENSITIVITY **
--*>
--*>      *
--*>
--*>      *   -.75 DEGREES
--*>

          if An_Instance.Receiving and Agas_Td > -0.3333 then
             --angle between a/c hdg & stn hdg
             Ac_To_Stn_Angle := An_Instance.Aircraft_True_Hdg -
               An_Instance.Station.Hdg;
             --distance gs ant fwd cg due to angl
             Gs_Dist_Forward :=
               (Gs_Antenna_Dist_X * Cos(Ac_To_Stn_Angle,360.0)) * Ft_To_Nm;
             --dist gs ant abv or below from pitch
             Gs_Dist_Above :=
               Sin(An_Instance.Aircraft_Pitch,360.0) * Gs_Antenna_Dist_X +
               Gs_Antenna_Dist_Z;
             --ht of gs ant
             Gs_Antenna_Height := Zgas + Gs_Dist_Above;
             --dist gs ant from gnd station
             Local_Gs_Dist := Agas_Td;
             --limit dist to flare the beam
             if Local_Gs_Dist < 0.07237 then
                Local_Gs_Dist := 0.07237;
             end if;
             --gs angle
             Gs_Angle := (Gs_Antenna_Height/Local_Gs_Dist) * 0.0094227;

             --compute gs deviation
             if Gs_Angle < (2.0 * An_Instance.Station.Gsa) then --normal gs
                if An_Instance.Station.Gsw > 0.0 then
                   Degs_To_Dots := 4.0/An_Instance.Station.Gsw;
                else
                   Degs_To_Dots := 2.857;
                end if;
                Temp := (Gs_Angle - An_Instance.Station.Gsa) * Degs_To_Dots;
             elsif Gs_Angle < (10.0 * An_Instance.Station.Gsa) then --false gs
                if An_Instance.Station.Gsa /= 0.0 then
                   Temp := -1.0 * Cos(
                      Ru.Xn180(Gs_Angle* 90.0/An_Instance.Station.Gsa),360.0);
                end if;
             else
                Temp := -8.0;
             end if;

             An_Instance.Dev_Dots := Ru.Flimit(Temp,-8.0,8.0);

          else
             An_Instance.Dev_Dots := 0.0;
          end if;



--      yg1(i) = yg1(i) + 0.1*(nigdv(i) - yg1(i))
--      yg2(i) = yg2(i) + 0.13333*(yg1(i) - yg2(i))
--      yg3(i) = yg3(i) + 0.27027*(yg2(i) - yg3(i))

--      nigdv(i) = yg3(i)

       end if;

   end Update;

end Glideslope;
