-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count : 0

-- The propeller interface unit (PIU) is an electro-hydromechanical
-- control unit mounted on the engine reduction gearbox.  The PIU
-- maintains propeller speed (Np ). The power manage-Ment unit (PMU)
-- controls the PIU based on the Np signal from the torque probe and
-- the electronic input from the PCLs. The PIU also contains a
-- flyweight-controlled mechanical overspeed governor and provides
-- propeller feathering through a feather dump solenoid activated by a
-- switch attached to the PCL.

-- Chapter 61 Model 3000 Maintenance Manual page 61-20-01

-------------------------------------------------------------------------------

-- DESCRIPTION

-- The PIU is an electro-hydro-mechanical control unit which modulates
-- the pitch of a single acting propeller over the entire operational
-- flight regine of the engine.  The PIU is controlled by the PMU.

-- The engine driven gear pump provides high pressure oil to the PIU.
-- Regulation of the pump delivery pressure is provided by a pressure
-- regulating valve which control pressure to a maximum of 700 PSI.
-- The regulated propeller control oil pressure is used by the primary
-- governor which consists of a Torque Motor (T/M) controlled Jet Pipe
-- Servo Valve assembly and the mechanical Overspeed Governor to
-- modulate -- the propeller servo pressure.

-- Normal Mode

-- In Normal mode, the Jet Pipe Servo Valve is electronically
-- commanded by the PMU via the T/M.  The servo valve modulated the
-- propeller servo pressure in order to maintain the propeller speed
-- at 100%.  The mechanical OSG act as a back-up to limit Np
-- overspeeds to 106%.

-- Manual Mode

-- In Manual mode, the NP OSG RESET solenoid and T/M are de-energized
-- which permits full governing of the propeller via the OSG at 200 +-
-- 2%. Np during transition between Normal and Manual mode is
-- effectively bumpless.  Minimum blade angle A reduction of power
-- causes the PMU to command a reduction of propeller pitch to
-- maintain the propeller at max RPM.  To insure a minimum thrust at
-- low power, a mechanical stop on the propeller causes the blade
-- angle to lock at minimum angle.  Feathering During shutdown,
-- retarding PCL to ro past the PROPELLER FEATHER position closes a
-- switch which commands a manual feather.  The command energizes the
-- FEATHER solemoid which dumps oil flow and feathers the propeller.
-- This is the only command the doesn't originate from the PMU.
-- Propeller Speeds 100% Np = 1995 RPM 106% Np = 2115 RPM 11.3.3
-- Propeller Interface Unit The propeller interface unit controls
-- propeller blade pitch.  It receives inputs from the PMU, from the
-- PMU mode selector, from the propeller mechanical speed, and from a
-- micro switch on the power lever angle which triggers the feather
-- solenoid at angles of 15 degrees or less.

-----------------------------------------------------------------------------------------
with Ada.Streams.Stream_Io;
with Jpats_Propeller_Types;
with Angle_Types;
with Force_Types;

package Propeller_Interface_Unit is

   type Instance is private;


-- Access Delta_Blade_Angle
   function  Delta_Blade_Angle
     (An_Instance : in Instance)
     return Angle_Types.Degrees_Per_Sec;
   procedure Set_Delta_Blade_Angle
     (An_Instance       : in out Instance;
      Delta_Blade_Angle : in     Angle_Types.Degrees_Per_Sec );

  -- Set Record_Test
  procedure Start_Recording
    (An_Instance : in out Instance;
     Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access);

  procedure Stop_Recording
    (An_Instance  : in out Instance);


-- Method Update
   procedure Update
     (Next_Instance                     : in out Instance;
      This_Instance                     : in     Instance;
      Iconst                            : in     Float;
      Overspeed_Governor_Reset_Solenoid : in     Boolean;
      Propeller_Feather_Solenoid        : in     Boolean;
      PMU_Blade_Angle_Rate              : in     Angle_Types.Degrees_Per_Sec;
      Prop_Rpm                          : in     Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
      Engine_Oil_Pressure               : in     Force_Types.Psi;
      Malf_Np_Lo                        : in     Boolean;
      Malf_Np_Hi                        : in     Boolean);

-- Method Initialize
   procedure Initialize (An_Instance : in out Instance);


private
  type Piu_Stream_Type is new Ada.Streams.Stream_IO.Stream_Access;

  procedure Nulloutput
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : in     Piu_Stream_Type);

  function Nullinput
    (Stream : access Ada.Streams.Root_Stream_Type'Class) return Piu_Stream_Type;

  procedure Nullread
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : out    Piu_Stream_Type);

  for Piu_Stream_Type'Write use Nulloutput;

  for Piu_Stream_Type'Read use Nullread;

  for Piu_Stream_Type'Output use Nulloutput;

  for Piu_Stream_Type'Input use Nullinput;

  type Instance is record
    Debug_Stream                              : Piu_Stream_Type;  -- Ada.Streams.Stream_Io.Stream_Access;
    Record_Debug_Information                  : Boolean;
    Prop_Rpm_Gov                              : Angle_Types.Rpm;
    Reset_Overspeed_Mechanical_Governor_Limit : Angle_Types.Rpm;
    Overspeed_Mechanical_Governor_Limit       : Angle_Types.Rpm;
    Delta_Blade_Angle                         : Angle_Types.Degrees_Per_Sec;
    Underspeed_Governor_Malfunction           : Angle_Types.Rpm;
    Overspeed_Governor_Malfunction            : Angle_Types.Rpm;
    Feather_Timer                             : Float;
  end record;


end Propeller_Interface_Unit;
