-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Ada.Text_Io;
with Stethoscope;
with Force_Types;
with Vector_Types;
with Log;

package body Propeller_Interface_Unit is

-- Access Delta_Blade_Angle
   function  Delta_Blade_Angle     (An_Instance : in Instance )
      return Angle_Types.Degrees is
   begin
      return An_Instance.Delta_Blade_Angle;
   end Delta_Blade_Angle;

   procedure Set_Delta_Blade_Angle
     (An_Instance       : in out Instance;
      Delta_Blade_Angle : in     Angle_Types.Degrees) is
   begin
      An_Instance.Delta_Blade_Angle := Delta_Blade_Angle;
   end Set_Delta_Blade_Angle;

  procedure Nulloutput
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : in     Piu_Stream_Type) is
  begin
    Integer'Write(Stream, 1);
  end Nulloutput;

  procedure Nullread
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : out    Piu_Stream_Type) is
    Dropped_Data : Integer;
  begin
    Integer'Read(Stream, Dropped_Data);
    Item := null;
  end Nullread;

  function Nullinput
    (Stream : access Ada.Streams.Root_Stream_Type'Class)
      return Piu_Stream_Type is
    Dropped_Data : Integer;
  begin
    Integer'Read(Stream, Dropped_Data);
    return null;
  end Nullinput;

  procedure Start_Recording
    (An_Instance  : in out Instance;
     Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access ) is

  begin
    An_Instance.Record_Debug_Information := True;
    An_Instance.Debug_Stream := Piu_Stream_Type(Debug_Stream);
    Log.Report("Recording PIU");
  end Start_Recording;

  procedure Stop_Recording
    (An_Instance : in out Instance) is
  begin
    An_Instance.Record_Debug_Information := False;
  end Stop_Recording;

-- Method Update
   procedure Update
     (Next_Instance                     : in out Instance;
      This_Instance                     : in     Instance;
      Iconst                            : in     Float;
      Overspeed_Governor_Reset_Solenoid : in     Boolean;
      Propeller_Feather_Solenoid        : in     Boolean;
      PMU_Blade_Angle_Rate              : in     Angle_Types.Degrees_Per_Sec;
      Prop_Rpm                          : in     Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
      Engine_Oil_Pressure               : in     Force_Types.Psi;
      Malf_Np_Lo                        : in     Boolean;
      Malf_Np_Hi                        : in     Boolean)
   is
     Blade_Angle_Rate_Req  : Angle_Types.Degrees_Per_Sec;
   begin
     -- Propeller pitch, which is automatically set by the PMU and
     -- PIU, may be defined by three basic conditions, feathered,
     -- low pitch (flat or fine), and high pitch (coarse). Each pitch
     -- condition is the measure of the angle between the plane of
     -- rotation of the propeller and the chord line of the blade.
     -- When feathered, the propeller blades are aligned nearly
     -- straight into the wind. Feather may be selected by placing
     -- the PCL to OFF to minimize drag if the propeller is not
     -- rotating while in flight (e.g. with an engine failure). When
     -- in low pitch, the propeller blade angle is approximately 15 deg.
     -- from the reference plane. The propeller blades will be at low
     -- pitch at low speeds and low throttle settings. High pitch is
     -- variable between feather and low pitch to maintain NP at a
     -- constant 2000 RPM for the given condition.

     -- The PMU controls propeller RPM by varying the propeller
     -- blade angle with pressure supplied by the engine oil system
     -- through the PIU. Pressurized engine oil supplied by the PIU
     -- forces the piston forward to decrease pitch towards fine.
     -- When oil pressure against the piston is reduced, the blades
     -- turn towards coarse pitch. Each blade root also incorporates
     -- a counter-weight which is positioned to counteract the natu-ral
     -- turning moment of the blade towards fine pitch. If oil
     -- pressure is lost, the propeller will automatically feather.
     -- In the event of PMU or PIU failure, NP is regulated by the
     -- mechanical flyweight overspeed governor. With the PMU
     -- functioning, the mechanical overspeed governor modulates
     -- oil pressure to the propeller pitch change piston to limit NP
     -- below 106% (approximately 2120 RPM), while the electronic
     -- governor will maintain NP at 100%. This arrangement
     -- keeps the mechanical governor in an underspeed condition
     -- with the PMU functioning. If PMU function is lost or deactivated,
     -- the mechanical overspeed governor resets to modulate
     -- oil pressure to the propeller pitch change piston to
     -- maintain NP at or below 100% (approximately 2000 RPM).
     -- With the PMU not functioning, NP may peak above 100%
     -- during power changes and then return to the governed
     -- range.

     -- In the event of an engine failure, oil pressure to the pitch
     -- change mechanism may be lost and the propeller will auto-matically
     -- return to the feathered condition. If an engine fail-ure
     -- is not due to a loss of oil pressure, the windmilling of the
     -- propeller and engine compressor will continue to drive the
     -- oil pumps, and the propeller may not feather. Placing the
     -- PCL in the OFF position triggers a microswitch which acti-vates
     -- the feather dump solenoid valve. The feather dump
     -- solenoid valve dumps propeller oil pressure, allowing the
     -- feathering spring and counterweights to feather the propel-ler.
     -- Power for the feather dump solenoid valve is provided
     -- through a circuit breaker placarded PROP SYS, located on
     -- the battery bus circuit breaker panel in the front cockpit.
     -- Refer to Section V for propeller limitations.

     -- Two PMU On propeller speed control malfunctions are simulated.
     -- One malfunction results in RPM rising to 106.%% (2130 Rpm);
     -- the PMU model responds by reverting to manual mode after 3
     -- seconds.  The second malfunction results in RPM dropping to
     -- 67% (1340 Rpm); the PMU will operate with this malfunction,
     -- but the torque limiting loop may activate to maintain output
     -- torque below the 106% limit.

     -- Note implementation of Raytheon math model has changes to
     -- incorporate all of the above functionality.  Some of these
     -- changes have been made by moving Delta_Blade_Angle calculations
     -- to the PMU where they are actually done.

     ------------------------------------
     -- Mechanical Flyweight Governor  --
     ------------------------------------

       -- PIU Malfunctioning
     if Overspeed_Governor_Reset_Solenoid then
       -- mechanical governor limit after 3 sec reset.
       -- timer handled in pmu code.  timer due pmu
       -- in state of losing power (i.e. capacitors are
       -- discharging) ?  -- 1980
       Next_Instance.Prop_Rpm_Gov  := Next_Instance.Reset_Overspeed_Mechanical_Governor_Limit; -- 1980
     else  --2130
       if Malf_Np_Hi then
       -- 106%  Overspeed governor limit
       -- PIU is possibly malfunctioning
         Next_Instance.Prop_Rpm_Gov := Next_Instance.Overspeed_Governor_Malfunction;  -- 2130 rpm or 106%
       elsif Malf_Np_Lo then
         -- Simple underspeed malfunction
         -- This could be due to PIU or
         -- mechanical propeller governor.
         Next_Instance.Prop_Rpm_Gov := Next_Instance.Underspeed_Governor_Malfunction;
       else
         Next_Instance.Prop_Rpm_Gov := Next_Instance.Overspeed_Mechanical_Governor_Limit;  -- 2005 rpm
       end if;
     end if;


     -- Note: The PMU off mechanical governing limit is set in the
     -- model at 1980 (99%).  This value is somewhat variable in
     -- flight Test, Tending to vary +/- % either side of 100%

     -- Step 2.  Calculate the blade pitch rate in response to the
     -- error between the commanded and the actual propeller speed.
     -- Propeller speed control is tighter when the PMU is on; a
     -- higher feedback gain together with a damping term are used to
     -- simulate RPM control in this mode.  However, If the power
     -- lever and is 15 degrees or less, or if the PMU is commanding
     -- blade pitch to feather (Propeller_feather_solenoid = true), or
     -- if there is a lack of sufficient engine oil pressure to supply
     -- the PIU boost pump, or if an uncommanded feather failure mode
     -- is activated (UNCOMMANDED_FEATHER = true), then the propeller
     -- is driven to feather at the rate of 25 degrees per second.
     -- Note that propeller feathering cannot be commanded if airframe
     -- power to the propeller feathering solenoid is unavailable.
     -- Airfram electrical power is available to the propeller
     -- feathering solenoid via the PROP SYS circuit breaker.

     if Engine_Oil_Pressure <= 0.0 then
       Next_Instance.Feather_Timer := This_Instance.Feather_Timer + Iconst;
     else
       Next_Instance.Feather_Timer := 0.0;
     end if;

     if Propeller_Feather_Solenoid  or Next_Instance.Feather_Timer > 90.0  then
       -- This simulates the action taken when the feather solenoid
       -- valve dumps the propeller oil pressure and allows the
       -- feather spring and courterweights on the propeller to
       -- feather the prop.  Note that if an engine malfunction
       -- is not due to the loss of oil pressure a windmilling
       -- propeller and compressor will maintaining oil pressure
       -- not allowing the propeller to feather.
       Blade_Angle_Rate_Req := 25.0;  -- deg/sec
--       Log.Report("Feathering.");
     else
      if not Overspeed_Governor_Reset_Solenoid and not Malf_Np_Hi and not Malf_Np_Lo then
--       if Prop_Rpm(Vector_Types.X) < Next_Instance.Prop_Rpm_Gov then -- 103.0*20.0 then
          Blade_Angle_Rate_Req := Pmu_Blade_Angle_Rate;
       else
         Blade_Angle_Rate_Req := 0.145*(Prop_Rpm(Vector_Types.X) - Next_Instance.Prop_Rpm_Gov);
       end if;
     end if;

     -- Next, The blade pitch rate is limited.  the minimum limit provides
     -- a realistic unfeathering rate for both automatic (i.e. pmu
     -- controlled) unfeathering and for natural (i.e. following manual
     -- mode starts with pmu off) unfeathering as rising piu oil pressure
     -- overcomes the feathering spring:

     Next_Instance.Delta_Blade_Angle :=
       Angle_Types.Degrees_Per_Sec'Max(-6.7, Angle_Types.Degrees_Per_Sec'Min
                                       ( 25.0, Blade_Angle_Rate_Req));

    if Next_Instance.Record_Debug_Information then
      declare
        Delta_Blade_Angle_Label : String := "Delta_Blade_Angle";

      begin
        String'Output(Next_Instance.Debug_Stream, Delta_Blade_Angle_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Blade_Angle);
      end;
    end if;

   end Update;

   -- Method Initialize
   procedure Initialize
     (An_Instance : in out Instance) is
   begin
     An_Instance.Prop_Rpm_Gov                              := Angle_Types.Rpm(2000.0);
     An_Instance.Reset_Overspeed_Mechanical_Governor_Limit := Angle_Types.Rpm(1980.0);
     An_Instance.Overspeed_Mechanical_Governor_Limit       := Angle_Types.Rpm(2005.0);
     An_Instance.Delta_Blade_Angle                         := Angle_Types.Degrees_Per_Sec(0.0);
     An_Instance.Underspeed_Governor_Malfunction           := Angle_Types.Rpm(1340.0);
     An_Instance.Overspeed_Governor_Malfunction           := Angle_Types.Rpm(2130.0);
     Stethoscope.Register_Signal
       (Name           => "Propeller/PIU/local/Dbeta075R",
        Units          => "float",
        Object_Address => An_Instance.Delta_Blade_Angle'Address,
        Value_Type     => "float",
        Scope_Index    => 0);
   end Initialize;

end Propeller_Interface_Unit;










