-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
--|  PROPELLER SUBSYSTEM
--|
--|  FUNCTION
--|  The function of the propeller subsystem is to convert the engine
--|  power into thrust in order to propel the aircraft through the air.
--|
--|  DESCRIPTION
--|  An electronically controlled Propeller Interface Unit
--|  (PIU) mounted on the reduction gearbox changes the blade angle
--|  automatically to maintain the propeller speed at maximum RPM
--|  throughout various power setting and ambient conditions.  When power
--|  increases, the angle of attack of the blade is increased
--|  automatically to allow the propeller to absorb additional energy
--|  without increase in propeller speed.
--|
--|  PROPELLER FEATURES
--|  o Hartzell
--|  o 4 bladed
--|  o Single acting
--|  o Variable pitch
--|  o Non reversing
--|  o Constant speed
--|  o Servo pressure drives the blade towards low pitch
--|  o Feathering spring and counterweights drive the blade
--|    towards feather position
--|
------------------------------------------------------------------------------
--|
--|  PROPELLER
--|  The aircraft is fitted with an aluminum, 97 inch, four
--|  bladed, constant speed, variable pitch, non-reversing, feath-ering
--|  propeller (Figure 1-12). The propeller is driven by the
--|  power turbine, through the reduction gearbox. The engine
--|  power management unit (PMU) and the propeller interface
--|  unit (PIU) automatically control the propeller blade angle
--|  (pitch) and propeller speed (Np ). Np is measured by the
--|  torque probe. The propeller system is designed to maintain A
--|  constant speed of 2000 RPM (100% Np ) during most flight
--|  conditions.
--|  The propeller pitch change mechanism is mounted to the
--|  front propeller hub and includes a sliding hydraulic piston,
--|  counterweights, and a feathering spring. The piston con-nects
--|  to each propeller blade with a fork which engages a
--|  cam follower on each blade root. The feathering spring con-stantly
--|  forces the blades towards the feathered position, and
--|  is overcome by the action of the hydraulic piston.
--|  Propeller pitch, which is automatically set by the PMU and
--|  PIU, may be defined by three basic conditions, feathered,
--|  low pitch (flat or fine), and high pitch (coarse). Each pitch
--|  condition is the measure of the angle between the plane of
--|  rotation of the propeller and the chord line of the blade.
--|  When feathered, the propeller blades are aligned nearly
--|  straight into the wind. Feather may be selected by placing
--|  the PCL to OFF to minimize drag if the propeller is not
--|  rotating while in flight (e.g. with an engine failure). When
--|  in low pitch, the propeller blade angle is approximately 15,A0(B
--|  from the reference plane. The propeller blades will be at low
--|  pitch at low speeds and low throttle settings. High pitch is
--|  variable between feather and low pitch to maintain N P at a
--|  constant 2000 RPM for the given condition.
--|  The PMU controls propeller RPM by varying the propeller
--|  blade angle with pressure supplied by the engine oil system
--|  through the PIU. Pressurized engine oil supplied by the PIU
--|  forces the piston forward to decrease pitch towards fine.
--|  When oil pressure against the piston is reduced, the blades
--|  turn towards coarse pitch. Each blade root also incorporates
--|  a counter-weight which is positioned to counteract the natu-ral
--|  turning moment of the blade towards fine pitch. If oil
--|  pressure is lost, the propeller will automatically feather.
--|  In the event of PMU or PIU failure, N P is regulated by the
--|  mechanical flyweight overspeed governor. With the PMU
--|  functioning, the mechanical overspeed governor modulates
--|  oil pressure to the propeller pitch change piston to limit Np
--|  below 106% (approximately 2120 RPM), while the elec-tronic
--|  governor will maintain Np at 100%. This arrangement
--|  keeps the mechanical governor in an underspeed condition
--|  with the PMU functioning. If PMU function is lost or deac-tivated,
--|  the mechanical overspeed governor resets to modu-late
--|  oil pressure to the propeller pitch change piston to
--|  maintain Np at or below 100% (approximately 2000 RPM).
--|  With the PMU not functioning, Np may peak above 100%
--|  during power changes and then return to the governed
--|  range.
--|  In the event of an engine failure, oil pressure to the pitch
--|  change mechanism may be lost and the propeller will auto-matically
--|  return to the feathered condition. If an engine fail-ure
--|  is not due to a loss of oil pressure, the windmilling of the
--|  propeller and engine compressor will continue to drive the
--|  oil pumps, and the propeller may not feather. Placing the
--|  PCL in the OFF position triggers a microswitch which acti-vates
--|  the feather dump solenoid valve. The feather dump
--|  solenoid valve dumps propeller oil pressure, allowing the
--|  feathering spring and counterweights to feather the propel-ler.
--|  Power for the feather dump solenoid valve is provided
--|  through a circuit breaker placarded PROP SYS, located on
--|  the battery bus circuit breaker panel in the front cockpit.
--|  Refer to Section V for propeller limitations.
--|
--|  Ref.  AIR FORCE TO 1T-6A-1
--|        NAVY (NAVAIR) A1-T6AAA-NFM-100
--|        Page 1-17 CVV DRAFT DECEMBER 1998
--|
-------------------------------------------------------------------------------

--|  PROPELLER MODEL
--|
--|  These propeller equations yield the force and moment contributions
--|  of the powerplant arising from the following:
--|
--|  o the thrust produced by the propeller in axial flow conditions,
--|  o the aerodynamic torque absorbed by the propeller in axial flow
--|    conditions,
--|  o the effect of roll rate upon propeller speed relative to the air,
--|    which contributes to roll damping,
--|  o the gyroscopic reaction moments for the combined propeller, power
--|    turbine and associated rotating components.
--|
--|  In addition, the aerodynamic torque absorbed by the propeller at the
--|  current operating condition is calculated, including a correction
--|  for conditions of non-axial flow, and the thrust coefficient is
--|  calculated for use in the aerodynamic model.
--|
--|  Note that other propeller-derived forces and moments are merged into
--|  the wing-body-plus model, including the correction to propeller thrust
--|  for deviation from axial flow conditions, and the in-plane forces
--|  (normal and sideforce) and hub moments (pitching and yawing) arising
--|  from non-axial flow conditions.  Pitch and yaw damping contributions
--|  of the propeller are similarly merged into the wing-body-plus model.
--|
--|  Static thrust and power coefficients are depicted as functions of blade
--|  angle and advance ratio.  The blade angle functionality permits full
--|  simulation of the powerplant's RPM governing loop, as well as operation
--|  on the low or high pitch stops.
--|
--|  The variables exported from this section are:   Lgyro, Mgyro, Ngyro,
--|                                                  Lprop, Mprop, Nprop,
--|                                                  Qprop, Tc'lim,
--|                                                  Xprop, Yprop, Zprop.
--|
--|  Reference: Kimball, D., "Flight Simulation Mathematical Model of
--|             the Beech MkII Joint Primary Aircraft Training System
--|             (JPATS)", Document 133E794 Rev. A, 18 February, 1998,
--|             FSI-SSD # 50181, pg 66.
--|
-------------------------------------------------------------------------------
with Jpats_Propeller_Types;
with Normalized_Types;
with Length_Types;
with Angle_Types;
with Temperature_Types;
with Force_Types;
with Ada.Streams.Stream_Io;

package Jpats_Propeller is

   type Instance is null record;

-- Accessor for X-component of the moment arm
   function Propeller_Moment_Arm
     return Length_Types.Feet;

-- Accessor Gyroscopic_Moments_Tensor  (i.e. Lgyro, Mgyro, Ngyro)
   function Gyroscopic_Moments_Tensor
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance;

-- Accessor Moment_Tensor  (i.e. Lprop, Mprop, Nprop)
   function Moment_Tensor
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance;

-- Accessor Force_Tensor  (i.e. Xprop, Yprop, Zprop)
   function Force_Tensor
     return Jpats_Propeller_Types.Force_Vector_Type.Instance;

-- Accessor Axial_Thrust_Coefficient_Limited
   function Axial_Thrust_Coefficient_Limited
     return Normalized_Types.Normalize;

-- Accessor Axial_Torque_Absorbed
   function Total_Torque_Absorbed
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance;

  procedure Start_Recording_Debug_Output
    (Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access);

  procedure Stop_Recording_Debug_Output;

  procedure Set_Aircraft_Angular_Rate
    ( Aircraft_Angular_Rate : in  Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance);

  procedure Set_Aircraft_Cg
    (Aircraft_Cg : in  Jpats_Propeller_Types.Coordinate_Type.Instance);

  procedure Set_Angle_Of_Attack
    (Alpha : in Angle_Types.Radians);

  procedure Set_Sideslip_Angle
    (Beta : in Angle_Types.Radians);

  procedure Set_Mach_Number
    (Mach_Number : in Length_Types.Mach);

  procedure Set_Ambient_Temperature
    (Temp : in Temperature_Types.Celsius);

  procedure Set_Ambient_Temperature_Ratio
    (Ambient_Temperature_Ratio : in Normalized_Types.Normalize);

  procedure Set_Ambient_Pressure
    (Ambient_Pressure : in Force_Types.Pounds_Per_Sq_Feet);

  procedure Set_Ambient_Pressure_Ratio
    (Ambient_Pressure_Ratio : in Normalized_Types.Normalize);

  procedure Set_Pressure_Altitude
    (Pressure_Altitude : in Length_Types.Feet);

  procedure Reset;

  procedure Reset_Test_Modes;
end  Jpats_Propeller;

