-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Torque_Types;
with Vector_Types;
use  Vector_Types;

with JPATS_Propeller.Container;
with JPATS_Propeller.Controller;
with Hartzell_4_Blade_Propeller;
with Geometry;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;

package body JPATS_Propeller is

-- Accessor for X-component of the moment arm
   function Propeller_Moment_Arm
     return Length_Types.Feet is
     Temp : Jpats_Propeller_Types.Coordinate_Type.Instance;
   begin
      Temp := Geometry.Moment_Arm
        (An_Instance => JPATS_Propeller.Container.This_Subsystem.The_Propeller_Installation_Geometry);
      return Temp(X);
   end Propeller_Moment_Arm;

-- Accessor Gyroscopic_Moments_Tensor  (i.e. Lgyro, Mgyro, Ngyro)
   function Gyroscopic_Moments_Tensor
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
   begin
     return Geometry.Gyroscopic_Moment_Tensor
       (An_Instance => JPATS_Propeller.Container.This_Subsystem.The_Propeller_Installation_Geometry);
   end Gyroscopic_Moments_Tensor;

-- Accessor Moment_Tensor  (i.e. Lprop, Mprop, Nprop)
   function Moment_Tensor
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
   begin
     return Geometry.Moment_Tensor
      (An_Instance => JPATS_Propeller.Container.This_Subsystem.The_Propeller_Installation_Geometry);
   end Moment_Tensor;

-- Accessor Force_Tensor  (i.e. Xprop, Yprop, Zprop)
   function Force_Tensor
     return Jpats_Propeller_Types.Force_Vector_Type.Instance is
   begin
     return Geometry.Force_Tensor
      (An_Instance => JPATS_Propeller.Container.This_Subsystem.The_Propeller_Installation_Geometry);
   end Force_Tensor;

-- Accessor Axial_Thrust_Coefficient_Limited
   function Axial_Thrust_Coefficient_Limited
     return Normalized_Types.Normalize is
   begin
     return Hartzell_4_Blade_Propeller.Aero_Axial_Thrust_Coefficient_Limited
       (An_Instance => JPATS_Propeller.Container.This_Subsystem.The_Propeller);
   end Axial_Thrust_Coefficient_Limited;

-- Accessor Axial_Torque_Absorbed
   function Total_Torque_Absorbed
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
   begin
     return Hartzell_4_Blade_Propeller.Total_Torque_Absorbed
       (An_Instance => JPATS_Propeller.Container.This_Subsystem.The_Propeller);
   end Total_Torque_Absorbed;

   procedure Start_Recording_Debug_Output
     (Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access) is
   begin
     JPATS_Propeller.Controller.Start_Recording_Debug_Output
       (A_Subsystem  => JPATS_Propeller.Container.Next_Subsystem,
        Debug_Stream => Debug_Stream);
   end Start_Recording_Debug_Output;

   procedure Stop_Recording_Debug_Output is
   begin
     JPATS_Propeller.Controller.Stop_Recording_Debug_Output
       (A_Subsystem  => JPATS_Propeller.Container.Next_Subsystem);
   end Stop_Recording_Debug_Output;

   procedure Set_Aircraft_Cg
     (Aircraft_Cg : in  Jpats_Propeller_Types.Coordinate_Type.Instance) is
   begin
     JPATS_Propeller.Controller.Aircraft_Auto_Driven := True;
     JPATS_Propeller.Controller.Aircraft_Cg := Aircraft_Cg;
   end Set_Aircraft_Cg;

   procedure Set_Aircraft_Angular_Rate
     ( Aircraft_Angular_Rate : in  Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance) is
   begin
     JPATS_Propeller.Controller.Aircraft_Auto_Driven := True;
     JPATS_Propeller.Controller.Aircraft_Angular_Rate := Aircraft_Angular_Rate;
   end Set_Aircraft_Angular_Rate;

   procedure Set_Mach_Number
     (Mach_Number : in Length_Types.Mach) is
   begin
     JPATS_Propeller.Controller.Aircraft_Auto_Driven := True;
     JPATS_Propeller.Controller.Mach := Mach_Number;
   end Set_Mach_Number;

   procedure Set_Ambient_Temperature
     (Temp : in Temperature_Types.Celsius) is
   begin
     JPATS_Propeller.Controller.Atmosphere_Auto_Driven := True;
     JPATS_Propeller.Controller.Ambient_Temperature := Temp;
     JPATS_Propeller.Controller.Ambient_Temperature_Ratio := (Temp+273.15)/288.15;
   end Set_Ambient_Temperature;

  procedure Set_Ambient_Temperature_Ratio
    (Ambient_Temperature_Ratio : in Normalized_Types.Normalize)is
  begin
    JPATS_Propeller.Controller.Atmosphere_Auto_Driven := True;
    JPATS_Propeller.Controller.Ambient_Temperature_Ratio := Ambient_Temperature_Ratio;
    JPATS_Propeller.Controller.Ambient_Temperature :=  Ambient_Temperature_Ratio*288.15 - 273.15;
  end Set_Ambient_Temperature_Ratio;

  procedure Set_Ambient_Pressure
    (Ambient_Pressure : in Force_Types.Pounds_Per_Sq_Feet) is
  begin
    JPATS_Propeller.Controller.Atmosphere_Auto_Driven := True;
    JPATS_Propeller.Controller.Ambient_Pressure := Ambient_Pressure;
    JPATS_Propeller.Controller.Ambient_Pressure_Ratio := Ambient_Pressure/2116.22;
    JPATS_Propeller.Controller.Pressure_Altitude := ((1.0 - (Ambient_Pressure/2116.22)**0.19018)*145447.0);
  end Set_Ambient_Pressure;

  procedure Set_Ambient_Pressure_Ratio
    (Ambient_Pressure_Ratio : in Normalized_Types.Normalize) is
  begin
    JPATS_Propeller.Controller.Atmosphere_Auto_Driven := True;
    JPATS_Propeller.Controller.Ambient_Pressure_Ratio := Ambient_Pressure_Ratio;
    JPATS_Propeller.Controller.Ambient_Pressure := Ambient_Pressure_Ratio*2116.22;
    JPATS_Propeller.Controller.Pressure_Altitude := ((1.0 - Ambient_Pressure_Ratio**0.19018)*145447.0);
  end Set_Ambient_Pressure_Ratio;

  procedure Set_Pressure_Altitude
    (Pressure_Altitude : in Length_Types.Feet) is
  begin
    JPATS_Propeller.Controller.Atmosphere_Auto_Driven := True;
    JPATS_Propeller.Controller.Pressure_Altitude  := Pressure_Altitude;
    JPATS_Propeller.Controller.Ambient_Pressure_Ratio := (1.0 - Pressure_Altitude/145447.0 )**(1.0/0.19018);
    JPATS_Propeller.Controller.Ambient_Pressure := JPATS_Propeller.Controller.Ambient_Pressure_Ratio*2116.22;
  end Set_Pressure_Altitude;

  procedure Set_Angle_Of_Attack
    (Alpha : in Angle_Types.Radians) is
  begin
     JPATS_Propeller.Controller.Aircraft_Auto_Driven := True;
     JPATS_Propeller.Controller.Aircraft_Angle_Of_Attack := Alpha;
  end Set_Angle_Of_Attack;

  procedure Set_Sideslip_Angle
    (Beta : in Angle_Types.Radians) is
  begin
     JPATS_Propeller.Controller.Aircraft_Auto_Driven := True;
     JPATS_Propeller.Controller.Aircraft_Sideslip_Angle := Beta;
  end Set_Sideslip_Angle;

  procedure Reset is
  begin
    JPATS_Propeller.Controller.Reset;
  end Reset;

  procedure Reset_Test_Modes is
  begin
    JPATS_Propeller.Controller.Aircraft_Auto_Driven := False;
    JPATS_Propeller.Controller.Atmosphere_Auto_Driven := False;
  end Reset_Test_Modes;

end JPATS_Propeller;










