-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0

-- Interface to External Classes
with Stethoscope;
with Ada.Text_Io;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;
with Jpats_Atmosphere;
with Jpats_Simulated_Aircraft;
with Jpats_Powerplant;
with JPATS_Propeller.Ios_Interface;
with Jpats_Auto_Test;
with Electrical_Units_Types;
with Jpats_Electrical_Types;
with Jpats_Electrical;

with Vector_Types;
use  Vector_Types;
with Coordinate_Types;

-- Interface to Local Subsystem
with Propeller_Interface_Unit;
with Hartzell_4_Blade_Propeller;
with Hartzell_4_Blade_Propeller_Geometry;
with Geometry;

package body JPATS_Propeller.Controller is

   Time_Step         : Integer;
   Propeller_Test    : Boolean := False;
--   Aircraft_Velocity : Coordinate_Types.Cartesian := (0.0,0.0,0.0);


   function Convert_To_Aircraft_Angular_Rate
     (Ct : in Coordinate_Types.Attitude)
      return Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance is
   begin
      return ( Roll  => Angle_Types.Radians_Per_Sec(Ct.Roll),
               Pitch => Angle_Types.Radians_Per_Sec(Ct.Pitch),
               Yaw   => Angle_Types.Radians_Per_Sec(Ct.Yaw));
   end Convert_To_Aircraft_Angular_Rate;

    -- procedure start recording debug stream
    procedure Start_Recording_Debug_Output
      (A_Subsystem  : in JPATS_Propeller.Container.Instance_Ptr;
       Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access) is
    begin
      Propeller_Interface_Unit.Start_Recording
        (An_Instance => A_Subsystem.The_PIU,
         Debug_Stream => Debug_Stream);
      Hartzell_4_Blade_Propeller.Start_Recording
        (An_Instance => A_Subsystem.The_Propeller,
         Debug_Stream => Debug_Stream);
    end Start_Recording_Debug_Output;

    procedure Stop_Recording_Debug_Output
      (A_Subsystem  : in JPATS_Propeller.Container.Instance_Ptr) is
    begin
      Propeller_Interface_Unit.Stop_Recording
        (An_Instance => A_Subsystem.The_PIU);
      Hartzell_4_Blade_Propeller.Stop_Recording
        (An_Instance => A_Subsystem.The_Propeller);
    end Stop_Recording_Debug_Output;


    Prop_Rpm                                : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
   Engine_Oil_Pressure                     : Force_Types.Psi;
   Overspeed_Governor_Reset_Solenoid       : Boolean;
   Manual_Feather_Signal                   : Boolean;
   PMU_Blade_Angle_Rate                    : Angle_Types.Rpm_Per_Sec;
   Low_Pitch_Stop                          : Angle_Types.Degrees;
   Propeller_Diameter                      : Length_Types.Feet;
   Wetted_Surface_Area                     : Length_Types.Sq_Feet;
   Delta_Blade_Angle                       : Angle_Types.Degrees_Per_Sec;
   Axial_Thrust                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
   Prop_Inertia                            : Mass_Types.Slugs_Feet_Sq;
   Axial_Torque_Absorbed                   : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
   Alpha_Propeller                         : Angle_Types.Degrees;
   Aircraft_Velocity                       : Coordinate_Types.Cartesian;
   Prop_Orientation                        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
   Propeller_Force                         : Jpats_Propeller_Types.Force_Vector_Type.Instance;
   Propeller_Moment                        : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
   Propeller_Gyro_Moment                   : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
   Piu_Failure                             : Boolean;
   Malf_Np_Hi                              : Boolean;
   Malf_Np_Lo                              : Boolean;
   Propeller_Reset                         : Boolean;
   Propeller_Reset_Engine_Torque           : Normalized_Types.Normalize;
   Propeller_Reset_Counter                 : Integer;

   procedure Reset is
   begin
     Propeller_Reset := True;
   end Reset;

   procedure Update
     (Next_Subsystem : in out Jpats_Propeller.Container.Instance_Ptr;
      This_Subsystem : in     Jpats_Propeller.Container.Instance_Ptr;
      Iconst         : in     Float) is

     use type Jpats_Propeller_Types.Torque_Vector_Type.Instance;

--      Malf_Np_Hi                              : Boolean renames Container.This_Ios_Interface.Malf_Np_Hi;
--      Malf_Np_Lo                              : Boolean renames Container.This_Ios_Interface.Malf_Np_Lo;
--      Aircraft_Cg : constant Jpats_Propeller_Types.Coordinate_Type.Instance
--        := (Jpats_Simulated_Aircraft.Get_X_Cg,
--            Jpats_Simulated_Aircraft.Get_Y_Cg,
--            Jpats_Simulated_Aircraft.Get_Z_Cg);
      --        := (Jpats_Simulated_Aircraft.Get_Fuselage_Station_Of_Current_Center_Of_Gravity_Of_Simulated_Aircraft,
      --            Jpats_Simulated_Aircraft.Get_Buttline_Of_Current_Center_Of_Gravity_Of_Simulated_Aircraft,
      --            Jpats_Simulated_Aircraft.Get_Waterline_Of_Current_Center_Of_Gravity_Of_Simulated_Aircraft);

--      Air_Density                             : Mass_Types.Slugs_Per_Cubic_Feet;
--      Aircraft_Angular_Rate                   : Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
--      Prop_Rpm                                : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
--      Engine_Oil_Pressure                     : Force_Types.Psi;
--      Ambient_Temperature_Ratio               : Normalized_Types.Normalize;
--      Aircraft_True_Airspeed                  : Length_Types.Feet_Per_Sec;
--      Aircraft_Angle_Of_Attack                : Angle_Types.Radians;
--      Aircraft_Sideslip_Angle                 : Angle_Types.Radians;
--      Dynamic_Pressure                        : Force_Types.Pounds_Per_Sq_Feet;
--      Mach                                    : Length_Types.Mach;
--      Overspeed_Governor_Reset_Solenoid       : Boolean;
--      Manual_Feather_Signal                   : Boolean;
--      PMU_Blade_Angle_Rate                    : Angle_Types.Rpm_Per_Sec;
--      Low_Pitch_Stop                          : Angle_Types.Degrees;
--      Propeller_Diameter                      : Length_Types.Feet;
--      Wetted_Surface_Area                     : Length_Types.Sq_Feet;
--      Delta_Blade_Angle                       : Angle_Types.Degrees_Per_Sec;
--      Axial_Thrust                            : Jpats_Propeller_Types.Force_Vector_Type.Instance;
--      Prop_Inertia                            : Mass_Types.Slugs_Feet_Sq;
--      Axial_Torque_Absorbed                   : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
--      Alpha_Propeller                         : Angle_Types.Degrees;
--      Aircraft_Velocity                       : Coordinate_Types.Cartesian;
--      Prop_Orientation                        : Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
   begin

       -- Atmosphere input
     if not Atmosphere_Auto_Driven then
       Air_Density                             := Jpats_Atmosphere.Density;
       Ambient_Temperature_Ratio               := Jpats_Atmosphere.Temperature_Ratio;
     else
       Air_Density := Ambient_Pressure_Ratio/Ambient_Temperature_Ratio*0.0023769;
     end if;

       -- Simulated Aircraft input
      -- Get Velocity in Body Axis Coordinate System
     if not Aircraft_Auto_Driven then
       Aircraft_Velocity                       := Jpats_Simulated_Aircraft.Get_V_B;
       Aircraft_Angular_Rate                   := Convert_To_Aircraft_Angular_Rate ( Ct => Jpats_Simulated_Aircraft.Get_Angular_Rate );
       Aircraft_Cg                             := (Jpats_Simulated_Aircraft.Get_X_Cg,
                                                   Jpats_Simulated_Aircraft.Get_Y_Cg,
                                                   Jpats_Simulated_Aircraft.Get_Z_Cg);
       Aircraft_True_Airspeed                  := Jpats_Simulated_Aircraft.Get_True_Airspeed;
       Aircraft_Angle_Of_Attack                := Jpats_Simulated_Aircraft.Get_Angle_Of_Attack;
       Aircraft_Sideslip_Angle                 := Jpats_Simulated_Aircraft.Get_Side_Slip_Angle;
       Dynamic_Pressure                        := Jpats_Simulated_Aircraft.Get_Dynamic_Pressure;  --need type casting?
       Mach                                    := Jpats_Simulated_Aircraft.Get_Mach_Number;
     else
       Aircraft_True_Airspeed := Mach * ((1.4 * 287.074 * Ambient_Temperature_Ratio*288.15)**0.5)*100.0/2.54/12.0;
       Dynamic_Pressure := 0.5 * Air_Density*Aircraft_True_Airspeed**2.0;
       Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
     end if;

       -- Powerplant input
       Prop_Rpm                                := Jpats_Powerplant.Reduction_Gear_Box_Rpm;
       Engine_Oil_Pressure                     := Jpats_Powerplant.Engine_Oil_Pressure;
       Overspeed_Governor_Reset_Solenoid       := Jpats_Powerplant.Overspeed_Governor_Reset_Solenoid;
       Manual_Feather_Signal                   := Jpats_Powerplant.Manual_Feather_Signal;
       PMU_Blade_Angle_Rate                    := Jpats_Powerplant.PMU_Blade_Angle_Rate;
--       Propeller_Reset                         := JPATS_Powerplant.Propeller_Reset;
--       if Prop_Check_Case = 1 then
--         Air_Density := 0.00237690;
--         Aircraft_Angular_Rate := (-1.0,1.0,1.0);
--         Aircraft_Cg := (164.0/12.0, 0.5/12.0, 85.0/12.0);
--         Aircraft_True_Airspeed := 120.0;
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(10.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(10.0);
--         Dynamic_Pressure := 17.113680;
--         Mach := 0.1074884;
--         Ambient_Temperature_Ratio := 1.0;
--         Prop_Rpm := (1700.0,0.0,0.0);
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Pmu_Blade_Angle_Rate := 0.1*(21.5 - Hartzell_4_Blade_Propeller.Blade_Angle ( Container.This_Subsystem.The_Propeller));
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Prop_Check_Case = 2  then
--         Air_Density := 0.0020480850;
--         Aircraft_Angular_Rate := (1.0,-1.0,1.0);
--         Aircraft_Cg := (169.0/12.0, 0.5/12.0, 90.0/12.0);
--         Aircraft_True_Airspeed := 170.0;
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(-10.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(10.0);
--         Dynamic_Pressure := 29.594840;
--         Mach := 0.1549620;
--         Ambient_Temperature_Ratio := 1.0;
--         Prop_Rpm := (2000.0,0.0,0.0);
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid := False;
--         Pmu_Blade_Angle_Rate := 0.1*(21.5 - Hartzell_4_Blade_Propeller.Blade_Angle ( Container.This_Subsystem.The_Propeller));
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Prop_Check_Case = 3  then
--         Aircraft_Cg := (169.0/12.0, 0.5/12.0, 90.0/12.0);
--         Aircraft_True_Airspeed := 250.0;
--         Prop_Rpm := (2000.0,0.0,0.0);
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(-2.0);
--         Aircraft_Angular_Rate := (1.0,1.0,-1.0);
--         Mach := 0.2278853;
--         Air_Density := 0.002048085;
--         Dynamic_Pressure := 64.002670;
--         Ambient_Temperature_Ratio := 1.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Pmu_Blade_Angle_Rate := 0.1*(21.5 - Hartzell_4_Blade_Propeller.Blade_Angle ( Container.This_Subsystem.The_Propeller));
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Prop_Check_Case = 4  then
--         Aircraft_Cg := (169.0/12.0, 0.5/12.0, 90.0/12.0);
--         Aircraft_True_Airspeed := 250.0;
--         Prop_Rpm := (2000.0,0.0,0.0);
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(-2.0);
--         Aircraft_Angular_Rate := (1.0,0.0,0.0);
--         Mach := 0.2320533;
--         Air_Density := 0.001755258;
--         Dynamic_Pressure := 54.851800;
--         Ambient_Temperature_Ratio := 1.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Pmu_Blade_Angle_Rate := 0.1*(21.5 - Hartzell_4_Blade_Propeller.Blade_Angle ( Container.This_Subsystem.The_Propeller));
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Prop_Check_Case = 5  then
--         Aircraft_Cg := (169.0/12.0, 0.5/12.0, 90.0/12.0);
--         Aircraft_True_Airspeed := 400.0;
--         Prop_Rpm := (2000.0,0.0,0.0);
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(-2.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.3712852;
--         Air_Density := 0.001755258;
--         Dynamic_Pressure := 140.420600;
--         Ambient_Temperature_Ratio := 1.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Pmu_Blade_Angle_Rate := 0.1*(21.5 - Hartzell_4_Blade_Propeller.Blade_Angle ( Container.This_Subsystem.The_Propeller));
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Prop_Check_Case = 6  then
--         Aircraft_Cg := (169.0/12.0, 0.5/12.0, 90.0/12.0);
--         Aircraft_True_Airspeed := 700.0;
--         Prop_Rpm := (2000.0,0.0,0.0);
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(-2.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.6497492;
--         Air_Density := 0.001755258;
--         Dynamic_Pressure := 430.038100;
--         Ambient_Temperature_Ratio := 1.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Pmu_Blade_Angle_Rate := 0.1*(21.5 - Hartzell_4_Blade_Propeller.Blade_Angle ( Container.This_Subsystem.The_Propeller));
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Prop_Check_Case = 7  then
--         Aircraft_Cg := (169.0/12.0, 0.5/12.0, 90.0/12.0);
--         Aircraft_True_Airspeed := 400.0;
--         Prop_Rpm := (300.0,0.0,0.0);
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(-2.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.3712852;
--         Air_Density := 0.001755258;
--         Dynamic_Pressure := 140.4206;
--         Ambient_Temperature_Ratio := 1.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Pmu_Blade_Angle_Rate := 0.1*(21.5 - Hartzell_4_Blade_Propeller.Blade_Angle ( Container.This_Subsystem.The_Propeller));
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--
--       end if;

--       if Engine_Check_Case = 1 then
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Aircraft_Cg := (169.0,0.5,90.0);
--         Mach := 0.0;
--         Ambient_Temperature_Ratio := 1.0;
--         Ambient_Pressure_Ratio := (1.0 - 0.00000687535*0.0)**5.2561;
--         Aircraft_True_Airspeed := Mach*(1.4*Ambient_Temperature_Ratio*(459.67+59.0)*53.35*32.2)**0.5;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Engine_Check_Case = 2 then
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Aircraft_Cg := (169.0,0.5,90.0);
--         Mach := 0.3;
--         Ambient_Temperature_Ratio := 1.0;
--         Ambient_Pressure_Ratio := (1.0 - 0.00000687535*5000.0)**5.2561;
--         Aircraft_True_Airspeed := Mach*(1.4*Ambient_Temperature_Ratio*(459.67+59.0)*53.35*32.2)**0.5;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Engine_Check_Case = 3 then
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Aircraft_Cg := (169.0,0.5,90.0);
--         Mach := 0.3;
--         Ambient_Temperature_Ratio := 1.0;
--         Ambient_Pressure_Ratio := (1.0 - 0.00000687535*5000.0)**5.2561;
--         Aircraft_True_Airspeed := Mach*(1.4*Ambient_Temperature_Ratio*(459.67+59.0)*53.35*32.2)**0.5;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;--
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Engine_Check_Case = 4 then
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Aircraft_Cg := (169.0,0.5,90.0);
--         Mach := 0.3;
--         Ambient_Temperature_Ratio := 1.0;
--         Ambient_Pressure_Ratio := (1.0 - 0.00000687535*5000.0)**5.2561;
--         Aircraft_True_Airspeed := Mach*(1.4*Ambient_Temperature_Ratio*(459.67+59.0)*53.35*32.2)**0.5;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Engine_Check_Case = 5 then
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Aircraft_Cg := (169.0,0.5,90.0);
--         Mach := 0.3;
--         Ambient_Temperature_Ratio := 1.0;
--         Ambient_Pressure_Ratio := (1.0 - 0.00000687535*5000.0)**5.2561;
--         Aircraft_True_Airspeed := Mach*(1.4*Ambient_Temperature_Ratio*(459.67+59.0)*53.35*32.2)**0.5;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Engine_Check_Case = 6 then
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Aircraft_Cg := (169.0,0.5,90.0);
--         Mach := 0.3;
--         Ambient_Temperature_Ratio := 1.0;
--         Ambient_Pressure_Ratio := (1.0 - 0.00000687535*5000.0)**5.2561;
--         Aircraft_True_Airspeed := Mach*(1.4*Ambient_Temperature_Ratio*(459.67+59.0)*53.35*32.2)**0.5;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Engine_Check_Case = 7 then
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Aircraft_Cg := (169.0,0.5,90.0);
--         Mach := 0.3;
--         Ambient_Temperature_Ratio := 1.0;
--         Ambient_Pressure_Ratio := (1.0 - 0.00000687535*5000.0)**5.2561;
--         Aircraft_True_Airspeed := Mach*(1.4*Ambient_Temperature_Ratio*(459.67+59.0)*53.35*32.2)**0.5;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Engine_Oil_Pressure := 101.0;
--         Overspeed_Governor_Reset_Solenoid :=False;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       end if;

--       if Pmu_Check_Case = 2 then
--         Aircraft_Cg := (165.29/12.0, 0.5/12.0, 91.31/12.0);
--         Aircraft_True_Airspeed := 130.0*1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.2;
--         Air_Density := 0.0023769;
--         Dynamic_Pressure := 17.46615;
--         Ambient_Temperature_Ratio := 0.9656236;
--         Engine_Oil_Pressure := 101.0;
----       Overspeed_Governor_Reset_Solenoid :=False;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Pmu_Check_Case = 4 then
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Prop_Rpm := (1060.2,0.0,0.0);
--         Aircraft_Cg := (164.7944/12.0, 0.5/12.0, 90.29/12.0);
--         Aircraft_True_Airspeed := 0.0; -- *1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.1411);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.0;
--         Air_Density := 0.0023769;
--         Ambient_Temperature_Ratio := 1.0;
--         Dynamic_Pressure := 0.0;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--         if Start_Test then
--           JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Quick_Start := False;
--           if Engine_Check_Case_Timer >= 0.0 and Engine_Check_Case_Timer <= 20.0 then
--             Open_Stream;
--             declare
--               Time_Label : String := "Time";
--             begin
--               String'Output(Debug_Stream, Time_Label);
--             end;
--             Float'Output(Debug_Stream, Engine_Check_Case_Timer);
--             Pratt_Whitney_Pt6a_68_Gas_Generator.Start_Recording
--               (An_Instance  => Next_Subsystem.The_Gas_Generator,
--                Debug_Stream => Debug_Stream);
--           else
--             Pratt_Whitney_Pt6a_68_Gas_Generator.Stop_Recording
--               (An_Instance => Next_Subsystem.The_Gas_Generator);
--             Start_Test := False;
--           end if;
--
--           if Engine_Check_Case_Timer > 2.0 and Engine_Check_Case_Timer <= 12.0 then
--             Fuel_Flow_Burn_Rate := 600.0;
--           elsif Engine_Check_Case_Timer > 12.0 then
--             Fuel_Flow_Burn_Rate := 160.0;
--           end if;
--           Engine_Check_Case_Timer := Engine_Check_Case_Timer + Iconst;
--
--         else
--           Engine_Check_Case_Timer := 0.0;
--         end if;
--       elsif Pmu_Check_Case = 5 then
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
----         Prop_Rpm := (1060.2,0.0,0.0);
--         Aircraft_Cg := (164.7944/12.0, 0.5/12.0, 90.29/12.0);
--         Aircraft_True_Airspeed := 0.0; -- *1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.1411);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.0;
--         Air_Density := 0.0023769;
--         Ambient_Temperature_Ratio := 1.0;
--         Dynamic_Pressure := 0.0;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Pmu_Check_Case = 6 then
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
----         Prop_Rpm := (1060.2,0.0,0.0);
--         Aircraft_Cg := (164.7944/12.0, 0.5/12.0, 90.29/12.0);
--         Aircraft_True_Airspeed := 0.0; -- *1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.1411);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.0;
--         Air_Density := 0.0023769;
--         Ambient_Temperature_Ratio := 1.0;
--         Dynamic_Pressure := 0.0;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Pmu_Check_Case = 12 then
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
----         Prop_Rpm := (1060.2,0.0,0.0);
--         Aircraft_Cg := (165.2900/12.0, 0.0/12.0, 91.31/12.0);
--         Aircraft_True_Airspeed := 255.16*1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(1.5);
----       Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.3999154;
--         Ambient_Temperature_Ratio := 0.9312473;
--         Ambient_Pressure_Ratio := 0.6876936;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Pmu_Check_Case = 17 then
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
----         Prop_Rpm := (1060.2,0.0,0.0);
--         Aircraft_Cg := (165.2900/12.0, 0.0/12.0, 91.31/12.0);
--         Aircraft_True_Airspeed := 368.5*1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(1.5);
----       Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.5999761;
--         Ambient_Temperature_Ratio := 0.8624945;
--         Ambient_Pressure_Ratio := 0.4595285;
--         Air_Density := 0.0023769*Ambient_Pressure_Ratio/Ambient_Temperature_Ratio;
--         Dynamic_Pressure := 0.5*Air_Density*Aircraft_True_Airspeed**2.0;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Pmu_Check_Case = 19 then
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
----         Prop_Rpm := (1060.2,0.0,0.0);
--         Aircraft_Cg := (164.7944/12.0, 0.5/12.0, 90.29/12.0);
--         Aircraft_True_Airspeed := 0.0; -- *1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.1411);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.0;
--         Air_Density := 0.0023769;
--         Ambient_Temperature_Ratio := 1.0;
--         Dynamic_Pressure := 0.0;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       elsif Pmu_Check_Case = 25 then
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
----         Prop_Rpm := (1060.2,0.0,0.0);
--         Aircraft_Cg := (164.7944/12.0, 0.5/12.0, 90.29/12.0);
--         Aircraft_True_Airspeed := 0.0; -- *1.689;  -- convert knots to ft/sec.
--         Aircraft_Angle_Of_Attack := Angle_Types.Degrees_To_Radians(2.1411);
--         Aircraft_Sideslip_Angle := Angle_Types.Degrees_To_Radians(0.0);
--         Aircraft_Angular_Rate := (0.0,0.0,0.0);
--         Mach := 0.0;
--         Air_Density := 0.0023769;
--         Ambient_Temperature_Ratio := 1.0;
--         Dynamic_Pressure := 0.0;
--         Aircraft_Velocity := (Aircraft_True_Airspeed*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Angle_Of_Attack)*Ada.Numerics.Elementary_Functions.Cos(Aircraft_Sideslip_Angle),0.0,0.0);
--       end if;

       --     Time_Step := Time_Step + 1;
     --     Ada.Text_Io.New_Line;
     --     Ada.Text_Io.Put ("Time Step := ");
     --     Ada.Text_Io.Put_Line(Integer'Image (Time_Step));
     Low_Pitch_Stop  := Hartzell_4_Blade_Propeller_Geometry.Low_Pitch_Stop(An_Instance => This_Subsystem.The_Propeller_Geometry);
     Propeller_Diameter := Hartzell_4_Blade_Propeller_Geometry.Propeller_Diameter(An_Instance => This_Subsystem.The_Propeller_Geometry);
     Wetted_Surface_Area := Hartzell_4_Blade_Propeller_Geometry.Wetted_Surface_Area(An_Instance => This_Subsystem.The_Propeller_Geometry);

     -- Hi prop rpm malfunction
     Malf_Np_Hi := JPATS_Propeller.Ios_Interface.This_Ios_Interface.Mfen20;

     -- Piu Failure
     Piu_Failure          := JPATS_Powerplant.PIU_Failure_Request;

     Propeller_Reset_Engine_Torque := JPATS_Powerplant.Engine_Torque_Pct*28.89;
     Propeller_Reset_Counter := 0;
     if Propeller_Reset then
       while (abs(Propeller_Reset_Engine_Torque - Axial_Torque_Absorbed(X)) > 0.5 )
         and Propeller_Reset_Counter < 50 loop
         Propeller_Reset_Counter := Propeller_Reset_Counter + 1;
         Overspeed_Governor_Reset_Solenoid := False;
         if (Propeller_Reset_Engine_Torque - Axial_Torque_Absorbed(X)) > 0.0 then
           Pmu_Blade_Angle_Rate := 0.02;
         else
           Pmu_Blade_Angle_Rate := -0.02;
         end if;

         Propeller_Interface_Unit.Update
           (Next_Instance                     => Next_Subsystem.The_Piu,
            This_Instance                     => This_Subsystem.The_Piu,
            Iconst                            => Iconst,
            Overspeed_Governor_Reset_Solenoid => Overspeed_Governor_Reset_Solenoid,
            Propeller_Feather_Solenoid        => Manual_Feather_Signal,
            PMU_Blade_Angle_Rate              => PMU_Blade_Angle_Rate,
            Prop_Rpm                          => Prop_Rpm,
            Engine_Oil_Pressure               => Engine_Oil_Pressure,
            Malf_Np_Lo                        => Malf_Np_Lo,
            Malf_Np_Hi                        => Malf_Np_Hi);

         Delta_Blade_Angle        := Propeller_Interface_Unit.Delta_Blade_Angle
           (An_Instance => Next_Subsystem.The_Piu);

         --|  Calculate the propeller orientation
         Prop_Orientation := Geometry.Rotational_Orientation
           (An_Instance => Next_Subsystem.The_Propeller_Installation_Geometry);

         --      Alpha_Propeller := 12.0;
         Alpha_Propeller :=
           Ada.Numerics.Elementary_Functions.Arccos((abs(Ada.Numerics.Elementary_Functions.Cos
                                                         (Aircraft_Angle_Of_Attack + Prop_Orientation(Vector_Types.Pitch)))
                                                     *Ada.Numerics.Elementary_Functions.Cos(-Aircraft_Sideslip_Angle
                                                                                            + Prop_Orientation(Vector_Types.Yaw))),360.0);  -- note prop orientation in yaw direction is specified in yaw, not beta. yaw=-beta


         Hartzell_4_Blade_Propeller.Update
           ( Next_Instance              => Next_Subsystem.The_Propeller,
             This_Instance              => This_Subsystem.The_Propeller,
             Iconst                     => Iconst,
             Air_Density                => Air_Density,
             Ambient_Temperature_Ratio  => Ambient_Temperature_Ratio,
             Mach                       => Mach,
             Propeller_Angle_Of_Attack  => Alpha_Propeller,
             Dynamic_Pressure           => Dynamic_Pressure,
             Aircraft_Angular_Rate      => Aircraft_Angular_Rate,
             Prop_Rpm                   => Prop_Rpm,
             Delta_Blade_Angle          => Delta_Blade_Angle,
             Aircraft_Forward_Velocity  => Aircraft_Velocity.X,
             Low_Pitch_Stop             => Low_Pitch_Stop,
             Propeller_Diameter         => Propeller_Diameter,
             Wetted_Surface_Area        => Wetted_Surface_Area,
             Degraded_Prop_Control      => Piu_Failure,
             Propeller_Reset            => Propeller_Reset,
             Quick_Start_Engine_Torque  => Propeller_Reset_Engine_Torque);

         Axial_Thrust := Hartzell_4_Blade_Propeller.Axial_Thrust
           (An_Instance => Next_Subsystem.The_Propeller);
         Axial_Torque_Absorbed := Hartzell_4_Blade_Propeller.Axial_Torque_Absorbed
           (An_Instance => Next_Subsystem.The_Propeller);
         Prop_Inertia := Hartzell_4_Blade_Propeller_Geometry.Propeller_Inertia_Ipgyro
           (An_Instance => Next_Subsystem.The_Propeller_Geometry);
       end loop;
       Geometry.Update
         (An_Instance           => Next_Subsystem.The_Propeller_Installation_Geometry,
          Axial_Thrust          => Axial_Thrust,
          Axial_Torque          => (0.0,0.0,0.0), --Float(-1.0)*Axial_Torque_Absorbed,
          Inertia               => Prop_Inertia,
          Rpm                   => (0.1,0.0,0.0),
          Aircraft_Angular_Rate => Aircraft_Angular_Rate,
          Aircraft_Cg           => Aircraft_Cg);
     else
       Propeller_Interface_Unit.Update
         (Next_Instance                     => Next_Subsystem.The_Piu,
          This_Instance                     => This_Subsystem.The_Piu,
          Iconst                            => Iconst,
          Overspeed_Governor_Reset_Solenoid => Overspeed_Governor_Reset_Solenoid,
          Propeller_Feather_Solenoid        => Manual_Feather_Signal,
          PMU_Blade_Angle_Rate              => PMU_Blade_Angle_Rate,
          Prop_Rpm                          => Prop_Rpm,
          Engine_Oil_Pressure               => Engine_Oil_Pressure,
          Malf_Np_Lo                        => Malf_Np_Lo,
          Malf_Np_Hi                        => Malf_Np_Hi);

       Delta_Blade_Angle        := Propeller_Interface_Unit.Delta_Blade_Angle
         (An_Instance => Next_Subsystem.The_Piu);

       --|  Calculate the propeller orientation
       Prop_Orientation := Geometry.Rotational_Orientation
         (An_Instance => Next_Subsystem.The_Propeller_Installation_Geometry);

       --      Alpha_Propeller := 12.0;
       Alpha_Propeller :=
         Ada.Numerics.Elementary_Functions.Arccos((abs(Ada.Numerics.Elementary_Functions.Cos
                                                       (Aircraft_Angle_Of_Attack + Prop_Orientation(Vector_Types.Pitch)))
                                                   *Ada.Numerics.Elementary_Functions.Cos(-Aircraft_Sideslip_Angle
                                                                                          + Prop_Orientation(Vector_Types.Yaw))),360.0);  -- note prop orientation in yaw direction is specified in yaw, not beta. yaw=-beta


       Hartzell_4_Blade_Propeller.Update
         ( Next_Instance              => Next_Subsystem.The_Propeller,
           This_Instance              => This_Subsystem.The_Propeller,
           Iconst                     => Iconst,
           Air_Density                => Air_Density,
           Ambient_Temperature_Ratio  => Ambient_Temperature_Ratio,
           Mach                       => Mach,
           Propeller_Angle_Of_Attack  => Alpha_Propeller,
           Dynamic_Pressure           => Dynamic_Pressure,
           Aircraft_Angular_Rate      => Aircraft_Angular_Rate,
           Prop_Rpm                   => Prop_Rpm,
           Delta_Blade_Angle          => Delta_Blade_Angle,
           Aircraft_Forward_Velocity  => Aircraft_Velocity.X,
           Low_Pitch_Stop             => Low_Pitch_Stop,
           Propeller_Diameter         => Propeller_Diameter,
           Wetted_Surface_Area        => Wetted_Surface_Area,
           Degraded_Prop_Control      => Piu_Failure,
           Propeller_Reset            => Propeller_Reset,
           Quick_Start_Engine_Torque  => Propeller_Reset_Engine_Torque);

       Axial_Thrust := Hartzell_4_Blade_Propeller.Axial_Thrust
         (An_Instance => Next_Subsystem.The_Propeller);
       Axial_Torque_Absorbed := Hartzell_4_Blade_Propeller.Axial_Torque_Absorbed
         (An_Instance => Next_Subsystem.The_Propeller);
       Prop_Inertia := Hartzell_4_Blade_Propeller_Geometry.Propeller_Inertia_Ipgyro
         (An_Instance => Next_Subsystem.The_Propeller_Geometry);

       Geometry.Update
         (An_Instance           => Next_Subsystem.The_Propeller_Installation_Geometry,
          Axial_Thrust          => Axial_Thrust,
          Axial_Torque          => Float(-1.0)*Axial_Torque_Absorbed,
          Inertia               => Prop_Inertia,
          Rpm                   => Prop_Rpm,
          Aircraft_Angular_Rate => Aircraft_Angular_Rate,
          Aircraft_Cg           => Aircraft_Cg);

     end if;

     Propeller_Force := Geometry.Force_Tensor
       (An_Instance => Jpats_Propeller.Container.This_Subsystem.The_Propeller_Installation_Geometry);
     Propeller_Moment := Geometry.Moment_Tensor
       (An_Instance => Jpats_Propeller.Container.This_Subsystem.The_Propeller_Installation_Geometry);
     Propeller_Gyro_Moment := Geometry.Gyroscopic_Moment_Tensor
       (An_Instance => Jpats_Propeller.Container.This_Subsystem.The_Propeller_Installation_Geometry);

     --     Container.Swap_Pointer := Container.This_Subsystem;
     --     Container.This_Subsystem := Container.Next_Subsystem;
     --     Container.Next_Subsystem := Container.Swap_Pointer;
     --     Container.Next_Subsystem.all := Container.This_Subsystem.all;

     if Propeller_Reset then
       Propeller_Reset := False;
     end if;

   end Update;

  procedure Update
      ( Integration_Constant : in Float ) is

  begin

    Update
      (Next_Subsystem => Container.Next_Subsystem,
       This_Subsystem => Container.This_Subsystem,
       Iconst         => Integration_Constant);

    Container.Swap_Ptr := Container.This_Subsystem;
    Container.This_Subsystem := Container.Next_Subsystem;
    Container.Next_Subsystem := Container.Swap_Ptr;
    Container.Next_Subsystem.all := Container.This_Subsystem.all;

  end Update;

-- Method Initialize
   procedure Initialize
     (A_Subsystem : in out Jpats_Propeller.Container.Instance_Ptr) is

   begin
     Propeller_Interface_Unit.Initialize
       (An_Instance => A_Subsystem.The_Piu);
     Hartzell_4_Blade_Propeller.Initialize
       (An_Instance => A_Subsystem.The_Propeller);
     Hartzell_4_Blade_Propeller_Geometry.Initialize
       (An_Instance => A_Subsystem.The_Propeller_Geometry);
     Geometry.Initialize
       (An_Instance => A_Subsystem.The_Propeller_Installation_Geometry);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Test",
       Units          => "Boolean",
       Object_Address => Propeller_Test'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Low_Pitch_Stop",
       Units          => "float",
       Object_Address => Low_Pitch_Stop'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Diameter",
       Units          => "float",
       Object_Address => Propeller_Diameter'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Wetted_Surface_Area",
       Units          => "float",
       Object_Address => Wetted_Surface_Area'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Aircraft_Velocity",
       Units          => "float",
       Object_Address => Aircraft_Velocity.X'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/delta_blade_angle",
       Units          => "float",
       Object_Address => Delta_Blade_angle'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Alpha_Propeller",
       Units          => "float",
       Object_Address => Alpha_Propeller'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Force(X)",
       Units          => "float",
       Object_Address => Propeller_Force(X)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Force(Y)",
       Units          => "float",
       Object_Address => Propeller_Force(Y)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Force(Z)",
       Units          => "float",
       Object_Address => Propeller_Force(Z)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Moment(X)",
       Units          => "float",
       Object_Address => Propeller_Moment(X)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Moment(Y)",
       Units          => "float",
       Object_Address => Propeller_Moment(Y)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_Moment(Z)",
       Units          => "float",
       Object_Address => Propeller_Moment(Z)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_gyro_moment(X)",
       Units          => "float",
       Object_Address => Propeller_gyro_moment(X)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_gyro_moment(Y)",
       Units          => "float",
       Object_Address => Propeller_gyro_moment(Y)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Propeller_gyro_moment(Z)",
       Units          => "float",
       Object_Address => Propeller_gyro_moment(Z)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
--    Stethoscope.Register_Signal
--      (Name           => "Propeller/Prop_Check_Case",
--       Units          => "integer",
--       Object_Address => Prop_Check_Case'Address,
--       Value_Type     => "int",
--       Scope_Index    => 0);
--    Stethoscope.Register_Signal
--      (Name           => "Propeller/Engine_Check_Case",
--       Units          => "integer",
--       Object_Address => Engine_Check_Case'Address,
--       Value_Type     => "int",
--       Scope_Index    => 0);
--    Stethoscope.Register_Signal
--      (Name           => "Propeller/Pmu_Check_Case",
--       Units          => "integer",
--       Object_Address => Pmu_Check_Case'Address,
--       Value_Type     => "int",
--       Scope_Index    => 0);
   end Initialize;

   -- Method Read_Tables
   procedure Read_Tables is
   begin
     Hartzell_4_Blade_Propeller.Read_Tables;
   end Read_Tables;

   procedure Initialize is
   begin
     Read_Tables;
     Initialize
       (A_Subsystem => Container.Next_Subsystem);
     Initialize
       (A_Subsystem => Container.This_Subsystem);

   end Initialize;

   begin
     null;
   exception
     when others =>
       Ada.Text_Io.Put_Line ("Error in Jpats_Propeller.Controller");
       raise;

   end JPATS_Propeller.Controller;



















