-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Ada.Streams.Stream_Io;
with Normalized_Types;
with Angle_Types;
with Torque_Types;
with Force_Types;
with Length_Types;
with Mass_Types;
with Jpats_Propeller_Types;
with Vector_Types;
use  Vector_Types;

with Hartzell_4_Blade_Propeller_Geometry;

package Hartzell_4_Blade_Propeller is

---------------------------------------------------------------------------
-- Reference: Kimball, D., "Flight Simulation Mathematical Model of
--            the Beech MkII Joint Primary Aircraft Training System
--            (JPATS)", Document 133E794 Rev. A, 18 February, 1998,
--            FSI-SSD # 50181, pg 66.
--
--
---------------------------------------------------------------------------

   type Instance is private;

-- Accessor Axial_Thrust
   function Axial_Thrust
     (An_Instance : in Instance) return Jpats_Propeller_Types.Force_Vector_Type.Instance;

-- Accessor Axial_Thrust_Coefficient_Limited
   function Aero_Axial_Thrust_Coefficient_Limited
     (An_Instance : in Instance) return Normalized_Types.Normalize;

-- Accessor Axial_Torque_Absorbed
   function Axial_Torque_Absorbed
     (An_Instance : in Instance) return Jpats_Propeller_Types.Torque_Vector_Type.Instance;
-- Accessor Total_Torque_Absorbed
   function Total_Torque_Absorbed
     (An_Instance : in Instance) return Jpats_Propeller_Types.Torque_Vector_Type.Instance;

-- Access Blade_Angle
   function Blade_Angle
     (An_Instance : Instance)
     return Angle_Types.Degrees;
   procedure Set_Blade_Angle
     (An_Instance : in out Instance;
      Blade_Angle : in     Angle_Types.Degrees);

  procedure Start_Recording
    (An_Instance  : in out Instance;
     Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access );

  procedure Stop_Recording
    (An_Instance : in out Instance);

-- Method Update
   procedure Update
     ( Next_Instance              : in out Instance;
       This_Instance              : in     Instance;
       Iconst                     : in     Float;
       Air_Density                : in     Mass_Types.Slugs_Per_Cubic_Feet;
       Ambient_Temperature_Ratio  : in     Normalized_Types.Normalize;
       Mach                       : in     Normalized_Types.Normalize;
       Propeller_Angle_Of_Attack  : in     Angle_Types.Degrees;
       Dynamic_Pressure           : in     Force_Types.Pounds_Per_Sq_Feet;
       Aircraft_Angular_Rate      : in     Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
       Prop_Rpm                   : in     Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
       Delta_Blade_Angle          : in     Angle_Types.Degrees;
       Aircraft_Forward_Velocity  : in     Length_Types.Feet_Per_Sec;
       Low_Pitch_Stop             : in     Angle_Types.Degrees;
       Propeller_Diameter         : in     Length_Types.Feet;
       Wetted_Surface_Area        : in     Length_Types.Sq_Feet;
       Degraded_Prop_Control      : in     Boolean;
       Propeller_Reset            : in     Boolean;
       Quick_Start_Engine_Torque  : in     Normalized_Types.Normalize);

   -- Method Initialize
   procedure Initialize (An_Instance : in out Instance);

   -- Method Read_Tables
   procedure Read_Tables;

private
  type Prop_Stream_Type is new Ada.Streams.Stream_IO.Stream_Access;

  procedure Nulloutput
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : in     Prop_Stream_Type);

  function Nullinput
    (Stream : access Ada.Streams.Root_Stream_Type'Class) return Prop_Stream_Type;

  procedure Nullread
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : out    Prop_Stream_Type);

  for Prop_Stream_Type'Write use Nulloutput;

  for Prop_Stream_Type'Read use Nullread;

  for Prop_Stream_Type'Output use Nulloutput;

  for Prop_Stream_Type'Input use Nullinput;

  type Instance is record
    Record_Debug_Information         : Boolean;
    Debug_Stream                     : Prop_Stream_Type; -- Ada.Streams.Stream_Io.Stream_Access;
    Hartzell_Geometry                : Hartzell_4_Blade_Propeller_Geometry.Instance;
    Prop_Rpm_Corrected               : Jpats_Propeller_Types.Propeller_Corrected_Angular_Rate_Vector_Type.Instance;
    Blade_Angle                      : Angle_Types.Degrees; -- Beta75R
    Blade_Angle_Hysterisis           : Angle_Types.Degrees;
    Blade_Angle_Step                 : Angle_Types.Degrees;
    Blade_Angle_Commanded            : Angle_Types.Degrees;
    Ship                             : Integer;
    Blade_Angle_Min                  : Angle_Types.Degrees;
    Blade_Angle_Max                  : Angle_Types.Degrees;
    Advance_Ratio                    : Normalized_Types.Normalize;
    Axial_Power_Coefficient          : Normalized_Types.Normalize;
    Total_Power_Coefficient          : Normalized_Types.Normalize;
    Axial_Torque_Absorbed            : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
    Total_Torque_Absorbed            : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
    Axial_Thrust_Coefficient         : Normalized_Types.Normalize;
    Aero_Axial_Thrust_Coefficient    : Normalized_Types.Normalize;
    Aero_Axial_Thrust_Coefficient_Limited : Normalized_Types.Normalize;
    Axial_Thrust                     : Jpats_Propeller_Types.Force_Vector_Type.Instance;
  end record;

end Hartzell_4_Blade_Propeller;




