-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Ada.Text_Io;
with Ada.Numerics;
use  Ada.Numerics;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;
with Interpolation_Table.Doubly_Indexed;
with Interpolation_Table.Singly_Indexed;
with Simulation_Dictionary;
with Stethoscope;
with Log;
with Jpats_Powerplant;

package body Hartzell_4_Blade_Propeller is

   Kcpaj_Table              : aliased Interpolation_Table.Singly_Indexed.Instance;
   Fcpaa_Table              : aliased Interpolation_Table.Singly_Indexed.Instance;
   Dcpn_Table               : aliased Interpolation_Table.Singly_Indexed.Instance;
   Kcpj_Table               : aliased Interpolation_Table.Singly_Indexed.Instance;
   Thrust_Coefficient_Table : aliased Interpolation_Table.Doubly_Indexed.Instance;
   Power_Coefficient_Table  : aliased Interpolation_Table.Doubly_Indexed.Instance;
   Ftmach_Table             : aliased Interpolation_Table.Doubly_Indexed.Instance;

   Test_Blade_Angle         : Boolean;
   Blade_Angle_During_Test  : Float := 21.5;

-- Accessor Axial_Thrust
   function Axial_Thrust
     (An_Instance : in Instance)
     return Jpats_Propeller_Types.Force_Vector_Type.Instance is
   begin
     return An_Instance.Axial_Thrust;
   end Axial_Thrust;

-- Accessor Axial_Thrust_Coefficient_Limited
   function Aero_Axial_Thrust_Coefficient_Limited
     (An_Instance : in Instance)
     return Normalized_Types.Normalize is
   begin
     return An_Instance.Aero_Axial_Thrust_Coefficient_Limited;
   end Aero_Axial_Thrust_Coefficient_Limited;


   -- Accessor Total_Torque
   function Total_Torque_Absorbed
     (An_Instance : in Instance)
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
   begin
     return An_Instance.Total_Torque_Absorbed;
   end Total_Torque_Absorbed;

   -- Accessor Axial_Torque
   function Axial_Torque_Absorbed
     (An_Instance : in Instance)
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
   begin
     return An_Instance.Axial_Torque_Absorbed;
   end Axial_Torque_Absorbed;

   -- Access_Blade_Angle
   function  Blade_Angle
     (An_Instance : in Instance )
      return Angle_Types.Degrees is
   begin
      return An_Instance.Blade_Angle;
   end Blade_Angle;

   procedure Set_Blade_Angle
     (An_Instance : in out Instance;
      Blade_Angle : in     Angle_Types.Degrees) is
   begin
      An_Instance.Blade_Angle := Blade_Angle;
   end Set_Blade_Angle;

   procedure Nulloutput
     (Stream : access Ada.Streams.Root_Stream_Type'Class;
      Item   : in     Prop_Stream_Type) is
   begin
     Integer'Write(Stream, 1);
   end Nulloutput;

   procedure Nullread
     (Stream : access Ada.Streams.Root_Stream_Type'Class;
      Item   : out    Prop_Stream_Type) is
     Dropped_Data : Integer;
   begin
     Integer'Read(Stream, Dropped_Data);
     Item := null;
   end Nullread;

  function Nullinput
    (Stream : access Ada.Streams.Root_Stream_Type'Class)
      return Prop_Stream_Type is
    Dropped_Data : Integer;
  begin
    Integer'Read(Stream, Dropped_Data);
    return null;
  end Nullinput;

  procedure Start_Recording
    (An_Instance  : in out Instance;
     Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access ) is

  begin
    An_Instance.Record_Debug_Information := True;
    An_Instance.Debug_Stream := Prop_Stream_Type(Debug_Stream);
    Log.Report("Recording Hartzell Propeller");
  end Start_Recording;

  procedure Stop_Recording
    (An_Instance : in out Instance) is
  begin
    An_Instance.Record_Debug_Information := False;
  end Stop_Recording;

-- Method Calculate_Thrust_Coefficient
   procedure Calculate_Thrust_Coefficient
     ( An_Instance         : in out Instance;
       Air_Density         : in     Mass_Types.Slugs_Per_Cubic_Feet;
       Prop_Diameter       : in     Length_Types.Feet;
       Dynamic_Pressure    : in     Force_Types.Pounds_Per_Sq_Feet;
       Wetted_Surface_Area : in     Length_Types.Sq_Feet;
       Mach                : in     Normalized_Types.Normalize;
       Advance_Ratio       : in     Normalized_Types.Normalize;
       Blade_Angle         : in     Angle_Types.Degrees)
   is
     Ct_Axial : Normalized_Types.Normalize                            renames An_Instance.Axial_Thrust_Coefficient;
     T_Axial  : Jpats_Propeller_Types.Force_Vector_Type.Instance      renames An_Instance.Axial_Thrust;
     Rho      : Mass_Types.Slugs_Per_Cubic_Feet                       renames Air_Density;
     Tc       : Normalized_Types.Normalize                            renames An_Instance.Aero_Axial_Thrust_Coefficient;
     Tc_Lim   : Normalized_Types.Normalize                            renames An_Instance.Aero_Axial_Thrust_Coefficient_Limited;
     Qbar     : Force_Types.Pounds_Per_Sq_Feet                        renames Dynamic_Pressure;
     Sw       : Length_Types.Sq_Feet                                  renames Wetted_Surface_Area;
     Nc       : Jpats_Propeller_Types.Propeller_Corrected_Angular_Rate_Vector_Type.Instance renames An_Instance.Prop_Rpm_Corrected;
     D        : Length_Types.Feet                                     renames Prop_Diameter;
     Ftmach   : Normalized_Types.Normalize;
   begin

-- Lookup the propeller thrust coefficient for axial
-- flow conditions, as a function of advance ratio and
-- blade pitch at the 3/4 radius station.
--
-- Table Thrust_Coefficient_Table
-- Prop thrust coefficient vs. Beta75R, at J' breakpoints of
--  0, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1,
--   1.2, 1.3, 1.4, 1.6, 1.8, 2.2, 2.4, 2.6, 2.8
--
-- Ct_Axial := Table CTJ = f(Beta75R, J')
   Ct_Axial := Interpolation_Table.Doubly_Indexed.Interpolate
     ( X => Blade_Angle,
       Y => Advance_Ratio,
       Table   => Thrust_Coefficient_Table'Access);

--| Calculate the thrust generated by the prop under conditions
--| of axial flow, the resulting thrust coefficient under axial
--| flow conditions and under the actual operating condition:

--| At some operating conditions compressibility effects reduce propeller
--| efficiency. Although these conditions are not normally encountered in
--| the T-6A flight envelope, they can occur in high-speed dives and in
--| extreme sub-ISA temperatures. To incorporate this effect, a factor is
--| applied to the lookup thrust value. The factor is a table lookup from
--| new table set FTMACH, and is a function of Mach number and
--| advance ratio. Also, the prop feathered drag coefficient is modified
--| based on flight test matching of propeller feathered sinks. Step 4
--| changes as follows:
   Ftmach := Interpolation_Table.Doubly_Indexed.Interpolate
     ( X => Advance_Ratio,
       Y => Mach,
       Table  => Ftmach_Table'Access);


   if An_Instance.Blade_Angle < 80.0 then
      T_Axial(X) := Ct_Axial*Rho*(Float(Nc(X))**Float(2.0))*(D**4.0)*Ftmach;
   else
      T_Axial(X) := -0.0060*Qbar*Sw;
   end if;

   Tc := T_Axial(X) / (Float'Max(1.0,Qbar) * Sw);  --
   Tc_lim := Float'max(-0.1, Float'min (5.0, Tc));

   exception
     when others =>
       Ada.Text_Io.Put_Line("An exception occurred in Hartzell_4_Blade_Propeller.Calculate_Thrust_Coefficient.");
       raise;

   end Calculate_Thrust_Coefficient;

-- Method Calculate_Power_Coefficient
   procedure Calculate_Power_Coefficient
    ( Next_Instance             : in out Instance;
      This_Instance             : in     Instance;
      Propeller_Angle_Of_Attack : in     Angle_Types.Degrees;
      Low_Pitch_Stop            : in     Angle_Types.Degrees) is

   Beta75r           : Angle_Types.Degrees                                    renames Next_Instance.Blade_Angle;
   Beta75r_Last      : Angle_Types.Degrees                                    renames This_Instance.Blade_Angle;
   J                 : Normalized_Types.Normalize                             renames Next_Instance.Advance_Ratio;
   Nc                : Jpats_Propeller_Types.Propeller_Corrected_Angular_Rate_Vector_Type.Instance  renames Next_Instance.Prop_Rpm_Corrected;
   Cp                : Normalized_Types.Normalize                             renames Next_Instance.Total_Power_Coefficient;
   Cp_Axial          : Normalized_Types.Normalize                             renames Next_Instance.Axial_Power_Coefficient;
   Delta_Cp_Alpha    : Float;
   K_Cp_Alpha        : Float;
   F_Cp_Alpha        : Float;
   Delta_Cp_N        : Float;
   K_Cp_J            : Float;
   Delta_Cp_Minpitch : Float;
   begin

-- Lookup the propeller power coefficient for axial
-- flow conditions, as a function of advance ratio and
-- blade pitch at the 3/4 radius station.
--
-- Table CPJ
-- Prop power coefficient vs. Beta75R, at J' breakpoints of
--  0, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1,
--   1.2, 1.3, 1.4, 1.6, 1.8, 2.2, 2.4, 2.6, 2.8
--
--  Axial_Power_Coefficient :=  Table CPJ = f(Beta75R, J')
    Cp_Axial := Interpolation_Table.Doubly_Indexed.Interpolate
      ( X => Beta75r,
        Y => J,
        Table   => Power_Coefficient_Table'Access);
--|
--| Calculate the correction to propeller power coefficient
--| for non-axial flow into the propeller. Sum this with the
--| axial flow power coefficient to obtain the power
--| coefficient for the operating condition:
--|
--| K_Cp_Alpha := Table KCPAJ = f(J')
    K_Cp_Alpha := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => J,
        Table => Kcpaj_Table'Access );

--| F_Cp_Alpha := Table FCPAA = f(abs(alphap))
    F_Cp_Alpha := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => abs(Propeller_Angle_Of_Attack),
        Table => Fcpaa_Table'Access );
    Delta_Cp_Alpha  := K_Cp_Alpha*F_Cp_Alpha;

--| Delta_Cp_N := Table DCPN = f(n')
    Delta_Cp_N := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => Nc(X),
        Table => DCPN_Table'Access );

--| K_Cp_J := Table KCPJ = f(J')
    K_Cp_J := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => J,
        Table => KCPJ_Table'Access );

    if Beta75r <= Low_Pitch_Stop  then
      Delta_Cp_Minpitch := Delta_Cp_N * K_Cp_J;
    else
      Delta_Cp_Minpitch := 0.0;
    end if;

    Cp := Cp_Axial + Delta_Cp_Alpha + Delta_Cp_Minpitch;
   exception
     when others =>
       Ada.Text_Io.Put_Line("An exception occurred in Hartzell_4_Blade_Propeller.Calculate_Power_Coefficient.");
       raise;

   end Calculate_Power_Coefficient;

-- Method Calculate_Propeller_Torque_Absorbed
   procedure Calculate_Propeller_Torque_Absorbed
     ( An_Instance             : in out Instance;
       Air_Density             : in     Mass_Types.Slugs_Per_Cubic_Feet;
       Prop_Diameter           : in     Length_Types.Feet) is

   Rho      : Float renames Air_Density;
   Cp       : Float renames An_Instance.Total_Power_Coefficient;
   Cp_Axial : Float renames An_Instance.Axial_Power_Coefficient;
   D        : Float renames Prop_Diameter;
   Nc       : Float renames An_Instance.Prop_Rpm_Corrected(X);
   Qaxial   : Float renames An_Instance.Axial_Torque_Absorbed(X); -- Axial torque absorbed in X axis of propeller
   Qprop    : Float renames An_Instance.Total_Torque_Absorbed(X); -- Total torque absorbed

   begin

-- Calculate the aerodynamic torque absorbed by the propeller
-- under axial flow conditions and under the actual operating
-- condition:

    Qaxial := (Cp_Axial*Rho*(Nc**2.0)*(D**5.0))/(2.0*pi);
    Qprop  := (Cp*Rho*(Nc**2.0)*(D**5.0))/(2.0*pi); -- about prop axis

   exception
     when others =>
       Ada.Text_Io.Put_Line("An exception occurred in Hartzell_4_Blade_Propeller.Calculate_Propeller_Torque_Absorbed.");
       raise;

   end Calculate_Propeller_Torque_Absorbed;

   -- Method Update
   procedure Update
     ( Next_Instance              : in out Instance;
       This_Instance              : in     Instance;
       Iconst                     : in     Float;
       Air_Density                : in     Mass_Types.Slugs_Per_Cubic_Feet;
       Ambient_Temperature_Ratio  : in     Normalized_Types.Normalize;
       Mach                       : in     Normalized_Types.Normalize;
       Propeller_Angle_Of_Attack  : in     Angle_Types.Degrees;
       Dynamic_Pressure           : in     Force_Types.Pounds_Per_Sq_Feet;
       Aircraft_Angular_Rate      : in     Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
       Prop_Rpm                   : in     Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
       Delta_Blade_Angle          : in     Angle_Types.Degrees_Per_Sec;
       Aircraft_Forward_Velocity  : in     Length_Types.Feet_Per_Sec;
       Low_Pitch_Stop             : in     Angle_Types.Degrees;
       Propeller_Diameter         : in     Length_Types.Feet;
       Wetted_Surface_Area        : in     Length_Types.Sq_Feet;
       Degraded_Prop_Control      : in     Boolean;
       Propeller_Reset            : in     Boolean;
       Quick_Start_Engine_Torque  : in     Normalized_Types.Normalize) is

     Nc    : Jpats_Propeller_Types.Propeller_Corrected_Angular_Rate_Vector_Type.Instance renames Next_Instance.Prop_Rpm_Corrected;
     J     : Normalized_Types.Normalize                            renames Next_Instance.Advance_Ratio;
     P     : Angle_Types.Radians_Per_Sec                           renames Aircraft_Angular_Rate(Roll);
     D     : Length_Types.Feet                                     renames Propeller_Diameter;
     Sw    : Length_Types.Sq_Feet                                  renames Wetted_Surface_Area;
     Gamma : constant Float := 1.4;
     Gas_Constant : constant Float := 1716.16;

   begin
--|  Calculate and limit the dynamic propeller advance ratio,
--|  based on the propeller speed corrected for roll rate.  Also calculate
--|  the angle of propeller shaft inclination from the freestream, and the

--|  propeller hub moment arms relative to the aircraft center of gravity.
--|
--|  corrected prop speed in revolutions per second
     Nc(X) :=  Angle_Types.Revolutions_Per_Sec'Max(0.1,((Float(Prop_Rpm(X))/60.0) + Float(P)/(2.0*Pi)));  -- limited to a minimum of 6 rpm.
     J := Normalized_Types.Normalize(Float'max(0.0, Float'min( 25.0,Float(Aircraft_Forward_Velocity)/(Nc(X)*Float(D)))));

     if Next_Instance.Ship < 3 then
       Next_Instance.Blade_Angle_Min := Angle_Types.Degrees(10.5);
     else
       if Next_Instance.Ship >= 3 then
         Next_Instance.Blade_Angle_Min := Angle_Types.Degrees(9.9);
       end if;
     end if;

     if not Degraded_Prop_Control then

--       if Propeller_Reset then
--         Next_Instance.Blade_Angle := Next_Instance.Blade_Angle_Min;
           --Angle_Types.Degrees'Max(Next_Instance.Blade_Angle_Min, Angle_Types.Degrees'Min
           --                                                   (Next_Instance.Blade_Angle_Max,
            --                                                   ((Next_Instance.Blade_Angle_Max - Next_Instance.Blade_Angle_Min)/
            --                                                    ((100.0 - 4.0)*28.89)*Quick_Start_Engine_Torque) - 10.0));
--       else
            if not (Jpats_Powerplant.Mfen29 and not jpats_Powerplant.Engine_Fired) then
               Next_Instance.Blade_Angle := Angle_Types.Degrees'Max(Next_Instance.Blade_Angle_Min, Angle_Types.Degrees'Min
                                                              (Next_Instance.Blade_Angle_Max, This_Instance.Blade_Angle + Delta_Blade_Angle*Iconst));
               Next_Instance.Blade_Angle_Commanded := Next_Instance.Blade_Angle;
               Next_Instance.Blade_Angle_Hysterisis := 1.0;
            end if;
--       end if;
     else
       -- Introduce hysterisis and degraded rate of motion into prop pitch
       -- response ( due to oil contamination).  The use of the (Rpm - int(Rpm))
       -- function introduces a degree of randomness in both the step prop pitch change
       -- and the intervals between pitch change motions.
       declare
         Prop_Rpm_Cmd : Float;
       begin


         Next_Instance.Blade_Angle_Commanded := Angle_Types.Degrees'Max(Next_Instance.Blade_Angle_Min, Angle_Types.Degrees'Min
                                                                       (Next_Instance.Blade_Angle_Max, This_Instance.Blade_Angle_Commanded
                                                                        + 0.15 * Delta_Blade_Angle * Iconst));
       end;
       if ((Next_Instance.Blade_Angle_Commanded - Next_Instance.Blade_Angle) >= Next_Instance.Blade_Angle_Hysterisis) then
         Next_Instance.Blade_Angle_Step := 7.5*(Prop_Rpm(X)  - Float(Integer(Prop_Rpm(X))));
         Next_Instance.Blade_Angle_Hysterisis := 1.0/(Float'Max(0.2,Float'Min(2.0,Next_Instance.Blade_Angle_Step)));
       elsif ((Next_Instance.Blade_Angle - Next_Instance.Blade_Angle_Commanded) >= Next_Instance.Blade_Angle_Hysterisis) then
         Next_Instance.Blade_Angle_Step := -7.5*(Prop_Rpm(X) - Float(Integer(Prop_Rpm(X))));
         Next_Instance.Blade_Angle_Hysterisis := 1.0/(Float'Max(0.2,Float'Min(2.0,-1.0*Next_Instance.Blade_Angle_Step)));
       else
         Next_Instance.Blade_Angle_Step := 0.0;
       end if;
       Next_Instance.Blade_Angle := Angle_Types.Degrees'Max(Next_Instance.Blade_Angle_Min, Angle_Types.Degrees'Min
                                                            (Next_Instance.Blade_Angle_Max, This_Instance.Blade_Angle + Next_Instance.Blade_Angle_Step));
       Next_Instance.Blade_Angle_Commanded := Next_Instance.Blade_Angle_Commanded - Next_Instance.Blade_Angle_Step;
     end if;

     if Test_Blade_Angle then
       Next_Instance.Blade_Angle := Blade_Angle_During_Test;
     end if;

     --|  Calculate the Aerodynamics of the propeller
     Calculate_Thrust_Coefficient
       ( An_Instance         => Next_Instance,
         Air_Density         => Air_Density,
         Prop_Diameter       => D,
         Dynamic_Pressure    => Dynamic_Pressure,
         Wetted_Surface_Area => Sw,
         Mach                => Mach,
         Advance_Ratio       => Next_Instance.Advance_Ratio,
         Blade_Angle         => Next_Instance.Blade_Angle);

     Calculate_Power_Coefficient
       ( Next_Instance             => Next_Instance,
         This_Instance             => This_Instance,
         Propeller_Angle_Of_Attack => Propeller_Angle_Of_Attack,
         Low_Pitch_Stop            => Low_Pitch_Stop);

     Calculate_Propeller_Torque_Absorbed
        ( An_Instance    => Next_Instance,
          Air_Density    => Air_Density,
          Prop_Diameter  => D );

     if Next_Instance.Record_Debug_Information then
       declare
         Blade_Angle_Label : String := "Blade_Angle";
       begin
         String'Output(Next_Instance.Debug_Stream, Blade_Angle_Label);
         Float'Output(Next_Instance.Debug_Stream, Next_Instance.Blade_Angle);
       end;
     end if;




   exception
     when others =>
       Ada.Text_Io.Put_Line("An exception occurred in Hartzell_4_Blade_Propeller.Update.");
       raise;

   end Update;


-- Method Initialize
   procedure Initialize
     (An_Instance : in out Instance) is
   begin
     An_Instance.Prop_Rpm_Corrected               := (0.0,0.0,0.0); -- Jpats_Propeller_Types.Propeller_Corrected_Angular_Rate_Vector_Type.Instance;
     An_Instance.Blade_Angle                      := Angle_Types.Degrees(84.0);
     An_Instance.Ship                             := 3;
     An_Instance.Blade_Angle_Min                  := Angle_Types.Degrees(9.9);
     An_Instance.Blade_Angle_Max                  := Angle_Types.Degrees(84.0);
     An_Instance.Advance_Ratio                    := Normalized_Types.Normalize(0.0);
     An_Instance.Axial_Power_Coefficient          := Normalized_Types.Normalize(0.0);
     An_Instance.Total_Power_Coefficient          := Normalized_Types.Normalize(0.0);
     An_Instance.Axial_Torque_Absorbed            := (0.0,0.0,0.0); -- Jpats_Propeller_Types.Torque_Vector_Type.Instance;
     An_Instance.Total_Torque_Absorbed            := (0.0,0.0,0.0); -- Jpats_Propeller_Types.Torque_Vector_Type.Instance;
     An_Instance.Axial_Thrust_Coefficient         := Normalized_Types.Normalize(0.0);
     An_Instance.Aero_Axial_Thrust_Coefficient_Limited := Normalized_Types.Normalize(0.0);
     An_Instance.Axial_Thrust                     := (0.0,0.0,0.0); -- Jpats_Propeller_Types.Force_Vector_Type.Instance
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Prop_Rpm_Corrected",
       Units          => "float",
       Object_Address => An_Instance.Prop_Rpm_Corrected'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Beta075Rmin",
       Units          => "float",
       Object_Address => An_Instance.Blade_Angle_Min'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Beta075R",
       Units          => "float",
       Object_Address => An_Instance.Blade_Angle'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/J",
       Units          => "float",
       Object_Address => An_Instance.Advance_Ratio'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Axial_Thrust_Coefficient",
       Units          => "float",
       Object_Address => An_Instance.Axial_Thrust_Coefficient'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Axial_Power_Coefficient",
       Units          => "float",
       Object_Address => An_Instance.Axial_Power_Coefficient'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Total_Power_Coefficient",
       Units          => "float",
       Object_Address => An_Instance.Total_Power_Coefficient'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Axial_Thrust",
       Units          => "float",
       Object_Address => An_Instance.Axial_Thrust'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Aero_Axial_Thrust_Coefficient",
       Units          => "float",
       Object_Address => An_Instance.Aero_Axial_Thrust_Coefficient'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Aero_Axial_Thrust_Coefficient_Limited",
       Units          => "float",
       Object_Address => An_Instance.Aero_Axial_Thrust_Coefficient_Limited'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Axial_Torque_Absorbed(X)",
       Units          => "float",
       Object_Address => An_Instance.Axial_Torque_Absorbed(X)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Axial_Torque_Absorbed(Y)",
       Units          => "float",
       Object_Address => An_Instance.Axial_Torque_Absorbed(Y)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Axial_Torque_Absorbed(Z)",
       Units          => "float",
       Object_Address => An_Instance.Axial_Torque_Absorbed(Z)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Total_Torque_Absorbed(X)",
       Units          => "float",
       Object_Address => An_Instance.Total_Torque_Absorbed(X)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Total_Torque_Absorbed(Y)",
       Units          => "float",
       Object_Address => An_Instance.Total_Torque_Absorbed(Y)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Total_Torque_Absorbed(Z)",
       Units          => "float",
       Object_Address => An_Instance.Total_Torque_Absorbed(Z)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Test_Blade_Angle",
       Units          => "boolean",
       Object_Address => Test_Blade_Angle'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Propeller/Hartzell/local/Blade_Angle_During_Test",
       Units          => "float",
       Object_Address => Blade_Angle_During_Test'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
   end Initialize;

   procedure Read_Tables is

     Propeller_Cat_Path  : String := Simulation_Dictionary.Lookup ("Propeller_Cat_Path");

   begin

     Interpolation_Table.Read
       ( File_Name => Propeller_Cat_Path & "fcpaa.ito",
         Table     =>  Fcpaa_Table );
     Interpolation_Table.Read
       ( File_Name => Propeller_Cat_Path & "kcpaj.ito",
         Table     =>  Kcpaj_Table );
     Interpolation_Table.Read
       ( File_Name => Propeller_Cat_Path & "dcpn.ito",
         Table     =>  Dcpn_Table );
     Interpolation_Table.Read
       ( File_Name => Propeller_Cat_Path & "kcpj.ito",
         Table     =>  Kcpj_Table );
     Interpolation_Table.Read
       ( File_Name => Propeller_Cat_Path & "ctj.ito",
         Table     =>  Thrust_Coefficient_Table );
     Interpolation_Table.Read
       ( File_Name => Propeller_Cat_Path & "cpj.ito",
         Table     =>  Power_Coefficient_Table );
     Interpolation_Table.Read
       ( File_Name => Propeller_Cat_Path & "ftmach.ito",
         Table     =>  Ftmach_Table );
   end Read_Tables;

end Hartzell_4_Blade_Propeller;
