-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer: Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Ada.Text_Io;
with Angle_Types;
with Ada.Numerics;
use  Ada.Numerics;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;

package body Geometry is

  -- Access Thrust_Reference_Point
   function  Thrust_Reference_Point
     (An_Instance : in Instance)
      return Jpats_Propeller_Types.Coordinate_Type.Instance is
   begin
      return An_Instance.Reference_Point;
   end Thrust_Reference_Point;

   procedure Set_Thrust_Reference_Point
     (An_Instance : in out Instance;
      Reference_Point : in    Jpats_Propeller_Types.Coordinate_Type.Instance) is
   begin
      An_Instance.Reference_Point  := Reference_Point;
   end Set_Thrust_Reference_Point;


-- Access Rotational_Orientation
   function  Rotational_Orientation
     (An_Instance : in Instance)
      return Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance is
   begin
      return An_Instance.Coordinate_Rotation; -- radians
   end Rotational_Orientation;

   procedure Set_Rotational_Orientation
     (An_Instance : in out Instance;
      Rotate      : in     Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance) is
   begin
      An_Instance.Coordinate_Rotation := Rotate;
   end Set_Rotational_Orientation;

-- Access Moment_Arm
   function  Moment_Arm
     (An_Instance : in Instance)
      return Jpats_Propeller_Types.Coordinate_Type.Instance is
   begin
      return An_Instance.Moment_Arm;
   end Moment_Arm;

   procedure Set_Moment_Arm
     (An_Instance : in out Instance;
      Moment_Arm  : in     Jpats_Propeller_Types.Coordinate_Type.Instance ) is
   begin
      An_Instance.Moment_Arm := Moment_Arm;
   end Set_Moment_Arm;


-- Access Moment_Tensor
   function  Moment_Tensor
     (An_Instance : in Instance)
      return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
   begin
      return An_Instance.Moment_Tensor;
   end Moment_Tensor;

   procedure Set_Moment_Tensor
     (An_Instance   : in out Instance;
      Moment_Tensor : in     Jpats_Propeller_Types.Torque_Vector_Type.Instance) is
   begin
      An_Instance.Moment_Tensor := Moment_Tensor;
   end Set_Moment_Tensor;

-- Access Force_Tensor
   function  Force_Tensor
     (An_Instance : in Instance)
      return Jpats_Propeller_Types.Force_Vector_Type.Instance is
   begin
      return An_Instance.Force_Tensor;
   end Force_Tensor;

   procedure Set_Force_Tensor
     (An_Instance : in out Instance;
      Force       : in     Jpats_Propeller_Types.Force_Vector_Type.Instance) is
   begin
      An_Instance.Force_Tensor := Force;
   end Set_Force_Tensor;

-- Access Gyroscopic_Moment_Tensor
   function  Gyroscopic_Moment_Tensor
     (An_Instance : in Instance)
      return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
   begin
      return An_Instance.Gyroscopic_Moment_Tensor;
   end Gyroscopic_Moment_Tensor;

   procedure Set_Gyroscopic_Moment_Tensor
     (An_Instance              : in out Instance;
      Gyroscopic_Moment_Tensor : in     Jpats_Propeller_Types.Torque_Vector_Type.Instance) is
   begin
      An_Instance.Gyroscopic_Moment_Tensor := Gyroscopic_Moment_Tensor;
   end Set_Gyroscopic_Moment_Tensor;

-- Method Update
   procedure Update
     (An_Instance           : in out Instance;
      Axial_Thrust          : in     Jpats_Propeller_Types.Force_Vector_Type.Instance;
      Axial_Torque          : in     Jpats_Propeller_Types.Torque_Vector_Type.Instance;
      Inertia               : in     Mass_Types.Slugs_Feet_Sq;
      Rpm                   : in     Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
      Aircraft_Angular_Rate : in     Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
      Aircraft_Cg           : in     Jpats_Propeller_Types.Coordinate_Type.Instance) is
   begin
     An_Instance.Force_Tensor  := Calculate_Body_Force
       ( An_Instance  => An_Instance,
         Axial_Thrust => Axial_Thrust);
     An_Instance.Moment_Arm    := Calculate_Moment_Arms
       (An_Instance => An_Instance,
        Aircraft_Cg => Aircraft_Cg);
     An_Instance.Moment_Tensor := Calculate_Body_Moment
       ( An_Instance => An_Instance,
         Axial_Torque => Axial_Torque );
     An_Instance.Gyroscopic_Moment_Tensor :=  Calculate_Gyro_Moment
       (An_Instance           => An_Instance,
        Inertia               => Inertia,
        Rpm                   => Rpm,
        Aircraft_Angular_Rate => Aircraft_Angular_Rate);

   end Update;


   --  Method Rotate Coordinate System
   --  These transfomations rotate from Body Axis to Wind Axis
   --  or Earth Axis to Body Axis.
   function Rotate_Coordinate_System
     (Cs        : in Jpats_Propeller_Types.Force_Vector_Type.Instance;
      Rotate    : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance)
     return Jpats_Propeller_Types.Force_Vector_Type.Instance is

      Phi        : Angle_Types.Degrees renames Rotate(Vector_Types.Roll);
      Theta      : Angle_Types.Degrees renames Rotate(Vector_Types.Pitch);
      Psi        : Angle_Types.Degrees renames Rotate(Vector_Types.Yaw);

   begin
        return
          (( Cs(Vector_Types.X)*Cos(Theta)*Cos(Psi) +
             Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Cos(Psi)-Cos(Phi)*Sin(Psi)) +
             Cs(Vector_Types.Z)*(Cos(Psi)*Sin(Theta)*Cos(Phi)+Sin(Phi)*Sin(Psi))),
           ( Cs(Vector_Types.X)*Cos(Theta)*Sin(Psi) +
             Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Sin(Psi)+Cos(Phi)*Cos(Psi)) +
             Cs(Vector_Types.Z)*(Cos(Phi)*Sin(Theta)*Sin(Psi) - Sin(Phi)*Cos(Psi)) ),
           (-Cs(Vector_Types.X)*Sin(Theta)          +
            Cs(Vector_Types.Y)* Sin(Phi)*Cos(Theta) +
            Cs(Vector_Types.Z)* Cos(Phi)*Cos(Theta) ));
   end Rotate_Coordinate_System;

      function Rotate_Coordinate_System
     (Cs        : in Jpats_Propeller_Types.Torque_Vector_Type.Instance;
      Rotate    : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance)
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is

      Phi        : Angle_Types.Degrees renames Rotate(Vector_Types.Roll);
      Theta      : Angle_Types.Degrees renames Rotate(Vector_Types.Pitch);
      Psi        : Angle_Types.Degrees renames Rotate(Vector_Types.Yaw);

   begin
      return
        (( Cs(Vector_Types.X)*Cos(Theta)*Cos(Psi) +
           Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Cos(Psi)-Cos(Phi)*Sin(Psi)) +
           Cs(Vector_Types.Z)*(Cos(Psi)*Sin(Theta)*Cos(Phi)+Sin(Phi)*Sin(Psi))),
         ( Cs(Vector_Types.X)*Cos(Theta)*Sin(Psi) +
           Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Sin(Psi)+Cos(Phi)*Cos(Psi)) +
           Cs(Vector_Types.Z)*(Cos(Phi)*Sin(Theta)*Sin(Psi) - Sin(Phi)*Cos(Psi)) ),
         (-Cs(Vector_Types.X)*Sin(Theta)          +
          Cs(Vector_Types.Y)* Sin(Phi)*Cos(Theta) +
          Cs(Vector_Types.Z)* Cos(Phi)*Cos(Theta) ));
   end Rotate_Coordinate_System;

   function Rotate_Coordinate_System
     (Cs        : in Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
      Rotate    : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance)
     return Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance is

      Phi        : Angle_Types.Degrees renames Rotate(Vector_Types.Roll);
      Theta      : Angle_Types.Degrees renames Rotate(Vector_Types.Pitch);
      Psi        : Angle_Types.Degrees renames Rotate(Vector_Types.Yaw);

   begin
      return
        (( Cs(Vector_Types.X)*Cos(Theta)*Cos(Psi) +
           Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Cos(Psi)-Cos(Phi)*Sin(Psi)) +
           Cs(Vector_Types.Z)*(Cos(Psi)*Sin(Theta)*Cos(Phi)+Sin(Phi)*Sin(Psi))),
         ( Cs(Vector_Types.X)*Cos(Theta)*Sin(Psi) +
           Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Sin(Psi)+Cos(Phi)*Cos(Psi)) +
           Cs(Vector_Types.Z)*(Cos(Phi)*Sin(Theta)*Sin(Psi) - Sin(Phi)*Cos(Psi)) ),
         (-Cs(Vector_Types.X)*Sin(Theta)          +
          Cs(Vector_Types.Y)* Sin(Phi)*Cos(Theta) +
          Cs(Vector_Types.Z)* Cos(Phi)*Cos(Theta) ));
   end Rotate_Coordinate_System;

   function Translate_Coordinate_System
     (Cs        : in Jpats_Propeller_Types.Force_Vector_Type.Instance;
      Translate : in Jpats_Propeller_Types.Coordinate_Translation_Type.Instance)
     return Jpats_Propeller_Types.Force_Vector_Type.Instance is


   begin
      return
        (( Cs(Vector_Types.X) + Translate(Vector_Types.X)),
         ( Cs(Vector_Types.Y) + Translate(Vector_Types.Y)),
         ( Cs(Vector_Types.Z) + Translate(Vector_Types.Z)));
   end Translate_Coordinate_System;

   function Translate_Coordinate_System
     (Cs        : in Jpats_Propeller_Types.Torque_Vector_Type.Instance;
      Translate : in Jpats_Propeller_Types.Coordinate_Translation_Type.Instance)
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is


   begin
      return
        (( Cs(Vector_Types.X) + Translate(Vector_Types.X)),
         ( Cs(Vector_Types.Y) + Translate(Vector_Types.Y)),
         ( Cs(Vector_Types.Z) + Translate(Vector_Types.Z)));
   end Translate_Coordinate_System;

   function Translate_Coordinate_System
     (Cs        : in Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
      Translate : in Jpats_Propeller_Types.Coordinate_Translation_Type.Instance)
     return Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance is


   begin
      return
        (( Cs(Vector_Types.X) + Translate(Vector_Types.X)),
         ( Cs(Vector_Types.Y) + Translate(Vector_Types.Y)),
         ( Cs(Vector_Types.Z) + Translate(Vector_Types.Z)));
   end Translate_Coordinate_System;

--   function Coordinate_System_Transform
--     (Cs        : in Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
--      Rotate    : in Jpats_Propeller_Types.Coordinate_Rotation_Type.Instance;
--      Translate : in Jpats_Propeller_Types.Coordinate_Translation_Type.Instance)
--     return Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance is
--
--     Phi        : Angle_Types.Degrees renames Rotate(Vector_Types.Roll);
--     Theta      : Angle_Types.Degrees renames Rotate(Vector_Types.Pitch);
--     Psi        : Angle_Types.Degrees renames Rotate(Vector_Types.Yaw);
--
--   begin
--     return
--       (( Cs(Vector_Types.X)*Cos(Theta)*Cos(Psi) +
--          Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Cos(Psi)-Cos(Phi)*Sin(Psi)) +
--          Cs(Vector_Types.Z)* Sin(Phi)*Sin(Psi) + Translate(Vector_Types.X)),
--        ( Cs(Vector_Types.X)*Cos(Theta)*Sin(Psi) +
--          Cs(Vector_Types.Y)*(Sin(Phi)*Sin(Theta)*Sin(Psi)+Cos(Phi)*Cos(Psi)) +
--          Cs(Vector_Types.Z)*(Cos(Phi)*Sin(Theta)*Sin(Psi) - Sin(Psi)*Sin(Psi)) +  Translate(Vector_Types.Y)),
--        (-Cs(Vector_Types.X)*Sin(Theta)          +
--         Cs(Vector_Types.Y)* Sin(Phi)*Cos(Theta)                             +
--         Cs(Vector_Types.Z)* Cos(Phi)*Cos(Theta) + Translate(Vector_Types.Z)) );
--   end Coordinate_System_Transform;

   -- Method Calculate_Body_Force
   function Calculate_Body_Force
     (An_Instance : in Instance;
      Axial_Thrust: in Jpats_Propeller_Types.Force_Vector_Type.Instance )
     return Jpats_Propeller_Types.Force_Vector_Type.Instance is

   begin

     return
       Rotate_Coordinate_System
       (Cs          => Axial_Thrust,
        Rotate      => An_Instance.Coordinate_Rotation);
   end Calculate_Body_Force;

-- Method Calculate_Moment_Arms
   function Calculate_Moment_Arms
     (An_Instance : in     Instance;
      Aircraft_Cg : in     Jpats_Propeller_Types.Coordinate_Type.Instance)
     return Jpats_Propeller_Types.Coordinate_Type.Instance is

     use type Jpats_Propeller_Types.Coordinate_Type.Instance;

   begin
     --| Alphap ranges between 0 and 180 degrees.  The propeller shaft in the
     --| Beech MkII is 2 degrees to the right and 1.5 degrees  downwards,
     --| relative to the body axes.

     --      lpx := ( FScg - FSprop)/12.0; -- (feet)
     --      lpy := -BLcg/12.0; --  (feet)
     --      lpz := (WLcg - WLprop)/12.0; --  (feet)
     return
       (Aircraft_Cg - An_Instance.Reference_Point);

   end Calculate_Moment_Arms;

-- Method Calculate_Body_Moment
   function Calculate_Body_Moment
     (An_Instance  : in Instance;
      Axial_Torque : in Jpats_Propeller_Types.Torque_Vector_Type.Instance)
     return Jpats_Propeller_Types.Torque_Vector_Type.Instance is
     use type Jpats_Propeller_Types.Torque_Vector_Type.Instance;

   begin
--| Calculate and sum the body-axis forces and moments due to the
--| axial-flow thrust, and the reactions to aerodynamic torque.
--| These equations account for the offset of the propeller hub
--| from the c.g., and for the orientation of the propeller shaft
--| (2 degrees to the right and 1.5 degrees downwards) relative to
--| the aircraft system:
--|
     return  ((An_Instance.Moment_Arm*An_Instance.Force_Tensor) +
              Rotate_Coordinate_System
       (Cs          => Axial_Torque,
        Rotate      => An_Instance.Coordinate_Rotation));
--     Lprop := -Qaxial -Yprop*lpz + Zprop*Lpy;
--     Mprop := -0.0349*Qaxial + Xprop*lpz - Zprop*Lpx;
--     Nprop := -0.0262*Qaxial - Xprop*lpy + Yprop*Lpx;
   end Calculate_Body_Moment;

 -- Method Calculate_Gyro_Moment
    function Calculate_Gyro_Moment
      (An_Instance           : in Instance;
       Inertia               : in Mass_Types.Slugs_Feet_Sq;
       Rpm                   : in Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
       Aircraft_Angular_Rate : in Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance)
      return Jpats_Propeller_Types.Torque_Vector_Type.Instance is

      P      : Angle_Types.Degrees_Per_Sec renames Aircraft_Angular_Rate(Vector_Types.Roll);
      Q      : Angle_Types.Degrees_Per_Sec renames Aircraft_Angular_Rate(Vector_Types.Pitch);
      R      : Angle_Types.Degrees_Per_Sec renames Aircraft_Angular_Rate(Vector_Types.Yaw);
      Ipgyro : Mass_Types.Slugs_Feet_Sq    renames Inertia;
      N      : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance renames Rpm;
    begin
 --| Calculate the gyroscopic reaction moments due to coupling of
 --| aircraft angular rates with the angular momentum of the
 --| rotating mass.  For these, the combined gyroscopic moment
 --| of inertia of the prop and engine power section is used.
 --| Due to the engine and propeller shaft orientation, these
 --| reactions create moments about each axis:
 --|

 --| Lgyro := Ipgyro*n'*2*pi*(-sin(1.5)*Q + sin(2)*R)     ?
 --| Mgyro := Ipgyro*n'*2*pi*(-cos(2.5)*R + sin(1.5)*P)   ?  What is this vector equation?
 --| Ngyro := Ipgyro*n'*2*pi*(-sin(2)*P + -cos(2.5)*Q)    ?
 --     return (Ipgyro*N *2.0*Ada.Numerics.Pi*(-Sin(1.5)*Q + Sin(2.0)*R),
 --             Ipgyro*N *2.0*Ada.Numerics.Pi*(-Cos(2.5)*R + Sin(1.5)*P),
 --             Ipgyro*N *2.0*Ada.Numerics.Pi*(-Sin(2.0)*P - Cos(2.5)*Q));

      -- Note: I am using a negative sign in front of Ipgyro since the
      -- cross product is torque needed to act on the prop to maintain
      -- the precession of the prop.  The moment reaction exerted on
      -- the airframe by the prop due to precession is equal and
      -- opposite to the that calculated by the cross product.
      return float(-1.0)*Ipgyro*(Aircraft_Angular_Rate*Rotate_Coordinate_System
                     (Cs        => (Angle_Types.Rpm_To_Radians_Per_Sec(N(Vector_Types.X)),
                                    Angle_Types.Rpm_To_Radians_Per_Sec(N(Vector_Types.Y)),
                                    Angle_Types.Rpm_To_Radians_Per_Sec(N(Vector_Types.Z))),
                      Rotate    => An_Instance.Coordinate_Rotation));

--      return Ipgyro*(Aircraft_Angular_Rate*
--                     Rotate_Coordinate_System
--                     (Cs        => (Angle_Types.Rpm_To_Radians_Per_Sec(N(Vector_Types.X)),
--                                    Angle_Types.Rpm_To_Radians_Per_Sec(N(Vector_Types.Y)),
--                                    Angle_Types.Rpm_To_Radians_Per_Sec(N(Vector_Types.Z))),
--                      Rotate    => An_Instance.Coordinate_Rotation));

    end Calculate_Gyro_Moment;

-- Method Initialize
   procedure Initialize
     (An_Instance : in out Instance) is
   begin
     An_Instance.Reference_Point := (Length_Types.Feet(35.41/12.0),Length_Types.Feet(0.0),Length_Types.Feet(92.26/12.0));
     -- Angles are measured from body axis to engine axis are -1.5
     -- degrees alpha and 2.0 degrees in Psi.  Note that these
     -- coordinates are used to convert from engine or wind axis back
     -- to body axis.  In that case the signs on all of the angles
     -- would need to be changed.  However, the transformation
     -- equations also transform from body to wind axis; therefore the
     -- signs of the angle would also need to be swapped.  The net sum
     -- would be use the equations and terms with their signs as you
     -- see here.
     An_Instance.Coordinate_Rotation := (Angle_Types.Degrees_To_Radians( 0.0),
                                         Angle_Types.Degrees_To_Radians(-1.5),
                                         Angle_Types.Degrees_To_Radians( 2.0));
     An_Instance.Coordinate_Translation := (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
     An_Instance.Moment_Arm := (Length_Types.Feet(0.0),Length_Types.Feet(0.0),Length_Types.Feet(0.0));
     An_Instance.Moment_Tensor := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
     An_Instance.Force_Tensor  := (Force_Types.Lbf(0.0),Force_Types.Lbf(0.0),Force_Types.Lbf(0.0));
     An_Instance.Gyroscopic_Moment_Tensor := (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));

   end Initialize;

end Geometry;



































