-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Ada.Numerics;
use  Ada.Numerics;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;
with Interpolation_Table.Doubly_Indexed;
with Interpolation_Table.Singly_Indexed;
with Vector_Types;
use  Vector_Types;
with Simulation_Dictionary;
with Ada.Text_Io;
with Log;
with Ada.Streams.Stream_Io;
with Stethoscope;
with Jpats_Powerplant;

package body Pratt_Whitney_Pt6a_68_Gas_Generator is

  ------------------------------------------------------------------------------
  --  BLEED AIR SUPPLY SYSTEM
  --  Engine bleed air is utilized for the canopy pressurization
  --  seal, anti-G system, environmental control system (ECS),
  --  pressurization, and on-board oxygen generation system
  --  (OBOGS). Bleed air is tapped from the engine compressor
  --  section at the left and right P3 bleed air ports. From the right
  --  P3 port, bleed air for the canopy seal and anti-G system is
  --  routed through the pneumatic system shutoff valve, a com-bined
  --  anti-G/OBOGS heat exchanger, and a water separator
  --  and then splits to feed the front and rear anti-G connections
  --  and the canopy seal pressure regulator. Bleed air for the
  --  ECS is also tapped at the right side P3 port, and is then
  --  routed through the bi-level control and shutoff valve, to the
  --  ECS heat exchanger, and on to the air distribution system.
  --  OBOGS bleed air is tapped at the left side P3 port, and uti-lizes
  --  one element of the anti-G/OBOGS heat exchanger.
  --
  --  Ref:  AIR FORCE TO 1T-6A-1
  --        NAVY (NAVAIR) A1-T6AAA-NFM-100
  --        Page 1-28  CVV DRAFT DECEMBER 1998
  --
  ------------------------------------------------------------------------------
  --
  --  OIL SYSTEM
  --  The engine oil system (Figure 1-8 and Figure 1-9), which
  --  includes a pressure system, a scavenge system, a cooling
  --  system, and a breather system, has a capacity of 18 U.S.
  --  quarts. The oil system provides a constant supply of filtered
  --  oil to the engine bearings, reduction gears, accessory drives
  --  and propeller throughout normal and aerobatic flight
  --  maneuvers. The oil tank is integrally cast with the compres-sor
  --  air inlet. Oil is cooled by routing it through an oil cooler
  --  located under the chin inlet.
  --  The engine oil system is designed to permit inverted flight
  --  operation. The pressure oil pump housing incorporates two
  --  elements. One element picks up oil near the center of the oil
  --  tank. This element is normally submerged in the oil. To pre-vent
  --  loss of oil pressure during inverted flight, a second ele-ment
  --  picks up oil near the top of the oil tank. This dual-element
  --  system ensures a constant flow of oil to the pressure
  --  regulating valve in all flight configurations. Scavenging of
  --  the bearing areas and gearboxes is accomplished with two
  --  dual-element gear type pumps. One dual-element pump is
  --  located inside the accessory gearbox and the other dual-ele-ment
  --  pump is located outside the gearbox. Scavenged oil is
  --  routed through the oil cooler before the oil is returned to the
  --  oil tank. The oil tank is vented into the accessory gearbox
  --  by a four point breather system, which includes a breather
  --  valve and centrifugal breather. Oil is filtered by the main Oil
  --  filter in the main oil tank, and by several oil strainers in the
  --  power, compressor, and accessory drive areas. If the main
  --  oil filter becomes clogged, a filter bypass valve allows unfil-tered
  --  oil to continue to lubricate the engine.
  --  Oil pressure downstream of the main oil pump and oil temperature
  --  downstream of the oil cooler are sensed by trans-ducers
  --  and sent to the engine data manager (EDM) which
  --  displays the information on the engine/systems electronic
  --  instrument display (EID) in each cockpit. The EDM will
  --  activate the red OIL PX annunciator on the annunciator
  --  panel if oil pressure falls below 40 psi when the engine is
  --  above IDLE power; or when oil pressure drops below 15 psi
  --  at IDLE power. Power for the oil pressure transducer is pro-vided
  --  through a circuit breaker placarded OIL TRX, located
  --  on the battery bus circuit breaker panel in the front cockpit.
  --
  --  Ref. AIR FORCE TO 1T-6A-1
  --       NAVY (NAVAIR) A1-T6AAA-NFM-100
  --       Page 1-6 - 1-12 CVV DRAFT DECEMBER 1998
  --
  -------------------------------------------------------------------------------

  -- Declare Blank Instances for Each Table
  -- Table Instance are filled with data using a table read
  -- during read_tables procedure.
  DITTHOT_Table    : aliased Interpolation_Table.Singly_Indexed.Instance;
  DSHPCNPNGC_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
  FNCSHPC_Table    : aliased Interpolation_Table.Doubly_Indexed.Instance;
  ITTCNGC_Table    : aliased Interpolation_Table.Doubly_Indexed.Instance;
  KITTNG_ACC_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
  KITTNG_BLD_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
  KNGFF_ACC_Table  : aliased Interpolation_Table.Singly_Indexed.Instance;
  KNGFF_BLD_Table  : aliased Interpolation_Table.Singly_Indexed.Instance;
  KSHPCNP_Table    : aliased Interpolation_Table.Singly_Indexed.Instance;
  KSHPNG_ACC_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
  KSHPNG_BLD_Table : aliased Interpolation_Table.Singly_Indexed.Instance;
  NGCFFC_Table     : aliased Interpolation_Table.Doubly_Indexed.Instance;
  PENGOIL_Table    : aliased Interpolation_Table.Doubly_Indexed.Instance;
  DPEOILNZ_Table   : aliased Interpolation_Table.Singly_Indexed.Instance;
  SHPCNGC_Table    : aliased Interpolation_Table.Doubly_Indexed.Instance;
  TACCNG_Table     : aliased Interpolation_Table.Doubly_Indexed.Instance;
  TDECNG_Table     : aliased Interpolation_Table.Singly_Indexed.Instance;
  TITTNGC_Table    : aliased Interpolation_Table.Doubly_Indexed.Instance;
  DNGSEIZ_Table    : aliased Interpolation_Table.Singly_Indexed.Instance;
  DT1MACH_Table    : aliased Interpolation_Table.Singly_Indexed.Instance;
  DQDWF_Table      : aliased Interpolation_Table.Singly_Indexed.Instance;

  Lag_Time_Exception : exception;
--  Eng1_Test_File    : Ada.Text_Io.File_Type;

  -- Access Rotor Reset
  procedure Rotor_Reset
    (An_Instance : in out Instance) is
  begin
--    An_Instance.Rotor_Reset := State;
    An_Instance.Rotor_Reset := True;
  end Rotor_Reset;

  -- Access Cold Soak Reset
  procedure Cold_Soak_Reset
    (An_Instance : in out Instance) is
  begin
--    An_Instance.Rotor_Reset := State;
    An_Instance.Cold_Soak_Reset := True;
  end Cold_Soak_Reset;

 -- Access Quick Start
  procedure Quick_Start
    (An_Instance : in out Instance) is
  begin
    An_Instance.Quick_Start := True;
  end Quick_Start;

  -- Access Internal Turbine Temperature
  function  Internal_Turbine_Temperature_Sensor
    (An_Instance : in Instance)
    return Temperature_Types.Celsius is
  begin
    return An_Instance.Internal_Turbine_Temperature;
  end Internal_Turbine_Temperature_Sensor;

  -- Gas Generator Rpm Sensor
  function Gas_Generator_Rpm_Sensor
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return Gas_Generator_Rpm(An_Instance => An_Instance);
  end Gas_Generator_Rpm_Sensor;

  -- Gas Generator Rpm
  function Gas_Generator_Rpm
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return An_Instance.Gas_Generator_Rpm;
  end Gas_Generator_Rpm;

  -- Gas Generator Speed Pct
  function Gas_Generator_Speed_Pct
    (An_Instance : in Instance)
    return Normalized_Types.Normalize is
  begin
    return An_Instance.Gas_Generator_Rpm/374.68;
  end Gas_Generator_Speed_Pct;

  -- Access Power_Turbine_Rpm_Sensor
  function  Power_Turbine_Rpm_Sensor
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return An_Instance.Power_Turbine_Rpm;
  end Power_Turbine_Rpm_Sensor;

  -- Access Power_Turbine_Rpm
  function  Power_Turbine_Rpm
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return An_Instance.Power_Turbine_Rpm;
  end Power_Turbine_Rpm;

  -- Access Power_Turbine_Speed_Pct
  function  Power_Turbine_Speed_Pct
    (An_Instance : in Instance)
    return Normalized_Types.Normalize is
  begin
    return An_Instance.Power_Turbine_Rpm/20.00;
  end Power_Turbine_Speed_Pct;

  -- Engine Boost Pump Shaft Rpm
  function Boost_Pump_Shaft_Rpm
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return Angle_Types.Rpm
      (Gas_Generator_Rpm(An_Instance => An_Instance)*0.1019);
  end Boost_Pump_Shaft_Rpm;

  -- Engine Boost Pump Shaft Rpm
  function Fcu_Shaft_Rpm
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return Angle_Types.Rpm
      (Gas_Generator_Rpm(An_Instance => An_Instance)*0.22413);
  end Fcu_Shaft_Rpm;

  -- Engine Generator Shaft Rpm
  function Generator_Shaft_Rpm
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return Angle_Types.Rpm
      (Gas_Generator_Rpm(An_Instance => An_Instance)*0.2931);
  end Generator_Shaft_Rpm;

  -- Engine Hydraulic Pump Shaft Rpm
  function Hyd_Shaft_Rpm
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return Angle_Types.Rpm
      (Gas_Generator_Rpm(An_Instance => An_Instance)*0.2041);
  end Hyd_Shaft_Rpm;

  --  Access Torque_Sensor
  function  Torque_Sensor
    (An_Instance : in Instance)
    return Torque_Types.Ft_lbf is
  begin
    return An_Instance.Torque(X);
  end Torque_Sensor;

  --  Access Torque_Sensor
  function  Engine_Torque_Pct
    (An_Instance : in Instance)
    return Normalized_Types.Normalize is
  begin
    return An_Instance.Torque(X)/28.89;
  end Engine_Torque_Pct;

  -- Access Bleed_Pressure
  function  Bleed_Pressure
    (An_Instance : in Instance)
    return Force_Types.Psi is
  begin
    return An_Instance.Bleed_Pressure;
  end Bleed_Pressure;

  procedure Set_Bleed_Pressure
    (An_Instance    : in out Instance;
     Bleed_Pressure : in     Force_Types.Psi) is
  begin
    An_Instance.Bleed_Pressure := Bleed_Pressure;
  end Set_Bleed_Pressure;

  -- Access Bleed_Temperature
  function  Bleed_Temperature
    (An_Instance : in Instance)
    return Temperature_Types.Celsius is
  begin
    return An_Instance.Bleed_Temperature;
  end Bleed_Temperature;

  procedure Set_Bleed_Temperature
    (An_Instance       : in out Instance;
     Bleed_Temperature : in     Temperature_Types.Celsius) is
  begin
    An_Instance.Bleed_Temperature :=  Bleed_Temperature;
  end Set_Bleed_Temperature;

  -- Access Exhaust_Thrust
  function  Exhaust_Thrust
    (An_Instance : in Instance)
    return Force_Types.Lbf is
  begin
    return An_Instance.Exhaust_Thrust;
  end Exhaust_Thrust;

  procedure Set_Exhaust_Thrust
    (An_Instance : in out Instance) is
  begin
    null;
  end Set_Exhaust_Thrust;

  -- Access Delta_Shp_C_Acc
  function  Delta_Shp_C_Acc
    (An_Instance : in Instance)
    return Torque_Types.Horsepower is
  begin
    return An_Instance.Delta_Shp_C_Acc;
  end Delta_Shp_C_Acc;

  -- Access Delta_Shp_C_Bleed
  function  Delta_Shp_C_Bleed
    (An_Instance : in Instance)
    return Torque_Types.Horsepower is
  begin
    return An_Instance.Delta_Shp_C_Bleed;
  end Delta_Shp_C_Bleed;

  -- Access Delta_Shp_C_Np
  function  Delta_Shp_C_Np
    (An_Instance : in Instance)
    return Torque_Types.Horsepower is
  begin
    return An_Instance.Delta_Shp_C_Np;
  end Delta_Shp_C_Np;

  -- Access Delta_Itt_C_Acc
  function  Delta_Itt_C_Acc
    (An_Instance : in Instance)
    return Temperature_Types.Kelvin is
  begin
    return An_Instance.Delta_Itt_C_Acc;
  end Delta_Itt_C_Acc;

  -- Access Delta_Itt_C_Bleed
  function  Delta_Itt_C_Bleed
    (An_Instance : in Instance)
    return Temperature_Types.Kelvin is
  begin
    return An_Instance.Delta_Itt_C_Bleed;
  end Delta_Itt_C_Bleed;

  -- Access Delta_Ng_C_Acc
  function  Delta_Ng_C_Acc
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return An_Instance.Delta_Ng_C_Acc;
  end Delta_Ng_C_Acc;

  -- Access Delta_Ng_C_Bleed
  function  Delta_Ng_C_Bleed
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return An_Instance.Delta_Ng_C_Bleed;
  end Delta_Ng_C_Bleed;

  -- Access Delta_Ng_Seizure
  function  Delta_Ng_Seizure
    (An_Instance : in Instance)
    return Angle_Types.Rpm is
  begin
    return An_Instance.Delta_Ng_Seizure;
  end Delta_Ng_Seizure;

  -- Access Shp
  function  Shp
    (An_Instance : in Instance)
    return Torque_Types.Horsepower is
  begin
    return An_Instance.Shaft_Horsepower;
  end Shp;

  -- Access Ignition_Exciter_Load
  function Ignition_Exciter_Load
    (An_Instance : in Instance)
    return Electrical_Units_Types.Amps is
  begin
    return An_Instance.Ignition_Exciter_Load;
  end Ignition_Exciter_Load;

  -- Access Engine_oil_pressure
  function Engine_Oil_Pressure
    (An_Instance : in Instance)
    return Force_Types.Psi is
  begin
    return An_Instance.Oil_Pressure;
  end Engine_Oil_Pressure;

  -- Access Engine_oil_temperature
  function Engine_Oil_Temperature
    (An_Instance : in Instance)
    return Temperature_Types.Celsius is
  begin
    return An_Instance.Oil_Temperature;
  end Engine_Oil_Temperature;

  -- Access Chip Detect
  function Chip_Sensor
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Chip_Sensor;
  end Chip_Sensor;

  function Fired
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Fired;
  end Fired;

  -- Function Itt_Start_Mode for IOS
  function Itt_Start_Mode
    (An_Instance : in Instance)
    return Boolean is
  begin
    return An_Instance.Itt_Start_Mode;
  end Itt_Start_Mode;

  Tau_Acc_Ng                 : Float;  -- Time in seconds
  Tau_Dec_Ng                 : Float;  -- time in seconds
  Tau_Ng                     : Float;  -- time in seconds
  Tau_Itt_Ng_C               : Float;  -- time in seconds
  Ng_C_Itt                   : Angle_Types.Rpm;
  Itt_C_Ng_C                 : Temperature_Types.Kelvin;
  Itt_C_Lead                 : Temperature_Types.Kelvin;
  Shp_C_Ng_C                 : Torque_Types.Horsepower;
  K_Shp_C_Np                 : Normalized_Types.Normalize;
  Shp_C                      : Torque_Types.Horsepower;
  Fn_C_Shp_C                 : Normalized_Types.Normalize;
  Ddnp_Shp_C                 : Normalized_Types.Normalize;
  Ng_Fire                    : Angle_Types.Rpm;
  K_Ng_Wf_Dot_Acc            : Normalized_Types.Normalize;  --  Units of Delta_rpm/Delta_shp = f(wf_dot)
  K_Ng_Wf_Dot_Bleed          : Normalized_Types.Normalize;  --  Units of Delta_rpm/Delta_bleed_flow_dot = f(wf_dot)
  Ng_C_Wf_Dot_C              : Angle_Types.Rpm;
  K_Itt_Ng_Acc               : Normalized_Types.Normalize;  -- Degrees K/Shp
  K_Itt_Ng_Bleed             : Normalized_Types.Normalize;
  Delta_Itt_Hot              : Temperature_Types.Celsius;
  Tau_Itt_Lag                : Float;  -- seconds
  K_Shp_Ng_Acc               : Normalized_Types.Normalize;
  K_Shp_Ng_Bleed             : Normalized_Types.Normalize;
  K_Ng_Malfunction           : Normalized_Types.Normalize;
  K_Itt_Malfunction          : Normalized_Types.Normalize;
  Starter_On                 : Boolean;
  Ng_Ss                      : Normalized_Types.Normalize;
  Ng_C                       : Normalized_Types.Normalize;
  Ng_C_Ss                    : Normalized_Types.Normalize;
  Ng_Dot                     : Normalized_Types.Normalize;
  Total_Temperature_Ratio    : Normalized_Types.Normalize;
  Total_Pressure_Ratio       : Normalized_Types.Normalize;
  Pressure_Altitude          : Length_Types.Feet;
  Delta_T1_Mach              : Normalized_Types.Normalize;
  Ng_For_Zero_Starter_Torque : Normalized_Types.Normalize;
  Tau_Acc_Tune               : Float := 1.0;

  input_Iconst                          : Float;
  input_Ambient_Temperature_Ratio       : Normalized_Types.Normalize;
  input_Ambient_Pressure_Ratio          : Normalized_Types.Normalize;
  input_Mach_Number                     : Length_Types.Mach;
  input_Fuel_Flow_Rate                  : Mass_Types.Pph; -- lbm/hr
  input_Bleed_Flow_Rate                 : Mass_Types.Ppm; -- lbm/min
  input_Propeller_Torque_Absorbed       : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
  input_Accessory_Torque_Absorbed       : Boolean;
  input_Air_Conditioner_Torque_Absorbed : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
  input_Accessory_Horsepower            : Torque_Types.Horsepower;
  input_Ignition_Exciter                : Boolean;
  input_Hot_Start                       : Boolean;
  input_No_Light_Request                : Boolean;

  procedure Nulloutput
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : in     Pratt_Stream_Type) is
  begin
    Integer'Write(Stream, 1);
  end Nulloutput;

  procedure Nullread
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : out    Pratt_Stream_Type) is
    Dropped_Data : Integer;
  begin
    Integer'Read(Stream, Dropped_Data);
    Item := null;
  end Nullread;

  function Nullinput
    (Stream : access Ada.Streams.Root_Stream_Type'Class)
    return Pratt_Stream_Type is
    Dropped_Data : Integer;
  begin
    Integer'Read(Stream, Dropped_Data);
    return null;
  end Nullinput;

  procedure Pratt_Queue_Write
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : in     Pratt_Queue) is
    Data     : Float;
    Iterator : Float_Container.Iterator := New_Iterator(For_The_Queue => Item);
  begin
    while not Float_Container.Is_Done(Obj => Iterator) loop
      Data := Float_Container.Current_Item(Obj => Iterator);
      Float'Write(Stream,Data);
      Float_Container.Next(Iterator);
    end loop;
--    Data := Front(Obj => Item);
--    Float'Write(Stream, Data);
  end Pratt_Queue_Write;

  procedure Pratt_Queue_Read
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : out    Pratt_Queue) is
    Data : Float;
--    Iterator : One_Second_Queue.Iterator := New_Iterator(For_The_Queue => Item);
  begin
    Clear(Item);
--    while not Is_Done(Obj => Iterator) loop
    for I in 1 .. Max_Queue_Size loop
      Float'Read(Stream, Data);
      Append
        (Obj  => Item,
         Elem => Data);
--      Next(Iterator);
    end loop;
  end Pratt_Queue_Read;

  procedure Pratt_Queue_Output
    (Stream : access Ada.Streams.Root_Stream_Type'Class;
     Item   : in     Pratt_Queue) is
    Data     : Float;
    Iterator : Float_Container.Iterator := New_Iterator(For_The_Queue => Item);
  begin
    while not Float_Container.Is_Done(Obj => Iterator) loop
      Data := Float_Container.Current_Item(Obj => Iterator);
      Float'Write(Stream,Data);
      Float_Container.Next(Iterator);
    end loop;
--    Data := Front(Item);
--    Float'Write(Stream, Data);
  end Pratt_Queue_Output;

  function Pratt_Queue_Input
    (Stream : access Ada.Streams.Root_Stream_Type'Class)
    return Pratt_Queue is
    Data : Float;
    Lagging_Queue_Holder : Pratt_Queue;
  begin
    --    Lagging_Queue.Clear(Lagging_Queue.Bounded_Queue(Item));
    for I in 1 .. Max_Queue_Size loop
      Float'Read(Stream, Data);
      Append
        (Obj  => Lagging_Queue_Holder,
         Elem => Data);
    end loop;
      return Lagging_Queue_Holder;
  end Pratt_Queue_Input;

  procedure Start_Recording
    (An_Instance  : in out Instance;
     Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access ) is

  begin
    An_Instance.Record_Debug_Information := True;
    An_Instance.Debug_Stream := Pratt_Stream_Type(Debug_Stream);
  end Start_Recording;

  procedure Stop_Recording
    (An_Instance : in out Instance) is
  begin
    An_Instance.Record_Debug_Information := False;
  end Stop_Recording;

  procedure Set_Engine_Test_Case
    (An_Instance      : in out Instance;
     Engine_Test_Case : in Integer) is
  begin
    An_Instance.Engine_Test_Case := Engine_Test_Case;
  end Set_Engine_Test_Case;

  function Inlet_Temperature_Sensor
    (An_Instance      : in Instance)
    return Temperature_Types.Celsius is
  begin
    return An_Instance.T1;
  end Inlet_Temperature_Sensor;

  Temp_Angle : Float := 0.0;

  -- Method Update
  procedure Update
    (Next_Instance                   : in out Instance;
     This_Instance                   : in     Instance;
     Iconst                          : in     Float;
     Ambient_Temperature_Ratio       : in     Normalized_Types.Normalize;
     Ambient_Pressure_Ratio          : in     Normalized_Types.Normalize;
     Mach_Number                     : in     Length_Types.Mach;
     Fuel_Flow_Rate                  : in     Mass_Types.Pph; -- lbm/hr
     Bleed_Flow_Rate                 : in     Mass_Types.Ppm; -- lbm/min
     Propeller_Torque_Absorbed       : in     Jpats_Propeller_Types.Torque_Vector_Type.Instance;
     Accessory_Torque_Absorbed       : in     Jpats_Propeller_Types.Torque_Vector_Type.Instance;
     Air_Conditioner_Torque_Absorbed : in     Jpats_Propeller_Types.Torque_Vector_Type.Instance;
     Accessory_Horsepower            : in     Torque_Types.Horsepower;
     Ignition_Exciter                : in     Boolean;
     Hot_Start                       : in     Boolean;
     No_Light_Request                : in     Boolean;
     Engine_Seizure                  : in     Boolean;
     Itt_Sensor_Malfunction          : in     Boolean;
     Overtemp                        : in     Boolean;
     Oil_Leak                        : in     Boolean;
     Oil_Pressure_Failure            : in     JPATS_Powerplant_Types.Oil_Pressure_Failure_Type;
     High_Oil_Temperature            : in     Boolean;
     Load_Factor                     : in     Coordinate_Types.Cartesian;
     Voltage                         : in     Electrical_Units_Types.Volts_Dc;
     Chip                            : in     Boolean;
     Flameout_Malfunction            : in     Boolean;
     Torque_Auto_Driven              : in     Boolean;
     Auto_Driven_Torque              : in     Torque_Types.Ft_Lbf ) is

    Theta_Amb                  : Normalized_Types.Normalize                             renames Ambient_Temperature_Ratio;
    Delta_Amb                  : Normalized_Types.Normalize                             renames Ambient_Pressure_Ratio;
    Mach                       : Length_Types.Mach                                      renames Mach_Number;
    --    Total_Temperature_Ratio    : Normalized_Types.Normalize;
    --    Total_Pressure_Ratio       : Normalized_Types.Normalize;
    Theta_T                    : Normalized_Types.Normalize                             renames Total_Temperature_Ratio;
    Delta_T                    : Normalized_Types.Normalize                             renames Total_Pressure_Ratio;
    Hp                         : Length_Types.Feet                                      renames Pressure_Altitude;
    Wbleed_Dot                 : Mass_Types.Ppm                                         renames Bleed_Flow_Rate;
    Q_Prop                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance      renames Propeller_Torque_Absorbed;
    Q_Air_Conditioner          : Jpats_Propeller_Types.Torque_Vector_Type.Instance      renames Air_Conditioner_Torque_Absorbed;
    Shp_Acc                    : Torque_Types.Horsepower                                renames Accessory_Horsepower;
    Wf_Dot_C                   : Mass_Types.Pph                                         renames Next_Instance.Fuel_Flow_Corrected;
    Ng                         : Normalized_Types.Normalize                             renames Next_Instance.Gas_Generator_Rpm_In_Percent;
    Ng_Last                    : Normalized_Types.Normalize                             renames This_Instance.Gas_Generator_Rpm_In_Percent;
    --    Ng_Ss                      : Normalized_Types.Normalize;
    --    Ng_C                       : Normalized_Types.Normalize;
    --    Ng_C_Ss                    : Normalized_Types.Normalize;
    --    Ng_Dot                     : Normalized_Types.Normalize;
    Np                         : Normalized_Types.Normalize                             renames Next_Instance.Power_Turbine_rpm_In_Percent;
    Np_Last                    : Normalized_Types.Normalize                             renames This_Instance.Power_Turbine_rpm_In_Percent;
    Itt_Lead                   : Temperature_Types.Celsius                              renames Next_Instance.Internal_Turbine_Temperature_Lead_Lag;
    Itt_Lead_Last              : Temperature_Types.Celsius                              renames This_Instance.Internal_Turbine_Temperature_Lead_Lag;
    Itt                        : Temperature_Types.Celsius                              renames Next_Instance.Internal_Turbine_Temperature;
    Itt_Last                   : Temperature_Types.Celsius                              renames This_Instance.Internal_Turbine_Temperature;
    Shp                        : Torque_Types.Horsepower                                renames Next_Instance.Shaft_Horsepower;
    Q_Eng                      : Jpats_Propeller_Types.Torque_Vector_Type.Instance      renames Next_Instance.Torque;
    --    Q_Pt                       : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
    Rpm                        : Angle_Types.Rpm                                        renames Next_Instance.Power_Turbine_rpm;
    Rpm_Last                   : Angle_Types.Rpm                                        renames This_Instance.Power_Turbine_rpm;
    Rpm_Dot                    : Angle_Types.Rpm_Per_Sec                                renames Next_Instance.Power_Turbine_Acceleration;
    Rpm_Dot_Last_Pass          : Angle_Types.Rpm_Per_Sec                                renames This_Instance.Power_Turbine_Acceleration;
    Ip_Accel                   : Mass_Types.Slugs_Feet_sq                               renames This_Instance.Power_Turbine_Inertia;
    Fn                         : Force_Types.Lbf                                        renames Next_Instance.Exhaust_Thrust;
    --    Tau_Acc_Ng                 : Float;  -- Time in seconds
    --    Tau_Dec_Ng                 : Float;  -- time in seconds
    --    Tau_Ng                     : Float;  -- time in seconds
    --    Tau_Itt_Ng_C               : Float;  -- time in seconds
    ----  Ng_C_Itt                   : Angle_Types.Rpm;
    --    Itt_C_Ng_C                 : Temperature_Types.Kelvin;
    --    Itt_C_Lead                 : Temperature_Types.Kelvin;
    --    Shp_C_Ng_C                 : Torque_Types.Horsepower;
    --    K_Shp_C_Np                 : Normalized_Types.Normalize;
    --    Shp_C                      : Torque_Types.Horsepower;
    --    Fn_C_Shp_C                 : Normalized_Types.Normalize;
    --    Ddnp_Shp_C                 : Normalized_Types.Normalize;
    --    Ng_Fire                    : Angle_Types.Rpm;
    --    K_Ng_Wf_Dot_Acc            : Normalized_Types.Normalize;  -- Units of Delta_rpm/Delta_shp = f(wf_dot)
    --    K_Ng_Wf_Dot_Bleed          : Normalized_Types.Normalize;  -- Units of Delta_rpm/Delta_bleed_flow_dot = f(wf_dot)
    --    Ng_C_Wf_Dot_C              : Angle_Types.Rpm;
    --    K_Itt_Ng_Acc               : Normalized_Types.Normalize;  -- Degrees K/Shp
    --    K_Itt_Ng_Bleed             : Normalized_Types.Normalize;
    --    Delta_Itt_Hot              : Temperature_Types.Celsius;
    --    Tau_Itt_Lag                : Float;  -- seconds
    --    K_Shp_Ng_Acc               : Normalized_Types.Normalize;
    --    K_Shp_Ng_Bleed             : Normalized_Types.Normalize;
    --    K_Ng_Malfunction           : Normalized_Types.Normalize;
    --    K_Itt_Malfunction          : Normalized_Types.Normalize;
    --    Starter_On                 : Boolean;
    Lowest_Voltage_Before_Degradation : Electrical_Units_Types.Volts_Dc := Electrical_Units_Types.Volts_Dc(21.5);
    Pullback                         : Boolean := False;

  begin
    input_Iconst                          :=  Iconst                            ;
    input_Ambient_Temperature_Ratio       :=  Ambient_Temperature_Ratio         ;
    input_Ambient_Pressure_Ratio          :=  Ambient_Pressure_Ratio            ;
    input_Mach_Number                     :=  Mach_Number                       ;
    input_Fuel_Flow_Rate                  :=  Fuel_Flow_Rate                    ;
    input_Bleed_Flow_Rate                 :=  Bleed_Flow_Rate                   ;
    input_Propeller_Torque_Absorbed       :=  Propeller_Torque_Absorbed         ;
    input_Accessory_Torque_Absorbed       :=  Accessory_Torque_Absorbed(X) < 0.0;
    input_Air_Conditioner_Torque_Absorbed :=  Air_Conditioner_Torque_Absorbed   ;
    input_Accessory_Horsepower            :=  Accessory_Horsepower              ;
    input_Ignition_Exciter                :=  Ignition_Exciter                  ;
    input_Hot_Start                       :=  Hot_Start                         ;
    input_No_Light_Request                :=  No_Light_Request                  ;


    -- Calculate the total temperature and pressure ratios, for
    -- use in correcting engine parameters. These carry the subscript "2"
    -- to denote the compressor face station:
    --
    -- The prefix "total" means ambient ratio + dynamic ratio obtained at the
    -- face of the engine.
    -- Customarily, this is temperature or pressure of the flow if the flow were
    -- slowed adiabatically and reversibly until stopped.

    Total_Temperature_Ratio := Theta_Amb*(1.0 + 0.2*Mach**2.0);
    Total_Pressure_Ratio    := Delta_Amb*(1.0 + 0.2*Mach**2.0)**3.5;

    -- Calculate Pressure Altitude from Ambient_Pressure_Ratio
    Pressure_Altitude := ((1.0 - Delta_Amb**0.19018)*145447.0);



    -- Step 1. Assign the power losses at the accessory gearbox, and
    -- the compressor bleed level. Also determine if the starter is
    -- engaged, if the ignitors are activated, and whether or not
    -- combustion is present.

    Starter_On := Accessory_Torque_Absorbed(X) < 0.0;

    -- *Accessory gearbox load*.
    -- The effects of non-variable accessory loads such as the fuel
    -- pump and the hydraulic pump load at zero flow rate are
    -- incorporated into the basic engine tables.  Variable accessory
    -- loads arise from the generator and hydraulic pump:

    -- Delta_Shp_Gen := Gen_Voltage*Generator_Output_In_Amps/0.7/746.0;  -- 746.0 W/HP 0.7 Peak to RMS conversion sqrt(2.0).
    -- Delta_Shp_Hyd :=  1.8*vdot_hyd;
    -- Shp_Acc := Delta_Shp_Gen + Delta_Shp_Hyd;

    -- *Compressor bleed level*.
    -- The basic engine tables incorporate the effects of the 5.2
    -- lb/hour bleed level arising from operation of the ECS in the
    -- "Normal" setting.

    -- *Combustion present*. The flag FIRED is used to represent whether
    -- combustion is present or not. Combustion initiates whenever
    -- sufficient fuel flow and gas generator speed occur while the
    -- ignitors are firing. Combustion ceases when fuel flow decreases
    -- below a minimum value, 50 lb/hour :
    --
    -- If ignitors_on then the the green IGN SEL light on the main
    --  annunciator panel is illuminated.

    Ng_Fire := 16.0 + 0.00016*Hp; -- minimum gas generator speed to sustain combustion.


    if Flameout_Malfunction and Fuel_Flow_Rate < 0.01 then
      Next_Instance.Flameout_Timer := Float'Min(10.0,This_Instance.Flameout_Timer + Iconst);
    else
      Next_Instance.Flameout_Timer := 0.0;
    end if;

    if Flameout_Malfunction then
      if Next_Instance.Flameout_Timer > 2.0 then
        Next_Instance.Wf_Dot := 0.0;
      else
        Next_Instance.Wf_Dot := This_Instance.Wf_Dot;
      end if;
    else
      Next_Instance.Wf_Dot := Fuel_Flow_Rate;
    end if;

    Next_Instance.Fired :=
      (This_Instance.Quick_Start or
      (Next_Instance.Wf_Dot >= 50.0
       and (Ignition_Exciter or This_Instance.Fired )
       and (Ng_Last > Ng_Fire)
       and not No_Light_Request));
--       or (Next_Instance.Flameout_Timer < 5.0)));

--    if not Next_Instance.Fired and Ng_Last > 35.0 then
--      Log.Report("Suspected flame out.");
--      Log.Report("wfdot = " & Float'Image(Wf_Dot))
--      Log.Report("this_instance.fired = " & Boolean'Image(This_Instance.Fired));
--      Log.Report("ng_last = " & Float'Image(Ng_Last));
--      Log.Report("ng_fire = " & Float'Image(Ng_Fire));
--      Log.Report("ignition exciter = " & Boolean'Image(Ignition_Exciter));
--    end if;
    if Ignition_Exciter then
      Next_Instance.Ignition_Exciter_Load := 1.0;
    else
      Next_Instance.Ignition_Exciter_Load := 0.0;
    end if;



    -- Step 2.  Calculated the total temperature and pressure ratios,
    -- for used in correcting engine parameters.  See Controller.


    -- Step 1. See Controller for pcl to pla conversion.

    -- Step 3.  Correct the fuel flow (calculated in the Fuel Metering
    -- System, section 11.3.1):

    Wf_Dot_C := Next_Instance.Wf_Dot /(Delta_T * Theta_T**0.1);  -- (lbm/hour)

    -- Step 4.  Obtain the steady state corrected gas generator speed
    -- as a function of corrected fuel flow, Mach number, accessory
    -- losses and bleed level, and whether or not combustion is
    -- present.

    -- the steady state corrected gas generator speed is calculated as:

    -- I am needing to match RAC's check cases, they implement
    -- K_Ng_Wf_Dot_Acc, K_Ng_Wf_Dot_Bleed, Next_Instance.Delta_Ng_C_Acc,
    -- and Next_Instance.Delta_Ng_C_Bleed above the test for fired condition.
    -- It ultimately doesn't affect the model since Ng_C_SS is still based on
    -- whether engine is fired or not.  Math model has these equations inside the
    -- if condition.

    K_Ng_Wf_Dot_Acc := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => Float(Next_Instance.Wf_Dot),
        Table => KNGFF_ACC_Table'Access );  -- not based on corrected wf_dot

    K_Ng_Wf_Dot_Bleed := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => Float(Next_Instance.Wf_Dot),
        Table => KNGFF_BLD_Table'Access );  -- not based on corrected wf_dot

    Next_Instance.Delta_Ng_C_Acc := (K_Ng_Wf_Dot_Acc * (Shp_Acc - 7.5))/Theta_T**0.5;
    -- shp_acc is actual horsepower drawn based on actual rpm on corrected rpm
    -- correct whole equation by theta_t**0,5

    Next_Instance.Delta_Ng_C_Bleed := K_Ng_Wf_Dot_Bleed * (Wbleed_Dot  - 5.2)/Theta_T**0.5;

    if Next_Instance.Fired  then


      Ng_C_Wf_Dot_C := Interpolation_Table.Doubly_Indexed.Interpolate
        ( X => Float(Wf_Dot_C),
          Y => Float(Mach),
          Table   => Ngcffc_Table'Access );


      -- The steady state gas generator speed is then calculated:

      Ng_C_Ss := Ng_C_Wf_Dot_C + Next_Instance.Delta_Ng_C_Acc + Next_Instance.Delta_Ng_C_Bleed;


    else
      -- if Starter_On and Electrical Power Available To The Starter
      -- generator, the starter drives the steady state corrected gas
      -- generator speed:
      if Starter_On then

        Ng_C_Ss := (18.0 + 0.00016 * Hp)/Theta_T**0.5; -- %

      else
        -- else The Steady State Gas Gnerator Speed is Dictated By
        -- Motoring Characteristics at The Flight Dynamic Pressure.

        --          Ng_C_Ss := 0.074*Qbar; -- %
        Ng_C_Ss := 32.3 * Mach; -- %

      end if;
    end if;

    -- Engine Seizure or Compressor Stall Malfunction
    if Engine_Seizure then
       Next_Instance.Seizure_Time := This_Instance.Seizure_Time + Iconst;
       Next_Instance.Delta_Ng_Seizure := Interpolation_Table.Singly_Indexed.Interpolate
       ( Input => Float(Next_Instance.Seizure_Time),
         Table => DNGSEIZ_Table'Access );
    else
       Next_Instance.Seizure_Time := 0.0;
       if Jpats_Powerplant.Compressor_Stall then
          Temp_Angle := Temp_Angle + (10.0 * Iconst);
          If Temp_Angle > 180.0 then
             Temp_Angle := 0.0;
          end if;
          Next_Instance.Delta_Ng_Seizure := 10.0 * Sin(Temp_Angle);
       else
          Next_Instance.Delta_Ng_Seizure := 0.0;
       end if;
    end if;

    Ng_C_Ss := Float'Max(0.0,(Ng_C_SS + Next_Instance.Delta_Ng_Seizure/Theta_T));


    -- The Steady State Gas Generator Speed is then Calculated:

    K_Ng_Malfunction := 1.0;

    Ng_Ss := K_Ng_Malfunction*Ng_C_Ss* Theta_T**0.5; -- steady state rpm in %


    -- Step 5.  Calculate the rate of change of gas generator
    -- speed, based on the difference between the steady
    -- state value and the current value, and on an
    -- acceleration time constant.  The acceleration time
    -- constant accounts for the effects of gas generator
    -- speed, pressure altitude and power turbine speed:

    -- Tau_Acc_ng        := Table TACCNG = F(Ng_Last, H)
    -- function of pressure altitude
    Tau_Acc_Ng := Interpolation_Table.Doubly_Indexed.Interpolate
      ( X => Float(Ng_Last),
        Y => Float(Hp),
        Table   => TACCNG_Table'Access);

    -- Tau_Dec_Ng        := Table TDECNG    = F(Ng_Last)
    Tau_Dec_Ng := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => Float(Ng_Last),
        Table => TDECNG_Table'Access);

    if Starter_On and not Next_Instance.Fired then
      -- Tau_Ng := 2.8;
      Tau_Ng := 2.8; --*((Lowest_Voltage_Before_Degradation)**2.0)/(Float'Max(1.0,Float'Min(Lowest_Voltage_Before_Degradation, Voltage)))**2.0;
--      Tau_Ng := Tau_Ng* Float'Min(5.0, Float'Max(1.0,21.7-Voltage));
    elsif Ng_Ss >= Ng_Last then
      Tau_Ng := Tau_Acc_Ng; -- (sec)
      -- voltage malfunction
      if ( Ng_Last < 50.0) then
        if not Starter_On then
           --          Tau_Ng := Tau_Ng * (13.5 - 0.25*Ng);
           Ng_For_Zero_Starter_Torque := 3.8 + 1.525*Voltage;
           --          Ng_For_Zero_Starter_Torque := 3.8 + 1.2*Voltage + 0.0*Voltage**2.0;
           --          Ng_For_Zero_Starter_Torque := 3.8 + 1.78*Voltage;
           Tau_Ng := Tau_Ng + Tau_Acc_Tune*Float'Max(0.0,(Ng_SS - Ng_For_Zero_Starter_torque)*Iconst);
           --        else
           --          Tau_Ng := Tau_Ng *
           --                    (Lowest_Voltage_Before_Degradation**2.0)/(Float'Max(5.0,Float'Min(Lowest_Voltage_Before_Degradation,Voltage)))**2.0;
        end if;
        --          Tau_Ng := Tau_Ng * (Lowest_Voltage_Before_Degradation**2.0)/(Float'Max(5.0,Float'Min(Lowest_Voltage_Before_Degradation,Voltage)))**2.0;
        Tau_Ng := Tau_Ng*Float'Min(5.0, Float'Max(1.0,18.5-Voltage));
      end if;
    elsif Ng_Ss < Ng_Last then
      if Engine_Seizure then
        Tau_Ng := 0.5;
      else
        Tau_Ng := Tau_Dec_Ng; -- (sec)
      end if;
    end if;

    -- Low Voltage Malfunction
    --  Capture Min Voltage during start
    --  Use min voltage so that can determine
    --  whether a low voltage start or normal start
    --  is occurring.
    if Starter_On then
       Next_Instance.Starter_Min_Voltage := Float'Min(Voltage, This_Instance.Starter_Min_Voltage);
       --  Log.Report("Pratt - Starter Min Voltage" & Float'Image(Next_Instance.Starter_Min_Voltage));
    else
       Next_Instance.Starter_Min_Voltage := 28.0;
    end if;

    if This_Instance.Rotor_Reset then
      Ng_Dot := 0.0;
    else
      Ng_Dot :=  (Ng_Ss - Ng_Last) / Tau_Ng;  -- (per sec)
    end if;

    -- Step 6.  Integrate the gas generator acceleration to
    -- obtain the new speed, and correct the result:
    if This_Instance.Rotor_Reset then
      Ng := Ng_SS;
    else
      Ng   := Float'Max(0.0,(Ng_Last + Ng_Dot*Iconst));  -- (%)
    end if;
    Ng_C := Ng / Theta_T**0.5/K_Ng_Malfunction; -- (%)
    Next_Instance.Gas_Generator_Rpm := Ng * 374.680;
    Next_Instance.Gas_Generator_Rpm_Corrected := Ng_C * 374.680;
    Next_Instance.Gas_Generator_Acceleration := Ng_Dot * 374.680;

    -- K_Ng_Malfunction Applies To Ng, but not to the corrected Ng.  Thus
    -- for this malfunction the PMU model and the cockpit indicator will register
    -- the malfunctioning value of Ng, but the engine model will continue to
    -- calculate normal levels of SHP and ITT, based on the non-malfunctioning
    -- value.

    -- Step 7.  Calculate the interstage turbine temperature.
    -- Statically, ITT is a function of the corrected gas generator
    -- speed, Mach number, accessory losses and bleed level.  Two
    -- additional devices have been included to achieve a realistic
    -- dynamic response.  First, the ITT lookup uses an adjusted gas
    -- generator speed which provides the temperature "lead" present
    -- during acceleration.  Second, a first-order lag has been
    -- imposed to smooth the ITT trace and to delay the decay of the
    -- ITT during engine deceleration.  Finally, the temperature
    -- depends on whether combination is present.

    if Next_Instance.Fired then

      -- Tau_Itt_Ng_C := Table TITTNGC = f(Ng_C)
      Tau_Itt_Ng_C := Interpolation_Table.Doubly_Indexed.Interpolate
        ( X  => Float(Ng_C),
          Y  => Float(Mach),
          Table => TITTNGC_Table'Access);

      -- Ng_C_Itt is the corrected gas generator speed adjusted upwards
      -- for acceleration.
      -- Ng_C_Itt := Ng_C + Tau_Itt_Ng_C * Float'Max(0.0,Ng_Dot);

      -- Itt_C_Ng_C := Table ITTCNGC = f(Ng_C_Itt)  ( degrees K )
      Itt_C_Ng_C := Interpolation_Table.Doubly_Indexed.Interpolate
        ( X => Float(Ng_C),
          Y => Float(Mach),
          Table   => ITTCNGC_Table'Access);

      -- K_Itt_Ng_Acc := Table KITTNG_ACC = f(Ng)
      K_Itt_Ng_Acc := Interpolation_Table.Singly_Indexed.Interpolate
        ( Input => Float(Ng),
          Table => KITTNG_ACC_Table'Access); -- deg K / Shp

      Next_Instance.Delta_Itt_C_Acc := (K_Itt_Ng_Acc * (Shp_Acc -7.5))/ Theta_T**0.4; --  deg. K

      -- K_Itt_Ng_Bleed := Table KITTNG_BLD = f(Ng)
      K_Itt_Ng_Bleed := Interpolation_Table.Singly_Indexed.Interpolate
        ( Input   => Float(Ng),
          Table   => KITTNG_BLD_Table'Access); -- deg K/lb/min

      Next_Instance.Delta_Itt_C_Bleed := (K_Itt_Ng_Bleed * (Wbleed_Dot - 5.2))/Theta_T**0.4; -- deg. K

      -- The corrected and uncorrected ITT with "lead" are then:
      --         Itt_C_Lead := Itt_C_Ng_C + Next_Instance.Delta_Itt_C_Acc + Delta_Itt_C_Bleed; -- (degrees K)
      Itt_C_Lead := Itt_C_Ng_C + Next_Instance.Delta_Itt_C_Acc + Next_Instance.Delta_Itt_C_Bleed + Tau_Itt_Ng_C*Float'Max(0.0,Ng_Dot); -- (degrees K)

      if Hot_Start then
        --  Delta_Itt_Hot  := F(Ng) = Table Ditthot -- deg c
        Delta_Itt_Hot :=  Interpolation_Table.Singly_Indexed.Interpolate
          (Input => Float(Ng),
           Table => DITTHOT_Table'Access ); -- deg C
      elsif Overtemp then
        Delta_Itt_Hot := Temperature_Types.Celsius(695.0);  -- previously was 650 deg c.
      elsif (Ng < 50.0) then
         --        if not Starter_On then -- more heat due to premature cutout
         ---            Delta_Itt_Hot := Temperature_Types.Celsius( 800.0 - 16.0*Ng );
         --            Delta_Itt_Hot := 5.0*(Ng_For_Zero_Starter_Torque - Ng);
         --             Delta_Itt_Hot := Float'Max(0.0,( Ng_For_Zero_Starter_Torque - Ng))*60.0*Float'min(5.0, Float'Max(Lowest_Voltage_Before_Degradation - Voltage,0.0));
         if (Ng <= 42.0 and Ng >= 18.0) and (Next_Instance.Starter_Min_Voltage < 19.8) and (Voltage >= 18.0 and Voltage <= 23.0) then
--            Delta_Itt_Hot := 455.0*Cos((Ng - 30.0)     *Ada.Numerics.Pi/24.0)
--                                  *Float'Max(Cos((Voltage-20.5)  *Ada.Numerics.Pi/5.0 ),
--                                             Float'Max(Cos((Voltage-21.0)  *Ada.Numerics.Pi/5.0 ),
--                                                       Float'Max(Cox((Voltage-21.5)*Ada.Numerics.Pi/5.0),
--                                                       Cos((Voltage-22.0)  *Ada.Numerics.Pi/5.0 ))));
            Delta_Itt_Hot := 455.0*Cos((Ng - 30.0)     *Ada.Numerics.Pi/24.0)
                                  *Cos((Voltage-21.25)  *Ada.Numerics.Pi/6.5 ); -- This piece of the eq changes the start characteristic drastically.
         --  Log.Report("Adding Delta_Itt_Hot = " & Float'Image(Delta_Itt_Hot));
         else
            Delta_Itt_Hot := 0.0;
         end if;
         --      elsif Voltage < Lowest_Voltage_Before_Degradation then -- more heat due to slow starter
         --         elsif Starter_On then
         --            Delta_Itt_Hot := 50.0*Float'min(5.0, Float'Max((Lowest_Voltage_Before_Degradation - Voltage)**2.0,0.0));
         --         end if;
         --          Delta_Itt_Hot := Temperature_Types.Celsius((800.0 - 16.0*Ng) * Float'Max(0.0,Float'Min(1.0, 2.625 - 0.125*Voltage)));
         --          Delta_Itt_Hot := Temperature_Types.Celsius((800.0 - 9.0*Ng) * Float'Max(0.0,Float'Min(1.0, 2.625 - 0.08*Voltage)));
         --            Delta_Itt_Hot := 0.0;
         --        else
         --          Delta_Itt_Hot := 0.0;
         --        end if;
      else
        Delta_Itt_Hot := 0.0;
      end if;

      if Itt_Sensor_Malfunction  then
        if Itt > 950.0 and not Pullback then  -- This mechanism permits an overtemp and pullback delay.
           Pullback := True;
--        elsif Itt <= 850.0 and Pullback then
--           Pullback := False;
        elsif not Pullback and (K_Itt_Malfunction <= 1.000001) then
           K_Itt_Malfunction := 1.5;
        end if;

        if (Pullback) then
           if Itt > 850.0 then
              K_Itt_Malfunction := ((850.0 - Delta_Itt_Hot)/(Itt_C_Lead*Theta_T**0.4 - 273.15));
           end if;
        end if;
      elsif Overtemp then
        K_Itt_Malfunction := 1.0;
      else
        K_Itt_Malfunction := 1.0;
        Pullback := False;
      end if;
--      Log.Report("Gas Gen Pullback = " & Boolean'Image(Pullback));
--      Log.Report("Gas Gen K_Itt_Malf = " & Float'Image(K_Itt_Malfunction));

      Itt_Lead   := K_Itt_Malfunction*(Itt_C_Lead * Theta_T**0.4 - 273.15) + Delta_Itt_Hot; -- ( deg C )

      -- The time constant for the ITT lag is 1.2 seconds during normal operation:
      Tau_Itt_Lag := 1.2;  -- sec;

    else
      -- With combustion absent, the ITT is set to decay toward the outside
      -- air temperature:
      Itt_Lead := Theta_Amb*288.15 - 273.15; -- deg C

      -- The time constant for the ITT lag during cool down is a
      -- function of gas generator speed, yielding a slowing rate of
      -- decay as the gas generator spools down:

      --      Tau_Itt_Lag := 280.0 / Float'Max(1.0, Ng);
      Tau_Itt_Lag := 600.0 / Float'Max(1.0, Ng);
    end if;

    -- Finally, the lag is imposed with the appropriate time
    -- constant to produce the output ITT

    if This_Instance.Rotor_Reset or This_Instance.Cold_Soak_Reset then
      Itt := Itt_Lead_Last;
    else
      Itt := Itt_Last + (Itt_Lead_Last - Itt_Last)*Iconst/ Tau_Itt_Lag; -- deg C
    end if;
    -- where: Itt_Last and Itt_Lead_Last are the last pass values of Itt and Itt_lead.

    -- Step 8.
    if Next_Instance.Fired then

      -- Calculate the horsepower and torque
      -- at the output shaft.  Corrected power is a
      -- function primarily of corrected gas generator
      -- speed, with adjustments for Mach number and
      -- power turbine speed, accessory load and bleed level:

      -- Shp_C_Ng_C := Table Shpcngc = F(Ng_C, Mach)
      Shp_C_Ng_C := Interpolation_Table.Doubly_Indexed.Interpolate
        ( X => Float(Ng_C),
          Y => Float(Mach),
          Table   => SHPCNGC_Table'Access);

      -- K_SHP_Ng_Acc := Table KSHPNG_ACC = f(Ng)
      K_Shp_Ng_Acc := Interpolation_Table.Singly_Indexed.Interpolate
        ( Input   => Float(Ng),
          Table   => KSHPNG_ACC_Table'Access);

      Next_Instance.Delta_Shp_C_Acc := (K_Shp_Ng_Acc*(Shp_Acc-7.5))/(Delta_T*Theta_T**0.5);

      -- K_SHP_Ng_Bleed := Table KSHPNG_BLD = f(Ng)
      K_Shp_Ng_Bleed := Interpolation_Table.Singly_Indexed.Interpolate
        ( Input => Float(Ng),
          Table => KSHPNG_BLD_Table'Access);

      Next_Instance.Delta_Shp_C_Bleed := (K_Shp_Ng_Bleed * (Wbleed_Dot - 5.2))/(Delta_T*Theta_T**0.5);

      -- DdNp_Shp_C := Table DSHPCNPNGC = F(Ng_C)
      Ddnp_Shp_C := Interpolation_Table.Singly_Indexed.Interpolate
        ( Input => Float(Ng_C),
          Table => DSHPCNPNGC_Table'Access);

      -- K_SHP_C_Np := Table K_SHP_C_Np = F(Np_last)
      K_Shp_C_Np := Interpolation_Table.Singly_Indexed.Interpolate
        ( Input => Float(Np_Last),
          Table => KSHPCNP_Table'Access);

      Next_Instance.Delta_Shp_C_Np := Ddnp_Shp_C * K_Shp_C_Np;

      -- The corrected shaft horsepower is then:
      Shp_C := Shp_C_Ng_C + Next_Instance.Delta_Shp_C_Np + Next_Instance.Delta_Shp_C_Acc + Next_Instance.Delta_Shp_C_Bleed;

      -- The output shaft horsepower and torque are:

      Shp := Shp_C * Delta_T * Theta_T**0.5;
      Next_Instance.Q_Pt(X) := 5252.1 * Shp / Float'Max(1.0,Rpm_Last);   -- (lb-ft)

    else
--      if Starter_On then
--        Next_Instance.Q_Pt(X) := (0.108*Ng**2.0 - 6.5*Np)*(Delta_T/Theta_T);
        Next_Instance.Q_Pt(X) := (0.108*(Float'Min(Ng,Float'Max(24.0,Np)))**2.0 - 6.5*Np)*(Delta_T/Theta_T);
--      else
--        Next_Instance.Q_Pt(X) := Torque_Types.Ft_Lbf'Min(((0.108*Ng**2.0 - 6.5*Np)*(Delta_T/Theta_T)), This_Instance.Q_Pt(X));
--      end if;
      -- RAC doesn't calculate Shp for this condition in the check cases
      -- although it is included in the model.
      Shp := Next_Instance.Q_Pt(X)*Rpm_Last/5252.1;
    end if;

    -- Step 9.  Calculate the load due to the air
    -- conditioner compressor, which is drawn from the
    -- propeller shaft:
    -- Note: This code was moved the Environmental subsystem
    --
    --  if Air_Conditioner_On then
    --    Q_air_Conditioner := 33.35;  -- lb-ft
    --  else
    --    Q_air_Conditioner := 0.0;    -- lb-ft
    --  end if;

    -- Step 10.  Calculate the rate of change of
    -- propeller speed from the imbalance between
    -- torque supplied and torque absorbed, and the
    -- inertia of the power train:


    if Torque_Auto_Driven then
      declare
        Temp_Torque : Float := 0.0;
      begin
        Temp_Torque := Auto_Driven_Torque;
        Next_Instance.Q_Pt(X) := Temp_Torque + 0.91*Rpm_Dot;
        Shp := Next_Instance.Q_Pt(X) * Rpm/5252.1;
        Shp_C := Shp/ Delta_T/Theta_T;
        Q_Eng(X) := Temp_Torque;
        Rpm_Dot := (30.0*(Next_Instance.Q_Pt(X) - Q_Prop(X) - Q_Air_Conditioner(X))/(Pi*Ip_Accel)); -- (rev/min/sec)
      end;
    else
      declare
        Temp : Float;
      begin
        if This_Instance.Engine_Test_Case >= 1 then
          Temp :=  Next_Instance.Q_Pt(X) - Q_Air_Conditioner(X);
          Rpm_Dot := 30.0*(Next_Instance.Q_Pt(X) - Temp - Q_Air_Conditioner(X))/(Pi*Ip_Accel ); -- (rev/min/sec)
        else
          Rpm_Dot := 30.0*(Next_Instance.Q_Pt(X) - Q_Prop(X) - Q_Air_Conditioner(X))/(Pi*Ip_Accel ); -- (rev/min/sec)
        end if;
      end;


      Q_Eng(X) := Next_Instance.Q_Pt(X) - 0.91*Rpm_Dot; -- lb-ft;
      if This_Instance.Quick_Start and (Q_Eng(X) > 100.0) then
        Q_Eng(X) := 100.0;
      end if;
    end if;


    -- Step 11.  Integrate the propeller acceleration
    -- to obtain the current propeller speed.  Also
    -- calculate the power turbine speed:

    if This_Instance.Quick_Start then
      Rpm := 2000.0;
    elsif This_Instance.Engine_Test_Case = 1 then
      Rpm := 0.0;
    elsif This_Instance.Engine_Test_Case = 2 then
      Rpm := 2000.0;
    elsif This_Instance.Engine_Test_Case = 3  then
      Rpm := 1800.0;
    elsif This_Instance.Engine_Test_Case = 4 then
      Rpm := 2050.0;
    elsif This_Instance.Engine_Test_Case = 5 then
      Rpm := 2000.0;
    elsif This_Instance.Engine_Test_Case = 6 then
      Rpm := 2000.0;
    elsif This_Instance.Engine_Test_Case = 7 then
      Rpm := 2000.0;
    else
      Rpm := Float'Min(3000.0,Float'Max(0.0,(Rpm_Last + (1.5*Rpm_Dot-0.5*Rpm_Dot_Last_Pass)*Iconst))); -- (rev/min/sec)
    end if;

    Np :=  Rpm / 20.0; -- (%)  -- 2000 rpm = 100%

    -- Step 12.  Calculate the exhaust thrust.  This
    -- is a function of corrected power, Mach number
    -- and the static pressure ratio:
    if Next_Instance.Fired then

      -- FN_C_Shp_c := table FNCSHPC = f(FNCSHPC)

      Fn_C_Shp_C := Interpolation_Table.Doubly_Indexed.Interpolate
        ( X => Float(Shp_C),
          Y => Float(Mach),
          Table   => Fncshpc_Table'Access);
      Fn := Fn_C_Shp_C * Delta_Amb; -- (lbs)
    else
      Fn := 0.0;
    end if;

    -- -- Step 13.  Transform the exhaust thrust into
    -- -- force components in the body reference system
    -- -- and pitching moment about the cg, after first
    -- -- calculating the moment arms for the exhaust
    -- -- thrust relative to the aircraft c.g.:  (ft)
    --                                            (ft)
    --                                            (lbs)
    --                                            (lbs)
    --                                            (lb-ft)

    -- Step 14. Oil Model
    -- Lookup the normal oil pressure, a function of gas generator speed
    -- since the oil pump is driven from the accessory gearbox:

    if Oil_Leak then
      -- oil supply and pressure to zero in about 1.0 minute
      Next_Instance.Oil_Leak_Timer := Float'Min(600.0,This_Instance.Oil_Leak_Timer + Iconst);
      Next_Instance.P_Engine_Oil_Loss := -1.8*Next_Instance.Oil_Leak_Timer;
    else
      Next_Instance.Oil_Leak_Timer := 0.0;
      Next_Instance.P_Engine_Oil_Loss := 0.0;
    end if;

--    Next_Instance.P_Engine_Oil_Ng := Interpolation_Table.Singly_Indexed.Interpolate
--      ( Input => Float(Ng),
--        Table => PENGOIL_Table'Access);  -- psi
      Next_Instance.P_Engine_Oil_Ng := Interpolation_Table.Doubly_Indexed.Interpolate
        ( X => Float(Ng),
          Y => Float(JPATS_Powerplant_Types.Oil_Pressure_Failure_Type'Pos(Oil_Pressure_Failure)),
          Table => PENGOIL_Table'Access);  -- psi
    -- Oil Pressure degradation due to load factor with 1.0 sec lag.
    Next_Instance.P_Engine_Oil_Nz_Prime := Interpolation_Table.Singly_Indexed.Interpolate
      (Input => Float(-Load_Factor.Z),
       Table => DPEOILNZ_Table'Access);

--    Ada.Text_IO.Put("Load Factor := " & Float'Image(Load_Factor.Z));
--    Next_Instance.P_Engine_Oil_Nz := This_Instance.P_Engine_Oil_Nz + (Next_Instance.P_Engine_Oil_Nz_Prime - This_Instance.P_Engine_Oil_Nz);
    Next_Instance.P_Engine_Oil_Nz := Front(Obj => Next_Instance.Engine_Oil_Pressure_Lag);
    Pop(Obj => Next_Instance.Engine_Oil_Pressure_Lag);
    Append
      (Obj  => Next_Instance.Engine_Oil_Pressure_Lag,
       Elem => Next_Instance.P_Engine_Oil_Nz_Prime);

      Next_Instance.Oil_Pressure := Float'Max(0.0,(Next_Instance.P_Engine_Oil_Ng + Next_Instance.P_Engine_Oil_Loss + Next_Instance.P_Engine_Oil_Nz ));

    if Next_Instance.Fired then
      Next_Instance.Oil_Temperature_Steady_State :=
        Float'Max(57.0, (52.0 + Ambient_Temperature_Ratio*288.15 - 273.15 +0.0018*Q_Eng(X)
                         + 0.0010*Hp - 0.3*Float'Min(0.0, Mach_Number*49.02*(Ambient_Temperature_Ratio*518.67)**0.5 )));
      if High_Oil_Temperature then
        Next_Instance.Oil_Temperature_Steady_State := Float'Min(127.0,Next_Instance.Oil_Temperature_Steady_State *2.05);
      end if;
    else
      Next_Instance.Oil_Temperature_Steady_State := Ambient_Temperature_Ratio*288.15-273.15;
    end if;

    if (Next_Instance.Fired and This_Instance.Oil_Temperature < 57.0) then -- or (Next_Instance.Fired and High_Oil_Temperature) then
      Next_Instance.Oil_Temperature := This_Instance.Oil_Temperature + 0.23*Iconst;
    else
      if High_Oil_Temperature or This_Instance.Oil_Temperature > 57.0 then
        Next_Instance.Oil_Temperature := This_Instance.Oil_Temperature
          + Float'Max((-0.80*Iconst),
                      Float'Min((0.80*Iconst),
                                (Next_Instance.Oil_Temperature_Steady_State - This_Instance.Oil_Temperature)));
      else
        Next_Instance.Oil_Temperature := This_Instance.Oil_Temperature
          + Float'Max((-0.055*Iconst),
                      Float'Min((0.055*Iconst),
                                (Next_Instance.Oil_Temperature_Steady_State - This_Instance.Oil_Temperature)));
      end if;
    end if;
    if This_Instance.Rotor_Reset or This_Instance.Cold_Soak_Reset then
      Next_Instance.Oil_Temperature := Next_Instance.Oil_Temperature_Steady_State;
    end if;

    -- Inlet Sensor
    declare
      Outside_Air_Temperature : Temperature_Types.Celsius := Ambient_Temperature_Ratio*288.15 - 273.15;
    begin
      Delta_T1_Mach := Interpolation_Table.Singly_Indexed.Interpolate
        ( Input => Float(Mach),
          Table => DT1MACH_Table'Access);

      if Next_Instance.Fired then
        if Mach < 0.07  and not This_Instance.Rotor_Reset then
          if (This_Instance.T1_Prime <= (Outside_Air_Temperature + Delta_T1_Mach)) then
            Next_Instance.T1_Prime := Float'Min((Outside_Air_Temperature + Delta_T1_Mach),(This_Instance.T1_Prime + 0.033*Iconst));
          else
            Next_Instance.T1_Prime := Float'Max((Outside_Air_Temperature + Delta_T1_Mach),(This_Instance.T1_Prime - 0.2*Iconst));
          end if;
        else
          Next_Instance.T1_Prime := Outside_Air_Temperature  + Delta_T1_Mach;
          -- where: Delta_t1_mach is f(Mach) from table DT1MACH
        end if;
        Next_Instance.Time_Since_Fired := 0.0;
      else
        Next_Instance.Time_Since_Fired := Float'Min(800.0,This_Instance.Time_Since_Fired + Iconst);
        if Mach < 0.07 then
          if This_Instance.Cold_Soak_Reset or This_Instance.Rotor_Reset then
            Next_Instance.T1_Prime := Outside_Air_Temperature;
            Next_Instance.Time_Since_Fired := 800.0;
          else
            if Next_Instance.Time_Since_Fired < 600.0 then
              Next_Instance.T1_Prime := This_Instance.T1_Prime + (0.01*Outside_Air_Temperature
                                                                  - 0.30*Float'Max(0.0,Float'Min(10.0, Ng - 10.0 )))*Iconst;
            else
              Next_Instance.T1_Prime :=  Exp(-Iconst/600.0)*This_Instance.T1_Prime
                                        + (1.0 - Exp(-Iconst/600.0))*Outside_Air_Temperature;
            end if;

            Next_Instance.T1_Prime := Float'Min(2.45*Outside_Air_Temperature, Float'Max(Outside_Air_Temperature, Next_Instance.T1_Prime));
            -- This code provides a temperature rise following
            -- shutdown to 2.45 * Outside Air Temperature; motoring
            -- the engine will cool this down temporarily and allow
            -- the pmu to function on a hot day.
          end if;
        else
          Next_Instance.T1_Prime := Outside_Air_Temperature + Delta_T1_Mach;
        end if;
      end if;
      if This_Instance.Rotor_Reset then
        Next_Instance.T1 := Next_Instance.T1_Prime;
      else
        Next_Instance.T1 := This_Instance.T1 + (Next_Instance.T1_Prime - This_Instance.T1)*Iconst/10.0;
      end if;
    end;

    -- Chip Sensor
    Next_Instance.Chip_Sensor := Chip;

    -- Bleed Model
    -- This is where the bleed model is calculated.

    -- Start Mode for Gauge to IOS
    if Ng < 40.0  then -- and not This_Instance.Itt_Start_Mode then
      Next_Instance.Itt_Start_Mode := True;
    end if;
    Pop (Obj => Next_Instance.Itt_Queue);
    Append
      ( Obj  => Next_Instance.Itt_Queue,
        Elem => Itt);
    Next_Instance.Itt_Trigger_Signal := (Front ( Obj => Next_Instance.Itt_Queue) - Itt) > 5.0;
    if Ng > 50.0 and Next_Instance.Itt_Trigger_Signal and Itt < 820.0 and This_Instance.Itt_Start_Mode then
      Next_Instance.Itt_Start_Mode := False;
    end if;

    if Next_Instance.Record_Debug_Information then
      declare
        Mach_Label : String := "Mach";
        Wf_Dot_Label : String := "wf_dot";
        Wf_Dot_C_Label : String := "wf_dot_c";
        Ng_C_SS_Label : String := "Ng_c_ss";
        Delta_Ng_C_Acc_Label : String := "delta_ng_c_acc";
        Delta_Ng_C_Bleed_Label : String := "Delta_ng_c_bleed";
        ng_ss_Label : String := "ng_ss";
        Ng_dot_Label : String := "Ng_dot";
        tau_ng_Label : String := "tau_ng";
        ng_Label : String := "ng";
        ng_c_Label : String := "ng_c";
        itt_c_lead_Label : String := "itt_c_lead";
        ng_c_itt_Label : String := "ng_c_itt";
        tau_itt_ng_c_Label : String := "tau_itt_ng_c";
        delta_itt_c_acc_Label : String := "delta_itt_c_acc";
        delta_itt_c_bleed_Label : String := "delta_itt_c_bleed";
        itt_lead_Label : String := "itt_lead";
        tau_itt_lag_Label : String := "tau_itt_lag";
        itt_Label : String := "itt";
        shp_c_Label : String := "shp_c";
        Delta_Shp_C_Np_Label : String := "delta_shp_c_np";
--      ddnp_shp_c_Label : String := "ddnp_shp_c";
        delta_shp_c_acc_Label : String := "delta_shp_c_acc";
        delta_shp_c_bleed_Label : String := "delta_shp_c_bleed";
        shp_Label : String := "shp";
        q_ptx_Label : String := "q_pt(x)";
        rpm_dot_Label : String := "rpm_dot";
        q_engx_Label : String := "q_eng(x)";
        rpm_Label : String := "rpm";
        np_Label : String := "np";
        fn_Label : String := "fn";
--        xex_Label : String := "xex";
--        zex_Label : String := "zex";
--        mex_Label : String := "mex";
        pengine_oil_Label : String := "pengine_oil";
        tengine_oil_Label : String := "tengine_oil";

      begin
        String'Output(Next_Instance.Debug_Stream, Mach_Label);
        Float'Output(Next_Instance.Debug_Stream, Input_Mach_Number);
        String'Output(Next_Instance.Debug_Stream, Wf_Dot_Label);
        Float'Output(Next_Instance.Debug_Stream, Input_Fuel_Flow_Rate);
        String'Output(Next_Instance.Debug_Stream, Wf_Dot_C_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Fuel_Flow_Corrected);
        String'Output(Next_Instance.Debug_Stream, Ng_C_Ss_Label);
        Float'Output(Next_Instance.Debug_Stream, Ng_C_Ss);
        String'Output(Next_Instance.Debug_Stream, Delta_Ng_C_Acc_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Ng_C_Acc);
        String'Output(Next_Instance.Debug_Stream, Delta_Ng_C_Bleed_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Ng_C_Bleed);
        String'Output(Next_Instance.Debug_Stream, Ng_Ss_Label);
        Float'Output(Next_Instance.Debug_Stream, Ng_SS);
        String'Output(Next_Instance.Debug_Stream, Ng_Dot_Label);
        Float'Output(Next_Instance.Debug_Stream, Ng_Dot);
        String'Output(Next_Instance.Debug_Stream, Tau_Ng_Label);
        Float'Output(Next_Instance.Debug_Stream, Tau_Ng);
        String'Output(Next_Instance.Debug_Stream, Ng_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Gas_Generator_Rpm_In_Percent);
        String'Output(Next_Instance.Debug_Stream, Ng_C_Label);
        Float'Output(Next_Instance.Debug_Stream, Ng_C);
        String'Output(Next_Instance.Debug_Stream, Itt_C_Lead_Label);
        Float'Output(Next_Instance.Debug_Stream, Itt_C_Lead);
        String'Output(Next_Instance.Debug_Stream, ng_c_itt_Label);
        Float'Output(Next_Instance.Debug_Stream, Ng_C_Itt);
        String'Output(Next_Instance.Debug_Stream, Tau_Itt_Ng_C_Label);
        Float'Output(Next_Instance.Debug_Stream, Tau_Itt_Ng_C);
        String'Output(Next_Instance.Debug_Stream, Delta_Itt_C_Acc_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Itt_C_Acc);
        String'Output(Next_Instance.Debug_Stream, Delta_Itt_C_Bleed_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Itt_C_Bleed);
        String'Output(Next_Instance.Debug_Stream, Itt_Lead_label      );
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Internal_Turbine_Temperature_Lead_Lag);
        String'Output(Next_Instance.Debug_Stream, Tau_Itt_Lag_Label);
        Float'Output(Next_Instance.Debug_Stream, Tau_Itt_Lag);
        String'Output(Next_Instance.Debug_Stream, Itt_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Internal_Turbine_Temperature);
        String'Output(Next_Instance.Debug_Stream, Shp_C_Label);
        Float'Output(Next_Instance.Debug_Stream, Shp_C);
--        String'Output(Next_Instance.Debug_Stream, Ddnp_Shp_C_Label);
        String'Output(Next_Instance.Debug_Stream, Delta_Shp_C_Np_Label);
--        Float'Output(Next_Instance.Debug_Stream, Ddnp_Shp_C);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Shp_C_Np);
        String'Output(Next_Instance.Debug_Stream, Delta_Shp_C_Acc_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Shp_C_Acc);
        String'Output(Next_Instance.Debug_Stream, Delta_Shp_C_Bleed_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Delta_Shp_C_Bleed);
        String'Output(Next_Instance.Debug_Stream, Shp_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Shaft_Horsepower);
        String'Output(Next_Instance.Debug_Stream, Q_ptx_label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Q_Pt(X));
        String'Output(Next_Instance.Debug_Stream, Rpm_Dot_label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Power_Turbine_Acceleration);
        String'Output(Next_Instance.Debug_Stream, Q_Engx_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Torque(X));
        String'Output(Next_Instance.Debug_Stream, Rpm_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Power_Turbine_Rpm);
        String'Output(Next_Instance.Debug_Stream, Np_label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Power_Turbine_Rpm_In_Percent);
        String'Output(Next_Instance.Debug_Stream, Fn_label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Exhaust_Thrust);
        String'Output(Next_Instance.Debug_Stream, Pengine_Oil_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Oil_Pressure);
        String'Output(Next_Instance.Debug_Stream, Tengine_Oil_Label);
        Float'Output(Next_Instance.Debug_Stream, Next_Instance.Oil_Temperature);

      end;
    end if;

    -- clear quick start is set
    -- write to next_instance rotor reset only if set.
    if This_Instance.Quick_Start then
      Next_Instance.Quick_Start := False;
    end if;

    -- clear cold soak reset is set
    -- write to next_instance cold soak reset only if set.
    if This_Instance.Cold_Soak_Reset then
      Next_Instance.Cold_Soak_Reset := False;
    end if;

    -- clear rotor reset if set
    -- write to next_instance rotor reset only if set.
    if This_Instance.Rotor_Reset then
      Next_Instance.Rotor_Reset := False;
    end if;

  exception
      when Lag_Time_Exception =>
      Ada.Text_Io.Put_Line("Lag_Time_Exception in Pratt_Whitney_Pt6a_68_Gas_Generator");
      Ada.Text_Io.Put_Line("The array size for the oil pressure lag is not large enough.");
      Ada.Text_Io.Put_Line("Increase Lag_Array_Size to a minimum of ");
      raise;
--    when others =>
--      Ada.Text_Io.Close(File => Eng1_Test_File);
--      raise;
  end Update;

  -- Method Initialize
  procedure Initialize
    (An_Instance : in out Instance) is
  begin
    An_Instance.Engine_Test_Case := 0;
    An_Instance.Record_Debug_Information := False;
    An_Instance.Fuel_Flow_Corrected := Mass_Types.Pph(0.0);
    An_Instance.Gas_Generator_Rpm := Angle_Types.Rpm(0.0);
    An_Instance.Gas_Generator_Rpm_Corrected := Angle_Types.Rpm(0.0);
    An_Instance.Gas_Generator_Rpm_In_Percent := Normalized_Types.Normalize(0.0);
    An_Instance.Gas_Generator_Acceleration := Angle_Types.Rpm_Per_Sec(0.0);
    An_Instance.Power_Turbine_Rpm := Angle_Types.Rpm(0.0);
    An_Instance.Power_Turbine_Rpm_Corrected := Angle_Types.Rpm(0.0);
    An_Instance.Power_Turbine_Rpm_In_Percent := Normalized_Types.Normalize(0.0);
    An_Instance.Power_Turbine_Acceleration := Angle_Types.Rpm_Per_Sec(0.0);
    An_Instance.Internal_Turbine_Temperature := Temperature_Types.Celsius(15.0);
    An_Instance.Internal_Turbine_Temperature_Lead_Lag := Temperature_Types.Celsius(15.0);
    An_Instance.Torque :=
      (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
    An_Instance.Shaft_Horsepower := Torque_Types.Horsepower(0.0);
    An_Instance.Propeller_Rpm := Angle_Types.Rpm(0.0);
    An_Instance.Exhaust_Thrust := Force_Types.Lbf(0.0);
    An_Instance.Fired := False;
    An_Instance.Bleed_Pressure := Force_Types.Psi(14.7);
    An_Instance.Bleed_Temperature := Temperature_Types.Celsius(15.0);
    An_Instance.Delta_Shp_C_Acc   := Torque_Types.Horsepower(0.0);
    An_Instance.Delta_Shp_C_Bleed := Torque_Types.Horsepower(0.0);
    An_Instance.Delta_Shp_C_Np    := Torque_Types.Horsepower(0.0);
    An_Instance.Delta_Itt_C_Acc   := Temperature_Types.Kelvin(0.0);
    An_Instance.Delta_Itt_C_Bleed := Temperature_Types.Kelvin(0.0);
    An_Instance.Delta_Ng_C_Acc    := Angle_Types.Rpm(0.0);
    An_Instance.Delta_Ng_C_Bleed  := Angle_Types.Rpm(0.0);
    An_Instance.Delta_Ng_Seizure  := Angle_Types.Rpm(0.0);
    An_Instance.Ignition_Exciter_Load := Electrical_Units_Types.Amps(0.0);
    An_Instance.Oil_Pressure      := Force_Types.Psi(0.0);
    An_Instance.Oil_Temperature   := Temperature_Types.celsius(15.0);
    An_Instance.Oil_Temperature_Steady_State   := Temperature_Types.celsius(15.0);
    An_Instance.Chip_Sensor       := False;
    An_Instance.P_Engine_Oil_Ng   := Force_Types.Psi(0.0);
    An_Instance.P_Engine_Oil_Nz_Prime   := Force_Types.Psi(0.0);
    An_Instance.P_Engine_Oil_Nz   := Force_Types.Psi(0.0);
    An_Instance.P_Engine_Oil_Loss := Force_Types.Psi(0.0);
    An_Instance.Q_Pt :=
      (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0));
    An_Instance.Seizure_Time      := 0.0;
    An_Instance.Oil_Leak_Timer    := 0.0;
    An_Instance.T1                := Temperature_Types.Celsius(15.0);
    An_Instance.T1_Prime          := Temperature_Types.Celsius(15.0);
    An_Instance.Rotor_Reset       := True;
    An_Instance.Cold_Soak_Reset   := True;
    An_Instance.Quick_Start       := False;
    An_Instance.Time_Since_Fired  := 600.0;
    An_Instance.Wf_Dot            := Mass_Types.Pph(0.0);
    An_Instance.Flameout_Timer    := 0.0;
    An_Instance.Itt_Start_Mode    := True;
    An_Instance.Itt_Trigger_Signal := False;
    while
      Available
      ( Obj => An_Instance.Engine_Oil_Pressure_Lag) >= 1
    loop
      Append
        (Obj  =>  An_Instance.Engine_Oil_Pressure_Lag,
         Elem => 0.0);
    end loop;
    while
      Available
      ( Obj => An_Instance.Itt_Queue) >= 1
    loop
      Append
        (Obj =>  An_Instance.Itt_Queue,
         Elem => 0.0);
    end loop;


    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Engine_Test_Case",
       Units          => "integer",
       Object_Address => An_Instance.Engine_Test_Case'Address,
       Value_Type     => "int",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Fuel_Flow_Corrected",
       Units          => "float",
       Object_Address => An_Instance.Fuel_Flow_Corrected'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Gas_Generator_Rpm",
       Units          => "float",
       Object_Address => An_Instance.Gas_Generator_Rpm'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Gas_Generator_Rpm_Corrected",
       Units          => "float",
       Object_Address => An_Instance.Gas_Generator_Rpm_Corrected'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Gas_Generator_Rpm_In_Percent",
       Units          => "float",
       Object_Address => An_Instance.Gas_Generator_Rpm_In_Percent'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Gas_Generator_Acceleration",
       Units          => "float",
       Object_Address => An_Instance.Gas_Generator_Acceleration'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Power_Turbine_Rpm",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Rpm'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Power_Turbine_Rpm_Corrected",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Rpm_Corrected'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Power_Turbine_Rpm_In_Percent",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Rpm_In_Percent'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Power_Turbine_Acceleration",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Acceleration'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Internal_Turbine_Temperature",
       Units          => "float",
       Object_Address => An_Instance.Internal_Turbine_Temperature'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Internal_Turbine_Temperature_Lead_Lag",
       Units          => "float",
       Object_Address => An_Instance.Internal_Turbine_Temperature_Lead_Lag'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Torque",
       Units          => "float",
       Object_Address => An_Instance.Torque'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Shaft_Horsepower",
       Units          => "float",
       Object_Address => An_Instance.Shaft_Horsepower'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Propeller_Rpm",
       Units          => "float",
       Object_Address => An_Instance.Propeller_Rpm'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Fired",
       Units          => "Boolean",
       Object_Address => An_Instance.Fired'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Bleed_Pressure",
       Units          => "float",
       Object_Address => An_Instance.Bleed_Pressure'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Bleed_Temperature",
       Units          => "float",
       Object_Address => An_Instance.Bleed_Temperature'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Delta_Shp_C_Acc",
       Units          => "float",
       Object_Address => An_Instance.Delta_Shp_C_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Delta_Shp_C_Bleed",
       Units          => "float",
       Object_Address => An_Instance.Delta_Shp_C_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Delta_Shp_C_Np",
       Units          => "float",
       Object_Address => An_Instance.Delta_Shp_C_Np'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Delta_Itt_C_Acc",
       Units          => "float",
       Object_Address => An_Instance.Delta_Itt_C_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Delta_Itt_C_Bleed",
       Units          => "float",
       Object_Address => An_Instance.Delta_Itt_C_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Delta_Ng_C_Acc",
       Units          => "float",
       Object_Address => An_Instance.Delta_Ng_C_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Delta_Ng_C_Bleed",
       Units          => "float",
       Object_Address => An_Instance.Delta_Ng_C_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Ignition_Exciter_Load",
       Units          => "float",
       Object_Address => An_Instance.Ignition_Exciter_Load'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Oil_Pressure",
       Units          => "float",
       Object_Address => An_Instance.Oil_Pressure'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Oil_Temperature",
       Units          => "float",
       Object_Address => An_Instance.Oil_Temperature'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Chip_Sensor",
       Units          => "Boolean",
       Object_Address => An_Instance.Chip_Sensor'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Torque",
       Units          => "float",
       Object_Address => An_Instance.Torque'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Power_Turbine_Inertia",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Inertia'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/Exhaust_Thrust",
       Units          => "float",
       Object_Address => An_Instance.Exhaust_Thrust'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    -----------------------------------------------------------
    --    Theta_Amb                  : Normalized_Types.Normalize                             renames Ambient_Temperature_Ratio;
    --    Delta_Amb                  : Normalized_Types.Normalize                             renames Ambient_Pressure_Ratio;
    --    Mach                       : Length_Types.Mach                                      renames Mach_Number;
    --    Theta_T                    : Normalized_Types.Normalize                             renames Total_Temperature_Ratio;
    --    Delta_T                    : Normalized_Types.Normalize                             renames Total_Pressure_Ratio;
    --    Wf_Dot                     : Mass_Types.Pph                                         renames Fuel_Flow_Rate;
    --    Hp                         : Length_Types.Feet                                      renames Pressure_Altitude;
    --    Wbleed_Dot                 : Mass_Types.Ppm                                         renames Bleed_Flow_Rate;
    --    Q_Prop                     : Jpats_Propeller_Types.Torque_Vector_Type.Instance      renames Propeller_Torque_Absorbed;
    --    Q_Air_Conditioner          : Jpats_Propeller_Types.Torque_Vector_Type.Instance      renames Air_Conditioner_Torque_Absorbed;
    --    Shp_Acc                    : Torque_Types.Horsepower                                renames Accessory_Horsepower;
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-wfdot",
       Units          => "float",
       Object_Address => Input_Fuel_Flow_Rate'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Theta_Ambient",
       Units          => "float",
       Object_Address => input_Ambient_Temperature_Ratio'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Ambient",
       Units          => "float",
       Object_Address => input_Ambient_Pressure_Ratio'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Mach",
       Units          => "float",
       Object_Address => Input_Mach_Number'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Hp",
       Units          => "float",
       Object_Address => Pressure_Altitude'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ip_Accel",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Inertia'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-iconst",
       Units          => "float",
       Object_Address => Input_Iconst'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-No_Light_Request",
       Units          => "Boolean",
       Object_Address => Input_No_Light_Request'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Shp_Acc",
       Units          => "float",
       Object_Address => Input_Accessory_Horsepower'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-wdotbleed",
       Units          => "float",
       Object_Address => Input_Bleed_Flow_Rate'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    --    Stethoscope.Register_Signal
    --      (Name           => "Powerplant/GG/local-Starter_On",
    --       Units          => "Boolean",
    --       Object_Address => Input_Accessory_Torque_Absorbed'Address,
    --       Value_Type     => "uchar",
    --       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Starter_On",
       Units          => "Boolean",
       Object_Address => Starter_On'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ignitors_On",
       Units          => "Boolean",
       Object_Address => Input_Ignition_Exciter'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Fired",
       Units          => "Boolean",
       Object_Address => An_Instance.Fired'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng_Fire",
       Units          => "float",
       Object_Address => Ng_Fire'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Theta_T",
       Units          => "float",
       Object_Address => Total_Temperature_Ratio'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_T",
       Units          => "float",
       Object_Address => Total_Pressure_Ratio'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-wfdot",
       Units          => "float",
       Object_Address => Input_Fuel_Flow_Rate'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Wf_Dot_C",
       Units          => "float",
       Object_Address => An_Instance.Fuel_Flow_Corrected'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng_C_SS",
       Units          => "float",
       Object_Address => Ng_C_SS'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng_C_Wf_Dot_C",
       Units          => "float",
       Object_Address => Ng_C_Wf_Dot_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Ng_C_Acc",
       Units          => "float",
       Object_Address => An_Instance.Delta_Ng_C_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Ng_C_Bleed",
       Units          => "float",
       Object_Address => An_Instance.Delta_Ng_C_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Ng_Wf_Dot_Acc",
       Units          => "float",
       Object_Address => K_Ng_Wf_Dot_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Ng_Wf_Dot_Bleed",
       Units          => "float",
       Object_Address => K_Ng_Wf_Dot_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng_SS",
       Units          => "float",
       Object_Address => Ng_SS'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng_Dot",
       Units          => "float",
       Object_Address => Ng_Dot'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Tau_Ng",
       Units          => "float",
       Object_Address => Tau_Ng'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Tau_Acc_Ng",
       Units          => "float",
       Object_Address => Tau_Acc_Ng'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Tau_Dec_Ng",
       Units          => "float",
       Object_Address => Tau_Dec_Ng'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Ng_Malfunction",
       Units          => "float",
       Object_Address => K_Ng_Malfunction'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng",
       Units          => "float",
       Object_Address => An_Instance.Gas_Generator_Rpm_In_Percent'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng_C",
       Units          => "float",
       Object_Address => Ng_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Itt_Malfunction",
       Units          => "float",
       Object_Address => K_Itt_Malfunction'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Itt_C_Lead",
       Units          => "float",
       Object_Address => Itt_C_Lead'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Itt_C_Ng_C",
       Units          => "float",
       Object_Address => Itt_C_Ng_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Tau_Itt_Ng_C",
       Units          => "float",
       Object_Address => Tau_Itt_Ng_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Itt_C_Acc",
       Units          => "float",
       Object_Address => An_Instance.Delta_Itt_C_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Itt_C_Bleed",
       Units          => "float",
       Object_Address => An_Instance.Delta_Itt_C_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Itt_Ng_Acc",
       Units          => "float",
       Object_Address => K_Itt_Ng_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Itt_Ng_Bleed",
       Units          => "float",
       Object_Address => K_Itt_Ng_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-ITT_Lead",
       Units          => "float",
       Object_Address => An_Instance.Internal_Turbine_Temperature_Lead_Lag'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Itt_Hot",
       Units          => "float",
       Object_Address => Delta_Itt_Hot'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Tau_Itt_Lag",
       Units          => "float",
       Object_Address => Tau_Itt_Lag'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-ITT",
       Units          => "float",
       Object_Address => An_Instance.Internal_Turbine_Temperature'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Shp_C",
       Units          => "float",
       Object_Address => Shp_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Shp_C_Ng_C",
       Units          => "float",
       Object_Address => Shp_C_Ng_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Shp_C_Np",
       Units          => "float",
       Object_Address => An_Instance.Delta_Shp_C_Np'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ddnp_Shp_C",
       Units          => "float",
       Object_Address => Ddnp_Shp_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Shp_C_Np",
       Units          => "float",
       Object_Address => K_Shp_C_Np'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Shp_C_Acc",
       Units          => "float",
       Object_Address => An_Instance.Delta_Shp_C_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_Shp_C_Bleed",
       Units          => "float",
       Object_Address => An_Instance.Delta_Shp_C_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Shp_Ng_Acc",
       Units          => "float",
       Object_Address => K_Shp_Ng_Acc'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-K_Shp_Ng_Bleed",
       Units          => "float",
       Object_Address => K_Shp_Ng_Bleed'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Shp",
       Units          => "float",
       Object_Address => An_Instance.Shaft_Horsepower'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
--    Stethoscope.Register_Signal
--      (Name           => "Powerplant/GG/Q_Pt(X)",
--       Units          => "float",
--       Object_Address => Next_Instance.Q_Pt(X)'Address,
--       Value_Type     => "float",
--       Scope_Index    => 0);
    --     input_Air_Conditioner_Torque_Absorbed : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
    --     input_Propeller_Torque_Absorbed       : Jpats_Propeller_Types.Torque_Vector_Type.Instance;
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Rpm_Dot",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Acceleration'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Q_eng(X)",
       Units          => "float",
       Object_Address => An_Instance.Torque(X)'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Rpm",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Rpm'Address,
       Value_Type     => "float",       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Np",
       Units          => "float",
       Object_Address => An_Instance.Power_Turbine_Rpm_In_Percent'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Fn_C_Shp_C",
       Units          => "float",
       Object_Address => Fn_C_Shp_C'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Fn",
       Units          => "float",
       Object_Address => An_Instance.Exhaust_Thrust'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Ng_C_Itt",
       Units          => "float",
       Object_Address => Ng_C_Itt'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Hot_Start",
       Units          => "Boolean",
       Object_Address => Input_Hot_Start'Address,
       Value_Type     => "uchar",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Delta_T1_Mach",
       Units          => "float",
       Object_Address => Delta_T1_Mach'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-T1_Prime",
       Units          => "float",
       Object_Address => An_Instance.T1_Prime'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-T1",
       Units          => "float",
      Object_Address => An_Instance.T1'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/GG/local-Tau_Acc_Tune",
       Units          => "float",
      Object_Address =>  Tau_Acc_Tune'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
  end Initialize;

  procedure Read_Tables is
    Powerplant_Cat_Path : String := Simulation_Dictionary.Lookup ("Powerplant_Cat_Path");
  begin
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "ngcffc.ito",
          Table     =>  NGCFFC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table ngcffc.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "taccng.ito",
          Table     =>  TACCNG_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table taccng.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "tdecng.ito",
          Table     =>  TDECNG_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table tdecng.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "ittcngc.ito",
          Table     =>  ITTCNGC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table tdecng.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "tittngc.ito",
          Table     =>  TITTNGC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table tittngc.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "shpcngc.ito",
          Table     =>  SHPCNGC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table shpcngc.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "dshpcnpn.ito",
          Table     =>  DSHPCNPNGC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table dshpcnpn.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "kshpcnp.ito",
          Table     =>  KSHPCNP_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table kshpcnp.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "fncshpc.ito",
          Table     =>  Fncshpc_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table fncshpc.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "kngffacc.ito",
          Table     =>  KNGFF_ACC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table kngffacc.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "ditthot.ito",
          Table     =>  DITTHOT_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table ditthot.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "kittngac.ito",
          Table     =>  KITTNG_ACC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table kittngac.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "kittngbl.ito",
          Table     =>  KITTNG_BLD_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table kittngbl.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "kshpngac.ito",
          Table     =>  KSHPNG_ACC_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table kshpngac.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "kshpngbl.ito",
          Table     =>  KSHPNG_BLD_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table kshpngl.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "kngffbld.ito",
          Table     =>  KNGFF_BLD_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table kngffbld.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "pengoil.ito",
          Table     =>  PENGOIL_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table pengoil.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "dpeoilnz.ito",
          Table     =>  DPEOILNZ_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table dpeoilnz.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "dngseizure.ito",
          Table     =>  DNGSEIZ_Table );
    exception
      when others =>
        Ada.Text_Io.Put_Line("Table dngseizure.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "dt1mach.ito",
          Table     => DT1MACH_Table );
    exception
      when others =>
        Ada.Text_IO.Put_Line("Table dt1mach.ito has an error.");
        raise;
    end;
    begin
      Interpolation_Table.Read
        ( File_Name => Powerplant_Cat_Path & "dqdwf.ito",
          Table     => DQDWF_Table );
    exception
      when others =>
        Ada.Text_IO.Put_Line("Table dqdwf.ito has an error.");
        raise;
    end;

  exception
    when others =>
      Ada.Text_Io.Put_Line("There is a table problem in Pratt_Whitney_Pt6a_68_Gas_Generator.Read_Tables procedure.");
      raise;
  end Read_Tables;

end Pratt_Whitney_Pt6a_68_Gas_Generator;
















