-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Interpolation_Table.Doubly_Indexed;
with Interpolation_Table.Singly_Indexed;
with Ada.Text_Io;
with Simulation_Dictionary;
with Stethoscope;

package body Metering_Valve is

  Cam_Profile_Table : aliased Interpolation_Table.Doubly_Indexed.Instance;
  Gain_Table        : aliased Interpolation_Table.Singly_Indexed.Instance;

  -- Access Output_Pressure
  function  Output_Pressure
    (An_Instance : in Instance)
    return Force_Types.Psi is
  begin
    return An_Instance.Output_Pressure;
  end Output_Pressure;

  procedure Set_Output_Pressure
    (An_Instance     : in out Instance;
     Output_Pressure : in     Force_Types.Psi) is
  begin
    An_Instance.Output_Pressure := Output_Pressure;
  end Set_Output_Pressure;

  -- Access Flow_Rate
  function Flow_Rate
    (An_Instance : in Instance)
    return Mass_Types.Pph is
  begin
    return An_Instance.Flow_Rate;
  end Flow_Rate;

  procedure Set_Flow_Rate
    (An_Instance : in out Instance;
     Flow_Rate   : in     Mass_Types.Pph) is
  begin
    An_Instance.Flow_Rate := Flow_Rate;
  end Set_Flow_Rate;

  -- Access Bypass_Flow_Rate
  function Bypass_Flow_Rate
    (An_Instance : in Instance)
    return Mass_Types.Pph is
  begin
    return An_Instance.Bypass_Flow_Rate;
  end Bypass_Flow_Rate;

  procedure Set_Bypass_Flow_Rate
    (An_Instance      : in out Instance;
     Bypass_Flow_Rate : in     Mass_Types.Pph) is
  begin
    An_Instance.Bypass_Flow_Rate := Bypass_Flow_Rate;
  end Set_Bypass_Flow_Rate;

  -- Access Min_Flow_Rate
  function  Min_Flow_Rate
    (An_Instance : in Instance)
    return Mass_Types.Pph is
  begin
    return An_Instance.Min_Flow_Rate;
  end Min_Flow_Rate;

  procedure Set_Min_Flow_Rate
    (An_Instance   : in out Instance;
     Min_Flow_Rate : in     Mass_Types.Pph) is
  begin
    An_Instance.Min_Flow_Rate := Min_Flow_Rate;
  end Set_Min_Flow_Rate;

  -- Access Altitude_Gain
  function  Altitude_Gain
    (An_Instance : in Instance)
    return Normalized_Types.Normalize is
  begin
    return An_Instance.Altitude_Gain;
  end Altitude_Gain;

  procedure Set_Altitude_Gain
    (An_Instance   : in out Instance;
     Altitude_Gain : in     Normalized_Types.Normalize) is
  begin
    An_Instance.Altitude_Gain := Altitude_Gain;
  end Set_Altitude_Gain;

  -- Method Update
  procedure Update
    ( Next_Instance                        : in out Instance;
      This_Instance                        : in     Instance;
      Power_Lever_Angle                    : in     Angle_Types.Degrees;
      Stepper_Motor_Position               : in     Step_Type;
      Shutdown_Solenoid_Energized          : in     Boolean;
      Ambient_Pressure_Ratio               : in     Normalized_Types.Normalize;
      Return_Line_Pressure                 : in     Force_Types.Psi;
      Inlet_Pressure                       : in     Force_Types.Psi;
      Inlet_Flow_Rate                      : in     Mass_Types.Pph;
      Restrain_Fuel_Flow                   : in     Boolean;
      Fmu_Malfunction                      : in     Boolean) is

    Rvs                  : Boolean                    renames Next_Instance.High_Pressure_Relief_Valve_State;
    Rvs_Set              : Force_Types.Psi            renames High_Pressure_Relief_Valve_Set_Point;
    Smp_Demanded         : Step_Type                  renames Stepper_Motor_Position;
    Delta_Pla            : Angle_Types.Degrees        renames Power_Lever_Angle;
    Delta_Ambient        : Normalized_Types.Normalize renames Ambient_Pressure_Ratio;
    Fuel_Flow_Dot        : Mass_Types.Pph             renames Next_Instance.Flow_Rate;
    K_Fmv_Delta          : Normalized_Types.Normalize renames Next_Instance.Altitude_Gain;
    Wf_C_3d_Dot          : Mass_Types.Pph             ;
    Standard_Atmospheric_Pressure : Force_Types.Psi  := Force_Types.Psi(14.696);
    K_Wfdot_Malfunction  : Normalized_Types.Normalize;
    Solenoid_Mechansim_Gain : Float :=  0.1;
  begin
--    Ada.Text_Io.Put("metering_valve0");
    --| high pressure relief valve
    Rvs := (Inlet_Pressure - Return_Line_Pressure) >  Rvs_Set;

    --| Calculate the bypass valve state.  The bypass valve maintains a constant
    --| 50 psi drop across the metering valve.  Modeling the state of this valve
    --| with a boolean is inaccurate, but is all that is needed currently.
    --| There is not much data in the training manual on this valve.  Excess fuel
    --| is returned to the high pressure pump inlet.
    Next_Instance.Bypass_Valve_State := (Inlet_Pressure - This_Instance.Output_Pressure ) > 50.0;


    --| Lookup the altitude compensation gain imposed by the action of
    --| the pressure sensing bellows upon the fuel metering valve:
    --|
    --| K_fmv_delta = f(damb) for table KFMVDELTA

    K_Fmv_Delta := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => Delta_Ambient,
        Table => Gain_Table'Access );

    --| Calculate the fuel flow, including the effect of the minimum
    --| flow orifice and the Manual and EEC shutdown valves.  Manual
    --| shutdown occurs at a power lever angle of 6 degrees or reversion
    --| to manual mode from normal PMU operating mode.
    --| Fuel_Flow_Dot has units of pph.

    if Shutdown_Solenoid_Energized  then
      Fuel_Flow_Dot := 0.0;  -- No Fuel or All fuel is bypassed to high pressure pump inlet.
      Next_Instance.Output_Pressure := Delta_Ambient*Standard_Atmospheric_Pressure;
    else
      --| Calculate the outlet pressure
      Next_Instance.Output_Pressure := Inlet_Pressure - 50.0; -- constant 50 psid across metering valve.
      if Next_Instance.Output_Pressure <= 0.0 then
        Fuel_Flow_Dot := 0.0;
      else
        --| Calculate the corrected fuel flow commanded by the power lever
        --| angle and stepper motor position via the 3D cam. The pressure
        --| drop across the valve is 50 Psi.  The cam controls the size of the
        --| valve opening.
        --|  Wf_C_3d_Dot := f(Delta_Pla, SMP) from Fuel Metering Unit table FF3DCAM
        --|                 in pound per hour, pph.

        Wf_C_3d_Dot := Interpolation_Table.Doubly_Indexed.Interpolate
          ( X => (Float(Delta_Pla)),
            Y => (Float(Smp_Demanded)),
            Table => Cam_Profile_Table'Access );

        --| Normalized Valve Position
        --| Actual Valve Position is not available.  Assume Flow rate data varies linearly
        --| with valve position.  Table generated from PMU software design document
        --| and software interface document.
        Next_Instance.Position := Normalized_Types.Normalize(Float(Wf_C_3d_Dot)/1130.5);


        --| A malfunction gain that factors fuel flow up or down from
        --| the commanded level is incorporated into the fuel management
        --| unit model, in Step 5 of Section 11.3.2. A value greater
        --| than unity, with the PMU ON, results in reduction in the
        --| stepper motor position to maintain commanded Ng or SHP, until
        --| the stepper motor reaches its lower limits. Thereafter, the
        --| PMU  may revert to manual mode if the abnormal fuel flow
        --| results in PMU limit exceedances.
        if Fmu_Malfunction then
           K_Wfdot_Malfunction := 0.850;
        else
           K_Wfdot_Malfunction := 1.0;
        end if;

        Fuel_Flow_Dot := K_Wfdot_Malfunction*(K_Fmv_Delta * Wf_C_3d_Dot + This_Instance.Min_Flow_Rate);
        if Restrain_Fuel_Flow then
          Fuel_Flow_Dot := Float'Min(83.0, Fuel_Flow_Dot);
        end if;
      end if;

    end if;
    --    Ada.Text_Io.Put_Line( Normalized_Types.Normalize'Image(K_Wfdot_Malfunction)); --Wf_C_3d_Dot, This_Instance.Min_Flow_Rate);
    --    Ada.Text_Io.Put_Line( Mass_Types.Pph'Image(Wf_C_3d_Dot)); --Wf_C_3d_Dot, This_Instance.Min_Flow_Rate);
    --| Calculate Bypass Flow Rate back to the center fuel tank
    if Next_Instance.High_Pressure_Relief_Valve_State or Next_Instance.Bypass_Valve_State then
      Next_Instance.Bypass_Flow_Rate := Inlet_Flow_Rate  - Fuel_Flow_Dot;
    else
      Next_Instance.Bypass_Flow_Rate := 0.0;
    end if;
    --    Ada.Text_Io.Put("metering_valve");

  end Update;

  -- Method Initialize
  procedure Initialize
    (An_Instance : in out Instance) is
  begin
    An_Instance.Position                             := Normalized_Types.Normalize(0.0);
    An_Instance.High_Pressure_Relief_Valve_State     := True;
    An_Instance.Bypass_Valve_State                   := True;
    An_Instance.Output_Pressure                      := Force_Types.Psi(0.0);
    An_Instance.Flow_Rate                            := Mass_Types.Pph(0.0);
    An_Instance.Bypass_Flow_Rate                     := Mass_Types.Pph(0.0);
    An_Instance.Min_Flow_Rate                        := Mass_Types.Pph(65.0);
    An_Instance.Altitude_Gain                        := Normalized_Types.Normalize(1.0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/Fmu/Metering_Valve/wfdot",
       Units          => "float",
       Object_Address => An_Instance.Flow_Rate'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/Fmu/Metering_Valve/Position",
       Units          => "float",
       Object_Address => An_Instance.Position'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/Fmu/Metering_Valve/Altitude_Gain",
       Units          => "float",
       Object_Address => An_Instance.Altitude_Gain'Address,
       Value_Type     => "float",
       Scope_Index    => 0);

  end Initialize;


   procedure Read_Tables is
     Powerplant_Cat_Path : String := Simulation_Dictionary.Lookup ("Powerplant_Cat_Path");
   begin

     Interpolation_Table.Read
       ( File_Name =>  Powerplant_Cat_Path & Cam_Profile_Table_Name,
         Table     =>  Cam_Profile_Table );

     Interpolation_Table.Read
       ( File_Name =>  Powerplant_Cat_Path & Gain_Table_Name,
         Table     =>  Gain_Table );
   end Read_Tables;

end Metering_Valve;
