-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Ada.Streams.Stream_Io;
with JPATS_Powerplant.Container;
with Jpats_Propeller_Types;
with Length_Types;
with Mass_Types;
with Torque_Types;
with Coordinate_Types;

private package JPATS_Powerplant.Controller is

--------------------------------------------------------------------
-- Reference: Kimball, D., "Flight Simulation Mathematical Model of
--            the Beech MkII Joint Primary Aircraft Training System
--            (JPATS)", Document 133E794 Rev. A, 18 February, 1998,
--            FSI-SSD # 50181, pg 66.
--
--------------------------------------------------------------------
-- Reference: Flight Manual USAF/USN Series T-6A Aircraft,
--            Air Force TO 1T-6A-1
--            Navy (NAVAIR) A1-T6AAA-NFM-100
--            CVV Draft December 1998
-------------------------------------------------------------------

  -- Accessor to Ignition_Select
  function Ignition_Select return Boolean;

  -- Accessor Manual_Feathe
  function Manual_Feather return Boolean;

  -- Accessor Pcl_Angle
  function Pcl_Angle return Angle_Types.Degrees;

  -- Accessor Fire_Warning_Request
  function Fire_Warning_Request return Boolean;

  -- Accessor PIU_Failure_Request
  function PIU_Failure_Request return Boolean;

  -- Accessor Return_Line_Pressure
  function Return_Line_Pressure return Force_Types.Psi;

  -- Accessor Engine_Vibration
  function Engine_Vibration return Normalized_Types.Normalize;

  -- Accessor Engine_Rough
  function Engine_Rough return Boolean;

  -- Accessor Compressor_Stall_Active
  function Compressor_Stall_Active return Boolean;

   -- Accessor Engine_Failure_Active
  function Engine_Failure_Active return Boolean;

  -- Accessor Oil_Pressure_Transducer_Signal
  function Oil_Pressure_Transducer_Signal return Force_Types.Psi;

  -- Accessor Oil_Temperature_Transducer_Signal
  function Oil_Temperature_Transducer_Signal return Temperature_Types.Celsius;

  -- Accessor Speed_Brake_Max_Power_Switch
  function Speed_Brake_Max_Power_Switch return Boolean;

  -- Accessor Landing_Gear_Warn_Low_Throttle_Switch
  function Landing_Gear_Warn_Low_Throttle_Switch return Boolean;

  -- Accessor Boost_Pump_Below_Idle_Switch
  function Boost_Pump_Below_Idle_Switch return Boolean;

  -- Method Update
  procedure Update
    (Integration_Constant : in Float);

  -- Method Initialize
  procedure Initialize;

  -- Method Start_Recording_Debug_Output
  procedure Start_Recording_Debug_Output
    (A_Subsystem  : in JPATS_Powerplant.Container.Instance_Ptr;
     Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access);

  -- Method Stop_Recording_Debug_Output
  procedure Stop_Recording_Debug_Output
    (A_Subsystem : in JPATS_Powerplant.Container.Instance_Ptr);

private
  type Interface is record
    --| Interface to Incoming Variables from Outside of Jpats_Powerplant.

    --| Interface from Hardware
    Manual_Select                         :  Boolean;
    Prop_Feather_Cut_Off_Switch           :  Boolean;   -- microswitch in throttle quadrant
    Speed_Brake_Max_Power_Switch          :  Boolean;   -- microswitch in throttle quadrant
    Landing_Gear_Warn_Low_Throttle_Switch :  Boolean;   -- microswitch in throttle quadrant
    Boost_Pump_Below_Idle_Switch          :  Boolean;   -- microswitch in throttle quadrant
    Forward_Starter_Switch_Manual         :  Boolean;
    Forward_Starter_Switch_In_Auto        :  Boolean;
    Forward_Ignition_Switch_On            :  Boolean;

    --|   Interface from Jpats_Electrical_Cat
    Edm_Batt_Cb                      :  Boolean;
    Edm_Gen_Cb                       :  Boolean;
    Ignition_Cb                      :  Boolean;
    Start_Cb                         :  Boolean;
    Oil_Trx_Cb                       :  Boolean;
    Pmu_Cb                           :  Boolean;
    Prop_Sys_Cb                      :  Boolean;
    Accessory_Torque_Absorbed        :  Jpats_Propeller_Types.Torque_Vector_Type.Instance;

    --| Interfaces from Jpats_Atmosphere_Cat
    Ambient_Pressure_Ratio           :  Normalized_Types.Normalize;
    Ambient_Temperature_Ratio        :  Normalized_Types.Normalize;
    Ambient_Temperature              :  Temperature_Types.Celsius;
    Ambient_Pressure                 :  Force_Types.Pounds_Per_Sq_Feet;
    Pressure_Altitude                :  Length_Types.Feet := 0.0;

    --| Interfaces from Jpats_Landing_Gear_Cat
    Left_Weight_On_Wheels            :  Boolean;
    Right_Weight_On_Wheels           :  Boolean;

    --| Interfaces from Jpats_Simulated_Aircraft_Cat
    Mach_Number                      :  Length_Types.Mach;
    Aircraft_Cg                      :  Jpats_Propeller_Types.Coordinate_Type.Instance;
    Aircraft_Angular_Rate            :  Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance;
    Load_Factor                      :  Coordinate_Types.Cartesian;
    Ground_Speed                     :  Length_Types.Feet_Per_Sec;

    --| Interfaces from Jpats_Propeller_Cat
    Propeller_Torque_Absorbed_Auto_Driven  :  Boolean;
    Propeller_Torque_Absorbed              :  Jpats_Propeller_Types.Torque_Vector_Type.Instance;

    --| Interfaces from Jpats_Hydraulics_system_cat
    Pump_Shaft_Horsepower_Absorbed   :  Torque_Types.Horsepower;

    --| Interface from Jpats_Fuel_Cat
    Fuel_Available                   :  Boolean;
    Fuel_Flow_Rate                   :  Mass_Types.Pph;
    Fuel_Flow_Burn_Rate              :  Mass_Types.Pph;

    --| Interface from Jpats_Environmental_Cat
    Bleed_Air_On                     :  Boolean;
    Bleed_Flow_Rate                  :  Mass_Types.Ppm;
    Air_Conditioner_Torque_Absorbed  :  Jpats_Propeller_Types.Torque_Vector_Type.Instance;

    -- Interface from Jpats_Flight_Instruments
    ADC_Pressure_Altitude            :  Length_Types.Feet;
    ADC_Mach_Number                  :  Length_Types.Mach;

    -- Interface to accessory horsepower
--    Accessory_Horsepower_Auto_Driven : Boolean;
    Accessory_Horsepower             : Torque_Types.Horsepower;

    -- Interface to torque driver
--    Torque_Auto_Driven               : Boolean;
    Torque                           : Torque_Types.Ft_Lbf;

    -- Interface to Warning System
    Ignition_Select                  : Boolean;

    -- Interface to Propeller
    Manual_Feather_Sig                 : Boolean;
    Oil_Temperature_Transducer_Failure : Boolean;
    Request_Fire_Warning               : Boolean;
    Piu_Failure                        : Boolean;
    Engine_Seizure                     : Boolean;
    Rough_Engine                       : Boolean;
    Compressor_Stall                   : Boolean;

    -- Interface to Engine Data Manager
    Power_Control_Lever_Angle_RVDT      : Angle_Types.Degrees := Angle_Types.Degrees(0.0);
    Power_Control_Lever_Angle_Lvdt      : Angle_Types.Degrees;
    Power_Control_Lever_Angle           : Angle_Types.Degrees;

    -- Interface to dcls
    Engine_Vibration_Magnitude          : Normalized_Types.Normalize;

    -- Interface to Fuel
    Pump_Return_Line_Pressure           : Force_Types.Psi := Force_Types.Psi(0.0);
    Fuel_Pressure                       : Force_Types.Psi;
    FMV_Return_Line_Pressure            : Force_Types.Psi := Force_Types.Psi(0.0);

    Linkage_Failure                     : Boolean := False;
    Check_Valve_Pressure_Drop           : Force_Types.Psi;
    Oil_Pressure_Transducer_Signal      : Force_Types.Psi;
    Oil_Temperature_Transducer_Signal   : Temperature_Types.Celsius;
  end record;

  This_Controller_Instance : Interface;

end JPATS_Powerplant.Controller;
