-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
with Ada.Numerics;
with Ada.Text_IO;
with Ada.Numerics.Elementary_Functions;
use  Ada.Numerics.Elementary_Functions;
with Ada.Streams.Stream_Io;
with Log;

with Stethoscope;
with JPATS_Atmosphere;
with JPATS_Landing_Gear;
with JPATS_Flight_Instruments;
with Jpats_Landing_Gear_Types;
with JPATS_Propeller;
with Jpats_Propeller_Types;
with JPATS_Simulated_Aircraft;
with JPATS_Electrical;
with Jpats_Electrical_Types;
with Jpats_Ios_Interface;
with JPATS_Hydraulics;
with JPATS_Environmental;
with JPATS_Environmental_Types;
with JPATS_Auto_test;
with JPATS_Fuel;
-- with JPATS_Avionics;
with Length_Types;
with Normalized_Types;
with Force_Types;
with Temperature_Types;
with Torque_Types;
with Mass_Types;
with Coordinate_Types;
with Vector_Types;
use  Vector_Types;

with JPATS_Powerplant.Container;
with JPATS_Powerplant.Io_Interface;
with JPATS_Powerplant.Ios_Interface;
with JPATS_Powerplant.Test_Controller;
with Jpats_Powerplant_Types;
with Power_Management_Unit;
with Gear_Pump;
with Fuel_Metering_Unit;
with Pratt_Whitney_Pt6a_68_Gas_Generator;
with Geometry;
with Electrical_Units_Types;
with Save_Restore;


package body JPATS_Powerplant.Controller is


  -- accessor to Ign_sel
  function Ignition_Select return Boolean is
  begin
    return This_Controller_Instance.Ignition_Select;
  end Ignition_Select;

  -- accessor to Manual_feather
  function Manual_Feather return Boolean is
  begin
    return This_Controller_Instance.Manual_Feather_Sig;
  end Manual_Feather;

  -- accessor to pcl angle
  function Pcl_Angle return Angle_Types.Degrees is
  begin
    return This_Controller_Instance.Power_Control_Lever_Angle;
  end Pcl_Angle;

  -- accessir to Fire_Warning_Request
  function Fire_Warning_Request return Boolean is
  begin
    return This_Controller_Instance.Request_Fire_Warning;
  end Fire_Warning_Request;

  -- accessor to Piu_Failure_Request
  function Piu_Failure_Request return Boolean is
  begin
    return This_Controller_Instance.Piu_Failure;
  end Piu_Failure_Request;

  -- accessor to Return_Line_Pressure
  function Return_Line_Pressure return Force_Types.Psi is
  begin
    return This_Controller_Instance.Pump_Return_Line_Pressure;
  end Return_Line_Pressure;

  -- Accessor to Engine_Vibration_Magnitude
  function Engine_Vibration return Normalized_Types.Normalize is
  begin
    return This_Controller_Instance.Engine_Vibration_Magnitude;
  end Engine_Vibration;

  -- accessor to Rough_Engine
  function Engine_Rough return Boolean is
  begin
    return This_Controller_Instance.Rough_Engine;
  end Engine_Rough;

  -- accessor to Compressor_Stall_Active
  function Compressor_Stall_Active return Boolean is
  begin
    return This_Controller_Instance.Compressor_Stall;
  end Compressor_Stall_Active;

  -- accessor to Engine_Failure_Active
  function Engine_Failure_Active return Boolean is
  begin
    return JPATS_Powerplant.Ios_Interface.This_Ios_Interface.mfen08;
  end Engine_Failure_Active;

  -- Accessor Oil_Pressure_Transducer_Signal
  function Oil_Pressure_Transducer_Signal
    return Force_Types.Psi is
  begin
    return This_Controller_Instance.Oil_Pressure_Transducer_Signal;
  end Oil_Pressure_Transducer_Signal;

  -- Accessor Oil_Temperature_Transducer_Signal
  function Oil_Temperature_Transducer_Signal
    return Temperature_Types.Celsius is
  begin
    return This_Controller_Instance.Oil_Temperature_Transducer_Signal;
  end Oil_Temperature_Transducer_Signal;

  -- Accessor Speed_Brake_Max_Power_Switch
  function Speed_Brake_Max_Power_Switch
    return Boolean is
  begin
    return This_Controller_Instance.Speed_Brake_Max_Power_Switch;
  end Speed_Brake_Max_Power_Switch;

  -- Accessor Landing_Gear_Warn_Low_Throttle_Switch
  function Landing_Gear_Warn_Low_Throttle_Switch
    return Boolean is
  begin
    return This_Controller_Instance.Landing_Gear_Warn_Low_Throttle_Switch;
  end Landing_Gear_Warn_Low_Throttle_Switch;

  -- Accessor Boost_Pump_Below_Idle_Switch
  function Boost_Pump_Below_Idle_Switch
    return Boolean is
  begin
    return This_Controller_Instance.Boost_Pump_Below_Idle_Switch;
  end Boost_Pump_Below_Idle_Switch;

  -- procedure start recording debug stream
  procedure Start_Recording_Debug_Output
    (A_Subsystem  : in JPATS_Powerplant.Container.Instance_Ptr;
     Debug_Stream : in out Ada.Streams.Stream_Io.Stream_Access) is
  begin
    Pratt_Whitney_Pt6a_68_Gas_Generator.Start_Recording
      (An_Instance => A_Subsystem.The_Gas_Generator,
       Debug_Stream => Debug_Stream);
    Power_Management_Unit.Start_Recording
      (An_Instance => A_Subsystem.The_Power_Management_Unit,
       Debug_Stream => Debug_Stream);
  end Start_Recording_Debug_Output;

  -- Procedure Stop Recording Debug Output
  procedure Stop_Recording_Debug_Output
    (A_Subsystem  : in JPATS_Powerplant.Container.Instance_Ptr) is
  begin
    Pratt_Whitney_Pt6a_68_Gas_Generator.Stop_Recording
      (An_Instance => A_Subsystem.The_Gas_Generator);
    Power_Management_Unit.Stop_Recording
      (An_Instance => A_Subsystem.The_Power_Management_Unit);
  end Stop_Recording_Debug_Output;

  -- function convert to aircraft angular rate
  function Convert_To_Aircraft_Angular_Rate
    (Ct : in Coordinate_Types.Attitude)
    return Jpats_Propeller_Types.Angular_Rate_Vector_Type.Instance is

  begin

    return ( Roll  => Angle_Types.Radians_Per_Sec(Ct.Roll),
             Pitch => Angle_Types.Radians_Per_Sec(Ct.Pitch),
             Yaw   => Angle_Types.Radians_Per_Sec(Ct.Yaw));

  end Convert_To_Aircraft_Angular_Rate;

  Comp_Stall_First_Pass               : Boolean := True;
  Comp_Stall_Onset_N1_Pct             : Normalized_Types.Normalize := 0.0;
  Comp_Stall_Minimum_N1_Pct           : Normalized_Types.Normalize := 0.0;
  Compressor_Stall_Enabled            : Boolean := False;
  Feather_Delay_Timer                 : Float;
  Pstd_Feather                        : Boolean := False;
  Pstd_Chip                           : Boolean := False;
  Pstd_Timer                          : Float;
  Improper_Start_Timer                : Float;
  Improper_Start_Enabled              : Boolean := False;
  N_One                               : Normalized_Types.Normalize := 0.0;

  -- Method Update
  procedure Update
    (Next_Subsystem : in out JPATS_Powerplant.Container.Instance_Ptr;
     This_Subsystem : in     JPATS_Powerplant.Container.Instance_Ptr;
     Iconst         : in     Float) is


    --|   Internal storage for interfaces to domain classes.
    --    Total_Pressure_Ratio               : Normalized_Types.Normalize;
    --    Total_Temperature_Ratio            : Normalized_Types.Normalize;
    Return_Line_Flow_Rate               : Mass_Types.Pph;
    Oil_Pressure                        : Force_Types.Psi := Force_Types.Psi(0.0);
    Hot_Start                           : Boolean := False;
    No_Light_Request                    : Boolean := False;
    Power_Turbine_Rpm_Sensor            : Angle_Types.Rpm
      := Pratt_Whitney_Pt6a_68_Gas_Generator.Power_Turbine_Rpm_Sensor
           (An_Instance => This_Subsystem.The_Gas_Generator);
    Power_Turbine_Torque_Sensor         : Torque_Types.Ft_Lbf
      := Pratt_Whitney_Pt6a_68_Gas_Generator.Torque_Sensor
           (An_Instance => This_Subsystem.The_Gas_Generator);
    Gas_Generator_Rpm_Sensor            : Angle_Types.Rpm
      := Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm_Sensor
           (An_Instance => This_Subsystem.The_Gas_Generator);
    Internal_Turbine_Temperature_Sensor : Temperature_Types.Celsius
      := Pratt_Whitney_Pt6a_68_Gas_Generator.Internal_Turbine_Temperature_Sensor
           (An_Instance =>  This_Subsystem.The_Gas_Generator);
    Inlet_Temperature_Sensor            : Temperature_Types.Celsius
      := Pratt_Whitney_Pt6a_68_Gas_Generator.Inlet_Temperature_Sensor
           (An_Instance =>  This_Subsystem.The_Gas_Generator);
    Fcu_Shaft_Rpm                       : Angle_Types.Rpm;
    RVDT_Malf                           : Boolean := False;
--  Hot_Start_Manual_Abort              : Boolean := False;
    Restrain_Fuel_Flow                  : Boolean := False;
    Itt_Sensor_Malfunction              : Boolean := False;
    Flameout_Malfunction                : Boolean := False;
    Flameout_Timer                      : Float   := 0.0;
    Overtemp                            : Boolean := False;
    Pmu_Critical_Fault                  : Boolean := False;
    Loss_Of_Useful_Power                : Boolean := False;
    Auto_Start_Switch_Off               : Boolean := False;
    High_Oil_Temperature                : Boolean := False;
    Oil_Pressure_Failure                : Jpats_Powerplant_Types.Oil_Pressure_Failure_Type
      := Jpats_Powerplant_Types.None;
    Oil_Leak                            : Boolean := False;
    Uncommanded_Feather                 : Boolean := False;
--  Pmu_Arinc_Fail                      : Boolean := False;
--  Linkage_Failure                     : Boolean := False;
    Slow_Start_Request                  : Boolean := False;
    Fired                               : Boolean := False;
    Delta_Ng_Seizure                    : Angle_Types.Rpm := 0.0;
    Shp                                 : Torque_Types.Horsepower := 0.0;
    K_Fmv_Delta                         : Normalized_Types.Normalize := 0.0;
    Start_Logic                         : Boolean := False;
    Chip                                : Boolean := False;
    Fmu_Malfunction                     : Boolean := False;

    -- Output from powerplant to pmu
    Delta_Shp_C_Acc                     : Torque_Types.Horsepower;
    Delta_Shp_C_Bleed                   : Torque_Types.Horsepower;
    Delta_Shp_C_Np                      : Torque_Types.Horsepower;
    Delta_Itt_C_Acc                     : Temperature_Types.Kelvin;
    Delta_Itt_C_Bleed                   : Temperature_Types.Kelvin;
    Delta_Ng_C_Acc                      : Angle_Types.Rpm;
    Delta_Ng_C_Bleed                    : Angle_Types.Rpm;
    Stepper_Motor_Position_Requested    : Jpats_Powerplant_Types.Step_Type;

    -- Interface to Propeller Category
    Propeller_Force_Vector          :  Jpats_Propeller_Types.Force_Vector_Type.Instance;
    Reset_Again                     :  Boolean := False;
  begin
     if (JPATS_Simulated_Aircraft.System_Reset or
         (Itt_Sensor_Malfunction and not
          JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen23)) then  --  or JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen06;

          if (Itt_Sensor_Malfunction and not
              JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen23) then
             Reset_Again := True;
          end if;

          Pratt_Whitney_Pt6a_68_Gas_Generator.Cold_Soak_Reset
            (An_Instance => This_Subsystem.The_Gas_Generator);
          Pratt_Whitney_Pt6a_68_Gas_Generator.Rotor_Reset
            (An_Instance => This_Subsystem.The_Gas_Generator);
          Fuel_Metering_Unit.Fmu_Reset
            ( An_Instance => This_Subsystem.The_Fuel_Metering_Unit);
          Power_Management_Unit.Pmu_Reset
            (An_Instance => This_Subsystem.The_Power_Management_Unit);
          JPATS_Propeller.Reset;
     end if;
     Propeller_Force_Vector := JPATS_Propeller.Force_Tensor;

     --  Quick Start Detect
     if JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Quick_Start_Request then
       Next_Subsystem.Quick_Start := True;
     end if;
     if Next_Subsystem.Quick_Start and Next_Subsystem.Quick_Start_Timer < 5.0  then
       Next_Subsystem.Quick_Start_Timer := Float'Min(15.0,This_Subsystem.Quick_Start_Timer + Iconst);
     else
       Next_Subsystem.Quick_Start_Timer := 0.0;
       Next_Subsystem.Quick_Start := False;
       JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Quick_Start_Request := False;
     end if;

     if JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Pmu_Mode_Off_Switch_Auto_Driven then
       This_Controller_Instance.Manual_Select                    := JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Pmu_Mode_Off_Switch;
     else
       This_Controller_Instance.Manual_Select                    := JPATS_Powerplant.Io_Interface.This_Io_Interface.Pmu_Mode_Off_Switch;
     end if;
     if JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Prop_Feather_Cut_Off_Switch_Auto_Driven then
       This_Controller_Instance.Prop_Feather_Cut_Off_Switch      := JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Prop_Feather_Cut_Off_Switch;
     else
       This_Controller_Instance.Prop_Feather_Cut_Off_Switch      := JPATS_Powerplant.Io_Interface.This_Io_Interface.Prop_Feather_Cut_Off_Switch;
     end if;
     if JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Starter_Switch_Auto_Driven then
       This_Controller_Instance.Forward_Starter_Switch_Manual    := JPATS_Powerplant.Io_Interface.auto_test_driver.Starter_Man_Switch;
       This_Controller_Instance.Forward_Starter_Switch_In_Auto   := JPATS_Powerplant.Io_Interface.auto_test_driver.Starter_Auto_Switch;
     else
       This_Controller_Instance.Forward_Starter_Switch_Manual    := JPATS_Powerplant.Io_Interface.This_Io_Interface.Starter_Man_Switch;
       This_Controller_Instance.Forward_Starter_Switch_In_Auto   := JPATS_Powerplant.Io_Interface.This_Io_Interface.Starter_Auto_Switch;
     end if;
     if JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Ignition_Switch_Auto_Driven then
       This_Controller_Instance.Forward_Ignition_Switch_On       := JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Ignition_On_Switch;
     else
       This_Controller_Instance.Forward_Ignition_Switch_On       := JPATS_Powerplant.Io_Interface.This_Io_Interface.Ignition_On_Switch;
     end if;

    --|   Interface from Jpats_Electrical_Cat
    if not JPATS_Powerplant.Test_Controller.Electrical_Auto_Driven then
      this_controller_instance.Edm_Gen_Cb                        := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Edm_Cb_Fwd_Gen_Bus);
      this_controller_instance.Ignition_Cb                       := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Ign_Cb);
      this_controller_instance.Start_Cb                          := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Start_Cb);
      this_controller_instance.Oil_Trx_Cb                        := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Oil_Trx_Cb);
      this_controller_instance.Pmu_Cb                            := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Pmu_Cb);
      this_controller_instance.Prop_Sys_Cb                       := JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Prop_Sys_Cb);
    else
      this_controller_instance.Edm_Gen_Cb                        := Jpats_Powerplant.Test_Controller.Edm_Gen_Cb;
      this_controller_instance.Ignition_Cb                       := Jpats_Powerplant.Test_Controller.Ignition_Cb;
      this_controller_instance.Start_Cb                          := Jpats_Powerplant.Test_Controller.Start_Cb;
      this_controller_instance.Oil_Trx_Cb                        := Jpats_Powerplant.Test_Controller.Oil_Trx_Cb;
      this_controller_instance.Pmu_Cb                            := Jpats_Powerplant.Test_Controller.Pmu_Cb;
      this_controller_instance.Prop_Sys_Cb                       := Jpats_Powerplant.Test_Controller.Prop_Sys_Cb;

    end if;

      --| Interfaces from Jpats_Atmosphere_Cat
    if not JPATS_Powerplant.Test_Controller.Atmosphere_Auto_Driven then
      This_Controller_Instance.Ambient_Pressure_Ratio           := JPATS_Atmosphere.Pressure_Ratio;
      This_Controller_Instance.Ambient_Temperature_Ratio        := JPATS_Atmosphere.Temperature_Ratio;
      This_Controller_Instance.Ambient_Temperature              := JPATS_Atmosphere.Temperature;  -- Celsius
      This_Controller_Instance.Ambient_Pressure                 := JPATS_Atmosphere.Pressure;

      --| Calculate Pressure Altitude from Ambient Pressure
      This_Controller_Instance.Pressure_Altitude := ((1.0 - This_Controller_Instance.Ambient_Pressure_Ratio**0.19018)*145447.0);
    else
      This_Controller_Instance.Ambient_Pressure_Ratio           := JPATS_Powerplant.Test_Controller.Ambient_Pressure_Ratio;
      This_Controller_Instance.Ambient_Temperature_Ratio        := JPATS_Powerplant.Test_Controller.Ambient_Temperature_Ratio;
      This_Controller_Instance.Ambient_Temperature              := JPATS_Powerplant.Test_Controller.Ambient_Temperature;  -- Celsius
      This_Controller_Instance.Ambient_Pressure                 := JPATS_Powerplant.Test_Controller.Ambient_Pressure;

      --| Calculate Pressure Altitude from Ambient Pressure
      This_Controller_Instance.Pressure_Altitude                := JPATS_Powerplant.Test_Controller.Pressure_Altitude;
    end if;

    --| Interfaces from Jpats_Landing_Gear_Cat
    if not JPATS_Powerplant.Test_Controller.Landing_Gear_Auto_Driven then
      This_Controller_Instance.Left_Weight_On_Wheels            := JPATS_Landing_Gear.Is_Closed(Jpats_Landing_Gear_Types.
                                                                       Left_Landing_Gear_A2_WOW_Relay);
      This_Controller_Instance.Right_Weight_On_Wheels           := JPATS_Landing_Gear.Is_Closed(Jpats_Landing_Gear_Types.
                                                                       Right_Landing_Gear_A2_WOW_Relay);
    else
      This_Controller_Instance.Left_Weight_On_Wheels            := JPATS_Powerplant.Test_Controller.Left_Weight_On_Wheels;

      This_Controller_Instance.Right_Weight_On_Wheels           := JPATS_Powerplant.Test_Controller.Right_Weight_On_Wheels;
    end if;

    --| Interfaces from Jpats_Simulated_Aircraft_Cat
    if not JPATS_Powerplant.Test_Controller.Aircraft_Auto_Driven then
      This_Controller_Instance.Mach_Number                      := JPATS_Simulated_Aircraft.Get_Mach_Number;
      This_Controller_Instance.Aircraft_Cg                      := (JPATS_Simulated_Aircraft.Get_X_Cg,
                                                                    JPATS_Simulated_Aircraft.Get_Y_Cg,
                                                                    JPATS_Simulated_Aircraft.Get_Z_Cg);
      This_Controller_Instance.Aircraft_Angular_Rate            := Convert_To_Aircraft_Angular_Rate
                                                                    ( Ct => JPATS_Simulated_Aircraft.Get_Angular_Rate );
      This_Controller_Instance.Load_Factor                      := JPATS_Simulated_Aircraft.Load_Factor;
      This_Controller_Instance.Ground_Speed                     := JPATS_Simulated_Aircraft.Get_Ground_Speed;
    else
      This_Controller_Instance.Mach_Number                      := JPATS_Powerplant.Test_Controller.Mach_Number;
      This_Controller_Instance.Aircraft_Cg                      := JPATS_Powerplant.Test_Controller.Aircraft_Cg;
      This_Controller_Instance.Aircraft_Angular_Rate            := JPATS_Powerplant.Test_Controller.Aircraft_Angular_Rate;
      This_Controller_Instance.Load_Factor                      := JPATS_Powerplant.Test_Controller.Load_Factor;
      This_Controller_Instance.Ground_Speed                     := JPATS_Powerplant.Test_Controller.Ground_Speed;
    end if;

    --| Interfaces from Jpats_Propeller_Cat
    if not This_Controller_Instance.Propeller_Torque_Absorbed_auto_driven then
      This_Controller_Instance.Propeller_Torque_Absorbed       := JPATS_Propeller.Total_Torque_Absorbed;
    end if;

    This_Controller_Instance.Accessory_Torque_Absorbed     := (-1.0*JPATS_Electrical.Starter_Torque,0.0,0.0);
    This_Controller_Instance.Pump_Shaft_Horsepower_Absorbed := -1.0*JPATS_Hydraulics.Pump_Shaft_Horsepower;

    --| Interface from Jpats_Fuel_Cat
    if not JPATS_Powerplant.test_controller.Fuel_Available_Auto_Driven then
      This_Controller_Instance.Fuel_Available                  := JPATS_Fuel.Fuel_Available;
    else
      This_Controller_Instance.Fuel_Available                  := JPATS_Powerplant.Test_Controller.Fuel_Available;
    end if;

    --| Interface from Jpats_Environmental_Cat
    if not JPATS_Powerplant.Test_Controller.Environmental_Auto_Driven then
      This_Controller_Instance.Bleed_Air_On                    := JPATS_Environmental.Bleed_Air_On;
      This_Controller_Instance.Bleed_Flow_Rate                 := JPATS_Environmental.Bleed_Flow;
      This_Controller_Instance.Air_Conditioner_Torque_Absorbed := (JPATS_Environmental.Air_Conditioner_Compressor_Torque,0.0,0.0);
    else
      This_Controller_Instance.Bleed_Air_On                    := JPATS_Powerplant.Test_Controller.Bleed_Air_On;
      This_Controller_Instance.Bleed_Flow_Rate                 := JPATS_Powerplant.Test_Controller.Bleed_Flow_Rate;
      This_Controller_Instance.Air_Conditioner_Torque_Absorbed := JPATS_Powerplant.Test_Controller.Air_Conditioner_Torque_Absorbed;

    end if;

      -- Interface from Jpats_Flight_Instruments
    if not JPATS_Powerplant.Test_Controller.Adc_Auto_Driven then
      This_Controller_Instance.ADC_Pressure_Altitude           := JPATS_Flight_Instruments.Primary_Alt;  -- indicated altitude
      This_Controller_Instance.ADC_Mach_Number                 := JPATS_Flight_Instruments.Primary_Ind_Mach;
    else
      This_Controller_Instance.ADC_Pressure_Altitude           := JPATS_Powerplant.Test_Controller.ADC_Pressure_Altitude;  -- indicated altitude
      This_Controller_Instance.ADC_Mach_Number                 := JPATS_Powerplant.Test_Controller.ADC_Mach_Number;
    end if;

    if JPATS_Powerplant.IOS_Interface.This_Ios_Interface.Pcl_Calibrate_Request then
      JPATS_Powerplant.IOS_Interface.This_Ios_Interface.Pcl_Calibrate_Request := False;
      JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Command := 2;
    end if;

     if JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Pcl_Position_Auto_Driven then
       This_Controller_Instance.Power_Control_Lever_Angle        := JPATS_Powerplant.Io_Interface.Auto_Test_Driver.Pcl_Position;
       if This_Controller_Instance.Power_Control_Lever_Angle <= 18.0 then
         JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Position := 18.0;
       elsif This_Controller_Instance.Power_Control_Lever_Angle >= 53.0 then
         JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Position := 53.0;
       else
         JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Position := This_Controller_Instance.Power_Control_Lever_Angle;
       end if;
       if JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Command /= 3 then
         JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Command := 3;
       end if;
     else
       This_Controller_Instance.Power_Control_Lever_Angle        := JPATS_Powerplant.Io_Interface.This_IO_Interface.Pcl_Position;
       -- PCL_Driver_Commands
       --
       --   Hex Code      Command Description
       --  -------------------------------------------------------------------
       --  0x00000000     No command
       --  0x00000001     Send reset to smart motor
       --  0x00000002     Calibrate smart motor
       --  0x00000003     Enable playback mode
       --  0x00000004     Disable playback mode

       if Save_Restore.Replay_In_Progress then
         if This_Controller_Instance.Power_Control_Lever_Angle <= 18.0 then
           JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Position := 18.0;
         elsif This_Controller_Instance.Power_Control_Lever_Angle >= 53.0 then
           JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Position := 53.0;
         else
           JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Position := This_Controller_Instance.Power_Control_Lever_Angle;
         end if;
         if JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Command /= 3 then
           JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Command := 3;
         end if;
       else
         JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Position := This_Controller_Instance.Power_Control_Lever_Angle;
         if JPATS_Powerplant.Io_Interface.This_Io_Interface.PCL_Driver_Command = 4 or
            JPATS_Powerplant.Io_Interface.This_Io_Interface.PCL_Driver_Command = 0 then
           JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Command := 0;
         else
           JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Driver_Command := 4;
         end if;
       end if;
     end if;

    if Next_Subsystem.Quick_Start then
      This_Controller_Instance.Edm_Batt_Cb                      := True;
      This_Controller_Instance.Edm_Gen_Cb                       := True;
      This_Controller_Instance.Ignition_Cb                      := True;
      This_Controller_Instance.Start_Cb                         := True;
      This_Controller_Instance.Oil_Trx_Cb                       := True;
      This_Controller_Instance.Pmu_Cb                           := True;
      This_Controller_Instance.Left_Weight_On_Wheels            := True;
      This_Controller_Instance.Right_Weight_On_Wheels           := True;
      if Next_Subsystem.Quick_Start_Timer > 0.5 and Next_Subsystem.Quick_Start_Timer < 1.0 then
        JPATS_Propeller.Reset;
        This_Controller_Instance.Prop_Feather_Cut_Off_Switch := False;
        This_Controller_Instance.Forward_Ignition_Switch_On := True;
        This_Controller_Instance.Forward_Starter_Switch_Manual := True;
        This_Controller_Instance.Forward_Starter_Switch_In_Auto := False;
      end if;
      if Next_Subsystem.Quick_Start_Timer < 2.0 then
        This_Controller_Instance.Manual_Select :=  True;
      end if;
      if Next_Subsystem.Quick_Start_Timer < 2.5 then
        JPATS_Propeller.Reset;
        Power_Management_Unit.Pmu_Reset
          (An_Instance => This_Subsystem.The_Power_Management_Unit);
        Pratt_Whitney_Pt6a_68_Gas_Generator.Quick_Start
          (An_Instance => This_Subsystem.The_Gas_Generator);
      end if;
      if Next_Subsystem.Quick_Start_Timer < 3.0 then
        Fuel_Metering_Unit.Fmu_Reset
          ( An_Instance => This_Subsystem.The_Fuel_Metering_Unit);
        Pratt_Whitney_Pt6a_68_Gas_Generator.Rotor_Reset
          (An_Instance => This_Subsystem.The_Gas_Generator);
      end if;
      JPATS_Propeller.Reset;
    end if;

    -- Reflect the hardware status back to the ios
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Pcl_Position := JPATS_Powerplant.Io_Interface.This_Io_Interface.Pcl_Position;
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Ignition_On_Switch := JPATS_Powerplant.Io_Interface.This_Io_Interface.Ignition_On_Switch;
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Ignition_Off_Switch := JPATS_Powerplant.Io_Interface.This_Io_Interface.Ignition_Off_Switch;
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Starter_Man_Switch := JPATS_Powerplant.Io_Interface.This_Io_Interface.Starter_Man_Switch;
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Starter_Auto_Switch := JPATS_Powerplant.Io_Interface.This_Io_Interface.Starter_Auto_Switch;
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Prop_Feather_Cut_Off_Switch := JPATS_Powerplant.Io_Interface.This_Io_Interface.Prop_Feather_Cut_Off_Switch;
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Pmu_Mode_Off_Switch := JPATS_Powerplant.Io_Interface.This_Io_Interface.Pmu_Mode_Off_Switch;
    JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Itt_Start_Mode := Pratt_Whitney_Pt6a_68_Gas_Generator.Itt_Start_Mode (An_Instance => This_Subsystem.The_Gas_Generator);

    if JPATS_Powerplant.Io_Interface.This_Io_Interface.Starter_Auto_Switch then
      JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Starter_Switch_Position := 1;
    elsif JPATS_Powerplant.Io_Interface.This_Io_Interface.Starter_Man_Switch then
      JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Starter_Switch_Position := 2;
    else
      JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Starter_Switch_Position := 0;
    end if;

    -- Hot Start Malfunctions
    Hot_Start := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen01
      or JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen02
      or JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen03;

    -- Hot Start with manual abort Malfunctions
    -- Hot_Start_Manual_Abort :=JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen02;

    -- Improper Start Enabled Malfunction
    if JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen25 then

       N_One := Normalized_Types.Normalize(Gas_Generator_Rpm_Sensor/374.680);

       if This_Controller_Instance.Left_Weight_On_Wheels or
          This_Controller_Instance.Right_Weight_On_Wheels then

          if N_One > 59.0 and Pcl_Angle < 7.250 and Improper_Start_Timer < 3.1 then
             Improper_Start_Enabled := True;  -- Changes PMU behavior
             Improper_Start_Timer := Improper_Start_Timer + Iconst;
          elsif N_One < 15.0 or Pcl_Angle < 4.50 then
             Improper_Start_Enabled := False;
             Improper_Start_Timer := 0.0;
          end if;
       else  -- Airborne
          if N_One > 13.0 and N_One < 55.0 and Pcl_Angle > 35.0 then
             Improper_Start_Enabled := True;
             Improper_Start_Timer := 0.2;
          elsif N_One < 13.0 or Pcl_Angle < 4.50 then
             Improper_Start_Enabled := False;
             Improper_Start_Timer := 0.0;
          end if;
       end if;
    else
       Improper_Start_Enabled := False;
       Improper_Start_Timer := 0.0;
    end if;

    -- No Start or Flameout or Improper Start Time Exceeded
    No_Light_Request := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen04 or
                        JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen26 or
                        Improper_Start_Timer > 3.0;

    -- Engine Seizure Malfunction or Hung Start
    Slow_Start_Request := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen03;
    -- or JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen08;


    -- This_Controller_Instance.Request_Fire_Warning := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen05
    --    or (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen06 and Internal_Turbine_Temperature_Sensor >= 920.0)
    --    or (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen07 and Power_Turbine_Torque_Sensor > 25.0*28.89 );

    This_Controller_Instance.Request_Fire_Warning :=
       ((JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen05 or JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen31)
       and Pratt_Whitney_Pt6a_68_Gas_Generator.Fired (An_Instance => This_Subsystem.The_Gas_Generator))
       or (This_Controller_Instance.Request_Fire_Warning and (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen05
       or JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen31))
       or (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen06 and Internal_Turbine_Temperature_Sensor >= 920.0)
       or (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen07 and Power_Turbine_Torque_Sensor > 25.0*28.89 );

    -- High ITT
    Overtemp := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen06 or
                (Improper_Start_Timer > 0.1 and Pcl_Angle > 16.0);

    -- Engine Seizure Malfunction
    This_Controller_Instance.Engine_Seizure := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen09
      or (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen18
          and Pratt_Whitney_Pt6a_68_Gas_Generator.Engine_Oil_Pressure
                (An_Instance => This_Subsystem.The_Gas_Generator) < Force_Types.Psi(0.6));
    This_Controller_Instance.Rough_Engine := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen09;

    -- Flameout Malfunction
    Flameout_Malfunction := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen08;

    -- Pmu Critical Failure
    Pmu_Critical_Fault := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen11;

    -- Loss of Useful Power
    Loss_Of_Useful_Power := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen12;

    -- Auto_Start Switch Malfunction
    Auto_Start_Switch_Off := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen02;

     -- Chip detect malfunction
    Chip := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen13 or Pstd_Chip;

    -- PIU Failure (Also Fire Warning with Erratic Engine/Fuel malfunction and PCL below engine cutoff angle)
    This_Controller_Instance.PIU_Failure := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen28 or
         (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen31 and
         This_Controller_Instance.Power_Control_Lever_Angle > Angle_Types.Degrees(6.0));

    -- Low Oil Pressure malfunction40
    if JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen14 then
      Oil_Pressure_Failure := Jpats_Powerplant_Types.DTP414;
      -- Low Oil Pressure at Idle malfunction
    elsif JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen15 then
      Oil_Pressure_Failure := Jpats_Powerplant_Types.DTP415;
      -- High Oil Pressure malfunction
    elsif JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen16 then
      Oil_Pressure_Failure := Jpats_Powerplant_Types.DTP416;
    else -- No oil pressure malfunction
      Oil_Pressure_Failure := Jpats_Powerplant_Types.None;
    end if;

    --  High Oil Temperature
    High_Oil_Temperature := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen17;

    -- Loss of Oil Malfunction
    Oil_Leak := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen18;

    -- Uncommanded Feather Malfunction
    Uncommanded_Feather := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen19 or Pstd_Feather;

      -- Oil Pressure Transducer Failure
    This_Controller_Instance.Oil_Temperature_Transducer_Failure := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen21;

    -- Itt Sensor Malfunction
    Itt_Sensor_Malfunction := JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen23; --  or JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen06;

    -- Linkage Failure Malfunction
    This_Controller_Instance.Linkage_Failure :=  (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen29 or
                                                  JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen27);

    -- Fuel Management Unit Malfunction
    Fmu_Malfunction := (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen30) and
                        This_Controller_Instance.Manual_Select;  -- PMU OFF

    -- Compressor Stall Malfunction
    if JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen22 then
       if Comp_Stall_First_Pass then
          Comp_Stall_Onset_N1_Pct := Normalized_Types.Normalize'Max(90.0,
                                     Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Speed_Pct
                                     (An_Instance => This_Subsystem.The_Gas_Generator));
          Comp_Stall_Minimum_N1_Pct := Comp_Stall_Onset_N1_Pct - Normalized_Types.Normalize(5.0);
          Comp_Stall_First_Pass := False;
       end if;
       if Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Speed_Pct
          (An_Instance => This_Subsystem.The_Gas_Generator) > Comp_Stall_Onset_N1_Pct and
          Jpats_Environmental.Is_Open(Jpats_Environmental_Types.Defog_Select_Relay) then
          Compressor_Stall_Enabled := True;
       elsif Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Speed_Pct
          (An_Instance => This_Subsystem.The_Gas_Generator) < Comp_Stall_Minimum_N1_Pct or
          Jpats_Environmental.Is_Closed(Jpats_Environmental_Types.Defog_Select_Relay) then
          Compressor_Stall_Enabled := False;
       end if;
    else
       Compressor_Stall_Enabled := False;
       Comp_Stall_First_Pass := True;
    end if;

    This_Controller_Instance.Compressor_Stall := Compressor_Stall_Enabled;

    -- Propeller Sleeve Touchdown Malfunction 
    if JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen24 and not
       JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen13 and not -- Chip
       JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen19 then    -- Uncommanded Feather

       if JPATS_Powerplant.Io_Interface.This_Io_Interface.Pmu_Mode_Off_Switch then
          Pstd_Feather := False;
       else
    
          if Pstd_Timer < 1.0 then
             Pstd_Feather := True;
          elsif Pstd_Timer < 10.0 then
             Pstd_Feather := False;
          elsif Pstd_Timer < 11.0 then
             Pstd_Feather := True;
          elsif Pstd_Timer < 16.0 then
             Pstd_Feather := False;
          else
             Pstd_Feather := True;
          end if;

       end if;

       if Pstd_Timer > 10.0 then
          Pstd_Chip := True;
       end if;

       if Pstd_Timer < 17.0 then
          Pstd_Timer := Pstd_Timer + Iconst;
       end if;

    else
       Pstd_Feather := False;
       Pstd_Chip    := False;
       Pstd_Timer   := 0.0;
    end if;

    This_Controller_Instance.Power_Control_Lever_Angle_Lvdt
      := Fuel_Metering_Unit.Power_Control_Lever_Angle_LVDT
      (An_Instance => This_Subsystem.The_Fuel_Metering_Unit);

    -- If Uncontrollable High Power malfunction is active, only allow power increases unless
    -- Uncontrollable High Power ( PCL Shutdown ) malfunction is active and PCL angle is less than 6 degrees (Engine Cutoff)
    if This_Controller_Instance.Linkage_Failure and 
       (This_Controller_Instance.Power_Control_Lever_Angle < This_Controller_Instance.Power_Control_Lever_Angle_Lvdt) and not
       (JPATS_Powerplant.Ios_Interface.This_Ios_Interface.Mfen27 and This_Controller_Instance.Power_Control_Lever_Angle <= Angle_Types.Degrees(6.0)) then
       This_Controller_Instance.Power_Control_Lever_Angle := This_Controller_Instance.Power_Control_Lever_Angle_Lvdt;
    end if;

    if not RVDT_Malf then
      This_Controller_Instance.Power_Control_Lever_Angle_RVDT := This_Controller_Instance.Power_Control_Lever_Angle;
    end if;

     -- Throttle quadrant microswitch simulation
     This_Controller_Instance.Prop_Feather_Cut_Off_Switch := This_Controller_Instance.Power_Control_Lever_Angle > Angle_Types.Degrees(6.875);

     This_Controller_Instance.Speed_Brake_Max_Power_Switch          := This_Controller_Instance.Power_Control_Lever_Angle > Angle_Types.Degrees(50.9);
     This_Controller_Instance.Landing_Gear_Warn_Low_Throttle_Switch := This_Controller_Instance.Power_Control_Lever_Angle < Angle_Types.Degrees(28.5);
     This_Controller_Instance.Boost_Pump_Below_Idle_Switch          := This_Controller_Instance.Power_Control_Lever_Angle < Angle_Types.Degrees(15.9);

    --| Calculate the total temperature and pressure ratios, for
    --| use in correcting engine parameters. These carry the subscript "2"
    --| to denote the compressor face station:
    --|
    --| The prefix "total" means ambient ratio + dynamic ratio obtained at the
    --| face of the engine.
    --| Customarily, this is temperature or pressure of the flow if the flow were
    --| slowed adiabatically and reversibly until stopped.

--    Total_Temperature_Ratio := Ambient_Temperature_Ratio*(1.0 + 0.2*Mach_Number**2.0);
--    Total_Pressure_Ratio    := Ambient_Pressure_Ratio*(1.0 + 0.2*Mach_Number**2.0)**3.5;


      Delta_Shp_C_Acc   := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Shp_C_Acc
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Delta_Shp_C_Bleed := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Shp_C_Bleed
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Delta_Itt_C_Acc   := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Itt_C_Acc
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Delta_Itt_C_Bleed := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Itt_C_Bleed
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Delta_Shp_C_Np    := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Shp_C_Np
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Delta_Ng_C_Acc    := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Ng_C_Acc
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Delta_Ng_C_Bleed  := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Ng_C_Bleed
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Delta_Ng_Seizure  := Pratt_Whitney_Pt6a_68_Gas_Generator.Delta_Ng_Seizure
        (An_Instance => This_Subsystem.The_Gas_Generator);
      Shp               := Pratt_Whitney_Pt6a_68_Gas_Generator.Shp
        (An_Instance => This_Subsystem.The_Gas_Generator);
      K_Fmv_Delta       := Fuel_Metering_Unit.Altitude_Gain
        (An_Instance => This_Subsystem.The_Fuel_Metering_Unit);
      Fired             := Pratt_Whitney_Pt6a_68_Gas_Generator.Fired
        (An_Instance => This_Subsystem.The_Gas_Generator);

      --  Call the Power management unit to update the fuel metering valve position requested.
      --  This signal is based on the last past state of the rest of the system.  By using the
      --  state vector of input.  Note: may need to move this call below the call for the gas
      --  generator so as to have two different time values to compute derivatives.  At the
      --  beginning of the frame the now n-2 time step has been thrown away.
      --      Ada.Text_Io.Put("pmu");
  Power_Management_Unit.Update
    (Next_Instance                             => Next_Subsystem.The_Power_Management_Unit,
     This_Instance                             => This_Subsystem.The_Power_Management_Unit,
     Iconst                                    => Iconst,
     Bleed_Air_On                              => This_Controller_Instance.Bleed_Flow_Rate > 5.2,                      -- Bleed Air High discrete
--   Bleed_Air_On                              => Bleed_Air_On,                                                         -- Pin Y   -- Cabin_Bleed            Interface_J2_Pin_Y;
     Left_Weight_On_Wheels                     => This_Controller_Instance.Left_Weight_On_Wheels,                       -- Pin_FF  -- Wow_Left               Interface_J2_Pin_F*;
     PMA_A                                     => Power_Turbine_Rpm_Sensor > 600.0,                                     -- Pin_A   -- Pma_A                  Interface_J1_Pin_A;
     PMA_B                                     => Power_Turbine_Rpm_Sensor > 600.0,                                     -- Pin_S   -- Pma_B                  Interface_J1_Pin_S;
     PMA_C                                     => Power_Turbine_Rpm_Sensor > 600.0,                                     -- Pin_T   -- Pma_C                  Interface_J1_Pin_T;
     Pmu_Switch                                => not This_Controller_Instance.Manual_Select,                           -- Pin B*  -- Manual_Sel             Interface_J2_Pin_LB;
     Power_28vdc2                              => This_Controller_Instance.Pmu_Cb,                                      -- Pin P   -- Power_28VDC            Interface_J2_Pin_P;
     Right_Weight_On_Wheels                    => This_Controller_Instance.Right_Weight_On_Wheels,                      -- Pin BB  -- Wow_Right              Interface_J2_Pin_B*;
     Start_Enable                              => This_Controller_Instance.Forward_Starter_Switch_In_Auto,              -- Pin U*  -- Start                  Interface_J2_Pin_LU;
     Engine_Rpm_Sensor_Fault                   => False,                                                                -- Malfunction
     Arinc_Airspeed_Fault                      => False,                                                                -- Malfunction
     Arinc_Hp_Fault                            => False,                                                                -- Malfunction
     Fuel_Flow_Sensor_Fault                    => False,                                                                -- Malfunction
     Prop_Speed_Sensor_Fault                   => False,                                                                -- Malfunction
     Pcl_Sensor_Fault                          => Improper_Start_Enabled,                                                                -- Malfunction
     --   Pma_Fault                            => False,                                                                -- Malfunction
     Torque_Sensor_Fault                       => False,                                                                -- Malfunction
     OAT_Sensor_Fault                          => False,                                                                -- Malfunction
     Wow_Fault                                 => False,                                                                -- Malfunction
     Hot_Start                                 => Hot_Start,                                                            -- Malfunction
     Voltage                                   => JPATS_Electrical.Voltage(Jpats_Electrical_Types.Starter_Distribution_Bus),                                     -- Malfunction
     PMU_Critical_Fault                        => Pmu_Critical_Fault,                                                   -- Malfunction
     PMU_Noncritical_Fault                     => False,                                                                -- Malfunction
     Slow_Start_Request                        => Slow_Start_Request,                                                   -- Malfunction
     Loss_Of_Useful_Power                      => Loss_Of_Useful_Power,                                                 -- Malfunction
     Auto_Start_Switch_Off                     => Auto_Start_Switch_Off,                                                -- Malfunction
     Itt_Sensor_Malfunction                    => Itt_Sensor_Malfunction,                                               -- Malfunction
     Overtemp                                  => Overtemp,                                                             -- Malfunction
     Ambient_Pressure_Ratio                    => This_Controller_Instance.Ambient_Pressure_Ratio,                      -- Ambient pressure ratio
     Engine_Rpm                                => Gas_Generator_Rpm_Sensor,                                             -- Pin_LB  -- Ng_Sig                 Interface_J1_Pin_LB;
     Internal_Turbine_Temperature_T5_Pos       => Internal_Turbine_Temperature_Sensor,                                  -- Pin_LZ  -- T5_Pos                 Interface_J1_Pin_LZ;
     Mach_Number                               => This_Controller_Instance.Mach_Number,                                 -- direct from gas generator
     ADC_Mach_Number                           => This_Controller_Instance.ADC_Mach_Number,                             -- Air data computer input over Arinc 429 on aircraft
     Power_Control_Lever_Angle_LVDT            => This_Controller_Instance.Power_Control_Lever_Angle_LVDT,              -- Pin_LN  -- BA_FB_E2               Interface_J1_Pin_LN;
     Power_Control_Lever_Angle_RVDT            => This_Controller_Instance.Power_Control_Lever_Angle_RVDT,              -- Pin CC  -- Excitation_Hi          Interface_J2_Pin_C*;
     Power_Turbine_Rpm                         => Power_Turbine_Rpm_Sensor,                                             -- Pin_LV  -- Q_Npt_Stg              Interface_J1_Pin_LV;
     Power_Turbine_Torque                      => Power_Turbine_Torque_Sensor,                                          -- Pin_LG  -- To_Hi                  Interface_J1_Pin_LG;
     ADC_Pressure_Altitude                     => This_Controller_Instance.ADC_Pressure_Altitude,                       -- Air data computer input over Arinc 429 on aircraft
     Total_Inlet_Temperature_T1_Hi             => Inlet_Temperature_Sensor,                                             -- Pin_LH  -- T1_Hi                  Interface_J1_Pin_LH;
     Delta_Shp_C_Acc                           => Delta_Shp_C_Acc,                                                      -- Torque_Types.Horsepower;               -- direct from gas generator
     Delta_Shp_C_Bleed                         => Delta_Shp_C_Bleed,                                                    -- Torque_Types.Horsepower;               -- direct from gas generator
     Delta_Shp_C_Np                            => Delta_Shp_C_Np,                                                       -- Torque_Types.Horsepower;               -- direct from gas generator
     Delta_Itt_C_Acc                           => Delta_Itt_C_Acc,                                                      -- Temperature_Types.Kelvin;              -- direct from gas generator
     Delta_Itt_C_Bleed                         => Delta_Itt_C_Bleed,                                                    -- Temperature_Types.Kelvin;              -- direct from gas generator
     Delta_Ng_C_Acc                            => Delta_Ng_C_Acc,                                                       -- Angle_Types.Rpm;                       -- direct from gas generator
     Delta_Ng_C_Bleed                          => Delta_Ng_C_Bleed,                                                     -- Angle_Types.Rpm;                       -- direct from gas generator
     Shp                                       => Shp,                                                                  -- Torque_Types.Horsepower;               -- direct from gas generator
     K_Fmv_Delta                               => K_Fmv_Delta,                                                          -- Normalized_Types.Normalize(1.0));      -- direct from gas generator
     Delta_Ng_Seizure                          => Delta_Ng_Seizure,                                                     -- Normalized_Types.normalize             -- direct from gas generator
     Fired                                     => Fired,                                                                -- boolean                                -- direct from gas generator
     Ambient_Temperature_Ratio                 => This_Controller_Instance.Ambient_Temperature_Ratio);

     -- This is the manual feather signal that is redundant to the pmu feather signal.
     -- The PMU causes the trim motors to also feather the prop when the pcl is reduced below 10.0 deg.
     -- The Uncommanded Feather malfunction feathers the propeller regardless of throttle position or
     -- circuit breaker status. With no malfunction, the manual feather signal is issued when the
     -- PCL is reduced below 7.0 degrees (about halfway between idle and off). If the PROP SYS circuit
     -- breaker is pulled before the PCL is reduced, the feathering action is delayed.

     if Uncommanded_Feather then

       This_Controller_Instance.Manual_Feather_Sig := This_Controller_Instance.Prop_Sys_Cb;
       Feather_Delay_Timer := 91.0;

     elsif This_Controller_Instance.Prop_Sys_Cb then

        if This_Controller_Instance.Prop_Feather_Cut_Off_Switch then

           This_Controller_Instance.Manual_Feather_Sig := False;
           Feather_Delay_Timer := 0.0;
        else
           This_Controller_Instance.Manual_Feather_Sig := True;
           Feather_Delay_Timer := 91.0;
        end if;

     elsif Feather_Delay_Timer < 90.0 then

        Feather_Delay_Timer := Feather_Delay_Timer + Iconst;
     else
        This_Controller_Instance.Manual_Feather_Sig := not This_Controller_Instance.Prop_Feather_Cut_Off_Switch;
     end if;

  -- The PMU Fail Relay
      Container.Auto_Return_Relay.Update
        ( An_Instance      => Next_Subsystem.The_Pmu_Fail_Relay,
          Power_And_Ground => Power_Management_Unit.Pmu_Fail
          (An_Instance => Next_Subsystem.The_Power_Management_Unit));


--| *Starter engagement*. The starter can be engaged either manually
--| (cockpit starter switch to MANUAL) or automatically by the PMU
--| model. If electrical power is available to the starter/generator
--| the starter model is controlled as follow:

-- The Auto Start Relay
      Container.Auto_Return_Relay.Update
        ( An_Instance      => Next_Subsystem.The_Auto_Start_Relay,
          Power_And_Ground => This_Controller_Instance.Start_Cb and Power_Management_Unit.Start
                               ( An_Instance => Next_Subsystem.The_Power_Management_Unit));

      -- Start Relay and Starter On Relay Logic
      -- Note: Both relays are tied together right before the
      -- Start relay hot lead.  Needed for potential malfunction of
      -- start relay or starter on relay independently.

      Start_Logic := This_Controller_Instance.Start_Cb and (Container.Auto_Return_Relay.Is_Energized
                                            (An_Instance => Next_Subsystem.The_Auto_Start_Relay)
                                            or This_Controller_Instance.Forward_Starter_Switch_Manual); -- or Aft_Start_Switch);

-- The Start Relay
      Container.Auto_Return_Relay.Update
        ( An_Instance      => Next_Subsystem.The_Start_Relay,
          Power_And_Ground => Start_Logic);

-- The Starter On Relay
      Container.Auto_Return_Relay.Update
        ( An_Instance      => Next_Subsystem.The_Starter_On_Relay,
          Power_And_Ground => Start_Logic);

--| *Ignition activation*. The ignition system can be activated either
--| manually (cockpit ignition switch to ON) or automatically by the
--| PMU model. If electrical power is available via the IGN_ELECT_PWR (Ignition_Cb)
--| circuit breaker the ignition is controlled as follows:

--|  Update the state of the auto ignition relay which depends on the whether
--|  the PMU detects the need for ignition.

--| Auto_Ignition_Relay
      Container.Auto_Return_Relay.Update
       (An_Instance => Next_Subsystem.The_Auto_Ignition_Relay,
        Power_And_Ground => This_Controller_Instance.Ignition_Cb and (Power_Management_Unit.Ignition
                              (An_Instance => Next_Subsystem.The_Power_Management_Unit)));

--| Ignition_Exciter
      JPATS_Powerplant.Container.Next_Subsystem.Ignition_Exciter := (This_Controller_Instance.Ignition_Cb and ((Container.Auto_Return_Relay.Is_Energized
                                (An_Instance => Next_Subsystem.The_Auto_Ignition_Relay))
                           or This_Controller_Instance.Forward_Ignition_Switch_On)); -- or Aft_Ignition_Switch_On );

--| Ignition Select Annunciator
      This_Controller_Instance.Ignition_Select := Next_Subsystem.Ignition_Exciter;


--| Convert Fuel Available to a minimum pressure from fuel_cat.
      if not Flameout_Malfunction then
        if Next_Subsystem.Quick_Start then
          This_Controller_Instance.Fuel_Pressure  := 27.0;
          This_Controller_Instance.Fuel_Flow_Rate := 1900.0;
        else
          This_Controller_Instance.Fuel_Pressure := Jpats_Fuel.Fuel_Pressure;
          This_Controller_Instance.Fuel_Flow_Rate := Jpats_Fuel.Fuel_Flow;  -- Mass_Types.Pph(2584.00);
        end if;
      else
        This_Controller_Instance.Fuel_Pressure := 0.0;
        This_Controller_Instance.Fuel_Flow_Rate := 0.0;
      end if;

--|  Check for fuel available from boost pumps.  Fuel available means appropriately
--|  pressurised fuel at the inlet of the intermediate pump.  This means that all
--|  timing considered with a cutoff the fuel firewall valve is the responsiblity
--|  of the systems Fuel_CAT category.
      Container.Intermediate_Pressure_Pump.Update
        (Next_Instance   => Next_Subsystem.The_Intermediate_Pressure_Pump,
         This_Instance   => This_Subsystem.The_Intermediate_Pressure_Pump,
         Iconst          => Iconst,
         Inlet_Pressure  => This_Controller_Instance.Fuel_Pressure,
         Rpm             => Pratt_Whitney_Pt6a_68_Gas_Generator.Boost_Pump_Shaft_Rpm
                              (An_Instance => This_Subsystem.The_Gas_Generator),
         Flow_Rate       => This_Controller_Instance.Fuel_Flow_Rate,
         Cavitating      => JPATS_Fuel.Pump_Degradation_Request);


--|  Update the engine driven high pressure fuel pump.  The pump takes in fuel from
--|  intermediate pump.  Depending on how fast it is running, excess fuel is pumped
--|  back to the collector tank.
      if Next_Subsystem.Quick_Start_Timer < 1.0 then
        Fcu_Shaft_Rpm := 2000.0;
      else
        Fcu_Shaft_Rpm :=  Pratt_Whitney_Pt6a_68_Gas_Generator.Fcu_Shaft_Rpm
          (An_Instance => This_Subsystem.The_Gas_Generator);
      end if;
      Container.High_Pressure_Pump.Update
        (An_Instance     => Next_Subsystem.The_High_Pressure_Pump,
         Inlet_Pressure  => Container.Intermediate_Pressure_Pump.Outlet_Pressure
                              (An_Instance => Next_Subsystem.The_Intermediate_Pressure_Pump),
         Rpm             => Fcu_Shaft_Rpm);


      Stepper_Motor_Position_Requested :=
        Power_Management_Unit.Stepper_Motor_Position_Demanded
        (An_Instance => Next_Subsystem.The_Power_Management_Unit);

      Fuel_Metering_Unit.Update
        ( Next_Instance                        => Next_Subsystem.The_Fuel_Metering_Unit,
          This_Instance                        => This_Subsystem.The_Fuel_Metering_Unit,
          Iconst                               => Iconst,
          Power_Control_Lever_Angle_In_Degrees => This_Controller_Instance.Power_Control_Lever_Angle,
          Stepper_Motor_Position_Requested     => Stepper_Motor_Position_Requested,
          Shutdown_Solenoid_Power              => This_Controller_Instance.Pmu_Cb,
          Energize_Shutdown_Solenoid           => Power_Management_Unit.Shutdown_Solenoid
                                                   (An_Instance => Next_Subsystem.The_Power_Management_Unit),
          Cam_Reset_Solenoid                   => This_Controller_Instance.Pmu_Cb or
                                                  Power_Turbine_Rpm_Sensor > 600.0,
          Ambient_Pressure_Ratio               => This_Controller_Instance.Ambient_Pressure_Ratio,
          Return_Line_Pressure                 => This_Controller_Instance.FMV_Return_Line_Pressure,
          Inlet_Pressure                       => Container.High_Pressure_Pump.Outlet_Pressure
                                                   ( An_Instance => Next_Subsystem.The_High_Pressure_Pump),
          Linkage_Failure                      => This_Controller_Instance.Linkage_Failure,
          Restrain_Fuel_Flow                   => (Slow_Start_Request and Gas_Generator_Rpm_Sensor < 50.0*374.68),
          Fmu_Malfunction                      => Fmu_Malfunction);

      if not JPATS_Powerplant.Test_Controller.Accessory_Horsepower_Auto_Driven then
--        Accessory_Horsepower := Torque_Types.Horsepower(7.5);
          This_Controller_Instance.Accessory_Horsepower := Torque_Types.Horsepower(Float(This_Controller_Instance.Accessory_Torque_Absorbed(X)) *
          Float(Pratt_Whitney_Pt6a_68_Gas_Generator.Generator_Shaft_Rpm
          (An_Instance => This_Subsystem.The_Gas_Generator))/550.0/60.0*2.0*Ada.Numerics.Pi)  -- Units of horsepower
             + This_Controller_Instance.Pump_Shaft_Horsepower_Absorbed;
      else
        This_Controller_Instance.Accessory_Horsepower := JPATS_Powerplant.Test_Controller.Accessory_Horsepower;
      end if;

      if Flameout_Malfunction then
          Flameout_Timer := Float'Min(10.0,Flameout_Timer + Iconst);
      else
        Flameout_Timer := 0.0;
      end if;


      if not JPATS_Powerplant.Test_Controller.Fuel_Flow_Auto_Driven then
        This_Controller_Instance.Fuel_Flow_Burn_Rate := Fuel_Metering_Unit.Fuel_Flow_Rate
          (An_Instance => Next_Subsystem.The_Fuel_Metering_Unit);

      else
        This_Controller_Instance.Fuel_Flow_Burn_Rate := JPATS_Powerplant.Test_Controller.Fuel_Flow_Burn_Rate;
      end if;

--      K_Check_Valve : = 0.0047;
--      Specific_Gravity_Of_Fuel := 0.819;  -- dimensionless
--      Density_Of_Water := 62.4; -- lbm/ft**3.0
--      Fuel_Density := Specific_Gravity_Of_Fuel*Density_Of_Water; -- lbm/ft**3.0;
      Return_Line_Flow_Rate := (This_Controller_Instance.Fuel_Flow_Rate - This_Controller_Instance.Fuel_Flow_Burn_Rate); -- lbm/hr
--      Check_Valve_Pressure_Drop := (3.0E-11*Return_Line_Flow_Rate**3.0 + 3.0E-8*Return_Line_Flow_Rate**2.0 + 4.0E-5*Return_Line_Flow_Rate + 0.3484) + 13.8/2454.0*Return_Line_Flow_Rate;
      This_Controller_Instance.Check_Valve_Pressure_Drop := (3.0E-11*Return_Line_Flow_Rate**3.0 + 3.0E-8*Return_Line_Flow_Rate**2.0 + 4.0E-5*Return_Line_Flow_Rate + 0.3484); --  + 13.8/2454.0*Return_Line_Flow_Rate;
--      Log.Report("check valve pressure drop := " & Float'Image(Check_Valve_Pressure_Drop));
      This_Controller_Instance.Pump_Return_Line_Pressure := Container.Intermediate_Pressure_Pump.Outlet_Pressure
        (An_Instance => Next_Subsystem.The_Intermediate_Pressure_Pump) - This_Controller_Instance.Check_Valve_Pressure_Drop;

      Pratt_Whitney_Pt6a_68_Gas_Generator.Update
        (Next_Instance                   => Next_Subsystem.The_Gas_Generator,
         This_Instance                   => This_Subsystem.The_Gas_Generator,
         Iconst                          => Iconst,
         Ambient_Temperature_Ratio       => This_Controller_Instance.Ambient_Temperature_Ratio,
         Ambient_Pressure_Ratio          => This_Controller_Instance.Ambient_Pressure_Ratio,
         Mach_Number                     => This_Controller_Instance.Mach_Number,
         Fuel_Flow_Rate                  => This_Controller_Instance.Fuel_Flow_Burn_Rate,
         Bleed_Flow_Rate                 => This_Controller_Instance.Bleed_Flow_Rate,
         Propeller_Torque_Absorbed       => This_Controller_Instance.Propeller_Torque_Absorbed,
         Accessory_Torque_Absorbed       => This_Controller_Instance.Accessory_Torque_Absorbed,
         Air_Conditioner_Torque_Absorbed => This_Controller_Instance.Air_Conditioner_Torque_Absorbed,
         Accessory_Horsepower            => This_Controller_Instance.Accessory_Horsepower,
         Ignition_Exciter                => Next_Subsystem.Ignition_Exciter,
         Hot_Start                       => Hot_Start,                                  -- malf interface
         No_Light_Request                => No_Light_Request,                           -- malf interface
         Engine_Seizure                  => This_Controller_Instance.Engine_Seizure,    -- malf interface
         Itt_Sensor_Malfunction          => Itt_Sensor_Malfunction,                     -- malf interface
         Overtemp                        => Overtemp,                                   -- malf interface
         Oil_Leak                        => Oil_Leak,
         Oil_Pressure_Failure            => Oil_Pressure_Failure,
         High_Oil_Temperature            => High_Oil_Temperature,
         Load_Factor                     => This_Controller_Instance.Load_Factor,
--       Voltage                         => Electrical_Units_Types.Volts_Dc(28.0),
         Voltage                         => JPATS_Electrical.Voltage(Jpats_Electrical_Types.Starter_Distribution_Bus),
         Chip                            => Chip,
         Flameout_Malfunction            => Flameout_Malfunction,
         Torque_Auto_Driven              => JPATS_Powerplant.Test_Controller.Torque_Auto_Driven,
         Auto_Driven_Torque              => JPATS_Powerplant.Test_Controller.Torque);

     -- Oil Transducer
     This_Controller_Instance.Oil_Pressure_Transducer_Signal := Pratt_Whitney_Pt6a_68_Gas_Generator.Engine_Oil_Pressure
                                (An_Instance => Next_Subsystem.The_Gas_Generator)
                                * Float(Boolean'Pos(This_Controller_Instance.Oil_Trx_Cb));

     if not This_Controller_Instance.Oil_Temperature_Transducer_Failure then
       This_Controller_Instance.Oil_Temperature_Transducer_Signal := Pratt_Whitney_Pt6a_68_Gas_Generator.Engine_Oil_Temperature
         (An_Instance => Next_Subsystem.The_Gas_Generator);
     else
       This_Controller_Instance.Oil_Temperature_Transducer_Signal := -75.0;
     end if;

     -- Engine Vibration
     declare
       Engine_Vibration_Gain : Float := 0.00120;
     begin

       if This_Controller_Instance.Left_Weight_On_Wheels and This_Controller_Instance.Right_Weight_On_Wheels then
         This_Controller_Instance.Engine_Vibration_Magnitude
           := Engine_Vibration_Gain * Float'Max(0.0, Float'Min(2889.0, Power_Turbine_Torque_Sensor-115.5))
                *(1.0 - Float'Max(0.0, Float'Min(1.0, Float(This_Controller_Instance.Ground_Speed)*0.14812 ))); --0.067))));
       else
         This_Controller_Instance.Engine_Vibration_Magnitude := 0.0;
       end if;
     end;

     declare
       Mach_Number_Limit          : Normalized_Types.Normalize := 2.0;
       Maximum_Afterburner_Thrust : Force_Types.lbf := 5500.0;
       Afterburner_Thrust         : Force_Types.Lbf;
     begin
--       Log.Report("Bool2 " & Boolean'Image(JPATS_Simulated_Aircraft.Bool2));
--       Log.Report("pcl > 40.0 " & Boolean'Image(Power_Control_Lever_Angle > 40.0));
       if JPATS_Powerplant.IOS_Interface.This_Ios_Interface.Srbool10
         and (This_Controller_Instance.Power_Control_Lever_Angle > 40.0) then

         Afterburner_Thrust := Maximum_Afterburner_Thrust * This_Controller_Instance.Ambient_Pressure_Ratio
                                 /This_Controller_Instance.Ambient_Temperature_Ratio*0.0023769
                                 *(( Mach_Number_Limit - This_Controller_Instance.Mach_Number )**2.0)**4.358;

       else
         Afterburner_Thrust := 0.0;
       end if;
--       Log.Report("afterburner thrust = " & Float'Image(Afterburner_Thrust));
       if not Next_Subsystem.Quick_Start then
         Geometry.Update
           (An_Instance           => Next_Subsystem.The_Powerplant_Geometry,
            Axial_Thrust          => ((Pratt_Whitney_Pt6a_68_Gas_Generator.Exhaust_Thrust
                                       (An_Instance => Next_Subsystem.The_Gas_Generator) + Afterburner_Thrust),
                                      Force_Types.Lbf(0.0),Force_Types.Lbf(0.0)),
            Axial_Torque          => (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0)),
            Inertia               => Mass_Types.Slugs_Feet_Sq(0.0),
            Rpm                   => (Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
                                      (An_Instance => Next_Subsystem.The_Gas_Generator),
                                      Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0)),
            Aircraft_Angular_Rate => This_Controller_Instance.Aircraft_Angular_Rate,
            Aircraft_Cg           => This_Controller_Instance.Aircraft_Cg );
       else
         Geometry.Update
           (An_Instance           => Next_Subsystem.The_Powerplant_Geometry,
            Axial_Thrust          => (Force_Types.Lbf(0.0),Force_Types.Lbf(0.0),Force_Types.Lbf(0.0)),
            Axial_Torque          => (Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0),Torque_Types.Ft_Lbf(0.0)),
            Inertia               => Mass_Types.Slugs_Feet_Sq(0.0),
            Rpm                   => (Pratt_Whitney_Pt6a_68_Gas_Generator.Gas_Generator_Rpm
                                      (An_Instance => Next_Subsystem.The_Gas_Generator),
                                      Angle_Types.Rpm(0.0),Angle_Types.Rpm(0.0)),
            Aircraft_Angular_Rate => This_Controller_Instance.Aircraft_Angular_Rate,
            Aircraft_Cg           => This_Controller_Instance.Aircraft_Cg );
       end if;
     end;

     -- Call Reset code again to prevent having to add a lag label to initiate system reset again
     -- when Itt_Sensor_Malfunction clears
     -- This code doesn't work the way it was described to work above.
     -- System Reset needs to be hit after the malfunction is cleared.
     -- This code should be removed, but it is not removed due to a time constraint for release.
     -- There is not enough time to remove and retest.
     if Reset_Again then
        Pratt_Whitney_Pt6a_68_Gas_Generator.Cold_Soak_Reset
          (An_Instance => This_Subsystem.The_Gas_Generator);
        Pratt_Whitney_Pt6a_68_Gas_Generator.Rotor_Reset
          (An_Instance => This_Subsystem.The_Gas_Generator);
        Fuel_Metering_Unit.Fmu_Reset
          ( An_Instance => This_Subsystem.The_Fuel_Metering_Unit);
        Power_Management_Unit.Pmu_Reset
          (An_Instance => This_Subsystem.The_Power_Management_Unit);
        JPATS_Propeller.Reset;
        Reset_Again := False;
     end if;
     -- Remove to here

  end Update;


  procedure Update
      ( Integration_Constant : in Float ) is

  begin

    Update
      (Next_Subsystem => Container.Next_Subsystem,
       This_Subsystem => Container.This_Subsystem,
       Iconst         => Integration_Constant);

    Container.Swap_Ptr := Container.This_Subsystem;
    Container.This_Subsystem := Container.Next_Subsystem;
    Container.Next_Subsystem := Container.Swap_Ptr;
    Container.Next_Subsystem.all := Container.This_Subsystem.all;

     -- This last statement is needed because all states are not written to every pass.
     -- This in effect implements a zero order hold on the instance variables.

  end Update;


  procedure Register_Stethoscope_Labels is

  begin

   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Manual_Select",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Manual_Select'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Prop_Feather_Cut_Off_Switch",
--      Units          => "Boolean",
--      Object_Address => Prop_Feather_Cut_Off_Switch'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Forward_Starter_Switch_Manual",
--      Units          => "Boolean",
--      Object_Address => Forward_Starter_Switch_Manual'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Forward_Starter_Switch_In_Auto",
--      Units          => "Boolean",
--      Object_Address => Forward_Starter_Switch_In_Auto'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Forward_Ignition_Switch_On",
--      Units          => "Boolean",
--      Object_Address => Forward_Ignition_Switch_On'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Electrical_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Electrical_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Edm_Batt_Cb",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Edm_Batt_Cb'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Edm_Gen_Cb",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Edm_Gen_Cb'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Ignition_Cb",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Ignition_Cb'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Start_Cb",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Start_Cb'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Oil_Trx_Cb",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Oil_Trx_Cb'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Pmu_Cb",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Pmu_Cb'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Atmosphere_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Atmosphere_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Ambient_Pressure_Ratio",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Ambient_Pressure_Ratio'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Ambient_Temperature_Ratio",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Ambient_Temperature_Ratio'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Pressure_Altitude",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Pressure_Altitude'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Ambient_Temperature",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Ambient_Temperature'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Ambient_Pressure",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Ambient_Pressure'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Landing_Gear_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Landing_Gear_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Left_Weight_On_Wheels",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Left_Weight_On_Wheels'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Right_Weight_On_Wheels",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Right_Weight_On_Wheels'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Aircraft_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Aircraft_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Mach_Number",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Mach_Number'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Aircraft_Cg(x)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Aircraft_Cg(X)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Aircraft_Cg(y)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Aircraft_Cg(Y)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Aircraft_Cg(z)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Aircraft_Cg(Z)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Aircraft_Angular_Rate(roll)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Aircraft_Angular_Rate(Roll)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Aircraft_Angular_Rate(pitch)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Aircraft_Angular_Rate(Pitch)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Aircraft_Angular_Rate(yaw)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Aircraft_Angular_Rate(Yaw)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Propeller_Torque_Absorbed_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Propeller_Torque_Absorbed_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Propeller_Torque_Absorbed(x)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Propeller_Torque_Absorbed(X)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Propeller_Torque_Absorbed(y)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Propeller_Torque_Absorbed(Y)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Propeller_Torque_Absorbed(z)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Propeller_Torque_Absorbed(Z)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Accessory_Torque_Absorbed(x)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Accessory_Torque_Absorbed(X)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Accessory_Torque_Absorbed(y)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Accessory_Torque_Absorbed(Y)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Accessory_Torque_Absorbed(z)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Accessory_Torque_Absorbed(Z)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Fuel_Available_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Fuel_Available_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Fuel_Available",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Fuel_Available'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Fuel_Flow_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Fuel_Flow_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Fuel_Flow_Burn_Rate",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Fuel_Flow_Burn_Rate'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Environmental_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Environmental_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Bleed_Air_On",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Bleed_Air_On'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Air_Conditioner_Torque_Absorbed(x)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Air_Conditioner_Torque_Absorbed(X)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Air_Conditioner_Torque_Absorbed(y)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Air_Conditioner_Torque_Absorbed(Y)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Air_Conditioner_Torque_Absorbed(z)",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Air_Conditioner_Torque_Absorbed(Z)'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Bleed_Flow_Rate",
      Units          => "Float",
      Object_Address => This_Controller_Instance.Bleed_Flow_Rate'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/ADC_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.ADC_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/ADC_Pressure_Altitude",
      Units          => "Float",
      Object_Address => This_Controller_Instance.ADC_Pressure_Altitude'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/ADC_Mach_Number",
      Units          => "Float",
      Object_Address => This_Controller_Instance.ADC_Mach_Number'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Ign_Sel",
      Units          => "Boolean",
      Object_Address => This_Controller_Instance.Ignition_Select'Address,
      Value_Type     => "uchar",
      Scope_Index    => 0);
--   Stethoscope.Register_Signal
--     (Name           => "Powerplant/Controller/Accessory_Horsepower_auto_driven",
--      Units          => "Boolean",
--      Object_Address => This_Controller_Instance.Accessory_Horsepower_auto_driven'Address,
--      Value_Type     => "uchar",
--      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Accessory_Horsepower",
      Units          => "float",
      Object_Address => This_Controller_Instance.Accessory_Horsepower'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Power_Control_Lever_Angle",
      Units          => "float",
      Object_Address => This_Controller_Instance.Power_Control_Lever_Angle'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Power_Control_Lever_Angle_RVDT",
      Units          => "float",
      Object_Address => This_Controller_Instance.Power_Control_Lever_Angle_Rvdt'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Stethoscope.Register_Signal
     (Name           => "Powerplant/Controller/Power_Control_Lever_Angle_LVDT",
      Units          => "float",
      Object_Address => This_Controller_Instance.Power_Control_Lever_Angle_LVDT'Address,
      Value_Type     => "float",
      Scope_Index    => 0);
   Jpats_Ios_Interface.Register
      ( Name        => "engine_seizure",
        Variable    => This_Controller_Instance.Engine_Seizure'Address
        );
   Jpats_Ios_Interface.Register
      ( Name        => "linkage",
        Variable    => This_Controller_Instance.Linkage_Failure'Address
        );
  end Register_Stethoscope_Labels;


-- Method Initialize
   procedure Initialize
     (A_Subsystem : in out JPATS_Powerplant.Container.Instance_Ptr) is

   begin
     --|  Initialize the Domain Classes
     --|

     Power_Management_Unit.Initialize
       (An_Instance => A_Subsystem.The_Power_Management_Unit);
     Container.Auto_Return_Relay.Initialize
       (An_Instance   => A_Subsystem.The_Auto_Ignition_Relay,
        Initial_State => False);
     Container.Auto_Return_Relay.Initialize
       (An_Instance   => A_Subsystem.The_Auto_Start_Relay,
        Initial_State => False);
     Container.Auto_Return_Relay.Initialize
       (An_Instance   => A_Subsystem.The_Start_Relay,
        Initial_State => False);
     Container.Intermediate_Pressure_Pump.Initialize
       (An_Instance => A_Subsystem.The_Intermediate_Pressure_Pump);
     Container.High_Pressure_Pump.Initialize
       (An_Instance => A_Subsystem.The_High_Pressure_Pump);
     Fuel_Metering_Unit.Initialize
       (An_Instance => A_Subsystem.The_Fuel_Metering_Unit);
     Pratt_Whitney_Pt6a_68_Gas_Generator.Initialize
       (An_Instance => A_Subsystem.The_Gas_Generator);
     Geometry.Initialize
       (An_Instance => A_Subsystem.The_Powerplant_Geometry);
     Container.Auto_Return_Relay.Initialize
       (An_Instance   => A_Subsystem.The_Pmu_Fail_Relay,
        Initial_State => False);

     Register_Stethoscope_Labels;

   end Initialize;

   -- Method Read_Tables
   procedure Read_Tables is
   begin
     Container.High_Pressure_Pump.Read_Tables;
     Container.Intermediate_Pressure_Pump.Read_Tables;
     Fuel_Metering_Unit.Read_Tables;
     Power_Management_Unit.Read_Tables;
     Pratt_Whitney_Pt6a_68_Gas_Generator.Read_Tables;
   end Read_Tables;

   procedure Initialize is
   begin
     Read_Tables;
     Initialize
       (A_Subsystem => Container.Next_Subsystem);
     Initialize
       (A_Subsystem => Container.This_Subsystem);

   end Initialize;


end JPATS_Powerplant.Controller;







