-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Steven D. Roberts
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Numerics;
use  Ada.Numerics;
with Gear;
with Angle_Types;

package body Gear_Box is


-- Access Chip_Detect
   function  Chip_Detect
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Chip_Detect;
   end Chip_Detect;

   procedure Set_Chip_Detect
     (An_Instance        : in out Instance;
      Chip_Detect_State  : in     Boolean) is
   begin
      An_Instance.Chip_Detect := Chip_Detect_State;
   end Set_Chip_Detect;

-- Access Ng_Sensor
   function  Ng_Sensor
     (An_Instance : in Instance)
      return Normalized_Types.Normalize is
   begin
      return Normalized_Types.Normalize(Shaft.Rpm(An_Instance.Shaft1)/37468.0 );
   end Ng_Sensor;

   procedure Set_Ng_Sensor
     (An_Instance : in out Instance;
      Ng_Sensor   : in     Normalized_Types.Normalize ) is
   begin
      An_Instance.Ng_Sensor := Ng_Sensor;
   end Set_Ng_Sensor;

   procedure Calculate_Horsepower_Transfer ( Input_Shaft       : in out Shaft.Instance;
                                             Input_Shaft_Gear  : in     Gear.Instance;
                                             Output_Shaft      : in out Shaft.Instance;
                                             Output_Shaft_Gear : in     Gear.Instance) is
    begin
       Shaft.Set_Torque(An_Instance  => Output_Shaft,
                        Torque_Value => -1.0*Shaft.Torque(Input_Shaft)*
                        (Gear.Radius(Output_Shaft_Gear)/Gear.Radius(Input_Shaft_Gear)));
       Shaft.Set_Rpm( An_Instance => Output_Shaft,
                      Rpm_Value   => -1.0*Shaft.Rpm(Input_Shaft)*
                                     (Gear.Radius(Input_Shaft_Gear)/Gear.Radius(Output_Shaft_Gear)));
    end Calculate_Horsepower_Transfer;

   -- Gas_Generator_Driven_Update Method
   procedure Gas_Generator_Driven_Update (An_Instance  : in out Instance) is

   begin

--      Total-Moment_balance
       Calculate_Horsepower_Transfer( An_Instance.Shaft1,
                                      Shaft.Gears(An_Instance.Shaft1,1),
                                      An_Instance.Shaft2,
                                      Shaft.Gears(An_Instance.Shaft2,1));
        Calculate_Horsepower_Transfer( An_Instance.Shaft2,
                                      Shaft.Gears(An_Instance.Shaft2,2),
                                       An_Instance.Shaft3,
                                      Shaft.Gears(An_Instance.Shaft3,2));
        Calculate_Horsepower_Transfer( An_Instance.Shaft3,
                                      Shaft.Gears(An_Instance.Shaft3,1),
                                       An_Instance.Shaft4,
                                      Shaft.Gears(An_Instance.Shaft4,2));
        Calculate_Horsepower_Transfer( An_Instance.Shaft4,
                                      Shaft.Gears(An_Instance.Shaft4,1),
                                       An_Instance.Shaft5,
                                      Shaft.Gears(An_Instance.Shaft5,1));
   end Gas_Generator_Driven_Update;

   -- Starter_Driven_Update Method
   procedure Starter_Driven_Update (An_Instance         : in out Instance;
                                    Input_Rpm           : in     Angle_Types.Rpm;
                                    Input_Torque        : in     Torque_Types.Ft_Lbf;
                                    Moment_Input_Shaft1 : in     Torque_Types.Ft_Lbf;
                                    Moment_Input_Shaft3 : in     Torque_Types.Ft_Lbf;
                                    Moment_Input_Shaft4 : in     Torque_Types.Ft_Lbf;
                                    Moment_Input_Shaft5 : in     Torque_Types.Ft_Lbf ) is

   begin

      Calculate_Horsepower_Transfer( An_Instance.Shaft2,
                                     Shaft.Gears(An_Instance.Shaft2,1),
                                     An_Instance.Shaft1,
                                     Shaft.Gears(An_Instance.Shaft1,1));
      Calculate_Horsepower_Transfer( An_Instance.Shaft2,
                                     Shaft.Gears(An_Instance.Shaft2,2),
                                     An_Instance.Shaft3,
                                     Shaft.Gears(An_Instance.Shaft3,2));
      Calculate_Horsepower_Transfer( An_Instance.Shaft3,
                                     Shaft.Gears(An_Instance.Shaft3,1),
                                     An_Instance.Shaft4,
                                     Shaft.Gears(An_Instance.Shaft4,2));
      Calculate_Horsepower_Transfer( An_Instance.Shaft4,
                                     Shaft.Gears(An_Instance.Shaft4,1),
                                     An_Instance.Shaft5,
                                     Shaft.Gears(An_Instance.Shaft5,1));

   end Starter_Driven_Update;

     Gear_Radii1 : Shaft.Radii_Type(1..1);
     Gear_Radii2 : Shaft.Radii_Type(1..2);

-- Method Initialize
   procedure Initialize
     (An_Instance : in out Instance) is
   begin
--|  Initialize the Shafts with the number of gears

      Gear_Radii1 := (others => 22.0);  -- gas_generator shaft.
      An_Instance.Shaft1 := Shaft.New_Shaft(Gear_Radii1);
      Gear_Radii2 := (75.0,52.0);  -- Starter_Generator shaft
      An_Instance.Shaft2 := Shaft.New_Shaft(Gear_Radii2);
      Gear_Radii2 := (75.0,20.0);  -- high pressure gear pump drive shaft
      An_Instance.Shaft3 := Shaft.New_Shaft(Gear_Radii2);
      Gear_Radii2 := (10.0,20.0);  -- Ng sensor shaft
      An_Instance.Shaft4 := Shaft.New_Shaft(Gear_Radii2);
      Gear_Radii2 := (10.0,20.0);  -- Intermediate pressure fuel pump shaft
      An_Instance.Shaft5 := Shaft.New_Shaft(Gear_Radii2);
      An_Instance.Chip_Detect := False;
   end Initialize;

   function Is_Positive (X : in Float) return Boolean is
   begin
      if X >= 0.0 then
         return True;
      else
         return False;
      end if;
   end Is_Positive;

end Gear_Box;
