-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
-- Update Count    : 0
--with Ada.Numerics;
--use  Ada.Numerics;
--with Ada.Numerics.Elementary_Functions;
--use  Ada.Numerics.Elementary_Functions;
with Interpolation_Table.Multiply_Indexed;
with Interpolation_Table.Singly_Indexed;
with Mass_Types;
with Simulation_Dictionary;
with Stethoscope;

package body Fuel_Metering_Unit is

  PCLPLA_Table     : aliased Interpolation_Table.Singly_Indexed.Instance;
  PLAPCL_Table     : aliased Interpolation_Table.Singly_Indexed.Instance;

  -- Accessor Altitude Gain
  function Altitude_Gain
    (An_Instance : in Instance)
    return Normalized_Types.Normalize
  is
  begin
    return Fuel_Metering_Valve.Altitude_Gain
      (An_Instance => An_Instance.Fmu_Metering_Valve);
  end Altitude_Gain;

  -- Accessor Power_Control_Lever_Angle_Lvdt
   function Power_Control_Lever_Angle_Lvdt
     (An_Instance : in Instance)
     return Angle_Types.Degrees
   is

   begin
     return An_Instance.Power_Control_Lever_Angle_Lvdt;
   end Power_Control_Lever_Angle_Lvdt;

   -- Accessor Power_Lever_Angle
   function Power_Lever_Angle
     (An_Instance : in Instance)
     return Angle_Types.Degrees
   is

   begin
     return New_Cam.Position
       ( An_Instance => An_Instance.Hydro_Mechanical_Cam);
   end Power_Lever_Angle;

   -- Accessor Fuel_Output_Pressure
   function Fuel_Output_Pressure
     (An_Instance : in Instance)
     return Force_Types.Psi
   is

   begin
     return Fuel_Metering_Valve.Output_Pressure
       (An_Instance => An_Instance.Fmu_Metering_Valve);
   end Fuel_Output_Pressure;


   -- Access Fuel_Flow_Rate
   function Fuel_Flow_Rate
     (An_Instance : in Instance)
     return Mass_Types.Pph
   is
   begin
     return Fuel_Metering_Valve.Flow_Rate
       (An_Instance => An_Instance.Fmu_Metering_Valve);
   end Fuel_Flow_Rate;


   -- Access Fuel_Bypass_Flow_Rate
   function Fuel_Bypass_Flow_Rate
     (An_Instance : in Instance)
     return Mass_Types.Pph
   is
   begin
     return Fuel_Metering_Valve.Bypass_Flow_Rate
       (An_Instance => An_Instance.Fmu_Metering_Valve);
   end Fuel_Bypass_Flow_Rate;

   procedure Fmu_Reset
     (An_Instance : in out Instance) is
   begin
     An_Instance.Fmu_Reset := True;
   end Fmu_Reset;

-- Method Update
   procedure Update
     ( Next_Instance                        : in out Instance;
       This_Instance                        : in     Instance;
       Iconst                               : in     Float;
       Power_Control_Lever_Angle_In_Degrees : in     Angle_Types.Degrees;
       Stepper_Motor_Position_Requested     : in     Jpats_Powerplant_Types.Step_Type;
       Shutdown_Solenoid_Power              : in     Boolean;
       Energize_Shutdown_Solenoid           : in     Boolean;
       Cam_Reset_Solenoid                   : in     Boolean;
       Ambient_Pressure_Ratio               : in     Normalized_Types.Normalize;
       Return_Line_Pressure                 : in     Force_Types.Psi;
       Inlet_Pressure                       : in     Force_Types.Psi;
       Linkage_Failure                      : in     Boolean;
       Restrain_Fuel_Flow                   : in     Boolean;
       Fmu_Malfunction                      : in     Boolean) is

     Smp                      : Jpats_Powerplant_Types.Step_Type;
     Delta_Pcl                : Angle_Types.Degrees               renames Power_Control_Lever_Angle_In_Degrees;
     Delta_Ambient            : Normalized_Types.Normalize        renames Ambient_Pressure_Ratio;
     Delta_Pla                : Angle_Types.Degrees;
     Delta_Pla_Prime          : Angle_Types.Degrees;
     Cam_Reset_Solenoid_State : Boolean;
     Shutdown_Solenoid_State  : Boolean;
     Smp_Req_Dot              : Integer;
   begin

      -- Reversion to Manual Mode is commanded by de-energizing the CAM RESET solenoid
      -- (referred to as Stepper Motor Reset Solenoid) which actuates the Nominal Reset Servo.
      -- The servo sets the S/M and 3D cam assembly to nominal fuel flow.

      Solenoid.Set_Is_Energized (An_Instance => Next_Instance.Cam_Reset_Solenoid,
                                 State       => not Cam_Reset_Solenoid);
      Cam_Reset_Solenoid_State := Solenoid.Is_Energized
                                    (An_Instance => Next_Instance.Cam_Reset_Solenoid);

--      if Linkage_Failure then
--        if Delta_Pcl < This_Instance.Delta_Pcl_Linkage then
--          Next_Instance.Delta_Pcl_Linkage := This_Instance.Delta_Pcl_Linkage;
--        end if;
--      else
        Next_Instance.Delta_Pcl_Linkage := Delta_Pcl;
--      end if;

      New_Cam.Update
        (Next_Instance             => Next_Instance.Hydro_Mechanical_Cam,
         This_Instance             => This_Instance.Hydro_Mechanical_Cam,
         Iconst                    => Iconst,
         Power_Control_Lever_Angle => Next_Instance.Delta_Pcl_Linkage,
         Linkage_Failure           => Linkage_Failure,
         Cam_Reset                 => This_Instance.Fmu_Reset);

      Delta_Pla_Prime := New_Cam.Position
        (An_Instance => Next_Instance.Hydro_Mechanical_Cam);

      Delta_Pla := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => Float(Next_Instance.Delta_Pcl_Linkage),
        Table => PLAPCL_Table'Access );
      Next_Instance.Power_Control_Lever_Angle_Lvdt := Interpolation_Table.Singly_Indexed.Interpolate
      ( Input => Float(Delta_Pla_Prime),
        Table => PCLPLA_Table'Access );


      -- Is this solenoid mechanically linked to the pcl lever?  No
      Solenoid.Set_Is_Energized (An_Instance => Next_Instance.Shutdown_Solenoid,
                                 State       => (Energize_Shutdown_Solenoid and
                                                 Shutdown_Solenoid_Power) or
                                                 Delta_Pla < Next_Instance.Cutoff_Pla_Angle);
      Shutdown_Solenoid_State := Solenoid.Is_Energized
        (An_Instance => Next_Instance.Shutdown_Solenoid)
          and not This_Instance.Fmu_Reset;


      -- If the stepper motor loses electrical power, it will fail in
      -- a fixed position.  The Rate limit switch and reset servos
      -- reset the stepper motor to a moninal fuel schedule.  Power is
      -- then controlled mechanically by the pilot.

      if Cam_Reset_Solenoid_State or This_Instance.Fmu_Reset then
        -- This next if block models the rate limited reset servos.
        if This_Instance.Smp_Req < 169 then
          Smp_Req_Dot := 100;
        else
          Smp_Req_Dot := -100;
        end if;
        if This_Instance.Fmu_Reset then
          Next_Instance.Smp_Req := 169;
        else
          Next_Instance.Smp_Req := This_Instance.Smp_Req +  Integer(Float(Smp_Req_Dot)*Iconst);
        end if;
      else
        Next_Instance.Smp_Req := Stepper_Motor_Position_Requested;
      end if;

      New_Stepper_Motor.Update
        (Next_Instance       => Next_Instance.Fmu_Stepper_Motor,
         This_Instance       => This_Instance.Fmu_Stepper_Motor,
         Iconst              => Iconst,
         Position_Requested  => Next_Instance.Smp_Req );

      Smp := New_Stepper_Motor.Position
        (An_Instance => Next_Instance.Fmu_Stepper_Motor);

      Fuel_Metering_Valve.Update
        (Next_Instance                    => Next_Instance.Fmu_Metering_Valve,
         This_Instance                    => This_Instance.Fmu_Metering_Valve,
         Power_Lever_Angle                => Delta_Pla_Prime,
         Stepper_Motor_Position           => Smp,
         Shutdown_Solenoid_Energized      => Shutdown_Solenoid_State,
         Ambient_Pressure_Ratio           => Ambient_Pressure_Ratio,
         Return_Line_Pressure             => Return_Line_Pressure,
         Inlet_Pressure                   => Inlet_Pressure,
         Inlet_Flow_Rate                  => Mass_Types.Pph(0.0),
         Restrain_Fuel_Flow               => Restrain_Fuel_Flow,
         Fmu_Malfunction                  => Fmu_Malfunction);

      if This_Instance.Fmu_Reset then
        Next_Instance.Fmu_Reset := False;
      end if;

   end Update;

-- Method Initialize
   procedure Initialize
     (An_Instance : in out Instance) is
   begin
--|  Initialize the Auto Return Relay Attributes
     An_Instance.Cutoff_Pla_Angle := Angle_Types.Degrees(7.3);
     An_Instance.Power_Control_Lever_Angle_Lvdt := Angle_Types.Degrees(0.0);
     New_Stepper_Motor.Initialize
       (An_Instance => An_Instance.FMU_Stepper_Motor);
     New_Cam.Initialize
       (This_Instance => An_Instance.Hydro_Mechanical_Cam,
        Initial_Value => Angle_Types.Degrees(0.0));
     Fuel_Metering_Valve.Initialize
       (An_Instance => An_Instance.FMU_Metering_Valve);
    Stethoscope.Register_Signal
      (Name           => "Powerplant/Fmu/Power_Control_Lever_Angle_LVDT",
       Units          => "float",
       Object_Address => An_Instance.Power_Control_Lever_Angle_LVDT'Address,
       Value_Type     => "float",
       Scope_Index    => 0);
   end Initialize;

-- Method Read_Tables
   procedure Read_Tables is
     Powerplant_Cat_Path : String := Simulation_Dictionary.Lookup ("Powerplant_Cat_Path");
   begin
     New_Cam.Read_Tables;
     Fuel_Metering_Valve.Read_Tables;
     Interpolation_Table.Read
       ( File_Name => Powerplant_Cat_Path & "pclpla.ito",
         Table     => PCLPLA_Table );
     Interpolation_Table.Read
       ( File_Name => Powerplant_Cat_Path & "plapcl.ito",
         Table     => PLAPCL_Table );

   end Read_Tables;

end Fuel_Metering_Unit;



