-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Log;

package body Standby_Mag_Compass is

   Turning_Error    : Float := 0.0;
   Accel_Error      : Float := 0.0;
   Turb_Calc        : Float := 0.0;
   Turbulence       : Float := 0.0;
   Accel_Damp_Const : constant := 10.0;
   Vel_Damp_Const   : constant := 0.10;
   Demanded_Heading : Float := 0.0;
   Card_Accel       : Float := 0.0;
   Card_Velocity    : Float := 0.0;
   Message_Logged   : Boolean := False;

   procedure Set_Translational_Acceleration
     (An_Instance                : in out Instance;
      Translational_Acceleration : in     Coordinate_Types.Cartesian) is
   begin
      An_Instance.Translational_Acceleration := Translational_Acceleration;
   end Set_Translational_Acceleration;

   procedure Set_Aircraft_Position
     (An_Instance       : in out Instance;
      Aircraft_Position : in     It.Position_Type) is
   begin
      An_Instance.Aircraft_Position := Aircraft_Position;
   end;

   procedure Set_Aircraft_Attitude
     (An_Instance       : in out Instance;
      Aircraft_Attitude : in     It.Attitude_Type) is
   begin
      An_Instance.Aircraft_Attitude := Aircraft_Attitude;
   end;

   procedure Set_Body_Axis_Velocity
     (An_Instance        : in out Instance;
      Body_Axis_Velocity : in     Coordinate_Types.Cartesian) is
   begin
      An_Instance.Body_Axis_Velocity := Body_Axis_Velocity;
   end Set_Body_Axis_Velocity;

   procedure Set_Turbulence_Level
     (An_Instance      : in out Instance;
      Turbulence_Level : in     Float) is
   begin
      An_Instance.Turbulence_Level := Turbulence_Level;
   end Set_Turbulence_Level;

   procedure Set_Weight_On_Wheels
     (An_Instance      : in out Instance;
      Weight_On_Wheels : in     Boolean) is
   begin
      An_Instance.Weight_On_Wheels :=Weight_On_Wheels;
   end Set_Weight_On_Wheels;

   procedure Set_Magnetic_Heading
     (An_Instance      : in out Instance;
      Magnetic_Heading : in     It.Heading_Type) is
   begin
      An_Instance.Magnetic_Heading :=Magnetic_Heading;
   end;

   procedure Set_Compass_Heading
     (An_Instance     : in out Instance;
      Compass_Heading : in     It.Heading_Type) is
   begin
      An_Instance.Compass_Heading := Compass_Heading;
   end;

   function Aircraft_Position
     (An_Instance : in Instance)
      return It.Position_Type is
   begin
      return An_Instance.Aircraft_Position;
   end;

   function Translational_Acceleration
     (An_Instance : in Instance)
      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.Translational_Acceleration;
   end;

   function Aircraft_Attitude
     (An_Instance : in Instance)
      return It.Attitude_Type is
   begin
      return An_Instance.Aircraft_Attitude;
   end;

   function Body_Axis_Velocity
     (An_Instance : in Instance)
      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.Body_Axis_Velocity;
   end;

   function Turbulence_Level
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Turbulence_Level;
   end;

   function Weight_On_Wheels
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Weight_On_Wheels;
   end;

   function Magnetic_Heading
     (An_Instance : in Instance)
      return It.Heading_Type is
   begin
      return An_Instance.Magnetic_Heading;
   end;

-- Access Compass_Heading
   function  Compass_Heading
     (An_Instance : in Instance)
      return It.Heading_Type is
   begin
      return An_Instance.Compass_Heading;
   end Compass_Heading;

   -- Method Init
   procedure Init
     (An_Instance : in out Instance) is

   begin

      An_Instance.Translational_Acceleration := (0.0,0.0,0.0);
      An_Instance.Aircraft_Attitude          := (0.0,0.0,0.0);
      An_Instance.Body_Axis_Velocity         := (0.0,0.0,0.0);
      An_Instance.Turbulence_Level           := 0.0;
      An_Instance.Weight_On_Wheels           := False;
      An_Instance.Magnetic_Heading           := 0.0;
      An_Instance.Compass_Heading            := 0.0;

   end Init;

-- Method Update
   procedure Update(Integration_Constant : in     Float;
                    An_Instance          : in out Instance) is

   begin

--  This module emulates the liquid filled standby compass by
--  driving a servo driven replica of the compass.  Abnormal
--  operation such as northerly turning errors, acceleration
--  errors, and turbulence effects.
--
--*********************************************************************
--# NORTHERLY TURNING ERRORS
--*********************************************************************
--
-- A turn from the north will produce an indication of a
-- turn to the north.  ( COMPASS LAG )
--
-- A turn from the south will produce an indication of a
-- turn from the south. (COMPASS LEAD).
--
-- The aircraft roll angle is used to approximate the
-- turn rates.  The effects are reversed for the southern
-- hemisphere.  At a bank angle of 20 degrees the error is
-- HALF  the latitude.  The cosine of magnetic heading is
-- used to fade the effects in and out according to the
-- indicated heading.
--
--  ERROR = A/C ROLL / 20.0 * A/C LATITUDE / 2 * COS( A/C HEADING )
--
      Turning_Error :=
        Ru.Flimit(An_Instance.Aircraft_Attitude.Roll,-30.0,30.0) *
        Cos(An_Instance.Magnetic_Heading,360.0)                  *
        Float(An_Instance.Aircraft_Position.Latitude)            *
        (-0.025);

--*********************************************************************
--# A/C ACCELERATION ERRORS
--*********************************************************************
--
-- An acceleration produces an indicated turn to the north
-- if on an easterly / westerly heading, and a zero error on
-- northerly / southerly heading.  The sine term is used to
-- fade the effects in or out at the desired headings.
--
--  0.600 =  20 DEG / 32 FT/SEC**2    (32 FT/SEC**2 = 1.0 G )
--
-- The ground velocity (fug) was included to make sure that the
-- acceleration error isn't included when the aircraft isn't
-- moving.  The flight acceleration label (fltaxa) is written
-- to even when the a/c is stationary with the brakes on.
--
      An_Instance.Body_Axis_Velocity.X :=
        Ru.Flimit(An_Instance.Body_Axis_Velocity.X,0.0,1.0);

      Accel_Error := An_Instance.Translational_Acceleration.X *
        (-0.6) * Sin(An_Instance.Magnetic_Heading,360.0)      *
        An_Instance.Body_Axis_Velocity.X;

--*********************************************************************
--# A/C TURBULENCE EFFECTS
--*********************************************************************
--
-- Turbulence errors are induced to swing the compass whenever the
-- A/C experiences turbulence effects.  A simple sine wave function
-- is used to produce this effect.  The turbulence level is used as
-- a gain to increase or decrease the amplitude of the error.
--
      if An_Instance.Turbulence_Level /= 0.0 and not
        An_Instance.Weight_On_Wheels then
         Turb_Calc := Turb_Calc + ( 60.0 * Integration_Constant);
         if Turb_Calc > 360.0 then
            Turb_Calc := Turb_Calc -360.0;
         end if;
         Turbulence := Sin(Turb_Calc,360.0) *
           An_Instance.Turbulence_Level * 0.3;
      else
         Turbulence := 0.0;
      end if;

--*********************************************************************
--# INDICATED MAGNETIC HEADING
--*********************************************************************
--
-- The compass model contains a dynamic model to provide
-- compass card overshoot and smoothing effects when windshield
-- heat is applied.  The acceleration constant and velocity
-- constants may be adjusted as required for the desired
-- effect.  NOTE: Be careful with these constants to prevent
-- the card from becoming unstable.  If the module rate is
-- changed for any reason, these terms may need to be adjusted
--
-- Note, the constant in NCACC may be adjusted for the compass
-- rate.  The constant in NVEL may be adjusted for the damping
-- rate.
--

      Demanded_Heading := Ru.Xn180(An_Instance.Magnetic_Heading
                                   + Accel_Error + Turning_Error
                                   + Turbulence);

-- The acceleration force on the card is a result of the compass
-- not being properly aligned with the magnetic field.  The larger
-- the difference between the magnetic field and the compass, the
-- larger the rotational force acting on the compass.  Since the
-- force is not literally the difference in position, an acceleration
-- damping constant is multiplied in.
-- Drag (mechanical and fluid) is proportional to the sqare of
-- the card rotational velocity.  The card velocity multiplied by
-- the absolute value of the card velocity gives a sqared value of
-- the velocity, and maintains the sign convention for velocity
-- direction.

      Card_Accel := Accel_Damp_Const
        * Ru.Xn180(Demanded_Heading - An_Instance.Compass_Heading)
        - ( Card_Velocity * abs(Card_Velocity) * Vel_Damp_Const);

-- The card velocity is the original velocity plus the forces
-- acting on the card multiplied by the time that force acted over.
-- The forces acting on the card are the acceleration produced
-- by the card not being aligned with the magnetic field, and drag.

      Card_Velocity := Card_Velocity
        + ( Card_Accel * Integration_Constant);

-- The new compass heading is found by taking the current compass heading,
-- and adding to it the velocity of the compass card multiplied by
-- the time that the compass card was rotating at that velocity.

      An_Instance.Compass_Heading :=
        Ru.Xn180(An_Instance.Compass_Heading
                 + (Card_Velocity * Integration_Constant));

   end Update;


end Standby_Mag_Compass;


