-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Radio_Utilities,
  Ada.Numerics.Float_Random;
with Ada.Text_Io,Log;
with Ada.Numerics.Elementary_Functions;

use Ada.Text_IO,Log;
use Ada.Numerics.Float_Random;
use Ada.Numerics.Elementary_Functions;

package body Standby_Attitude_Indicator is

   package Ru renames Radio_Utilities;

   Roll : Float := 0.0;
   Pitch : Float := 0.0;
   Bounce : Float := 0.0;
   Bounce_Magnitude : Float := 0.0;
   Bounce_Timer : Float := 0.0;
   Hold_Timer : Float := 0.0;
   Spin_Timer : Float := 0.0;
   Gyro_Stable_Time : constant := 10.0;
   Minutes : constant := 60.0;
   Seconds : constant := 1.0;
   Engine_Quick_Start : Boolean := False; --Local for now
   Ahrs_Fast_Align : Boolean := False;    --Local for now
   System_Reset : Boolean := False;       --Local for now
   Roll_Erect : Float := 0.0;
   Pitch_Erect : Float := 0.0;
   Pitch_Indicated : Float := -40.0;
   Roll_Indicated : Float  := -40.0;
   Tumble_Timer   : Float := 0.0;
   Gen : Ada.Numerics.Float_Random.Generator;
   Pitch_Delta : Float := 0.0;
   Roll_Delta : Float := 0.0;
   Pitch_N1 : Float := 0.0;
   Roll_N1 : Float := 0.0;
   Cage_N1 : Boolean := False;
   Test_Timer : Float := 0.0;

   procedure Set_Power
     (An_Instance : in out Instance;
      Power       : in     Boolean) is
   begin
      An_Instance.Power := Power;
   end Set_Power;

   procedure Set_Cage
     (An_Instance : in out Instance;
      Cage        : in     Boolean) is
   begin
      An_Instance.Cage := Cage;
   end Set_Cage;

   procedure Set_Off_Flag
     (An_Instance : in out Instance;
      Off_Flag    : in     Boolean) is
   begin
      An_Instance.Off_Flag := Off_Flag;
   end;

   procedure Set_Aircraft_Attitude
     (An_Instance       : in out Instance;
      Aircraft_Attitude : in     Nav_Instruments_Types.Attitude_Type) is
   begin
      An_Instance.Aircraft_Attitude := Aircraft_Attitude;
   end Set_Aircraft_Attitude;

   procedure Set_Indicated_Attitude
     (An_Instance        : in out Instance;
      Indicated_Attitude : in     Nav_Instruments_Types.Attitude_Type) is
   begin
      An_Instance.Indicated_Attitude := Indicated_Attitude;
   end;

   function Power
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Power;
   end;

   function Cage
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Cage;
   end;

   function Off_Flag
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Off_Flag;
   end;

   function Aircraft_Attitude
     (An_Instance : in Instance)
      return Nav_Instruments_Types.Attitude_Type is
   begin
      return An_Instance.Aircraft_Attitude;
   end;

   -- Access Indicated_Attitude
   function  Indicated_Attitude
     (An_Instance : in Instance)
      return Nav_Instruments_Types.Attitude_Type is
   begin
      return An_Instance.Indicated_Attitude;
   end Indicated_Attitude;

   -- Method Init-------------------------------------------------------------
   procedure Init
     (An_Instance : in out Instance) is

   begin

      -- initialize adi
      Hold_Timer := 0.0 * Minutes;
      Spin_Timer := 0.0 * Minutes;
      Tumble_Timer := 0.0 * Minutes;
      Pitch_Indicated := -69.0;
      Roll_Indicated  := -90.0;

      -- Initialize instance components:
      An_Instance.Power              := False;
      An_Instance.Cage               := False;
      An_Instance.Off_Flag           := False;
      An_Instance.Aircraft_Attitude  := (0.0,0.0,0.0);
      An_Instance.Indicated_Attitude := (0.0,0.0,0.0);

   end Init;
------------------------------------------------------------------------------

   -- Method Update
   procedure Update(Iconst      : in     Float;
                    An_Instance : in out Instance) is


   begin

--  This module simulates the standby attitude indicator,
--  initially showing right wing down and nose down indications.
--  Upon application of power, the standby horizon will erect
--  in 2 minutes to the proper indication (not necessarily wings
--  level).  Caging the standby horizon will decrease the time
--  to proper indication.  Upon loss of power the standby horizon
--  will remain operational for 9 minutes, then slowly tumble to
--  a spun down position.  If spun down, then caged and released, the
--  standby horizon will fall to left wing down and nose down with a
--  significant bounce off the stops in the pitch axis.
--
--
--*********************************************************************
--# FLIGHT ROLL AND PITCH
--*********************************************************************
--
-- Calculate actual aircraft roll and pitch from terms derived from
-- flight software, and limit those terms to 180 degrees.
--
      Pitch := Ru.Xn180(An_Instance.Aircraft_Attitude.Pitch);
      Roll  := Ru.Xn180(An_Instance.Aircraft_Attitude.Roll);

--*********************************************************************
--# SPIN UP
--*********************************************************************
--
-- This section is only executed when the standby horizon has power.
-- When power is applied to a gyro hold timer and a spin
-- timer are incremented.  The purpose of the hold timer is to keep
-- the gyro holding its simulated spin for 9 minutes when power is
-- lost. The spin timer keeps track of the 10 seconds spin-up time.
-- Whe increment section of the spin timer will generate the bounce
-- function and gyro erection terms.
--
      if An_Instance.Power then
         --reset bounce term
         Bounce := 0.0;

         if (Engine_Quick_Start or Ahrs_Fast_Align) then
            Hold_Timer := 3.0 * Minutes;
            Spin_Timer := 8.0 * Minutes;
            Pitch_Indicated := Pitch;
            Roll_Indicated  := Roll;
            Bounce_Magnitude := 0.0;
            Bounce := 0.0;
         end if;

         --run up gyro-hold timer
         if Hold_Timer >= 1.0 * Minutes then
            Hold_Timer := 3.0 * Minutes;
         else
            Hold_Timer := Hold_Timer + Iconst;
         end if;

         --run up gyro-spin timer
         if Spin_Timer >= 1.0 * Minutes then
            Spin_Timer := 8.0 * Minutes;
         --if tumble tmr active let
         elsif Tumble_Timer > 0.0 then
            --gyro fall.
            Tumble_Timer := Tumble_Timer - Iconst;
            --spin up timer
            Spin_Timer := Spin_Timer + Iconst;
         else
            --Normal spin up branch
            Spin_Timer := Spin_Timer + Iconst;

            Roll_Erect := Ru.Flimit(Roll - Roll_Indicated,
                                    (-90.0/Minutes)*Iconst,
                                    ( 90.0/Minutes)*Iconst);
            Pitch_Erect := Ru.Flimit(Pitch - Pitch_Indicated,
                                     (-70.0/Minutes)*Iconst,
                                      (70.0/Minutes)*Iconst);

         end if;

      else

--*********************************************************************
--# SPIN DOWN
--*********************************************************************
--
-- The spin down cycle lasts a total of 11 minutes.  The first 9
-- minutes the hold timer is decremented.  When the gyro hold timer
-- runs down the gyro starts to topple.  It will take 2 minutes to
-- topple completely.  If the gyro is caged during gyro toppled or
-- after the gyro has toppled the indicator will go to wings and
-- pitch level then fall hard to the previously toppled position.
-- This section is only executed if the standby horizon has no power.
--
         if System_Reset then  --simply used for debug and test guide
            Hold_Timer := 0.0; --if not powered and the system
            Spin_Timer := 0.0; --reset is selected, then spin the
         end if;               --gyro down completely.

         if Hold_Timer <= 0.0 then --hold timer expired ?
              Hold_Timer := 0.0;     --hold at zero

              if Spin_Timer <= 0.0 then --gyro spun down ?
                   Spin_Timer := 0.0;     --hold spun down

                   if Tumble_Timer > 0.0 then --decrement tumble
                        Tumble_Timer := Tumble_Timer - Iconst;
                   else
                      if Pitch_Indicated >= -69.0 then --all the way down?
                         --not yet -easy fall over
                         Pitch_Erect := -5.0 * Iconst;
                      else
                         Pitch_Erect := 0.0;
                      end if;

                      if Roll_Indicated > -90.0 then
                         Roll_Erect := -5.0 * Iconst;
                      else
                         Roll_Erect := 0.0;
                      end if;

                   end if;

              else
               --decrease gyro spin
               Spin_Timer := Spin_Timer - Iconst;

               --decrement tumble
               if Tumble_Timer > 0.0 then
                  Tumble_Timer := Tumble_Timer - Iconst;
               else
                  if Pitch_Indicated >= -69.0 then
                     Pitch_Erect := -0.75/Minutes * Iconst;
                  else
                     Pitch_Erect := 0.0;
                  end if;

                  if Roll_Indicated >= -90.0 then
                     Roll_Erect := -0.75/Minutes * Iconst;
                  else
                     Roll_Erect := 0.0;
                  end if;

--                  Bounce_Magnitude :=
--                    20.0 * ((8.0*Minutes - Spin_Timer)/8.0*Minutes);
--
--                  Bounce_Timer := Bounce_Timer + (Iconst*360.0);
--                  if Bounce_Timer >= 360.0 then Bounce_Timer := 0.0; end if;
--
--                  Bounce := Bounce_Magnitude * Sin(Bounce_Timer,360.0);

               end if;

              end if;

         else
            Hold_Timer  := Hold_Timer - Iconst;
            Pitch_Erect := 0.0;
            Roll_Erect  := 0.0;
            Bounce_Magnitude := 0.0;
            Bounce := 0.0;
         end if;
      end if;

--*********************************************************************
--# FLIGHT DELTA PITCH AND ROLL
--*********************************************************************
--
-- Calculate the delta terms for roll angle and pitch angle of the
-- aircraft.  The roll and pitch delta terms are dampened during spin
-- up or spin down cycles by multiplying in the spin timer.
--
      Roll_Delta := Ru.Xn180(Roll - Roll_N1) * (Spin_Timer/(8.0*Minutes));

      Pitch_Delta := Ru.Xn180(Pitch - Pitch_N1) * (Spin_Timer/(8.0*Minutes));

      Pitch_N1 := Pitch; --update pitch (n-1)
      Roll_N1  := Roll;  --update roll  (n-1)

--*********************************************************************
--# FLIGHT ALIGNMENTS
--*********************************************************************
--
-- This section integrates a roll precession term into the aircraft
-- roll.  The pitch axis of the gyro has no precession.  The precess-
-- ion term is small and is only factored in when the simulated gyro
-- is fully spun up.  The erection rate for the pitch and roll axes
-- is 30 degrees / 4 minutes or 7.5 degrees / 1 minute.
--
      --gyro spun up ??
      if (Spin_Timer >= 8.0*Minutes and An_Instance.Power) then
         Roll_Erect := Ru.Xn180( Roll * 0.2 - Roll_Indicated);
         Roll_Erect := Ru.Flimit(Roll_Erect,-0.001*Iconst,0.001*Iconst);
         Pitch_Erect := 0.0;
      end if;

--*********************************************************************
--# CAGING
--*********************************************************************
--
-- This section forces the indicator to wings and pitch level.  When
-- the gyro has less than 1 minute of spin a large precession term
-- (a negative erection) is set to make the gyro fall over after the
-- cage.  Bounce magnitude after fall is also set here.
--
      --if gyro caged & spun-up
      if An_Instance.Cage
        and Spin_Timer > Gyro_Stable_Time then

         Pitch_Indicated := 0.0; --yes, set pitch to zero
         Roll_Indicated  := 0.0; --set roll to zero
         Pitch_Erect     := 0.0; --no pitch erection
         Roll_Erect      := 0.0; --no roll erection
         Bounce          := 0.0; --no bounce
         Bounce_Timer    := 5.0/Iconst; --set bounce tmr to max

      --else if caged & not spun-up
      elsif An_Instance.Cage and Spin_Timer <= Gyro_Stable_Time then
         --bounce in pitch.
         Bounce_Magnitude := 20.0;
         Bounce_Timer := 0.0;
         Bounce := 0.0;
         --set tumble timer
         Tumble_Timer :=
           ((Gyro_Stable_Time - Spin_Timer) / Gyro_Stable_Time);
         --big pitch erection
         Pitch_Erect := -70.0 * Iconst;
         --big roll erection
         Roll_Erect := -90.0 * Iconst;
         Pitch_Indicated := 0.0; --set pitch to zero
         Roll_Indicated := 0.0;  --set roll to zero

      elsif Bounce_Timer >= 5.0/Iconst then
         Bounce_Timer := 5.0/Iconst;
      end if;

--*********************************************************************
--# FINAL PITCH AND ROLL COMPUTATIONS
--*********************************************************************
--
-- The final indicated pitch and roll are factored here by summing
-- the previous indicated value, the delta term, and the erection
-- term.
--
      if not An_Instance.Cage then
         Pitch_Indicated :=
           Ru.Xn180(Pitch_Indicated + Pitch_Delta + Pitch_Erect);
         Roll_Indicated :=
           Ru.Xn180(Roll_Indicated + Roll_Delta + Roll_Erect);
      end if;

--*********************************************************************
--# OUTPUT SECTION
--*********************************************************************
--
      An_Instance.Off_Flag := An_Instance.Power and not An_Instance.Cage;

      An_Instance.Indicated_Attitude.Pitch :=
        Ru.Flimit(Pitch_Indicated + Bounce, (-69.0), 70.0);

      An_Instance.Indicated_Attitude.Roll := Roll_Indicated;

--*********************************************************************
--# N-1 LABELS
--*********************************************************************
--
      Cage_N1 := An_Instance.Cage;

   end Update;


end Standby_Attitude_Indicator;
