-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-- DATA SOURCE:  USAF TO 1T-6A-1
--               Raytheon Wiring Diagrams-Avionics and Electrical Systems
--               Flight Safety Simulation Models
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions;
use ada.Numerics.Elementary_Functions;

with JPATS_Nav_Instruments.Container,
  Nav_Instruments_Types,
  Standby_Attitude_Indicator,
  Standby_Mag_Compass,
  Jpats_Electrical_Types,
  Jpats_Electrical,
  Jpats_Landing_Gear_Types,
  Jpats_Landing_Gear,
  Jpats_Simulated_Aircraft,
  Jpats_Atmosphere,
  Jpats_Radio_Db_If;

with Jpats_Io_Types, Jpats_Io, Io_Types;

package body JPATS_Nav_Instruments.Controller is

   package Att renames Standby_Attitude_Indicator;
   package Mag renames Standby_Mag_Compass;
   Io : Container.Io_Interface_Instance renames Container.This_Io_Interface;
   package Gear renames Jpats_Landing_Gear;
   package Gear_Types renames Jpats_Landing_Gear_Types;
   package Jiot renames Jpats_Io_Types;
   package Iot renames Io_Types;

   Attitude : Nav_Instruments_Types.Attitude_Type := (0.0,0.0,0.0);
   Display_Attitude : Nav_Instruments_Types.Attitude_Type := (0.0,0.0,0.0);
   Display_Heading : Nav_Instruments_Types.Heading_Type := 0.0;
   Position : Nav_Instruments_Types.Position_Type := (45.0,-45.0,0.0);
   Magnetic_Heading : Nav_Instruments_Types.Heading_Type := 0.0;

   procedure Register_Io_Variables is
   begin

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "STBY_COMPASS_COSINE",
         A_Direction     => Iot.Output,
         An_Address      => Io.Stby_Compass_Cos'Address,
         A_Size          => Io.Stby_Compass_Cos'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "STBY_COMPASS_SINE",
         A_Direction     => Iot.Output,
         An_Address      => Io.Stby_Compass_Sin'Address,
         A_Size          => Io.Stby_Compass_Sin'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "STBY_HORIZON_SINE_ROLL",
         A_Direction     => Iot.Output,
         An_Address      => Io.Stby_Horizon_Roll_Sin'Address,
         A_Size          => Io.Stby_Horizon_Roll_Sin'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "STBY_HORIZON_COSINE_ROLL",
         A_Direction     => Iot.Output,
         An_Address      => Io.Stby_Horizon_Roll_Cos'Address,
         A_Size          => Io.Stby_Horizon_Roll_Cos'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "STBY_HORIZON_PITCH",
         A_Direction     => Iot.Output,
         An_Address      => Io.Stby_Horizon_Pitch'Address,
         A_Size          => Io.Stby_Horizon_Pitch'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "STBY_HORIZON_PULL_TO_CAGE",
         A_Direction     => Iot.Input,
         An_Address      => Io.Stby_Horizon_Cage'Address,
         A_Size          => Io.Stby_Horizon_Cage'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "STBY_HORIZON_OFF_FLAG",
         A_Direction     => Iot.Output,
         An_Address      => Io.Stby_Horizon_Flag'Address,
         A_Size          => Io.Stby_Horizon_Flag'Size);
   end;

   procedure Initialize is
   begin

      --register io variables:
      Register_Io_Variables;

      --Initialize standby attitude indicator attributes:
      Standby_Attitude_Indicator.Init(Container.This_Subsystem.The_Stby_Att);

      --Initialize standby mag compass attributes:
      Standby_Mag_Compass.Init(Container.This_Subsystem.The_Stby_Mag);

   end Initialize;

   procedure Update
     (Integration_Constant : in Float) is
   begin

      --common inputs from other categories
      Attitude.Pitch := JPATS_Simulated_Aircraft.Get_Pitch_Angle * 57.3;
      Attitude.Roll  := JPATS_Simulated_Aircraft.Get_Roll_Angle * 57.3;


--Standby Attitude Indicator section---------------------------------------

      --Inputs from other categories
      Att.Set_Power(Container.This_Subsystem.The_Stby_Att,
         JPATS_Electrical.Is_Powered(Jpats_Electrical_Types.Gyro_CB));
      Att.Set_Cage(Container.This_Subsystem.The_Stby_Att,
                   Io.Stby_Horizon_Cage);
      Att.Set_Aircraft_Attitude(Container.This_Subsystem.The_Stby_Att,
                                Attitude);


      Att.Update(Integration_Constant,Container.This_Subsystem.The_Stby_Att);

      --Outputs from Standby Attitude Indicator
      Display_Attitude :=
        Att.Indicated_Attitude(Container.This_Subsystem.The_Stby_Att);
      Io.Stby_Horizon_Roll_Sin := -sin(Display_Attitude.Roll,360.0) * 10.0;
      Io.Stby_Horizon_Roll_Cos := Cos(Display_Attitude.Roll,360.0) * 10.0;
      Io.Stby_Horizon_Pitch    := Display_Attitude.Pitch / 7.0;

      Io.Stby_Horizon_Flag :=
        Att.Off_Flag(Container.This_Subsystem.The_Stby_Att);

--Standby Magnetic Compass section------------------------------------------

      --Inputs from other categories
      Position.Latitude := JPATS_Simulated_Aircraft.Get_North;
      Position.Longitude := JPATS_Simulated_Aircraft.Get_East;


      Mag.Set_Aircraft_Position
        (Container.This_Subsystem.The_Stby_Mag,Position);
      Mag.Set_Translational_Acceleration
        (Container.This_Subsystem.The_Stby_Mag,
         JPATS_Simulated_Aircraft.Get_Acceleration_Body_Axis);
      Mag.Set_Aircraft_Attitude
        (Container.This_Subsystem.The_Stby_Mag, Attitude);
      Mag.Set_Body_Axis_Velocity
        (Container.This_Subsystem.The_Stby_Mag,
         JPATS_Simulated_Aircraft.Get_V_B);
      Mag.Set_Turbulence_Level
        (Container.This_Subsystem.The_Stby_Mag,
         Jpats_atmosphere.Get_Total_Turbulence_Intensity);
      Mag.Set_Weight_On_Wheels(Container.This_Subsystem.The_Stby_Mag,
        Gear.Is_Closed(Gear_Types.Left_Gear_Wow_Switch));

      Magnetic_Heading := Jpats_Simulated_Aircraft.Get_Hdg_Angle * 57.3 -
        Jpats_Radio_Db_If.Magnetic_Variation;

      Mag.Set_Magnetic_Heading
        (Container.This_Subsystem.The_Stby_Mag,Magnetic_Heading);

      Mag.Update(Integration_Constant,Container.This_Subsystem.The_Stby_Mag);

      Display_Heading := Mag.Compass_Heading(
                         Container.This_Subsystem.The_Stby_Mag);

      Io.Stby_Compass_Sin := Sin(Display_Heading,360.0) * 10.0;
      Io.Stby_Compass_Cos := Cos(Display_Heading,360.0) * 10.0;

   end Update;

end JPATS_Nav_Instruments.Controller;
