-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Interpolation_Table.Singly_Indexed;
with Simulation_Dictionary;
with Ada.Numerics.Elementary_Functions; use Ada.Numerics.Elementary_Functions;
with Log;

package body Landing_Gear_G is

   Gear_Up_Rate   : aliased Interpolation_Table.Singly_Indexed.Instance;
   Gear_Down_Rate : aliased Interpolation_Table.Singly_Indexed.Instance;

   procedure Extend
     (An_Instance       : in out Instance;
      Iconst            : in     Float;
      Hyd_Fcn           : in     Hyd_Fcn_Type)
   is
      package Single renames Interpolation_Table.Singly_Indexed;
      Position      : Gear_Position_Type renames An_Instance.Position;
      Last_Position : Gear_Position_Type renames An_Instance.Last_Position;
      Hyd_Load      : Hyd_Load_Type      renames An_Instance.Hyd_Load;
   begin
      Last_Position := Position;
      Position      := Gear_Position_Type'Min(Gear_Position_Type'Last,
                                              Position +
                                              Gear_Position_Type(Hyd_Fcn) *
                                              Gear_Position_Type(Iconst) *
                                              Gear_Position_Type(Single.Interpolate (Float(Position), Gear_Down_Rate'Access)));
--      Hyd_Load      := Hyd_Load_Type(Hyd_Load_Type((1.639/Iconst)**0.33))*Hyd_Load_Type(Position-Last_Position);

      if Position = Gear_Position_Type'Last then
         Hyd_Load := 0.0;
      else
         Hyd_Load := 0.217 * Hyd_Load_Type(Hyd_Load_Type(1.639/Iconst)*Hyd_Load_Type(Position-Last_Position));
      end if;
   exception
      when others =>
         Log.Report(Package_Name & ".Extend()");
         raise;
   end Extend;

   procedure Retract
     (An_Instance       : in out Instance;
      Iconst            : in     Float;
      Hyd_Fcn           : in     Hyd_Fcn_Type)
   is
      package Single renames Interpolation_Table.Singly_Indexed;
      Position      : Gear_Position_Type renames An_Instance.Position;
      Last_Position : Gear_Position_Type renames An_Instance.Last_Position;
      Hyd_Load      : Hyd_Load_Type      renames An_Instance.Hyd_Load;
   begin
      Last_Position := Position;
      Position      := Gear_Position_Type'Max(Gear_Position_Type'First,
                                              Position -
                                              Gear_Position_Type(Hyd_Fcn) *
                                              Gear_Position_Type(Iconst) *
                                              Gear_Position_Type(Single.Interpolate (Float(Position), Gear_Up_Rate'Access)));
      if Position = Gear_Position_Type'First then
         Hyd_Load := 0.0;
      else
         Hyd_Load := 1.5*Hyd_Load_Type(-0.847/Iconst)*Hyd_Load_Type(Position-Last_Position);
      end if;
   exception
      when others =>
         Log.Report(Package_Name & ".Retract()");
         raise;
   end Retract;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Position      := Gear_Position_Type'Last;
      An_Instance.Last_Position := 0.0;
      An_Instance.Hyd_Load      := 0.0;
   exception
      when others =>
         Log.Report(Package_Name & ".Initialize()");
         raise;
   end Initialize;

   function Position
     (An_Instance : in Instance)
     return Gear_Position_Type
   is
   begin
      return An_Instance.Position;
   exception
      when others =>
         Log.Report(Package_Name & ".Position()");
         raise;
   end Position;

   function Up_Locked
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Position = Gear_Position_Type'First;
   exception
      when others =>
         Log.Report(Package_Name & ".Up_Locked()");
         raise;
   end Up_Locked;

   function Down_Locked
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Position = Gear_Position_Type'Last;
   exception
      when others =>
         Log.Report(Package_Name & ".Down_Locked()");
         raise;
   end Down_Locked;

   function In_Transit
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return not Up_Locked (An_Instance) and not Down_Locked (An_Instance);
   exception
      when others =>
         Log.Report(Package_Name & ".In_Transit()");
         raise;
   end In_Transit;

   function Hydraulic_Load
     (An_Instance : in Instance)
     return Hyd_Load_Type
   is
   begin
      return An_Instance.Hyd_Load;
   exception
      when others =>
         Log.Report(Package_Name & ".Hydraulic_Load()");
         raise;
   end Hydraulic_Load;

   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Landing_Gear_Dir") & Gear_Up_Table & ".ito",
         Table     => Gear_Up_Rate);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Landing_Gear_Dir") & Gear_Down_Table & ".ito",
         Table     => Gear_Down_Rate);
   exception
      when others =>
         Log.Report(Package_Name & ".Read_Tables()");
         raise;
   end Read_Tables;

end Landing_Gear_G;
