-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Numerics.Elementary_Functions; use Ada.Numerics.Elementary_Functions;
with Log;

package body Landing_Gear_Door_G is

   procedure Extend
     (An_Instance       : in out Instance;
      Iconst            : in     Float;
      Hyd_Fcn           : in     Hyd_Fcn_Type)
   is
      Position      : Position_Type renames An_Instance.Position;
      Last_Position : Position_Type renames An_Instance.Last_Position;
   begin
      Last_Position := Position;
      Position := (Position_Type'Min(Position_Type'Last,
                                     Position + Position_Type(Hyd_Fcn) *
                                     Position_Type(0.75) * Position_Type(Iconst)));

      if Position = Position_Type'Last then
         An_Instance.Hyd_Load := 0.0;
      else
         An_Instance.Hyd_Load := 1.5*(Hyd_Load_Type ((0.943/Iconst)) *
                                      Hyd_Load_Type(Position-Last_Position));
      end if;
   exception
      when others =>
         Log.Report(Package_Name & ".Extend()");
         raise;
   end Extend;

   procedure Retract
     (An_Instance       : in out Instance;
      Iconst            : in     Float;
      Hyd_Fcn           : in     Hyd_Fcn_Type)
   is
      Position      : Position_Type renames An_Instance.Position;
      Last_Position : Position_Type renames An_Instance.Last_Position;
   begin
      Last_Position := Position;
      Position := Position_Type'Max(Position_Type'First, Position - Position_Type(Hyd_Fcn) *
                                    Position_Type(0.75) * Position_Type(Iconst));
      if Position = Position_Type'First then
         An_Instance.Hyd_Load := 0.0;
      else
         An_Instance.Hyd_Load := 1.5*(Hyd_Load_Type ((-0.930/Iconst)) *
                                      Hyd_Load_Type(Position-Last_Position));
      end if;
   exception
      when others =>
         Log.Report(Package_Name & ".Retract()");
         raise;
   end Retract;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Position      := 0.0;
      An_Instance.Last_Position := 0.0;
      An_Instance.Hyd_Load      := 0.0;
   exception
      when others =>
         Log.Report(Package_Name & ".Initialize()");
         raise;
   end Initialize;

   function Position
     (An_Instance : in Instance)
     return Position_Type
   is
   begin
      return An_Instance.Position;
   exception
      when others =>
         Log.Report(Package_Name & ".Position()");
         raise;
   end Position;

   function Up_Locked
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Position = Position_Type'First;
   exception
      when others =>
         Log.Report(Package_Name & ".Up_Locked()");
         raise;
   end Up_Locked;

   function Open
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Position = Position_Type'Last;
   exception
      when others =>
         Log.Report(Package_Name & ".Open()");
         raise;
   end Open;

   function In_Transit
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return not Up_Locked (An_Instance) and not Open (An_Instance);
   exception
      when others =>
         Log.Report(Package_Name & ".In_Transit()");
         raise;
   end In_Transit;

   function Hydraulic_Load
     (An_Instance : in Instance)
     return Hyd_Load_Type
   is
   begin
      return An_Instance.Hyd_Load;
   exception
      when others =>
         Log.Report(Package_Name & ".Hydraulic_Load()");
         raise;
   end Hydraulic_Load;

end Landing_Gear_Door_G;

