-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Landing_Gear.Container;
with Jpats_Electrical;
with Jpats_Hydraulics;
with Jpats_Auto_Test;
with Jpats_Aircraft_Body;
with Jpats_Secondary_Flight_Controls;
with Jpats_Electrical_Types;
with Jpats_Hydraulics_Types;
with Jpats_Landing_Gear_Types;
with Jpats_Lights;
with Jpats_Flight_Instruments;
with Length_Types;
with Jpats_Secondary_Flight_Controls_Types;
with Jpats_Engine_Data_Manager;
with Jpats_Dcls;
with JPATS_Reposition;
with Log;
with Jpats_Powerplant;
with JPATS_Simulated_Aircraft;
with Jpats_IOS_Pilot;

package body Jpats_Landing_Gear.Controller is

   Left : Container.MLG renames Container.Left;
   Right : Container.MLG renames Container.Right;
   G : Container.Random_MLG.Generator;-- renames Container.This_Subsystem.LGC_G;
   Random_MLG_Generator_State : Container.Random_MLG.State renames Container.This_Subsystem.LGC_Random_MLG_Generator_State;

   procedure Reset_Random_MLG_Generator
   is
   begin
      Container.Random_MLG.Reset(G,Random_MLG_Generator_State);
   exception
      when others =>
         Log.Report("Jpats_Landing_Gear.Controller.Reset_Random_MLG_Generator()");
         raise;
   end Reset_Random_MLG_Generator;

   procedure Initialize is
      package Ctnr       renames Container;
      package Main_Gear  renames Ctnr.Main_Gear;
      package Nose_Gear  renames Ctnr.Nose_Gear;
      package Gear_Door  renames Ctnr.Gear_Door;
      package Switches   renames Ctnr.Position_Switches;
      package Relays     renames Ctnr.Relays;
      package Left_Brake  renames Ctnr.Left_Brake;
      package Right_Brake renames Ctnr.Right_Brake;
      package Indicators renames Ctnr.Gear_Position_Indicators;

      The_Left_Gear       : Main_Gear.Instance renames Ctnr.This_Subsystem.The_Left_Main_Landing_Gear;
      The_Right_Gear      : Main_Gear.Instance renames Ctnr.This_Subsystem.The_Right_Main_Landing_Gear;
      The_Nose_Gear       : Nose_Gear.Instance renames Ctnr.This_Subsystem.The_Nose_Landing_Gear;
      The_Left_Gear_Door  : Gear_Door.Instance renames Ctnr.This_Subsystem.The_Left_Main_Gear_Door;
      The_Right_Gear_Door : Gear_Door.Instance renames Ctnr.This_Subsystem.The_Right_Main_Gear_Door;
      The_Switches        : Switches.Instance  renames Ctnr.This_Subsystem.The_Gear_Position_Switches;
   begin

      Main_Gear.Initialize(The_Left_Gear);
      Main_Gear.Initialize(The_Right_Gear);
      Nose_Gear.Initialize(The_Nose_Gear);
      Gear_Door.Initialize(The_Left_Gear_Door);
      Gear_Door.Initialize(The_Right_Gear_Door);
      Switches.Initialize(The_Switches);
      Relays.Initialize(Ctnr.This_Subsystem.The_Gear_Position_Relays);
      Left_Brake.Initialize(Ctnr.This_Subsystem.The_Left_Brake);
      Right_Brake.Initialize(Ctnr.This_Subsystem.The_Right_Brake);
      Indicators.Initialize(Ctnr.This_Subsystem.The_Position_Indicators);

      Main_Gear.Read_Tables;
      Nose_Gear.Read_Tables;
      Left_Brake.Read_Tables;
      Right_Brake.Read_Tables;

      Container.Random_MLG.Reset(G);

   exception
      when others =>
         Log.Report("Jpats_Landing_Gear.Controller.Initialize()");
         raise;
   end Initialize;


   Old_Gear_Commanded : Jpats_Landing_Gear_Types.Gear_Position_Type renames Container.This_Subsystem.LGC_Old_Gear_Commanded;
   Nose_Wheel_Steering_Enable : Boolean renames Container.This_Subsystem.LGC_Nose_Wheel_Steering_Enable;
   Nws_Switch_Engaged         : Boolean renames Container.This_Subsystem.LGC_Nws_Switch_Engaged;
   Nws_Switch_Closed          : Boolean renames Container.This_Subsystem.LGC_Nws_Switch_Closed;
   Nose_Wheel_Steer_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Nose_Wheel_Steer_Last_Pass;
   Main_Gear_Stuck_Up_Malf_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Main_Gear_Stuck_Up_Malf_Last_Pass;
   Main_Gear_Stuck_Down_Malf_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Main_Gear_Stuck_Down_Malf_Last_Pass;
   Main_Gear_Door_Stuck_Open_Malf_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Main_Gear_Door_Stuck_Open_Malf_Last_Pass;
   Main_Gear_Down_Indicator_Fail_Malf_Last_Pass  : Boolean renames Container.This_Subsystem.LGC_Main_Gear_Down_Indicator_Fail_Malf_Last_Pass;
   Main_Gear_Inboard_Door_Fail_Malf_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Main_Gear_Inboard_Door_Fail_Malf_Last_Pass;
   Gear_Sequencing_Fail_Malf_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Gear_Sequencing_Fail_Malf_Last_Pass;
   Gear_Control_Unit_Malf_One_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Gear_Control_Unit_Malf_One_Last_Pass;
   Gear_Control_Unit_Malf_Two_Last_Pass : Boolean renames Container.This_Subsystem.LGC_Gear_Control_Unit_Malf_Two_Last_Pass;
   Fail_Left_Main_Up         : Boolean renames Container.This_Subsystem.LGC_Fail_Left_Main_Up;
   Fail_Right_Main_Up        : Boolean renames Container.This_Subsystem.LGC_Fail_Right_Main_Up;
   Fail_Left_Main_Down       : Boolean renames Container.This_Subsystem.LGC_Fail_Left_Main_Down;
   Fail_Right_Main_Down      : Boolean renames Container.This_Subsystem.LGC_Fail_Right_Main_Down;
   Fail_Left_Door_Open       : Boolean renames Container.This_Subsystem.LGC_Fail_Left_Door_Open;
   Fail_Right_Door_Open      : Boolean renames Container.This_Subsystem.LGC_Fail_Right_Door_Open;
   Fail_Left_Main_Down_Ind   : Boolean renames Container.This_Subsystem.LGC_Fail_Left_Main_Down_Ind;
   Fail_Right_Main_Down_Ind  : Boolean renames Container.This_Subsystem.LGC_Fail_Right_Main_Down_Ind;
   Fail_Left_Door_Closed     : Boolean renames Container.This_Subsystem.LGC_Fail_Left_Door_Closed;
   Fail_Right_Door_Closed    : Boolean renames Container.This_Subsystem.LGC_Fail_Right_Door_Closed;
   Fail_Nose_Up              : Boolean renames Container.This_Subsystem.LGC_Fail_Nose_Up;

   function NWS_Enable
     return Boolean
   is
   begin
      return Nose_Wheel_Steering_Enable;
   exception
      when others =>
         Log.Report("Jpats_Landing_Gear.Controller.NWS_Enable()");
         raise;
   end NWS_Enable;

   procedure Update
     (Iconst : in Float) is

      package Ctnr       renames Container;
      package Ele        renames Jpats_Electrical;
      package Ele_T      renames Jpats_Electrical_Types;
      package Ele_U      renames Electrical_Units_Types;
      package Main_Gear  renames Ctnr.Main_Gear;
      package Nose_Gear  renames Ctnr.Nose_Gear;
      package Gear_T     renames Jpats_Landing_Gear_Types;
      package Gear_Door  renames Ctnr.Gear_Door;
      package Hyd        renames Jpats_Hydraulics;
      package Hyd_T      renames Jpats_Hydraulics_Types;
      package Switches   renames Ctnr.Position_Switches;
      package Relays     renames Ctnr.Relays;
      package Left_Brake  renames Ctnr.Left_Brake;
      package Right_Brake renames Ctnr.Right_Brake;
      package Indicators renames Ctnr.Gear_Position_Indicators;
      package Auto_Test  renames Jpats_Auto_Test;
      package Sfc        renames Jpats_Secondary_Flight_Controls;
      package Sfc_T      renames Jpats_Secondary_Flight_Controls_Types;
      package Lights     renames Jpats_Lights;
      package Edm        renames Jpats_Engine_Data_Manager;
      package Dcls       renames Jpats_Dcls;
      package Pwrplant   renames Jpats_Powerplant;
      package Jip        renames Jpats_IOS_Pilot;

      The_Left_Gear       : Main_Gear.Instance renames Ctnr.This_Subsystem.The_Left_Main_Landing_Gear;
      The_Right_Gear      : Main_Gear.Instance renames Ctnr.This_Subsystem.The_Right_Main_Landing_Gear;
      The_Nose_Gear       : Nose_Gear.Instance renames Ctnr.This_Subsystem.The_Nose_Landing_Gear;
      The_Left_Gear_Door  : Gear_Door.Instance renames Ctnr.This_Subsystem.The_Left_Main_Gear_Door;
      The_Right_Gear_Door : Gear_Door.Instance renames Ctnr.This_Subsystem.The_Right_Main_Gear_Door;
      The_Switches        : Switches.Instance  renames Ctnr.This_Subsystem.The_Gear_Position_Switches;
      The_Indicators      : Indicators.Instance renames Ctnr.This_Subsystem.The_Position_Indicators;
      The_Relays          : Relays.Instance renames Ctnr.This_Subsystem.The_Gear_Position_Relays;

      Io : Ctnr.Io_Interface_Instance renames Ctnr.This_Io_Interface;
      Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;
      Sound : Ctnr.Sound_Interface_Instance renames Ctnr.This_Sound_Interface;


      Emergency_Handle_Pulled : constant Boolean := Io.Emer_Landing_Gear_Sw;
      Gear_Selector_Up        : constant Boolean := not Io.Landing_Gear_Down_Sw;
      Gear_Selector_Down      : constant Boolean := Io.Landing_Gear_Down_Sw;

      Primary_Hyd_Fcn   : constant Hyd_T.Normalized_Pressure_Type := Hyd.Primary_System_Pressure_Function;
      Emergency_Hyd_Fcn : constant Hyd_T.Normalized_Pressure_Type := Hyd.Emergency_System_Pressure_Function;

      K_Hyd_Gear     : Hyd_T.Normalized_Pressure_Type;
      Gear_Commanded : Gear_T.Gear_Position_Type := 0.0;

      Ldg_Gr_Cont_Cb : constant Boolean := Ele.Is_Powered (Ele_T.Ldg_Gr_Cont_Cb);
      Test_Lts_Cb    : constant Boolean := Ele.Is_Powered (Ele_T.Test_Lts_Cb);

      Kias_Adc      : constant Length_Types.Knots       := Jpats_Flight_Instruments.Primary_Ias;
      Flap_Position : constant Sfc_T.Flap_Position_Type := Sfc.Mean_Flap_Position;

      Auto_Test_Selected           : constant Boolean := Auto_Test.On;
      Auto_Test_Position_Commanded : constant Boolean := Auto_Test.LG_Pos;
      Auto_Test_Emer_Gear          : constant Boolean := Auto_Test.Emer_Gear;

      Switch_Request             : Ctnr.Gear_Switch_Array_Of_Boolean;
      Relay_Pwr_And_Ground       : Ctnr.Relay_Array_Of_Boolean;
      Annunciator_Pwr_And_Ground : Ctnr.Position_Indicator_Array_Of_Boolean;
      Gear_Handle_Disagreement : Boolean := False;

      Nws_Reset_Relay : Boolean;
      Nws_Holding_Relay : Boolean;


      Gear_Handle_Light_Command : BOOLEAN := False;
      Warn_Lights_Command : Boolean := False;

      Ios_Pilot_Enable : constant Boolean := Jip.Ios_Pilot_Enable;
      Ios_Pilot_Gear_Down_Requested :constant Boolean := Jip.Request_Gear_Down;
      Ios_Pilot_Gear_Up_Requested :constant Boolean := Jip.Request_Gear_Up;

   begin

      -----------------
      -- GEAR HANDLE --
      -----------------

      if Ios.Gear_Control_Unit_Malf_Two and not Emergency_Handle_Pulled then

         Gear_Handle_Light_Command := False;
      else
         Gear_Handle_Light_Command := (not (Gear_Door.Up_Locked(The_Left_Gear_Door) and Gear_Door.Up_Locked(The_Right_Gear_Door))
                                      or (Pwrplant.Landing_Gear_Warn_Low_Throttle_Switch and Main_Gear.Up_Locked(The_Left_Gear)
                                      and Main_Gear.Up_Locked(The_Right_Gear) and Nose_Gear.Up_Locked(The_Nose_Gear))
                                      or (Gear_Selector_UP and Relays.Is_Closed (The_Relays,Gear_T.Left_Landing_Gear_A1_WOW_Relay)));
      end if;


      Io.Landing_Gear_Handle_Lt := (Gear_Handle_Light_Command and Ldg_Gr_Cont_Cb) or
                                   (Lights.Lamp_Test_Sw and Test_Lts_Cb);

      -------------------------------------
      -- HYD FUNCTION FOR DOORS AND GEAR --
      -------------------------------------

      if Auto_Test_Selected and not Auto_Test_Emer_Gear then

         K_Hyd_Gear := 1.0;

      else

         if Emergency_Handle_Pulled or Auto_Test_Emer_Gear then
            K_Hyd_Gear := Emergency_Hyd_Fcn;
         else
            if Ldg_Gr_Cont_Cb then
               K_Hyd_Gear := Primary_Hyd_Fcn;
            else
               K_Hyd_Gear := 0.0;
            end if;
         end if;

      end if;

      ------------------------
      -- COMMANDED POSITION --
      ------------------------

      if Auto_Test_Selected then
         if Auto_Test_Position_Commanded or Auto_Test_Emer_Gear then
            Gear_Commanded := 1.0;
         elsif not Auto_Test_Position_Commanded then
            Gear_Commanded := 0.0;
         end if;
      elsif Ios_Pilot_Enable then
         if Ios_Pilot_Gear_Down_Requested then
            Ios.Ios_Pilot_Gear := 1;
         end if;
         if Ios_Pilot_Gear_Up_Requested then
            Ios.Ios_Pilot_Gear := 0;
         end if;
         if Ios.Ios_Pilot_Gear = 0 then
            Gear_Commanded := 0.0;
            Io.Landing_Gear_Down_Sw := False;
         elsif Ios.Ios_Pilot_Gear = 1 then
            Gear_Commanded := 1.0;
            Io.Landing_Gear_Down_Sw := True;
         end if;
      else
         if Gear_Selector_Down or Emergency_Handle_Pulled then
            Gear_Commanded := 1.0;
         elsif Gear_Selector_Up then
            Gear_Commanded := 0.0;
         else
            Gear_Commanded := Old_Gear_Commanded;
         end if;
      end if;

      Old_Gear_Commanded := Gear_Commanded;

      ----------------------------------------
      -- GEAR DOOR MALF DETERMINATION LOGIC --
      ----------------------------------------

      if Ios.Main_Gear_Door_Stuck_Open_Malf and not Main_Gear_Door_Stuck_Open_Malf_Last_Pass then

         case Container.Random_MLG.Random(G) is
            when Left =>
               Fail_Left_Door_Open  := True;
               Fail_Right_Door_Open := False;
            when Right =>
               Fail_Left_Door_Open  := False;
               Fail_Right_Door_Open := True;
            when others =>
               Fail_Left_Door_Open  := False;
               Fail_Right_Door_Open := False;
         end case;
      elsif not Ios.Main_Gear_Door_Stuck_Open_Malf and Main_Gear_Door_Stuck_Open_Malf_Last_Pass then
         Fail_Left_Door_Open  := False;
         Fail_Right_Door_Open := False;
      end if;

      Main_Gear_Door_Stuck_Open_Malf_Last_Pass := Ios.Main_Gear_Door_Stuck_Open_Malf;

      if Ios.Main_Gear_Inboard_Door_Fail_Malf and not Main_Gear_Inboard_Door_Fail_Malf_Last_Pass then

         case Container.Random_MLG.Random(G) is
            when Left =>
               Fail_Left_Door_Closed  := True;
               Fail_Right_Door_Closed := False;
            when Right =>
               Fail_Left_Door_Closed  := False;
               Fail_Right_Door_Closed := True;
            when others =>
               Fail_Left_Door_Closed  := False;
               Fail_Right_Door_Closed := False;
         end case;
      elsif not Ios.Main_Gear_Inboard_Door_Fail_Malf and Main_Gear_Inboard_Door_Fail_Malf_Last_Pass then
         Fail_Left_Door_Closed  := False;
         Fail_Right_Door_Closed := False;
      end if;

      Main_Gear_Inboard_Door_Fail_Malf_Last_Pass := Ios.Main_Gear_Inboard_Door_Fail_Malf;

      if not Ios.Main_Gear_Inboard_Door_Fail_Malf and not Ios.Gear_Sequencing_Fail_Malf
         and not Ios.Main_Gear_Door_Stuck_Open_Malf then

         if Emergency_Handle_Pulled then
            Fail_Left_Door_Closed  := False;
            Fail_Right_Door_Closed := False;
         else
            if Ios.Gear_Control_Unit_Malf_Two and not Gear_Control_Unit_Malf_Two_Last_Pass then
               Fail_Left_Door_Closed  := True;
               Fail_Right_Door_Closed := True;
            elsif not Ios.Gear_Control_Unit_Malf_Two and Gear_Control_Unit_Malf_Two_Last_Pass then
               Fail_Left_Door_Closed  := False;
               Fail_Right_Door_Closed := False;
            end if;

         end if;

      end if;

      ----------------
      -- GEAR DOORS --
      ----------------

      if Gear_Commanded = 1.0 then
         if (Nose_Gear.Down_Locked (The_Nose_Gear) and
             Main_Gear.Down_Locked (The_Left_Gear) and
             Main_Gear.Down_Locked (The_Right_Gear) and
             not Emergency_Handle_Pulled) then
            if not Fail_Left_Door_Open then
               Gear_Door.Retract
                 (An_Instance => The_Left_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
            if not Fail_Right_Door_Open then
               Gear_Door.Retract
                 (An_Instance => The_Right_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
         else
            if not Fail_Left_Door_Closed then
               Gear_Door.Extend
                 (An_Instance => The_Left_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
            if not Fail_Right_Door_Closed then
               Gear_Door.Extend
                 (An_Instance => The_Right_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
         end if;
      else
         if (not Nose_Gear.Up_Locked (The_Nose_Gear) or
            not Main_Gear.Up_Locked (The_Left_Gear) or
            not Main_Gear.Up_Locked (The_Right_Gear)) and
            Switches.Is_Open (The_Switches,Gear_T.Left_Gear_WOW_Switch) then
            if not Fail_Left_Door_Closed then
               Gear_Door.Extend
                 (An_Instance => The_Left_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
            if not Fail_Right_Door_Closed then
               Gear_Door.Extend
                 (An_Instance => The_Right_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
         else
            if not Fail_Left_Door_Open then
               Gear_Door.Retract
                 (An_Instance => The_Left_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
            if not Fail_Right_Door_Open then
               Gear_Door.Retract
                 (An_Instance => The_Right_Gear_Door,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;
         end if;
      end if;

      -----------------------------------
      -- GEAR MALF DETERMINATION LOGIC --
      -----------------------------------

      if Ios.Main_Gear_Stuck_Up_Malf and not Main_Gear_Stuck_Up_Malf_Last_Pass then

         case Container.Random_MLG.Random(G) is
            when Left =>
               Fail_Left_Main_Up  := True;
               Fail_Right_Main_Up := False;
            when Right =>
               Fail_Left_Main_Up  := False;
               Fail_Right_Main_Up := True;
            when others =>
               Fail_Left_Main_Up  := False;
               Fail_Right_Main_Up := False;
         end case;
      elsif not Ios.Main_Gear_Stuck_Up_Malf and Main_Gear_Stuck_Up_Malf_Last_Pass then
         Fail_Left_Main_Up  := False;
         Fail_Right_Main_Up := False;
      end if;

      Main_Gear_Stuck_Up_Malf_Last_Pass := Ios.Main_Gear_Stuck_Up_Malf;

      Fail_Left_Main_Down := Ios.Main_Gear_Stuck_Down_Malf;

      if not Ios.Main_Gear_Stuck_Up_Malf and not Ios.Main_Gear_Stuck_Down_Malf then

         if Emergency_Handle_Pulled then
            Fail_Nose_Up       := False;
            Fail_Left_Main_Up  := False;
            Fail_Right_Main_Up := False;
            Gear_Control_Unit_Malf_One_Last_Pass := False;
         else
            if Ios.Gear_Control_Unit_Malf_One and not Gear_Control_Unit_Malf_One_Last_Pass then
               Fail_Nose_Up       := True;
               Fail_Left_Main_Up  := True;
               Fail_Right_Main_Up := True;
            elsif not Ios.Gear_Control_Unit_Malf_One and Gear_Control_Unit_Malf_One_Last_Pass then
               Fail_Nose_Up       := False;
               Fail_Left_Main_Up  := False;
               Fail_Right_Main_Up := False;
            end if;

            Gear_Control_Unit_Malf_One_Last_Pass := Ios.Gear_Control_Unit_Malf_One;

         end if;

      end if;

      if not Ios.Main_Gear_Stuck_Up_Malf and not Ios.Main_Gear_Stuck_Down_Malf 
         and not Ios.Gear_Control_Unit_Malf_One then

         if Emergency_Handle_Pulled then
            Fail_Nose_Up       := False;
            Fail_Left_Main_Up  := False;
            Fail_Right_Main_Up := False;
            Gear_Control_Unit_Malf_Two_Last_Pass := False;
         else
            if Ios.Gear_Control_Unit_Malf_Two and not Gear_Control_Unit_Malf_Two_Last_Pass then
               Fail_Nose_Up       := True;
               Fail_Left_Main_Up  := True;
               Fail_Right_Main_Up := True;
            elsif not Ios.Gear_Control_Unit_Malf_Two and Gear_Control_Unit_Malf_Two_Last_Pass then
               Fail_Nose_Up       := False;
               Fail_Left_Main_Up  := False;
               Fail_Right_Main_Up := False;
            end if;

            Gear_Control_Unit_Malf_Two_Last_Pass := Ios.Gear_Control_Unit_Malf_Two;

         end if;

      end if;

      ----------
      -- GEAR --
      ----------

      if Gear_Commanded = 1.0 then

         if (((Gear_Door.Open (The_Left_Gear_Door) and
              Gear_Door.Open (The_Right_Gear_Door)) and not
              Ios.Gear_Sequencing_Fail_Malf) or
             (Emergency_Handle_Pulled or Auto_Test_Emer_Gear)) then

            if not Ios.Nose_Gear_Stuck_Up_Malf and not Fail_Nose_Up then
               Nose_Gear.Extend
                 (An_Instance => The_Nose_Gear,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;

         end if;

         if (Gear_Door.Open (The_Left_Gear_Door) and
             Gear_Door.Open (The_Right_Gear_Door) and not 
             (Ios.Gear_Sequencing_Fail_Malf and not Emergency_Handle_Pulled)) then

            if not Fail_Left_Main_Up then
               Main_Gear.Extend
                 (An_Instance => The_Left_Gear,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;

            if not Fail_Right_Main_Up then
               Main_Gear.Extend
                 (An_Instance => The_Right_Gear,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;

         end if;

      else

         if ((Gear_Door.Open (The_Left_Gear_Door) and
              Gear_Door.Open (The_Right_Gear_Door)) and
             Switches.Is_Open (The_Switches,Gear_T.Right_Gear_WOW_Switch)) then

            if not Ios.Nose_Gear_Stuck_Down_Malf then
               Nose_Gear.Retract
                 (An_Instance => The_Nose_Gear,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;

            if not Fail_Left_Main_Down then
               Main_Gear.Retract
                 (An_Instance => The_Left_Gear,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;

            if not Fail_Right_Main_Down then
               Main_Gear.Retract
                 (An_Instance => The_Right_Gear,
                  Iconst      => Iconst,
                  Hyd_Fcn     => K_Hyd_Gear);
            end if;

         end if;

      end if;

      -------------------
      -- GEAR SWITCHES --
      -------------------

      --    Left_Door_Lock_Switch
      Switch_Request (Gear_T.Left_Door_Lock_Switch) := Gear_Door.Up_Locked (The_Left_Gear_Door);

      --    Left_Main_Gear_Switch
      Switch_Request (Gear_T.Left_Main_Gear_Switch) := Main_Gear.Up_Locked (The_Left_Gear);

      --    Left_Gear_Up_And_Locked_Switch
      Switch_Request (Gear_T.Left_Gear_Up_And_Locked_Switch) := Main_Gear.Up_Locked (The_Left_Gear);

      --    Left_Gear_Down_And_Locked_Switch
      Switch_Request (Gear_T.Left_Gear_Down_And_Locked_Switch) := Main_Gear.Down_Locked (The_Left_Gear);

      --    Left_Gear_WOW_Switch
      Switch_Request (Gear_T.Left_Gear_WOW_Switch) :=
        Main_Gear.Down_Locked (The_Left_Gear) and
        Jpats_Aircraft_Body.Get_Left_WOW;

      --    Nose_Gear_Down_And_Locked_Switch
      Switch_Request (Gear_T.Nose_Gear_Down_And_Locked_Switch) := Nose_Gear.Down_Locked (The_Nose_Gear);

      --    Nose_Gear_Up_And_Locked_Switch
      Switch_Request (Gear_T.Nose_Gear_Up_And_Locked_Switch) := Nose_Gear.Up_Locked (The_Nose_Gear);

      --    Right_Door_Lock_Switch
      Switch_Request (Gear_T.Right_Door_Lock_Switch) := Gear_Door.Up_Locked (The_Right_Gear_Door);

      --    Right_Main_Gear_Switch
      Switch_Request (Gear_T.Right_Main_Gear_Switch) := Main_Gear.Up_Locked (The_Right_Gear);

      --    Right_Gear_Up_And_Locked_Switch
      Switch_Request (Gear_T.Right_Gear_Up_And_Locked_Switch) := Main_Gear.Up_Locked (The_Right_Gear);

      --    Right_Gear_Down_And_Locked_Switch
      Switch_Request (Gear_T.Right_Gear_Down_And_Locked_Switch) := Main_Gear.Down_Locked (The_Right_Gear);

      --    Right_Gear_WOW_Switch
      Switch_Request (Gear_T.Right_Gear_WOW_Switch) :=
        Main_Gear.Down_Locked (The_Right_Gear) and
        Jpats_Aircraft_Body.Get_Right_WOW;



     Switches.Update
       (An_Instance => The_Switches,
        Request     => Switch_Request);


     -- ios checklist gear_posiiton
     if Main_Gear.Up_Locked(The_Right_Gear) and
       Main_Gear.Up_Locked(The_Left_Gear) and
       Nose_Gear.Up_Locked(The_Nose_Gear) then
        Ios.Gear_Position := 0;
     elsif Nose_Gear.Down_Locked(The_Nose_Gear) and
       Main_Gear.Down_Locked(The_Left_Gear) and
       Main_Gear.Down_Locked(The_Right_Gear) then
        Ios.Gear_Position := 1;
     else
        Ios.Gear_Position := 2;
     end if;

     -----------------
     -- GEAR RELAYS --
     -----------------

     -- Left_Landing_Gear_A1_WOW_Relay
     Relay_Pwr_And_Ground (Gear_T.Left_Landing_Gear_A1_WOW_Relay) :=
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name => Gear_T.Left_Gear_WOW_Switch) and Ldg_Gr_Cont_Cb;

     -- Right_Landing_Gear_A1_WOW_Relay
     Relay_Pwr_And_Ground (Gear_T.Right_Landing_Gear_A1_WOW_Relay) :=
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name => Gear_T.Right_Gear_WOW_Switch) and Ldg_Gr_Cont_Cb;

     -- Left_Landing_Gear_A2_WOW_Relay
     Relay_Pwr_And_Ground (Gear_T.Left_Landing_Gear_A2_WOW_Relay) :=
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name => Gear_T.Left_Gear_WOW_Switch) and Ldg_Gr_Cont_Cb;

     -- Right_Landing_Gear_A2_WOW_Relay
     Relay_Pwr_And_Ground (Gear_T.Right_Landing_Gear_A2_WOW_Relay) :=
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name => Gear_T.Right_Gear_WOW_Switch) and Ldg_Gr_Cont_Cb;

     -- Q_Switch_Control_Relay
     Relay_Pwr_And_Ground (Gear_T.Q_Switch_Control_Relay) :=
       Ldg_Gr_Cont_Cb and ((Flap_Position > 23.0) or (Kias_Adc < 120.0));

     -- Landing_Gear_Down_And_Locked
     Relay_Pwr_And_Ground (Gear_T.Landing_Gear_Down_And_Locked) :=
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name => Gear_T.Left_Gear_Down_And_Locked_Switch) and
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name => Gear_T.Right_Gear_Down_And_Locked_Switch) and
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name => Gear_T.Nose_Gear_Down_And_Locked_Switch) and
       Ldg_Gr_Cont_Cb;


     Relays.Update
       (An_Instance      => The_Relays,
        Power_And_Ground => Relay_Pwr_And_Ground);

     --------------------------
     -- GROUND LOCK SOLENOID --
     --------------------------

     IO.Landing_Gear_Lock := (Ldg_Gr_Cont_Cb and IO.Downlock_Override_Sw) or
        (Relays.Is_Open(The_Relays,Gear_T.Right_Landing_Gear_A1_WOW_Relay) and
        not Ios.Gear_Control_Unit_Malf_Two);

     --------------------------------
     -- GEAR POSITION ANNUNCIATORS --
     --------------------------------

     Annunciator_Pwr_And_Ground (Ctnr.Left_Main_Gear_Down_And_Locked_Indicator) :=
       Ldg_Gr_Cont_Cb and
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name        => Gear_T.Left_Gear_Down_And_Locked_Switch) and
       Switches.Is_Open
       (An_Instance => The_Switches,
        Name        => Gear_T.Left_Gear_Up_And_Locked_Switch);

     Annunciator_Pwr_And_Ground (Ctnr.Right_Main_Gear_Down_And_Locked_Indicator) :=
       Ldg_Gr_Cont_Cb and
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name        => Gear_T.Right_Gear_Down_And_Locked_Switch) and
       Switches.Is_Open
       (An_Instance => The_Switches,
        Name        => Gear_T.Right_Gear_Up_And_Locked_Switch);

     Annunciator_Pwr_And_Ground (Ctnr.Nose_Gear_Down_And_Locked_Indicator) :=
       Ldg_Gr_Cont_Cb and
       Switches.Is_Closed
       (An_Instance => The_Switches,
        Name        => Gear_T.Nose_Gear_Down_And_Locked_Switch);

     Annunciator_Pwr_And_Ground (Ctnr.Left_Main_Gear_In_Transit_Indicator) :=
       Ldg_Gr_Cont_Cb and
       (Switches.Is_Open
        (An_Instance => The_Switches,
         Name        => Gear_T.Left_Door_Lock_Switch) or
        (Switches.Is_Open
         (An_Instance => The_Switches,
          Name        => Gear_T.Left_Gear_Down_And_Locked_Switch) and
         Switches.Is_Open
         (An_Instance => The_Switches,
          Name        => Gear_T.Left_Gear_Up_And_Locked_Switch)));

     Annunciator_Pwr_And_Ground (Ctnr.Right_Main_Gear_In_Transit_Indicator) :=
       Ldg_Gr_Cont_Cb and
       (Switches.Is_Open
        (An_Instance => The_Switches,
         Name        => Gear_T.Right_Door_Lock_Switch) or
        (Switches.Is_Open
         (An_Instance => The_Switches,
          Name        => Gear_T.Right_Gear_Down_And_Locked_Switch) and
         Switches.Is_Open
         (An_Instance => The_Switches,
          Name        => Gear_T.Right_Gear_Up_And_Locked_Switch)));

     Annunciator_Pwr_And_Ground (Ctnr.Nose_Gear_In_Transit_Indicator) :=
       Ldg_Gr_Cont_Cb and
       Switches.Is_Open
       (An_Instance => The_Switches,
        Name        => Gear_T.Nose_Gear_Up_And_Locked_Switch) and
       Switches.Is_Open
       (An_Instance => The_Switches,
        Name        => Gear_T.Nose_Gear_Down_And_Locked_Switch);

     -- update collection
     Indicators.Update
       (An_Instance => Ctnr.This_Subsystem.The_Position_Indicators,
        Request     => Annunciator_Pwr_And_Ground);

     ----------------------------------------------
     -- GEAR INDICATION MALF DETERMINATION LOGIC --
     ----------------------------------------------

     if Ios.Main_Gear_Down_Indicator_Fail_Malf and not Main_Gear_Down_Indicator_Fail_Malf_Last_Pass then

        case Container.Random_MLG.Random(G) is
           when Left =>
              Fail_Left_Main_Down_Ind  := True;
              Fail_Right_Main_Down_Ind := False;
           when Right =>
              Fail_Left_Main_Down_Ind  := False;
              Fail_Right_Main_Down_Ind := True;
           when others =>
              Fail_Left_Main_Down_Ind  := False;
              Fail_Right_Main_Down_Ind := False;
        end case;
     elsif not Ios.Main_Gear_Down_Indicator_Fail_Malf and Main_Gear_Down_Indicator_Fail_Malf_Last_Pass then
        Fail_Left_Main_Down_Ind  := False;
        Fail_Right_Main_Down_Ind := False;
     end if;

     Main_Gear_Down_Indicator_Fail_Malf_Last_Pass := Ios.Main_Gear_Down_Indicator_Fail_Malf;

     ---------------------
     -- GEAR INDICATION --
     ---------------------

     Warn_Lights_Command := (Pwrplant.Landing_Gear_Warn_Low_Throttle_Switch
                             and Main_Gear.Up_Locked(The_Left_Gear)
                             and Main_Gear.Up_Locked(The_Right_Gear)
                             and Nose_Gear.Up_Locked(The_Nose_Gear)) or
                            (Gear_Selector_UP and Relays.Is_Closed (The_Relays,Gear_T.Left_Landing_Gear_A1_WOW_Relay));

     Io.Lh_Gear_Green   := (Indicators.Is_On (The_Indicators,Ctnr.Left_Main_Gear_Down_And_Locked_Indicator) or
                            (Lights.Lamp_Test_Sw and Test_Lts_Cb)) and not Fail_Left_Main_Down_Ind;

     Io.Lh_Gear_Red     := (Indicators.Is_On (The_Indicators,Ctnr.Left_Main_Gear_In_Transit_Indicator) or
                            (Lights.Lamp_Test_Sw and Test_Lts_Cb) or (Warn_Lights_Command and Ldg_Gr_Cont_Cb));

     Io.Nose_Gear_Green := (Indicators.Is_On (The_Indicators,Ctnr.Nose_Gear_Down_And_Locked_Indicator) or
                            (Lights.Lamp_Test_Sw and Test_Lts_Cb));

     Io.Nose_Gear_Red   := (Indicators.Is_On (The_Indicators,Ctnr.Nose_Gear_In_Transit_Indicator) or
                            (Lights.Lamp_Test_Sw and Test_Lts_Cb) or (Warn_Lights_Command and Ldg_Gr_Cont_Cb));

     Io.Rh_Gear_Green   := (Indicators.Is_On (The_Indicators,Ctnr.Right_Main_Gear_Down_And_Locked_Indicator) or
                            (Lights.Lamp_Test_Sw and Test_Lts_Cb)) and not Fail_Right_Main_Down_Ind;

     Io.Rh_Gear_Red     := (Indicators.Is_On (The_Indicators,Ctnr.Right_Main_Gear_In_Transit_Indicator) or
                            (Lights.Lamp_Test_Sw and Test_Lts_Cb) or (Warn_Lights_Command and Ldg_Gr_Cont_Cb));

     ------------
     -- BRAKES --
     ------------

     declare

        Left_Brake_Position  : constant Float := Dcls.Left_Brake_Pos;
        Right_Brake_Position : constant Float := Dcls.Right_Brake_Pos;

     begin

        Left_Brake.Update
          (An_Instance   => Ctnr.This_Subsystem.The_Left_Brake,
           Brake_Pos     => Left_Brake_Position,
           Px_Norm       => 1.0,
           Vol_Norm      => 1.0,
           Parking_Brake => Io.Parking_Brake,
           Failure_Malf  => Ios.Total_Brake_Pressure_Loss_Malf or
                            Ios.Left_Brake_Pressure_Loss_Malf or
                            Ios.Right_Tire_Left_Brake_Malf);

        Right_Brake.Update
          (An_Instance => Ctnr.This_Subsystem.The_Right_Brake,
           Brake_Pos     => Right_Brake_Position,
           Px_Norm       => 1.0,
           Vol_Norm      => 1.0,
           Parking_Brake => Io.Parking_Brake,
           Failure_Malf  => Ios.Total_Brake_Pressure_Loss_Malf or
                            Ios.Right_Brake_Pressure_Loss_Malf or
                            Ios.Left_Tire_Right_Brake_Malf);

     end;

     -------------------------
     -- NOSE WHEEL STEERING --
     -------------------------

     Nws_Switch_Engaged := Io.Nose_Wheel_Steer and not Nose_Wheel_Steer_Last_Pass;

     if Nws_Switch_Engaged and not Nws_Switch_Closed then
        Nws_Switch_Closed := True;
     elsif Nws_Switch_Engaged and Nws_Switch_Closed then
        Nws_Switch_Closed := False;
     end if;

     if not Ele.Is_Powered(Ele_T.Nws_Cb)
       or not Relays.Is_Closed (The_Relays,Gear_T.Left_Landing_Gear_A1_WOW_Relay) then
        Nws_Switch_Closed := False;
     end if;

     Nose_Wheel_Steer_Last_Pass := Io.Nose_Wheel_Steer;

     Nws_Reset_Relay := (Ele.Is_Powered(Ele_T.Nws_Cb) and
                         Relays.Is_Closed (The_Relays,Gear_T.Left_Landing_Gear_A1_WOW_Relay));

     Nws_Holding_Relay := Ele.Is_Powered(Ele_T.Nws_Cb) and Nws_Reset_Relay and Nws_Switch_Closed;

     Nose_Wheel_Steering_Enable := (((Ele.Is_Powered(Ele_T.Nws_Cb) and
                                      Nws_Holding_Relay and
                                      Relays.Is_Closed(The_Relays, Gear_T.Right_Landing_Gear_A1_WOW_Relay))) or
                                    Ios.Nws_Enable);

     Io.Nose_Wheel_Steering_Ann :=  ((Nose_Wheel_Steering_Enable or (Lights.Lamp_Test_Sw and Test_Lts_Cb))
                                      and not Ios.Loss_NWS_Indication_Malf) or Ios.Nws_Enable;


     Ios.Nws_Engaged := Io.Nose_Wheel_Steering_Ann;

     ------------------
     -- wheel chocks --
     ------------------

     declare
        package Sim_Acft renames Jpats_Simulated_Aircraft;
        package Alt renames Length_Types;
        package Vel renames Length_Types;
        Altitude     : constant Alt.Feet         := Sim_Acft.Get_Aircraft_Height_Above_Local_Terrain;
        Ground_Speed : constant Vel.Feet_per_Sec := Sim_Acft.Get_Ground_Speed;
        Ground_Service_Available : constant Boolean := Altitude < 10.0 and Ground_Speed < 1.5;
     begin

        if JPATS_Reposition.Reposition_Command.Active then

           Ctnr.This_Subsystem.Wheel_Chock_On := Ios.Wheel_Chocks;

        else

           if Ground_Service_Available then
              Ctnr.This_Subsystem.Wheel_Chock_On := Ios.Wheel_Chocks;
           else
              Ctnr.This_Subsystem.Wheel_Chock_On := False;
           end if;

        end if;

     exception
           when others =>
              Log.Report("wheel chocks");
              raise;
     end;

     -----------
     -- SOUND --
     -----------

     Sound.Left_Main_Gear_Door_Position  := Gear_Door.Position(The_Left_Gear_Door);
     Sound.Right_Main_Gear_Door_Position := Gear_Door.Position(The_Right_Gear_Door);
     Sound.Nose_Gear_Door_Position       := Nose_Gear.Position(The_Nose_Gear);
     Sound.Left_Main_Gear_Position       := Main_Gear.Position(The_Left_Gear);
     Sound.Right_Main_Gear_Position      := Main_Gear.Position(The_Right_Gear);
     Sound.Nose_Gear_Position            := Nose_Gear.Position(The_Nose_Gear);

     ----------------------------
     -- RANDOM GENERATOR STATE --
     ----------------------------

     Container.Random_MLG.Save(G,Random_MLG_Generator_State);

   exception
      when others =>
         Log.Report("Jpats_Landing_Gear.Controller.Update()");
         raise;
   end Update;

end Jpats_Landing_Gear.Controller;
