-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Log;
with Interpolation_Table.Singly_Indexed;
with Simulation_Dictionary;

package body Brake_G is

   Brake_Pressure : aliased Interpolation_Table.Singly_Indexed.Instance;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Pressure := 0.0;
      An_Instance.Back_Pressure := 0.0;
      An_Instance.K_Pump := 1.0;
   exception
      when others =>
         Log.Report(Package_Name & ".Initialize()");
         raise;
   end Initialize;

   function Pressure
     (An_Instance : in Instance)
     return Pressure_Type
   is
   begin
      return An_Instance.Pressure;
   exception
      when others =>
         Log.Report(Package_Name & ".Pressure()");
         raise;
   end Pressure;

   function Back_Pressure
     (An_Instance : in Instance)
     return Pressure_Type
   is
   begin
      return An_Instance.Back_Pressure;
   exception
      when others =>
         Log.Report(Package_Name & ".Back_Pressure()");
         raise;
   end Back_Pressure;

   function K_Pump
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.K_Pump;
   exception
      when others =>
         Log.Report(Package_Name & ".K_Pump()");
         raise;
   end K_Pump;


   procedure Update
     (An_Instance   : in out Instance;
      Brake_Pos     : in     Float;
      Px_Norm       : in     Float;
      Vol_Norm      : in     Float;
      Parking_Brake : in     Boolean;
      Failure_Malf  : in     Boolean)
   is

      package Single renames Interpolation_Table.Singly_Indexed;

      Px : Pressure_Type; -- := Max_Px*Pressure_Type((Brake_Pos/(-10.0))*Px_Norm*Vol_Norm);

   begin

      Px := (Pressure_Type(Single.Interpolate (abs(Brake_Pos), Brake_Pressure'Access)) *
             Pressure_Type(Vol_Norm) *
             Pressure_Type(Px_Norm));

      if Failure_Malf then
         An_Instance.Pressure := 0.0;
         An_Instance.Back_Pressure := 0.0;
      elsif Parking_Brake then
         if Px > An_Instance.Pressure then
            An_Instance.Pressure := Px;
         end if;
         An_Instance.Back_Pressure := 5.0;
      else
         An_Instance.Pressure := Px;
         An_Instance.Back_Pressure := 5.0;
      end if;

   exception
      when others =>
         Log.Report(Package_Name & ".Update()");
         raise;
   end Update;

   procedure UpdateWPump
     (An_Instance   : in out Instance;
      Iconst        : in     Float;
      Auto_Test_On  : in     Boolean;
      Brake_Pos     : in     Float;
      Px_Norm       : in     Float;
      Vol_Norm      : in     Float;
      Parking_Brake : in     Boolean;
      Failure_Malf  : in     Boolean)
   is

      package Single renames Interpolation_Table.Singly_Indexed;

      Last_Px : constant Pressure_Type := An_Instance.Pressure;
      Last_K_Pump : constant Float := An_Instance.K_Pump;


      Px : Pressure_Type; -- := Max_Px*Pressure_Type((Brake_Pos/(-10.0))*Px_Norm*Vol_Norm);
      Kp : Float;

   begin

      Kp :=
        Float'Max(1.0,
                  Float'Min(1.4,
                            Last_K_Pump + 0.001 * Float'Max(0.0,
                                                            Float(Last_Px) - 25.0) * Iconst - 0.003 * Iconst));

      if Auto_Test_On then
         An_Instance.K_Pump := 1.0;
      else
         An_Instance.K_Pump := Float'Max(1.0, Kp*Float(Vol_Norm)*Float(Px_Norm));
      end if;

      Px := (Pressure_Type(Single.Interpolate (abs(Brake_Pos), Brake_Pressure'Access)) *
             Pressure_Type(Vol_Norm) *
             Pressure_Type(Px_Norm));

      Px := Px * Pressure_Type(An_Instance.K_Pump);

      if Failure_Malf then
         An_Instance.Pressure := 0.0;
         An_Instance.Back_Pressure := 0.0;
      elsif Parking_Brake then
         if Px > An_Instance.Pressure then
            An_Instance.Pressure := Px;
         end if;
         An_Instance.Back_Pressure := 5.0;
      else
         An_Instance.Pressure := Px;
         An_Instance.Back_Pressure := 5.0;
      end if;

   exception
      when others =>
         Log.Report(Package_Name & ".Update()");
         raise;
   end UpdateWPump;


   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("Landing_Gear_Dir") & Brake_Pressure_Table & ".ito",
         Table     => Brake_Pressure);
   exception
      when others =>
         Log.Report(Package_Name & ".Read_Tables()");
         raise;
   end Read_Tables;


end Brake_G;
