-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Log;

--debug
with Jpats_Simulated_Aircraft;

package body Yaw_Driver  is
   --| General definitions -----------------------------------------------------
   Body_File_Name : constant String := "IOS_pilot/Yaw_Driver.adb";


   procedure Set_Gain
     (Tuning_Mode      :in     Integer;
      KP_Sideslip      :in     Float;
      KP_Lat_Acc       :in     Float;
      KI_Lat_Acc       :in     Float;
      KD_Lat_Acc       :in     Float;
      Yaw_Gain         :in     Float;
      Heading_Gain     :in     Float;
      An_Instance      :in out Instance) is
   begin
      An_Instance.Tuning_Mode := Tuning_Mode;
      An_Instance.KP_Sideslip := KP_Sideslip;
      An_Instance.KP_Lat_Acc  := KP_Lat_Acc;
      An_Instance.KI_Lat_Acc  := KI_Lat_Acc;
      An_Instance.KD_Lat_Acc  := KD_Lat_Acc;
      An_Instance.Yaw_Gain    := Yaw_Gain;
      An_Instance.Heading_Gain:= Heading_Gain;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize",
                    Severity => Log.ERROR);
         raise;

   end Set_Gain;


   procedure initialize
     (An_Instance            :in out  Instance) is
   begin

      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Yaw_Rate);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Lat_Acc);
      An_Instance.Mode := 0;
      An_Instance.Tuning_Mode := 0;

      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Force);
      First_Order_Filter.Initialize(0.0,An_Instance.Washout_Force);

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize",
                    Severity => Log.ERROR);
         raise;

   end Initialize;


   procedure Set_Mode
     (Mode                    :in       Integer;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.Mode := Mode;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_mode",
                    Severity => Log.ERROR);
         raise;

   end Set_mode;

   procedure Reset_Internal_variables
     (Yaw_Rate           :in      Angle_Types.Degrees_Per_Sec;
      Lateral_Acc        :in      Length_Types.Feet_Per_Sec_Sq;
      Rudder_Deflection  :in      Angle_Types.Degrees;
      An_Instance        :in out  Instance) is
   begin

      An_Instance.Integral_Lat_Acc         := 0.0;
      An_Instance.Last_Lat_Acc             := 0.0;
      An_Instance.Commanded_Rudder_Deflection := Rudder_Deflection;
      First_Order_Filter.Initialize(Yaw_rate,An_Instance.Filtered_Yaw_Rate);
      First_Order_Filter.Initialize(Lateral_acc,An_Instance.Filtered_Lat_Acc);
      An_Instance.Fader   := 0.0;
      An_Instance.Error   := 0.0;
      An_Instance.Error_L := 0.0;
      An_Instance.T_TO    := 0.0;
      An_Instance.Lat_TO  := 0.0;
      An_Instance.Long_TO := 0.0;


      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Force);
      First_Order_Filter.Initialize(0.0,An_Instance.Washout_Force);

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Reset_Internal_variables",
                    Severity => Log.ERROR);
         raise;

   end Reset_Internal_variables;


   procedure Set_Reference
     (Mode                    :in      Integer;
      Selected_Heading        :in      Angle_Types.Degrees;
      An_Instance             :in out  Instance) is

   begin

      An_Instance.Mode := Mode;

      An_Instance.Selected_Heading := Selected_Heading;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_reference",
                    Severity => Log.ERROR);
         raise;

   end Set_Reference;



   procedure Update
     (Heading                    :in      Angle_Types.Degrees;
      Altitude_AGL               :in      Length_Types.Feet;
      TAS                        :in      Length_Types.Feet_Per_Sec;
      Roll_Angle                 :in      Angle_Types.Degrees;
      Pitch_Angle                :in      Angle_Types.Degrees;
      Yaw_Rate                   :in      Angle_Types.Degrees_Per_Sec;
      Sideslip_Angle             :in      Angle_Types.Degrees;
      Lateral_Acc                :in      Length_Types.Feet_Per_Sec_Sq;
      V_North                    :in      Length_Types.Feet_Per_Sec;
      V_East                     :in      Length_Types.Feet_Per_Sec;
      Q_Bar                      :in      Float;
      Nose_Wow                   :in      Boolean;
      Force_Body_Z               :in      Float;
      Rudder_Deflection          :in      Angle_Types.Degrees;
      Pilot_Force                :in      Float;
      Dt                         :in      Float;
      An_Instance                :in out  Instance) is


      Gain_Yaw_Damper             : Float := 0.0;
      Gain_Heading_Error          : Float := 1.0;

      G                           : Float := 32.174 ;

      Temp1                       : Float := 0.0;
      Temp2                       : Float := 0.0;


      P_Heading_Comp              : Float := 0.0;
      I_Heading_Comp              : Float := 0.0;
      Yaw_Rate_Comp               : Float := 0.0;
      P_Sideslip_Comp             : Float := 0.0;

      P_Lat_Acc_Comp              : Float := 0.0;
      D_Lat_Acc_Comp              : Float := 0.0;

      Tho_Washout_Force           : Float := 10.0;
      Gain_Backdrive_Force        : Float := 3.0;

      package JSA         renames Jpats_Simulated_Aircraft;

      --| local variables for heading hold function
      Psi_TO      : Float := 0.0;
      R_Dot_Dr    : Float := 0.0;
      Error_L2    : Float := 0.0;
      R_Errd      : Float := 0.0;
      R_Err       : Float := 0.0;
      RDot_errdot : Float := 0.0;
      Dr_Dot      : Float := 0.0;
      D_Rud       : Float := 0.0;
      Er_Psi      : Float := 0.0;
      Q_Bar_T     : Float := Q_Bar;

   begin


      if Q_Bar_T < 1.0 then
         Q_Bar_T := 1.0;
      end if;

      -- -------------------------------------------------------------------
      --| Process the fadering in
      -- -------------------------------------------------------------------
      if An_Instance.Fader < 1.0 then
         An_Instance.Fader := An_Instance.Fader + Dt/Float'Max(Fader_Time,0.1);
         if An_Instance.Fader >= 1.0 then
            An_Instance.Fader := 1.0;
         end if;
      end if;


      -- -------------------------------------------------------------------
      --| Process the gains to ramp up and ramp down
      -- -------------------------------------------------------------------
      if (Altitude_AGL < Gain_Heading_Error_X1) then
         Gain_Heading_Error  := 1.0;
      elsif (Altitude_AGL > Gain_Heading_Error_X2) then
         Gain_Heading_Error := 0.0;
      else
         Gain_Heading_Error :=
           (Gain_Heading_Error_X2 - Altitude_AGL )/ (Gain_Heading_Error_X2-Gain_Heading_Error_X1);
      end if;

      if (abs(Altitude_AGL) < Gain_Yaw_Damper_x1) then
         Gain_Yaw_Damper := 0.0;
      elsif (abs(Altitude_AGL) > Gain_Yaw_Damper_x2) then
         Gain_Yaw_Damper := 1.0;
      else
         Gain_Yaw_Damper :=
           (Altitude_AGL - Gain_Yaw_Damper_x1)/(Gain_Yaw_Damper_X2 - Gain_Yaw_Damper_X1);
      end if;



      -- -------------------------------------------------------------------
      --| Determine heading component of the control law (PID)
      -- -------------------------------------------------------------------

      -- Determine the deviation from the center of the runway
      -- during auto takeoff (considered to last less than 200 s)
      if An_instance.T_TO < 200.0 then
         An_Instance.T_TO    := An_Instance.T_TO    + Dt;
         An_Instance.Lat_TO  := An_Instance.Lat_TO  + V_North*Dt;
         An_Instance.Long_TO := An_Instance.Long_TO + V_East*Dt;

         if An_Instance.T_TO > 2.0 then

            if abs(An_Instance.Lat_TO) < 0.1 then
               if An_Instance.Long_TO > 0.0 then
                  Psi_TO := 90.0;
               else
                  Psi_TO := 270.0;
               end if;
            else
               Psi_TO := Arctan(An_Instance.Long_TO,An_Instance.Lat_TO)*57.3;
            end if;
         else
              Psi_TO :=  An_Instance.Selected_Heading;
         end if;
      else
         Psi_TO :=  An_Instance.Selected_Heading;
      end if;


      if Nose_Wow then
          R_Dot_Dr := -0.0015 * Float'Max(15.0,Q_Bar_t) +
                0.000_06 * Force_Body_Z;
       else
          R_Dot_Dr := -0.0015 *Float'Max(15.0,Q_Bar_t);
       end if;

       Error_L2 := An_Instance.Error_L;
       An_Instance.Error_L := An_Instance.Error;
       Er_Psi := 2.0*An_Instance.Selected_Heading - Heading - Psi_TO;
       if Er_Psi > 180.0 then
          Er_Psi := Er_Psi - 360.0;
       elsif Er_Psi < -180.0 then
          Er_Psi := Er_Psi + 360.0;
       end if;
       An_Instance.Error := 0.0175 * Er_Psi * Cos(Roll_angle,360.0) * Cos(Pitch_angle,360.0);

       R_Err := (An_Instance.Error - An_Instance.Error_L)/Dt + 1.43 * An_Instance.Error_L * An_Instance.Heading_Gain;
       R_Errd := (An_Instance.Error - 2.0 * An_Instance.Error_L + Error_L2)/(Dt * Dt);
       Rdot_Errdot := 9.0 * An_Instance.Heading_Gain * (9.0 * An_Instance.Heading_Gain * R_Err + 2.0 * R_Errd);
       Dr_dot := - Float'Max(-100.0,Float'Min(100.0,Rdot_Errdot/R_dot_dr));
       D_Rud  := Dr_Dot * Dt;




      P_Heading_Comp := Float'Max(-Max_Rud_Rate,
                                           Float'Min(Max_Rud_Rate,Gain_Heading_Error*D_Rud));


      -- -------------------------------------------------------------------
      --| Determine lateral_acc  component of the control law
      -- -------------------------------------------------------------------
      Temp1 := Float'Min(Max_Lat_Acc*G , Float'Max(-Max_Lat_Acc*G,-Lateral_Acc));

      First_Order_Filter.Update(Temp1,
                                Tho_Filtered_Lat_Acc,
                                Dt,
                                An_Instance.Filtered_Lat_Acc);
      Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Lat_Acc);

      if (Rudder_Deflection  < 24.0) and
        (Rudder_Deflection  > -24.0) then
         An_Instance.Integral_Lat_acc := An_Instance.Integral_Lat_acc + temp2*Dt;
      end if;

      Temp1 := (Temp2 -  An_Instance.Last_Lat_acc)/Float'Max(1.0/60.0,Dt);
      An_Instance.Last_Lat_acc := temp2;


      P_Lat_Acc_Comp := An_Instance.KP_Lat_Acc*Temp2 + An_Instance.KI_Lat_Acc*An_Instance.Integral_Lat_acc;
      D_Lat_Acc_Comp := An_Instance.KD_Lat_Acc*Temp1;






      -- -------------------------------------------------------------------
      --| Determine Yaw rate component of the control law
      -- -------------------------------------------------------------------
      Temp1 := G*sin(Roll_Angle,360.0)/Float'Max(60.0*1.6787,TAS)*57.3 - Yaw_Rate ;


      First_Order_Filter.Update(Temp1,
                                Tho_Filtered_Yaw_Rate,
                                Dt,
                                An_Instance.Filtered_Yaw_Rate);
      Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Yaw_rate)/Tho_Filtered_Yaw_Rate;


      Yaw_Rate_Comp := An_Instance.Yaw_Gain*(Temp2);


      -- -------------------------------------------------------------------
      --| Determine Sideslip  component of the control law
      -- -------------------------------------------------------------------

      P_Sideslip_Comp := An_Instance.KP_Sideslip*Sideslip_Angle;


      -- -------------------------------------------------------------------
      --| Determine control  component of the yaw damper
      -- -------------------------------------------------------------------

      Temp1 := Yaw_Rate_Comp + P_Sideslip_Comp +
        P_Lat_Acc_Comp       + D_Lat_Acc_Comp ;



      Temp2 := Gain_Yaw_Damper*(Temp1 - Rudder_Deflection);



      -- -------------------------------------------------------------------
      --| Determine Rudder command
      -- -------------------------------------------------------------------
      Temp1 := An_Instance.Fader*Float'Max(-Max_Rud_Rate,
                         Float'Min(Max_Rud_Rate,Temp2 ));

      An_Instance.Commanded_Rudder_Deflection :=
        Float'Max(-24.0, Float'Min(24.0, P_Heading_Comp + Temp1 + Rudder_Deflection));





      --| -----------------------------------------------------------------------
      --| Determine the backdrive force
      --| -----------------------------------------------------------------------
      if An_Instance.Backdrive then

         if (Integer(JSA.Float1) = 124)  and JSA.Bool1 then
            Gain_Backdrive_Force  := Jsa.Float2;
            Tho_Washout_Force     := Jsa.Float3;
         end if;

         First_Order_Filter.Update(Pilot_force,
                                   Tho_Washout_Force/10.0,
                                   Dt,
                                   An_Instance.Filtered_Force);

         Temp1 := Gain_Backdrive_force*First_Order_Filter.Output(An_Instance.Filtered_Force);

         First_Order_Filter.Update(temp1,
                                   Tho_Washout_force,
                                   Dt,
                                   An_Instance.Washout_force);
         temp1  := First_Order_Filter.Output(An_Instance.Washout_force);


         An_Instance.Backdrive_force := Float'Min(100.0, Float'Max(-100.0,Temp1));

      else

         An_Instance.Backdrive_force := 0.0;

      end if;


      -- ### debug #########################################################

    if (Integer(JSA.Float1) > 120) or (Integer(JSA.Float1) < 130) then
       if    (Integer(JSA.Float8) = 1) then
          An_Instance.Debug1 := Yaw_Rate_Comp + P_Sideslip_Comp +
            P_Lat_Acc_Comp + D_Lat_Acc_comp;
       elsif (Integer(JSA.Float8) = 2) then
          An_Instance.Debug1 := Temp2;
       elsif (Integer(JSA.Float8) = 3) then
          An_Instance.Debug1 := temp1;
       elsif (Integer(JSA.Float8) = 4) then
          An_Instance.Debug1 := P_Heading_Comp;
       elsif (Integer(JSA.Float8) = 5) then
          An_Instance.Debug1 := I_Heading_Comp;
       elsif (Integer(JSA.Float8) = 6) then
          An_Instance.Debug1 := An_Instance.Commanded_Rudder_Deflection;
       elsif (Integer(JSA.Float8) = 7) then
          An_Instance.Debug1 := Rudder_deflection;
       elsif (Integer(JSA.Float8) = 8) then
          An_Instance.Debug1 := Yaw_Rate_Comp;
       elsif (Integer(JSA.Float8) = 9) then
          An_Instance.Debug1 := P_Sideslip_Comp;
       elsif (Integer(JSA.Float8) = 10) then
          An_Instance.Debug1 := P_Lat_Acc_Comp;
       elsif (Integer(JSA.Float8) = 11) then
          An_Instance.Debug1 := D_Lat_Acc_Comp;
       elsif (Integer(JSA.Float8) = 12) then
          An_Instance.Debug1 := Gain_Heading_Error;
       elsif (Integer(JSA.Float8) = 13) then
          An_Instance.Debug1 := Gain_Yaw_Damper;
       elsif (Integer(JSA.Float8) = 14) then
          An_Instance.Debug1 := Float(Boolean'Pos(Nose_Wow));
       elsif (Integer(JSA.Float8) = 15) then
          An_Instance.Debug1 := An_Instance.LAT_TO;
       elsif (Integer(JSA.Float8) = 16) then
          An_Instance.Debug1 := An_Instance.Long_TO;
       elsif (Integer(JSA.Float8) = 17) then
          An_Instance.Debug1 := An_Instance.T_TO;
       elsif (Integer(JSA.Float8) = 18) then
          An_Instance.Debug1 := Psi_TO;

       elsif (Integer(JSA.Float8) = 23) then
          An_Instance.Debug1 := An_Instance.KP_Sideslip;
       elsif (Integer(JSA.Float8) = 24) then
          An_Instance.Debug1 := An_Instance.KP_Lat_Acc;
       elsif (Integer(JSA.Float8) = 25) then
          An_Instance.Debug1 := An_Instance.KI_Lat_Acc;
       elsif (Integer(JSA.Float8) = 26) then
          An_Instance.Debug1 := An_Instance.KD_Lat_Acc;
       elsif (Integer(JSA.Float8) = 27) then
          An_Instance.Debug1 :=  An_Instance.Yaw_Gain;
       elsif (Integer(JSA.Float8) = 28) then
          An_Instance.Debug1 :=Lateral_Acc  ;
       elsif (Integer(JSA.Float8) = 30) then
          An_Instance.Debug1 :=  An_Instance.Backdrive_Force;
       elsif (Integer(JSA.Float8) = 31) then
          An_Instance.Debug1 := Pilot_Force;
       end if;


    end if;
    -- ### debug End #####################################################

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Update",
                    Severity => Log.ERROR);
         raise;
   end Update;




   function Commanded_Rudder_deflection(An_Instance :in Instance)
                                         return Angle_Types.Degrees is
   begin

      return An_Instance.Commanded_Rudder_Deflection;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Commanded_Rudder_deflection",
                    Severity => Log.ERROR);
         raise;
   end Commanded_Rudder_Deflection;
   function Is_on(An_Instance :in Instance)
                  return Boolean is
   begin

      return (An_Instance.Mode > 0);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": is_on",
                    Severity => Log.ERROR);
         raise;
   end Is_On;

   function Mode(An_Instance :in Instance) return Integer is
   begin
      return An_Instance.mode;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Mode",
                    Severity => Log.ERROR);
         raise;
   end Mode;

   function Debug1(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug1;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug1",
                    Severity => Log.ERROR);
         raise;
   end debug1;

   function Debug2(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug2;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug2",
                    Severity => Log.ERROR);
         raise;
   end debug2;

   function Debug3(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug3;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug3",
                    Severity => Log.ERROR);
         raise;
   end debug3;

   function Debug4(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug4;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug1",
                    Severity => Log.ERROR);
         raise;
   end debug4;

   procedure Set_Backdrive_Mode
     (Is_On                   :in       Boolean;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.backdrive := Is_on;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_backdrive_",
                    Severity => Log.ERROR);
         raise;

   end Set_Backdrive_mode;

   function Backdrive(An_Instance :in Instance) return Boolean  is
   begin
      return An_Instance.Backdrive;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Backdrive",
                    Severity => Log.ERROR);
         raise;
   end Backdrive;
   function Backdrive_Force (An_Instance :in Instance) return float  is
   begin
      return An_Instance.Backdrive_Force ;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Backdrive_force",
                    Severity => Log.ERROR);
         raise;
   end Backdrive_Force;

end  Yaw_Driver;



