-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Log;
with Jpats_Simulated_Aircraft;

package body torque_Driver  is
   -- General definitions -----------------------------------------------------
   Body_File_Name : constant String := "IOS_pilot/Torque_Driver.adb";


   procedure Set_Gain
     (Tuning_Mode              :in     Integer;
      KP_Torque                :in     Float;
      KI_Torque                :in     Float;
      KD_Torque                :in     Float;
      Max_Torque_Err           :in     Float;

      KP_IAS                   :in     Float;
      KI_IAS                   :in     Float;
      KD_IAS                   :in     Float;
      Max_IAS_Err              :in     Float;

      Pitch_Gain               :in     Float;
      Nominal_Torque_Comp      :in     Float;
      An_Instance              :in out Instance) is
   begin
      An_Instance.Tuning_Mode          := Tuning_Mode;
      An_Instance.KP_Torque            := KP_Torque;
      An_Instance.KI_Torque            := KI_Torque;
      An_Instance.KD_Torque            := KD_Torque;
      An_Instance.Max_Torque_Err       := Max_Torque_Err;

      An_Instance.KP_IAS               := KP_IAS;
      An_Instance.KI_IAS               := KI_IAS;
      An_Instance.KD_IAS               := KD_IAS;
      An_Instance.Max_IAS_Err          := Max_IAS_Err;

      An_Instance.Pitch_Gain           := Pitch_Gain;
      An_Instance.Nominal_Torque_Comp  := Nominal_Torque_Comp;


   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": set_gain",
                    Severity => Log.ERROR);
         raise;

   end Set_Gain;

   procedure initialize
     (An_Instance             :in out  Instance) is
   begin
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Pitch);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_IAS);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Torque);
      An_Instance.Mode := 0;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize",
                    Severity => Log.ERROR);
         raise;

   end Initialize;

   procedure Set_Mode
     (Mode                    :in       Integer;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.Mode := Mode;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_mode",
                    Severity => Log.ERROR);
         raise;

   end Set_mode;


   procedure Reset_Internal_variables
     (Pitch_Angle        :in      Angle_Types.Degrees;
      PLA                :in      Angle_Types.Degrees;
      An_Instance        :in out  Instance) is
   begin

      An_Instance.Integral_IAS_Error  := 0.0;
      An_Instance.Last_IAS_Error  := 0.0;
      An_Instance.Integral_Torque_Error  := 0.0;
      An_Instance.Commanded_Pla := PLA;
      First_Order_Filter.Initialize(Pitch_Angle,An_Instance.Filtered_Pitch);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_IAS);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Torque);
      An_Instance.Fader := 0.0;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Reset_Internal_variables",
                    Severity => Log.ERROR);
         raise;

   end Reset_Internal_variables;

   procedure Set_Reference
     (Mode                    :in      Integer;
      Selected_Torque         :in      float;
      Selected_IAS            :in      Length_Types.Knots;
      Selected_Time_Constant  :in      float;
      An_Instance             :in out  Instance) is


   begin
      if not( An_Instance.Mode = Mode) then
         An_Instance.Easy_On  := 0.0;
      end if;
      An_Instance.Mode := Mode;
      An_Instance.Selected_torque := Selected_torque;
      An_Instance.Selected_IAS := Selected_IAS;
      An_Instance.Time_constant := Selected_Time_constant;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_reference",
                    Severity => Log.ERROR);
         raise;

   end Set_Reference;


   procedure Update
     (Torque                     :in      Normalized_Types.Normalize;
      PLA                        :in      Angle_Types.Degrees;
      IAS                        :in      Length_Types.Knots;
      Pitch_Angle                :in      Angle_Types.Degrees;
      Flap_Pos                   :in      Angle_Types.Degrees;
      Gear_Pos                   :in      Float;
      Nx                         :in      Float;
      Dt                         :in      Float;
      An_Instance                :in out  Instance) is

      Temp1                       : Float := 0.0;
      Temp2                       : Float := 0.0;
      Comm_Torque                 : Float := 0.0;

      Tho_Filtered_Torque         : Float := 2.0;

      Pitch_Comp                  : Float := 0.0;
      P_IAS_Comp                  : Float := 0.0;
      I_IAS_Comp                  : Float := 0.0;
      D_IAS_Comp                  : Float := 0.0;
      P_Torque_Comp               : Float := 0.0;
      I_Torque_Comp               : Float := 0.0;
      D_Torque_Comp               : Float := 0.0;

       -- ### debug #########################################################
      package JSA         renames Jpats_Simulated_Aircraft;
      -- ### debug End #####################################################

   begin
      -- ### debug #########################################################
      if An_Instance.Tuning_Mode = 2 then
         An_Instance.Mode := 2;
      elsif An_Instance.Tuning_Mode = 1 then
         An_Instance.Mode := 1;

      end if;
      -- ### debug End #####################################################

      -- Process the fadering in
      if An_Instance.Fader < 1.0 then
         An_Instance.Fader := An_Instance.Fader + Dt/Float'Max(Fader_Time,0.1);
         if An_Instance.Fader >= 1.0 then
            An_Instance.Fader := 1.0;
         end if;
      end if;

      --| -----------------------------------------------------------------------
      --|  IAS SELECT MODE
      --| -----------------------------------------------------------------------

      -- Determine the contribution of IAS for commanded torque
      if An_Instance.Mode = 1 then
         An_Instance.Selected_IAS := IAS;
         An_Instance.Integral_IAS_Error := An_Instance.Integral_IAS_Error*(1.0 - 0.33*Dt);
         First_Order_Filter.Initialize(0.0,An_Instance.Filtered_IAS);
         First_Order_Filter.Initialize(Pitch_Angle,An_Instance.Filtered_Pitch);
         First_Order_Filter.Initialize(An_Instance.Pitch_Gain*Pitch_Angle,An_Instance.Washout_Pitch);

      else
         Temp1 :=  An_Instance.Selected_IAS - IAS;
         Temp2 := Float'Max(-An_Instance.Max_IAS_Err,Float'Min(An_Instance.Max_IAS_Err,Temp1));
         First_Order_Filter.Update(Temp2,
                                   Tho_Filtered_IAS,
                                   Dt,
                                   An_Instance.Filtered_IAS);

         Temp1 := First_Order_Filter.Output(An_Instance.Filtered_IAS);

         if ((Torque > 1.0) and (Torque < 99.0)) and
           ((PLA > 19.0) and  (PLA < 52.0))  and
           (abs(An_Instance.Selected_Torque - Torque)< An_Instance.Max_Torque_Err) then
            An_Instance.Integral_IAS_Error := An_Instance.Integral_IAS_Error + Temp1*Dt;
         end if;

         P_IAS_Comp := An_Instance.KP_IAS*Temp1;
         I_IAS_Comp := An_Instance.KI_IAS*An_Instance.Integral_IAS_Error;
         D_IAS_Comp := An_Instance.KD_IAS*Nx;


         -- Determine the contribution of Pitch for commanded PLA

         First_Order_Filter.Update(Pitch_Angle,
                                   Tho_Filtered_Pitch,
                                   Dt,
                                   An_Instance.Filtered_Pitch);

         Temp1 := First_Order_Filter.Output(An_Instance.Filtered_Pitch);

         Temp2 := An_Instance.Pitch_Gain*Temp1;

         First_Order_Filter.Update(temp2,
                                   Tho_Washout_Pitch,
                                   Dt,
                                   An_Instance.Washout_Pitch);
         Pitch_Comp  := First_Order_Filter.Output(An_Instance.Washout_Pitch);



         An_Instance.Selected_Torque :=
           Float'Max(0.0,Float'Min(100.0,An_Instance.Nominal_Torque_Comp + Pitch_Comp +
                                   P_IAS_Comp + I_IAS_Comp + D_IAS_Comp)) ;


      end if;

      --| -----------------------------------------------------------------------
      --| TORQUE SELECT MODE
      --| -----------------------------------------------------------------------

      Comm_torque := An_Instance.Selected_Torque;

      -- Determine the contribution of torque for commanded PLA
      Temp1 :=  Comm_Torque  - Torque;

      Temp2 := Float'Max(-An_Instance.Max_Torque_Err,Float'Min(An_Instance.Max_Torque_Err,Temp1));

      Tho_Filtered_Torque := An_Instance.Time_constant;

      --| -----------------------------------------------------------------------
      --| Process the torque easy-on during mode change
      --| -----------------------------------------------------------------------

      if An_Instance.Easy_On  < 1.0 then
         An_Instance.Easy_On  := An_Instance.Easy_On + Dt/12.0;
         Tho_Filtered_Torque :=2.5;
         if An_Instance.Easy_On >= 1.0 then
            An_Instance.Easy_On  := 1.0;
         end if;
      end if;

      First_Order_Filter.Update(Temp2,
                                Tho_Filtered_Torque,
                                Dt,
                                An_Instance.Filtered_torque);

      Temp1 := First_Order_Filter.Output(An_Instance.Filtered_torque);

      Temp2 := (Temp1 - An_Instance.Last_Torque_Error)/Float'Max(1.0/60.0,Dt);

      if (PLA > 19.0) and
        (PLA < 52.0) then
         An_Instance.Integral_Torque_Error := An_Instance.Integral_Torque_Error + Temp1*Dt;
      end if;



      P_Torque_Comp := An_Instance.KP_Torque*Temp1/10.0;
      I_Torque_Comp := An_Instance.KI_Torque*An_Instance.Integral_Torque_Error/10.0;
      I_Torque_Comp := An_Instance.KD_Torque*Temp2/10.0;

      An_Instance.Last_Torque_Error := Temp1;
      Temp1 := P_Torque_Comp + I_Torque_Comp + D_Torque_Comp;

      Temp2 := An_Instance.Fader*Float'Max(-Max_Delta_PLA*Dt,Float'Min(Max_Delta_PLA*Dt,Temp1));

      An_Instance.Commanded_PLA := Float'Max(18.0, Float'Min(53.0, Temp2 + PLA));






      -- ### debug #########################################################
      if (Integer(JSA.Float1) = 132)  then
         JSA.Set_Float2to9
           (Float2 => An_Instance.Selected_IAS,
            Float3 => An_Instance.Selected_torque,
            Float4 => An_Instance.Nominal_Torque_comp,
            Float5 => Pitch_Comp ,
            Float6 => P_IAS_Comp,
            Float7 => I_IAS_Comp,
            Float8 => D_IAS_Comp,
            Float9 => torque,
            Float10=> IAS);
      end if;


      if (Integer(JSA.Float1) = 131)  then
         if    (Integer(JSA.Float8) = 1) then
            An_Instance.Debug1 := An_Instance.Selected_IAS;
         elsif (Integer(JSA.Float8) = 2) then
            An_Instance.Debug1 := An_Instance.Selected_Torque;
         elsif (Integer(JSA.Float8) = 3) then
            An_Instance.Debug1 := An_Instance.Nominal_Torque_Comp;
         elsif (Integer(JSA.Float8) = 4) then
            An_Instance.Debug1 := Pitch_Comp;
         elsif (Integer(JSA.Float8) = 5) then
            An_Instance.Debug1 := P_IAS_Comp;
         elsif (Integer(JSA.Float8) = 6) then
            An_Instance.Debug1 := I_IAS_Comp;
         elsif (Integer(JSA.Float8) = 7) then
            An_Instance.Debug1 := D_IAS_Comp;
         elsif (Integer(JSA.Float8) = 8) then
            An_Instance.Debug1 := Torque;
         elsif (Integer(JSA.Float8) = 9) then
            An_Instance.Debug1 := Pitch_Angle;
         elsif (Integer(JSA.Float8) = 10) then
            An_Instance.Debug1 := An_Instance.Integral_IAS_Error;
         elsif (Integer(JSA.Float8) = 11) then
            An_Instance.Debug1 := First_Order_Filter.Output(An_Instance.Filtered_Pitch);



         elsif (Integer(JSA.Float8) = 12) then
            An_Instance.Debug1 := First_Order_Filter.Output(An_Instance.Washout_Pitch);
         elsif (Integer(JSA.Float8) = 13) then
            An_Instance.Debug1 := First_Order_Filter.Output(An_Instance.Filtered_IAS);
         elsif (Integer(JSA.Float8) = 14) then
            An_Instance.Debug1 := Nx;
         elsif (Integer(JSA.Float8) = 15) then
            An_Instance.Debug1 := An_Instance.Easy_On;

         end if;


      end if;
      -- ### debug End #####################################################


   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Update",
                    Severity => Log.ERROR);
         raise;
   end Update;

   function Commanded_PLA(An_Instance :in Instance)
                          return Angle_Types.Degrees is
   begin

      return An_Instance.Commanded_PLA;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Commanded__PLA",
                    Severity => Log.ERROR);
         raise;
   end Commanded_PLA;

   function Is_on(An_Instance :in Instance)
                  return Boolean is
   begin

      return (An_Instance.Mode > 0);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": is_on",
                    Severity => Log.ERROR);
         raise;
   end Is_On;

   function Established(An_Instance :in Instance)
                     return Boolean is
   begin

      return (An_Instance.fader > 0.9);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Fadering",
                    Severity => Log.ERROR);
         raise;
   end Established;

   function Mode(An_Instance :in Instance) return Integer is
   begin
      return An_Instance.mode;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Mode",
                    Severity => Log.ERROR);
         raise;
   end Mode;

   procedure Set_Debug1
     (Debug1                  :in       float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.debug1 := debug1;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_debug1 ",
                    Severity => Log.ERROR);
         raise;

   end Set_debug1;

   function Debug1(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug1;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug1",
                    Severity => Log.ERROR);
         raise;
   end debug1;

   procedure Set_Debug2
     (Debug2                  :in       float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.debug2 := debug2;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_debug2 ",
                    Severity => Log.ERROR);
         raise;

   end Set_debug2;

   function Debug2(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug2;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug2",
                    Severity => Log.ERROR);
         raise;
   end debug2;

   function Debug3(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug3;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug3",
                    Severity => Log.ERROR);
         raise;
   end debug3;

   function Debug4(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug4;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug1",
                    Severity => Log.ERROR);
         raise;
   end debug4;


   function Selected_IAS(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.Selected_IAS;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Selected_IAS",
                    Severity => Log.ERROR);
         raise;
   end Selected_IAS;

   function Selected_Torque(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.Selected_Torque;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Selected_Torque",
                    Severity => Log.ERROR);
         raise;
   end Selected_Torque;
end  Torque_Driver;



