-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Log;
with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;

-- ### debug #########################################################
with Jpats_Simulated_Aircraft;
-- ### debug End #####################################################

package body Roll_Driver  is

   -- General definitions -----------------------------------------------------
   Body_File_Name : constant String := "IOS_pilot/Roll_Driver.adb";

   -- Package Local variables needed for turn off/on mode
   Max_Heading                 : Float := 360.0 - HEADING_DELTA(2);

   package IT          renames Interpolation_Table;

   IAS20_T             :aliased  IT.doubly_Indexed.Instance;
   GAINTURNX2_T        :aliased  IT.doubly_Indexed.Instance;

   procedure Set_Gain
     (Tuning_Mode             : in Integer;
      Comm_Roll               : in Float;
      KP_Roll                 : in Float;
      KI_Roll                 : in Float;
      KD_Roll                 : in Float;
      Max_Roll_Rate           : in Float;
      Comm_Turn_Rate          : in Float;
      KP_Turn_Rate            : in Float;
      KI_Turn_Rate            : in Float;
      Delta_Max_Selected_Turn : in Float;
      Max_Turn_Err            : in Float;
      KP_Heading              : in Float;
      KI_Heading              : in Float;
      KD_Heading              : in Float;
      Max_Heading_Err         : in Float;
      An_Instance             : in out Instance) is
   begin
      An_Instance.Tuning_Mode             := Tuning_Mode;
      An_Instance.Comm_Roll               := Comm_Roll;
      An_Instance.KP_Roll                 := KP_Roll;
      An_Instance.KI_Roll                 := KI_Roll;
      An_Instance.KD_Roll                 := KD_Roll;
      An_Instance.Max_Roll_Rate           := Max_Roll_Rate;
      An_Instance.Comm_Turn_Rate          := Comm_Turn_Rate;
      An_Instance.KP_Turn_Rate            := KP_Turn_Rate;
      An_Instance.KI_Turn_Rate            := KI_Turn_Rate;
      An_Instance.Delta_Max_Selected_Turn := Delta_Max_Selected_Turn;
      An_Instance.Max_Turn_Err            := Max_Turn_Err;
      An_Instance.KP_Heading              := KP_Heading;
      An_Instance.KI_Heading              := KI_Heading;
      An_Instance.KD_Heading              := KD_Heading;
      An_Instance.Max_Heading_Err         := Max_Heading_Err;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize",
                    Severity => Log.ERROR);
         raise;

   end Set_Gain;

   procedure initialize
     (An_Instance             :in out  Instance) is
      -- Point to data table directory
      File_Path : String := Simulation_Dictionary.Lookup("IOS_Pilot_Dir");
   begin
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Roll);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Turn );
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Heading_Error);
      An_Instance.Mode := 0;

      It.Read(File_Path & "ias20.ito",IAS20_T);
      It.Read(File_Path & "gturnx2.ito",GAINTURNX2_T);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize",
                    Severity => Log.ERROR);
         raise;

   end Initialize;

   procedure Init_Rapid_Direction_Change(
      Roll_Angle              :in Angle_Types.Degrees;      
      An_Instance             :in out  Instance) is
   begin
      First_Order_Filter.Initialize(Roll_Angle,An_Instance.Filtered_Selected_Roll);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Turn );
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Heading_Error);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Init_Rapid_Direction_Change",
                    Severity => Log.ERROR);
         raise;

   end Init_Rapid_Direction_Change;

  procedure Set_Mode
     (Mode                    :in       Integer;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.Mode := Mode;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_mode",
                    Severity => Log.ERROR);
         raise;

   end Set_mode;


  procedure Reset_Internal_variables
     (Roll_Angle         :in      Angle_Types.Degrees;
      Roll_Rate          :in      Angle_Types.Degrees_Per_Sec;
      Aileron_Deflection :in      Angle_Types.Degrees;
      An_Instance        :in out   Instance) is
   begin
      An_Instance.Commanded_Roll           := Roll_Angle;
      An_Instance.Commanded_Roll_Rate      := Roll_Rate;
      An_Instance.Commanded_Aileron_Deflection   := Aileron_Deflection;
      An_Instance.Integral_Turn_Rate_error := 0.0;
      An_Instance.Integral_Roll_Error      := 0.0;
      An_Instance.Integral_Heading_Error   := 0.0;
      An_Instance.Fader := 0.0;
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Heading_Error);
      An_Instance.Easy_On  := 0.0;


   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Reset_Internal_variables",
                    Severity => Log.ERROR);
         raise;

   end Reset_Internal_variables;

   procedure Set_Reference
     (Mode                    :in      Integer;
      Steady_Turn             :in      Boolean;
      Selected_Heading        :in      Angle_Types.Degrees;
      Heading                 :in      Angle_Types.Degrees;
      Selected_Turn_Rate      :in      Integer;
      Selected_Turn_Direction :in      Integer;
      Selected_Climb_rate     :in      Length_Types.Feet_Per_Min;
      Selected_IAS            :in      Length_Types.Knots;
      Selected_Torque         :in      Float;
      Performance_Mode        :in      Integer;
      An_Instance             :in out  Instance) is

      Turn_Direction  : Float := 1.0;
      IAS             : Float := 100.0;
      Target_Heading  : Float := Selected_Heading;

   begin
      An_Instance.Mode := Mode;
      Max_Heading := 360.0 - HEADING_DELTA(Selected_Turn_Rate);
      Heading_Buffer := HEADING_DELTA(Selected_Turn_Rate);
      if (Selected_Turn_Direction = 3) then
         --| Right turn
         if Steady_Turn then
            if Heading <= Max_Heading then
               Target_Heading := Heading + Heading_Buffer;
            else 
               Target_Heading := Heading_Buffer - (360.0 - Heading);
            end if;
         end if;
         --| Target in the left side but the turn is to the right
         if (Heading >  Target_Heading + 1.0 ) and
            ((Heading - Target_Heading ) <  (360.0 +  Target_Heading - Heading)) then
            An_Instance.Mode_Catch_Up_Is_On := False;
         else
            An_Instance.Mode_Catch_Up_Is_On := True;
         end if;
         Turn_Direction := 1.0;
      elsif (Selected_Turn_Direction = 2) then
         if Steady_Turn then
            if Heading >= Heading_Buffer then
               Target_Heading := Heading - Heading_Buffer;
            else 
               Target_Heading := 360.0 - (Heading_Buffer - Heading);
            end if;
         end if;
         --| Target in the right side but the turn is to the left
         if (Heading <  Target_Heading - 1.0) and
           ((Target_Heading - Heading) < (360.0 + Heading - Target_Heading )) then
            An_Instance.Mode_Catch_Up_Is_On := False;
         else
            An_Instance.Mode_Catch_Up_Is_On := True;
         end if;
         Turn_Direction := -1.0;
      else
         if ((Target_Heading - Heading) > 0.0) then
            --| Target in the right side
            if ((Target_Heading - Heading) < (360.0 + Heading - Target_Heading )) then
               Turn_Direction := 1.0;
            else
               Turn_Direction := -1.0;
            end if;
         else
            --| Target in the left side
            if ((Heading - Target_Heading ) <  (360.0 +  Target_Heading - Heading)) then
               Turn_Direction  := -1.0;
            else
               Turn_Direction  := 1.0;
            end if;
         end if;
         An_Instance.Mode_Catch_Up_Is_On := True;
      end if;
      An_Instance.Selected_Turn_Rate := 1.5*Float(Selected_Turn_Rate)*Turn_Direction;
      An_Instance.Selected_Heading := Target_Heading;
      An_Instance.Heading_Hold_Is_On := false;


      case Performance_mode is
         when 2 =>
            IAS        := It.Doubly_Indexed.Interpolate(Selected_torque,Selected_Climb_rate,IAS20_T'access);
         when others =>
            IAS        := Selected_IAS;
      end case;

      An_Instance.Gain_Selected_Turn_Rate_X2 := It.Doubly_Indexed.Interpolate
        (abs(An_Instance.Selected_Turn_Rate),IAS,GAINTURNX2_T'access);

      An_Instance.Gain_Selected_Turn_Rate_X1 := An_Instance.Gain_Selected_Turn_Rate_X2/20.0;

      An_Instance.Gain_Heading_Error_X2      := An_Instance.Gain_Selected_Turn_Rate_X2;
      An_Instance.Gain_Heading_Error_X1      := An_Instance.Gain_Selected_Turn_Rate_X1;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_Reference",
                    Severity => Log.ERROR);
         raise;

   end Set_Reference;



   procedure Update
     (Turn_Rate          : in      Angle_Types.Degrees_Per_Sec;
      Steady_Right_Turn  : in      Boolean;
      Steady_Left_Turn   : in      Boolean;
      Heading            : in      Angle_Types.Degrees;
      Roll_Rate          : in      Angle_Types.Degrees_Per_Sec;
      Roll_Angle         : in      Angle_Types.Degrees;
      Pitch_Angle        : in      Angle_Types.Degrees;
      Aileron_Deflection : in      Angle_Types.Degrees;
      TAS                : in      Length_Types.Feet_Per_Sec;
      Altitude_AGL       : in      Length_Types.Feet;
      Flap_Pos           : in      Float;
      Gear_Pos           : in      Float;
      Pilot_Force        : in      Float;
      Dt                 : in      Float;
      An_Instance        : in out  Instance) is


      Gain_Selected_Turn_Rate     : Float := 0.0;


      Gain_Heading_Error          : Float := 1.0;


      Gain_Wing_level             : Float := 0.0;
      Gain_Aileron_Effectiveness  : Float := 0.0;

      G                           : Float := 32.174;

      Selected_Turn_Rate          : Float := 0.0;

      Heading_Err                 : Float := 0.0;
      Corrected_Heading_Err       : Float := 0.0;
      Turn_Rate_Err               : Float := 0.0;
      Temp1                       : Float := 0.0;
      Temp2                       : Float := 0.0;

      P_Heading_Comp              : Float := 0.0;
      I_Heading_Comp              : Float := 0.0;
      D_Roll_Comp                 : Float := 0.0;
      P_Roll_Comp                 : Float := 0.0;
      I_Roll_Comp                 : Float := 0.0;
      P_Turn_Rate_Comp            : Float := 0.0;
      I_Turn_Rate_Comp            : Float := 0.0;


      Comm_Roll                   : Float := 0.0;
      Gain_Backdrive_Force        : Float := 2.5;
      Tho_Filtered_Force          : Float := 2.0;



      -- ### debug #########################################################
      package JSA         renames Jpats_Simulated_Aircraft;
      -- ### debug End #####################################################

   begin

      --| -----------------------------------------------------------------------
      --| Check for Steady Turn
      --| -----------------------------------------------------------------------

      if Steady_Right_Turn then
         if Heading <= Max_Heading then
            An_Instance.Selected_Heading := Heading + Heading_Buffer;
         else 
            An_Instance.Selected_Heading := Heading_Buffer - (360.0 - Heading);
         end if;
      elsif Steady_Left_Turn then
         if Heading >= Heading_Buffer then
            An_Instance.Selected_Heading := Heading - Heading_Buffer;
         else 
            An_Instance.Selected_Heading := 360.0 - (Heading_Buffer - Heading);
         end if;
      end if;

      --| -----------------------------------------------------------------------
      --| Process the roll easy-on during mode change (12 s)
      --| -----------------------------------------------------------------------

      if An_Instance.Easy_On  < 1.0 then
         An_Instance.Easy_On  := An_Instance.Easy_On + Dt/12.0;
         if An_Instance.Easy_On >= 1.0 then
            An_Instance.Easy_On  := 1.0;
         end if;
      end if;

      -- -------------------------------------------------------------------
      -- Process the fadering in
      -- -------------------------------------------------------------------
      if An_Instance.Fader < 1.0 then
         An_Instance.Fader := An_Instance.Fader + Dt/Float'Max(Fader_Time,0.1);
         if An_Instance.Fader >= 1.0 then
            An_Instance.Fader := 1.0;
         end if;
      end if;

      Heading_Err := Heading - An_Instance.Selected_Heading;

      if Heading_err > 180.0 then
         Heading_err := Heading_err - 360.0;
      elsif Heading_err < -180.0 then
         Heading_err := Heading_err + 360.0;
      end if;


      -- -------------------------------------------------------------------
      -- The following logic is to enable turn to the right or to
      -- the left by jumping the threshold of heading hold zone.
      -- -------------------------------------------------------------------
      if (not An_Instance.Mode_Catch_Up_Is_On) then
         if (Heading_err >  0.0) then
            Corrected_Heading_Err := Heading_Err + 90.0;
         else
            Corrected_Heading_Err := Heading_Err - 90.0;
         end if;
         if abs(Corrected_Heading_Err) > 90.0 + 2.0*An_Instance.Gain_Selected_Turn_Rate_X2  then
            An_Instance.Mode_Catch_Up_Is_On := True;
         end if;
      else
         Corrected_Heading_Err := Heading_Err;
      end if;

      -- -------------------------------------------------------------------
      -- Update the gains to ramp up or ramp down
      -- -------------------------------------------------------------------

      if (abs(Corrected_Heading_Err)< An_Instance.Gain_Selected_Turn_Rate_X1) then
         Gain_Selected_Turn_Rate := 0.0;
      elsif (abs(Corrected_Heading_Err) >  An_Instance.Gain_Selected_Turn_Rate_X2) then
         Gain_Selected_Turn_Rate := 1.0;
      else
         Gain_Selected_Turn_Rate := (abs(Corrected_Heading_Err) - An_Instance.Gain_Selected_Turn_Rate_X1)/
           (An_Instance.Gain_Selected_Turn_Rate_X2 -  An_Instance.Gain_Selected_Turn_Rate_X1);
      end if;

      if (abs(Corrected_Heading_Err) < An_Instance.Gain_Heading_Error_X1) then
         Gain_Heading_Error := 1.0;
      elsif (abs(Corrected_Heading_Err) > An_Instance.Gain_Heading_Error_X2) then
         Gain_Heading_Error := 0.0;
      else
         Gain_Heading_Error := (An_Instance.Gain_Heading_Error_X2 - abs(Corrected_Heading_Err))/
           (An_Instance.Gain_Heading_Error_X2 - An_Instance.Gain_Heading_Error_X1);
      end if;

      if (abs(Altitude_AGL)< Gain_Wing_Level_X1) then
         Gain_Wing_Level := 0.0;
      elsif (abs(Altitude_AGL) >  Gain_Wing_Level_X2) then
         Gain_Wing_Level := 1.0;
      else
         Gain_Wing_Level := (abs(Altitude_AGL) - Gain_Wing_Level_X1)/
           (Gain_Wing_Level_X2 -  Gain_Wing_Level_X1);
      end if;


      if (TAS < Gain_Aileron_Effectiveness_X1) then
         Gain_Aileron_Effectiveness := 0.0;
      elsif (TAS  >  Gain_Aileron_Effectiveness_X2) then
         Gain_Aileron_Effectiveness := 1.0;
      else
         Gain_Aileron_Effectiveness := (TAS - Gain_Aileron_Effectiveness_X1)/
           (Gain_Aileron_Effectiveness_X2 -  Gain_Aileron_Effectiveness_X1);
      end if;



      --| Heading hold is engaged?
      --| Once it is engaged, it will keep engaged, until a new heading
      --| target is inputed.

      if An_Instance.Mode_Catch_Up_Is_On then
         if not An_Instance.Heading_Hold_Is_On then
            if (abs(Corrected_Heading_Err) <  An_Instance.Gain_Heading_Error_X2) then
               An_Instance.Heading_Hold_Is_On := True;
		An_Instance.Heading_Hold_Is_Locked := false;
            else
               An_Instance.Heading_Hold_Is_Locked := false;
            end if;

         else
            if not An_Instance.Heading_Hold_Is_Locked then
               if (Gain_Heading_Error > 0.9) then
                  An_Instance.Heading_Hold_Is_Locked := True;
               end if;
            else
               Gain_Heading_Error := 1.0;
               Gain_Selected_Turn_Rate := 0.0;
            end if;

         end if;
      else
         An_Instance.Heading_Hold_Is_On := false;
         Gain_Heading_Error             := 0.0;
         Gain_Selected_Turn_Rate        := 1.0;
      end if;

       -- ### debug #########################################################
      if An_Instance.Tuning_Mode = 2 then
         An_Instance.Selected_Turn_Rate   := An_Instance.Comm_Turn_Rate;
         Corrected_Heading_Err            := 30.0;
         An_Instance.KP_Heading                       := 0.0; -- 60.0
         An_Instance.KI_Heading                       := 0.0; --10.0;
         An_Instance.KD_Heading                       := 0.0; -- 60.0

         Gain_Heading_Error := 0.0;
         Gain_Selected_Turn_Rate := 1.0;
      end if;
      -- ### debug End #####################################################





      -- -------------------------------------------------------------------
      -- Process the contribution of the turn rate component
      -- -------------------------------------------------------------------

        if An_Instance.Heading_Hold_Is_On then                                   
             An_Instance.Selected_Turn_Rate := Gain_Selected_Turn_Rate*An_Instance.Selected_Turn_Rate;
        end if;



      Temp2 := Gain_Selected_Turn_Rate*An_Instance.Selected_Turn_Rate;
      Temp1 :=  Temp2 - Heading_Err*Gain_Heading_Error*An_Instance.KD_Heading;


      if abs(Temp1 - First_Order_Filter.Output
             (An_Instance.Filtered_Selected_turn)) > An_Instance.Delta_Max_Selected_Turn*Dt then
         if Temp1 > First_Order_Filter.Output
             (An_Instance.Filtered_Selected_turn) then
            Temp1 := First_Order_Filter.Output
             (An_Instance.Filtered_Selected_turn) + An_Instance.Delta_Max_Selected_Turn*Dt;
         else
            Temp1 := First_Order_Filter.Output
             (An_Instance.Filtered_Selected_turn) - An_Instance.Delta_Max_Selected_Turn*Dt;
         end if;
      end if;


      First_Order_Filter.Update(temp1,
                                Tho_Filtered_Selected_turn,
                                Dt,
                                An_Instance.Filtered_Selected_turn);

      Selected_Turn_Rate := First_Order_Filter.Output(An_Instance.Filtered_Selected_turn);


      Turn_Rate_Err :=Float'Max(-An_Instance.Max_Turn_Err,
                                Float'Min(An_Instance.Max_Turn_Err,
                                          (Selected_Turn_Rate - Turn_Rate)));

      if (Gain_Wing_Level < 0.1) or (Gain_Aileron_Effectiveness < 0.1) then
         An_Instance.Integral_Turn_Rate_Error :=
           An_Instance.Integral_Turn_Rate_Error*(1.0 - 0.33*Dt);
      else
         if ((An_Instance.Commanded_Roll < Max_Roll_Angle) and (An_Instance.Commanded_Roll > -Max_Roll_Angle)) and
           ((Aileron_Deflection > -20.0) and (Aileron_Deflection < 11.0)) then
            An_Instance.Integral_Turn_Rate_Error :=
              An_Instance.Integral_Turn_Rate_Error + Turn_Rate_Err*Dt;
         end if;
      end if;
      P_Turn_Rate_Comp  := An_Instance.KP_Turn_Rate*Turn_Rate_Err;
      I_Turn_Rate_Comp  := An_Instance.KI_Turn_Rate*An_Instance.Integral_Turn_Rate_Error;



      Temp2 := Arctan(TAS*Gain_Selected_Turn_Rate*Selected_Turn_Rate/(G*57.3))*57.3;



      Comm_Roll := Gain_Wing_Level*(P_Turn_Rate_Comp +  I_Turn_Rate_Comp + Temp2);


      -- ### TUNING #########################################################
      if An_Instance.Tuning_Mode = 1 then
         An_Instance.KI_Heading           := 0.0; --10.0;
         An_Instance.KP_Heading           := 0.0; -- 60.0;
         Comm_Roll            := An_Instance.Comm_Roll;
         Gain_Heading_Error   := 0.0;
      end if;

      -- ### TUNING End #####################################################


      -- -------------------------------------------------------------------
      -- Process the commanded roll angle and rate
      -- -------------------------------------------------------------------
      if (abs(An_Instance.Selected_Turn_Rate) > 4.0) and
        (Flap_Pos < 20.0) and (Gear_Pos < 0.5) then
         Temp1 := Float'Min(Max_Roll_Angle*1.5 , Float'Max(-Max_Roll_Angle*1.5, Comm_Roll));
      else
         Temp1 := Float'Min(Max_Roll_Angle , Float'Max(-Max_Roll_Angle, Comm_Roll));
      end if;


      if An_Instance.Easy_On  < 1.0 then
        temp2 := 10.0;
      else
        temp2 := 40.0;
      end if;

      if abs(Temp1 - An_Instance.Commanded_Roll) > temp2*An_Instance.Max_Roll_Rate*dt then
         if (Temp1 - An_Instance.Commanded_Roll) > 0.0 then
            Temp1 := An_Instance.Commanded_Roll + temp2*An_Instance.Max_Roll_Rate*dt;
         else
            Temp1 := An_Instance.Commanded_Roll  - temp2*An_Instance.Max_Roll_Rate*dt;
         end if;
      end if;

      First_Order_Filter.Update(Temp1,
                                Tho_Filtered_Selected_Roll,
                                Dt,
                                An_Instance.Filtered_Selected_Roll);



      Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Selected_Roll);


      Temp1 := (Temp2 - An_Instance.Commanded_Roll)/Float'Max(Dt,1.0/60.0);
      An_Instance.Commanded_Roll_Rate := Gain_Wing_Level*
        (Float'Min(An_Instance.Max_Roll_Rate,Float'Max(-An_Instance.Max_Roll_Rate,Temp1)));
      An_Instance.Commanded_Roll      := (An_Instance.Commanded_Roll + An_Instance.Commanded_Roll_Rate* Dt);

      -- -------------------------------------------------------------------
      -- Process the contribution of the heading hold component
      -- -------------------------------------------------------------------
      First_Order_Filter.Update(Corrected_Heading_Err*Gain_Heading_Error*Gain_Wing_Level,
                                Tho_Filtered_Heading_error,
                                Dt,
                                An_Instance.Filtered_Heading_Error);
      Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Heading_Error);

      Temp1 := Float'Max(-An_Instance.Max_Heading_Err,Float'Min(An_Instance.Max_Heading_Err,Temp2));



      if (Gain_Heading_Error  < 0.1) or (Gain_Aileron_Effectiveness < 0.1) then
         An_Instance.Integral_Heading_Error :=
           An_Instance.Integral_Heading_Error *(1.0 - 0.33*Dt);
      else
         if (Aileron_Deflection > -20.0) and
           (Aileron_Deflection < 11.0) then
            An_Instance.Integral_Heading_Error :=
              An_Instance.Integral_Heading_Error + temp1*Dt;

                if (abs(An_Instance.KI_Heading*An_Instance.Integral_Heading_Error) >20.0) then
                        if (An_Instance.KI_Heading*An_Instance.Integral_Heading_Error > 0.0) then
                                An_Instance.Integral_Heading_Error := 20.0/An_Instance.KI_Heading;
                        else
                                An_Instance.Integral_Heading_Error := -20.0/An_Instance.KI_Heading;
                        end if;
                end if;


         end if;
      end if;




      P_Heading_Comp := An_Instance.KP_Heading*Temp1;
      I_Heading_Comp := An_Instance.KI_Heading*An_Instance.Integral_Heading_Error;


      -- -------------------------------------------------------------------
      -- Process the P.I.D. component of roll hold
      -- -------------------------------------------------------------------
      Temp2 := Roll_Rate - An_Instance.Commanded_Roll_Rate;


      D_Roll_Comp := Temp2*An_Instance.KD_Roll;


      Temp2 := Roll_Angle - An_Instance.Commanded_Roll;

      if  (Gain_Aileron_Effectiveness < 0.1)  then
         An_Instance.Integral_Roll_Error :=
           An_Instance.Integral_Roll_Error *(1.0 - 0.1*Dt);
      else
         if (Aileron_Deflection > -20.0) and
           (Aileron_Deflection < 11.0) then

            An_Instance.Integral_Roll_Error :=
              An_Instance.Integral_Roll_Error + Temp2*Dt;

            if (abs(An_Instance.Integral_Roll_Error * An_Instance.KI_Roll) >20.0) then
               if (An_Instance.Integral_Roll_Error * An_Instance.KI_Roll > 0.0) then
                  An_Instance.Integral_Roll_Error := 20.0/An_Instance.KI_Roll;
               else
                  An_Instance.Integral_Roll_Error := -20.0/An_Instance.KI_Roll;
               end if;
            end if;



         end if;
      end if;

      P_Roll_Comp := Temp2*An_Instance.KP_Roll;
      I_Roll_Comp := An_Instance.Integral_Roll_Error * An_Instance.KI_Roll;


      -- -------------------------------------------------------------------
      -- update the aileron command
      -- -------------------------------------------------------------------


      Temp1 := P_Roll_Comp  + I_Roll_Comp + D_Roll_Comp +
        P_Heading_Comp + I_Heading_Comp;



      Temp2 := Gain_Aileron_Effectiveness*
        Float'Max(-Max_Ail_Rate,Float'Min(Max_Ail_Rate,Temp1));

      An_Instance.Commanded_Aileron_Deflection :=
        Float'Max(-20.0, Float'Min(11.0, Temp2 + Aileron_Deflection));



      --| -----------------------------------------------------------------------
      --| Determine the backdrive force
      --| -----------------------------------------------------------------------
      if An_Instance.Backdrive then

         if (Integer(JSA.Float1) = 114)  and JSA.Bool1 then
            Gain_Backdrive_Force  := Jsa.Float4;
            Tho_Filtered_Force    := JSA.Float5;
         end if;

         Temp1 := Gain_Wing_Level*Float'Min( 100.0, Float'Max(- 100.0,Gain_Backdrive_Force*Pilot_force));

         An_Instance.Backdrive_Force := An_Instance.Backdrive_Force + Tho_Filtered_Force*Dt*(Temp1 - An_Instance.Backdrive_Force);


      else

         An_Instance.Backdrive_force := 0.0;

      end if;




      -- ### debug #########################################################

      if (Integer(JSA.Float1) = 200)  and JSA.Bool1 then
         JSA.Set_Float2to9(
                           Float2 => Comm_Roll,
                           Float3 => P_Turn_Rate_Comp,
                           Float4 => I_Turn_Rate_Comp,
                           Float5 => P_Heading_Comp,
                           Float6 => I_Heading_Comp,
                           Float7 => P_Roll_Comp,
                           Float8 => I_Roll_Comp,
                           Float9 => D_Roll_Comp,
                           Float10=> An_Instance.Commanded_Aileron_Deflection);
      end if;



      for I in 1 .. 2 loop
         if I = 1 then
            Temp1 := an_instance.Debug1;
         else
            Temp1 := an_instance.Debug2;
         end if;

         case Integer(Temp1) is
            when 1 =>
               Temp2 := Arctan(TAS*Gain_Selected_Turn_Rate*Selected_Turn_Rate/(G*57.3))*57.3;
            when 2 =>
               Temp2 := Comm_Roll;
            when 3 =>
               Temp2 := P_Turn_Rate_Comp;
            when 4 =>
               Temp2 := I_Turn_Rate_Comp;
            when 5 =>
               Temp2 := An_Instance.Commanded_Aileron_Deflection;
            when 6 =>
               Temp2 := Aileron_deflection;
            when 7 =>
               Temp2 := An_Instance.Integral_Turn_Rate_Error;
            when 8 =>
               Temp2 := Gain_Selected_Turn_Rate;
            when 9 =>
               Temp2 := An_Instance.Commanded_Roll;
            when 10 =>
               Temp2 := Selected_Turn_rate;
            when 11 =>
               Temp2 := An_Instance.Selected_heading;
            when 12 =>
               Temp2 := An_Instance.Selected_Turn_rate;
            when 13 =>
               Temp2 := P_Heading_Comp;
            when 14 =>
               Temp2 := I_Heading_Comp;
            when 15 =>
               Temp2 := Corrected_Heading_Err*Gain_Heading_Error;
            when 16 =>
               Temp2 := P_roll_Comp;
            when 17 =>
               Temp2 := I_roll_Comp;
            when 18 =>
               Temp2 := D_roll_Comp;
            when 19 =>
               Temp2 := Corrected_Heading_err;
            when 20 =>
               Temp2 := Gain_Wing_level;
            when 21 =>
               Temp2 := P_Roll_Comp  + I_Roll_Comp + D_Roll_Comp +
              P_Heading_Comp + I_Heading_Comp;
            when 22 =>
               Temp2 := Altitude_agl;
            when 23 =>
               Temp2 := An_Instance.Gain_Selected_Turn_Rate_X2;
            when 24 =>
               Temp2 := An_Instance.KI_Roll ;
            when 25 =>
               Temp2 :=  An_Instance.KP_Roll ;
            when 26 =>
               Temp2 := An_Instance.KD_Roll  ;
            when 27 =>
               Temp2 := An_Instance.KP_Turn_Rate ;
            when 28 =>

               Temp2 := An_Instance.KI_Turn_Rate;
            when 29 =>
               Temp2 := An_Instance.KP_Heading ;
            when 30 =>
               Temp2 := An_Instance.KI_Heading ;
            when 31 =>
               Temp2 := An_Instance.KD_Heading ;
            when 32 =>
               if An_Instance.Heading_Hold_Is_On then
                  Temp2 :=  1.0;
               else
                  Temp2 :=  0.0;
               end if;
            when 33 =>
               if An_Instance.Heading_Hold_Is_locked then
                  Temp2 :=  1.0;
               else
                  Temp2 :=  0.0;
               end if;

            when 39 =>
               Temp2 :=  An_Instance.Backdrive_Force;
            when 40 =>
               Temp2 := Pilot_Force;
            when 41 =>
               Temp2 := 999.9;
            when Others => null;
         end case;

         if I = 1 then
            an_instance.Debug3 := Temp2;
         else
            an_instance.Debug4 := Temp2;
         end if;

      end loop;




      -- ### debug End #####################################################
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Update",
                    Severity => Log.ERROR);
         raise;
   end Update;

   function Commanded_Aileron_deflection(An_Instance :in Instance)
                                         return Angle_Types.Degrees is
   begin

      return An_Instance.Commanded_Aileron_Deflection;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Commanded_Aileron_deflection",
                    Severity => Log.ERROR);
         raise;
   end Commanded_Aileron_Deflection;
   function Is_on(An_Instance :in Instance)
                  return Boolean is
   begin

      return (An_Instance.Mode > 0);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": is_on",
                    Severity => Log.ERROR);
         raise;
   end Is_On;



   function Selected_heading(An_Instance :in Instance)
                             return Angle_Types.Degrees is
   begin
      return An_Instance.Selected_Heading;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Selected_heading",
                    Severity => Log.ERROR);
         raise;
   end Selected_Heading;

   function Mode(An_Instance :in Instance) return Integer is
   begin
      return An_Instance.Mode;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Mode",
                    Severity => Log.ERROR);
         raise;
   end Mode;

   procedure Set_Debug1
     (Debug1                  :in       float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.debug1 := debug1;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_debug1 ",
                    Severity => Log.ERROR);
         raise;

   end Set_debug1;

   function Debug1(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug1;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug1",
                    Severity => Log.ERROR);
         raise;
   end debug1;

   procedure Set_Debug2
     (Debug2                  :in       float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.debug2 := debug2;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_debug2 ",
                    Severity => Log.ERROR);
         raise;

   end Set_debug2;

   function Debug2(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug2;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug2",
                    Severity => Log.ERROR);
         raise;
   end debug2;

   function Debug3(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug3;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug3",
                    Severity => Log.ERROR);
         raise;
   end debug3;

   function Debug4(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug4;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug1",
                    Severity => Log.ERROR);
         raise;
   end debug4;

   function Selected_Turn_Rate(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.Selected_Turn_Rate;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Selected_Turn_Rate",
                    Severity => Log.ERROR);
         raise;
   end Selected_Turn_Rate;

   function Maximum_Roll_Angle(An_Instance :in Instance) return Float  is
   begin
      return Max_Roll_Angle;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Maximum_Roll_Angle",
                    Severity => Log.ERROR);
         raise;
   end Maximum_Roll_Angle;

   procedure Set_Backdrive_Mode
     (Is_On                   :in       Boolean;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.backdrive := Is_on;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_backdrive_mode",
                    Severity => Log.ERROR);
         raise;

   end Set_Backdrive_mode;

   procedure Set_Selected_Heading
     (Heading                 :in       Float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.Selected_Heading := Heading;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_Selected_Heading",
                    Severity => Log.ERROR);
         raise;

   end Set_Selected_Heading;


   function Backdrive(An_Instance :in Instance) return Boolean  is
   begin
      return An_Instance.Backdrive;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Backdrive",
                    Severity => Log.ERROR);
         raise;
   end Backdrive;
   function Backdrive_Force (An_Instance :in Instance) return float  is
   begin
      return An_Instance.Backdrive_Force;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Backdrive_force",
                    Severity => Log.ERROR);
         raise;
   end Backdrive_Force;

end  Roll_Driver;



