-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
-- ### debug #########################################################
with Jpats_Simulated_Aircraft;
-- ### debug End #####################################################
with Log;
with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;

package body Pitch_Driver  is
   -- General definitions -----------------------------------------------------
   Body_File_Name : constant String := "IOS_pilot/Pitch_Driver.adb";


   package JSA         renames Jpats_Simulated_Aircraft;
   package IT          renames Interpolation_Table;

   -- Data
   IAS20_T        :aliased  IT.doubly_Indexed.Instance;
   IAS21_T        :aliased  IT.doubly_Indexed.Instance;
   IAS22_T        :aliased  IT.doubly_Indexed.Instance;
   CLIMB2_T       :aliased  IT.doubly_Indexed.Instance;



   procedure Set_Gain
     (Tuning_Mode                    :in     Integer;
      Comm_Pitch                     :in     Float;
      KP_Pitch                       :in     Float;
      KI_Pitch                       :in     Float;
      KD_Pitch                       :in     Float;
      Max_Pitch_Rate                 :in     Float;

      KP_Climb                       :in     Float;
      KI_Climb                       :in     Float;
      Max_Climb_rate                 :in     Float;

      KP_IAS                         :in     Float;
      KI_IAS                         :in     Float;
      KD_IAS                         :in     Float;
      Max_IAS_Err                    :in     Float;
      Max_IAS_Rate                   :in     Float;

      KP_Altitude                    :in     Float;
      KI_Altitude                    :in     Float;
      KD_Altitude                    :in     Float;
      Max_Alt_Err                    :in     Float;

      Gain_Roll_compensator          :in     Float;
      Gain_Torque_compensator        :in     Float;
      Gain_Stall_Correction          :in     Float;

      Limited_AOA                    :in     Float;
      An_Instance                    :in out Instance) is
   begin
      An_Instance.Tuning_Mode                    := Tuning_Mode;
      An_Instance.Comm_Pitch                     := Comm_Pitch;
      An_Instance.KP_Pitch                       := KP_Pitch;
      An_Instance.KI_Pitch                       := KI_Pitch;
      An_Instance.KD_Pitch                       := KD_Pitch;
      An_Instance.Max_Pitch_Rate                 := Max_Pitch_Rate;

      An_Instance.KP_Climb                       := KP_Climb;
      An_Instance.KI_Climb                       := KI_Climb;
      An_Instance.Max_Climb_Rate                 := Max_Climb_Rate;

      An_Instance.KP_IAS                         := KP_IAS;
      An_Instance.KI_IAS                         := KI_IAS;
      An_Instance.KD_IAS                         := KD_IAS;
      An_Instance.Max_IAS_Err                    := Max_IAS_Err;
      An_Instance.Max_IAS_Rate                   := Max_IAS_Rate;

      An_Instance.KP_Altitude                    := KP_Altitude;
      An_Instance.KI_Altitude                    := KI_Altitude ;
      An_Instance.KD_Altitude                    := KD_Altitude;
      An_Instance.Max_Alt_Err                    := Max_Alt_Err;

      An_Instance.Gain_Roll_Compensator          := Gain_Roll_Compensator;
      An_Instance.Gain_Torque_Compensator        := Gain_Torque_Compensator;
      An_Instance.Gain_Stall_Correction          := Gain_Stall_Correction;

      An_Instance.Limited_AOA                    := Limited_AOA;


   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": set_gain",
                    Severity => Log.ERROR);
         raise;

   end Set_Gain;
   procedure initialize
     (An_Instance             :in out  Instance) is
      -- Point to data table directory
      File_Path : String := Simulation_Dictionary.Lookup("IOS_Pilot_Dir");
   begin
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Pitch);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_IAS_Err);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Climb);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Altitude);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_AOA);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_torque);
      First_Order_Filter.Initialize(0.0,An_Instance.Washout_torque);
      An_Instance.Mode := 0;

      It.Read(File_Path & "ias20.ito",IAS20_T);
      It.Read(File_Path & "ias21.ito",IAS21_T);
      It.Read(File_Path & "ias22.ito",IAS22_T);
      It.Read(File_Path & "climb2.ito",CLIMB2_T);

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize",
                    Severity => Log.ERROR);
         raise;

   end Initialize;

   procedure Set_Mode
     (Mode                    :in       Integer;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.Mode := Mode;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_mode",
                    Severity => Log.ERROR);
         raise;

   end Set_mode;

   procedure Reset_Internal_variables
     (Pitch_Angle        :in      Angle_Types.Degrees;
      Pitch_Rate         :in      Angle_Types.Degrees_Per_Sec;
      AOA                :in      Angle_Types.Degrees;
      Altitude           :in      Length_Types.Feet;
      Climb_rate         :in      Length_Types.Feet_Per_Min;
      IAS                :in      Length_Types.Knots;
      Torque             :in      Float;
      Elevator_Deflection:in      Angle_Types.Degrees;
      An_Instance        :in out  Instance) is
   begin
      An_Instance.Commanded_Pitch_Angle    := Pitch_angle;
      An_Instance.Commanded_Pitch_Rate     := 0.0;
      An_Instance.IAS_Last                 := 0.0;
      An_Instance.Integral_Pitch_Error     := 0.0;
      An_Instance.Integral_Altitude_Error  := 0.0;
      An_Instance.Integral_IAS_Error       := 0.0;
      An_Instance.Integral_Climb_Error     := 0.0;
      An_Instance.Commanded_Elevator_Deflection := Elevator_Deflection;
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Pitch);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_IAS_Err);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Climb);
      First_Order_Filter.Initialize(0.0,An_Instance.Filtered_Selected_Altitude);
      First_Order_Filter.Initialize(AOA,An_Instance.Filtered_AOA);
      First_Order_Filter.Initialize(torque,An_Instance.Filtered_torque);
      First_Order_Filter.Initialize(torque,An_Instance.Washout_torque);
      An_Instance.Altitude_Hold_Enabled   := False;
      An_Instance.Altitude_Hold_Is_On     := False;
      An_Instance.Altitude_Hold_Is_Locked := False;
      An_Instance.Fader                   := 0.0;
      An_Instance.Easy_On                 := 0.0;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Reset_Internal_variables",
                    Severity => Log.ERROR);
         raise;

   end Reset_Internal_variables;


   procedure Set_Reference
     (Mode                    :in      Integer;
      Selected_Altitude       :in      Length_Types.Feet;
      Selected_Climb_rate     :in      Length_Types.Feet_Per_Min;
      Selected_IAS            :in      Length_Types.Knots;
      Selected_Torque         :in      Float;
      Performance_Mode        :in      Integer;
      An_Instance             :in out  Instance) is

      Altitude_To_Climb_Hold_Gain_X2 : Float := 200.0;
      Tlevel       : Float := 0.0;
      IAS          : Float := 100.0;
      Climb_Rate   : Float := 0.0;
   begin

      if not ( An_Instance.Mode = Mode)  then
         An_Instance.Easy_On  := 0.0;
      end if;


      An_Instance.Mode := Mode;

      An_Instance.Selected_Climb_rate := Selected_Climb_rate;
      An_Instance.Selected_IAS        := Selected_IAS;

      case Performance_mode is
         when 1 =>
            An_Instance.Performance_Mode := IAS_CLIMB_HOLD;
            IAS        := Selected_IAS;
            Climb_Rate := Selected_Climb_Rate;


         when 2 =>
            An_Instance.Performance_Mode := CLIMB_TORQUE_HOLD;
            IAS        := It.Doubly_Indexed.Interpolate(Selected_torque,Selected_Climb_rate,IAS20_T'access);
            Climb_Rate := Selected_Climb_Rate;

         when others =>
            An_Instance.Performance_Mode := IAS_TORQUE_HOLD;
            IAS        := Selected_IAS;
            Climb_Rate := It.Doubly_Indexed.Interpolate(Selected_IAS,Selected_Torque,CLIMB2_T'access);
      end case;

      Tlevel :=
        Arcsin(Float'Max(-0.9,
                         Float'Min(0.9, abs(Climb_Rate)/
                                   (Float'Max(40.0,IAS*1.6878)*60.0))))*57.3/1.5;
      -- divided by 1.5 to represent a leveling with 2 degree/s of flight path angle

      Altitude_To_Climb_Hold_Gain_X2 :=
        150.0 + Float'Max(0.0, abs(Climb_rate)*Tlevel/60.0);


      An_Instance.Altitude_To_Climb_Hold_Gain_X2 := Altitude_To_Climb_Hold_Gain_X2;
      An_Instance.Altitude_To_Climb_Hold_Gain_X1 := 10.0 +
              (Altitude_To_Climb_Hold_Gain_X2 - 150.0)/10.0 ;

      An_Instance.Altitude_Hold_Gain_X1          := An_Instance.Altitude_To_Climb_Hold_Gain_X1;
      An_Instance.Altitude_Hold_Gain_X2          := An_Instance.Altitude_To_Climb_Hold_Gain_X2;


      if An_Instance.Altitude_Hold_Enabled then
         if abs(Selected_Altitude - An_Instance.Selected_Altitude) > Altitude_To_Climb_Hold_Gain_X2 then
            An_Instance.Altitude_Hold_Is_On := False;
            An_Instance.Altitude_Hold_Is_Locked := False;
         end if;

         An_Instance.Selected_Altitude   := Selected_Altitude;
      end if;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_reference",
                    Severity => Log.ERROR);
         raise;

   end Set_Reference;


   procedure Update
     (Altitude                   :in      Length_Types.Feet;
      Climb_rate                 :in      Length_Types.Feet_Per_Min;
      IAS                        :in      Length_Types.Knots;
      Roll_Angle                 :in      Angle_Types.Degrees;
      Pitch_Angle                :in      Angle_Types.Degrees;
      AOA                        :in      Angle_Types.Degrees;
      Pitch_Rate                 :in      Angle_Types.Degrees_Per_Sec;
      TAS                        :in      Length_Types.Feet_Per_Sec;
      Altitude_AGL               :in      Length_Types.Feet;
      Elevator_Deflection        :in      Angle_Types.Degrees;
      Flap_Pos                   :in      Angle_Types.Degrees;
      Nx                         :in      Float;
      Torque                     :in      Float;
      Pilot_Force                :in      Float;
      Dt                         :in      Float;
      Bypass                     :in      Boolean;
      An_Instance                :in out  Instance) is


      Altitude_to_Climb_Hold_Gain : Float := 0.0;


      Altitude_Hold_Gain          : Float := 0.0;

      Tho_Washout_Torque          : Float := 5.0;
      Gain_Backdrive_Force        : Float := 2.0;

      G                           : Float := 32.174 ;

      Temp1                       : Float := 0.0;
      Temp2                       : Float := 0.0;
      IAS_Err                     : Float := 0.0;
      IAS_Rate                    : Float := 0.0;
      Alt_Err                     : Float := 0.0;
      Limited_AOA                 : Float := 0.0;

      Comm_Pitch                  : Float := 0.0;
      Torque_Comp_For_Pitch       : Float := 0.0;
      Roll_Comp_For_Pitch         : Float := 0.0;
      P_IAS_Comp                  : Float := 0.0;
      I_IAS_Comp                  : Float := 0.0;
      D_IAS_Comp                  : Float := 0.0;
      P_Climb_Comp                : Float := 0.0;
      I_Climb_Comp                : Float := 0.0;
      Climb_Comp_For_Pitch        : Float := 0.0;
      P_Pitch_Comp                : Float := 0.0;
      I_Pitch_Comp                : Float := 0.0;
      D_Pitch_Comp                : Float := 0.0;
      P_Altitude_Comp             : Float := 0.0;
      I_Altitude_Comp             : Float := 0.0;
      Stall_Correction            : Float := 0.0;

      Selected_IAS                : Float := 0.0;
      Selected_Climb              : Float := 0.0;

      Max_Pitch                   : Float := 45.0;




   begin


      if not Bypass then

      --| -----------------------------------------------------------------------
      --| Process the fadering in
      --| -----------------------------------------------------------------------
      if An_Instance.Fader < 1.0 then
         An_Instance.Fader := An_Instance.Fader + Dt/Fader_Time ;
         if An_Instance.Fader >= 1.0 then
            An_Instance.Fader := 1.0;
         end if;
      end if;

      --| -----------------------------------------------------------------------
      --| Process the pitch easy-on during mode change (12 s)
      --| -----------------------------------------------------------------------

      if Altitude_AGL < 100.0 then
         An_Instance.Easy_On  := 1.0;
      end if;

      if An_Instance.Easy_On  < 1.0 then
         An_Instance.Easy_On  := An_Instance.Easy_On + Dt/12.0;
         if An_Instance.Easy_On >= 1.0 then
            An_Instance.Easy_On  := 1.0;
         end if;
      end if;

      --| -----------------------------------------------------------------------
      --| Based on altitude error, calculate the ramp up/down
      --| gains.
      --| -----------------------------------------------------------------------
      Alt_Err := An_Instance.Selected_Altitude -Altitude;

      if (abs(Alt_Err)< An_Instance.Altitude_to_Climb_Hold_Gain_X1) then
         Altitude_to_Climb_Hold_Gain := 0.0;
      elsif (abs(Alt_Err) >  An_Instance.Altitude_to_Climb_Hold_Gain_X2) then
         Altitude_to_Climb_Hold_Gain := 1.0;
      else
         Altitude_to_Climb_Hold_Gain := (abs(Alt_Err) - An_Instance.Altitude_to_Climb_Hold_Gain_X1)/
           (An_Instance.Altitude_to_Climb_Hold_Gain_X2 -  An_Instance.Altitude_to_Climb_Hold_Gain_X1);
      end if;


      if (abs(Alt_Err)< An_Instance.Altitude_Hold_Gain_X1) then
         Altitude_Hold_Gain := 1.0;
      elsif (abs(Alt_Err) >  An_Instance.Altitude_Hold_Gain_X2) then
         Altitude_Hold_Gain := 0.0;
      else
         Altitude_Hold_Gain := (An_Instance.Altitude_Hold_Gain_X2 - abs(Alt_Err))/
           (An_Instance.Altitude_Hold_Gain_X2 - An_Instance.Altitude_Hold_Gain_X1);
      end if;


      if not An_Instance.Altitude_Hold_Enabled then
         An_Instance.Altitude_Hold_Is_On := False;
         An_Instance.Altitude_Hold_Is_Locked := False;
         Altitude_to_Climb_Hold_Gain := 1.0;
         Altitude_Hold_Gain := 0.0;
      else
         --| Altitude hold is engaged?
         --| Once it is engaged, it will keep engaged, until a new altitude
         --| target is inputed.

         if (not An_Instance.Altitude_Hold_Is_On) and (not An_Instance.Altitude_Hold_Is_Locked) then
            if (abs(Alt_Err) <  An_Instance.Altitude_Hold_Gain_X2) then
               An_Instance.Altitude_Hold_Is_On := True;

            end if;
         else
            if not An_Instance.Altitude_Hold_Is_Locked then
               if (Altitude_Hold_Gain > 0.9) then
                  An_Instance.Altitude_Hold_Is_Locked := True;
               end if;
            else
               Altitude_Hold_Gain := 1.0;
            end if;

         end if;
      end if;

      --| -----------------------------------------------------------------------
      --| The following are to determine the AOA reference (Limited_AOA).
      --| It consist of providing AOA reference, which corresponds to a value
      --| associated with a stabilized cruise condition.
      --| -----------------------------------------------------------------------

      Limited_AOA := An_Instance.Limited_AOA;

      --| Filter the AOA of reference
      First_Order_Filter.Update(Limited_AOA,
                                1.5,
                                Dt,
                                An_Instance.Filtered_AOA);
      Limited_AOA := First_Order_Filter.Output(An_Instance.Filtered_AOA);

      --| -----------------------------------------------------------------------
      --| Determine the stall correction if AOA is greater than AOA stall
      --| -----------------------------------------------------------------------
      Temp1 := AOA_Stall;
      if (Integer(JSA.Float1) = 116)  and JSA.Bool1 then
         An_Instance.Gain_Stall_Correction := Jsa.Float2;
         Temp1 := Jsa.Float3;
      end if;
      Stall_Correction := -An_Instance.Gain_Stall_Correction*Float'Max(0.0, AOA - temp1);

      --| -----------------------------------------------------------------------
      --| Determine the contribution of roll for selecting
      --| the necessary pitch commande
      --| -----------------------------------------------------------------------


      Temp1 := (1.0 - Cos(Roll_Angle,360.0));

      Roll_Comp_For_Pitch := An_Instance.Gain_Roll_Compensator*Temp1;



      --| -----------------------------------------------------------------------
      --| Determine the contribution of torque change for selecting
      --| the necessary pitch commande
      --| -----------------------------------------------------------------------
      if (Integer(JSA.Float1) = 115)  and JSA.Bool1 then
         An_Instance.Gain_Torque_Compensator := Jsa.Float4;
      end if;

      First_Order_Filter.Update(torque,
                                Tho_Washout_Torque/10.0,
                                Dt,
                                An_Instance.Filtered_torque);

      Temp1 := First_Order_Filter.Output(An_Instance.Filtered_torque);

      First_Order_Filter.Update(temp1,
                                Tho_Washout_torque,
                                Dt,
                                An_Instance.Washout_torque);
      temp1  := First_Order_Filter.Output(An_Instance.Washout_torque);

      if (An_Instance.Altitude_Hold_Is_On and
        (not ( An_Instance.Performance_Mode = IAS_CLIMB_HOLD))) or
        ((An_Instance.Mode = 1) and (not An_Instance.Altitude_Hold_Is_On))  then
         torque_Comp_For_Pitch := -An_Instance.Gain_torque_Compensator*Temp1;
      else
         torque_Comp_For_Pitch := 0.0;
      end if;



      --| -----------------------------------------------------------------------
      --| Determine the contribution of IAS for selecting
      --| the necessary pitch commande
      --| -----------------------------------------------------------------------


      -- ### debug #########################################################
      if An_Instance.Tuning_Mode = 3 then
         Altitude_Hold_Gain              := 0.0;
         Altitude_To_Climb_Hold_Gain     := 1.0;
         An_Instance.Mode                := 1;
         An_Instance.Altitude_Hold_Is_On := False;
      end if;
      -- ### debug End #####################################################

      if (not (An_Instance.Mode = 1)) or
        An_Instance.Altitude_Hold_Is_On then
         An_Instance.Selected_IAS := IAS;
      end if;


      Temp2 := An_Instance.selected_IAS;

      -- Last selected_IAS
      Temp1 := IAS - First_Order_Filter.Output(An_Instance.Filtered_IAS_Err);

      if abs(Temp2 - Temp1) > An_Instance.Max_Ias_Rate then
         if (Temp2 - Temp1) > 0.0 then
            Temp2 := Temp1  + An_Instance.Max_Ias_Rate;
         else
            Temp2 := Temp1 - An_Instance.Max_Ias_Rate;
         end if;
      end if;


      Selected_IAS  := Temp2;

      Temp1 :=Float'Min(An_Instance.Max_Ias_Err,Float'Max(-An_Instance.Max_Ias_Err,IAS - Selected_IAS));


      --| Filter the change on selected IAS
      First_Order_Filter.Update(Temp1,
                                Tho_Filtered_IAS_Err,
                                Dt,
                                An_Instance.Filtered_IAS_Err);
      Temp2:= First_Order_Filter.Output(An_Instance.Filtered_IAS_Err);

      --IAS_Err := Altitude_To_Climb_Hold_Gain*
      --  Float'Min(An_Instance.Max_Ias_Err,Float'Max(-An_Instance.Max_Ias_Err, Temp2));
      IAS_Err  := Altitude_To_Climb_Hold_Gain*Temp2;
      IAS_Rate := Altitude_To_Climb_Hold_Gain*(IAS - An_Instance.IAS_last)/(Float'Max(Dt,1.0/60.0));
      An_Instance.IAS_Last := IAS;

      if (not (An_Instance.Mode = 1)) or An_Instance.Altitude_Hold_Is_On then
         An_Instance.Integral_IAS_Error := An_Instance.Integral_IAS_Error*(1.0 - 0.5*Dt);

         P_IAS_Comp := An_Instance.KP_IAS*IAS_Err ;
         I_IAS_Comp := An_Instance.KI_IAS*An_Instance.Integral_IAS_Error;
         D_IAS_Comp := 0.0;

      else

         if ((First_Order_Filter.Output(An_Instance.Filtered_Selected_Pitch) > - Max_Pitch) and
             (First_Order_Filter.Output(An_Instance.Filtered_Selected_Pitch) < Max_Pitch )) and
           ((Elevator_Deflection > -15.9) and  (Elevator_Deflection < 17.9))    and
           (abs(An_Instance.Selected_Climb_Rate) < 4990.0)  and
           (abs(IAS_Err) < 5.0) then

            Temp2 :=10.0;
            if (Integer(JSA.Float1) = 115)  and JSA.Bool1 then
               Temp2 := Jsa.Float2;
            end if;

            An_Instance.Integral_IAS_Error := Float'Max(-temp2, Float'Min(temp2,An_Instance.Integral_IAS_Error + IAS_Err*Dt));
         else
            An_Instance.Integral_IAS_Error := An_Instance.Integral_IAS_Error*(1.0 - 0.5*Dt);
         end if;



         P_IAS_Comp := An_Instance.KP_IAS*IAS_Err ;

         I_IAS_Comp := An_Instance.KI_IAS*An_Instance.Integral_IAS_Error;

         D_IAS_Comp := An_Instance.KD_IAS*IAS_Rate;


      end if;



      --| -----------------------------------------------------------------------
      --| Determine the contribution of Climb rate error for selecting pitch
      --| -----------------------------------------------------------------------

      -- ### debug #########################################################
      if An_Instance.Tuning_Mode = 2 then
         Altitude_Hold_Gain              := 0.0;
         Altitude_To_Climb_Hold_Gain     := 1.0;
         An_Instance.Mode                := 2;
         An_Instance.Altitude_Hold_Is_On := False;
      end if;
      -- ### debug End #####################################################





      if (An_Instance.Mode = 1) and (not An_Instance.Altitude_Hold_Is_On) then
         Temp1 := It.Doubly_Indexed.Interpolate(Selected_IAS,Torque,CLIMB2_T'access);
         An_Instance.Selected_Climb_Rate := Float'Max(-5000.0, Float'Min(5000.0,Temp1));


      elsif An_Instance.Altitude_Hold_Is_On then
         --| -----------------------------------------------------------------------
         --| If in altitude hold, update the selected climb rate
         --| -----------------------------------------------------------------------
         if (abs(Alt_Err)< An_Instance.Altitude_to_Climb_Hold_Gain_X1) then
            Altitude_to_Climb_Hold_Gain := 0.0;
         elsif (Alt_Err >  An_Instance.Altitude_to_Climb_Hold_Gain_X2) then
            Altitude_to_Climb_Hold_Gain := 1.0;
         elsif (Alt_Err >  An_Instance.Altitude_to_Climb_Hold_Gain_X1) and (Alt_Err <  An_Instance.Altitude_to_Climb_Hold_Gain_X2) then
            Altitude_to_Climb_Hold_Gain := (Alt_Err - An_Instance.Altitude_to_Climb_Hold_Gain_X1)/
              (An_Instance.Altitude_to_Climb_Hold_Gain_X2 -  An_Instance.Altitude_to_Climb_Hold_Gain_X1);
         elsif(Alt_Err <  -An_Instance.Altitude_to_Climb_Hold_Gain_X1) and (Alt_Err >  -An_Instance.Altitude_to_Climb_Hold_Gain_X2) then
            Altitude_to_Climb_Hold_Gain := (Alt_Err + An_Instance.Altitude_to_Climb_Hold_Gain_X1)/
              (An_Instance.Altitude_to_Climb_Hold_Gain_X2 -  An_Instance.Altitude_to_Climb_Hold_Gain_X1);
         elsif (Alt_Err <  -An_Instance.Altitude_to_Climb_Hold_Gain_X2) then
            Altitude_to_Climb_Hold_Gain := -1.0;
         end if;

         An_Instance.Selected_Climb_Rate := Altitude_To_Climb_Hold_Gain*An_Instance.Selected_Climb_Rate;

      end if;

      Temp1 := An_Instance.Selected_Climb_Rate;

      Temp2 := Altitude_To_Climb_Hold_Gain*Temp1 + An_Instance.Kd_Altitude*Altitude_Hold_Gain*Alt_Err;

      if abs(Temp2 - First_Order_Filter.Output(An_Instance.Filtered_Selected_Climb)) > An_Instance.Max_Climb_Rate then
         if (Temp2 - First_Order_Filter.Output(An_Instance.Filtered_Selected_Climb)) > 0.0 then
            Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Selected_Climb) + An_Instance.Max_Climb_Rate;
         else
            Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Selected_Climb) - An_Instance.Max_Climb_Rate;
         end if;
      end if;

      --| Filter the selected climb rate
      if (An_Instance.Mode = 1) and (not An_Instance.Altitude_Hold_Is_On) then
         Temp1 := 0.5;
         if (Integer(JSA.Float1) = 115)  and JSA.Bool1 then
            Temp1 := Jsa.Float3;
         end if;
      else
         Temp1 := 1.0;
      end if;

      First_Order_Filter.Update(Temp2,
                                Temp1*Tho_Filtered_Selected_Climb,
                                Dt,
                                An_Instance.Filtered_Selected_Climb);
      Selected_Climb := First_Order_Filter.Output(An_Instance.Filtered_Selected_Climb);

      Temp1 := Selected_Climb - Climb_Rate;

      if (An_Instance.Mode = 1) and (not An_Instance.Altitude_Hold_Is_On) then
         P_Climb_Comp := An_Instance.KP_Climb*Temp1/60.0;
         An_Instance.Integral_Climb_Error := An_Instance.Integral_Climb_Error*(1.0 - 0.5*Dt);
         I_Climb_Comp := An_Instance.KI_Climb*An_Instance.Integral_Climb_Error/60.0;


      else
         if ((First_Order_Filter.Output(An_Instance.Filtered_Selected_Pitch) > - Max_Pitch) and
             (First_Order_Filter.Output(An_Instance.Filtered_Selected_Pitch) < Max_Pitch )) and
           ((Elevator_Deflection > -15.9) and  (Elevator_Deflection < 17.9))   and
           (abs(An_Instance.Selected_Climb_Rate) < 4990.0)   then
            An_Instance.Integral_Climb_Error := An_Instance.Integral_Climb_Error + Temp1*Dt;
         end if;


         P_Climb_Comp := An_Instance.KP_Climb*Temp1/60.0;
         I_Climb_Comp := An_Instance.KI_Climb*An_Instance.Integral_Climb_Error/60.0;
      end if;




      --| -----------------------------------------------------------------------
      --| Determine the contribution of Climb rate for selecting pitch
      --| -----------------------------------------------------------------------

      if  ( An_Instance.Performance_Mode = CLIMB_TORQUE_HOLD) then
         if (Flap_Pos > 45.0)   then
            Selected_IAS := It.Doubly_Indexed.Interpolate(torque,Selected_Climb,IAS22_T'access);
         elsif (Flap_Pos > 15.0)  then
            Selected_IAS := It.Doubly_Indexed.Interpolate(torque,Selected_Climb,IAS21_T'access);
         else
            Selected_IAS := It.Doubly_Indexed.Interpolate(torque,Selected_Climb,IAS20_T'access);
         end if;
      end if;

      Climb_Comp_For_Pitch :=
        (Arcsin(Float'Max(-0.9, Float'Min(0.9,Selected_Climb/(Float'Max(20.0,Selected_IAS*1.6878)*60.0))))*57.3 +
        Limited_AOA);



      --| -----------------------------------------------------------------------
      --| Determine the Commanded Pitch Angle and Rate
      --| -----------------------------------------------------------------------



      Comm_Pitch :=Roll_Comp_For_Pitch + P_IAS_Comp + I_IAS_Comp  + D_IAS_Comp +
        P_Climb_Comp + I_Climb_Comp + Climb_Comp_For_Pitch + Torque_Comp_For_Pitch ;


      -- ### debug #########################################################
      if An_Instance.Tuning_Mode = 1 then


         Comm_Pitch      := An_Instance.Comm_Pitch;


         Altitude_Hold_Gain := 0.0;
         Altitude_To_Climb_Hold_Gain := 1.0;
         An_Instance.Altitude_Hold_Is_On := False;

      end if;
      -- ### debug End #####################################################

      --| -----------------------------------------------------------------------
      --| Process the pitch easy-on during mode change (12 s)
      --| -----------------------------------------------------------------------
      Temp1 := 50.0;
      Temp2 := 1.0;
      if An_Instance.Easy_On  < 1.0 then
         Temp1 := 20.0;
         Temp2 := 2.0;

      end if;

      if (Integer(JSA.Float1) = 112)  and JSA.Bool1 then
            Temp1 := JSA.float2;
            Temp2 := JSA.float3;
            if An_Instance.Easy_On  < 1.0 then
               Temp1 := JSA.Float4;
               Temp2 := JSA.Float5;

            end if;

      end if;

      if abs(Comm_Pitch - An_Instance.Commanded_Pitch_Angle) > temp1*An_Instance.Max_Pitch_Rate*dt then
         if (Comm_Pitch - An_Instance.Commanded_Pitch_Angle) > 0.0 then
            Comm_Pitch := An_Instance.Commanded_Pitch_Angle + temp1*An_Instance.Max_Pitch_Rate*dt;
         else
            Comm_Pitch := An_Instance.Commanded_Pitch_Angle  - temp1*An_Instance.Max_Pitch_Rate*dt;
         end if;
      end if;

      First_Order_Filter.Update(Comm_Pitch,
                                Temp2*Tho_Filtered_Selected_Pitch,
                                Dt,
                                An_Instance.Filtered_Selected_Pitch);
      Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Selected_Pitch);

      Temp1 := Float'Max(-Max_Pitch, Float'Min(Max_Pitch,Temp2));

      Temp2 := (Temp1 - An_Instance.Commanded_Pitch_Angle)/Float'Max(Dt,1.0/60.0);

      Temp1 := Float'Max(-An_Instance.Max_Pitch_Rate ,Float'Min(An_Instance.Max_Pitch_Rate,Temp2));

      An_Instance.Commanded_Pitch_Rate := An_Instance.Commanded_Pitch_Rate +
        (1.0/6.0)* (Temp1 - An_Instance.Commanded_Pitch_Rate) ;

      An_Instance.Commanded_Pitch_Angle := An_Instance.Commanded_Pitch_Angle +
        An_Instance.Commanded_Pitch_Rate*Dt;

      --| -----------------------------------------------------------------------
      --| Determine the contribution of Pitch rate error for selecting
      --| the elevator command
      --| -----------------------------------------------------------------------
      --Temp1 := An_Instance.Commanded_Pitch_Rate - Pitch_Rate;
      Temp1 := - Pitch_Rate;
      D_Pitch_Comp  := An_Instance.KD_Pitch*Temp1;


      --| -----------------------------------------------------------------------
      --| Determine the contribution of Pitch error for selecting elevator command
      --| -----------------------------------------------------------------------
      Temp1 := An_Instance.Commanded_Pitch_Angle - Pitch_Angle;

      if (Elevator_Deflection > -15.9) and
        (Elevator_Deflection < 17.9) then
         An_Instance.Integral_Pitch_Error := An_Instance.Integral_Pitch_Error  + Temp1*Dt;
      elsif Stall_Correction < - 0.1 then
         An_Instance.Integral_Pitch_Error := An_Instance.Integral_Pitch_Error *(1.0 - 0.33*Dt);
      end if;
      P_Pitch_Comp := An_Instance.KP_Pitch*Temp1;
      I_Pitch_Comp := An_Instance.KI_Pitch*An_Instance.Integral_Pitch_Error;


      --| -----------------------------------------------------------------------
      --| Determine the contribution of altitude error for selecting
      --| the commanded elevator
      --| -----------------------------------------------------------------------
      Temp1 := Altitude_Hold_Gain*Alt_Err; -- Altitude error
      First_Order_Filter.Update(Temp1,
                                Tho_Filtered_Selected_Altitude,
                                Dt,
                                An_Instance.Filtered_Selected_Altitude);
      Temp2 := First_Order_Filter.Output(An_Instance.Filtered_Selected_Altitude);
      Temp1 :=Float'Min(An_Instance.Max_Alt_Err, Float'Max(-An_Instance.Max_Alt_Err,Temp2));


      if not  An_Instance.Altitude_Hold_Is_On then
         An_Instance.Integral_Altitude_Error := An_Instance.Integral_Altitude_Error *(1.0 - 0.33*Dt);
      else
         if (Elevator_Deflection > -15.9) and
           (Elevator_Deflection < 17.9) then
            An_Instance.Integral_Altitude_Error := An_Instance.Integral_Altitude_Error + Temp1*Dt;
         elsif Stall_Correction < - 0.1 then
            An_Instance.Integral_Altitude_Error := An_Instance.Integral_Altitude_Error *(1.0 - 0.33*Dt);
         end if;
      end if;


      P_Altitude_Comp := An_Instance.Kp_Altitude*Temp1;
      I_Altitude_Comp := An_Instance.KI_Altitude*An_Instance.Integral_Altitude_Error;



      Temp1 := P_Altitude_Comp + I_Altitude_Comp +
        P_Pitch_Comp + I_Pitch_Comp+ D_Pitch_Comp +
        Stall_Correction;


      Temp2 := Float'Max(-Max_Delta_elev,Float'Min(Max_Delta_elev,Temp1));


      An_Instance.Commanded_Elevator_Deflection :=
        Float'Max(-16.0, Float'Min(18.0, Temp2 + Elevator_Deflection));



      --| -----------------------------------------------------------------------
      --| Determine the backdrive force
      --| -----------------------------------------------------------------------
      if An_Instance.Backdrive then


         if Altitude_AGL < 10.0 then
            if IAS < 60.0 then
               Temp2 := -40.0;
            elsif IAS < 70.0 then
               Temp2 := -40.0*(70.0 - IAS)/10.0;
            else
               Temp2 := 0.0;
            end if;
         else
            Temp2 := 0.0;
         end if;

         if (Integer(JSA.Float1) = 114)  and JSA.Bool1 then
            Gain_Backdrive_Force  := Jsa.Float2;
            Temp2 := Temp2*Jsa.Float3;
         end if;


         An_Instance.Backdrive_force := Float'Min( 100.0, Float'Max(- 100.0,-Gain_Backdrive_force*Pilot_force)) + temp2;


      else

         An_Instance.Backdrive_force := 0.0;

      end if;


      end if;
      -- ### debug #########################################################
      if (Integer(JSA.Float1) = 110) and not JSA.Bool1 then
         JSA.Set_Float2to9
           (Float2 => Comm_Pitch,
            Float3 => Pitch_Angle,
            Float4 => Roll_Comp_For_Pitch,
            Float5 => P_IAS_Comp,
            Float6 => D_IAS_Comp,
            Float7 => I_IAS_Comp,
            Float8 => P_Climb_Comp,
            Float9 => I_Climb_Comp,
            Float10=> Climb_Comp_For_Pitch);
      end if;
      if (Integer(JSA.Float1) = 110) and JSA.Bool1 then
         JSA.Set_Float2to9
           (Float2 => Comm_Pitch,
            Float3 => Pitch_Angle,
            Float4 => P_Altitude_Comp,
            Float5 => I_Altitude_Comp,
            Float6 => P_Pitch_Comp,
            Float7 => I_Pitch_Comp,
            Float8 => D_Pitch_Comp,
            Float9 => P_Altitude_Comp + I_Altitude_Comp + P_Pitch_Comp + I_Pitch_Comp+ D_Pitch_Comp,
            Float10=> An_Instance.Commanded_Elevator_Deflection);
      end if;
      for I in 1 .. 2 loop
         if I = 1 then
            Temp1 := an_instance.Debug1;
         else
            Temp1 := an_instance.Debug2;
         end if;

         case Integer(Temp1) is
            when 1 =>
               Temp2 := P_Climb_Comp;
            when 2 =>
               Temp2 := I_Climb_Comp;
            when 3 =>
               Temp2 := Climb_Comp_For_Pitch;
            when 4 =>
               Temp2 := Selected_Climb;
            when 5 =>
               Temp2 := Comm_Pitch;
            when 6 =>
               Temp2 := An_Instance.Commanded_Pitch_Angle;
            when 7 =>
               Temp2 := An_Instance.Commanded_Pitch_Rate;
            when 8 =>
               Temp2 := Limited_AOA;
            when 9 =>
               Temp2 := An_Instance.Commanded_Elevator_Deflection ;
            when 10 =>
               Temp2 := Elevator_Deflection;
            when 11 =>
               Temp2 := An_Instance.Selected_IAS;
            when 12 =>
               Temp2 := P_IAS_Comp;
            when 13 =>
               Temp2 := I_IAS_Comp;
            when 14 =>
               Temp2 := Pitch_angle;
            when 15 =>
               Temp2 := IAS_Err;
            when 16 =>
               Temp2 := Selected_IAS;
            when 17 =>
               Temp2 := P_Altitude_Comp;
            when 18 =>
               Temp2 := I_Altitude_Comp;
            when 19 =>
               Temp2 := Alt_Err;
            when 20 =>
               Temp2 := Altitude_Hold_Gain;
            when 21 =>
               Temp2 := Altitude_To_Climb_Hold_Gain;
            when 22 =>
               Temp2 :=Roll_angle;
            when 23 =>
               Temp2 :=An_Instance.fader;
            when 24 =>
               Temp2 :=
                 Float'Max(-4000.0,
                           Float'Min(4000.0,
                                     Selected_IAS*Sin(Pitch_Angle - Limited_AOA, 360.0)*1.6878*60.0)) ;
            when 25 =>
               if An_Instance.Performance_Mode = IAS_CLIMB_HOLD then
                  Temp2 := 1.0;

               elsif An_Instance.Performance_Mode = IAS_TORQUE_HOLD then
                  Temp2 := 3.0;
               else
                  Temp2 := 2.0;
               end if;

            when 26 =>
               Temp2 := An_Instance.Selected_Climb_Rate;
            when 27 =>

               Temp2 := An_Instance.Altitude_To_Climb_Hold_Gain_X2;
            when 28 =>

               Temp2 := An_Instance.Altitude_To_Climb_Hold_Gain_X1;
            when 29 =>
               if An_Instance.Altitude_Hold_Is_On then
                  Temp2 := 1.0;

               else
                  Temp2 := 0.0;
               end if;
            when 30 =>
               if An_Instance.Altitude_Hold_Is_Locked then
                  Temp2 := 1.0;

               else
                  Temp2 := 0.0;
               end if;
            when 31 =>
               if An_Instance.Altitude_Hold_Enabled  then
                  Temp2 := 1.0;

               else
                  Temp2 := 0.0;
               end if;

            when 32 =>
               Temp2 :=  First_Order_Filter.Output(An_Instance.Washout_torque);
            when 33 =>
               Temp2 := Climb_rate;
            when 34 =>
               Temp2 := P_Pitch_Comp;
            when 35 =>
               Temp2 := I_Pitch_Comp;
            when 36 =>
               Temp2 := D_Pitch_Comp;
            when 37 =>
               Temp2 := An_Instance.Easy_On;
            when 38 =>
               Temp2 :=  Torque_Comp_For_Pitch;
            when 39 =>
               Temp2 :=  An_Instance.Backdrive_Force;
            when 40 =>
               Temp2 := Pilot_Force;
            when 41 =>
               Temp2 := IAS;
            when 42 =>
               Temp2 := Nx;
            when 43 =>
               Temp2 := D_IAS_Comp;
            when 44 =>
               Temp2 := IAS_Rate;
            when 45 =>
               Temp2 := Pitch_Rate;
            when 46 =>
               Temp2 := Float( An_Instance.Tuning_Mode);
            when 47 =>
               Temp2 := An_Instance.Max_Ias_Err;
            when 48 =>
               Temp2 := P_IAS_Comp + D_IAS_Comp;
            when 49 =>
               Temp2 := An_Instance.KD_IAS;
            when 50 =>
               Temp2 := P_IAS_Comp + An_Instance.KD_IAS*IAS_rate;
            when 51 =>
               Temp2 := P_IAS_Comp + P_Climb_Comp + Climb_Comp_For_Pitch;
            when 52 =>
               Temp2 := Stall_correction;
            when 53 =>
               Temp2 := AOA;

            when Others => null;
         end case;

         if I = 1 then
            an_instance.Debug3 := Temp2;
         else
            an_instance.Debug4 := Temp2;
         end if;

      end loop;



      -- ### debug End #####################################################


   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Update",
                    Severity => Log.ERROR);
         raise;
   end Update;

   function Commanded_Elevator_Deflection(An_Instance :in Instance)
                                         return Angle_Types.Degrees is
   begin

      return An_Instance.Commanded_Elevator_Deflection;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Commanded_Elevator_deflection",
                    Severity => Log.ERROR);
         raise;
   end Commanded_Elevator_Deflection;

   function Is_on(An_Instance :in Instance)
                  return Boolean is
   begin

      return (An_Instance.Mode > 0);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": is_on",
                    Severity => Log.ERROR);
         raise;
   end Is_On;



   function Mode(An_Instance :in Instance) return Integer is
   begin
      return An_Instance.mode;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Mode",
                    Severity => Log.ERROR);
         raise;
   end Mode;

   function Performance_Mode(An_Instance :in Instance) return Integer is
      Temp  : Integer := 1;
   begin
      case An_Instance.Performance_mode is
         when IAS_CLIMB_HOLD =>
            Temp := 1;
         when CLIMB_TORQUE_HOLD =>
            Temp := 2;
         when others =>
            Temp := 3;
      end case;
      return temp;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Performance_Mode",
                    Severity => Log.ERROR);
         raise;
   end Performance_Mode;

   procedure Set_Debug1
     (Debug1                  :in       float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.debug1 := debug1;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_debug1 ",
                    Severity => Log.ERROR);
         raise;

   end Set_debug1;



   function Debug1(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug1;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug1",
                    Severity => Log.ERROR);
         raise;
   end debug1;

   procedure Set_Debug2
     (Debug2                  :in       float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.debug2 := debug2;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_debug2 ",
                    Severity => Log.ERROR);
         raise;

   end Set_debug2;

   function Debug2(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug2;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug2",
                    Severity => Log.ERROR);
         raise;
   end debug2;

   function Debug3(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug3;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug3",
                    Severity => Log.ERROR);
         raise;
   end debug3;

   function Debug4(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.debug4;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": debug4",
                    Severity => Log.ERROR);
         raise;
   end debug4;

   function Selected_Climb_Rate(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.Selected_Climb_Rate;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Selected_Climb_Rate",
                    Severity => Log.ERROR);
         raise;
   end Selected_Climb_Rate;

   function Selected_IAS(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.Selected_IAS;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Selected_IAS",
                    Severity => Log.ERROR);
         raise;
   end Selected_IAS;

   procedure Set_Selected_Altitude
     (Selected_Altitude       :in       float;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.Selected_Altitude := Selected_Altitude;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_Selected_Altitude ",
                    Severity => Log.ERROR);
         raise;

   end Set_Selected_Altitude;

  function Selected_Altitude(An_Instance :in Instance) return Float  is
   begin
      return An_Instance.Selected_Altitude;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Selected_Altitude",
                    Severity => Log.ERROR);
         raise;
   end Selected_Altitude;

   procedure Set_Backdrive_Mode
     (Is_On                   :in       Boolean;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.backdrive := Is_on;

   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_backdrive_mode",
                    Severity => Log.ERROR);
         raise;

   end Set_Backdrive_mode;

   function Backdrive(An_Instance :in Instance) return Boolean  is
   begin
      return An_Instance.Backdrive;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Backdrive",
                    Severity => Log.ERROR);
         raise;
   end Backdrive;

   function Backdrive_Force (An_Instance :in Instance) return float  is
   begin
      return An_Instance.Backdrive_Force ;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Backdrive_force",
                    Severity => Log.ERROR);
         raise;
   end Backdrive_Force;

   procedure Set_Altitude_Hold_Enable
     (Is_enabled              :in       Boolean;
      An_Instance             :in out   Instance) is
   begin

      An_Instance.Altitude_Hold_Enabled := Is_enabled;
      if not An_Instance.Altitude_Hold_Enabled then
         An_Instance.Altitude_Hold_Is_On := False;
         An_Instance.Altitude_Hold_Is_Locked := False;
      end if;


   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Set_Altitude_Hold_Enable",
                    Severity => Log.ERROR);
         raise;

   end Set_Altitude_Hold_Enable;

   function Altitude_Hold_Is_on(An_Instance :in Instance)
                                return Boolean is
   begin

      return An_Instance.Altitude_Hold_Is_On;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Altitude_Hold_Is_On",
                    Severity => Log.ERROR);
         raise;
   end Altitude_Hold_Is_On;


   function Get_AOA_Stall return Float  is
   begin
      return AOA_Stall;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Get_AOA_Stall",
                    Severity => Log.ERROR);
         raise;
   end Get_AOA_Stall ;

end  Pitch_Driver;



