-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Asep Rahmat
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

--| Note :
--| I used the following convention :
--|          - All angle type is in degreees;
--|          - Altitude type is positive ascendant

with Jpats_Ios_Pilot.Container;
with Jpats_Ios_Pilot_Types;           use  Jpats_Ios_Pilot_Types;
with JPATS_IOS_Pilot.IOS_Interface;

with Jpats_Secondary_Flight_Controls;
with Jpats_Landing_Gear;
with Jpats_Powerplant;
with Jpats_Powerplant.Test;
with Jpats_Aircraft_Body;
with Jpats_Simulated_Aircraft;
with Jpats_Dcls;
with Jpats_Flight_Instruments;
with JPATS_Visual_Buffer;
with Jpats_Radio_Db_If;

with JPATS_Auto_Test;
with JPATS_Reposition;
with JPATS_Reposition_Types;
with Save_Restore;



with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Angle_Types;
with Length_Types;
with Simulation_Dictionary;
with Log;

package body Jpats_IOS_Pilot.Controller is

   Body_File_Name : constant String  := "IOS_pilot/Jpats_ios_pilot-controller.adb";
   
   FADERING_HEADING_THRESHOLD        : constant Integer := 50;
   P_DOT_THRESHOLD                   : constant Float   := 0.025;
   ROLL_ANGLE_THRESHOLD              : constant Float   := 2.0;

   --| -------------------------------------------------------------------------------------------------
   --| Package renaming
   --| -------------------------------------------------------------------------------------------------
   package IT          renames Interpolation_Table;
   package JAB         renames JPATS_Aircraft_Body;
   package JSA         renames Jpats_Simulated_Aircraft;
   package JAT         renames JPATS_Auto_test;
   package DCLS        renames JPATS_Dcls;
   package CNT         renames JPATS_IOS_Pilot.Container;
   package IOS         renames JPATS_IOS_Pilot.IOS_Interface;

   Ios_Pilot              : CNT.Ios_Interface_Instance renames CNT.This_Ios_Interface;
   Turn_Left_Last_Pass    : Boolean := False;
   Turn_Right_Last_Pass   : Boolean := False;
   Fadering_Left          : Boolean := False;
   Fadering_Right         : Boolean := False;
   Fadering_Heading_Count : Integer := 0;

   --| -------------------------------------------------------------------------------------------------
   --| Declare Lookup table
   --| -------------------------------------------------------------------------------------------------
   --| Pitch driver data
   KD_PITCH_T           :aliased  IT.Singly_Indexed.Instance;
   KP_CLIMB_T           :aliased  IT.Singly_Indexed.Instance;
   KI_CLIMB_T           :aliased  IT.Singly_Indexed.Instance;
   KI_IAS_T             :aliased  IT.doubly_Indexed.Instance;
   KP_IAS_T             :aliased  IT.doubly_Indexed.Instance;
   KD_IAS_T             :aliased  IT.doubly_Indexed.Instance;
   MAXIASER_T           :aliased  IT.doubly_Indexed.Instance;
   KI_ALTITUDE_T        :aliased  IT.Singly_Indexed.Instance;
   MAXCLIMBRATE_T       :aliased  IT.Singly_Indexed.Instance;
   MAXIASRATE_T         :aliased  IT.Singly_Indexed.Instance;
   LIMITEDAOA0_T        :aliased  IT.doubly_Indexed.Instance;
   LIMITEDAOA1_T        :aliased  IT.doubly_Indexed.Instance;
   LIMITEDAOA2_T        :aliased  IT.doubly_Indexed.Instance;

   --| Roll driver data
   KI_TURN_T            :aliased  IT.Singly_Indexed.Instance;
   KP_HEADING_T         :aliased  IT.Singly_Indexed.Instance;
   KI_HEADING_T         :aliased  IT.doubly_Indexed.Instance;
   KD_HEADING_T         :aliased  IT.Singly_Indexed.Instance;
   DMAXTURN_T           :aliased  IT.Singly_Indexed.Instance;
   MAXTURN_T            :aliased  IT.Singly_Indexed.Instance;
   MAXHDG_T             :aliased  IT.doubly_Indexed.Instance;


   --| Yaw driver data
   KP_LAT_T             :aliased  IT.Singly_Indexed.Instance;
   KI_LAT_T             :aliased  IT.Singly_Indexed.Instance;
   KD_LAT_T             :aliased  IT.Singly_Indexed.Instance;
   YAW_GAIN_T           :aliased  IT.Singly_Indexed.Instance;

   --| Torque driver data
   NOMTORQ20_T          :aliased  IT.doubly_Indexed.Instance;
   NOMTORQ21_T          :aliased  IT.doubly_Indexed.Instance;
   NOMTORQ22_T          :aliased  IT.doubly_Indexed.Instance;

   KP_IAST_T            :aliased  IT.Singly_Indexed.Instance;
   KD_IAST_T            :aliased  IT.Singly_Indexed.Instance;


   procedure Initialize  is
      --| Point to data table directory
      File_Path : String := Simulation_Dictionary.Lookup("IOS_Pilot_Dir");
   begin

      CNT.IOS_Pilot_Roll_Driver.Initialize(CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
      CNT.IOS_Pilot_Pitch_Driver.Initialize(CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
      CNT.IOS_Pilot_Yaw_Driver.Initialize(CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);
      CNT.Ios_Pilot_Torque_Driver.Initialize(CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);



      CNT.This_Subsystem.TO_Phase := -1;
      CNT.IOS_Pilot_Pitch_Driver.Set_Mode(0,CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
      CNT.IOS_Pilot_Roll_Driver.Set_Mode(0,CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
      CNT.IOS_Pilot_Yaw_Driver.Set_Mode(0,CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);

      CNT.Ios_Pilot_Torque_Driver.Set_Mode(0,CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);


      --| Load data table
      It.Read(File_Path & "kdpitch.ito",KD_PITCH_T);
      It.Read(File_Path & "kpclimb.ito",KP_CLIMB_T);
      It.Read(File_Path & "kiclimb.ito",KI_CLIMB_T);
      It.Read(File_Path & "kiias.ito",KI_IAS_T);
      It.Read(File_Path & "kpias.ito",KP_IAS_T);
      It.Read(File_Path & "kdias.ito",KD_IAS_T);
      It.Read(File_Path & "maxiaser.ito",MAXIASER_T);
      It.Read(File_Path & "kialt.ito",KI_ALTITUDE_T);
      It.Read(File_Path & "maxclr.ito",MAXCLIMBRATE_T);
      It.Read(File_Path & "maxiasr.ito",MAXIASRATE_T);
      It.Read(File_Path & "limaoa0.ito",LIMITEDAOA0_T);
      It.Read(File_Path & "limaoa1.ito",LIMITEDAOA1_T);
      It.Read(File_Path & "limaoa2.ito",LIMITEDAOA2_T);

      It.Read(File_Path & "kiturn.ito",KI_TURN_T);
      It.Read(File_Path & "kphdg.ito",KP_HEADING_T);
      It.Read(File_Path & "kihdg.ito",KI_HEADING_T);
      It.Read(File_Path & "kdhdg.ito",KD_HEADING_T);
      It.Read(File_Path & "dmaxturn.ito",DMAXTURN_T);
      It.Read(File_Path & "maxturn.ito",MAXTURN_T);
      It.Read(File_Path & "maxhdg.ito",MAXHDG_T);

      It.Read(File_Path & "kplat.ito",KP_LAT_T);
      It.Read(File_Path & "kilat.ito",KI_LAT_T);
      It.Read(File_Path & "kdlat.ito",KD_LAT_T);
      It.Read(File_Path & "kyawg.ito",Yaw_Gain_T);

      It.Read(File_Path & "nomtorq20.ito",NOMTORQ20_T);
      It.Read(File_Path & "nomtorq21.ito",NOMTORQ21_T);
      It.Read(File_Path & "nomtorq22.ito",NOMTORQ22_T);
      It.Read(File_Path & "kpiast.ito",KP_IAST_T);
      It.Read(File_Path & "kdiast.ito",KD_IAST_T);
      IOS.Register_Ios_Variables;

   exception
      when others =>
         Log.Report(Body_File_Name & ".Initialize()");
         raise;
   end Initialize;


   procedure Update (Iconst : in Float) is

      --|----------------------------------------------------------------------------
      --| Roll driver local variables
      --|----------------------------------------------------------------------------
      Turn_Rate                  : constant Angle_Types.Degrees_Per_Sec  :=
        (JSA.Get_Yaw_Rate*Cos(JSA.Get_Roll_Angle)
         + JSA.Get_Pitch_Rate*Sin(JSA.Get_Roll_Angle))*57.3/cos(JSA.Get_Pitch_Angle);
      Heading                    : constant Angle_Types.Degrees          := JSA.Get_Hdg_Angle*57.3;
      Roll_Rate                  : constant Angle_Types.Degrees_Per_Sec  := JSA.Get_Roll_Rate*57.3;
      Roll_Angle                 : constant Angle_Types.Degrees          := JSA.Get_Roll_Angle*57.3;
      Aileron_Deflection         : constant Angle_Types.Degrees          := DCLS.Get_Right_Aileron_Position;
      TAS                        : constant Length_Types.Feet_Per_Sec    := JSA.Get_True_Airspeed;
      Altitude_AGL               : constant Length_Types.Feet            :=-JSA.Get_Aircraft_Height_Above_Local_Terrain;

      Magvar                     : constant Angle_Types.Degrees          := Jpats_Radio_Db_If.Magnetic_Variation;
      Roll_Pilot_Force           : constant Float                        := DCLS.Lat_Stick_Force;

      --| Tuning Variables ----------------------------------------------------------
      Comm_Roll                  : Float := 0.0;
      KP_Roll                    : Float := 1.0;
      KI_Roll                    : Float := 0.5;
      KD_Roll                    : Float := 1.0;
      Max_Roll_Rate              : Float := 19.5;

      Comm_Turn_Rate             : Float := 0.0;
      KP_Turn_Rate               : Float := 0.1;
      KI_Turn_Rate               : Float := 0.1;
      Delta_Max_Selected_Turn    : Float := 30.0;
      Max_Turn_Err               : Float := 0.3;

      KP_Heading                 : Float := 1.0;
      KI_Heading                 : Float := 1.0;
      KD_Heading                 : Float := 0.05;
      Max_Heading_Err            : Float := 0.05;


      --|----------------------------------------------------------------------------
      --| Pitch driver local variables
      --|----------------------------------------------------------------------------
      Altitude                   : constant Length_Types.Feet            := JSA.Get_Aircraft_Geometric_Altitude;
      Climb_rate                 : constant Length_Types.Feet_Per_Min    := JSA.Get_Rate_Of_Climb;
      IAS                        : constant Length_Types.Knots           := Jpats_Flight_instruments.Primary_IAS;
      Pitch_Angle                : constant Angle_Types.Degrees          := JSA.Get_Pitch_Angle*57.3;
      AOA                        : constant Angle_Types.Degrees          := JSA.Get_Angle_Of_Attack*57.3;
      Pitch_Rate                 : constant Angle_Types.Degrees_Per_Sec  := JSA.Get_Pitch_Rate*57.3;
      Elevator_Deflection        : constant Angle_Types.Degrees          := -DCLS.Get_Elev_Pos;
      Nz                         : constant Float                        := -JSA.Load_Factor.Z;
      Nx                         : constant Float                        := JSA.Load_Factor.X;
      Pitch_Pilot_Force          : constant Float                        := DCLS.Long_Stick_Force;
      GW                         : constant Float                        := JSA.GW;

      --| Tuning Variables ----------------------------------------------------------
      Comm_Pitch                 : Float := 0.0;
      KP_Pitch                   : Float := 5.0;
      KI_Pitch                   : Float := 5.0;
      KD_Pitch                   : Float := 3.0;
      Max_Pitch_Rate             : Float := 30.0*80.0/Float'Max(80.0,TAS);

      KP_Climb                   : Float := 0.01;
      KI_Climb                   : Float := 0.005;
      Max_Climb_rate             : Float := 400.0;

      KP_IAS                     : Float := 5.0;
      KI_IAS                     : Float := 0.3;
      KD_IAS                     : Float := 2.0;
      Max_IAS_Err                : Float := 7.5;
      Max_IAS_Rate               : Float := 7.5;

      KP_Altitude                : Float := 0.5;
      KI_Altitude                : Float := 0.055;
      KD_Altitude                : Float := 10.0;
      Max_Alt_Err                : Float := 10.0;

      Gain_Roll_compensator      : Float := 2.0;
      Gain_Torque_Compensator    : Float := 0.2;
      Gain_Stall_Correction      : Float := 1.5;

      Limited_AOA                : Float := 0.0;

      --|----------------------------------------------------------------------------
      --| Yaw driver local variables
      --|----------------------------------------------------------------------------
      Yaw_Rate                   : constant Angle_Types.Degrees_Per_Sec  := JSA.Get_Yaw_Rate*57.3;
      Sideslip_Angle             : constant Angle_Types.Degrees          := JSA.Get_Side_Slip_Angle*57.3;
      Lateral_Acc                : constant Length_Types.Feet_Per_Sec_Sq := JSA.Load_Factor.Y*32.2;
      Rudder_Deflection          : constant Angle_Types.Degrees          := -DCLS.Get_Rudder_Position;
      V_North                    : constant Length_Types.Feet_Per_Sec    := JSA.Get_Vel_Ea.X;
      V_East                     : constant Length_Types.Feet_Per_Sec    := JSA.Get_Vel_Ea.Y;
      Q_Bar                      : Float                                 := Jsa.Get_Dynamic_Pressure;

      Nose_Wow                   : constant Boolean                      := JAB.Get_Nose_Wow;
      Force_Body_Z               : constant Float                        := JAB.Get_Nose_Force_Body_Axis.Z;
      Yaw_Pilot_Force            : constant Float                        := DCLS.Get_Pedal_Force;


      --| Tuning Variables ----------------------------------------------------------
      KP_Sideslip                : Float := 1.0;
      KP_Lat_Acc                 : Float := 1.0;
      KI_Lat_Acc                 : Float := 0.10;
      KD_Lat_Acc                 : Float := 1.0;
      Yaw_Gain                   : Float := 1.5;
      Heading_Gain               : Float := 1.5;

      --|----------------------------------------------------------------------------
      --| Torque driver local variables
      --|----------------------------------------------------------------------------
      Torque                     : constant Float                        := Jpats_powerplant.Engine_Torque_Pct;
      PLA                        : constant Angle_Types.Degrees          := Jpats_powerplant.Power_Control_Lever_Angle;

      --| Tuning Variables ----------------------------------------------------------
      KP_Torque                  : Float := 0.3;
      KI_Torque                  : Float := 0.1;
      KD_Torque                  : Float := 0.5;
      Max_Torque_Err             : Float :=  2.5;

      KP_IAS_Torque              : Float := 10.0;
      KI_IAS_Torque              : Float := 0.1;
      KD_IAS_Torque              : Float := 1.0;
      Max_IAS_Err_Torque         : Float := 5.0;

      Pitch_Gain                 : Float := 3.0;
      Nominal_Torque_Comp        : Float := 35.0;

      --|----------------------------------------------------------------------------
      --| Secondary control local variables
      --|----------------------------------------------------------------------------
      Dt                         : constant Float                        := Iconst;

      Flap_Pos                   : constant float                        := Float(Jpats_Secondary_Flight_Controls.Mean_Flap_Position);
      Gear_Pos                   : constant Float                        := Float(Jpats_Landing_Gear.Mean_Gear_Position);
      Speedbrake_Pos             : constant Float                        := Float(Jpats_Secondary_Flight_Controls.Speedbrake_Position);




      --| Temporary IOS variables
      Roll_Mode                  : Integer                               := 0;
      Selected_Heading           : Angle_Types.Degrees                   := 0.0;
      Selected_Turn_Rate         : Integer                               := 1;
      Selected_Turn_Direction    : Integer                               := 0;

      Pitch_Mode                 : Integer                               := 0;
      Selected_Altitude          : Length_Types.Feet                     := 2000.0;
      Selected_Climb_rate        : Length_Types.Feet_Per_Min             := 1500.0;
      Selected_IAS               : Length_Types.Knots                    := 160.0;

      Yaw_Mode                   : Integer                               := 0;

      Torque_Mode                : Integer                               := 0;
      Selected_Torque            : Float                                 := 0.0;
      Selected_Time_Constant     : Float                                 := 1.5;
      Tuning_Mode                : Integer                               := 0;


      Temp1                      : Float                                 := 0.0;
      Temp2                      : Float                                 := 0.0;
      Bypass                     : Boolean                               := True;
      G                          : constant Float                        := 32.174;

      --|----------------------------------------------------------------------------
      --| Auto takeoff  local variables
      --|----------------------------------------------------------------------------
      TO_Ready                   : Boolean                               := false;
      Rotation_Speed             : Float                                 := 85.0;
      Rotation_Climb             : Float                                 := 2000.0;
      Altitude_Off               : Float                                 := 20.0;
      IAS_Hold_Start             : Float                                 := 139.0;

      Ft_East                    : constant Float                        := Float(Jsa.Get_East - CNT.This_Subsystem.Start_long) * Cos(Float(Jsa.Get_North),360.0) * 364566.0;
      Ft_North                   : constant Float                        := float(Jsa.Get_North - CNT.This_Subsystem.Start_lat) * 364566.0;

      --|----------------------------------------------------------------------------
      --| Engage/Disengage parameter
      --|----------------------------------------------------------------------------
      Engage_Max_Roll_Rate       : Float                                 := 30.0;
      Engage_Max_Pitch_Rate      : Float                                 := 30.0;
      DisEngage_Max_Roll_Rate    : Float                                 := 30.0;
      DisEngage_Max_Pitch_Rate   : Float                                 := 30.0;

      procedure Reset_Internal_Variables is
      begin

         CNT.IOS_Pilot_Pitch_Driver.Reset_Internal_variables
           (Pitch_Angle,
            0.0,
            AOA,
            Altitude,
            Climb_Rate,
            IAS,
            Torque,
            Elevator_Deflection,
            CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

         CNT.IOS_Pilot_Roll_Driver.Reset_Internal_variables
           (Roll_Angle,
            0.0,
            Aileron_Deflection,
            CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);

         CNT.IOS_Pilot_Yaw_Driver.Reset_Internal_variables
           (0.0,
            0.0,
            Rudder_Deflection,
            CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);

         CNT.Ios_Pilot_Torque_Driver.Reset_Internal_variables
           (Pitch_Angle,
            PLA,
            CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);

      end Reset_Internal_Variables;

      procedure Reset_Fader is
      begin
         Fadering_Left := False;
         Fadering_Right := False;
         Fadering_Heading_Count := 0;
      end Reset_Fader;

      procedure Reset_On_Off_Flags is
      begin
         --| Reset on/off flags
         Ios_Pilot.Turn_Left := False;
         Ios_Pilot.Turn_Right := False;
         Ios_Pilot.Turn_Right_Enabled := False;
         Ios_Pilot.Turn_Left_Enabled := False;
         Reset_Fader;
         Turn_Left_Last_Pass := False;
         Turn_Right_Last_Pass := False;
      end Reset_On_Off_Flags;

      

   begin

      ------------------------------------------------------------------
      --| DEBUG ||||||||||||||||||||||||||||||||||||||||||||||||||||||||

      --| Use this if NO HARDWARE
      --if IOS_Pilot.Ios_Pilot_Debug then
      --   if (not (JAT.At_Phase = 9)) and
      --     ((not CNT.Ios_Pilot_Torque_Driver.Is_On(CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver))  or
      --      (CNT.Ios_Pilot_Torque_Driver.Is_On(CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver) and
      --       JSA.Get_Flight_Freeze and
      --       (CNT.Ios_Pilot_Torque_Driver.Mode(CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver)=2))) then

      --      Jpats_Powerplant.Test.Set_Power_Control_Lever_Angle (Position => PLA);

      --   end if;
      --end if;

      --||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||


      if IOS_Pilot.Ios_Pilot_Debug  then
         Rotation_Speed := Ios_Pilot.Rotation_Speed;
         Rotation_Climb := Ios_Pilot.Rotation_Climb;
         Altitude_Off   := Ios_Pilot.Altitude_Off;

         if IOS_Pilot.Ios_Pilot_Mode = 0 then
            if Integer(IOS_Pilot.Misc_Var1) = 5 then
               Engage_Max_Roll_Rate     := IOS_Pilot.Misc_Var2;
               Engage_Max_Pitch_Rate    := IOS_Pilot.Misc_Var3;
               DisEngage_Max_Roll_Rate  := IOS_Pilot.Misc_Var4;
               DisEngage_Max_Pitch_Rate := IOS_Pilot.Misc_Var5;
            end if;
            if Integer(IOS_Pilot.Misc_Var1) = 6 then
               IAS_Hold_Start := IOS_Pilot.Misc_Var2;
            end if;
         end if;


      else
         Ios_Pilot.Rotation_Speed := Rotation_Speed;
         Ios_Pilot.Rotation_Climb := Rotation_Climb;
         Ios_Pilot.Altitude_Off   := Altitude_Off;

      end if;

      --|----------------------------------------------------------------------------
      --| Set backdrive mode of DCLS
      --|----------------------------------------------------------------------------

      CNT.Ios_Pilot_Pitch_Driver.Set_Backdrive_Mode
        (True,CNT.This_Subsystem.The_Ios_Pilot_Pitch_Driver);

      CNT.Ios_Pilot_Roll_Driver.Set_Backdrive_Mode
        (True,CNT.This_Subsystem.The_Ios_Pilot_Roll_Driver);

      CNT.Ios_Pilot_Yaw_Driver.Set_Backdrive_Mode
        (True,CNT.This_Subsystem.The_Ios_Pilot_Yaw_Driver);

      --|----------------------------------------------------------------------------
      --| Determine the functional mode of the ios pilot
      --|----------------------------------------------------------------------------
      if CNT.This_Subsystem.Previous_Mode = 0 then
         if Ios_Pilot.Enable and (Altitude_AGL > 20.0) then
            --| IOS Pilot is set on inflight ----------------------------------------

            if abs(Roll_Angle)  > CNT.IOS_Pilot_Roll_Driver.Maximum_Roll_Angle(
               CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver) then  IOS_Pilot.Error_Number := 1;
            elsif IOS_Pilot.Error_Number = 1  then  IOS_Pilot.Error_Number := 0;
            end if;

            if abs(Pitch_Angle) > 30.0        then IOS_Pilot.Error_Number := 2;
            elsif IOS_Pilot.Error_Number = 2  then  IOS_Pilot.Error_Number := 0;
            end if;
            if abs(Roll_rate)   > 15.0         then IOS_Pilot.Error_Number := 3;
            elsif IOS_Pilot.Error_Number = 3  then  IOS_Pilot.Error_Number := 0;
            end if;
            if abs(Pitch_rate)  > 15.0         then IOS_Pilot.Error_Number := 4;
            elsif IOS_Pilot.Error_Number = 4  then  IOS_Pilot.Error_Number := 0;
            end if;
            if (IAS < 80.0) or (IAS > 240.0) then IOS_Pilot.Error_Number := 5;
            elsif IOS_Pilot.Error_Number = 5  then  IOS_Pilot.Error_Number := 0;
            end if;
            if (Nz  < 0.0 ) or (Nz  > 2.5  ) then IOS_Pilot.Error_Number := 6;
            elsif IOS_Pilot.Error_Number = 6  then  IOS_Pilot.Error_Number := 0;
            end if;

            if (IOS_Pilot.Error_Number = 0) then
               CNT.This_Subsystem.Mode := 2 ;
               JSA.Set_System_Freeze;
               IOS_Pilot.Apply_Enabled := True;
               Ios_Pilot.Turn_Left_Enabled := True;
               Ios_Pilot.Turn_Right_Enabled := True;
               Reset_Internal_Variables;

            end if;

         elsif Ios_Pilot.Enable and (Altitude_AGL < 20.0) then

            --| IOS Pilot is set on ground ----------------------------------------

            CNT.This_Subsystem.Mode := 1 ;
            CNT.IOS_Pilot_Roll_Driver.Initialize(CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
            CNT.IOS_Pilot_Pitch_Driver.Initialize(CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
            CNT.IOS_Pilot_Yaw_Driver.Initialize(CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);
            CNT.Ios_Pilot_Torque_Driver.Initialize(CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);

            Reset_Internal_Variables;

         else
            CNT.This_Subsystem.Mode := 0 ;
            IOS_Pilot.Apply_Enabled := false;
            Ios_Pilot.Enable        := False;
            Ios_Pilot.Takeoff       := False;
            Reset_On_Off_Flags;
         end if;
      else
         if not Ios_Pilot.Enable then
            CNT.This_Subsystem.Mode := 0 ;
            IOS_Pilot.Apply_Enabled := false;
            Reset_On_Off_Flags;
         else
            --| check if IOS Pilot is beyond the disengaged limits
            if abs(Roll_Angle)  > CNT.IOS_Pilot_Roll_Driver.Maximum_Roll_Angle(
               CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver) then IOS_Pilot.Error_Number := 7;
            elsif IOS_Pilot.Error_Number = 7  then  IOS_Pilot.Error_Number := 0;
            end if;
            if abs(Pitch_Angle) > 50.0        then IOS_Pilot.Error_Number := 8;
            elsif IOS_Pilot.Error_Number = 8  then  IOS_Pilot.Error_Number := 0;
            end if;
            if abs(Roll_rate)   > 60.0        then IOS_Pilot.Error_Number := 9;
            elsif IOS_Pilot.Error_Number = 9  then  IOS_Pilot.Error_Number := 0;
            end if;
            if abs(Pitch_rate)  > 60.0         then IOS_Pilot.Error_Number := 10;
            elsif IOS_Pilot.Error_Number = 10  then  IOS_Pilot.Error_Number := 0;
            end if;
            if (IAS  < 70.0) and (Altitude_AGL > 20.0) and
              (not (JAT.At_Phase = 9))         then IOS_Pilot.Error_Number := 11;
            elsif IOS_Pilot.Error_Number = 11  then  IOS_Pilot.Error_Number := 0;
            end if;
            if Climb_rate  < - 9000.0       then IOS_Pilot.Error_Number := 12;
            elsif IOS_Pilot.Error_Number = 12  then  IOS_Pilot.Error_Number := 0;
            end if;
         end if;
      end if;

      if not (CNT.This_Subsystem.Mode = 0) then

         if CNT.This_Subsystem.Mode = 1 then
            --| ===================================================================================
            --| IOS PILOT is in Takeoff mode
            --| ===================================================================================

            --| First time on, go to Takeoff mode
            if CNT.This_Subsystem.TO_Phase = -1 then
               CNT.This_Subsystem.TO_Phase := 0;
               IOS_Pilot.TO_Is_started     := False;
               Ios_Pilot.TO_Is_Done        := False;
               IOS_Pilot.TO_Is_Ready       := False;
               IOS_Pilot.Takeoff           := False;
               IOS_Pilot.Request_Gear_Down := true;
            end if;

            if (CNT.This_Subsystem.TO_Phase >= 0) and
              (CNT.This_Subsystem.TO_Phase < 3) then

               --| Disable altitude hold
               IOS_Pilot.Altitude_Engaged := False;

               --| disable ios pilot if takeoff is uncheck after takeoff mode
               --| has been initiated.
               if (CNT.This_Subsystem.TO_Phase > 0) and
                 (CNT.This_Subsystem.TO_Phase < 3) and
                 (not IOS_Pilot.Takeoff) then
                  CNT.This_Subsystem.Mode := 0 ;
                  IOS_Pilot.Apply_Enabled := false;
                  Ios_Pilot.Enable        := False;
                  Ios_Pilot.Takeoff       := False;
                  Reset_On_Off_Flags;
                  IOS_Pilot.Error_Number  := 13;
               end if;

               case CNT.This_Subsystem.TO_Phase is
                  when 0 =>
                     --| TO Phase = 0 -------------------------------------

                     if Ios_Pilot.Takeoff then
                        if   not Ios_Pilot.TO_Is_started  then
                           JSA.Set_System_Freeze;
                           --| Check if we are already reposition to TO point
                           if (abs(Heading - JPATS_Reposition.Reference_Runway.Hdg)< 0.5) and
                             (Torque >0.0) and (Torque < 10.0) and
                             (Sqrt(V_North*V_North + V_East*V_East) < 10.0) and
                             (Altitude_AGL > 0.0) and (Altitude_AGL < 6.0) and
                             (not JSA.Crashed) then
                              Ios_Pilot.TO_Is_Started := True;
                              CNT.This_Subsystem.Reposition_Active := False;
                           else
                              --| Reposition To Takeoff
                              Jpats_Reposition.Takeoff;
                              Ios_Pilot.TO_Is_Started := True;
                              CNT.This_Subsystem.Reposition_Active := True;
                           end if;
                        end if;

                        TO_Ready := (Gear_Pos > 0.9) and (Torque > 3.0) and (Jpats_Landing_Gear.IOS_Pilot_Gear = 1);
                        if not TO_Ready then
                           --| Tell operator that TO mode is not ready if takeoff checkbox is checked
                           if (Gear_Pos < 0.9) or (Jpats_Landing_Gear.IOS_Pilot_Gear = 0) then
                              IOS_Pilot.Error_Number  := 21;
                           end if;
                           if (Torque < 3.0) then
                              IOS_Pilot.Error_Number  := 22;
                           end if;
                        elsif  IOS_Pilot.Error_Number  = 21 or IOS_Pilot.Error_Number  = 22  then
                           IOS_Pilot.Error_Number  := 0;
                        end if;

                        if (JPATS_Reposition.Reposition_Command.Active or
                            CNT.This_Subsystem.Reposition_Active)  then

                           CNT.This_Subsystem.Reposition_Active := True;

                           --| Wait until the  reposition is finished
                           if not (JPATS_Reposition.Reposition_Command.Active) and
                             (JAT.At_Phase /= 9)  and CNT.This_Subsystem.Reposition_Active  then
                              --| Reposition is finished check if To is ready
                              CNT.This_Subsystem.Reposition_Active := False;
                           end if;

                        elsif (JSA.Get_Flight_Freeze or ( not IOS_Pilot.TO_Is_Ready)) and
                          (not CNT.This_Subsystem.Reposition_Active)  and
                          TO_Ready  then

                           if JSA.Get_Flight_Freeze then
                              IOS_Pilot.TO_Is_Ready := True;
                           end if;

                           --| set the performance to climb and torque hold
                           IOS_Pilot.Performance_mode := 2;
                           Ios_Pilot.Ias_Enabled :=False;
                           Ios_Pilot.Climb_Rate_Enabled := True;
                           Ios_Pilot.Torque_Enabled :=True;

                           --| Pitch driver set
                           --| set the stick forward down
                           Pitch_Mode          := 2;
                           Selected_Altitude   := 5000.0;
                           Selected_Climb_rate := -100.0;
                           Selected_IAS        := 140.0;

                           --| Roll Driver set
                           Roll_Mode                := 1;
                           Selected_Heading         := JPATS_Reposition.Reference_Runway.Hdg;
                           Selected_Turn_Rate       := 2;
                           Selected_Turn_Direction  := 1;

                           --| Yaw Driver set
                           Yaw_Mode      := 1;

                           --| Torque Driver set
                           Torque_Mode            := 1;
                           Selected_Torque        := 30.0;
                           Selected_Time_Constant := 1.0;

                           CNT.IOS_Pilot_Pitch_Driver.Set_Reference
                             (Pitch_Mode,
                              Selected_Altitude,
                              Selected_Climb_Rate,
                              Selected_IAS,
                              Selected_Torque,
                              2,
                              CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

                           CNT.IOS_Pilot_Roll_Driver.Set_Reference
                             (Roll_Mode,
                              False,
                              Selected_Heading,
                              Heading,
                              Selected_Turn_Rate,
                              Selected_Turn_Direction,
                              Selected_Climb_Rate,
                              Selected_IAS,
                              Selected_Torque,
                              2,                              
                              CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);

                           CNT.IOS_Pilot_Yaw_Driver.Set_Reference
                             (Yaw_Mode,
                              Selected_Heading,
                              CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);

                           CNT.Ios_Pilot_Torque_Driver.Set_Reference
                             (Torque_Mode,
                              Selected_Torque,
                              Selected_IAS,
                              Selected_Time_Constant,
                              CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);

                           --| Set the reference for plotting the flightpath on the runway
                           CNT.This_Subsystem.start_Lat := Jsa.Get_North;
                           CNT.This_Subsystem.start_Long := Jsa.Get_East;

                        elsif (not JSA.Get_Flight_Freeze) and
                          (not CNT.This_Subsystem.Reposition_Active)  and
                          TO_Ready  and IOS_Pilot.TO_Is_Ready then
                           CNT.Ios_Pilot_Torque_Driver.Set_Reference
                             (1,
                              100.0,
                              140.0,
                              1.0,
                              CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);
                           CNT.This_Subsystem.TO_Phase := 1;
                           --| set the performance to climb and torque  hold
                           IOS_Pilot.Performance_mode := 2;
                           Ios_Pilot.Ias_Enabled :=false;
                           Ios_Pilot.Climb_Rate_Enabled := true;
                           Ios_Pilot.Torque_Enabled :=true;

                        end if;
                     else
                        --| disable ios pilot if takeoff is uncheck after takeoff mode
                        --| has been initiated.
                        if (IOS_Pilot.TO_Is_Ready) and
                          (not IOS_Pilot.Takeoff) then
                           CNT.This_Subsystem.Mode := 0 ;
                           IOS_Pilot.Apply_Enabled := false;
                           Ios_Pilot.Enable        := False;
                           Ios_Pilot.Takeoff       := False;
                           Reset_On_Off_Flags;
                           IOS_Pilot.Error_Number  := 13;
                        end if;

                        --| Do not let reposition on air once
                        --| IOS PILOT is already started on ground.
                        if JPATS_Reposition.Reposition_Command.Active and (Altitude_AGL > 20.0) then
                           CNT.This_Subsystem.Mode := 0 ;
                           IOS_Pilot.Apply_Enabled := false;
                           Ios_Pilot.Enable        := False;
                           Ios_Pilot.Takeoff       := False;
                           Reset_On_Off_Flags;
                           IOS_Pilot.Error_Number  := 13;
                        end if;
                     end if;

                  when 1 =>
                     --| TO Phase = 1 -------------------------------------

                     if (IAS > Rotation_Speed*0.85) then
                        --| Pitch driver set
                        --| set the performance to climb and torque hold
                        IOS_Pilot.Performance_mode := 2;
                        Ios_Pilot.Ias_Enabled :=False;
                        Ios_Pilot.Climb_Rate_Enabled := True;
                        Ios_Pilot.Torque_Enabled :=True;

                        Pitch_Mode          := 2;
                        Selected_Altitude   := 5000.0;
                        Selected_Climb_rate := Rotation_climb;
                        Selected_IAS        := 140.0;
                        Selected_Torque     := 100.0;
                        CNT.IOS_Pilot_Pitch_Driver.Set_Reference
                          (Pitch_Mode,
                           Selected_Altitude,
                           Selected_Climb_Rate,
                           Selected_IAS,
                           Selected_Torque,
                           2,
                           CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

                         CNT.This_Subsystem.TO_Phase := 2;
                     end if;

                  when 2 =>
                     --| TO Phase = 2 -------------------------------------

                     if (Altitude_AGL > Altitude_off) then
                        IOS_Pilot.Apply_Enabled     := true;
                        Ios_Pilot.Turn_Left_Enabled := True;
                        Ios_Pilot.Turn_Right_Enabled := True;
                        CNT.This_Subsystem.TO_Phase := 3;
                     end if;

                  when others =>
                     Log.Report(Event    => Body_File_Name & ": Case TO Phase",
                                Severity => Log.ERROR);
               end case;
            else
               CNT.This_Subsystem.Mode     := 2;
               IOS_Pilot.Takeoff           := False;
            end if;

            --| Update IOS Page data ---------------------------------------------------------
            if not IOS_Pilot.Altitude_Engaged then
               IOS_Pilot.Altitude := Altitude;
            end if;
            Ios_Pilot.Torque := torque;
            Ios_Pilot.IAS := IAS;
            Ios_pilot.Climb_Rate := Climb_Rate;

            Ios_Pilot.Heading := Heading - Magvar;

            if  (CNT.This_Subsystem.Mode = 2) or (Altitude_AGL > Altitude_off) then
               Ios_Pilot.Altitude := 5000.0;
               Ios_Pilot.Torque := 100.0;
               Ios_Pilot.IAS := 140.0;
               Ios_Pilot.Heading := JPATS_Reposition.Reference_Runway.Hdg - Magvar;

               IOS_Pilot.Performance_mode := 3;
               Ios_Pilot.Ias_Enabled :=True;
               Ios_Pilot.Climb_Rate_Enabled := False;
               Ios_Pilot.Torque_Enabled :=True;
            end if;

            if Ios_Pilot.Heading > 360.0 then
               Ios_Pilot.Heading := Ios_Pilot.Heading - 360.0;
            elsif Ios_Pilot.Heading < 0.0 then
               Ios_Pilot.Heading := Ios_Pilot.Heading + 360.0;
            end if;

            --| ===================================================================================
            --| End of IOS PILOT  in Takeoff mode
            --| ===================================================================================

         elsif CNT.This_Subsystem.Mode = 2 then
            --| ==================================================================================
            --| IOS PILOT is inflight mode
            --| ==================================================================================

            --| Turn off IOS PILOT if reposition to TO while it was already on in flight mode
            if  (JAT.At_Phase = 9) and JPATS_Reposition.Reposition_Command.On_Ground then
               CNT.This_Subsystem.Mode := 0 ;
               IOS_Pilot.Apply_Enabled := false;
               Ios_Pilot.Enable        := False;
               Reset_On_Off_Flags;
               Ios_Pilot.Takeoff       := False;
               IOS_Pilot.Error_Number  := 13;
            end if;

            --| announce warning if engine is off

            if  (IOS_Pilot.Error_Number  = 22) and (Torque > 3.0) then
               IOS_Pilot.Error_Number  := 0;
            end if;

            --| Go to TO mode if Takeoff is checked -------------
            if IOS_Pilot.Takeoff then
               CNT.This_Subsystem.Mode       := 1;
               CNT.This_Subsystem.TO_Phase   := 0;
               Ios_Pilot.TO_Is_started       := False;
               IOS_Pilot.TO_Is_Ready         := False;
               Ios_Pilot.TO_Is_Done          := False;
               IOS_Pilot.Apply_Enabled       := false;
               Ios_Pilot.Turn_Left_Enabled   := False;
               Ios_Pilot.Turn_Right_Enabled  := False;
               Reset_Internal_Variables;
            end if;

            --| The following is from takeoff mode ---------------

            --| TO Phase = 3 -------------------------------------
            if (CNT.This_Subsystem.TO_Phase = 3)  then
               IOS_Pilot.TO_Is_Done        := True;
               IOS_Pilot.Apply_Enabled     := true;
               Ios_Pilot.Turn_Left_Enabled := True;
               Ios_Pilot.Turn_Right_Enabled := True;
               CNT.This_Subsystem.TO_Phase := 4;

            elsif (CNT.This_Subsystem.TO_Phase = 4) then
               if  (IAS > IAS_Hold_Start)  then

                  --| set the performance to IAS and torque hold
                  IOS_Pilot.Performance_mode := 3;
                  Ios_Pilot.Ias_Enabled :=True;
                  Ios_Pilot.Climb_Rate_Enabled := False;
                  Ios_Pilot.Torque_Enabled :=True;

                  Pitch_Mode          := 1;
                  Selected_Altitude   := 5000.0;
                  Selected_Climb_rate := 1500.0;
                  Selected_IAS        := 140.0;
                  Torque_Mode         := 1;
                  Selected_Torque     := 100.0;
                  CNT.IOS_Pilot_Pitch_Driver.Set_Reference
                    (Pitch_Mode,
                     Selected_Altitude,
                     Selected_Climb_Rate,
                     Selected_IAS,
                     Selected_Torque,
                     3,
                     CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

                  CNT.Ios_Pilot_Torque_Driver.Set_Reference
                    (Torque_Mode,
                     Selected_Torque,
                     Selected_IAS,
                     1.0,
                     CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);

               end if;

               if IOS_Pilot.Apply or (IAS > 139.0) then
                  CNT.This_Subsystem.TO_Phase := 5;
                  Ios_Pilot.Apply := True;
               end if;
            end if;

            --| The following is when IOS Pilot starts in air.
            if (CNT.This_Subsystem.TO_Phase = -1) then
               CNT.This_Subsystem.TO_Phase := 5;
               Ios_Pilot.TO_Is_Done        := True;
               Ios_Pilot.Apply := True;
               --| set the performance to ias and climb hold
               IOS_Pilot.Performance_mode   := 1;
               Ios_Pilot.Ias_Enabled        :=true;
               Ios_Pilot.Climb_Rate_Enabled := True;
               Ios_Pilot.Torque_Enabled     :=false;
               --| set turn direction to closest
               Ios_Pilot.Turn_Direction     := 1;
               Ios_Pilot.Turn_Rate          := 2;

               --| set gear up
               IOS_Pilot.Request_Gear_Up := true;

               --| give info if engine torque is not sufficient.
               if (Torque < 3.0) then
                  IOS_Pilot.Error_Number  := 22;
               end if;
            end if;


            --| Process selected references, when apply button is pressed or turn left or right or fadering in -----------------------
            if (Ios_Pilot.Apply) or
               (Ios_Pilot.Turn_Left and not Turn_Left_Last_Pass) or 
               (Ios_Pilot.Turn_Right and not Turn_Right_Last_Pass) or
               (Fadering_Left) or
               (Fadering_Right) then

               --| Roll and Yaw  Driver set -----------------------------------------------------
               Roll_Mode := 1;
               Yaw_Mode  := 1;

               Selected_Turn_Rate := Ios_Pilot.Turn_Rate;

               if ( (not (Ios_Pilot.Turn_Left and Ios_Pilot.Synchronize_Heading)) and
                    (not (Ios_Pilot.Turn_Right and Ios_Pilot.Synchronize_Heading)) and
                    (not (Fadering_Left and Ios_Pilot.Synchronize_Heading)) and
                    (not (Fadering_Right and Ios_Pilot.Synchronize_Heading)) and 
                    (Ios_Pilot.Apply) ) then
                  Ios_Pilot.Turn_Left := False;
                  Ios_Pilot.Turn_Right := False;
                  Turn_Left_Last_Pass := False;
                  Turn_Right_Last_Pass := False;
                  Reset_Fader;
                  Selected_Turn_Direction := Ios_Pilot.Turn_Direction;
                  Selected_Heading := Ios_Pilot.Heading + Magvar;
               elsif (Ios_Pilot.Turn_Left and not Turn_Left_Last_Pass) then
                  if Fadering_Right or Fadering_Left or Ios_Pilot.Turn_Right then
                     CNT.IOS_Pilot_Roll_Driver.Init_Rapid_Direction_Change(
                                           Roll_Angle,
                                           CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
                  end if;
                  Selected_Heading := Heading - CNT.IOS_Pilot_Roll_Driver.Heading_Buffer;
                  Reset_Fader;                
                  Turn_Left_Last_Pass := True;
                  Selected_Turn_Direction  := 2;
                  IOS_Pilot.Turn_Right := False;
                  Turn_Right_Last_Pass := False;      
               elsif (Ios_Pilot.Turn_Right and not Turn_Right_Last_Pass) then
                  if Fadering_Left or Fadering_Right or Ios_Pilot.Turn_Left then
                     CNT.IOS_Pilot_Roll_Driver.Init_Rapid_Direction_Change(
                                           Roll_Angle,
                                           CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
                  end if;
                  Reset_Fader;
                  Selected_Heading := Heading + CNT.IOS_Pilot_Roll_Driver.Heading_Buffer;
                  Turn_Right_Last_Pass := True;
                  Selected_Turn_Direction  := 3;
                  IOS_Pilot.Turn_Left := False;
                  Turn_Left_Last_Pass := False;
               elsif Fadering_Left then
                  Selected_Turn_Direction := 2;
                  Selected_Heading := Heading;
               elsif Fadering_Right then
                  Selected_Turn_Direction := 3;
                  Selected_Heading := Heading;
               elsif Ios_Pilot.Turn_Left then
                  Selected_Turn_Direction := 2;
                  Turn_Left_Last_Pass := True;
               elsif Ios_Pilot.Turn_Right then
                  Selected_Turn_Direction  := 3;
                  Turn_Right_Last_Pass := True;
               else
                  Log.Report("Shouldn't be here");
               end if;
               if Selected_Heading > 360.0 then
                  Selected_Heading := Selected_Heading - 360.0;
               elsif Selected_Heading < 0.0 then
                  Selected_Heading := Selected_Heading + 360.0;
               end if;

               case Ios_pilot.Performance_mode is
                  when 1 =>
                     --| IAS and Climb Rate hold -------------------------------------------------

                     --| Pitch driver set ----------
                     --| set Pitch driver to climb hold
                     Pitch_Mode               := 2;
                     Selected_Altitude        := Ios_Pilot.Altitude;
                     Selected_Climb_rate      := Ios_pilot.Climb_Rate;
                     Selected_IAS             := Ios_Pilot.IAS;


                     --| Torque Driver set --------
                     --| set torque driver to IAS hold
                     Torque_Mode            := 2;
                     Selected_Torque        := Ios_Pilot.Torque;
                     Selected_Time_Constant := 1.0;
                  when 2 =>
                     --| Climb Rate and Torque hold ----------------------------------------------

                     --| Pitch driver set ---------
                     --| set Pitch driver to climb hold
                     Pitch_Mode               := 2;
                     Selected_Altitude        := Ios_Pilot.Altitude;
                     Selected_Climb_rate      := Ios_Pilot.Climb_Rate;
                     Selected_IAS             := Ios_Pilot.IAS;


                     --| Torque Driver set --------
                     --| set torque driver to Torque hold
                     Torque_Mode            := 1;
                     Selected_Torque        := Ios_Pilot.Torque;
                     Selected_Time_Constant := 1.0;
                  when 3 =>
                     --| Torque and IAS hold ----------------------------------------------------

                     --| Pitch driver set --------
                     --| set Pitch driver to IAS hold
                     Pitch_Mode               := 1;
                     Selected_Altitude        := Ios_Pilot.Altitude;
                     Selected_Climb_rate      := Ios_pilot.Climb_Rate;
                     Selected_IAS             := Ios_Pilot.IAS;


                     --| Torque Driver set -------
                     --| set torque driver to Torque hold
                     Torque_Mode            := 1;
                     Selected_Torque        := Ios_Pilot.Torque;
                     Selected_Time_Constant := 1.0;
                  when others =>
                     null;
               end case;

               CNT.IOS_Pilot_Pitch_Driver.Set_Reference
                 (Pitch_Mode,
                  Selected_Altitude,
                  Selected_Climb_Rate,
                  Selected_IAS,
                  Selected_Torque,
                  Ios_pilot.Performance_Mode,
                  CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

               CNT.IOS_Pilot_Roll_Driver.Set_Reference
                 (Roll_Mode,
                  (Ios_Pilot.Turn_Left or Ios_Pilot.Turn_Right),
                  Selected_Heading,
                  Heading,
                  Selected_Turn_Rate,
                  Selected_Turn_Direction,
                  Selected_Climb_Rate,
                  Selected_IAS,
                  Selected_Torque,
                  Ios_pilot.Performance_Mode,
                  CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);

               CNT.IOS_Pilot_Yaw_Driver.Set_Reference
                 (Yaw_Mode,
                  Selected_Heading,
                  CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);

               CNT.Ios_Pilot_Torque_Driver.Set_Reference
                 (Torque_Mode,
                  Selected_Torque,
                  Selected_IAS,
                  Selected_Time_Constant,
                  CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);
            end if;

            if (not Ios_Pilot.Turn_Left) and Turn_Left_Last_Pass then
               Turn_Left_Last_Pass := False;
               if (not Ios_Pilot.Turn_Right) then
                  CNT.IOS_Pilot_Roll_Driver.Init_Rapid_Direction_Change(
                        Roll_Angle,
                        CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
                  Fadering_Left := True;
                  Fadering_Heading_Count := 0;
               end if;
            elsif (not Ios_Pilot.Turn_Right) and Turn_Right_Last_Pass then
               Turn_Right_Last_Pass := False;
               if (not Ios_Pilot.Turn_Left) then
                  CNT.IOS_Pilot_Roll_Driver.Init_Rapid_Direction_Change(
                        Roll_Angle,
                        CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
                  Fadering_Right := True;
                  Fadering_Heading_Count := 0;
               end if;
            end if;
            if Ios_Pilot.Turn_Left or Ios_Pilot.Turn_Right or Fadering_Left or Fadering_Right then
               if Fadering_Left or Fadering_Right then
                  if (abs(JSA.P_Dot) <= P_DOT_THRESHOLD) and (abs(Roll_Angle) <= ROLL_ANGLE_THRESHOLD) then
                     Fadering_Heading_Count := Fadering_Heading_Count + 1;
                  end if;
                  if (Fadering_Heading_Count >= FADERING_HEADING_THRESHOLD) then
                     Reset_Fader;
                  end if;
               end if;
               if Ios_Pilot.Synchronize_Heading then
                  Ios_Pilot.Heading := Heading - Magvar;
               end if;
            end if;
         end if;


         --| #################################################################################
         --| PITCH DRIVER UPDATE PROCESS
         --| #################################################################################

         if CNT.IOS_Pilot_Pitch_Driver.Is_On (CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver) then

            --| -------------------------------------------------------------------
            --| Update the necessary gain changes
            --| -------------------------------------------------------------------
            KP_Pitch   := 5.0;
            KI_Pitch   := 10.0;


            Gain_Torque_Compensator     := 0.1;

            Max_Pitch_Rate :=Float'Min(9.0,Max_Pitch_Rate);


            KD_Pitch :=It.Singly_Indexed.Interpolate(TAS/1.6787,KD_PITCH_T'access);
            KP_Climb :=It.Singly_Indexed.Interpolate(TAS/1.6787,KP_CLIMB_T'access);
            KI_Climb :=It.Singly_Indexed.Interpolate(TAS/1.6787,KI_CLIMB_T'access);
            Max_Climb_Rate :=It.Singly_Indexed.Interpolate(TAS/1.6787,MAXCLIMBRATE_T'access);
            KI_IAS :=It.Doubly_Indexed.Interpolate(TAS/1.6787,Torque,KI_IAS_T'access);
            KP_IAS :=It.Doubly_Indexed.Interpolate(TAS/1.6787,Torque,KP_IAS_T'access);
            KD_IAS :=It.Doubly_Indexed.Interpolate(TAS/1.6787,Torque,KD_IAS_T'access);
            Max_IAS_Err :=It.Doubly_Indexed.Interpolate(TAS/1.6787,Torque,MAXIASER_T'access);
            Max_IAS_Rate := It.Singly_Indexed.Interpolate(Torque,MAXIASRATE_T'access);
            KI_Altitude :=It.Singly_Indexed.Interpolate(TAS/1.6787,KI_ALTITUDE_T'access);

            if Ios_Pilot.TO_Is_Done then

               if (Flap_Pos < 5.0) then
                  Limited_AOA := It.Doubly_Indexed.Interpolate(IAS,GW,LIMITEDAOA0_T'access);
               elsif (Flap_Pos < 30.0) then
                  Limited_AOA := It.Doubly_Indexed.Interpolate(IAS,GW,LIMITEDAOA1_T'access);
               else
                  Limited_AOA := It.Doubly_Indexed.Interpolate(IAS,GW,LIMITEDAOA2_T'access);
               end if;
               Gain_Stall_Correction := 10.0;
            else
               Limited_AOA := Float'Max(-4.0,Float'Min(5.0,AOA));
               Gain_Stall_Correction := 0.0;
            end if;


            --| Stall warning
            if (AOA > CNT.IOS_Pilot_Pitch_Driver.Get_AOA_Stall )  and
              (Climb_Rate < -750.0) then
               IOS_Pilot.Error_Number  := 23;
               IOS_Pilot.Altitude_Engaged := False;
            elsif (IOS_Pilot.Error_Number  = 23) and
              (AOA < CNT.IOS_Pilot_Pitch_Driver.Get_AOA_Stall )  then
               IOS_Pilot.Error_Number  := 0;
            end if;

            --| Set the altitude hold
            CNT.IOS_Pilot_Pitch_Driver.Set_Altitude_Hold_Enable
              (IOS_Pilot.Altitude_Engaged,CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

            if not IOS_Pilot.Altitude_Engaged then
               CNT.IOS_Pilot_Pitch_Driver.Set_Selected_Altitude
                 (-10000.0,CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
            end if;


            if IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 1) then

               --| -------------------------------------------------------------------------------------
               --| TUNING Pitch DRIVER - Pitch Hold Loop
               --| -------------------------------------------------------------------------------------
               Tuning_Mode                    := 1;
               --| Update from IOS Pilot Debug Page
               Comm_Pitch      := IOS_Pilot.Comm_Pitch;
               KP_Pitch        := IOS_Pilot.KP_Pitch;
               KI_Pitch        := IOS_Pilot.KI_Pitch;
               KD_Pitch        := IOS_Pilot.KD_Pitch;
               Max_Pitch_Rate  := IOS_Pilot.Max_Pitch_Rate;

               IOS_Pilot.Comm_Climb      :=Climb_Rate;
               IOS_Pilot.KP_Climb        := KP_Climb;
               IOS_Pilot.KI_Climb        := KI_Climb;
               IOS_Pilot.Max_Delta_Climb := Max_Climb_Rate;

               IOS_Pilot.Comm_IAS        := KD_IAS;
               IOS_Pilot.KP_IAS          := KP_IAS;
               IOS_Pilot.KI_IAS          := KI_IAS;

               IOS_Pilot.Max_Delta_IAS     := Max_IAS_Err;
               --IOS_Pilot.Max_Delta_IAS    := Max_IAS_Rate;

               IOS_Pilot.KP_Altitude     := KP_Altitude;
               IOS_Pilot.KI_Altitude     := KI_Altitude ;
               IOS_Pilot.KD_Altitude     := KD_Altitude;
               IOS_Pilot.Max_Altitude_Err:= Max_Alt_Err;





            elsif IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 2) then

               --| -------------------------------------------------------------------------------------
               --| TUNING PITCH DRIVER - Climb Hold Loop
               --| -------------------------------------------------------------------------------------

               Tuning_Mode                    := 2;
               --| Update from IOS Pilot Debug Page
               IOS_Pilot.Comm_Pitch      := Pitch_angle;
               IOS_Pilot.KP_Pitch        := KP_Pitch;
               IOS_Pilot.KI_Pitch        := KI_Pitch;
               IOS_Pilot.KD_Pitch        := KD_Pitch;
               IOS_Pilot.Max_Pitch_Rate  := Max_Pitch_Rate;

               IOS_Pilot.Comm_Climb      := Climb_Rate;
               KP_Climb                  := IOS_Pilot.KP_Climb;
               KI_Climb                  := IOS_Pilot.KI_Climb;
               Max_Climb_Rate            := IOS_Pilot.Max_Delta_Climb;

               IOS_Pilot.KP_IAS          := KP_IAS;
               IOS_Pilot.KI_IAS          := KI_IAS;
               IOS_Pilot.Comm_IAS        := KD_IAS;
               IOS_Pilot.Max_Delta_IAS   := Max_IAS_Err;
               --IOS_Pilot.Max_Delta_IAS    := Max_IAS_Rate;

               IOS_Pilot.KP_Altitude     := KP_Altitude;
               IOS_Pilot.KI_Altitude     := KI_Altitude ;
               IOS_Pilot.KD_Altitude     := KD_Altitude;
               IOS_Pilot.Max_Altitude_Err:= Max_Alt_Err;






            elsif IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 3) then


               --| -------------------------------------------------------------------------------------
               --| TUNING PITCH DRIVER - IAS Hold Loop
               --| -------------------------------------------------------------------------------------
               Tuning_Mode                    := 3;
               --| Update from IOS Pilot Debug Page
               IOS_Pilot.Comm_Pitch      := Pitch_Angle;
               IOS_Pilot.KP_Pitch        := KP_Pitch;
               IOS_Pilot.KI_Pitch        := KI_Pitch;
               IOS_Pilot.KD_Pitch        := KD_Pitch;
               IOS_Pilot.Max_Pitch_Rate  := Max_Pitch_Rate;

               IOS_Pilot.Comm_Climb      := Climb_Rate;
               IOS_Pilot.KP_Climb        := KP_Climb;
               IOS_Pilot.KI_Climb        := KI_Climb;
               IOS_Pilot.Max_Delta_Climb := Max_Climb_Rate;

               KP_IAS          := IOS_Pilot.KP_IAS;
               KI_IAS          := IOS_Pilot.KI_IAS;
               KD_IAS          := IOS_Pilot.Comm_IAS;
               Max_IAS_Err     := IOS_Pilot.Max_Delta_IAS;
               --IOS_Pilot.Max_Delta_IAS    := Max_IAS_Rate;

               IOS_Pilot.KP_Altitude     := KP_Altitude;
               IOS_Pilot.KI_Altitude     := KI_Altitude;
               IOS_Pilot.KD_Altitude     := KD_Altitude;
               IOS_Pilot.Max_Altitude_Err:= Max_Alt_Err;




            elsif IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 4) then


               --| -------------------------------------------------------------------------------------
               --| TUNING Pitch DRIVER - Altitude Hold Loop
               --| -------------------------------------------------------------------------------------
               Tuning_Mode                    := 4;
               --| Update from IOS Pilot Debug Page
               IOS_Pilot.Comm_Pitch      := Pitch_Angle;
               IOS_Pilot.KP_Pitch        := KP_Pitch;
               IOS_Pilot.KI_Pitch        := KI_Pitch;
               IOS_Pilot.KD_Pitch        := KD_Pitch;
               IOS_Pilot.Max_Pitch_Rate  := Max_Pitch_Rate;

               IOS_Pilot.Comm_Climb      := Climb_Rate;
               IOS_Pilot.KP_Climb        := KP_Climb;
               IOS_Pilot.KI_Climb        := KI_Climb;
               IOS_Pilot.Max_Delta_Climb := Max_Climb_Rate;

               IOS_Pilot.KP_IAS          := KP_IAS;
               IOS_Pilot.KI_IAS          := KI_IAS;
               IOS_Pilot.Comm_IAS        := KD_IAS;
               IOS_Pilot.Max_Delta_IAS     := Max_IAS_Err;
               --IOS_Pilot.Max_Delta_IAS    := Max_IAS_Rate;

               KP_Altitude     := IOS_Pilot.KP_Altitude;
               KI_Altitude     := IOS_Pilot.KI_Altitude;
               KD_Altitude     := IOS_Pilot.KD_Altitude;
               Max_Alt_Err     := IOS_Pilot.Max_Altitude_Err;







            else

               --| -------------------------------------------------------------------------------------
               --| NORMAL MODE
               --| -------------------------------------------------------------------------------------
               Tuning_Mode                    := 0;
               --| Update IOS Pilot Debug Page
               IOS_Pilot.Comm_Pitch      := Pitch_Angle;
               IOS_Pilot.KP_Pitch        := KP_Pitch;
               IOS_Pilot.KI_Pitch        := KI_Pitch;
               IOS_Pilot.KD_Pitch        := KD_Pitch;
               IOS_Pilot.Max_Pitch_Rate  := Max_Pitch_Rate;

               IOS_Pilot.Comm_Climb      := Climb_Rate;
               IOS_Pilot.KP_Climb        := KP_Climb;
               IOS_Pilot.KI_Climb        := KI_Climb;
               IOS_Pilot.Max_Delta_Climb := Max_Climb_Rate;

               IOS_Pilot.KP_IAS          := KP_IAS;
               IOS_Pilot.KI_IAS          := KI_IAS;
               IOS_Pilot.Comm_IAS        := KD_IAS;
               IOS_Pilot.Max_Delta_IAS   := Max_IAS_Err;
               --IOS_Pilot.Max_Delta_IAS    := Max_IAS_Rate;

               IOS_Pilot.KP_Altitude     := KP_Altitude;
               IOS_Pilot.KI_Altitude     := KI_Altitude ;
               IOS_Pilot.KD_Altitude     := KD_Altitude;
               IOS_Pilot.Max_Altitude_Err:= Max_Alt_Err;


            end if;


            case IOS_Pilot.Ios_Pilot_Mode is
               when 1 =>
                  --| Set the plot variables
                  IOS_Pilot.Misc_Var1 := Pitch_Angle;
                  IOS_Pilot.Misc_Var2 := IOS_Pilot.Comm_Pitch;
                  IOS_Pilot.Misc_Var3 := Pitch_Rate;
                  IOS_Pilot.Misc_Var4 := 0.0;
                  IOS_Pilot.Misc_Var5 := Elevator_Deflection;

               when 2  =>
                  --| Set the plot variables
                  IOS_Pilot.Misc_Var1 := Pitch_Angle;
                  IOS_Pilot.Misc_Var2 := Climb_rate;
                  IOS_Pilot.Misc_Var3 := CNT.IOS_Pilot_Pitch_Driver.Selected_Climb_Rate
                    (An_Instance  => CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
                  IOS_Pilot.Misc_Var4 := 0.0;
                  IOS_Pilot.Misc_Var5 := Elevator_Deflection;
               when 3  =>
                  --| Set the plot variables
                  IOS_Pilot.Misc_Var1 := Pitch_Angle;
                  IOS_Pilot.Misc_Var2 := Climb_Rate;
                  IOS_Pilot.Misc_Var3 := IOS_Pilot.Comm_Climb;
                  IOS_Pilot.Misc_Var4 := IAS;
                  IOS_Pilot.Misc_Var5 := CNT.IOS_Pilot_Pitch_Driver.Selected_IAS
                    (An_Instance  => CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
               when 4  =>
                  --| Set the plot variables
                  IOS_Pilot.Misc_Var1 := Pitch_Angle;
                  IOS_Pilot.Misc_Var2 := Climb_Rate;
                  IOS_Pilot.Misc_Var3 := IAS;
                  IOS_Pilot.Misc_Var4 := Altitude;
                  IOS_Pilot.Misc_Var5 := CNT.IOS_Pilot_Pitch_Driver.Selected_Altitude
                    (An_Instance  => CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
               when others =>
                  null;
            end case;

            CNT.IOS_Pilot_Pitch_Driver.Set_Gain
              (Tuning_Mode,
               Comm_Pitch,
               KP_Pitch,
               KI_Pitch,
               KD_Pitch,
               Max_Pitch_Rate,

               KP_Climb,
               KI_Climb,
               Max_Climb_Rate,

               KP_IAS,
               KI_IAS,
               KD_IAS,
               Max_IAS_Err,
               Max_IAS_Rate,

               KP_Altitude,
               KI_Altitude,
               KD_Altitude,
               Max_Alt_Err,

               Gain_Roll_Compensator,
               Gain_torque_Compensator,
               Gain_Stall_Correction,


               Limited_Aoa,
               CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

            if  (not (JSA.Get_Flight_Freeze)) and (CNT.This_Subsystem.TO_Phase > 0) then
               Bypass := False;
            else
               Bypass := True;
            end if;

            CNT.IOS_Pilot_Pitch_Driver.Update
              (Altitude,
               Climb_Rate,
               IAS,
               Roll_Angle,
               Pitch_Angle,
               AOA,
               Pitch_Rate,
               TAS,
               Altitude_AGL,
               Elevator_Deflection,
               Flap_Pos,
               Nx,
               Torque,
               Pitch_Pilot_Force,
               Dt,
               Bypass,
               CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);


         end if;



         --| #################################################################################
         --| ROLL DRIVER UPDATE PROCESS
         --| #################################################################################

         if CNT.IOS_Pilot_Roll_Driver.Is_On(CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver) or
            Ios_Pilot.Turn_Right or
            Ios_Pilot.Turn_Left or
            Fadering_Left or
            Fadering_Right then


            --| -------------------------------------------------------------------
            --| Update the necessary gain changes
            --| -------------------------------------------------------------------

            KI_Turn_Rate :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,KI_TURN_T'access);

            Delta_Max_Selected_Turn :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,DMAXTURN_T'access);

            Max_Turn_Err :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,MAXTURN_T'access);

            KP_Heading :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,KP_HEADING_T'access);

            KI_Heading :=It.Doubly_Indexed.Interpolate
              (TAS/1.6787,Flap_Pos,KI_HEADING_T'access);

            KD_Heading :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,KD_HEADING_T'access);

            Max_Heading_Err := It.Doubly_Indexed.Interpolate
              (TAS/1.6787,Flap_Pos,MAXHDG_T'access);



            if IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 5) then

               --| -------------------------------------------------------------------------------------
               --| TUNING ROLL DRIVER - Roll Hold Loop
               --| -------------------------------------------------------------------------------------

               Tuning_Mode             := 1;
               Comm_Roll               := IOS_Pilot.Comm_Roll;
               KP_Roll                 := IOS_Pilot.KP_Roll;
               KI_Roll                 := IOS_Pilot.KI_Roll;
               KD_Roll                 := IOS_Pilot.KD_Roll;
               Max_Roll_Rate           := IOS_Pilot.Max_Roll_Rate;

               IOS_Pilot.Comm_Turn_Rate:= Turn_Rate;
               IOS_Pilot.KP_Turn_Rate  := KP_Turn_Rate;
               IOS_Pilot.KI_Turn_Rate  := KI_Turn_Rate;
               IOS_Pilot.Max_Delta_Turn:= Max_Turn_Err;

               IOS_Pilot.KP_Heading    := KP_Heading;
               IOS_Pilot.KI_Heading    := KI_Heading;
               IOS_Pilot.KD_Heading    := KD_Heading;
               IOS_Pilot.Max_Heading_Err := Max_Heading_Err;



               --| Set the plot variables
               IOS_Pilot.Misc_Var1 := Roll_Angle;
               IOS_Pilot.Misc_Var2 := IOS_Pilot.Comm_Roll;
               IOS_Pilot.Misc_Var3 := Roll_Rate;
               IOS_Pilot.Misc_Var4 := 0.0;
               IOS_Pilot.Misc_Var5 := Aileron_Deflection;

            elsif IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 6) then

               --| -------------------------------------------------------------------------------------
               --| TUNING ROLL DRIVER - Turn Rate Hold Loop
               --| -------------------------------------------------------------------------------------

               Tuning_Mode            := 2;
               IOS_Pilot.Comm_Roll    := Roll_angle;
               IOS_Pilot.KP_Roll      := KP_Roll;
               IOS_Pilot.KI_Roll      := KI_Roll;
               IOS_Pilot.KD_Roll      := KD_Roll;
               IOS_Pilot.Max_Roll_Rate:= Max_Roll_Rate;

               IOS_Pilot.Comm_Turn_Rate :=  Turn_Rate;
               KP_Turn_Rate           := IOS_Pilot.KP_Turn_Rate;
               KI_Turn_Rate           := IOS_Pilot.KI_Turn_Rate;
               Max_Turn_Err           := IOS_Pilot.Max_Delta_Turn;

               IOS_Pilot.KP_Heading   := KP_Heading;
               IOS_Pilot.KI_Heading   := KI_Heading;
               IOS_Pilot.KD_Heading   := KD_Heading;
               IOS_Pilot.Max_Heading_Err := Max_Heading_Err;

               --| Set the plot variables
               IOS_Pilot.Misc_Var1 := Roll_angle;
               IOS_Pilot.Misc_Var2 := Turn_Rate;
               IOS_Pilot.Misc_Var3 := IOS_Pilot.Comm_Turn_Rate;
               IOS_Pilot.Misc_Var4 := Aileron_Deflection;
               IOS_Pilot.Misc_Var5 := Rudder_Deflection;



            elsif IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 7) then
               --| -------------------------------------------------------------------------------------

               --| TUNING ROLL DRIVER - Heading Hold Loop
               --| -------------------------------------------------------------------------------------

               Tuning_Mode            := 3;
               IOS_Pilot.Comm_Roll    := Roll_angle;
               IOS_Pilot.KP_Roll      := KP_Roll;
               IOS_Pilot.KI_Roll      := KI_Roll;
               IOS_Pilot.KD_Roll      := KD_Roll;
               IOS_Pilot.Max_Roll_Rate:= Max_Roll_Rate;

               IOS_Pilot.Comm_Turn_Rate:= Turn_Rate;
               IOS_Pilot.KP_Turn_Rate := KP_Turn_Rate;
               IOS_Pilot.KI_Turn_Rate := KI_Turn_Rate;
               IOS_Pilot.Max_Delta_Turn := Max_Turn_Err;

               KP_Heading             := IOS_Pilot.KP_Heading;
               KI_Heading             := IOS_Pilot.KI_Heading;
               KD_Heading             := IOS_Pilot.KD_Heading;
               Max_Heading_Err        := IOS_Pilot.Max_Heading_Err;


               --| Set the plot variables
               IOS_Pilot.Misc_Var1 := Turn_Rate;
               IOS_Pilot.Misc_Var2 := Heading  - Magvar;
               IOS_Pilot.Misc_Var3 := CNT.Ios_Pilot_Roll_Driver.Selected_Heading
                 (CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver) - Magvar;
               IOS_Pilot.Misc_Var4 := Aileron_Deflection;
               IOS_Pilot.Misc_Var5 := Rudder_Deflection;
            else

               --| -------------------------------------------------------------------------------------
               --| NORMAL MODE
               --| -------------------------------------------------------------------------------------

               Tuning_Mode            := 0;
               IOS_Pilot.Comm_Roll    := Roll_angle;
               IOS_Pilot.KP_Roll      := KP_Roll;
               IOS_Pilot.KI_Roll      := KI_Roll;
               IOS_Pilot.KD_Roll      := KD_Roll;
               IOS_Pilot.Max_Roll_Rate:= Max_Roll_Rate;

               IOS_Pilot.Comm_Turn_Rate:= Turn_Rate;
               IOS_Pilot.KP_Turn_Rate := KP_Turn_Rate;
               IOS_Pilot.KI_Turn_Rate := KI_Turn_Rate;
               IOS_Pilot.Max_Delta_Turn := Max_Turn_Err;

               IOS_Pilot.KP_Heading   := KP_Heading;
               IOS_Pilot.KI_Heading   := KI_Heading;
               IOS_Pilot.KD_Heading   := KD_Heading;
               IOS_Pilot.Max_Heading_Err := Max_Heading_Err;


            end if;

            case IOS_Pilot.Ios_Pilot_Mode is
               when 5 =>
                  --| Set the plot variables
                  IOS_Pilot.Misc_Var1 := Roll_Angle;
                  IOS_Pilot.Misc_Var2 := IOS_Pilot.Comm_Roll;
                  IOS_Pilot.Misc_Var3 := Roll_Rate;
                  IOS_Pilot.Misc_Var4 := 0.0;
                  IOS_Pilot.Misc_Var5 := Aileron_Deflection;

               when 6  =>
                  --| Set the plot variables
                  IOS_Pilot.Misc_Var1 := Roll_angle;
                  IOS_Pilot.Misc_Var2 := Turn_Rate;
                  IOS_Pilot.Misc_Var3 := IOS_Pilot.Comm_Turn_Rate;
                  IOS_Pilot.Misc_Var4 := Aileron_Deflection;
                  IOS_Pilot.Misc_Var5 := Rudder_Deflection;
               when 7  =>
                  --| Set the plot variables
                  IOS_Pilot.Misc_Var1 := Turn_Rate;
                  IOS_Pilot.Misc_Var2 := Heading  - Magvar;
                  IOS_Pilot.Misc_Var3 := CNT.Ios_Pilot_Roll_Driver.Selected_Heading
                    (CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver) - Magvar;
                  IOS_Pilot.Misc_Var4 := Aileron_Deflection;
                  IOS_Pilot.Misc_Var5 := Rudder_Deflection;

               when others =>
                  null;
            end case;

            CNT.IOS_Pilot_Roll_Driver.Set_Gain
              (Tuning_Mode  => Tuning_Mode,
               Comm_Roll    => Comm_Roll,
               KP_Roll      => KP_Roll,
               KI_Roll      => KI_Roll,
               KD_Roll      => KD_Roll,
               Max_Roll_Rate=> Max_Roll_Rate,
               Comm_Turn_Rate=> Comm_Turn_Rate,
               KP_Turn_Rate => KP_Turn_Rate,
               KI_Turn_Rate => KI_Turn_Rate,
               Delta_Max_Selected_Turn=> Delta_Max_Selected_Turn,
               Max_Turn_Err => Max_Turn_Err,
               KP_Heading   => KP_Heading,
               KI_Heading   => KI_Heading,
               KD_Heading   => KD_Heading,
               Max_Heading_Err => Max_Heading_Err,
               An_Instance  => CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);

            if (not (JSA.Get_Flight_Freeze)) and (CNT.This_Subsystem.TO_Phase > 0) then

               CNT.IOS_Pilot_Roll_Driver.Update
                 (Turn_Rate,
                  Ios_Pilot.Turn_Right,
                  Ios_Pilot.Turn_Left,
                  Heading,
                  Roll_Rate,
                  Roll_Angle,
                  Pitch_Angle,
                  Aileron_Deflection,
                  TAS,
                  Altitude_AGL,
                  Flap_Pos,
                  Gear_Pos,
                  Roll_Pilot_Force,
                  Dt,
                  CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);

            end if;

         end if;


         --| #################################################################################
         --| YAW DRIVER UPDATE PROCESS
         --| #################################################################################
         if CNT.IOS_Pilot_Yaw_Driver.Is_On (CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver) then



            --| -------------------------------------------------------------------
            --| Update the necessary gain changes
            --| -------------------------------------------------------------------
            KP_Lat_Acc :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,KP_LAT_T'access);

            KI_Lat_Acc :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,KI_LAT_T'access);

            KD_Lat_Acc :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,KD_LAT_T'access);

            Yaw_gain :=It.Singly_Indexed.Interpolate
              (TAS/1.6787,YAW_GAIN_T'access);




            if IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 8) then
               --| TUNING YAW DRIVER
               Tuning_Mode   := 1;
               KP_Sideslip   := IOS_Pilot.KP_Sideslip;
               KP_Lat_Acc    := IOS_Pilot.KP_Lat_Acc;
               KI_Lat_Acc    := IOS_Pilot.KP_Slipball;
               KD_Lat_Acc    := IOS_Pilot.KI_Slipball;
               Yaw_Gain      := IOS_Pilot.Yaw_Gain;
               Heading_Gain  := IOS_Pilot.KP_Yaw_Heading;

               --| Set the plot variables
               IOS_Pilot.Misc_Var1 := Lateral_Acc;
               IOS_Pilot.Misc_Var2 := Sideslip_Angle;
               IOS_Pilot.Misc_Var3 := 0.0;
               IOS_Pilot.Misc_Var4 := 0.0;
               IOS_Pilot.Misc_Var5 := Rudder_Deflection;
            elsif IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 10) then

               --| TUNING HEADING ON GROUND
               Tuning_Mode   := 1;
               KP_Sideslip   := IOS_Pilot.KP_Sideslip;
               KP_Lat_Acc    := IOS_Pilot.KP_Lat_Acc;
               KI_Lat_Acc    := IOS_Pilot.KP_Slipball;
               KD_Lat_Acc    := IOS_Pilot.KI_Slipball;
               Yaw_Gain      := IOS_Pilot.Yaw_Gain;
               Heading_Gain  := IOS_Pilot.KP_Yaw_Heading;


               --| Set the plot variables
               IOS_Pilot.Misc_Var5 := Rudder_Deflection;

            else

               Tuning_Mode := 0;
               IOS_Pilot.KP_Sideslip   := KP_Sideslip;
               IOS_Pilot.KP_Lat_Acc    := KP_Lat_Acc;
               IOS_Pilot.KP_Slipball   := KI_Lat_Acc;
               IOS_Pilot.KI_Slipball   := KD_Lat_Acc;
               IOS_Pilot.Yaw_Gain      := Yaw_Gain;
               IOS_Pilot.KP_Yaw_Heading:= Heading_Gain;
            end if;

            CNT.IOS_Pilot_Yaw_Driver.Set_Gain
                 (Tuning_Mode  => Tuning_Mode,
                  KP_Sideslip  => KP_Sideslip,
                  KP_Lat_Acc   => KP_Lat_Acc,
                  KI_Lat_Acc   => KI_Lat_Acc,
                  KD_Lat_Acc   => KD_Lat_Acc,
                  Yaw_Gain     => Yaw_Gain,
                  Heading_Gain => Heading_Gain,
                  An_Instance  => CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);

            if (not (JSA.Get_Flight_Freeze)) and  (CNT.This_Subsystem.TO_Phase > 0) then

               CNT.IOS_Pilot_Yaw_Driver.Update
                 (Heading,
                  Altitude_AGL,
                  TAS,
                  Roll_Angle,
                  Pitch_Angle,
                  Yaw_Rate,
                  Sideslip_Angle,
                  Lateral_Acc,
                  V_North,
                  V_East,
                  Q_Bar,
                  Nose_Wow,
                  Force_Body_Z,
                  Rudder_Deflection,
                  Yaw_Pilot_Force,
                  Dt,
                  CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);
            end if;

         end if;

         --| #################################################################################
         --| TORQUE DRIVER UPDATE PROCESS
         --| #################################################################################

         if CNT.Ios_Pilot_Torque_Driver.Is_On(CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver) then


            --| -------------------------------------------------------------------
            --| Update the necessary gain changes
            --| -------------------------------------------------------------------
            --| Determine the contribution of nominal torque component
            if (Flap_Pos > 45.0)   then
               Nominal_Torque_Comp := It.Doubly_Indexed.Interpolate(CNT.IOS_Pilot_Torque_Driver.Selected_IAS
                 (CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver),Climb_Rate,NOMTORQ22_T'access);
            elsif (Flap_Pos > 15.0)  then
               Nominal_Torque_Comp := It.Doubly_Indexed.Interpolate(CNT.IOS_Pilot_Torque_Driver.Selected_IAS
                 (CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver),Climb_Rate,NOMTORQ21_T'access);
            else
               Nominal_Torque_Comp := It.Doubly_Indexed.Interpolate(CNT.IOS_Pilot_Torque_Driver.Selected_IAS
                 (CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver),Climb_Rate,NOMTORQ20_T'access);
            end if;


            KP_IAS_Torque :=It.Singly_Indexed.Interpolate(TAS/1.6787,KP_IAST_T'access);
            KD_IAS_Torque :=It.Singly_Indexed.Interpolate(TAS/1.6787,KD_IAST_T'access);

            if IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 9) then

               --| -------------------------------------------------------------------------------------
               --| TUNING TORQUE DRIVER - Torque Hold Loop
               --| -------------------------------------------------------------------------------------
               Tuning_Mode                      := 1;
               --| Update from IOS Pilot Debug Page
               KP_Torque              := IOS_Pilot.KP_Torque;
               KI_Torque              := IOS_Pilot.KI_Torque;
               KD_Torque              := IOS_Pilot.KD_Torque;
               Max_Torque_Err         := IOS_Pilot.Max_Torque_Rate ;

               IOS_Pilot.KP_IAS_Torque          := KP_IAS_Torque;
               IOS_Pilot.KI_IAS_Torque          := KI_IAS_Torque;
               IOS_Pilot.KD_IAS_Torque          := KD_IAS_Torque;
               IOS_Pilot.Max_Delta_IAS_Torque   := Max_IAS_Err_Torque;




               --| Set the plot variables
               IOS_Pilot.Misc_Var1 := Torque;
               IOS_Pilot.Misc_Var2 := CNT.IOS_Pilot_Torque_Driver.Selected_Torque
                 (CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver);
               IOS_Pilot.Misc_Var3 := IAS;
               IOS_Pilot.Misc_Var4 := CNT.IOS_Pilot_Torque_Driver.Selected_IAS
                 (CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver);
               IOS_Pilot.Misc_Var5 := PLA;



            elsif IOS_Pilot.Ios_Pilot_Debug and
              (IOS_Pilot.Ios_Pilot_Mode = 10) then
               --| -------------------------------------------------------------------------------------
               --| TUNING TORQUE DRIVER - IAS Torque Hold Loop
               --| -------------------------------------------------------------------------------------

               Tuning_Mode                      := 2;
               --| Update from IOS Pilot Debug Page
               IOS_Pilot.KP_Torque              := KP_Torque;
               IOS_Pilot.KI_Torque              := KI_Torque;
               IOS_Pilot.KD_Torque              := KD_Torque;
               IOS_Pilot.Max_Torque_Rate        := Max_Torque_Err;

               KP_IAS_Torque         := IOS_Pilot.KP_IAS_Torque ;
               KI_IAS_Torque         := IOS_Pilot.KI_IAS_Torque ;
               KD_IAS_Torque         := IOS_Pilot.KD_IAS_Torque ;
               Max_IAS_Err_Torque    := IOS_Pilot.Max_Delta_IAS_Torque ;

               --| Set the plot variables
               IOS_Pilot.Misc_Var1 := Torque;
               IOS_Pilot.Misc_Var2 := CNT.IOS_Pilot_Torque_Driver.Selected_Torque
                 (CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver);
               IOS_Pilot.Misc_Var3 := IAS;
               IOS_Pilot.Misc_Var4 := CNT.IOS_Pilot_Torque_Driver.Selected_IAS
                 (CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver);
               IOS_Pilot.Misc_Var5 := PLA;

            else

               --| -------------------------------------------------------------------------------------
               --| NORMAL MODE
               --| -------------------------------------------------------------------------------------
               Tuning_Mode                      := 0;
               --| Update from IOS Pilot Debug Page
               IOS_Pilot.KP_Torque              := KP_Torque;
               IOS_Pilot.KI_Torque              := KI_Torque;
               IOS_Pilot.KD_Torque              := KD_Torque;
               IOS_Pilot.Max_Torque_Rate        := Max_Torque_Err;

               IOS_Pilot.KP_IAS_Torque          := KP_IAS_Torque;
               IOS_Pilot.KI_IAS_Torque          := KI_IAS_Torque;
               IOS_Pilot.KD_IAS_Torque          := KD_IAS_Torque;
               IOS_Pilot.Max_Delta_IAS_Torque   := Max_IAS_Err_Torque;



            end if;

            CNT.IOS_Pilot_Torque_Driver.Set_Gain
              (Tuning_Mode,
               KP_Torque,
               KI_Torque,
               KD_Torque,
               Max_Torque_Err,

               KP_IAS_Torque,
               KI_IAS_Torque,
               KD_IAS_Torque,
               Max_IAS_Err_Torque,

               Pitch_Gain,
               Nominal_Torque_Comp,
               CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver);


            if  (not (JAT.At_Phase = 9)) and
              (not (JSA.Get_Flight_Freeze and
                    (CNT.Ios_Pilot_Torque_Driver.Mode
                     (CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver)=2))) and
              (CNT.This_Subsystem.TO_Phase > -1) then

               CNT.Ios_Pilot_Torque_Driver.Update
                 (Torque,
                  PLA,
                  IAS,
                  Pitch_Angle,
                  Flap_Pos,
                  Gear_Pos,
                  Nx,
                  Dt,
                  CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);


               Jpats_Powerplant.Test.Set_Power_Control_Lever_Angle
                 (Position => CNT.Ios_Pilot_Torque_Driver.Commanded_PLA
                  (CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver));
            end if;

         end if;


         --| Update IOS Page data ---------------------------------------------------------
         if not IOS_Pilot.Altitude_Engaged then
            IOS_Pilot.Altitude := Altitude;
         end if;
         case Ios_pilot.Performance_mode is
            when 1 =>
               --| IAS and Climb Rate hold -------------------------------------------------
               Ios_Pilot.Torque := torque;

               --| if Altitude hold engaged
               if CNT.IOS_Pilot_Pitch_Driver.Altitude_Hold_Is_on
                 (CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver) then

                  Ios_Pilot.Climb_Rate_Enabled := False;
                  Ios_pilot.Climb_Rate := Climb_Rate;
               else
                  Ios_Pilot.Climb_Rate_Enabled := True;
               end if;

            when 2 =>
               --| Climb Rate and Torque hold ----------------------------------------------
               Ios_Pilot.IAS := IAS;
               --| if Altitude hold engaged
               if CNT.IOS_Pilot_Pitch_Driver.Altitude_Hold_Is_on
                 (CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver) then

                  Ios_Pilot.Climb_Rate_Enabled := False;
                  Ios_pilot.Climb_Rate := Climb_Rate;
               else
                  Ios_Pilot.Climb_Rate_Enabled := True;
               end if;


            when 3 =>
               --| Torque and IAS hold ----------------------------------------------------
               Ios_pilot.Climb_Rate := Climb_Rate;

               --| if Altitude hold engaged
               if CNT.IOS_Pilot_Pitch_Driver.Altitude_Hold_Is_on
                 (CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver) then

                  Ios_Pilot.IAS_Enabled := False;
                  Ios_pilot.IAS := IAS;
               else
                  Ios_Pilot.IAS_Enabled := True;
               end if;
            when others =>
               null;
         end case;



         --| Turn Off IOS Pilot
         if (IOS_Pilot.Error_Number > 0) and
           (IOS_Pilot.Error_Number < 21) then
            CNT.This_Subsystem.Mode := 0 ;
            IOS_Pilot.Apply_Enabled := false;
            Ios_Pilot.Enable        := False;
            Ios_Pilot.Takeoff       := False;
            CNT.This_Subsystem.TO_Phase := -1;
            JSA.Set_System_Freeze;
            CNT.This_Subsystem.Error_Counter := 0;
            Reset_On_Off_Flags;
         end if;



      else
         --| ===================================================================================
         --| IOS PILOT is off
         --| ===================================================================================
         if not (CNT.This_Subsystem.Previous_Mode = CNT.This_Subsystem.Mode) then
            JSA.Set_System_Freeze;
            CNT.IOS_Pilot_Pitch_Driver.Set_Mode(0,CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
            CNT.IOS_Pilot_Roll_Driver.Set_Mode(0,CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
            CNT.IOS_Pilot_Yaw_Driver.Set_Mode(0,CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);
            CNT.Ios_Pilot_Torque_Driver.Set_Mode(0,CNT.This_Subsystem.The_Ios_Pilot_Torque_Driver);
            CNT.This_Subsystem.TO_Phase          := -1;
            CNT.This_Subsystem.Reposition_Active := False;
            Ios_Pilot.Takeoff                    := False;
            Ios_Pilot.TO_Is_started              := False;
            Ios_Pilot.TO_Is_Done                 := False;
            Ios_Pilot.Enable                     := False;
            --| Disable altitude hold
            IOS_Pilot.Altitude_Engaged           := False;
            Reset_On_Off_Flags;
         end if;

         --| Update IOS Page data ---------------------------------------------------------
         Ios_Pilot.Heading := Heading - Magvar;
         if Ios_Pilot.Heading > 360.0 then
            Ios_Pilot.Heading := Ios_Pilot.Heading - 360.0;
         elsif Ios_Pilot.Heading < 0.0 then
            Ios_Pilot.Heading := Ios_Pilot.Heading + 360.0;
         end if;


         Ios_Pilot.Torque := torque;
         Ios_Pilot.IAS := IAS;
         Ios_pilot.Climb_Rate := Climb_Rate;


         --| Reset Error
         --| Turn Off IOS Pilot
         if (IOS_Pilot.Error_Number > 0) and
           (IOS_Pilot.Error_Number < 21) then
            IOS_Pilot.Apply_Enabled              := false;
            Ios_Pilot.Enable                     := False;
            Ios_Pilot.Takeoff                    := False;
            Ios_Pilot.TO_Is_started              := False;
            Ios_Pilot.TO_Is_Done                 := False;
            CNT.This_Subsystem.TO_Phase          := -1;
            CNT.This_Subsystem.Reposition_Active := False;
            Reset_On_Off_Flags;
         end if;

         if (IOS_Pilot.Error_Number > 0) then
            if CNT.This_Subsystem.Error_Counter < 300 then
               CNT.This_Subsystem.Error_Counter := CNT.This_Subsystem.Error_Counter + 1 ;
               if CNT.This_Subsystem.Error_Counter = 300 then
                  CNT.This_Subsystem.Error_Counter := 0;
                  IOS_Pilot.Error_Number := 0;
               end if;
            end if;
         else
            CNT.This_Subsystem.Error_Counter := 0;

         end if;

      end if;

      if Ios_Pilot.Apply then
         Ios_Pilot.Apply := False;
      end if;

      --| Reset request gear up or down
      if IOS_Pilot.Request_Gear_Down or IOS_Pilot.Request_Gear_Up  then
         if Jpats_Landing_Gear.Ios_Pilot_Gear = 1 then
            IOS_Pilot.Request_Gear_Down := False;
         end if;
         if Jpats_Landing_Gear.Ios_Pilot_Gear = 0 then
            IOS_Pilot.Request_Gear_Up  := False;
         end if;
      end if;

      --| Reset internal variable when trimming ---------------------------------------
      if (JAT.At_Phase = 9) or  Save_Restore.Replay_Trimming then
         Reset_Internal_Variables;
         Ios_Pilot.Apply := True;

      end if;


      CNT.This_Subsystem.Previous_Mode := CNT.This_Subsystem.Mode;

      --| ### debug #########################################################
      if Integer(JSA.Float1) = 100 then

         if       (Integer(JSA.Float8) = 1) then
            CNT.Debug1 := Float(Boolean'Pos(Ios_Pilot.Enable));
         elsif    (Integer(JSA.Float8) = 2) then
            CNT.Debug1 := Float(CNT.This_Subsystem.Mode);
         elsif    (Integer(JSA.Float8) = 3) then
            CNT.Debug1 := Float(CNT.This_Subsystem.TO_Phase);
         elsif    (Integer(JSA.Float8) = 4) then
            CNT.Debug1 := Float(Boolean'Pos(Ios_Pilot.takeoff));
         elsif    (Integer(JSA.Float8) = 5) then
            CNT.Debug1 := Float(Boolean'Pos(Ios_Pilot.To_Is_started));
         elsif    (Integer(JSA.Float8) = 6) then
            CNT.Debug1 := Float(Boolean'Pos(CNT.This_Subsystem.Reposition_Active));
         elsif    (Integer(JSA.Float8) = 7) then
            CNT.Debug1 := Altitude_AGL;
         elsif    (Integer(JSA.Float8) = 8) then
            CNT.Debug1 := TAS;
         elsif    (Integer(JSA.Float8) = 9) then
            CNT.Debug1 := Float(CNT.This_Subsystem.Previous_Mode);
         elsif    (Integer(JSA.Float8) = 13) then
            CNT.Debug1 := Float( IOS_Pilot.Ios_Pilot_Mode);
         elsif    (Integer(JSA.Float8) = 14) then
            CNT.Debug1 := Float(Boolean'Pos(IOS_Pilot.Ios_Pilot_Debug));
         elsif    (Integer(JSA.Float8) = 15) then
            CNT.Debug1 := IOS_Pilot.KP_Pitch;
         elsif    (Integer(JSA.Float8) = 16) then
            CNT.Debug1 := Float(IOS_Pilot.Error_Number);
         elsif    (Integer(JSA.Float8) = 16) then
            if IOS_Pilot.Altitude_Engaged then
               CNT.Debug1 := 1.0;
            else
               CNT.Debug1 := 0.0;
            end if;
         end if;

      elsif (Integer(JSA.Float1) > 100) or (Integer(JSA.Float1) < 110) then

         if JSA.Bool2 then
            CNT.Debug3 := CNT.IOS_Pilot_Roll_Driver.Debug1
              (CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
         end if;



      elsif (Integer(JSA.Float1) > 110) or (Integer(JSA.Float1) < 120) then
         if JSA.Bool2 then
            CNT.Debug3 := CNT.IOS_Pilot_Pitch_Driver.Debug1
              (CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
         end if;


      elsif (Integer(JSA.Float1) > 120) or (Integer(JSA.Float1) < 130) then
         if JSA.Bool2 then
            CNT.Debug3 := CNT.IOS_Pilot_Yaw_Driver.Debug1
              (An_Instance  => CNT.This_Subsystem.The_IOS_Pilot_Yaw_Driver);
         end if;

      elsif (Integer(JSA.Float1) > 130) or (Integer(JSA.Float1) < 140) then
         if JSA.Bool2 then
            CNT.Debug3 := CNT.IOS_Pilot_Torque_Driver.Debug1
              (An_Instance  => CNT.This_Subsystem.The_IOS_Pilot_Torque_Driver);
         end if;


      end if;


      --| ### debug End #####################################################

      if IOS_Pilot.Ios_Pilot_Mode = 0 then
         if Integer(IOS_Pilot.Misc_Var1) = 0 then
            for I in 1 .. 2 loop
               if I = 1 then
                  Temp1 := IOS_Pilot.Misc_Var2 ;
               else
                  Temp1 := IOS_Pilot.Misc_Var3 ;
               end if;
               case Integer(Temp1) is
                  when 1 =>
                     Temp2 := float(CNT.This_Subsystem.Mode);
                  when 2 =>
                     Temp2 := float(CNT.This_Subsystem.TO_Phase);
                  when 3 =>
                     Temp2 := float(CNT.This_Subsystem.TO_Phase);
                  when 4 =>
                     Temp2 := Float(Boolean'Pos(IOS_Pilot.TO_Is_Started));
                  when 5 =>
                     Temp2 := Float(Boolean'Pos(IOS_Pilot.takeoff));
                  when 6 =>
                     Temp2 := Heading;
                  when 7 =>
                     Temp2 := JPATS_Reposition.Reference_Runway.Hdg;
                  when 8 =>
                     Temp2 := Torque;
                  when 9 =>
                     Temp2 := Sqrt(V_North*V_North + V_East*V_East);
                  when 10 =>
                     Temp2 := Altitude_Agl;
                  when 11 =>
                     Temp2 := Float(Boolean'Pos(CNT.This_Subsystem.Reposition_Active));
                  when 12 =>
                     Temp2 := Float(Boolean'Pos(TO_Ready));
                  when 13 =>
                     Temp2 := JSA.GW;
                  when 14 =>
                     Temp2 := Ft_East;
                  when 15 =>
                     Temp2 := Ft_north;

                  when Others => null;
               end case;

               if I = 1 then
                  IOS_Pilot.Misc_Var4 := Temp2;
               else
                  IOS_Pilot.Misc_Var5 := Temp2;
               end if;

            end loop;

         elsif Integer(IOS_Pilot.Misc_Var1) = 1 then
            CNT.IOS_Pilot_Pitch_Driver.Set_Debug1
              (IOS_Pilot.Misc_Var2,CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
            CNT.IOS_Pilot_Pitch_Driver.Set_Debug2
              (IOS_Pilot.Misc_Var3,CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
            IOS_Pilot.Misc_Var4 := CNT.IOS_Pilot_Pitch_Driver.Debug3
              (CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);
            IOS_Pilot.Misc_Var5 := CNT.IOS_Pilot_Pitch_Driver.Debug4
              (CNT.This_Subsystem.The_IOS_Pilot_Pitch_Driver);

         elsif Integer(IOS_Pilot.Misc_Var1) = 2 then
            CNT.IOS_Pilot_Roll_Driver.Set_Debug1
              (IOS_Pilot.Misc_Var2,CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
            CNT.IOS_Pilot_Roll_Driver.Set_Debug2
              (IOS_Pilot.Misc_Var3,CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
            IOS_Pilot.Misc_Var4 := CNT.IOS_Pilot_Roll_Driver.Debug3
              (CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
            IOS_Pilot.Misc_Var5 := CNT.IOS_Pilot_Roll_Driver.Debug4
              (CNT.This_Subsystem.The_IOS_Pilot_Roll_Driver);
         else
               null;
         end if;
      end if;



      if (IOS_Pilot.Error_Number > 0) and (CNT.This_Subsystem.Error_Counter < 250) then
         IOS_Pilot.Misc_Var5 := Float(IOS_Pilot.Error_Number);
      elsif (IOS_Pilot.Error_Number > 0) and (CNT.This_Subsystem.Error_Counter > 250) then
         IOS_Pilot.Misc_Var5 := 0.0;
      end if;

   exception
      when others =>
         Log.Report(Body_File_Name & ".Update()");
         raise;
   end Update;






end Jpats_IOS_Pilot.Controller;



