-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|

with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Log;

package body First_Order_Filter  is
   -- General definitions -----------------------------------------------------
   Body_File_Name : constant String := "IOS_pilot/first_order_filter.adb";

   -- First order lowpass filter using Euler integration method
   procedure Initialize
     (Uk0             :in      Float;
      A_Lowpass       :in out  Lowpass) is
   begin
      A_Lowpass.Last_Output := Uk0;
      A_Lowpass.Output := Uk0;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize - lowpass",
                    Severity => Log.ERROR);
         raise;
   end Initialize;

   procedure Update
     (Uk              :in     Float;
      Time_Constant   :in     Float;
      Dt              :in     Float;
      A_Lowpass       :in out Lowpass) is
      Tho :Float := Float'Max(Time_Constant,1.0/60.0);
   begin
      A_Lowpass.Last_Output := A_Lowpass.Output;
      A_Lowpass.Output := A_Lowpass.Last_Output +
        (Dt/Tho)*(Uk-A_Lowpass.Last_Output);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": update - lowpass",
                    Severity => Log.ERROR);
         raise;
   end Update;

   procedure Update
     (Uk              :in     Float;
      Rate_Limit      :in     Float;
      Time_Constant   :in     Float;
      Dt              :in     Float;
      A_Lowpass       :in out Lowpass) is
      Tho :Float := Float'Max(Time_Constant,1.0/60.0);
      Limited_Input  : Float;
   begin

      A_Lowpass.Last_Output := A_Lowpass.Output;

      if abs(Uk - A_Lowpass.Last_Output ) > Rate_Limit then
         if (Uk - A_Lowpass.Last_Output ) > 0.0 then
            Limited_Input := A_Lowpass.Last_Output + Rate_Limit;
         else
            Limited_Input := A_Lowpass.Last_Output - Rate_Limit;
         end if;
      else
         Limited_Input := Uk;
      end if;


      A_Lowpass.Output := A_Lowpass.Last_Output +
        (Dt/Tho)*(Limited_Input - A_Lowpass.Last_Output);
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": update - lowpass",
                    Severity => Log.ERROR);
         raise;
   end Update;

   function Output (A_Lowpass       :in Lowpass)
                    return Float is
   begin
      return A_Lowpass.Output;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": output - lowpass",
                    Severity => Log.ERROR);
         raise;
   end Output;


   -- First order highpass filter using Bilinear transformation method
   procedure Initialize
     (Uk0             :in      Float;
      A_Washout       :in out  Washout)is

   begin
      A_Washout.Last_1_Input  := Uk0;
      A_Washout.Last_2_Input  := Uk0;
      A_Washout.Last_Output   := 0.0;
      A_Washout.Output        := 0.0;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": Initialize - washout",
                    Severity => Log.ERROR);
         raise;
   end Initialize;

   procedure Update
     (Uk              :in     Float;
      Time_Constant   :in     Float;
      Dt              :in     Float;
      A_Washout       :in out  Washout) is
      Tho :Float := Float'Max(Time_Constant,1.0/60.0);
   begin
      A_Washout.Last_Output := A_Washout.Output;
      A_Washout.Output := ((2.0 - Dt/Tho)*A_Washout.Last_Output
                           + 2.0*A_Washout.Last_1_Input
                           - 2.0*A_Washout.Last_2_Input)/(2.0 + Dt/Tho);
      A_Washout.Last_2_Input := A_Washout.Last_1_Input;
      A_Washout.Last_1_Input := Uk;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": update - washout",
                    Severity => Log.ERROR);
         raise;
   end Update;

   function Output ( A_Washout       :in Washout)
                    return Float is
   begin
      return A_Washout.Output;
   exception
      when others =>
         Log.Report(Event    => Body_File_Name & ": output - washout",
                    Severity => Log.ERROR);
         raise;
   end Output;


end First_Order_Filter;





