// SortieFilesSystemInterface.cpp: implementation of the CSortieFilesSystemInterface class.
//
//////////////////////////////////////////////////////////////////////
#include "..\core\stdafx.h"
#include "SortieFilesSystemInterface.h"
#include "SortieCommsAction.h"
#include "..\comms\commsShared.h"

#include "AircraftConfigRecordSet.h"
#include "RepositionRecordSet.h"
#include "WeatherArea1RecordSet.h"
#include "WeatherArea2RecordSet.h"
#include "WeatherArea3RecordSet.h"
#include "WeatherArea4RecordSet.h"
#include "WeatherGlobalsRecordSet.h"
#include "ThreatsRecordSet.h"
#include "AreaMapRecordSet.h"
#include "SymbolDisplayRecordSet.h"
#include "RecordingsRecordSet.h"
#include "WorkingDlg.h"

#include "..\malfunctions\MalfFieldAction.h"
#include "..\malfunctions\MalfCommsAction.h"
#include "..\sortieedit\MalfPreselectTypeRecordSet.h"
#include "MalfunctionsRecordSet.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSortieFilesSystemInterface::CSortieFilesSystemInterface()
{

}

CSortieFilesSystemInterface::~CSortieFilesSystemInterface()
{
}

void CSortieFilesSystemInterface::Clean()
{
    // Delete any comms actions that were created.
    _FSI_STL::map<_FSI_STL::string, CSortieCommsAction*>::iterator scaIt     = NULL;
    _FSI_STL::map<_FSI_STL::string, CSortieCommsAction*>::iterator scaendIt  = NULL;
    scaIt       = m_mapCommsActions.begin();
    scaendIt    = m_mapCommsActions.end();
    CWidget* pWidget = NULL;
    while (scaIt != scaendIt)
    {
        pWidget = dynamic_cast<CWidget*>((*scaIt).second);
        if (pWidget != NULL)
        {
            if (CWidget::IsValidAddress(pWidget) != INVALID)
            {
                pWidget->Deleting(true);
            }
        }

        scaIt++;
    }

    m_mapCommsActions.clear();
}

void CSortieFilesSystemInterface::LoadData(const _FSI_STL::string& rstlStrDataFile)
{
   Clean();

   m_pVariableMapSet    =  new   CVariableMapRecordSet(NULL);
   m_pVariableMapSet->Open();


   CVariant var;
   CCommsShared::GetLocalValue(var, _FSI_STL::string("selected_sortie_base"));
   CString  base(((_FSI_STL::string)var).c_str());

   CCommsShared::GetLocalValue(var, _FSI_STL::string("selected_sortie_lesson"));
   CString  lesson(((_FSI_STL::string)var).c_str());

   m_condition_for_find.Format("[Base ID] = '%s' AND [Lesson ID] = '%s'", base, lesson);

   SetRecordedData();

   ResetMissionTime();

   SetAircraftConfig();
   SetReposition();

   SetWeatherAreas();   // Call after repositioning
   SetWeatherGlobals(); // Call after SetWeatherAreas

   SetThreats();

   SetAreaMap();        // Call after repositioning
   SetSymDisp();

   SetMalfunctions();

   m_pVariableMapSet->Close();
   delete   m_pVariableMapSet;

   // bump the count of number of sorties loaded (so GeoCharts can know if the sortie file changed)

   CCommsShared::GetLocalValue(var, _FSI_STL::string("sortie_load_count"));
   long  old_type =  var.Type();
   var.Value((unsigned long)var + 1UL);
   var.ChangeType(old_type);
   CCommsShared::SetLocalValue(var, _FSI_STL::string("sortie_load_count"), 0);
}

//////////////////////////////////////////////////////////////////////
// Accessors
//////////////////////////////////////////////////////////////////////

_FSI_STL::string CSortieFilesSystemInterface::SortieFile()
{
    return m_stlStrSortieFile;
}

//////////////////////////////////////////////////////////////////////
// Mutators
//////////////////////////////////////////////////////////////////////

void CSortieFilesSystemInterface::SortieFile(const _FSI_STL::string& rstlStrSortieFile)
{
    if (m_stlStrSortieFile != rstlStrSortieFile)
    {
        m_stlStrSortieFile = rstlStrSortieFile;
        LoadData(rstlStrSortieFile);
    }
}


void CSortieFilesSystemInterface::ChangeData(BOOL isLocal, _FSI_STL::string stlStrVariable, _FSI_STL::string stlStrValue)
{
   CVariant var;

   if (isLocal)
   {  
      CCommsShared::GetLocalValue(var, stlStrVariable);
      long  old_type =  var.Type();
      var.Value(stlStrValue);
      var.ChangeType(old_type);
      CCommsShared::SetLocalValue(var, stlStrVariable, 0);
   }
   else
   {
      CCommsShared::GetHostValue(var, stlStrVariable);
      long  old_type =  var.Type();
      var.Value(stlStrValue);
      var.ChangeType(old_type);
      if (old_type   == 0)
         CCommsShared::SetHostValue(var, stlStrVariable, -1);
      else
         CCommsShared::SetHostValue(var, stlStrVariable, 0);
   }
}


void  CSortieFilesSystemInterface::SetReposition(void)
{
   CRepositionRecordSet *pRepositionRecordSet   =  new   CRepositionRecordSet(NULL);
   pRepositionRecordSet->Open();

   BOOL  found =  pRepositionRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);
   if (found)
   {
      if ( pRepositionRecordSet->m_Reposition_Command == "0" )          // No Reposition
      {
         // Do nothing.
      }
      else if ( pRepositionRecordSet->m_Reposition_Command == "20" )    // Lat/Long Reposition
      {
         LoadItem( pRepositionRecordSet->m_Reposition_Latitude,         "[Sortie Field] = 'Reposition Latitude'"              );
         LoadItem( pRepositionRecordSet->m_Reposition_Longitude,        "[Sortie Field] = 'Reposition Longitude'"             );

         LoadItem( pRepositionRecordSet->m_Reposition_Altitude_MSL,     "[Sortie Field] = 'Reposition Altitude MSL'"          );
         LoadItem( pRepositionRecordSet->m_Reposition_Heading,          "[Sortie Field] = 'Reposition Heading'"               );
         LoadItem( pRepositionRecordSet->m_Reposition_Airspeed,         "[Sortie Field] = 'Reposition Airspeed'"              );
      }
      else if ( pRepositionRecordSet->m_Reposition_Command == "21" )    // NAVAID Reposition
      {
         // Only bother with these for LatLon or NAVAID reposition types
         LoadItem( pRepositionRecordSet->m_Reposition_Navaid_ID,        "[Sortie Field] = 'Reposition Navaid ID'"             );
         LoadItem( pRepositionRecordSet->m_Reposition_Navaid_Radial,    "[Sortie Field] = 'Reposition Navaid Radial'"         );
         LoadItem( pRepositionRecordSet->m_Reposition_Navaid_Distance,  "[Sortie Field] = 'Reposition Navaid Distance'"       );

         LoadItem( pRepositionRecordSet->m_Reposition_Altitude_MSL,     "[Sortie Field] = 'Reposition Altitude MSL'"          );
         LoadItem( pRepositionRecordSet->m_Reposition_Heading,          "[Sortie Field] = 'Reposition Heading'"               );
         LoadItem( pRepositionRecordSet->m_Reposition_Airspeed,         "[Sortie Field] = 'Reposition Airspeed'"              );
      }
      else if ( pRepositionRecordSet->m_Reposition_Command == "22" )    // Formation Reposition
      {
         LoadItem( pRepositionRecordSet->m_Formation_Command,           "[Sortie Field] = 'Formation Command'"                );
      }
      else
      {
         LoadItem( pRepositionRecordSet->m_Active_Airport,              "[Sortie Field] = 'Active Airport'"                   );
         LoadItem( pRepositionRecordSet->m_Active_Runway,               "[Sortie Field] = 'Active Runway'"                    );
         ChangeData( FALSE, "Reposition_Airport Change", "1");
      }

      LoadItem( pRepositionRecordSet->m_Reposition_Command,             "[Sortie Field] = 'Reposition Command'"               );
   }

   pRepositionRecordSet->Close();
   delete   pRepositionRecordSet;
}


void  CSortieFilesSystemInterface::SetAreaMap(void)
{
   CAreaMapRecordSet *pAreaMapRecordSet   =  new   CAreaMapRecordSet(NULL);
   pAreaMapRecordSet->Open();

   BOOL  found =  pAreaMapRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);
   if (found)
   {
      LoadItem( pAreaMapRecordSet->m_Approach_Plot_Scale,               "[Sortie Field] = 'Approach Plot Scale'"              );
      LoadItem( pAreaMapRecordSet->m_Area_Map_Auto_Center,              "[Sortie Field] = 'Area Map Auto Center'"             ); 
      LoadItem( pAreaMapRecordSet->m_Area_Map_Auto_Scale,               "[Sortie Field] = 'Area Map Auto Scale'"              ); 
      LoadItem( pAreaMapRecordSet->m_Area_Map_Center_Airport_ID,        "[Sortie Field] = 'Area Map Center Airport ID'"       ); 
      LoadItem( pAreaMapRecordSet->m_Area_Map_Center_On_Aircraft,       "[Sortie Field] = 'Area Map Center On Aircraft'"      ); 
      LoadItem( pAreaMapRecordSet->m_Area_Map_Center_On_Airport,        "[Sortie Field] = 'Area Map Center On Airport'"       ); 
      LoadItem( pAreaMapRecordSet->m_Area_Map_Scale,                    "[Sortie Field] = 'Area Map Scale'"                   );
      LoadItem( pAreaMapRecordSet->m_Area_Map_Spider_Web_Airport_ID,    "[Sortie Field] = 'Area Map Spider Web Airport ID'"   ); 
      LoadItem( pAreaMapRecordSet->m_Area_Map_Spider_Web_Navaid_ID,     "[Sortie Field] = 'Area Map Spider Web Navaid ID'"    ); 
      LoadItem( pAreaMapRecordSet->m_Area_Map_Spider_Web_Type,          "[Sortie Field] = 'Area Map Spider Web Type'"         ); 
      LoadItem( pAreaMapRecordSet->m_GeoChart_Init_Data,                "[Sortie Field] = 'GeoChart Init Data'"               );
   }

   pAreaMapRecordSet->Close();
   delete   pAreaMapRecordSet;

   ChangeData(TRUE, "sortie_was_loaded", "1");
}


void  CSortieFilesSystemInterface::SetSymDisp(void)
{
   CSymbolDisplayRecordSet *pSymbolDisplayRecordSet   =  new   CSymbolDisplayRecordSet(NULL);
   pSymbolDisplayRecordSet->Open();

   BOOL  found =  pSymbolDisplayRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);
   if (found)
   {
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_ILS,                "[Sortie Field] = 'Symbol Display ILS'"                  );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_ILS_Text,           "[Sortie Field] = 'Symbol Display ILS Text'"             );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_VOR,                "[Sortie Field] = 'Symbol Display VOR'"                  );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_VOR_Text,           "[Sortie Field] = 'Symbol Display VOR Text'"             );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_NDB,                "[Sortie Field] = 'Symbol Display NDB'"                  );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_NDB_Text,           "[Sortie Field] = 'Symbol Display NDB Text'"             );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Airports,           "[Sortie Field] = 'Symbol Display Airports'"             );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Airport_Text,       "[Sortie Field] = 'Symbol Display Airport Text'"         );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Markers,            "[Sortie Field] = 'Symbol Display Markers'"              );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Waypoints,          "[Sortie Field] = 'Symbol Display Waypoints'"            );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Jet_Routes,         "[Sortie Field] = 'Symbol Display Jet Routes'"           );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Victor_Routes,      "[Sortie Field] = 'Symbol Display Victor Routes'"        );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Lead_Aircraft,      "[Sortie Field] = 'Symbol Display Lead Aircraft'"        );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Wing_Aircraft,      "[Sortie Field] = 'Symbol Display Wing Aircraft'"        );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_LatLon_Grid,        "[Sortie Field] = 'Symbol Display LatLon Grid'"          );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Mouse_LatLon,       "[Sortie Field] = 'Symbol Display Mouse LatLon'"         );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Weather_Areas,      "[Sortie Field] = 'Symbol Display Weather Areas'"        );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Decimal_Degrees,    "[Sortie Field] = 'Symbol Display Decimal Degrees'"      );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Obstacles,          "[Sortie Field] = 'Symbol Display Obstacles'"            );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Obstacles_Text,     "[Sortie Field] = 'Symbol Display Obstacles Text'"       );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Practice_Areas,     "[Sortie Field] = 'Symbol Display Practice Areas'"       );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Practice_Areas_T,   "[Sortie Field] = 'Symbol Display Practice Areas Te'"    );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Lines,              "[Sortie Field] = 'Symbol Display Lines'"                );
      LoadItem( pSymbolDisplayRecordSet->m_Symbol_Display_Lines_Text,         "[Sortie Field] = 'Symbol Display Lines Text'"           );
   }

   pSymbolDisplayRecordSet->Close();
   delete   pSymbolDisplayRecordSet;
}

void  CSortieFilesSystemInterface::SetThreats(void)
{
   CThreatsRecordSet *pThreatsRecordSet   =  new   CThreatsRecordSet(NULL);
   pThreatsRecordSet->Open();

   BOOL  found =  pThreatsRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);
   if (found)
   {
      LoadItem( pThreatsRecordSet->m_Threat_1_Aircraft_Type,            "[Sortie Field] = 'Threat 1 Aircraft Type'"           );
      LoadItem( pThreatsRecordSet->m_Threat_1_Bearing,                  "[Sortie Field] = 'Threat 1 Bearing'"                 );
      LoadItem( pThreatsRecordSet->m_Threat_1_Relative_Altitude,        "[Sortie Field] = 'Threat 1 Relative Altitude'"       );
      LoadItem( pThreatsRecordSet->m_Threat_1_Distance,                 "[Sortie Field] = 'Threat 1 Distance'"                );
      LoadItem( pThreatsRecordSet->m_Threat_1_Time,                     "[Sortie Field] = 'Threat 1 Time'"                    );
      LoadItem( pThreatsRecordSet->m_Threat_1_Vertical_Start_Position,  "[Sortie Field] = 'Threat 1 Vertical Start Position'" );
      LoadItem( pThreatsRecordSet->m_Threat_1_Speed,                    "[Sortie Field] = 'Threat 1 Speed'"                   );
      LoadItem( pThreatsRecordSet->m_Threat_1_Vertical_Closure_Rate,    "[Sortie Field] = 'Threat 1 Vertical Closure Rate'"   );
      LoadItem( pThreatsRecordSet->m_Threat_1_Display_on_NACWS,         "[Sortie Field] = 'Threat 1 Display on NACWS'"        );
      LoadItem( pThreatsRecordSet->m_Threat_1_Enable,                   "[Sortie Field] = 'Threat 1 Enable'"                  );

      LoadItem( pThreatsRecordSet->m_Threat_2_Aircraft_Type,            "[Sortie Field] = 'Threat 2 Aircraft Type'"           );
      LoadItem( pThreatsRecordSet->m_Threat_2_Bearing,                  "[Sortie Field] = 'Threat 2 Bearing'"                 );
      LoadItem( pThreatsRecordSet->m_Threat_2_Relative_Altitude,        "[Sortie Field] = 'Threat 2 Relative Altitude'"       );
      LoadItem( pThreatsRecordSet->m_Threat_2_Distance,                 "[Sortie Field] = 'Threat 2 Distance'"                );
      LoadItem( pThreatsRecordSet->m_Threat_2_Time,                     "[Sortie Field] = 'Threat 2 Time'"                    );
      LoadItem( pThreatsRecordSet->m_Threat_2_Vertical_Start_Position,  "[Sortie Field] = 'Threat 2 Vertical Start Position'" );
      LoadItem( pThreatsRecordSet->m_Threat_2_Speed,                    "[Sortie Field] = 'Threat 2 Speed'"                   );
      LoadItem( pThreatsRecordSet->m_Threat_2_Vertical_Closure_Rate,    "[Sortie Field] = 'Threat 2 Vertical Closure Rate'"   );
      LoadItem( pThreatsRecordSet->m_Threat_2_Display_on_NACWS,         "[Sortie Field] = 'Threat 2 Display on NACWS'"        );
      LoadItem( pThreatsRecordSet->m_Threat_2_Enable,                   "[Sortie Field] = 'Threat 2 Enable'"                  );

      LoadItem( pThreatsRecordSet->m_Ground_Hazard_Vehicle_Type,        "[Sortie Field] = 'Ground Hazard Vehicle Type'"       );
      LoadItem( pThreatsRecordSet->m_Ground_Hazard_Path,                "[Sortie Field] = 'Ground Hazard Path'"               );
      LoadItem( pThreatsRecordSet->m_Ground_Hazard_Enable,              "[Sortie Field] = 'Ground Hazard Enable'"             );
      LoadItem( pThreatsRecordSet->m_Ground_Hazard_Pause,               "[Sortie Field] = 'Ground Hazard Pause'"              );
   }

   pThreatsRecordSet->Close();
   delete   pThreatsRecordSet;
}


void  CSortieFilesSystemInterface::SetAircraftConfig(void)
{
   CAircraftConfigRecordSet   *pAircraftConfigRecordSet  =  new   CAircraftConfigRecordSet(NULL);
   pAircraftConfigRecordSet->Open();

   BOOL  found =  pAircraftConfigRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);
   if (found)
   {
      LoadItem( pAircraftConfigRecordSet->m_External_Power,                   "[Sortie Field] = 'External Power'"                   );
      LoadItem( pAircraftConfigRecordSet->m_Quickstart,                       "[Sortie Field] = 'Quickstart'"                       );
      LoadItem( pAircraftConfigRecordSet->m_Wheel_Chocks,                     "[Sortie Field] = 'Wheel Chocks'"                     );
      LoadItem( pAircraftConfigRecordSet->m_Number_of_Pilots,                 "[Sortie Field] = 'Number of Pilots'"                 );
      LoadItem( pAircraftConfigRecordSet->m_Overspeed_Mute,                   "[Sortie Field] = 'Overspeed Mute'"                   );
      LoadItem( pAircraftConfigRecordSet->m_Gear_Warning_Mute,                "[Sortie Field] = 'Gear Warning Mute'"                );
      LoadItem( pAircraftConfigRecordSet->m_Left_Fuel_Quantity,               "[Sortie Field] = 'Left Fuel Quantity'"               );
      LoadItem( pAircraftConfigRecordSet->m_Right_Fuel_Quantity,              "[Sortie Field] = 'Right Fuel Quantity'"              );
      LoadItem( pAircraftConfigRecordSet->m_Chatter_Rate,                     "[Sortie Field] = 'Chatter Rate'"                     );
      LoadItem( pAircraftConfigRecordSet->m_Master_Light_Level,               "[Sortie Field] = 'Master Light Level'"               );
      LoadItem( pAircraftConfigRecordSet->m_Master_Light_Level_Activate,      "[Sortie Field] = 'Master Light Level Activate'"      );
      LoadItem( pAircraftConfigRecordSet->m_Display_Propeller,                "[Sortie Field] = 'Display Propeller'"                );
      LoadItem( pAircraftConfigRecordSet->m_Rear_Seat_View,                   "[Sortie Field] = 'Rear Seat View'"                   );
   }

   pAircraftConfigRecordSet->Close();
   delete   pAircraftConfigRecordSet;
}


void  CSortieFilesSystemInterface::SetWeatherAreas(void)
{
   bool  found =  false;

   CWeatherArea1RecordSet  *pWeatherArea1RecordSet =  new   CWeatherArea1RecordSet(NULL);
   pWeatherArea1RecordSet->Open();

   if (pWeatherArea1RecordSet->Find(AFX_DAO_FIRST, m_condition_for_find))
   {
      LoadItem( pWeatherArea1RecordSet->m_Use_Weather_Area_1,                 "[Sortie Field] = 'Use Weather Area 1'"               );

      if (pWeatherArea1RecordSet->m_Use_Weather_Area_1 == "1")
      {
         found =  true;
         LoadItem( pWeatherArea1RecordSet->m_Active_Airport_1,                "[Sortie Field] = 'Active Airport 1'"                 );
         LoadItem( pWeatherArea1RecordSet->m_Active_Runway_1,                 "[Sortie Field] = 'Active Runway 1'"                  );
         LoadItem( pWeatherArea1RecordSet->m_Weather_Area_Radius_1,           "[Sortie Field] = 'Weather Area Radius 1'"            );
         LoadItem( pWeatherArea1RecordSet->m_Field_Pressure_Hg_1,             "[Sortie Field] = 'Field Pressure Hg 1'"              );
         LoadItem( pWeatherArea1RecordSet->m_Field_Temperature_F_1,           "[Sortie Field] = 'Field Temperature F 1'"            );
         LoadItem( pWeatherArea1RecordSet->m_RVR_Feet_1,                      "[Sortie Field] = 'RVR Feet 1'"                       );
         LoadItem( pWeatherArea1RecordSet->m_Visibility_sm_1,                 "[Sortie Field] = 'Visibility sm 1'"                  );
         LoadItem( pWeatherArea1RecordSet->m_Time_of_Day_1,                   "[Sortie Field] = 'Time of Day 1'"                    );
         LoadItem( pWeatherArea1RecordSet->m_Upper_Cloud_Top_1,               "[Sortie Field] = 'Upper Cloud Top 1'"                );
         LoadItem( pWeatherArea1RecordSet->m_Upper_Cloud_Bottom_1,            "[Sortie Field] = 'Upper Cloud Bottom 1'"             );
         LoadItem( pWeatherArea1RecordSet->m_Upper_Cloud_Type_1,              "[Sortie Field] = 'Upper Cloud Type 1'"               );
         LoadItem( pWeatherArea1RecordSet->m_Lower_Cloud_Top_1,               "[Sortie Field] = 'Lower Cloud Top 1'"                );
         LoadItem( pWeatherArea1RecordSet->m_Lower_Cloud_Bottom_1,            "[Sortie Field] = 'Lower Cloud Bottom 1'"             );
         LoadItem( pWeatherArea1RecordSet->m_Lower_Cloud_Type_1,              "[Sortie Field] = 'Lower Cloud Type 1'"               );
         LoadItem( pWeatherArea1RecordSet->m_Cirrus_1,                        "[Sortie Field] = 'Cirrus 1'"                         );
         LoadItem( pWeatherArea1RecordSet->m_Scud_1,                          "[Sortie Field] = 'Scud 1'"                           );
         LoadItem( pWeatherArea1RecordSet->m_Fog_Type_1,                      "[Sortie Field] = 'Fog Type 1'"                       );
         LoadItem( pWeatherArea1RecordSet->m_Fog_Top_1,                       "[Sortie Field] = 'Fog Top 1'"                        );
         LoadItem( pWeatherArea1RecordSet->m_Surface_Wind_Direction_1,        "[Sortie Field] = 'Surface Wind Direction 1'"         );
         LoadItem( pWeatherArea1RecordSet->m_Surface_Wind_Speed_1,            "[Sortie Field] = 'Surface Wind Speed 1'"             );
         LoadItem( pWeatherArea1RecordSet->m_Wind_Gust_Speed_1,               "[Sortie Field] = 'Wind Gust Speed 1'"                );
         LoadItem( pWeatherArea1RecordSet->m_Wind_Gust_Variation_1,           "[Sortie Field] = 'Wind Gust Variation 1'"            );
         LoadItem( pWeatherArea1RecordSet->m_Precipitation_Type_1,            "[Sortie Field] = 'Precipitation Type 1'"             );
         LoadItem( pWeatherArea1RecordSet->m_Precipitation_Intensity_1,       "[Sortie Field] = 'Precipitation Intensity 1'"        );
         LoadItem( pWeatherArea1RecordSet->m_Lightning_1,                     "[Sortie Field] = 'Lightning 1'"                      );
         LoadItem( pWeatherArea1RecordSet->m_Hail_1,                          "[Sortie Field] = 'Hail 1'"                           );
         LoadItem( pWeatherArea1RecordSet->m_Runway_Condition_Rating_1,       "[Sortie Field] = 'Runway Condition Rating 1'"        );
         LoadItem( pWeatherArea1RecordSet->m_Runway_Contamination_1,          "[Sortie Field] = 'Runway Contamination 1'"           );
         LoadItem( pWeatherArea1RecordSet->m_Blowing_Contaminant_1,           "[Sortie Field] = 'Blowing Contaminant 1'"            );
         LoadItem( pWeatherArea1RecordSet->m_Runway_Skidmarks_1,              "[Sortie Field] = 'Runway Skidmarks 1'"               );
         LoadItem( pWeatherArea1RecordSet->m_Ice_Quantity_1,                  "[Sortie Field] = 'Ice Quantity 1'"                   );
         LoadItem( pWeatherArea1RecordSet->m_Ice_Time_1,                      "[Sortie Field] = 'Ice Time 1'"                       );
         LoadItem( pWeatherArea1RecordSet->m_Turbulence_Intensity_1,          "[Sortie Field] = 'Turbulence Intensity 1'"           );
         LoadItem( pWeatherArea1RecordSet->m_Turbulence_Enable_1,             "[Sortie Field] = 'Turbulence Enable 1'"              );
      }
   }
   pWeatherArea1RecordSet->Close();
   delete   pWeatherArea1RecordSet;


   CWeatherArea2RecordSet  *pWeatherArea2RecordSet =  new   CWeatherArea2RecordSet(NULL);
   pWeatherArea2RecordSet->Open();

   if (pWeatherArea2RecordSet->Find(AFX_DAO_FIRST, m_condition_for_find))
   {
      LoadItem( pWeatherArea2RecordSet->m_Use_Weather_Area_2,                 "[Sortie Field] = 'Use Weather Area 2'"               );

      if (pWeatherArea2RecordSet->m_Use_Weather_Area_2 == "1")
      {
         found =  true;
         LoadItem( pWeatherArea2RecordSet->m_Active_Airport_2,                "[Sortie Field] = 'Active Airport 2'"                 );
         LoadItem( pWeatherArea2RecordSet->m_Active_Runway_2,                 "[Sortie Field] = 'Active Runway 2'"                  );
         LoadItem( pWeatherArea2RecordSet->m_Weather_Area_Radius_2,           "[Sortie Field] = 'Weather Area Radius 2'"            );
         LoadItem( pWeatherArea2RecordSet->m_Field_Pressure_Hg_2,             "[Sortie Field] = 'Field Pressure Hg 2'"              );
         LoadItem( pWeatherArea2RecordSet->m_Field_Temperature_F_2,           "[Sortie Field] = 'Field Temperature F 2'"            );
         LoadItem( pWeatherArea2RecordSet->m_RVR_Feet_2,                      "[Sortie Field] = 'RVR Feet 2'"                       );
         LoadItem( pWeatherArea2RecordSet->m_Visibility_sm_2,                 "[Sortie Field] = 'Visibility sm 2'"                  );
         LoadItem( pWeatherArea2RecordSet->m_Time_of_Day_2,                   "[Sortie Field] = 'Time of Day 2'"                    );
         LoadItem( pWeatherArea2RecordSet->m_Upper_Cloud_Top_2,               "[Sortie Field] = 'Upper Cloud Top 2'"                );
         LoadItem( pWeatherArea2RecordSet->m_Upper_Cloud_Bottom_2,            "[Sortie Field] = 'Upper Cloud Bottom 2'"             );
         LoadItem( pWeatherArea2RecordSet->m_Upper_Cloud_Type_2,              "[Sortie Field] = 'Upper Cloud Type 2'"               );
         LoadItem( pWeatherArea2RecordSet->m_Lower_Cloud_Top_2,               "[Sortie Field] = 'Lower Cloud Top 2'"                );
         LoadItem( pWeatherArea2RecordSet->m_Lower_Cloud_Bottom_2,            "[Sortie Field] = 'Lower Cloud Bottom 2'"             );
         LoadItem( pWeatherArea2RecordSet->m_Lower_Cloud_Type_2,              "[Sortie Field] = 'Lower Cloud Type 2'"               );
         LoadItem( pWeatherArea2RecordSet->m_Cirrus_2,                        "[Sortie Field] = 'Cirrus 2'"                         );
         LoadItem( pWeatherArea2RecordSet->m_Scud_2,                          "[Sortie Field] = 'Scud 2'"                           );
         LoadItem( pWeatherArea2RecordSet->m_Fog_Type_2,                      "[Sortie Field] = 'Fog Type 2'"                       );
         LoadItem( pWeatherArea2RecordSet->m_Fog_Top_2,                       "[Sortie Field] = 'Fog Top 2'"                        );
         LoadItem( pWeatherArea2RecordSet->m_Surface_Wind_Direction_2,        "[Sortie Field] = 'Surface Wind Direction 2'"         );
         LoadItem( pWeatherArea2RecordSet->m_Surface_Wind_Speed_2,            "[Sortie Field] = 'Surface Wind Speed 2'"             );
         LoadItem( pWeatherArea2RecordSet->m_Wind_Gust_Speed_2,               "[Sortie Field] = 'Wind Gust Speed 2'"                );
         LoadItem( pWeatherArea2RecordSet->m_Wind_Gust_Variation_2,           "[Sortie Field] = 'Wind Gust Variation 2'"            );
         LoadItem( pWeatherArea2RecordSet->m_Precipitation_Type_2,            "[Sortie Field] = 'Precipitation Type 2'"             );
         LoadItem( pWeatherArea2RecordSet->m_Precipitation_Intensity_2,       "[Sortie Field] = 'Precipitation Intensity 2'"        );
         LoadItem( pWeatherArea2RecordSet->m_Lightning_2,                     "[Sortie Field] = 'Lightning 2'"                      );
         LoadItem( pWeatherArea2RecordSet->m_Hail_2,                          "[Sortie Field] = 'Hail 2'"                           );
         LoadItem( pWeatherArea2RecordSet->m_Runway_Condition_Rating_2,       "[Sortie Field] = 'Runway Condition Rating 2'"        );
         LoadItem( pWeatherArea2RecordSet->m_Runway_Contamination_2,          "[Sortie Field] = 'Runway Contamination 2'"           );
         LoadItem( pWeatherArea2RecordSet->m_Blowing_Contaminant_2,           "[Sortie Field] = 'Blowing Contaminant 2'"            );
         LoadItem( pWeatherArea2RecordSet->m_Runway_Skidmarks_2,              "[Sortie Field] = 'Runway Skidmarks 2'"               );
         LoadItem( pWeatherArea2RecordSet->m_Ice_Quantity_2,                  "[Sortie Field] = 'Ice Quantity 2'"                   );
         LoadItem( pWeatherArea2RecordSet->m_Ice_Time_2,                      "[Sortie Field] = 'Ice Time 2'"                       );
         LoadItem( pWeatherArea2RecordSet->m_Turbulence_Intensity_2,          "[Sortie Field] = 'Turbulence Intensity 2'"           );
         LoadItem( pWeatherArea2RecordSet->m_Turbulence_Enable_2,             "[Sortie Field] = 'Turbulence Enable 2'"              );
      }
   }
   pWeatherArea2RecordSet->Close();
   delete   pWeatherArea2RecordSet;


   CWeatherArea3RecordSet  *pWeatherArea3RecordSet =  new   CWeatherArea3RecordSet(NULL);
   pWeatherArea3RecordSet->Open();

   if (pWeatherArea3RecordSet->Find(AFX_DAO_FIRST, m_condition_for_find))
   {
      LoadItem( pWeatherArea3RecordSet->m_Use_Weather_Area_3,                 "[Sortie Field] = 'Use Weather Area 3'"               );

      if (pWeatherArea3RecordSet->m_Use_Weather_Area_3 == "1")
      {
         found =  true;
         LoadItem( pWeatherArea3RecordSet->m_Active_Airport_3,                "[Sortie Field] = 'Active Airport 3'"                 );
         LoadItem( pWeatherArea3RecordSet->m_Active_Runway_3,                 "[Sortie Field] = 'Active Runway 3'"                  );
         LoadItem( pWeatherArea3RecordSet->m_Weather_Area_Radius_3,           "[Sortie Field] = 'Weather Area Radius 3'"            );
         LoadItem( pWeatherArea3RecordSet->m_Field_Pressure_Hg_3,             "[Sortie Field] = 'Field Pressure Hg 3'"              );
         LoadItem( pWeatherArea3RecordSet->m_Field_Temperature_F_3,           "[Sortie Field] = 'Field Temperature F 3'"            );
         LoadItem( pWeatherArea3RecordSet->m_RVR_Feet_3,                      "[Sortie Field] = 'RVR Feet 3'"                       );
         LoadItem( pWeatherArea3RecordSet->m_Visibility_sm_3,                 "[Sortie Field] = 'Visibility sm 3'"                  );
         LoadItem( pWeatherArea3RecordSet->m_Time_of_Day_3,                   "[Sortie Field] = 'Time of Day 3'"                    );
         LoadItem( pWeatherArea3RecordSet->m_Upper_Cloud_Top_3,               "[Sortie Field] = 'Upper Cloud Top 3'"                );
         LoadItem( pWeatherArea3RecordSet->m_Upper_Cloud_Bottom_3,            "[Sortie Field] = 'Upper Cloud Bottom 3'"             );
         LoadItem( pWeatherArea3RecordSet->m_Upper_Cloud_Type_3,              "[Sortie Field] = 'Upper Cloud Type 3'"               );
         LoadItem( pWeatherArea3RecordSet->m_Lower_Cloud_Top_3,               "[Sortie Field] = 'Lower Cloud Top 3'"                );
         LoadItem( pWeatherArea3RecordSet->m_Lower_Cloud_Bottom_3,            "[Sortie Field] = 'Lower Cloud Bottom 3'"             );
         LoadItem( pWeatherArea3RecordSet->m_Lower_Cloud_Type_3,              "[Sortie Field] = 'Lower Cloud Type 3'"               );
         LoadItem( pWeatherArea3RecordSet->m_Cirrus_3,                        "[Sortie Field] = 'Cirrus 3'"                         );
         LoadItem( pWeatherArea3RecordSet->m_Scud_3,                          "[Sortie Field] = 'Scud 3'"                           );
         LoadItem( pWeatherArea3RecordSet->m_Fog_Type_3,                      "[Sortie Field] = 'Fog Type 3'"                       );
         LoadItem( pWeatherArea3RecordSet->m_Fog_Top_3,                       "[Sortie Field] = 'Fog Top 3'"                        );
         LoadItem( pWeatherArea3RecordSet->m_Surface_Wind_Direction_3,        "[Sortie Field] = 'Surface Wind Direction 3'"         );
         LoadItem( pWeatherArea3RecordSet->m_Surface_Wind_Speed_3,            "[Sortie Field] = 'Surface Wind Speed 3'"             );
         LoadItem( pWeatherArea3RecordSet->m_Wind_Gust_Speed_3,               "[Sortie Field] = 'Wind Gust Speed 3'"                );
         LoadItem( pWeatherArea3RecordSet->m_Wind_Gust_Variation_3,           "[Sortie Field] = 'Wind Gust Variation 3'"            );
         LoadItem( pWeatherArea3RecordSet->m_Precipitation_Type_3,            "[Sortie Field] = 'Precipitation Type 3'"             );
         LoadItem( pWeatherArea3RecordSet->m_Precipitation_Intensity_3,       "[Sortie Field] = 'Precipitation Intensity 3'"        );
         LoadItem( pWeatherArea3RecordSet->m_Lightning_3,                     "[Sortie Field] = 'Lightning 3'"                      );
         LoadItem( pWeatherArea3RecordSet->m_Hail_3,                          "[Sortie Field] = 'Hail 3'"                           );
         LoadItem( pWeatherArea3RecordSet->m_Runway_Condition_Rating_3,       "[Sortie Field] = 'Runway Condition Rating 3'"        );
         LoadItem( pWeatherArea3RecordSet->m_Runway_Contamination_3,          "[Sortie Field] = 'Runway Contamination 3'"           );
         LoadItem( pWeatherArea3RecordSet->m_Blowing_Contaminant_3,           "[Sortie Field] = 'Blowing Contaminant 3'"            );
         LoadItem( pWeatherArea3RecordSet->m_Runway_Skidmarks_3,              "[Sortie Field] = 'Runway Skidmarks 3'"               );
         LoadItem( pWeatherArea3RecordSet->m_Ice_Quantity_3,                  "[Sortie Field] = 'Ice Quantity 3'"                   );
         LoadItem( pWeatherArea3RecordSet->m_Ice_Time_3,                      "[Sortie Field] = 'Ice Time 3'"                       );
         LoadItem( pWeatherArea3RecordSet->m_Turbulence_Intensity_3,          "[Sortie Field] = 'Turbulence Intensity 3'"           );
         LoadItem( pWeatherArea3RecordSet->m_Turbulence_Enable_3,             "[Sortie Field] = 'Turbulence Enable 3'"              );
      }
   }
   pWeatherArea3RecordSet->Close();
   delete   pWeatherArea3RecordSet;


   CWeatherArea4RecordSet  *pWeatherArea4RecordSet =  new   CWeatherArea4RecordSet(NULL);
   pWeatherArea4RecordSet->Open();

   if (pWeatherArea4RecordSet->Find(AFX_DAO_FIRST, m_condition_for_find))
   {
      LoadItem( pWeatherArea4RecordSet->m_Use_Weather_Area_4,                 "[Sortie Field] = 'Use Weather Area 4'"               );

      if (pWeatherArea4RecordSet->m_Use_Weather_Area_4 == "1")
      {
         found =  true;
         LoadItem( pWeatherArea4RecordSet->m_Active_Airport_4,                "[Sortie Field] = 'Active Airport 4'"                 );
         LoadItem( pWeatherArea4RecordSet->m_Active_Runway_4,                 "[Sortie Field] = 'Active Runway 4'"                  );
         LoadItem( pWeatherArea4RecordSet->m_Weather_Area_Radius_4,           "[Sortie Field] = 'Weather Area Radius 4'"            );
         LoadItem( pWeatherArea4RecordSet->m_Field_Pressure_Hg_4,             "[Sortie Field] = 'Field Pressure Hg 4'"              );
         LoadItem( pWeatherArea4RecordSet->m_Field_Temperature_F_4,           "[Sortie Field] = 'Field Temperature F 4'"            );
         LoadItem( pWeatherArea4RecordSet->m_RVR_Feet_4,                      "[Sortie Field] = 'RVR Feet 4'"                       );
         LoadItem( pWeatherArea4RecordSet->m_Visibility_sm_4,                 "[Sortie Field] = 'Visibility sm 4'"                  );
         LoadItem( pWeatherArea4RecordSet->m_Time_of_Day_4,                   "[Sortie Field] = 'Time of Day 4'"                    );
         LoadItem( pWeatherArea4RecordSet->m_Upper_Cloud_Top_4,               "[Sortie Field] = 'Upper Cloud Top 4'"                );
         LoadItem( pWeatherArea4RecordSet->m_Upper_Cloud_Bottom_4,            "[Sortie Field] = 'Upper Cloud Bottom 4'"             );
         LoadItem( pWeatherArea4RecordSet->m_Upper_Cloud_Type_4,              "[Sortie Field] = 'Upper Cloud Type 4'"               );
         LoadItem( pWeatherArea4RecordSet->m_Lower_Cloud_Top_4,               "[Sortie Field] = 'Lower Cloud Top 4'"                );
         LoadItem( pWeatherArea4RecordSet->m_Lower_Cloud_Bottom_4,            "[Sortie Field] = 'Lower Cloud Bottom 4'"             );
         LoadItem( pWeatherArea4RecordSet->m_Lower_Cloud_Type_4,              "[Sortie Field] = 'Lower Cloud Type 4'"               );
         LoadItem( pWeatherArea4RecordSet->m_Cirrus_4,                        "[Sortie Field] = 'Cirrus 4'"                         );
         LoadItem( pWeatherArea4RecordSet->m_Scud_4,                          "[Sortie Field] = 'Scud 4'"                           );
         LoadItem( pWeatherArea4RecordSet->m_Fog_Type_4,                      "[Sortie Field] = 'Fog Type 4'"                       );
         LoadItem( pWeatherArea4RecordSet->m_Fog_Top_4,                       "[Sortie Field] = 'Fog Top 4'"                        );
         LoadItem( pWeatherArea4RecordSet->m_Surface_Wind_Direction_4,        "[Sortie Field] = 'Surface Wind Direction 4'"         );
         LoadItem( pWeatherArea4RecordSet->m_Surface_Wind_Speed_4,            "[Sortie Field] = 'Surface Wind Speed 4'"             );
         LoadItem( pWeatherArea4RecordSet->m_Wind_Gust_Speed_4,               "[Sortie Field] = 'Wind Gust Speed 4'"                );
         LoadItem( pWeatherArea4RecordSet->m_Wind_Gust_Variation_4,           "[Sortie Field] = 'Wind Gust Variation 4'"            );
         LoadItem( pWeatherArea4RecordSet->m_Precipitation_Type_4,            "[Sortie Field] = 'Precipitation Type 4'"             );
         LoadItem( pWeatherArea4RecordSet->m_Precipitation_Intensity_4,       "[Sortie Field] = 'Precipitation Intensity 4'"        );
         LoadItem( pWeatherArea4RecordSet->m_Lightning_4,                     "[Sortie Field] = 'Lightning 4'"                      );
         LoadItem( pWeatherArea4RecordSet->m_Hail_4,                          "[Sortie Field] = 'Hail 4'"                           );
         LoadItem( pWeatherArea4RecordSet->m_Runway_Condition_Rating_4,       "[Sortie Field] = 'Runway Condition Rating 4'"        );
         LoadItem( pWeatherArea4RecordSet->m_Runway_Contamination_4,          "[Sortie Field] = 'Runway Contamination 4'"           );
         LoadItem( pWeatherArea4RecordSet->m_Blowing_Contaminant_4,           "[Sortie Field] = 'Blowing Contaminant 4'"            );
         LoadItem( pWeatherArea4RecordSet->m_Runway_Skidmarks_4,              "[Sortie Field] = 'Runway Skidmarks 4'"               );
         LoadItem( pWeatherArea4RecordSet->m_Ice_Quantity_4,                  "[Sortie Field] = 'Ice Quantity 4'"                   );
         LoadItem( pWeatherArea4RecordSet->m_Ice_Time_4,                      "[Sortie Field] = 'Ice Time 4'"                       );
         LoadItem( pWeatherArea4RecordSet->m_Turbulence_Intensity_4,          "[Sortie Field] = 'Turbulence Intensity 4'"           );
         LoadItem( pWeatherArea4RecordSet->m_Turbulence_Enable_4,             "[Sortie Field] = 'Turbulence Enable 4'"              );
      }
   }
   pWeatherArea4RecordSet->Close();
   delete   pWeatherArea4RecordSet;

   if (found)
      ChangeData(TRUE, "wx_load_cmd", "1");
}


void CSortieFilesSystemInterface::SetWeatherGlobals(void)
{
   CWeatherGlobalsRecordSet   *pWeatherGlobalsRecordSet  =  new   CWeatherGlobalsRecordSet(NULL);
   pWeatherGlobalsRecordSet->Open();

   BOOL  found =  pWeatherGlobalsRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);
   if (found)
   {
      LoadItem( pWeatherGlobalsRecordSet->m_Weather_Mode,               "[Sortie Field] = 'Weather Mode'"                     );

      LoadItem( pWeatherGlobalsRecordSet->m_Surface_Wind_Altitude_AGL,  "[Sortie Field] = 'Surface Wind Altitude AGL'"        );
      LoadItem( pWeatherGlobalsRecordSet->m_Cruise_Wind_Altitude_MSL,   "[Sortie Field] = 'Cruise Wind Altitude MSL'"         );
      LoadItem( pWeatherGlobalsRecordSet->m_Cruise_Wind_Speed___knots,  "[Sortie Field] = 'Cruise Wind Speed - knots'"        );
      LoadItem( pWeatherGlobalsRecordSet->m_Cruise_Wind_Direction,      "[Sortie Field] = 'Cruise Wind Direction'"            );

      LoadItem( pWeatherGlobalsRecordSet->m_Moon_Azimuth,               "[Sortie Field] = 'Moon Azimuth'"                     );
      LoadItem( pWeatherGlobalsRecordSet->m_Moon_Elevation,             "[Sortie Field] = 'Moon Elevation'"                   );
      LoadItem( pWeatherGlobalsRecordSet->m_Moon_Phase,                 "[Sortie Field] = 'Moon Phase'"                       );

      LoadItem( pWeatherGlobalsRecordSet->m_Windshear_Model,            "[Sortie Field] = 'Windshear Model'"                  );
      LoadItem( pWeatherGlobalsRecordSet->m_Windshear_Intensity,        "[Sortie Field] = 'Windshear Intensity'"              );
      LoadItem( pWeatherGlobalsRecordSet->m_Windshear_Enable,           "[Sortie Field] = 'Windshear Enable'"                 );
      LoadItem( pWeatherGlobalsRecordSet->m_Turbulence_Enable,          "[Sortie Field] = 'Turbulence Enable'"                );
      LoadItem( pWeatherGlobalsRecordSet->m_Turbulence_Intensity,       "[Sortie Field] = 'Turbulence Intensity'"             );

      LoadItem( pWeatherGlobalsRecordSet->m_Microburst_Model,           "[Sortie Field] = 'Microburst Model'"                 );
      LoadItem( pWeatherGlobalsRecordSet->m_Microburst_Intensity,       "[Sortie Field] = 'Microburst Intensity'"             );
      LoadItem( pWeatherGlobalsRecordSet->m_Microburst_Position,        "[Sortie Field] = 'Microburst Position'"              );
      LoadItem( pWeatherGlobalsRecordSet->m_Microburst_Enable,          "[Sortie Field] = 'Microburst Enable'"                );
   }

   pWeatherGlobalsRecordSet->Close();
   delete   pWeatherGlobalsRecordSet;

   {
      CVariant var;
      CCommsShared::GetHostValue(var, _FSI_STL::string("Storm_Type"));
      long  old_type =  var.Type();
      var.Value(0L);
      var.ChangeType(old_type);
      CCommsShared::SetHostValue(var, _FSI_STL::string("Storm_Type"), 0);

      CCommsShared::GetHostValue(var, _FSI_STL::string("Storm_Select"));
      old_type =  var.Type();
      var.Value(1L);
      var.ChangeType(old_type);
      CCommsShared::SetHostValue(var, _FSI_STL::string("Storm_Select"), 0);
   }

}


void  CSortieFilesSystemInterface::SetRecordedData(void)
{
   CRecordingsRecordSet *pRecordingsRecordSet   =  new   CRecordingsRecordSet(NULL);
   pRecordingsRecordSet->Open();

   BOOL  found =  pRecordingsRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);
   if (found)
   {
      HKEY     hKey;
      RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\FlightSafety\\Common"), 0, KEY_READ, &hKey);

      CString  path_to_demos;
      CString  path_to_formation_lead;

      if (hKey != NULL)
      {
         CString  sws_name;
         CString  strValue;
         // get SWS name
         DWORD    dwType, dwCount;
         LONG     lResult;
         lResult  =  RegQueryValueEx(hKey,      _T("SWS Name"),      NULL, &dwType, NULL, &dwCount);
         if (lResult == ERROR_SUCCESS)
         {
            ASSERT(dwType  == REG_SZ);
            lResult  =  RegQueryValueEx(hKey,   _T("SWS Name"),      NULL, &dwType, (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)), &dwCount);
            strValue.ReleaseBuffer();
         }
         sws_name =  strValue;

         // get Demonstrations Folder
         // lResult  =  RegQueryValueEx(hKey,      _T("Demonstrations Folder"),   NULL, &dwType, NULL, &dwCount);
         // if (lResult == ERROR_SUCCESS)
         // {
         //    ASSERT(dwType  == REG_SZ);
         //    lResult  =  RegQueryValueEx(hKey,   _T("Demonstrations Folder"),   NULL, &dwType, (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)), &dwCount);
         //    strValue.ReleaseBuffer();
         // }
         // path_to_demos  =  sws_name + strValue;
         path_to_demos  =  sws_name + "Saved Recordings";

         // get Formation Lead Folder
         // lResult  =  RegQueryValueEx(hKey,      _T("Formation Lead Folder"),   NULL, &dwType, NULL, &dwCount);
         // if (lResult == ERROR_SUCCESS)
         // {
         //    ASSERT(dwType  == REG_SZ);
         //    lResult  =  RegQueryValueEx(hKey,   _T("Formation Lead Folder"),   NULL, &dwType, (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)), &dwCount);
         //    strValue.ReleaseBuffer();
         // }
         // path_to_formation_lead  =  sws_name + strValue;
         path_to_formation_lead  =  sws_name + "Saved Recordings";

         // clean up
         RegCloseKey(hKey);
      }

      CWorkingDlg workingDlg;

      if (!pRecordingsRecordSet->m_Demo_File_1.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_1;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_1;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_1);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_1);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_2.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_2;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_2;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_2);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_2);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_3.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_3;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_3;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_3);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_3);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_4.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_4;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_4;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_4);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_4);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_5.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_5;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_5;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_5);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_5);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_6.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_6;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_6;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_6);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_6);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_7.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_7;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_7;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_7);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_7);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_8.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_8;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_8;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_8);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_8);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_9.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_9;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_9;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_9);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_9);

         workingDlg.DoModal();
      }

      if (!pRecordingsRecordSet->m_Demo_File_10.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_demos  +  "\\"  +  pRecordingsRecordSet->m_Demo_File_10;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\demos\\"  +  pRecordingsRecordSet->m_Demo_File_10;
         workingDlg.m_CStr_primary_extension    =  "dmo";
         workingDlg.m_CStr_secondary_extension  =  "";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] demostration file...", pRecordingsRecordSet->m_Demo_File_10);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] demostration file.", pRecordingsRecordSet->m_Demo_File_10);

         workingDlg.DoModal();
      }
      if (!pRecordingsRecordSet->m_Formation_Lead_File_1.IsEmpty())
      {
         workingDlg.m_CStr_source               =  path_to_formation_lead  +  "\\"  +  pRecordingsRecordSet->m_Formation_Lead_File_1;
         workingDlg.m_CStr_destination          =  "c:\\ftd data\\formation_lead\\"  +  pRecordingsRecordSet->m_Formation_Lead_File_1;
         workingDlg.m_CStr_primary_extension    =  "dat";
         workingDlg.m_CStr_secondary_extension  =  "idx";
         workingDlg.m_CStr_start_caption.Format("Copying [%s] formation lead profile...", pRecordingsRecordSet->m_Formation_Lead_File_1);
         workingDlg.m_CStr_finish_caption.Format("Finished copying [%s] formation lead profile.", pRecordingsRecordSet->m_Formation_Lead_File_1);

         workingDlg.DoModal();
      }
   }

   pRecordingsRecordSet->Close();
   delete   pRecordingsRecordSet;
}


void CSortieFilesSystemInterface::LoadItem(CString& member, char *row)
{
   static _FSI_STL::string  stlStrVariable;
   static _FSI_STL::string  stlStrValue;

   stlStrValue = (LPCTSTR)member;
   m_pVariableMapSet->Find(AFX_DAO_FIRST, row);
   stlStrVariable = (LPCTSTR)m_pVariableMapSet->m_Variable;
   ChangeData(m_pVariableMapSet->m_Local, stlStrVariable, stlStrValue);
}


void CSortieFilesSystemInterface::ResetMissionTime()
{
   ChangeData(FALSE, "mission_time", "0");
}


void  CSortieFilesSystemInterface::SetMalfunctions(void)
{
   // Clear all armed and active malfunctions.
   CMalfFieldAction::m_malfunction.ClearAllPreselects();
   CMalfFieldAction::m_malfunction.ClearAll();

   Sleep(1000);   // Allow time for "clears" to finish.

   CMalfunctionsRecordSet  *pCMalfunctionsRecordSet   =  new   CMalfunctionsRecordSet(NULL);
   pCMalfunctionsRecordSet->Open();

   BOOL  found =  pCMalfunctionsRecordSet->Find(AFX_DAO_FIRST, m_condition_for_find);

   if (found)
   {
      _FSI_STL::string  stlStrTrigger1 =  "";
      char              lConditional1  =  0;
      double            dValue1        =  0.0;
      _FSI_STL::string  stlStrTrigger2 =  "";
      char              lConditional2  =  0;
      double            dValue2        =  0.0;
      long              lAndResults    =  0;
      _FSI_STL::string  stlStrLabel    =  "";

      CMalfPreselectTypeRecordSet      *pMalfPreselectTypeRecordSet  =  new   CMalfPreselectTypeRecordSet(NULL);
      pMalfPreselectTypeRecordSet->Open();

      stlStrLabel    =  (LPCTSTR)pCMalfunctionsRecordSet->m_Malfunction_1_Label;
      if (pCMalfunctionsRecordSet->m_Malfunction_1_Activation        == "1")                 // Preselect
      {
         CString  condition_for_find;
         condition_for_find.Format("[Ordinal] = %d", atoi(pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_1_Type));

         if (pMalfPreselectTypeRecordSet->Find(AFX_DAO_FIRST, condition_for_find))
         {
            stlStrTrigger1    =  (LPCTSTR)pMalfPreselectTypeRecordSet->m_Preselect_Type;
            if (pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_1_Operator == "0")        // less than
               lConditional1  =  LESS_THAN;
            else
               lConditional1  =  GREATER_THAN;
            dValue1           =  atof(pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_1_Value);

            bool  compound_condition   =  true;                                              // Assume it's a compound condition...
            if (atoi(pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_Logical) ==  0)      // Neither AND or OR, so it's not 
               compound_condition      =  false;                                             // a compound condition.

            if (compound_condition)
            {
               // Find out whether to && or to || the trigger conditions.
               if (atoi(pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_Logical) ==  2)   // OR
                  lAndResults    =  0;                                                       // ||
               else                                                                          // AND
                  lAndResults    =  1;                                                       // &&

               condition_for_find.Format("[Ordinal] = %d", atoi(pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_2_Type));

               if (pMalfPreselectTypeRecordSet->Find(AFX_DAO_FIRST, condition_for_find))
               {
                  stlStrTrigger2    =  (LPCTSTR)pMalfPreselectTypeRecordSet->m_Preselect_Type;
                  if (pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_2_Operator == "0")  // less than
                     lConditional2  =  LESS_THAN;
                  else
                     lConditional2  =  GREATER_THAN;
                  dValue2           =  atof(pCMalfunctionsRecordSet->m_Malfunction_1_Preselect_2_Value);
               }
            }
            else
            {
               stlStrTrigger2    =  "";
               lConditional2     =  LESS_THAN;
               dValue2           =  0.0;
               lAndResults       =  0;
            }

            bool  bPreselectValid   =  CMalfFieldAction::m_malfunction.AddPreselect(stlStrTrigger1,   lConditional1, dValue1, 
                                                                                    stlStrTrigger2,   lConditional2, dValue2, 
                                                                                    stlStrLabel,      lAndResults);

            if (bPreselectValid)
            {
               if (CMalfFieldAction::m_malfunction.m_mapCommsActions.find(stlStrLabel) != CMalfFieldAction::m_malfunction.m_mapCommsActions.end())
                  CMalfFieldAction::m_malfunction.m_mapCommsActions[stlStrLabel]->ShowAsPreselect(true);
            }
         }
      }
      else if (pCMalfunctionsRecordSet->m_Malfunction_1_Activation   == "2")  // Activate now
      {
         ChangeData(FALSE, stlStrLabel, "1");
      }


      stlStrLabel    =  (LPCTSTR)pCMalfunctionsRecordSet->m_Malfunction_2_Label;
      if (pCMalfunctionsRecordSet->m_Malfunction_2_Activation        == "1")                 // Preselect
      {
         CString  condition_for_find;
         condition_for_find.Format("[Ordinal] = %d", atoi(pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_1_Type));

         if (pMalfPreselectTypeRecordSet->Find(AFX_DAO_FIRST, condition_for_find))
         {
            stlStrTrigger1    =  (LPCTSTR)pMalfPreselectTypeRecordSet->m_Preselect_Type;
            if (pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_1_Operator == "0")        // less than
               lConditional1  =  LESS_THAN;
            else
               lConditional1  =  GREATER_THAN;
            dValue1           =  atof(pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_1_Value);

            bool  compound_condition   =  true;                                              // Assume it's a compound condition...
            if (atoi(pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_Logical) ==  0)      // Neither AND or OR, so it's not 
               compound_condition      =  false;                                             // a compound condition.

            if (compound_condition)
            {
               // Find out whether to && or to || the trigger conditions.
               if (atoi(pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_Logical) ==  2)   // OR
                  lAndResults    =  0;                                                       // ||
               else                                                                          // AND
                  lAndResults    =  1;                                                       // &&

               condition_for_find.Format("[Ordinal] = %d", atoi(pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_2_Type));

               if (pMalfPreselectTypeRecordSet->Find(AFX_DAO_FIRST, condition_for_find))
               {
                  stlStrTrigger2    =  (LPCTSTR)pMalfPreselectTypeRecordSet->m_Preselect_Type;
                  if (pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_2_Operator == "0")  // less than
                     lConditional2  =  LESS_THAN;
                  else
                     lConditional2  =  GREATER_THAN;
                  dValue2           =  atof(pCMalfunctionsRecordSet->m_Malfunction_2_Preselect_2_Value);
               }
            }
            else
            {
               stlStrTrigger2    =  "";
               lConditional2     =  LESS_THAN;
               dValue2           =  0.0;
               lAndResults       =  0;
            }

            bool  bPreselectValid   =  CMalfFieldAction::m_malfunction.AddPreselect(stlStrTrigger1,   lConditional1, dValue1, 
                                                                                    stlStrTrigger2,   lConditional2, dValue2, 
                                                                                    stlStrLabel,      lAndResults);

            if (bPreselectValid)
            {
               if (CMalfFieldAction::m_malfunction.m_mapCommsActions.find(stlStrLabel) != CMalfFieldAction::m_malfunction.m_mapCommsActions.end())
                  CMalfFieldAction::m_malfunction.m_mapCommsActions[stlStrLabel]->ShowAsPreselect(true);
            }
         }
      }
      else if (pCMalfunctionsRecordSet->m_Malfunction_2_Activation   == "2")  // Activate now
      {
         ChangeData(FALSE, stlStrLabel, "1");
      }


      stlStrLabel    =  (LPCTSTR)pCMalfunctionsRecordSet->m_Malfunction_3_Label;
      if (pCMalfunctionsRecordSet->m_Malfunction_3_Activation        == "1")                 // Preselect
      {
         CString  condition_for_find;
         condition_for_find.Format("[Ordinal] = %d", atoi(pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_1_Type));

         if (pMalfPreselectTypeRecordSet->Find(AFX_DAO_FIRST, condition_for_find))
         {
            stlStrTrigger1    =  (LPCTSTR)pMalfPreselectTypeRecordSet->m_Preselect_Type;
            if (pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_1_Operator == "0")        // less than
               lConditional1  =  LESS_THAN;
            else
               lConditional1  =  GREATER_THAN;
            dValue1           =  atof(pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_1_Value);

            bool  compound_condition   =  true;                                              // Assume it's a compound condition...
            if (atoi(pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_Logical) ==  0)      // Neither AND or OR, so it's not 
               compound_condition      =  false;                                             // a compound condition.

            if (compound_condition)
            {
               // Find out whether to && or to || the trigger conditions.
               if (atoi(pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_Logical) ==  2)   // OR
                  lAndResults    =  0;                                                       // ||
               else                                                                          // AND
                  lAndResults    =  1;                                                       // &&

               condition_for_find.Format("[Ordinal] = %d", atoi(pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_2_Type));

               if (pMalfPreselectTypeRecordSet->Find(AFX_DAO_FIRST, condition_for_find))
               {
                  stlStrTrigger2    =  (LPCTSTR)pMalfPreselectTypeRecordSet->m_Preselect_Type;
                  if (pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_2_Operator == "0")  // less than
                     lConditional2  =  LESS_THAN;
                  else
                     lConditional2  =  GREATER_THAN;
                  dValue2           =  atof(pCMalfunctionsRecordSet->m_Malfunction_3_Preselect_2_Value);
               }
            }
            else
            {
               stlStrTrigger2    =  "";
               lConditional2     =  LESS_THAN;
               dValue2           =  0.0;
               lAndResults       =  0;
            }

            bool  bPreselectValid   =  CMalfFieldAction::m_malfunction.AddPreselect(stlStrTrigger1,   lConditional1, dValue1, 
                                                                                    stlStrTrigger2,   lConditional2, dValue2, 
                                                                                    stlStrLabel,      lAndResults);

            if (bPreselectValid)
            {
               if (CMalfFieldAction::m_malfunction.m_mapCommsActions.find(stlStrLabel) != CMalfFieldAction::m_malfunction.m_mapCommsActions.end())
                  CMalfFieldAction::m_malfunction.m_mapCommsActions[stlStrLabel]->ShowAsPreselect(true);
            }
         }
      }
      else if (pCMalfunctionsRecordSet->m_Malfunction_3_Activation   == "2")  // Activate now
      {
         ChangeData(FALSE, stlStrLabel, "1");
      }

      pMalfPreselectTypeRecordSet->Close();
      delete   pMalfPreselectTypeRecordSet;
   }

   pCMalfunctionsRecordSet->Close();
   delete   pCMalfunctionsRecordSet;
}

