// XMLArchive.h

#ifndef XMLARCHIVE_H
#define XMLARCHIVE_H

#include "comdef.h"
#include <string>

class CXMLArchive;

///////////////////////////////////////////////////////////////////////////
// CXMLNode encapsulates an IXMLDOMNode interface pointer to manage
// that resource and provide convenience functions to the
// IXMLDOMNode interface
///////////////////////////////////////////////////////////////////////////
class CXMLNode
{
public:
	CXMLNode();
	CXMLNode(IXMLDOMNode * node);
	~CXMLNode();
	bool operator==(const CXMLNode& rhs);
	CXMLNode& operator=(const CXMLNode& rhs);
	CXMLNode& operator=(IXMLDOMNode *node);
	IXMLDOMNode *GetNode() { return m_pNode; }

	bool SetAttribute(const BSTR bstrName, VARIANT v);
	bool SetAttribute(const BSTR bstrName, long l);
	bool SetAttribute(const BSTR bstrName, unsigned char c);
	bool SetAttribute(const BSTR bstrName, const char *pszVal);
	bool SetAttribute(const BSTR bstrName, COLORREF rgbVal);
	bool SetAttribute(const BSTR bstrName, const RECT& rVal);

	bool GetAttribute(const BSTR bstrName, VARIANT& v);
	bool GetAttribute(const BSTR bstrName, long& l);
	bool GetAttribute(const BSTR bstrName, unsigned char& c);
	bool GetAttribute(const BSTR bstrName, _bstr_t& bsVal);
	bool GetAttribute(const BSTR bstrName, COLORREF& rgbVal);
	bool GetAttribute(const BSTR bstrName, RECT& rVal);

	bool GetName(_bstr_t& bsName);


protected:

private:
	IXMLDOMNode *m_pNode;
};


///////////////////////////////////////////////////////////////////////////
// CXMLNodeList encapsulates the IXMLDOMNodeList interface pointer to manage
// that resource and provide convenience functions to the
// IXMLDOMNodeList interface
///////////////////////////////////////////////////////////////////////////
class CXMLNodeList
{
public:
	CXMLNodeList(IXMLDOMNodeList *pNodeList);
	CXMLNodeList(CXMLNode *pParent);
	CXMLNodeList(CXMLNode *pParent, const BSTR bstrName);
	~CXMLNodeList();
	int Length(void);
	void Reset(void);
	CXMLNode *operator[](long nIndex);
	CXMLNode *NextNode(void);

private:
	IXMLDOMNodeList *m_pNodeList;
};


// forward reference so that this file doesn't require MFC
class CDocument;

///////////////////////////////////////////////////////////////////////////
// CXMLArchive encapsulates the IXMLDOMDocument interface pointer to manage
// that resource and provide convenience functions to the
// IXMLDOMDocument interface
///////////////////////////////////////////////////////////////////////////
class CXMLArchive
{
public:
	enum Mode { store = 0, load = 1 };

	CXMLArchive(const char *pszFilename, UINT nMode);
	~CXMLArchive();
	IXMLDOMDocument *GetDoc() { return m_pDoc; }

	bool CreateNode(DOMNodeType type, const BSTR bstrName, CXMLNode& newNode);
	CXMLNode * CreateNode(DOMNodeType type, const BSTR bstrName);
	bool InsertNewElement(CXMLNode& pParent, const BSTR bstrName, CXMLNode& newNode);
	CXMLNode * InsertNewElement(CXMLNode *pParent, const BSTR bstrName);
	CXMLNode * InsertNewTextElement(CXMLNode *pParent, const BSTR bstrName, const char *pszText);
	bool InsertNewCData(CXMLNode *pParent, const char *pszData);
	bool RetrieveCData(CXMLNode *pNode, _bstr_t& bsCData);
	bool AppendChild(CXMLNode * pChild);
	bool LoadDocument(void);
	bool SaveDocument(void);
	bool IsStoring(void) { return m_nMode == CXMLArchive::store; }

	bool SelectNode(CXMLNode& parent, const BSTR bstrName, CXMLNode& newNode);
	CXMLNode * SelectNode(CXMLNode *pParent, const BSTR bstrName);
	CXMLNode * SelectNextNode(CXMLNode *pParent);

	CDocument* m_pDocument;

protected:

private:
	_bstr_t			m_bsFilename;
	IXMLDOMDocument *m_pDoc;
	CXMLNode		*m_pCurrParent;
	UINT			m_nMode;

	HRESULT CheckLoad(void);
	HRESULT ReportError(IXMLDOMParseError *pXMLError);
};


///////////////////////////////////////////////////////////////////////////
// The CNamedData template class provides for the convenient
// association of a XML Node name and a variable for input/output of
// that Node's value via CXMLArchive serialization functions.
//
// NOTE: if you use the CNamedData<T> template with a new data type, you'll
// have to implement the pair of IO functions (operator<< and operator>>).
///////////////////////////////////////////////////////////////////////////
template<class _T> class CNamedData
{
public:
	CNamedData(CXMLNode& parent, const BSTR bstrName, _T & tVal):
	  m_pParent(&parent), m_bsName(bstrName), m_tVal(tVal) {}
	~CNamedData(){}

    friend CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData & namedData);
    friend CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData & namedData);

//private:
	CXMLNode	*m_pParent;
	_bstr_t		m_bsName;
	_T&			m_tVal;
};

///////////////////////////////////////////////////////////////////////////
// The CNamedCData class is similar to the CNamedData<T> template class,
// but was provided separately since there was not a unique data type to
// distinguish XML CDATA from other string-based data.
///////////////////////////////////////////////////////////////////////////
template<class _T> class CNamedCData
{
public:
	CNamedCData(CXMLNode& parent, const BSTR bstrName, _T & tVal):
	  m_pParent(&parent), m_bsName(bstrName), m_tVal(tVal) {}
	~CNamedCData(){}

    friend CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedCData & namedCData);
    friend CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedCData & namedCData);

private:
	CXMLNode	*m_pParent;
	_bstr_t		m_bsName;
	_T&			m_tVal;
};

///////////////////////////////////////////////////////////////////////////
// The CNamedAttribute template class provides for the convenient
// association of an XML Attribute name and a variable for input/output of
// that Attribute's value.
//
// NOTE: if you use the CNamedAttribute<T> template with a new data type, you'll
// have to implement the pair of IO functions (operator<< and operator>>).
///////////////////////////////////////////////////////////////////////////
template<class _T> class CNamedAttribute
{
public:
	CNamedAttribute(CXMLNode& parent, const BSTR bstrName, _T & tVal):
	  m_pParent(&parent), m_bsName(bstrName), m_tVal(tVal) {}
	~CNamedAttribute(){}

    friend CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute & namedAttr);
    friend CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute & namedAttr);

private:
	CXMLNode	*m_pParent;
	_bstr_t		m_bsName;
	_T&			m_tVal;
};


#endif // #ifndef XMLARCHIVE_H

