///////////////////////////////////////////////////////////////////////////
// XMLArchive.cpp

#include "..\core\stdafx.h"
#include "XMLArchive.h"
#include <vector>


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

static void DumpFormattedTree(FILE *pFile, IXMLDOMNode* node, int level, bool& bTextChild);
static void DumpXMLString(FILE *pFile, const BSTR bstr);


///////////////////////////////////////////////////////////////////////////
#define CHECKHR(x) {hr = x; if (FAILED(hr)) goto CleanUp;}
#define SAFERELEASE(p) {if (p) {(p)->Release(); p = NULL;}}

// table for converting hex characters to their numeric equivalent
// handles both lower and upper case for A to F.
static char tblHexChars[] = {
	0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  0,  0,  0,  0,  0,  0,
	0, 10, 11, 12, 13, 14, 15,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	0, 10, 11, 12, 13, 14, 15,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
};


///////////////////////////////////////////////////////////////////////////
// CXMLNode encapsulates an IXMLDOMNode interface pointer to manage
// that resource and provide convenience functions to the
// IXMLDOMNode interface
///////////////////////////////////////////////////////////////////////////

CXMLNode::CXMLNode() :
	m_pNode(0)
{
}
CXMLNode::CXMLNode(IXMLDOMNode * node) :
	m_pNode(node)
{
}
	
CXMLNode::~CXMLNode()
{
	SAFERELEASE(m_pNode);
}

bool CXMLNode::operator==(const CXMLNode& rhs)
{
	return m_pNode == rhs.m_pNode;
}

CXMLNode& CXMLNode::operator=(IXMLDOMNode *node)
{
	SAFERELEASE(m_pNode);

	m_pNode = node;
	if (m_pNode)
		m_pNode->AddRef();

	return *this;
}

CXMLNode& CXMLNode::operator=(const CXMLNode& rhs)
{
	if (*this == rhs)
		return *this;

	SAFERELEASE(m_pNode);

	m_pNode = rhs.m_pNode;
	if (m_pNode)
		m_pNode->AddRef();

	return *this;
}

//
// CXMLNode::GetAttribute
// retrieve node name in a friendly way
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::GetName(_bstr_t& bsName)
{
	BSTR nodeName;

	if (SUCCEEDED(m_pNode->get_nodeName(&nodeName)))
	{
		bsName = nodeName;
		return true;
	}
	return false;
}

//
// CXMLNode::GetAttribute
// retrieve a named 'VARIANT' attribute from this node
// this is the low-level worker function
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::GetAttribute(const BSTR bstrName, VARIANT& v)
{
	IXMLDOMElement* pEle = NULL;

	if (SUCCEEDED(m_pNode->QueryInterface(IID_IXMLDOMElement, (void**)&pEle)))
	{
		HRESULT hr = pEle->getAttribute(bstrName, &v);
		SAFERELEASE(pEle);
		return (S_OK == hr);
	}
	return false;
}

//
// CXMLNode::GetAttribute
// retrieve a named 'long' attribute from this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::GetAttribute(const BSTR bstrName, long& l)
{
	VARIANT v;
	if (GetAttribute(bstrName, v))
	{
		HRESULT hr = VariantChangeType(&v, &v, 0, VT_I4);
		l = V_I4(&v);
		return true;
	}
	return false;
}

//
// CXMLNode::GetAttribute
// retrieve a named 'unsigned char' attribute from this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::GetAttribute(const BSTR bstrName, unsigned char& c)
{
	long l;
	if (GetAttribute(bstrName, l))
	{
		c = (unsigned char)l;
		return true;
	}
	return false;
}

//
// CXMLNode::GetAttribute
// retrieve a named '_bstr_t' attribute from this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::GetAttribute(const BSTR bstrName, _bstr_t& bsVal)
{
	VARIANT v;
	if (GetAttribute(bstrName, v) && VT_NULL != v.vt)
	{
		bsVal = V_BSTR(&v);
		return true;
	}
	return false;
}

//
// CXMLNode::GetAttribute
// retrieve a 'COLORREF' attribute from this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::GetAttribute(const BSTR bstrName, COLORREF& rgbVal)
{
	VARIANT v;

	if (!GetAttribute(bstrName, v) || VT_NULL == v.vt)
	{
		return false;
	}
	BSTR bs = V_BSTR(&v);
	if (SysStringLen(bs) < 7)	// e.g. "#FF00FF" as rrggbb
	{
		return false;
	}

	// sanity check before using characters as indices into tblHexChars
	ASSERT(	bs[1] >= '0' && bs[1] <= 'f' &&
			bs[2] >= '0' && bs[2] <= 'f' &&
			bs[3] >= '0' && bs[3] <= 'f' &&
			bs[4] >= '0' && bs[4] <= 'f' &&
			bs[5] >= '0' && bs[5] <= 'f' &&
			bs[6] >= '0' && bs[6] <= 'f');

	rgbVal =	(tblHexChars[bs[1]] <<  4) + (tblHexChars[bs[2]] <<  0) +
				(tblHexChars[bs[3]] << 12) + (tblHexChars[bs[4]] <<  8) +
				(tblHexChars[bs[5]] << 20) + (tblHexChars[bs[6]] << 16);

	return true;
}

//
// CXMLNode::GetAttribute
// retrieve a 'RECT' attribute from this node
// note that this is obtained as four separate attributes
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::GetAttribute(const BSTR bstrName, RECT& rVal)
{
	int nLen = wcslen(bstrName);
	wchar_t wszTemp[100];
	ASSERT(nLen+3 < sizeof(wszTemp) / sizeof(wchar_t));

	wcscpy(wszTemp, bstrName);
	wszTemp[nLen++] = L'_';
	wszTemp[nLen+1] = 0;

	long x = 0;
	long y = 0;
	long w = 0;
	long h = 0;

	wszTemp[nLen] = L'X';
	if (GetAttribute(wszTemp, x))
	{
		wszTemp[nLen] = L'Y';
		if (GetAttribute(wszTemp, y))
		{
			wszTemp[nLen] = L'W';
			if (GetAttribute(wszTemp, w))
			{
				wszTemp[nLen] = L'H';
				if (GetAttribute(wszTemp, h))
				{
					rVal.left   = x;
					rVal.top    = y;
					rVal.right  = x + w;
					rVal.bottom = y + h;
				}
			}
		}
	}

	return false;
}

//
// CXMLNode::SetAttribute
// set a named 'VARIANT' attribute on this node
// this is the low-level worker function
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::SetAttribute(const BSTR bstrName, VARIANT v)
{
	IXMLDOMElement* pEle = NULL;

	if (SUCCEEDED(m_pNode->QueryInterface(IID_IXMLDOMElement, (void**)&pEle)))
	{
		HRESULT hr = pEle->setAttribute(bstrName, v);
		SAFERELEASE(pEle);
		return SUCCEEDED(hr);
	}
	return false;
}

//
// CXMLNode::SetAttribute
// set a named 'long' attribute on this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::SetAttribute(const BSTR bstrName, long l)
{
	VARIANT v;
	v.vt = VT_I4;
	V_I4(&v) = l;
	return SetAttribute(bstrName, v);
}

//
// CXMLNode::SetAttribute
// set a named 'unsigned char' attribute on this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::SetAttribute(const BSTR bstrName, unsigned char c)
{
	VARIANT v;
	v.vt = VT_I2;
	V_I2(&v) = c;
	return SetAttribute(bstrName, v);
}

//
// CXMLNode::SetAttribute
// set a named 'const char *' attribute on this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::SetAttribute(const BSTR bstrName, const char *pszVal)
{
	VARIANT v;
	v.vt = VT_BSTR;
	_bstr_t bs(pszVal);
	V_BSTR(&v) = bs;
	return SetAttribute(bstrName, v);
}

//
// CXMLNode::SetAttribute
// set a 'COLORREF' attribute on this node
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::SetAttribute(const BSTR bstrName, COLORREF rgbVal)
{
	char szTemp[10];
	sprintf(szTemp, "#%02X%02X%02X",
					GetRValue(rgbVal),
					GetGValue(rgbVal),
					GetBValue(rgbVal));
	return SetAttribute(bstrName, szTemp);
}

//
// CXMLNode::SetAttribute
// set a 'RECT' attribute on this node
// note that this uses four separate attributes which are internally named
// Entry   : 
// Returns : true if successful
//
bool CXMLNode::SetAttribute(const BSTR bstrName, const RECT& rVal)
{
	IXMLDOMElement* pEle = NULL;
	if (SUCCEEDED(m_pNode->QueryInterface(IID_IXMLDOMElement, (void**)&pEle)))
	{
		VARIANT v;
		v.vt = VT_I4;

		int nLen = wcslen(bstrName);
		wchar_t wszTemp[100];
		ASSERT(nLen+3 < sizeof(wszTemp) / sizeof(wchar_t));

		wcscpy(wszTemp, bstrName);
		wszTemp[nLen++] = L'_';
		wszTemp[nLen+1] = 0;

		// would like to normalize the rect before writing, but can't because
		// some objects (e.g. CIosObjLine) expect a de-normalized rect.
		// <RECT>_X
		wszTemp[nLen] = L'X';
		V_I4(&v) = rVal.left;				// V_I4(&v) = min(rVal.left, rVal.right);
		pEle->setAttribute(wszTemp, v);

		// <RECT>_Y
		wszTemp[nLen] = L'Y';
		V_I4(&v) = rVal.top;				// V_I4(&v) = min(rVal.top, rVal.bottom);
		pEle->setAttribute(wszTemp, v);

		// <RECT>_W
		wszTemp[nLen] = L'W';
		V_I4(&v) = rVal.right - rVal.left;	// V_I4(&v) = abs(rVal.right - rVal.left);
		pEle->setAttribute(wszTemp, v);

		// <RECT>_H
		wszTemp[nLen] = L'H';
		V_I4(&v) = rVal.bottom - rVal.top;	// V_I4(&v) = abs(rVal.bottom - rVal.top);
		pEle->setAttribute(wszTemp, v);

		SAFERELEASE(pEle);
		return true;
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////
// CXMLNodeList encapsulates the IXMLDOMNodeList interface pointer to manage
// that resource and provide convenience functions to the
// IXMLDOMNodeList interface
///////////////////////////////////////////////////////////////////////////

CXMLNodeList::CXMLNodeList(IXMLDOMNodeList *pNodeList) :
	m_pNodeList(pNodeList)
{
}

CXMLNodeList::CXMLNodeList(CXMLNode *pParent)
{
	m_pNodeList = 0;
	HRESULT hr = pParent->GetNode()->get_childNodes(&m_pNodeList);
}

CXMLNodeList::CXMLNodeList(CXMLNode *pParent, const BSTR bstrName)
{
	m_pNodeList = 0;
	HRESULT hr = pParent->GetNode()->selectNodes(bstrName, &m_pNodeList);
}

CXMLNodeList::~CXMLNodeList()
{
	SAFERELEASE(m_pNodeList);
}

int CXMLNodeList::Length(void)
{
	long lLen = 0L;
	HRESULT hr = m_pNodeList->get_length(&lLen);
	return SUCCEEDED(hr) ? lLen : 0L;
}
void CXMLNodeList::Reset(void)
{
	HRESULT hr = m_pNodeList->reset();
}
CXMLNode *CXMLNodeList::operator[](long nIndex)
{
    IXMLDOMNode * node = NULL;
	HRESULT hr = m_pNodeList->get_item(nIndex, &node);
	return SUCCEEDED(hr) ? new CXMLNode(node) : NULL;
}
CXMLNode *CXMLNodeList::NextNode(void)
{
    IXMLDOMNode * node = NULL;
	HRESULT hr = m_pNodeList->nextNode(&node);
	return SUCCEEDED(hr) ? new CXMLNode(node) : NULL;
}

///////////////////////////////////////////////////////////////////////////
// CXMLArchive encapsulates the IXMLDOMDocument interface pointer to manage
// that resource and provide convenience functions to the
// IXMLDOMDocument interface
///////////////////////////////////////////////////////////////////////////

CXMLArchive::CXMLArchive(const char *pszFilename, UINT nMode) :
	m_bsFilename(pszFilename),
	m_nMode(nMode),
	m_pDoc(0),
	m_pCurrParent(0),
	m_pDocument(0)
{
	CoInitialize(NULL);

	// Create an empty XML document
	HRESULT hr = CoCreateInstance(CLSID_DOMDocument, NULL, CLSCTX_INPROC_SERVER,
								IID_IXMLDOMDocument, (void**)&m_pDoc);

	if (m_pDoc)
	{
		m_pDoc->put_preserveWhiteSpace(true);
//		if (IsStoring())
//		{
//			// Create the root node
//			CXMLNode m_ndRoot;
//			CreateNode(NODE_ELEMENT, L"xxx", m_ndRoot);
//			AppendChild(&m_ndRoot);
//		}
	}
}

CXMLArchive::~CXMLArchive()
{
	SAFERELEASE(m_pDoc);

	CoUninitialize();
}

bool CXMLArchive::CreateNode(DOMNodeType type, const BSTR bstrName, CXMLNode& newNode)
{
	ASSERT(m_pDoc);
	ASSERT(IsStoring());
	if (!m_pDoc)
	{
		return false;
	}

	IXMLDOMNode * node = NULL;
	VARIANT vtype;

	vtype.vt = VT_I4;
	V_I4(&vtype) = (int)type;

	HRESULT hr = m_pDoc->createNode(vtype, bstrName, NULL, &node);

	newNode = node;
	return SUCCEEDED(hr);
}

CXMLNode * CXMLArchive::CreateNode(DOMNodeType type, const BSTR bstrName)
{
	CXMLNode *pNode = new CXMLNode();
	if (CreateNode(type, bstrName, *pNode))
	{
		return pNode;
	}
	else
	{
		delete pNode;
		return 0;
	}
#ifdef NOT_NOW
	ASSERT(m_pDoc);
	ASSERT(IsStoring());
	if (!m_pDoc)
	{
		return 0;
	}

	IXMLDOMNode * node = NULL;
	VARIANT vtype;

	vtype.vt = VT_I4;
	V_I4(&vtype) = (int)type;

	HRESULT hr = m_pDoc->createNode(vtype, bstrName, NULL, &node);

	return SUCCEEDED(hr) ? new CXMLNode(node) : 0;
#endif
}

bool CXMLArchive::SelectNode(CXMLNode& parent, const BSTR bstrName, CXMLNode& newNode)
{
	ASSERT(m_pDoc);
	if (!m_pDoc)
	{
		return false;
	}

	IXMLDOMNode* pNode = NULL;
	HRESULT hr;
	if (parent.GetNode() == NULL)
	{
		hr = m_pDoc->QueryInterface(IID_IXMLDOMNode,(void**)&pNode);
	}
	else
	{
		pNode = parent.GetNode();
	}

	IXMLDOMNode* pResult = NULL;

	hr = pNode->selectSingleNode(bstrName, &pResult);
	newNode = pResult;
	return SUCCEEDED(hr) && pResult;
}

CXMLNode * CXMLArchive::SelectNode(CXMLNode *pParent, const BSTR bstrName)
{
	ASSERT(m_pDoc);
	if (!m_pDoc)
	{
		return 0;
	}

	IXMLDOMNode* pNode = NULL;
	HRESULT hr;
	if (pParent == NULL)
	{
		hr = m_pDoc->QueryInterface(IID_IXMLDOMNode,(void**)&pNode);
	}
	else
	{
		pNode = pParent->GetNode();
	}

	IXMLDOMNode* pResult = NULL;

	hr = pNode->selectSingleNode(bstrName, &pResult);
	return (SUCCEEDED(hr) && pResult) ? new CXMLNode(pResult) : 0;
}

CXMLNode * CXMLArchive::SelectNextNode(CXMLNode *pParent)
{
	ASSERT(m_pDoc);
	ASSERT(pParent);
	// not implemented yet
	return 0;
}

bool CXMLArchive::InsertNewElement(CXMLNode& pParent, const BSTR bstrName, CXMLNode& newNode)
{
	ASSERT(m_pDoc);
	ASSERT(IsStoring());
	if (!m_pDoc)
	{
		return false;
	}

	if (CreateNode(NODE_ELEMENT, bstrName, newNode))
	{
		VARIANT after;
		after.vt = VT_EMPTY;
		IXMLDOMNode *p = NULL;
		HRESULT hr = pParent.GetNode()->insertBefore(newNode.GetNode(), after, &p);
		return SUCCEEDED(hr);
	}
	return false;
}

CXMLNode * CXMLArchive::InsertNewElement(CXMLNode *pParent, const BSTR bstrName)
{
	ASSERT(m_pDoc);
	ASSERT(pParent);
	ASSERT(IsStoring());
	if (!m_pDoc)
	{
		return false;
	}

	CXMLNode *pNode = CreateNode(NODE_ELEMENT, bstrName);
	if (pNode)
	{
		VARIANT after;
		after.vt = VT_EMPTY;
		IXMLDOMNode *p = NULL;
		HRESULT hr = pParent->GetNode()->insertBefore(pNode->GetNode(), after, &p);
	}
	return pNode;
}

CXMLNode *CXMLArchive::InsertNewTextElement(CXMLNode *pParent, const BSTR bstrName, const char *pszText)
{
	ASSERT(m_pDoc);
	ASSERT(pParent);
	ASSERT(IsStoring());
	if (!m_pDoc)
	{
		return false;
	}

	CXMLNode *pNode = InsertNewElement(pParent, bstrName);
	CXMLNode *pText = CreateNode(NODE_TEXT, L"");

	if (pText)
	{
		VARIANT v;
		v.vt = VT_BSTR;
		_bstr_t bs(pszText);
		V_BSTR(&v) = bs;
		pText->GetNode()->put_nodeValue(v);

		VARIANT after;
		after.vt = VT_EMPTY;
		IXMLDOMNode *p = NULL;
		HRESULT hr = pNode->GetNode()->insertBefore(pText->GetNode(), after, &p);
	}

	delete pText;	// discard local text - not accessible to caller

	return pNode;
}

bool CXMLArchive::InsertNewCData(CXMLNode *pParent, const char *pszData)
{
	ASSERT(m_pDoc);
	ASSERT(pParent);
	ASSERT(IsStoring());
	if (!m_pDoc)
	{
		return false;
	}

	CXMLNode *pNode = CreateNode(NODE_CDATA_SECTION, L"Code");

	if (pNode)
	{
		VARIANT v;
		v.vt = VT_BSTR;
		_bstr_t bs(pszData);
		V_BSTR(&v) = bs;
		pNode->GetNode()->put_nodeValue(v);

		VARIANT after;
		after.vt = VT_EMPTY;
		IXMLDOMNode *p = NULL;
		HRESULT hr = pParent->GetNode()->insertBefore(pNode->GetNode(), after, &p);
	}

	delete pNode;  // discard local node - not accessible to caller

	return true;
}

bool CXMLArchive::RetrieveCData(CXMLNode *pNode, _bstr_t& bsCData)
{
	ASSERT(m_pDoc);
	ASSERT(pNode);
	if (!m_pDoc)
	{
		return false;
	}

	IXMLDOMNode *pCDataNode = NULL;
	HRESULT hr = pNode->GetNode()->get_firstChild(&pCDataNode);

	if (SUCCEEDED(hr) && pCDataNode)
	{
		VARIANT v;
		if (SUCCEEDED(pCDataNode->get_nodeValue(&v)) && VT_NULL != v.vt)
		{
			bsCData = V_BSTR(&v);
		}
		pCDataNode->Release();
		return true;
	}
	return false;
}

bool CXMLArchive::LoadDocument(void)
{
	ASSERT(m_pDoc);
	if (!m_pDoc)
	{
		return false;
	}

	ASSERT(!IsStoring());

	IXMLDOMParseError  *pXMLError = NULL;
	VARIANT         vURL;
	VARIANT_BOOL    vb;
	HRESULT         hr;

	CHECKHR(m_pDoc->put_async(VARIANT_FALSE));

	if (m_bsFilename.length() <= 0)
	{
#ifdef IOS_RUNTIME
		((CIosrtApp*)AfxGetApp())->LogMessage("CXMLArchive Error: No filename provided.", IDB_ERROR);
#endif
		return false;
	}

	// Load xml document from the given URL or file path
	VariantInit(&vURL);
	vURL.vt = VT_BSTR;
	V_BSTR(&vURL) = m_bsFilename;
	CHECKHR(m_pDoc->load(vURL, &vb));

	CHECKHR(CheckLoad());
                
CleanUp:
	SAFERELEASE(pXMLError);

	return SUCCEEDED(hr);
}

bool CXMLArchive::SaveDocument(void)
{
	ASSERT(m_pDoc);
	if (!m_pDoc)
	{
		return false;
	}

	ASSERT(IsStoring());
#ifdef NOT_NOW
	VARIANT vName;

	vName.vt = VT_BSTR;
	V_BSTR(&vName) = m_bsFilename;
	HRESULT hr = m_pDoc->save(vName);
#endif

	FILE *pFile = fopen(m_bsFilename, "w+b");
	if (pFile)
	{
		fprintf(pFile, "<?xml version=\"1.0\"?>\n");
		fprintf(pFile, "<!-- FlightSafety International XML File -->\n");
		CTime t = CTime::GetCurrentTime();
//		fprintf(pFile, t.FormatGmt("<!-- GMT Date: %x  Time: %X -->"));
		fprintf(pFile, t.FormatGmt("<!-- Last modified - Date: %Y-%b-%d  Time: %H:%M:%S (GMT) -->"));

		IXMLDOMNode* firstNode;
		GetDoc()->get_firstChild(&firstNode);
		bool bTextChild;
		DumpFormattedTree(pFile, firstNode, 0, bTextChild);
		firstNode->Release();

		fclose(pFile);
		return true;
	}
	return false;
}

HRESULT CXMLArchive::CheckLoad(void)
{
	// And since we don't have the VARIANT_BOOL from load we
	// need to check the parse Error errorCode property to see
	// if everything went ok.
	IXMLDOMParseError  *pXMLError = NULL;
	LONG errorCode = E_FAIL;
	HRESULT hr;

	CHECKHR(m_pDoc->get_parseError(&pXMLError));
	CHECKHR(pXMLError->get_errorCode(&errorCode));
	if (errorCode != 0)
	{
		hr = ReportError(pXMLError);
		goto CleanUp;
	}

CleanUp:
	SAFERELEASE(pXMLError);
	return errorCode;
}


HRESULT CXMLArchive::ReportError(IXMLDOMParseError *pXMLError)
{
	long line, linePos;
	LONG errorCode;
	BSTR pBURL, pBReason;
	HRESULT hr;
#ifdef IOS_RUNTIME
	char szMsg[1000];
#endif

	CHECKHR(pXMLError->get_line(&line));
	CHECKHR(pXMLError->get_linepos(&linePos));
	CHECKHR(pXMLError->get_errorCode(&errorCode));
	CHECKHR(pXMLError->get_url(&pBURL));
	CHECKHR(pXMLError->get_reason(&pBReason));

#ifdef IOS_RUNTIME
	sprintf(szMsg, "CXMLArchive Error: %S", pBReason);
	((CIosrtApp*)AfxGetApp())->LogMessage(szMsg, IDB_ERROR);
#endif
	TRACE1("%S", pBReason);
	if (line > 0)
	{
#ifdef IOS_RUNTIME
		sprintf(szMsg, "Error on line %d, position %d in \"%S\".", line, linePos, pBURL);
		((CIosrtApp*)AfxGetApp())->LogMessage(szMsg, IDB_ERROR);
#endif
		TRACE3("Error on line %d, position %d in \"%S\".\n", line, linePos, pBURL);
	}

CleanUp:
	SysFreeString(pBURL);
	SysFreeString(pBReason);

	return E_FAIL;
}


bool CXMLArchive::AppendChild(CXMLNode * pChild)
{
	HRESULT hr = m_pDoc->appendChild(pChild->GetNode(), NULL);
	return SUCCEEDED(hr);
}



///////////////////////////////////////////////////////////////////////////
// This set of functions provides for 'serialized' IO of XML Nodes.
// See the CNamedData<T> template class for access to these.
// These functions are intended to parallel the MFC CArchive serialization
// functions.  As such, the output operators create XML Nodes in the XML
// DOM hierarchy and populate them with data.  The input operators locate
// XML Nodes by name and retrieve their associated data.
//
// NOTE: if you use the CNamedData<T> template with a new data type, you'll
// have to implement the pair of IO functions (operator<< and operator>>).
///////////////////////////////////////////////////////////////////////////

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<double> & namedData)
{
	ASSERT(rArchive.IsStoring());
	char szTemp[100];
	sprintf(szTemp, "%f", namedData.m_tVal);
	CXMLNode *pNode = rArchive.InsertNewTextElement(namedData.m_pParent, namedData.m_bsName, szTemp);
	delete pNode;

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<double> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	CXMLNode *pNode = rArchive.SelectNode(namedData.m_pParent, namedData.m_bsName);
	if (pNode)
	{
		BSTR bs;
		pNode->GetNode()->get_text(&bs);

		VARIANT v;
		v.vt = VT_BSTR;
		V_BSTR(&v) = bs;
		HRESULT hr = VariantChangeType(&v, &v, 0, VT_R8);
		namedData.m_tVal = V_R8(&v);

		delete pNode;
	}

	return rArchive;
}


CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<bool> & namedData)
{
	ASSERT(rArchive.IsStoring());
	CXMLNode *pNode = rArchive.InsertNewTextElement(namedData.m_pParent, namedData.m_bsName, namedData.m_tVal ? "True" : "False");
	delete pNode;

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<bool> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	CXMLNode *pNode = rArchive.SelectNode(namedData.m_pParent, namedData.m_bsName);
	if (pNode)
	{
		BSTR bs;
		pNode->GetNode()->get_text(&bs);
		_bstr_t bsTemp(bs);
		namedData.m_tVal = (_bstr_t("True") == bsTemp || _bstr_t("1") == bsTemp);
		delete pNode;
	}

	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<int> & namedData)
{
	ASSERT(rArchive.IsStoring());
	char szTemp[100];
	sprintf(szTemp, "%d", namedData.m_tVal);
	CXMLNode *pNode = rArchive.InsertNewTextElement(namedData.m_pParent, namedData.m_bsName, szTemp);
	delete pNode;

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<int> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	CXMLNode *pNode = rArchive.SelectNode(namedData.m_pParent, namedData.m_bsName);
	if (pNode)
	{
		BSTR bs;
		pNode->GetNode()->get_text(&bs);

		VARIANT v;
		v.vt = VT_BSTR;
		V_BSTR(&v) = bs;
		HRESULT hr = VariantChangeType(&v, &v, 0, VT_I4);
		namedData.m_tVal = V_I4(&v);

		delete pNode;
	}

	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<unsigned int> & namedData)
{
	ASSERT(rArchive.IsStoring());
	int n = namedData.m_tVal;
	return rArchive << CNamedData<int>(*namedData.m_pParent, namedData.m_bsName, n);
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<unsigned int> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	int n;
	CXMLArchive& r = rArchive >> CNamedData<int>(*namedData.m_pParent, namedData.m_bsName, n);
	namedData.m_tVal = n;
	return r;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<std::string> & namedData)
{
	ASSERT(rArchive.IsStoring());
	CXMLNode *pNode = rArchive.InsertNewTextElement(namedData.m_pParent, namedData.m_bsName, namedData.m_tVal.c_str());
	delete pNode;

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<std::string> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	CXMLNode *pNode = rArchive.SelectNode(namedData.m_pParent, namedData.m_bsName);
	if (pNode)
	{
		BSTR bs;
		pNode->GetNode()->get_text(&bs);
		namedData.m_tVal = (const char *)_bstr_t(bs);
		delete pNode;
	}

	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<COLORREF> & namedData)
{
	ASSERT(rArchive.IsStoring());
	char szTemp[10];
	sprintf(szTemp, "#%02X%02X%02X",
					GetRValue(namedData.m_tVal),
					GetGValue(namedData.m_tVal),
					GetBValue(namedData.m_tVal));
	CXMLNode *pNode = rArchive.InsertNewTextElement(namedData.m_pParent, namedData.m_bsName, szTemp);
// enable this code for RGB values as attributes
//	if (pNode)
//	{
//		pNode->SetAttribute(namedData.m_tVal);
//	}
	delete pNode;

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<COLORREF> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	CXMLNode *pNode = rArchive.SelectNode(namedData.m_pParent, namedData.m_bsName);
	if (pNode)
	{
		BSTR bs;
		pNode->GetNode()->get_text(&bs);

		// sanity check
		ASSERT(	bs[1] >= '0' && bs[1] <= 'f' &&
				bs[2] >= '0' && bs[2] <= 'f' &&
				bs[3] >= '0' && bs[3] <= 'f' &&
				bs[4] >= '0' && bs[4] <= 'f' &&
				bs[5] >= '0' && bs[5] <= 'f' &&
				bs[6] >= '0' && bs[6] <= 'f');

		namedData.m_tVal =	(tblHexChars[bs[1]] <<  4) + (tblHexChars[bs[2]] <<  0) +
							(tblHexChars[bs[3]] << 12) + (tblHexChars[bs[4]] <<  8) +
							(tblHexChars[bs[5]] << 20) + (tblHexChars[bs[6]] << 16);

		delete pNode;
	}

// enable this code for RGB values as attributes
//	CXMLNode *pNode = rArchive.SelectNode(namedData.m_pParent, namedData.m_bsName);
//	if (pNode)
//	{
//		pNode->GetAttribute(namedData.m_tVal);
//		delete pNode;
//	}
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<RECT> & namedData)
{
	ASSERT(rArchive.IsStoring());
	CXMLNode ndRect;
	if (rArchive.InsertNewElement(*namedData.m_pParent, namedData.m_bsName, ndRect))
	{
		ndRect.SetAttribute(namedData.m_bsName, namedData.m_tVal);
	}

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<RECT> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	CXMLNode ndRect;
	if (rArchive.SelectNode(*namedData.m_pParent, namedData.m_bsName, ndRect))
	{
		ndRect.GetAttribute(namedData.m_bsName, namedData.m_tVal);
	}

	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedData<LOGFONT> & namedData)
{
	ASSERT(rArchive.IsStoring());
	CXMLNode ndFont;
	if (rArchive.InsertNewElement(*namedData.m_pParent, namedData.m_bsName, ndFont))
	{
		ndFont.SetAttribute(L"Height",         namedData.m_tVal.lfHeight);
		ndFont.SetAttribute(L"Width",          namedData.m_tVal.lfWidth);
		ndFont.SetAttribute(L"Escapement",     namedData.m_tVal.lfEscapement);
		ndFont.SetAttribute(L"Orientation",    namedData.m_tVal.lfOrientation);
		ndFont.SetAttribute(L"Weight",         namedData.m_tVal.lfWeight);
		ndFont.SetAttribute(L"Italic",         namedData.m_tVal.lfItalic);
		ndFont.SetAttribute(L"Underline",      namedData.m_tVal.lfUnderline);
		ndFont.SetAttribute(L"Strikeout",      namedData.m_tVal.lfStrikeOut);
		ndFont.SetAttribute(L"CharSet",        namedData.m_tVal.lfCharSet);
		ndFont.SetAttribute(L"OutPrecision",   namedData.m_tVal.lfOutPrecision);
		ndFont.SetAttribute(L"ClipPrecision",  namedData.m_tVal.lfClipPrecision);
		ndFont.SetAttribute(L"Quality",        namedData.m_tVal.lfQuality);
		ndFont.SetAttribute(L"PitchAndFamily", namedData.m_tVal.lfPitchAndFamily);
		ndFont.SetAttribute(L"FaceName",       namedData.m_tVal.lfFaceName);
	}

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedData<LOGFONT> & namedData)
{
	ASSERT(!rArchive.IsStoring());
	CXMLNode ndFont;
	if (rArchive.SelectNode(*namedData.m_pParent, namedData.m_bsName, ndFont))
	{
		ndFont.GetAttribute(L"Height",         namedData.m_tVal.lfHeight);
		ndFont.GetAttribute(L"Width",          namedData.m_tVal.lfWidth);
		ndFont.GetAttribute(L"Escapement",     namedData.m_tVal.lfEscapement);
		ndFont.GetAttribute(L"Orientation",    namedData.m_tVal.lfOrientation);
		ndFont.GetAttribute(L"Weight",         namedData.m_tVal.lfWeight);
		ndFont.GetAttribute(L"Italic",         namedData.m_tVal.lfItalic);
		ndFont.GetAttribute(L"Underline",      namedData.m_tVal.lfUnderline);
		ndFont.GetAttribute(L"Strikeout",      namedData.m_tVal.lfStrikeOut);
		ndFont.GetAttribute(L"CharSet",        namedData.m_tVal.lfCharSet);
		ndFont.GetAttribute(L"OutPrecision",   namedData.m_tVal.lfOutPrecision);
		ndFont.GetAttribute(L"ClipPrecision",  namedData.m_tVal.lfClipPrecision);
		ndFont.GetAttribute(L"Quality",        namedData.m_tVal.lfQuality);
		ndFont.GetAttribute(L"PitchAndFamily", namedData.m_tVal.lfPitchAndFamily);
		_bstr_t bsTemp;
		if (ndFont.GetAttribute(L"FaceName", bsTemp))
		{
			int nLen = sizeof(namedData.m_tVal.lfFaceName)-1;
			strncpy(namedData.m_tVal.lfFaceName, (const char *)bsTemp, nLen);
			namedData.m_tVal.lfFaceName[nLen] = '\0';
		}
	}

	return rArchive;
}

///////////////////////////////////////////////////////////////////////////
// This set of functions provides for 'serialized' IO of XML Attributes.
// See the CNamedAttribute<T> template class for access to these.
// Note that these don't actually require access to the CXMLArchive,
// but implementing them on the CXMLArchive led to more consistent and
// readable code for those already familiar with the MFC CArchive class
//
// NOTE: if you use the CNamedAttribute<T> template with a new data type, you'll
// have to implement the pair of IO functions (operator<< and operator>>).
///////////////////////////////////////////////////////////////////////////

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<const int> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, (long)namedAttr.m_tVal);
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<const char *> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, namedAttr.m_tVal);
	return rArchive;
}


CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<int> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, (long)namedAttr.m_tVal);
	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<int> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	long l;
	if (namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, l))
	{
		namedAttr.m_tVal = l;
	}
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<unsigned int> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, (long)namedAttr.m_tVal);
	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<unsigned int> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	long l;
	if (namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, l))
	{
		namedAttr.m_tVal = l;
	}
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<long> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, namedAttr.m_tVal);
	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<long> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, namedAttr.m_tVal);
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<COLORREF> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, namedAttr.m_tVal);
	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<COLORREF> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, namedAttr.m_tVal);
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<RECT> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, namedAttr.m_tVal);

	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<RECT> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, namedAttr.m_tVal);

	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<std::string> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, namedAttr.m_tVal.c_str());
	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<std::string> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	VARIANT v;
	if (namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, v) && VT_NULL != v.vt)
	{
		namedAttr.m_tVal = _bstr_t(V_BSTR(&v));
	}
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<bool> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, namedAttr.m_tVal ? "True" : "False");
	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<bool> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	_bstr_t bsTemp;
	if (namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, bsTemp))
	{
		namedAttr.m_tVal = (_bstr_t("True") == bsTemp || _bstr_t("1") == bsTemp);
	}
	return rArchive;
}

CXMLArchive& operator<<(CXMLArchive& rArchive, const CNamedAttribute<double> & namedAttr)
{
	ASSERT(rArchive.IsStoring());
	char szTemp[100];
	sprintf(szTemp, "%.8g", namedAttr.m_tVal);
	namedAttr.m_pParent->SetAttribute(namedAttr.m_bsName, szTemp);
	return rArchive;
}

CXMLArchive& operator>>(CXMLArchive& rArchive, CNamedAttribute<double> & namedAttr)
{
	ASSERT(!rArchive.IsStoring());
	_bstr_t bsTemp;
	if (namedAttr.m_pParent->GetAttribute(namedAttr.m_bsName, bsTemp))
	{
		namedAttr.m_tVal = atof(bsTemp);
	}
	return rArchive;
}


//
// DumpXMLString
// Convert special XML characters to legal escape sequences:
//		&	= &amp;
//		<	= &lt;
//		>	= &gt;
//		'	= &apos;
//		"	= &quot;
// Entry   : file pointer and the string that may need converting
// Returns : nothing
//
static void DumpXMLString(FILE *pFile, const BSTR bstr)
{
	_bstr_t bs(bstr, false);

	// automatically convert any linked *.ios files to .xml
	int nLen = bs.length();
	char *pStr = (char *)bs;
	if (stricmp(pStr + nLen - 4, ".ios") == 0)
	{
		strcpy(pStr + nLen - 4, ".xml");
	}

	// escape special characters before writing
	for (char *pIn = bs; *pIn; pIn++)
	{
		switch(*pIn)
		{
		case '&':	fputs("&amp;",  pFile);	break;
		case '<':	fputs("&lt;",   pFile);	break;
		case '>':	fputs("&gt;",   pFile);	break;
		case '"':	fputs("&quot;", pFile);	break;
		case '\'':	fputs("&apos;", pFile);	break;
		default:	fputc(*pIn,     pFile);	break;
		}
	}
}

//
// DumpFormattedTree
// Utility function to format and dump an XML tree to an ASCII file.
// Applies tags, linefeeds and indentation for a more readable format than
// the single-line output of the IXMLDOMDocument::save() function.
// Entry   : 
// Returns : nothing
//
static void DumpFormattedTree(FILE *pFile, IXMLDOMNode* node, int level, bool& bTextChild)
{
	bTextChild = false;
    IXMLDOMNode* pChild;
	IXMLDOMNode* pNext;
    BSTR nodeName;
    IXMLDOMNamedNodeMap* pattrs;

    node->get_nodeName(&nodeName);
	_bstr_t bsTemp(nodeName);
	if (stricmp("#text", bsTemp) == 0)
	{
		// for #text nodes, we'll dump the text contents
		BSTR bs;
		node->get_text(&bs);
		fprintf(pFile, "%S", bs);
		SysFreeString(bs);
		bTextChild = true;
		return;
	}
	else if (stricmp("#cdata-section", bsTemp) == 0)
	{
		// for #cdata nodes, we'll dump the node 'value'
		VARIANT value;
		node->get_nodeValue(&value);
		fprintf(pFile, "<![CDATA[%S]]>", V_BSTR(&value));
        VariantClear(&value);
		return;
	}

	// for all other node types, dump the node name and continue
	fprintf(pFile, "\r\n");
    for (int i = 0; i < level; i++)
	{
		fprintf(pFile, "   ");
	}
	fprintf(pFile, "<%S",nodeName);

	// dump all of the attributes for this node
    if (SUCCEEDED(node->get_attributes(&pattrs)) && pattrs != NULL)
    {
        pattrs->nextNode(&pChild);
        while (pChild)
        {
			// attribute name
            BSTR name;
            pChild->get_nodeName(&name);
			fprintf(pFile, " %S=\"", name);
            ::SysFreeString(name);

			// attribute value
            VARIANT value;
            pChild->get_nodeValue(&value);
            if (value.vt == VT_BSTR)
            {
				DumpXMLString(pFile, V_BSTR(&value));
            }
			fprintf(pFile, "\"");
            VariantClear(&value);
            pChild->Release();
            pattrs->nextNode(&pChild);
        }
        pattrs->Release();
    }

	// dump all of the child nodes (recursively)
	bool bHasChildren = false;
    node->get_firstChild(&pChild);
	if (pChild)
	{
		bHasChildren = true;
		fprintf(pFile, ">");
	}
    while (pChild)
    {
        DumpFormattedTree(pFile, pChild, level+1, bTextChild);
        pChild->get_nextSibling(&pNext);
        pChild->Release();
        pChild = pNext;
    }

	// if there were child nodes, dump the closing tag
	if (bHasChildren)
	{
		// for children other than simple text, apply indentation
		if (!bTextChild)
		{
			fprintf(pFile, "\r\n");
			for (i = 0; i < level; i++)
			{
				fprintf(pFile, "   ");
			}
		}
		fprintf(pFile, "</%S>",nodeName);
	}
	else
	{
		fprintf(pFile, "/>");
	}
    SysFreeString(nodeName);

	// one last linefeed at the end
	if (0 == level)
	{
		fprintf(pFile, "\r\n");
	}

    return;
}
