/////////////////////////////////////////////////////////////////////////////
//
//           F L I G H T S A F E T Y   I N T E R N A T I O N A L
//                     Simulation Systems Division
//                      2700 North Hemlock Circle
//                     Broken Arrow, Oklahoma 74012
//                          (918) 259-4000
/////////////////////////////////////////////////////////////////////////////
//
// DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
// Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
// technical or operational data or information from automatic dissemination
// under the International Exchange Program or by other means.  This protection
// covers information required solely for administrative or operational
// purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
//
// WARNING:  This document contains technical data whose export is restricted
// by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
// Executive Order 12470.  Violation of these export control laws is subject
// to severe criminal penalties.  Dissemination of this document is controlled
// under DOD Directive 5230.25
//
/////////////////////////////////////////////////////////////////////////////
//
//
// Filename         : weatherControl.cpp
//
// Date             : 30 March 2000
//
// Engineer         : Michael F Lawson
//
// Revision         : $
//
// Description      : Controls the simulator weather based on the pre-loaded
//                    sortie data.  This exists in the radiodatabase project
//                    because it needs to look up lat/longs in the radio
//                    database.
//
// Classification   : UNCLASSIFIED
//
// Requirements     : None.
//
// Components Used  : 

// Environment      : Operating system(s) - Microsoft Windows NT 4.0 with
//                                              NT service pack 3, 4, or 5
//                                          Microsoft Windows NT 2000
//
//                    Compiler(s) - Visual C++ 6.0
//
//                    Architechure(s) - Intel Pentium, Pentium II
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
//
//                              R e v i s i o n   H i s t o r y
//
/////////////////////////////////////////////////////////////////////////////
//
//
/////////////////////////////////////////////////////////////////////////////


#include "..\core\stdafx.h"
#include "WeatherControl.h"
#include "..\comms\CommsShared.h"
#include "CommUtils.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

// These define some of the host variable names to use.  I don't like this,
// because it seems like there should be a better place to put them, other than
// hard-coding the names here.

const _FSI_STL::string lat_host_name = "ac_latitude";
const _FSI_STL::string lon_host_name = "ac_longitude";
const _FSI_STL::string wx_mode_name  = "wx_mode";
const _FSI_STL::string wx_load_name  = "wx_load_cmd";

enum  {AUTO_MODE=0, MANUAL_MODE=1, WX_OFF_MODE=2};


// Define some other static class variables... Yes, I made them all
// static so that we wouldn't actually need to create an instance of
// CWeatherControl and have to worry about multiple weather controls
// being active at once.

int          CWeatherControl::m_nHostRate = CCommUtils::HostRateFromRegistry();
double       CWeatherControl::m_dLatitude;
double       CWeatherControl::m_dLongitude;
CWeatherArea CWeatherControl::m_area[m_num_wx_areas];
int          CWeatherControl::m_ActiveArea = 0;
long         CWeatherControl::m_last_wx_mode = MANUAL_MODE;
long         CWeatherControl::m_wx_mode = MANUAL_MODE;
bool         CWeatherControl::m_wxOff = false;
int          CWeatherControl::m_wxApplyTimer = 0;
bool         CWeatherControl::m_isAnyActive = false;
bool         CWeatherControl::m_wx_loadCmd = false;


// Update is called every time new data arrives from the host.  This function
// address is registered with a call to CCommsShared::AddUpdateCallback in the
// radiodatabasecomponentinterface.

void CWeatherControl::Update()
{
   static int one_sec_timer = 0;
   
   // Check to see if we're delaying before sending the Apply command to the
   // host.  If the timer reaches zero, then send the Apply.
   if (m_wxApplyTimer > 0)
   {
      m_wxApplyTimer--;
      if (m_wxApplyTimer == 0)
      {
            CCommUtils::SetVariable( HostComms, "wx_apply",     true );
            CCommUtils::SetVariable( HostComms, "wx_apply_flt", true );
      }
   }

   // Update only once every couple of seconds so we don't increase the CPU's
   // load very much.

   if ((++one_sec_timer)%(m_nHostRate*2))
      return;

   // I'm going to save these pointers to the CVariants so I won't have to do
   // lookups by variable name each time.  Note: this makes the assumption
   // that the pointers for a particular variant remain constant after they're
   // looked-up.

   static CVariant *pVarLatitude  = NULL,
                   *pVarLongitude = NULL,
                   *pVarWxMode    = NULL,
                   *pLoadCommand  = NULL;

   
   // Look up current ownship latitude, saving pointer for future reference.

   if (pVarLatitude != NULL) 
         m_dLatitude = (double)(*pVarLatitude);
   else
   {
     pVarLatitude = CCommsShared::FindVariant(lat_host_name);
     ASSERT(pVarLatitude != NULL);
     if (pVarLatitude != NULL)
         m_dLatitude = (double)(*pVarLatitude);
   }


   // Look up current ownship longitude, saving pointer for future reference.

   if (pVarLongitude != NULL) 
         m_dLongitude = (double)(*pVarLongitude);
   else
   {
     pVarLongitude = CCommsShared::FindVariant(lon_host_name);
     ASSERT(pVarLongitude != NULL);
     if (pVarLongitude != NULL)
         m_dLongitude = (double)(*pVarLongitude);
   }


   // Look up current weather control mode, saving pointer for future reference.
   
   if (pVarWxMode != NULL)
   {
       m_wx_mode = (long)(*pVarWxMode);
       if (m_wx_mode == 0)
           CCommUtils::SetVariable( HostComms, "weather_manual", false );
       else
           CCommUtils::SetVariable( HostComms, "weather_manual", true );
   }
   else
   {
      pVarWxMode = CCommsShared::FindLocalVariant(wx_mode_name);
      ASSERT(pVarWxMode != NULL);
      if (pVarWxMode != NULL)
      {
          m_wx_mode = (long)(*pVarWxMode);
          if (m_wx_mode == 0)
              CCommUtils::SetVariable( HostComms, "weather_manual", false );
          else
              CCommUtils::SetVariable( HostComms, "weather_manual", true );
      }
   }


   // Look up current sortie load command boolean, saving pointer for future
   
   if (pLoadCommand != NULL) 
         m_wx_loadCmd = (bool)(*pLoadCommand);
   else
   {
     pLoadCommand = CCommsShared::FindLocalVariant(wx_load_name);
     ASSERT(pLoadCommand != NULL);
     if (pLoadCommand != NULL)
         m_wx_loadCmd = (bool)(*pLoadCommand);
   } 
 
   bool areaInfoChanged = false;

   // Check to see if a sortie load is happening.  If so, go do it, using
   // local comms variables, and reset the flag to false.

   if (m_wx_loadCmd == true)
   { 
      for (int i=0; i<m_num_wx_areas; i++)
         m_area[i].Deactivate();

      LoadSortieAreaData();

      CCommUtils::SetVariable( LocalComms, (char *)wx_load_name.c_str(), false);
      m_wx_loadCmd = false;
	  areaInfoChanged = true;
   }


   // For each of the defined weather areas, call their Update method, so they
   // can calculate if the Ownship is within their area.  If it IS in the area,
   // mark that area as the active area, and mark the others as inactive.

   for (int i=0; i < m_num_wx_areas; i++)
   {
      m_area[i].Update(m_dLatitude, m_dLongitude);
      
      if (m_area[i].isNowInRange())
      {
         for (int j=0; j < m_num_wx_areas; j++)
             m_area[j].m_isActive = false;

         m_ActiveArea = i;
         m_isAnyActive = true;

         if (m_wx_mode == AUTO_MODE)
         {
           m_area[m_ActiveArea].m_isActive = true;
           SendNewWeatherToHost();
         }
      }

      if (m_area[i].hasChanged()) 
         areaInfoChanged = true;
   }

   if (areaInfoChanged) 
   {
      SendAreaInfoToMap();
      SendATISInformation();
   }


   // Test here for weather control logic.  This section tests if we are
   // changing from either Off mode or Manual Mode, to Auto Mode.

   if (m_wx_mode      == AUTO_MODE && 
       m_last_wx_mode != AUTO_MODE    )
   {
      CCommUtils::SetVariable( HostComms, "wx_manual", false);
      SetApplyTimer(0.5f);

      if (m_last_wx_mode == WX_OFF_MODE)
      {
         CCommUtils::SetVariable( HostComms, "wx_off", false);
		   //SetApplyTimer(0.5f);
      } /*
      else if (m_last_wx_mode == MANUAL_MODE)
      {
         // Nothing special needed for this case
      } */
      if (m_isAnyActive == false)
      {
         // Check for the case where we're already sitting inside an area but
         // haven't actually entered it yet.  This is the case if the sim
         // begins in MANUAL_MODE, but you're inside an airport, and change
         // the mode to AUTO_MODE
         for (int j=0; j < m_num_wx_areas; j++)
            if (m_area[j].m_isInRange)
            {
               m_ActiveArea = j;
               m_isAnyActive = true;
            }
      }

      if (m_isAnyActive == true)
      {
         m_area[m_ActiveArea].m_isActive = true;
         SendNewWeatherToHost();
      }
   }

   // Check to see if we're going to OFF mode from either Auto mode or Manaual
   // mode.

   else if (m_wx_mode      == MANUAL_MODE &&
            m_last_wx_mode != MANUAL_MODE   )
   {
      CCommUtils::SetVariable( HostComms, "wx_manual", true);
		SetApplyTimer(0.5f);

      if (m_last_wx_mode == WX_OFF_MODE)
      {
          CCommUtils::SetVariable( HostComms, "wx_off", false);
		    //SetApplyTimer(0.5f);
      } /*
      else if (m_last_wx_mode == AUTO_MODE)
      {
         // nothing special needed for this case
      } */

      // Turn off active weather areas
      for (int j=0; j < m_num_wx_areas; j++)
         m_area[j].m_isActive = false;

   }

   else if (m_wx_mode      == WX_OFF_MODE &&
            m_last_wx_mode != WX_OFF_MODE    )
   {
     // Both cases use this code
     CCommUtils::SetVariable( HostComms, "wx_manual", false);
     CCommUtils::SetVariable( HostComms, "wx_off", true);
	  SetApplyTimer(0.5f);

     // Turn off active weather areas
     for (int j=0; j < m_num_wx_areas; j++)
        m_area[j].m_isActive = false;
   }

   m_last_wx_mode = m_wx_mode;
}


// SendAreaInfoToMap() sends the details of the weather areas via local comms
// whenever they change due to becoming active, etc.

void CWeatherControl::SendAreaInfoToMap()
{
   CCommUtils::SetVariable( LocalComms, "wx_area1_latitude_center",  m_area[0].m_area_latitude  );
   CCommUtils::SetVariable( LocalComms, "wx_area1_longitude_center", m_area[0].m_area_longitude );
   CCommUtils::SetVariable( LocalComms, "wx_area1_radius",           m_area[0].m_wx_radius      );
   CCommUtils::SetVariable( LocalComms, "wx_area1_loaded",           m_area[0].m_isLoaded       );
   CCommUtils::SetVariable( LocalComms, "wx_area1_active",           m_area[0].m_isActive       );

   CCommUtils::SetVariable( LocalComms, "wx_area2_latitude_center",  m_area[1].m_area_latitude  );
   CCommUtils::SetVariable( LocalComms, "wx_area2_longitude_center", m_area[1].m_area_longitude );
   CCommUtils::SetVariable( LocalComms, "wx_area2_radius",           m_area[1].m_wx_radius      );
   CCommUtils::SetVariable( LocalComms, "wx_area2_loaded",           m_area[1].m_isLoaded       );
   CCommUtils::SetVariable( LocalComms, "wx_area2_active",           m_area[1].m_isActive       );

   CCommUtils::SetVariable( LocalComms, "wx_area3_latitude_center",  m_area[2].m_area_latitude  );
   CCommUtils::SetVariable( LocalComms, "wx_area3_longitude_center", m_area[2].m_area_longitude );
   CCommUtils::SetVariable( LocalComms, "wx_area3_radius",           m_area[2].m_wx_radius      );
   CCommUtils::SetVariable( LocalComms, "wx_area3_loaded",           m_area[2].m_isLoaded       );
   CCommUtils::SetVariable( LocalComms, "wx_area3_active",           m_area[2].m_isActive       );

   CCommUtils::SetVariable( LocalComms, "wx_area4_latitude_center",  m_area[3].m_area_latitude  );
   CCommUtils::SetVariable( LocalComms, "wx_area4_longitude_center", m_area[3].m_area_longitude );
   CCommUtils::SetVariable( LocalComms, "wx_area4_radius",           m_area[3].m_wx_radius      );
   CCommUtils::SetVariable( LocalComms, "wx_area4_loaded",           m_area[3].m_isLoaded       );
   CCommUtils::SetVariable( LocalComms, "wx_area4_active",           m_area[3].m_isActive       );
}


// Okay, we finally have new data to send to the host, so let's do it.
// This is another area which should probably be done differently, since
// there are hard-coded host labels in here.  Maybe we need a list of
// variables to send or something clever like that someday.  Note that
// the "Apply" command must be sent towards the end...

void CWeatherControl::SendNewWeatherToHost()
{
   ASSERT(m_ActiveArea >= 0);
   ASSERT(m_ActiveArea <  m_num_wx_areas);

   // Get a shorthand pointer for readability
   CWeatherArea *pArea = &m_area[m_ActiveArea];
   
   // Change active airport & active runway
   CCommUtils::SetVariable( HostComms, "Reposition_Airport",        pArea-> m_airport, -1            );
   CCommUtils::SetVariable( HostComms, "Reposition_Runway",         pArea-> m_runway,  -1            );
   CCommUtils::SetVariable( HostComms, "Reposition_Environment_Change", true                         );

   CCommUtils::SetVariable( HostComms, "wx_field_press",            pArea-> m_field_pressure_inHg    );
   CCommUtils::SetVariable( HostComms, "wx_field_temp",             pArea-> m_field_temperature_f    );
   CCommUtils::SetVariable( HostComms, "wx_rvrFt",                  pArea-> m_RVR_ft                 );
   CCommUtils::SetVariable( HostComms, "wx_visibSm",                pArea-> m_visibility_sm          );
   CCommUtils::SetVariable( HostComms, "wx_timeOfDay",              pArea-> m_visual_scene           );
   CCommUtils::SetVariable( HostComms, "wx_cloud2Top",              pArea-> m_cloud_2_top            );
   CCommUtils::SetVariable( HostComms, "wx_cloud2Bottom",           pArea-> m_cloud_2_bottom         );
   CCommUtils::SetVariable( HostComms, "wx_cloud2Type",             pArea-> m_cloud_2_type           );
   CCommUtils::SetVariable( HostComms, "wx_cloud1Top",              pArea-> m_cloud_1_top            );
   CCommUtils::SetVariable( HostComms, "wx_cloud1Bottom",           pArea-> m_cloud_1_bottom         );
   CCommUtils::SetVariable( HostComms, "wx_cloud1Type",             pArea-> m_cloud_1_type           );
   CCommUtils::SetVariable( HostComms, "wx_cirrus",                 pArea-> m_cirrus                 );
   CCommUtils::SetVariable( HostComms, "wx_scud",                   pArea-> m_scud                   );
   CCommUtils::SetVariable( HostComms, "wx_fogType",                pArea-> m_fog_type               );
   CCommUtils::SetVariable( HostComms, "wx_fogTop",                 pArea-> m_fog_top                );
   CCommUtils::SetVariable( HostComms, "wx_sfc_wind_dir",           pArea-> m_surface_wind_direction );
   CCommUtils::SetVariable( HostComms, "wx_sfc_wind_spd",           pArea-> m_surface_wind_speed     );
   CCommUtils::SetVariable( HostComms, "wx_gust_spd",               pArea-> m_wind_gust_speed        );
   CCommUtils::SetVariable( HostComms, "wx_gust_var",               pArea-> m_wind_gust_variation    );
   CCommUtils::SetVariable( HostComms, "wx_precipType",             pArea-> m_precip_type            );
   CCommUtils::SetVariable( HostComms, "wx_precipIntensity",        pArea-> m_precip_intensity       );
   CCommUtils::SetVariable( HostComms, "wx_lightning",              pArea-> m_lightning              );
   CCommUtils::SetVariable( HostComms, "wx_hail",                   pArea-> m_hail                   );
   CCommUtils::SetVariable( HostComms, "wx_RCR",                    pArea-> m_RCR                    );
   CCommUtils::SetVariable( HostComms, "wx_contamination",          pArea-> m_runway_condition       );
   CCommUtils::SetVariable( HostComms, "vis_blowing_contamination", pArea-> m_blowing_contamination  );
   CCommUtils::SetVariable( HostComms, "vis_aptl_skidmarks",        pArea-> m_wet_skidmarks          );
   CCommUtils::SetVariable( HostComms, "wx_ice_quantity",           pArea-> m_ice_quantity           );
   CCommUtils::SetVariable( HostComms, "wx_ice_time",               pArea-> m_ice_time               );
   CCommUtils::SetVariable( HostComms, "turbulence_intensity",      pArea-> m_turbulence_intensity   );
   CCommUtils::SetVariable( HostComms, "turbulence_enable",         pArea-> m_turbulence_enable      );

   // Do the Apply for the visual system.
   // Set timer to half a second before applying the changes.  This is juat a
   // guess; the value of a half second has no special meaning.
   SetApplyTimer(0.5f);
}


void CWeatherControl::SendATISInformation()
{
   CCommUtils::SetVariable( HostComms, "wxa1_airport_ident", m_area[0].m_airport, -1 );
   CCommUtils::SetVariable( HostComms, "wxa2_airport_ident", m_area[1].m_airport, -1 );
   CCommUtils::SetVariable( HostComms, "wxa3_airport_ident", m_area[2].m_airport, -1 );
   CCommUtils::SetVariable( HostComms, "wxa4_airport_ident", m_area[3].m_airport, -1 );

   CCommUtils::SetVariable( HostComms, "wxa1_active_rwy", m_area[0].m_runway, -1 );
   CCommUtils::SetVariable( HostComms, "wxa2_active_rwy", m_area[1].m_runway, -1 );
   CCommUtils::SetVariable( HostComms, "wxa3_active_rwy", m_area[2].m_runway, -1 );
   CCommUtils::SetVariable( HostComms, "wxa4_active_rwy", m_area[3].m_runway, -1 );

   CCommUtils::SetVariable( HostComms, "wxa1_ceiling1", m_area[0].m_cloud_2_bottom );
   CCommUtils::SetVariable( HostComms, "wxa2_ceiling1", m_area[1].m_cloud_2_bottom );
   CCommUtils::SetVariable( HostComms, "wxa3_ceiling1", m_area[2].m_cloud_2_bottom );
   CCommUtils::SetVariable( HostComms, "wxa4_ceiling1", m_area[3].m_cloud_2_bottom );

   CCommUtils::SetVariable( HostComms, "wxa1_clouds1", m_area[0].m_cloud_2_type );
   CCommUtils::SetVariable( HostComms, "wxa2_clouds1", m_area[1].m_cloud_2_type );
   CCommUtils::SetVariable( HostComms, "wxa3_clouds1", m_area[2].m_cloud_2_type );
   CCommUtils::SetVariable( HostComms, "wxa4_clouds1", m_area[3].m_cloud_2_type );

   CCommUtils::SetVariable( HostComms, "wxa1_visibility_sm", m_area[0].m_visibility_sm );
   CCommUtils::SetVariable( HostComms, "wxa2_visibility_sm", m_area[1].m_visibility_sm );
   CCommUtils::SetVariable( HostComms, "wxa3_visibility_sm", m_area[2].m_visibility_sm );
   CCommUtils::SetVariable( HostComms, "wxa4_visibility_sm", m_area[3].m_visibility_sm );

   CCommUtils::SetVariable( HostComms, "wxa1_temperature_f", m_area[0].m_field_temperature_f );
   CCommUtils::SetVariable( HostComms, "wxa2_temperature_f", m_area[1].m_field_temperature_f );
   CCommUtils::SetVariable( HostComms, "wxa3_temperature_f", m_area[2].m_field_temperature_f );
   CCommUtils::SetVariable( HostComms, "wxa4_temperature_f", m_area[3].m_field_temperature_f );

   CCommUtils::SetVariable( HostComms, "wxa1_wind_dir", m_area[0].m_surface_wind_direction );
   CCommUtils::SetVariable( HostComms, "wxa2_wind_dir", m_area[1].m_surface_wind_direction );
   CCommUtils::SetVariable( HostComms, "wxa3_wind_dir", m_area[2].m_surface_wind_direction );
   CCommUtils::SetVariable( HostComms, "wxa4_wind_dir", m_area[3].m_surface_wind_direction );

   CCommUtils::SetVariable( HostComms, "wxa1_wind_speed", m_area[0].m_surface_wind_speed );
   CCommUtils::SetVariable( HostComms, "wxa2_wind_speed", m_area[1].m_surface_wind_speed );
   CCommUtils::SetVariable( HostComms, "wxa3_wind_speed", m_area[2].m_surface_wind_speed );
   CCommUtils::SetVariable( HostComms, "wxa4_wind_speed", m_area[3].m_surface_wind_speed );

   CCommUtils::SetVariable( HostComms, "wxa1_altimeter_inhg", m_area[0].m_field_pressure_inHg );
   CCommUtils::SetVariable( HostComms, "wxa2_altimeter_inhg", m_area[1].m_field_pressure_inHg );
   CCommUtils::SetVariable( HostComms, "wxa3_altimeter_inhg", m_area[2].m_field_pressure_inHg );
   CCommUtils::SetVariable( HostComms, "wxa4_altimeter_inhg", m_area[3].m_field_pressure_inHg );

   CCommUtils::SetVariable( HostComms, "wxa1_precip", m_area[0].m_precip_type );
   CCommUtils::SetVariable( HostComms, "wxa2_precip", m_area[1].m_precip_type );
   CCommUtils::SetVariable( HostComms, "wxa3_precip", m_area[2].m_precip_type );
   CCommUtils::SetVariable( HostComms, "wxa4_precip", m_area[3].m_precip_type );
}


// Set the "apply" timer for a delayed activation to the visual system

void CWeatherControl::SetApplyTimer(float numSeconds)
{
   m_wxApplyTimer = (int)(m_nHostRate * numSeconds);
}


void CWeatherControl::LoadSortieAreaData()
{

   CWeatherArea *pArea = &m_area[0];

   CCommUtils::GetVariable( LocalComms, "wxld_use_area_1",                  pArea->m_use_area                );

   if (pArea->m_use_area)
   {

      CCommUtils::GetVariable( LocalComms, "wxld_radius_1",                    pArea-> m_wx_radius              );
      CCommUtils::GetVariable( LocalComms, "wxld_airport_1",                   pArea-> m_airport, -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_runway_1",                    pArea-> m_runway,  -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_press_1",            pArea-> m_field_pressure_inHg    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_temp_1",             pArea-> m_field_temperature_f    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_rvrFt_1",                  pArea-> m_RVR_ft                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_visibSm_1",                pArea-> m_visibility_sm          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_timeOfDay_1",              pArea-> m_visual_scene           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Top_1",              pArea-> m_cloud_2_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Bottom_1",           pArea-> m_cloud_2_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Type_1",             pArea-> m_cloud_2_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Top_1",              pArea-> m_cloud_1_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Bottom_1",           pArea-> m_cloud_1_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Type_1",             pArea-> m_cloud_1_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cirrus_1",                 pArea-> m_cirrus                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_scud_1",                   pArea-> m_scud                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogType_1",                pArea-> m_fog_type               );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogTop_1",                 pArea-> m_fog_top                );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_dir_1",           pArea-> m_surface_wind_direction );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_spd_1",           pArea-> m_surface_wind_speed     );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_spd_1",               pArea-> m_wind_gust_speed        );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_var_1",               pArea-> m_wind_gust_variation    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipType_1",             pArea-> m_precip_type            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipIntensity_1",        pArea-> m_precip_intensity       );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_lightning_1",              pArea-> m_lightning              );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_hail_1",                   pArea-> m_hail                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_RCR_1",                    pArea-> m_RCR                    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_contamination_1",          pArea-> m_runway_condition       );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_blowing_contamination_1", pArea-> m_blowing_contamination  );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_aptl_skidmarks_1",        pArea-> m_wet_skidmarks          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_quantity_1",           pArea-> m_ice_quantity           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_time_1",               pArea-> m_ice_time               );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_intensity_1",      pArea-> m_turbulence_intensity   );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_enable_1",         pArea-> m_turbulence_enable      );

      pArea->Activate();
   }

   else
   {
      pArea->Deactivate();
   }


   pArea = &m_area[1];

   CCommUtils::GetVariable( LocalComms, "wxld_use_area_2",                  pArea->m_use_area                );

   if (pArea->m_use_area)
   {
      CCommUtils::GetVariable( LocalComms, "wxld_radius_2",                    pArea-> m_wx_radius              );
      CCommUtils::GetVariable( LocalComms, "wxld_airport_2",                   pArea-> m_airport, -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_runway_2",                    pArea-> m_runway,  -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_press_2",            pArea-> m_field_pressure_inHg    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_temp_2",             pArea-> m_field_temperature_f    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_rvrFt_2",                  pArea-> m_RVR_ft                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_visibSm_2",                pArea-> m_visibility_sm          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_timeOfDay_2",              pArea-> m_visual_scene           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Top_2",              pArea-> m_cloud_2_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Bottom_2",           pArea-> m_cloud_2_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Type_2",             pArea-> m_cloud_2_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Top_2",              pArea-> m_cloud_1_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Bottom_2",           pArea-> m_cloud_1_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Type_2",             pArea-> m_cloud_1_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cirrus_2",                 pArea-> m_cirrus                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_scud_2",                   pArea-> m_scud                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogType_2",                pArea-> m_fog_type               );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogTop_2",                 pArea-> m_fog_top                );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_dir_2",           pArea-> m_surface_wind_direction );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_spd_2",           pArea-> m_surface_wind_speed     );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_spd_2",               pArea-> m_wind_gust_speed        );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_var_2",               pArea-> m_wind_gust_variation    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipType_2",             pArea-> m_precip_type            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipIntensity_2",        pArea-> m_precip_intensity       );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_lightning_2",              pArea-> m_lightning              );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_hail_2",                   pArea-> m_hail                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_RCR_2",                    pArea-> m_RCR                    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_contamination_2",          pArea-> m_runway_condition       );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_blowing_contamination_2", pArea-> m_blowing_contamination  );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_aptl_skidmarks_2",        pArea-> m_wet_skidmarks          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_quantity_2",           pArea-> m_ice_quantity           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_time_2",               pArea-> m_ice_time               );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_intensity_2",      pArea-> m_turbulence_intensity   );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_enable_2",         pArea-> m_turbulence_enable      );

      pArea->Activate();
   }

   else
   {
      pArea->Deactivate();
   }



   pArea = &m_area[2];

   CCommUtils::GetVariable( LocalComms, "wxld_use_area_3",                  pArea->m_use_area                );

   if (pArea->m_use_area)
   {
      CCommUtils::GetVariable( LocalComms, "wxld_radius_3",                    pArea-> m_wx_radius              );
      CCommUtils::GetVariable( LocalComms, "wxld_airport_3",                   pArea-> m_airport, -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_runway_3",                    pArea-> m_runway,  -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_press_3",            pArea-> m_field_pressure_inHg    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_temp_3",             pArea-> m_field_temperature_f    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_rvrFt_3",                  pArea-> m_RVR_ft                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_visibSm_3",                pArea-> m_visibility_sm          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_timeOfDay_3",              pArea-> m_visual_scene           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Top_3",              pArea-> m_cloud_2_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Bottom_3",           pArea-> m_cloud_2_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Type_3",             pArea-> m_cloud_2_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Top_3",              pArea-> m_cloud_1_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Bottom_3",           pArea-> m_cloud_1_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Type_3",             pArea-> m_cloud_1_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cirrus_3",                 pArea-> m_cirrus                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_scud_3",                   pArea-> m_scud                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogType_3",                pArea-> m_fog_type               );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogTop_3",                 pArea-> m_fog_top                );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_dir_3",           pArea-> m_surface_wind_direction );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_spd_3",           pArea-> m_surface_wind_speed     );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_spd_3",               pArea-> m_wind_gust_speed        );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_var_3",               pArea-> m_wind_gust_variation    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipType_3",             pArea-> m_precip_type            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipIntensity_3",        pArea-> m_precip_intensity       );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_lightning_3",              pArea-> m_lightning              );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_hail_3",                   pArea-> m_hail                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_RCR_3",                    pArea-> m_RCR                    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_contamination_3",          pArea-> m_runway_condition       );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_blowing_contamination_3", pArea-> m_blowing_contamination  );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_aptl_skidmarks_3",        pArea-> m_wet_skidmarks          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_quantity_3",           pArea-> m_ice_quantity           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_time_3",               pArea-> m_ice_time               );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_intensity_3",      pArea-> m_turbulence_intensity   );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_enable_3",         pArea-> m_turbulence_enable      );

      pArea->Activate();
   }

   else
   {
      pArea->Deactivate();
   }



   pArea = &m_area[3];

   CCommUtils::GetVariable( LocalComms, "wxld_use_area_4",                  pArea->m_use_area                );

   if (pArea->m_use_area)
   {
      CCommUtils::GetVariable( LocalComms, "wxld_radius_4",                    pArea-> m_wx_radius              );
      CCommUtils::GetVariable( LocalComms, "wxld_airport_4",                   pArea-> m_airport, -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_runway_4",                    pArea-> m_runway,  -1            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_press_4",            pArea-> m_field_pressure_inHg    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_field_temp_4",             pArea-> m_field_temperature_f    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_rvrFt_4",                  pArea-> m_RVR_ft                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_visibSm_4",                pArea-> m_visibility_sm          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_timeOfDay_4",              pArea-> m_visual_scene           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Top_4",              pArea-> m_cloud_2_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Bottom_4",           pArea-> m_cloud_2_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud2Type_4",             pArea-> m_cloud_2_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Top_4",              pArea-> m_cloud_1_top            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Bottom_4",           pArea-> m_cloud_1_bottom         );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cloud1Type_4",             pArea-> m_cloud_1_type           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_cirrus_4",                 pArea-> m_cirrus                 );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_scud_4",                   pArea-> m_scud                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogType_4",                pArea-> m_fog_type               );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_fogTop_4",                 pArea-> m_fog_top                );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_dir_4",           pArea-> m_surface_wind_direction );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_sfc_wind_spd_4",           pArea-> m_surface_wind_speed     );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_spd_4",               pArea-> m_wind_gust_speed        );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_gust_var_4",               pArea-> m_wind_gust_variation    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipType_4",             pArea-> m_precip_type            );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_precipIntensity_4",        pArea-> m_precip_intensity       );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_lightning_4",              pArea-> m_lightning              );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_hail_4",                   pArea-> m_hail                   );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_RCR_4",                    pArea-> m_RCR                    );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_contamination_4",          pArea-> m_runway_condition       );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_blowing_contamination_4", pArea-> m_blowing_contamination  );
      CCommUtils::GetVariable( LocalComms, "wxld_vis_aptl_skidmarks_4",        pArea-> m_wet_skidmarks          );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_quantity_4",           pArea-> m_ice_quantity           );
      CCommUtils::GetVariable( LocalComms, "wxld_wx_ice_time_4",               pArea-> m_ice_time               );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_intensity_4",      pArea-> m_turbulence_intensity   );
      CCommUtils::GetVariable( LocalComms, "wxld_turbulence_enable_4",         pArea-> m_turbulence_enable      );

      pArea->Activate();
   }

   else
   {
      pArea->Deactivate();
   }

   // Let the host know that the weather area data has changed
   CCommUtils::SetVariable( HostComms, "wx_areas_changed", true );
}

