// LatLonToleranceData.h: interface for the CLatLonToleranceData class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(_LATLONTOLERANCEDATA_H_)
#define _LATLONTOLERANCEDATA_H_

#include <string>
#include <list>
#include <map>

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

class CVariant;

struct TolData
{
    unsigned long   s_ulTime;
    double          s_dLat;
    double          s_dLon;
    float           s_fAlt;
    float           s_fAirpseed;
//    float         s_fHeading;
};

enum TolDataClients
{
    AreaMapLead,
    AreaMapWing,
    ApproachPlot,
    GeoChartLead,
    GeoChartWing
};

enum TolDataResets
{
    RESET_ADDED     = 1,
    RESET_START_IT  = 2
};

class __declspec(dllexport) CToleranceData
{
private:
    // "Time" - first pass is 0, increments by one for each
    // call to Update() unless a freeze condition is in effect.
    // Used to make sure that the length of a track can be 
    // based on time.
    static unsigned long                        m_ulTime;

    // Pointer to the variant for the value of system freeze.
    // System freeze is used to determine whether to increment
    // m_ulTime.
    static CVariant*                            m_pSystemFreeze;

    static CVariant*                            m_pSortieLoaded;

    // Tolerances for the ground and altitude distances.
    static double                               m_dLatLonTolerance;
    static float                                m_fAltTolerance;

    // Maximum number of toldata elements to keep around.
    static long                                 m_lMaxNumberOfElementsLead;
    static long                                 m_lMaxNumberOfElementsWing;

    // Pointers to data from the host for the lead aircraft.
    static CVariant*                            m_pLatVarLead;
    static CVariant*                            m_pLonVarLead;
    static CVariant*                            m_pAltVarLead;
    static CVariant*                            m_pAirspeedVarLead;
//    static CVariant*                            m_pHeadingVarLead;

    // Pointers to data form the host for the wing aircraft.
    static CVariant*                            m_pLatVarWing;
    static CVariant*                            m_pLonVarWing;
    static CVariant*                            m_pAltVarWing;
    static CVariant*                            m_pAirspeedVarWing;
//    static CVariant*                            m_pHeadingVarWing;

    // The toldata elements.
    static _FSI_STL::list<struct TolData>       m_listTolDataLead;
    static _FSI_STL::list<struct TolData>       m_listTolDataWing;

    // Storage of iterators into the lists that keep track of the last data
    // that was displayed so that only the new data has to be obtained and displayed
    // and the start position after a clear.
    static _FSI_STL::map<HWND, _FSI_STL::map<int, _FSI_STL::list<struct TolData>::iterator> > m_mapStartIts;
    static _FSI_STL::map<HWND, _FSI_STL::map<int, _FSI_STL::list<struct TolData>::iterator> > m_mapClearIts;

    static _FSI_STL::map<HWND, _FSI_STL::map<int, bool> > m_mapClearSinceLastUpdate;

    // Storage of how many elements have been added or deleted since the last
    // time that data was retrieved from the lists.
    static _FSI_STL::map<HWND, _FSI_STL::map<int, unsigned short> > m_mapAdded;

    static _FSI_STL::map<HWND, _FSI_STL::map<int, int> > m_mapTolDataReset;

    static _FSI_STL::list<struct TolData>::iterator m_itLastAddedLead;
    static _FSI_STL::list<struct TolData>::iterator m_itLastAddedWing;

    // The name of the variable used to get the system freeze data.
    static _FSI_STL::string                         m_stlStrSystemFreeze;

    static _FSI_STL::string                         m_stlStrSortieLoaded;

    // The names of the variables to use for obtaining data.
    static _FSI_STL::string                         m_stlStrLatHostNameLead;
    static _FSI_STL::string                         m_stlStrLonHostNameLead;
    static _FSI_STL::string                         m_stlStrAltHostNameLead;
    static _FSI_STL::string                         m_stlStrAirspeedHostNameLead;
//    static _FSI_STL::string                            m_stlStrHeadingHostNameLead;

    static _FSI_STL::string                         m_stlStrLatHostNameWing;
    static _FSI_STL::string                         m_stlStrLonHostNameWing;
    static _FSI_STL::string                         m_stlStrAltHostNameWing;
    static _FSI_STL::string                         m_stlStrAirspeedHostNameWing;
//    static _FSI_STL::string                            m_stlStrHeadingHostNameWing;

    static void ResetData();

    static bool                                     m_bResetClearSinceLast;

public:
	 CToleranceData();
    virtual ~CToleranceData(){};

    static void Update();

    static void Clear(HWND hWnd, enum TolDataClients eTDC);
    static void ClearAll();
    static bool ClearSinceLastUpdate(HWND hWnd, enum TolDataClients eTDC);
 
    // Accessors
    static double LatLonTolerance();
    static float  AltTolerance();
    static unsigned short Added(HWND hWnd, enum TolDataClients eTDC);
    static void StartIt(HWND hWnd, enum TolDataClients eTDC, bool& bOk, _FSI_STL::list<struct TolData>::iterator& it);
    static void ClearIt(HWND hWnd, enum TolDataClients eTDC, bool& bOk,_FSI_STL::list<struct TolData>::iterator& it);

    // Mutators
    static void LatLonTolerance(const double& rdLatLonTolerance);
    static void AltTolerance(const float fAltTolerance);
};

#endif // !defined(_LATLONTOLERANCEDATA_H_)
