// Texan_Model.cpp: implementation of the Texan_Model class.
//
//////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <gl/gl.h>
#include <gl/glu.h>
#include <assert.h>
#include "Texan_Model.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Include model geometry
//////////////////////////////////////////////////////////////////////
#include "texan_data.h"


//////////////////////////////////////////////////////////////////////
// Construction
//////////////////////////////////////////////////////////////////////
Texan_Model::Texan_Model()
{
	m_gear_position[NOSE_GEAR] = 1.0;
	m_gear_position[LEFT_GEAR] = 1.0;
	m_gear_position[RIGHT_GEAR] = 1.0;
	
	m_gear_door_position[NOSE_GEAR] = 1.0;
	m_gear_door_position[LEFT_GEAR] = 0.0;
	m_gear_door_position[RIGHT_GEAR] = 0.0;

	m_elevator_position = 0.0;
	m_rudder_position = 0.0;
	m_prop_position = 0.0;
	m_prop_pitch = 45.0;
	m_steering_angle = 0.0;

    m_bDeleteLists = false;
}


//////////////////////////////////////////////////////////////////////
// Destruction
//////////////////////////////////////////////////////////////////////
Texan_Model::~Texan_Model()
{
    if (m_bDeleteLists == true)
    {
	    glDeleteLists(m_aircraft_list, 1);
	    glDeleteLists(m_elevator_list, 1);
	    glDeleteLists(m_nose_door_list, 1);
	    glDeleteLists(m_nosegear_list, 1);
	    glDeleteLists(m_prop_list, 1);
	    glDeleteLists(m_rudder_list, 1);
	    glDeleteLists(m_l_gear_list, 1);
	    glDeleteLists(m_r_gear_list, 1);
	    glDeleteLists(m_l_outer_gear_door_list, 1);
	    glDeleteLists(m_r_outer_gear_door_list, 1);
	    glDeleteLists(m_inner_gear_door_list, 1);
	    glDeleteLists(m_aileron1_list, 1);
	    glDeleteLists(m_aileron2_list, 1);
    }
}


//////////////////////////////////////////////////////////////////////
// The main drawing routine
//////////////////////////////////////////////////////////////////////
void Texan_Model::Render()
{
	glPushAttrib(GL_TEXTURE_BIT);

	drawACBody();
	drawElevator();
	drawRudder();
//	drawProp();
	drawGear();
	drawAilerons();

	glPopAttrib();
}


//////////////////////////////////////////////////////////////////////
// Initialization routine; called once per instance.  Builds textures,
//   creates display lists for speed.
//////////////////////////////////////////////////////////////////////
void Texan_Model::Initialize(whichPaintScheme paintScheme)
{
   if (m_bDeleteLists == true)
   {
      glDeleteLists(m_aircraft_list, 1);
      glDeleteLists(m_elevator_list, 1);
      glDeleteLists(m_nose_door_list, 1);
      glDeleteLists(m_nosegear_list, 1);
      glDeleteLists(m_prop_list, 1);
      glDeleteLists(m_rudder_list, 1);
      glDeleteLists(m_l_gear_list, 1);
      glDeleteLists(m_r_gear_list, 1);
      glDeleteLists(m_l_outer_gear_door_list, 1);
      glDeleteLists(m_r_outer_gear_door_list, 1);
      glDeleteLists(m_inner_gear_door_list, 1);
      glDeleteLists(m_aileron1_list, 1);
      glDeleteLists(m_aileron2_list, 1);
   }
   else
   {
      m_bDeleteLists = true;
   }

	assert (paintScheme == NAVY || paintScheme == GENERIC || paintScheme == NOT_ANY);

	char navy_lh_texture[] = "flhnavy.bmp";
	char navy_rh_texture[] = "frhnavy.bmp";
	char navy_wt_texture[] = "navywt.bmp";
	char navy_wb_texture[] = "navywb.bmp";

	switch(paintScheme)
		{
		case NAVY:
			m_bottom_texture.LoadBMP(navy_wb_texture);
			m_top_texture.LoadBMP(navy_wt_texture);
			// continue into GENERIC, no 'break'
		case GENERIC:
			l_fuselageTex.LoadBMP(navy_lh_texture);
			r_fuselageTex.LoadBMP(navy_rh_texture);
			break;
		case NOT_ANY:
			break;
		}

	glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	l_fuselageTex.TexParameter(GL_TEXTURE_WRAP_S, GL_CLAMP);
	l_fuselageTex.TexParameter(GL_TEXTURE_WRAP_T, GL_CLAMP);
	l_fuselageTex.TexParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	l_fuselageTex.TexParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);

	r_fuselageTex.TexParameter(GL_TEXTURE_WRAP_S, GL_CLAMP);
	r_fuselageTex.TexParameter(GL_TEXTURE_WRAP_T, GL_CLAMP);
	r_fuselageTex.TexParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	r_fuselageTex.TexParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);

	m_bottom_texture.TexParameter(GL_TEXTURE_WRAP_S, GL_CLAMP);
	m_bottom_texture.TexParameter(GL_TEXTURE_WRAP_T, GL_CLAMP);
	m_bottom_texture.TexParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	m_bottom_texture.TexParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);

	m_top_texture.TexParameter(GL_TEXTURE_WRAP_S, GL_CLAMP);
	m_top_texture.TexParameter(GL_TEXTURE_WRAP_T, GL_CLAMP);
	m_top_texture.TexParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	m_top_texture.TexParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);

	// Generate aircraft display lists ///////////////////////////////
	m_aircraft_list = glGenLists(1);
	glNewList(m_aircraft_list, GL_COMPILE);

		glEnableClientState(GL_VERTEX_ARRAY);
		glEnableClientState(GL_NORMAL_ARRAY);

		glColor4f(JPATS_NOSECONE);

		glVertexPointer(3, GL_FLOAT, 0, &NOSECONE);
		glNormalPointer(GL_FLOAT, 0, &NOSECONE_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 180);

		glColor4f(JPATS_CANOPY);
		glVertexPointer(3, GL_FLOAT, 0, &CANOPY);
		glNormalPointer(GL_FLOAT, 0, &CANOPY_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 339);

		glColor4f(JPATS_WHITE);
	
		glVertexPointer(3, GL_FLOAT, 0, &VTAIL);
		glNormalPointer(GL_FLOAT, 0, &VTAIL_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 87);
		
		glVertexPointer(3, GL_FLOAT, 0, &FINS);
		glNormalPointer(GL_FLOAT, 0, &FINS_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 60);

		// WINGS /////////////////////////////////////////////////////
		if (paintScheme == NAVY)
			{
			glEnable(GL_TEXTURE_2D);
			glColor4f(JPATS_WHITE);
			m_bottom_texture.SetActive();
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glTexCoordPointer(2,GL_FLOAT, 0, &WING_BOTTOM_TEXTURE);
			}
		else
			{
			glDisable(GL_TEXTURE_2D);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
			glColor4f(JPATS_BLUE);
			}

		glVertexPointer(3, GL_FLOAT, 0, &WING_BOTTOM);
		glNormalPointer(GL_FLOAT, 0, &WING_BOTTOM_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 270); 
	
		if (paintScheme == NAVY)
			{
			m_top_texture.SetActive();
			glTexCoordPointer(2,GL_FLOAT, 0, &WING_TOP_TEXTURE);
			}
		else
			{
			glEnableClientState(GL_COLOR_ARRAY);
			glColorPointer(4, GL_FLOAT, 0, &WING_TOP_COLORS);
			}

		glVertexPointer(3, GL_FLOAT, 0, &WING_TOP);
		glNormalPointer(GL_FLOAT,0,&WING_TOP_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 234);

		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		glDisable(GL_TEXTURE_2D);

		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_FLOAT, 0, &WING_EDGE_COLORS);
		glNormalPointer(GL_FLOAT, 0, &WING_EDGE_NORMALS);
		glVertexPointer(3, GL_FLOAT, 0, &WING_EDGE);
		glDrawArrays(GL_TRIANGLES, 0, 132);

		glDisableClientState(GL_COLOR_ARRAY);

		// END OF WING ///////////////////////////////////////////////

		glPushMatrix();
		glDisableClientState(GL_COLOR_ARRAY);
		if (paintScheme != NOT_ANY)
			{
			glEnable(GL_TEXTURE_2D);
			l_fuselageTex.SetActive();
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glTexCoordPointer(2,GL_FLOAT, 0, &L_FUSELAGE_TEXCOORD);
			}
		else {
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
			glDisable(GL_TEXTURE_2D);
			}
		glColor4f(JPATS_WHITE);
		glVertexPointer(3, GL_FLOAT, 0, &L_FUSELAGE_FACES_X);
		glNormalPointer(GL_FLOAT, 0, &L_FUSELAGE_FACES_NORM);
		glDrawArrays(GL_TRIANGLES, 0, 834);

		if (paintScheme != NOT_ANY)
		{
			r_fuselageTex.SetActive();
			glTexCoordPointer(2,GL_FLOAT, 0, &R_FUSELAGE_TEXCOORD);
		}
		glVertexPointer(3, GL_FLOAT, 0, &R_FUSELAGE_FACES_X);
		glNormalPointer(GL_FLOAT, 0, &R_FUSELAGE_FACES_NORM);
		glDrawArrays(GL_TRIANGLES, 0, 987);

		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		glDisable(GL_TEXTURE_2D);
		glPopMatrix();

		glColor4f(JPATS_BLACK);

		glVertexPointer(3, GL_FLOAT, 0, &BLACK_TOP);
		glNormalPointer(GL_FLOAT, 0, &BLACK_TOP_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 129);

		glEnableClientState(GL_COLOR_ARRAY);

		glVertexPointer(3, GL_FLOAT, 0, &HTAIL);
		glNormalPointer(GL_FLOAT, 0, &HTAIL_NORMALS);
		glColorPointer(4, GL_FLOAT, 0, &HTAIL_COLORS);
		glDrawArrays(GL_TRIANGLES, 0, 126);

		glDisableClientState(GL_COLOR_ARRAY);

		if (paintScheme != NOT_ANY) // Disable exhaust for maximum performance
		{
			glColor4f(JPATS_EXHAUST);
//			glDisable(GL_CULL_FACE);
			glNormalPointer(GL_FLOAT, 0, &EXHAUST_NORMALS);
			glVertexPointer(3, GL_FLOAT, 0, &EXHAUST_RH);
			glDrawArrays(GL_TRIANGLES, 0, 378);
			glVertexPointer(3, GL_FLOAT, 0, &EXHAUST_LH);
			glDrawArrays(GL_TRIANGLES, 0, 378);
//			glEnable(GL_CULL_FACE);
		}

	glEndList();

	m_nose_door_list = glGenLists(1);
	glNewList(m_nose_door_list, GL_COMPILE);
		glColor4f(JPATS_BLUE);
//		glDisable(GL_CULL_FACE);
		glNormalPointer(GL_FLOAT, 0, &NOSEDOOR_NORMALS);
		glVertexPointer(3,GL_FLOAT,0,&NOSEDOOR);
		glDrawArrays(GL_TRIANGLES,0,6);
//		glEnable(GL_CULL_FACE);
	glEndList();

	m_elevator_list = glGenLists(1);
	glNewList(m_elevator_list, GL_COMPILE);
		glEnableClientState(GL_NORMAL_ARRAY);
		glEnableClientState(GL_COLOR_ARRAY);
		glVertexPointer(3, GL_FLOAT, 0, &ELEVATOR);
		glNormalPointer(GL_FLOAT, 0, &ELEVATOR_NORMALS);
		glVertexPointer(3, GL_FLOAT, 0, &ELEVATOR);
		glColorPointer(4, GL_FLOAT, 0, &ELEVATOR_COLORS);
		glDrawArrays(GL_TRIANGLES, 0, 90);
		glDisableClientState(GL_COLOR_ARRAY);
	glEndList();

	m_rudder_list = glGenLists(1);
	glNewList(m_rudder_list, GL_COMPILE);
		glColor4f(JPATS_WHITE);
		glVertexPointer(3, GL_FLOAT, 0, &RUDDER);
		glNormalPointer(GL_FLOAT, 0, &RUDDER_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 57);
	glEndList();

	m_prop_list = glGenLists(1);
	glNewList(m_prop_list, GL_COMPILE);
		glColor4f(JPATS_PROP); // This is only the plain prop
		glVertexPointer(3, GL_FLOAT, 0, &PROP);
		glNormalPointer(GL_FLOAT, 0, &PROP_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 312);
	glEndList();

	m_nosegear_list = glGenLists(1);
	glNewList(m_nosegear_list, GL_COMPILE);
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_FLOAT, 0, &NOSEGEAR_COLORS);
		glVertexPointer(3, GL_FLOAT, 0, &NOSEGEAR);
		glNormalPointer(GL_FLOAT, 0, &NOSEGEAR_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 540);
		glDisableClientState(GL_COLOR_ARRAY);
	glEndList();

	m_l_gear_list = glGenLists(1);
	glNewList(m_l_gear_list, GL_COMPILE);
		glEnableClientState(GL_COLOR_ARRAY);
		glVertexPointer(3, GL_FLOAT, 0, &GEAR);
		glNormalPointer(GL_FLOAT, 0, &GEAR_NORMALS);	
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_FLOAT, 0, &GEAR_COLORS);
		glDrawArrays(GL_TRIANGLES, 0, 468);
		glDisableClientState(GL_COLOR_ARRAY);
	glEndList();

	m_r_gear_list = glGenLists(1);
	glNewList(m_r_gear_list, GL_COMPILE);
     glPushMatrix();
		glScalef(-1.0f, 1.0f, -1.0f);
		glCallList(m_l_gear_list);
     glPopMatrix();
	glEndList();

	m_l_outer_gear_door_list = glGenLists(1);
	glNewList(m_l_outer_gear_door_list, GL_COMPILE);
		glColor4f(JPATS_BLUE);
		glVertexPointer(3, GL_FLOAT, 0, &GEAR_DOOR);
		glNormalPointer(GL_FLOAT, 0, &GEAR_DOOR_NORMALS);
		glDrawArrays(GL_TRIANGLES, 0, 36);
	glEndList();

	m_r_outer_gear_door_list = glGenLists(1);
	glNewList(m_r_outer_gear_door_list, GL_COMPILE);
     glPushMatrix();
		glScalef(-1.0f, 1.0f, -1.0f);
		glCallList(m_l_outer_gear_door_list);
     glPopMatrix();
	glEndList();

	m_inner_gear_door_list = glGenLists(1);
	glNewList(m_inner_gear_door_list, GL_COMPILE);
		glColor4f(JPATS_BLUE);
		glVertexPointer(3, GL_FLOAT, 0, &INNER_DOOR);
		glNormalPointer(GL_FLOAT, 0, &INNER_DOOR_NORMALS);
		glDrawArrays(GL_TRIANGLES,0,36);
	glEndList();

	m_aileron1_list = glGenLists(1);
	glNewList(m_aileron1_list, GL_COMPILE);
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_FLOAT, 0, &AILERON1_COLORS);
		glNormalPointer(GL_FLOAT, 0, &AILERON1_NORMALS);
		glVertexPointer(3, GL_FLOAT, 0, &AILERON1);
		glDrawArrays(GL_TRIANGLES, 0, 24);
	glEndList();

	m_aileron2_list = glGenLists(2);
	glNewList(m_aileron2_list, GL_COMPILE);
		glColorPointer(4, GL_FLOAT, 0, &AILERON2_COLORS);
		glNormalPointer(GL_FLOAT, 0, &AILERON2_NORMALS);
		glVertexPointer(3, GL_FLOAT, 0, &AILERON2);
		glDrawArrays(GL_TRIANGLES, 0, 24);
		glDisableClientState(GL_COLOR_ARRAY);
	glEndList(); 
}


//////////////////////////////////////////////////////////////////////
// "Set" methods for the class
//////////////////////////////////////////////////////////////////////

void Texan_Model::SetPropPitch(double newPropPitch)
{
	// Propeller pitch ranges between 0 and 90.0 degrees (TENTATIVE)
	assert( newPropPitch >= 0.0 && newPropPitch <= 90.0 );
	m_prop_pitch = newPropPitch;
}

void Texan_Model::SetPropPosition(double newPropPosition)
{
	// Propeller rotational position ranges between 0.0 and 360.0 degrees
	assert( newPropPosition >= 0.0 && newPropPosition < 360.0 );
	m_prop_position = newPropPosition;
}

void Texan_Model::SetRudderPosition(double newRudderPosition)
{
	// Rudder position ranges between +90.0 and -90.0 degrees
	assert( newRudderPosition >= -90.0 && newRudderPosition <= 90.0 );
	m_rudder_position = newRudderPosition;
}

void Texan_Model::SetElevatorPosition(double newElevatorPosition)
{
	// Elevator position ranges between +90.0 and -90.0 degrees
	assert( newElevatorPosition >= -90.0 && newElevatorPosition <= 90.0);
	m_elevator_position = newElevatorPosition;
}

void Texan_Model::SetGearPosition(whichGear gearIndex, double newGearPosition)
{
	// Gear Position ranges from 0.0 (UP) to 1.0 (DOWN).
	assert( gearIndex >=0  && gearIndex <= 2);
	assert( m_gear_position[gearIndex] >= 0 && m_gear_position[gearIndex] <= 1.0 );
	m_gear_position[gearIndex] = newGearPosition ;  // Convert into degrees
}

void Texan_Model::SetGearDoorPosition(whichGear gearDoorIndex, double newGearDoorPosition)
{
	// Gear Door position ranges from 0.0 (UP) to 1.0 (DOWN).
	assert( gearDoorIndex >=0  && gearDoorIndex <= 2);
	assert( m_gear_door_position[gearDoorIndex] >= 0 && m_gear_door_position[gearDoorIndex] <= 1.0 );
	m_gear_door_position[gearDoorIndex] = newGearDoorPosition ;  // Convert into degrees
}


void Texan_Model::SetSteeringAngle(double newAngle)
{
	// Sets nosewheel steering angle.  Ranges from -90 to +90.
	assert( newAngle > -90.0);
	assert( newAngle <  90.0);
	m_steering_angle = newAngle;
}


//////////////////////////////////////////////////////////////////////
// Render-time drawing routines.  Inline for speed.
//////////////////////////////////////////////////////////////////////

inline void Texan_Model::drawACBody()
{
	glCallList(m_aircraft_list);
}


inline void Texan_Model::drawElevator()
{
	glPushMatrix();
		{
		glTranslatef(0.576f,19.9f,-370.0f);
		glRotatef((float)m_elevator_position, 1.0f, 0.0f, 0.0f);
		glCallList(m_elevator_list);
		}
	glPopMatrix();
}


inline void Texan_Model::drawRudder()
{
	glPushMatrix();
		{
		glTranslatef(0,0,-324);
		glRotatef((float)m_rudder_position,0.0f, 1.0f, 0.0f);
		glTranslatef(-2,-110,324);
		glCallList(m_rudder_list);
		}
	glPopMatrix();
}


inline void Texan_Model::drawProp()
{
	glPushMatrix();
#if 1 // PLAIN PROP
	glTranslatef(0.0f,3.0f,0.0f);
	glRotatef((float)m_prop_position, 0.0f, 0.0f, 1.0f);
	glCallList(m_prop_list);

#else // FANCY PROP WITH VARIABLE PITCH
	glColor4f(JPATS_PROP);		
	glTranslatef(0.0f,3.0f,262.0f);
	glRotatef((float)m_prop_position, 0.0f, 0.0f, 1.0f);
	glPushMatrix();
	glRotatef((float)m_prop_pitch,1.0f,0.0f,0.0f);
	glCallList(m_prop_list);	
	glVertexPointer(3, GL_FLOAT, 0, &PROP2);
	glNormalPointer(GL_FLOAT, 0, &PROP2_NORMALS);
	glDrawArrays(GL_TRIANGLES, 0, 126);// draw blade #1
	glPopMatrix();
	glPushMatrix();
	glRotatef(90.0f,0.0,0.0,1.0);
	glRotatef((float)m_prop_pitch,1.0f,0.0f,0.0f);
	glDrawArrays(GL_TRIANGLES, 0, 126);// draw blade #2
	glPopMatrix();
	glPushMatrix();
	glRotatef(180.0f,0.0,0.0,1.0);
	glRotatef((float)m_prop_pitch,1.0f,0.0f,0.0f);
	glDrawArrays(GL_TRIANGLES, 0, 126);// draw blade #3
	glPopMatrix();
	glRotatef(270.0f,0.0,0.0,1.0);
	glRotatef((float)m_prop_pitch,1.0f,0.0f,0.0f);
	glDrawArrays(GL_TRIANGLES, 0, 126);// draw blade #4
#endif
	glPopMatrix();
}


inline void Texan_Model::drawGear()
{
	// DRAW NOSE GEAR ////////////////////////////////////////////////
	// Don't bother drawing gear if it is retracted
	if ( m_gear_position[NOSE_GEAR] > 0.01f )
		{
		glPushMatrix();
			glTranslatef(0.5f, -6.0f, 155.0f);
			glRotatef((float)m_steering_angle, 0.0f, 1.0f, 0.0f);
			// Convert to degrees.  The "0.9" is due to the fact that it swings out a little past vertical.
			glRotatef((float)((0.9 - m_gear_position[NOSE_GEAR]) * 90.0), 1.0f, 0.0f, 0.0f);
			glCallList(m_nosegear_list);
		glPopMatrix();
		}
	
	// DRAW NOSE GEAR DOORS //////////////////////////////////////////
		//   Here, door position is keyed to gear position, so 
		//   m_gear_door_position[NOSE_GEAR] input is ignored
	if (m_gear_position[NOSE_GEAR] > 0.01f)  
		{
		float door_degrees;
		if (m_gear_position[NOSE_GEAR] >= 0.1)
			door_degrees = 0.0; // (this is fully open)
		else
			door_degrees = (float) (90.0 - (900.0 * m_gear_position[NOSE_GEAR]));

		glPushMatrix();
		glTranslatef(-7.5f, -36.0f, 85.0f);
		glRotatef(door_degrees, 0.0f, 0.0, 1.0f);
		glCallList(m_nose_door_list);		// First nose door
		glPopMatrix();

		glPushMatrix();
		glTranslatef(10.5f, -36.0f, 85.0f);
		glRotatef(-door_degrees, 0.0f, 0.0, 1.0f);
		glCallList(m_nose_door_list);		// Second nose door
		glPopMatrix();
		}

	// POSITION AND DRAW LEFT GEAR /////////////////////////////////////
//m_gear_position[LEFT_GEAR] = m_gear_position[RIGHT_GEAR] = m_gear_position[NOSE_GEAR] = temps;
	if (m_gear_position[LEFT_GEAR] > 0.01)
		{
		glPushMatrix();
		glTranslatef(67.0f,-24.0f,0.0f);
		glRotatef((float)(90.0  * (m_gear_position[LEFT_GEAR] - 1.0)), 0.0f, 0.0f, 1.0f);
		glCallList(m_l_gear_list);
		glPopMatrix();
		
		// POSITION AND DRAW LEFT OUTBOARD GEAR DOOR /////////////////////
		glPushMatrix();
		glTranslatef(79.5, -35.0, 0.0);
		glRotatef((float)(120.0 * (m_gear_position[LEFT_GEAR] - 1.0)),0.0f,0.0f,1.0f);
		glCallList(m_l_outer_gear_door_list);
		glPopMatrix();
		}

	// POSITION AND DRAW LEFT INNER GEAR DOOR ////////////////////////
	//
	if (m_gear_door_position[LEFT_GEAR] > 0.01)
		{
		glPushMatrix();
		glTranslatef(3.0, -37.0, 0.0);
		glRotatef((float)(-90.0 * m_gear_door_position[LEFT_GEAR]), 0.0, 0.0, 1.0);
		glCallList(m_inner_gear_door_list);
		glPopMatrix();
		}

	// POSITION AND DRAW RIGHT GEAR ////////////////////////////////////
	if (m_gear_position[RIGHT_GEAR] > 0.01)
		{
		glPushMatrix();
		glTranslatef(-65.5f, -24.0f, 0.0f);
		glRotatef((float)(90.0  * (1.0 - m_gear_position[RIGHT_GEAR])), 0.0f, 0.0f, 1.0f);
		glCallList(m_r_gear_list);
		glPopMatrix();

		// POSITION AND DRAW RIGHT OUTBOARD GEAR DOOR /////////////////////
		glPushMatrix();
		glTranslatef(-78.0, -35.0, 0.0);
		glRotatef((float)(120.0 * (1.0 - m_gear_position[RIGHT_GEAR])),0.0f,0.0f,1.0f);
		glCallList(m_r_outer_gear_door_list);
		glPopMatrix();
		}
	
	// POSITION AND DRAW RIGHT INNER GEAR DOOR ////////////////////////
	if (m_gear_door_position[RIGHT_GEAR] > 0.01)
		{
		glPushMatrix();
		glTranslatef(-3.0, -37.0, 0.0);
		glRotatef((float)(180.0 + 90.0 * m_gear_door_position[RIGHT_GEAR]), 0.0, 0.0, 1.0);
		glCallList(m_inner_gear_door_list);
		glPopMatrix();
		}

}

inline void Texan_Model::drawAilerons()
{
		glPushMatrix();
		glTranslatef(-226.0f, -13.0f, -23.0f);
		glRotatef(-(float)m_rudder_position, -74.658f, 11.170f, 6.189f);
		glScalef(1.729f, 1.729f, 1.729f);
		glCallList(m_aileron1_list);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(226.0f, -13.0f, -23.0f);
		glRotatef(-(float)m_rudder_position, 74.659f, 11.170f, 6.190f);
		glScalef(1.729f, 1.729f, 1.729f);
		glCallList(m_aileron2_list);
		glPopMatrix();
}
