// NavaidSystemInterface.cpp: implementation of the CNavaidSystemInterface class.
//
//////////////////////////////////////////////////////////////////////

#include "..\core\stdafx.h"
#include "..\core\Action.h"
#include "NavaidSystemInterface.h"
#include "StationKillAction.h"
#include "NavaidCommsAction.h"


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CNavaidSystemInterface::CNavaidSystemInterface()
{
   m_int_configuration  =  0x0a;

   m_pAction            =  NULL;

   m_pending_status     =  0;
}

CNavaidSystemInterface::~CNavaidSystemInterface()
{
   delete   m_XMLWidget_latitude;
   delete   m_XMLWidget_longitude;
   delete   m_XMLWidget_host_status_array;
   delete   m_XMLWidget_host_idents_array;

   if (CWidget::IsValidAddress(m_CommsAction_latitude)  != INVALID)
      m_CommsAction_latitude->Deleting(true);

   if (CWidget::IsValidAddress(m_CommsAction_longitude) != INVALID)
      m_CommsAction_longitude->Deleting(true);

   if (CWidget::IsValidAddress(m_CommsAction_host_idents_array) != INVALID)
      m_CommsAction_host_idents_array->Deleting(true);

   if (CWidget::IsValidAddress(m_CommsAction_host_status_array) != INVALID)
      m_CommsAction_host_status_array->Deleting(true);
}


void  CNavaidSystemInterface::FixupAndConvertIdent(CString &CStr_ident)
{
   CStr_ident.TrimLeft();                                      // Remove leading spaces.
   CStr_ident.TrimRight();                                     // Remove trailing spaces.
   CStr_ident   =  CStr_ident.Mid(CStr_ident.Find(" ") + 1);   // Move to ident which is possibly after a frequency.
   CStr_ident.MakeUpper();                                     // Uppercase it.

   int   length   =  CStr_ident.GetLength();
   for (; length < 4; length++)
   {
      CStr_ident += " ";                                       // Add necessary spaces at the end.
   }
}


void  CNavaidSystemInterface::SetCurrentIdent(_FSI_STL::string &ident, bool reload)
{
   m_int_configuration  =  GetIdentConfig(ident);

   CString  cStr_ident(ident.c_str());       // Initialize a CString with the parameter (ident).
   FixupAndConvertIdent(cStr_ident);         // Upper case it, pad it...

   // Update current ident string.
   m_stl_str_ident   =  cStr_ident;

   if (CWidget::IsValidAddress(m_CommsAction_host_idents_array) != INVALID)
      m_CommsAction_host_idents_array->OnLButtonUp();

   if (m_pAction  && reload)
      m_pAction->OnUpdate();

   // Update user interface to reflect configuration options.
   _FSI_STL::list<CAction*>::iterator lIt    = m_listActions.begin();
   _FSI_STL::list<CAction*>::iterator lendIt = m_listActions.end();
   while (lIt != lendIt)
   {
      (*lIt)->OnUpdate();
      lIt++;
   }
}


_FSI_STL::string  CNavaidSystemInterface::GetCurrentIdent(void)
{
   return   m_stl_str_ident;
}


int   CNavaidSystemInterface::GetCurrentConfiguration(void)
{
   return   m_int_configuration;
}


long  CNavaidSystemInterface::GetCurrentStatus(void)
{
   int   ii;
   for (ii = 0; ii < MAX_STATION_KILL; ii++)
   {
      if (m_CStr_idents[ii]   == m_stl_str_ident.c_str())
         break;
   }

   if (ii < MAX_STATION_KILL)
      return   m_long_status[ii];

   return 0;
}


long  CNavaidSystemInterface::GetIdentStatus(const _FSI_STL::string &ident)
{
   int   ii;
   for (ii = 0; ii < MAX_STATION_KILL; ii++)
   {
      if (m_CStr_idents[ii]   == ident.c_str())
         break;
   }

   if (ii < MAX_STATION_KILL)
      return   m_long_status[ii];

   return 0;
}


long  CNavaidSystemInterface::GetIdentConfig(const _FSI_STL::string &ident)
{
   long     configuration  =  0;
   CString  cStr_ident(ident.c_str());          // Initialize a CString with the parameter (ident).
   FixupAndConvertIdent(cStr_ident);            // Upper case it, pad it...

   // Validate Ident - check to see if Ident exists in Radio Database.
   CString  temp(cStr_ident);
   temp.MakeReverse();
   int   int_ident         =  (int)*(int *)LPCTSTR(temp);
   int   nNumNavAids       =  0;
   rdbStation  *pStation   =  COpenGLMap::rdbRadioDB.NearestNavAid(int_ident, m_latitude, m_longitude, nNumNavAids);

   if (pStation)
   {
      configuration  |= DetermineConfiguration(pStation);

      VecStation vecpStations;
      const double dDistance = 10.0;

      // locate any other nav aids of the same ident within some specified distance
      COpenGLMap::rdbRadioDB.AdjacentNavAids(pStation, dDistance, vecpStations);
      if (0 < vecpStations.size())
      {
         // found some others, so combine these facilities with the others
         for (VecStation::iterator iter = vecpStations.begin(); iter != vecpStations.end(); iter++)
         {
            configuration  |= DetermineConfiguration(*iter);
         }
      }
   }
   else
   {
      CString  message;
      message.Format("Sorry, no NAVAID with an ident of  \"%s\" could be found.", cStr_ident);
      COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
      COpenGLMap::m_stlStrCaption   =  "Invalid Ident";
      AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
      return   0;
   }

   return   configuration;
}


long  CNavaidSystemInterface::DetermineConfiguration(rdbStation *pStation)
{
   long  configuration  =  0;

   // Look up configuration options from radio database
   DWORD dwTypePrimary     =  pStation->nType1;
   DWORD dwTypeSecondary   =  pStation->nType2;

   dwTypePrimary &= 0xfffffe00u;

   switch   (dwTypePrimary)
   {
      case  rdbType1ILSDME:
      case  rdbType1ILS:
      {
         configuration        |= dwTypePrimary;
         if (dwTypeSecondary  &  0x00100000)
            configuration     |= 0x00000010;       // LOC
         if (dwTypeSecondary  &  0x00200000)
            configuration     |= 0x00000008;       // GS

         rdbILS   *pILS    =  (rdbILS *)pStation;
         if (pILS->nOmLat  || pILS->nOmLon)
            configuration  |= 0x00000004;          // OM
         if (pILS->nMmLat  || pILS->nMmLon)
            configuration  |= 0x00000002;          // MM
         if (pILS->nImLat  || pILS->nImLon)
            configuration  |= 0x00000001;          // IM
      }
      break;

      case rdbType1NDB:
      case rdbType1VOR:
      case rdbType1DME:
      case rdbType1TAC:
      case rdbType1VORDME:
      case rdbType1VORTAC:
      case rdbType1DMETAC:
      case rdbType1VORDMETAC:
      {
         configuration     |= dwTypePrimary;
      }
      break;
   }

   return   configuration;
}


void CNavaidSystemInterface::SetPendingStatus(long status)
{
   long  configuration  =  GetCurrentConfiguration();                      // See what is available to Fail or Reset.
   long  current_status =  GetCurrentStatus();                             // See what is currently Failed or Reset.

   if (!(current_status &  rdbType1ILS)   && (status  &  rdbType1ILS))     // Current status has ILS Reset and new status has ILS Failed.
   {
      if (status  &  0x08000000)                                           // If new status has DME failed make sure it stays failed but we
         configuration  &= 0x480000FF;                                     // don't care about the MLS, VOR, TAC, NDB, AWM, COM...
      else
         configuration  &= 0x400000FF;                                     // don't care if it has MLS, VOR, TAC, NDB, AWM, COM...
      status         =  configuration;                                     // fail all the other ILS related items.
   }
   else if (current_status &  rdbType1ILS && !(status &  rdbType1ILS))     // Current status has ILS Failed and new status has ILS Reset.
   {
      status         &= 0xBFFFFF00;                                        // Reset all the other ILS related items.
   }
   else if ((current_status   &  0x000000FF) != (status  &  0x000000FF))   // If current status and new status lower bits have changed.
   {
      if ((status &  0x000000FF) == (configuration &  0x000000FF))         // All is Failed that can be so,
         status      |= 0x40000000;                                        // set the total bit
      else                                                                 // else
         status      &= 0xBFFFFFFF;                                        // clear the total bit.
   }

   m_pending_status  =  status;

   if (CWidget::IsValidAddress(m_CommsAction_host_status_array) != INVALID)
      m_CommsAction_host_status_array->OnLButtonUp();
}


long CNavaidSystemInterface::GetPendingStatus()
{
   return   m_pending_status;
}


void  CNavaidSystemInterface::CreateComms()
{
   CXMLElement *pXMLElement   =  NULL;
   POSITION    pos            =  NULL;
   m_XMLWidget_latitude       =  NULL;
   m_CommsAction_latitude     =  new   CNavaidCommsAction;

   if (m_CommsAction_latitude)
   {
      m_CommsAction_latitude->Initialize(m_XMLWidget_latitude, NULL, 1, false);

      if (m_XMLWidget_latitude)
      {
         // Set the read mode to changed and the write mode to NO.
         if (m_XMLWidget_latitude->FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
         {
            pXMLElement->AddElementValue("ac_latitude");    // Set IOS alias name to READ from.

            STRING2STRING_MAP::iterator s2sIt = NULL;
            CString strValue;
            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("READ_MODE")) == true)
            {
               CString strValue;
               strValue.Format("%d",READ_CHANGED);
               (*s2sIt).second = _FSI_STL::string((LPCTSTR)strValue);
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("WRITE_MODE")) == true)
            {
               (*s2sIt).second = _FSI_STL::string("NO");
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
               (*s2sIt).second = "ac_latitude";              // Set the ChangeValue string.
            }
         }

         pXMLElement = NULL;
         if (m_XMLWidget_latitude->FindElement(pXMLElement, pos, _FSI_STL::string("RANGE_DATA")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("NONE"));
         }

         pXMLElement = NULL;
         if (m_XMLWidget_latitude->FindElement(pXMLElement, pos, _FSI_STL::string("FORMAT")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("Generic_No_Conversions"));
         }

         m_CommsAction_latitude->Initialize(m_XMLWidget_latitude, NULL, 1, false);
      }
   }

   pXMLElement             =  NULL;
   pos                     =  NULL;
   m_XMLWidget_longitude   =  NULL;
   m_CommsAction_longitude =  new   CNavaidCommsAction;

   if (m_CommsAction_longitude)
   {
      m_CommsAction_longitude->Initialize(m_XMLWidget_longitude, NULL, 1, false);

      if (m_XMLWidget_longitude)
      {
         // Set the read mode to changed and the write mode to NO.
         if (m_XMLWidget_longitude->FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
         {
            pXMLElement->AddElementValue("ac_longitude");      // Set IOS alias name to READ from.

            STRING2STRING_MAP::iterator s2sIt = NULL;
            CString strValue;
            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("READ_MODE")) == true)
            {
               CString strValue;
               strValue.Format("%d",READ_CHANGED);
               (*s2sIt).second = _FSI_STL::string((LPCTSTR)strValue);
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("WRITE_MODE")) == true)
            {
               (*s2sIt).second = _FSI_STL::string("NO");
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
               (*s2sIt).second = "ac_longitude";               // Set the ChangeValue string.
            }
         }

         pXMLElement = NULL;
         if (m_XMLWidget_longitude->FindElement(pXMLElement, pos, _FSI_STL::string("RANGE_DATA")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("NONE"));
         }

         pXMLElement = NULL;
         if (m_XMLWidget_longitude->FindElement(pXMLElement, pos, _FSI_STL::string("FORMAT")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("Generic_No_Conversions"));
         }

         m_CommsAction_longitude->Initialize(m_XMLWidget_longitude, NULL, 1, false);
      }
   }


   // Look up aliases for the navaid ident and status arrays from the host.
   HKEY hKey;
    // Open the registry.
   RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\FlightSafety\\FSISuite\\Radio Database"), 0, KEY_READ, &hKey);

   CString  CStr_navaid_status;
   CString  CStr_navaid_idents;

   if (hKey != NULL)
   {
      CString strValue;
      DWORD dwType, dwCount;

      LONG lResult = RegQueryValueEx(hKey, _T("NavaidIdents"), NULL, &dwType, NULL, &dwCount);
      if (lResult == ERROR_SUCCESS)
      {
         ASSERT(dwType == REG_SZ);
         lResult = RegQueryValueEx(hKey, _T("NavaidIdents"), NULL, &dwType, (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)), &dwCount);
         strValue.ReleaseBuffer();
      }
      CStr_navaid_idents = strValue;

      lResult = RegQueryValueEx(hKey, _T("NavaidStatus"), NULL, &dwType, NULL, &dwCount);
      if (lResult == ERROR_SUCCESS)
      {
         ASSERT(dwType == REG_SZ);
         lResult = RegQueryValueEx(hKey, _T("NavaidStatus"), NULL, &dwType, (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)), &dwCount);
         strValue.ReleaseBuffer();
      }
      CStr_navaid_status = strValue;

      RegCloseKey(hKey);
   }


   pXMLElement                      =  NULL;
   pos                              =  NULL;
   m_XMLWidget_host_status_array    =  NULL;
   m_CommsAction_host_status_array  =  new   CNavaidCommsAction;

   if (m_CommsAction_host_status_array)
   {
      m_CommsAction_host_status_array->Initialize(m_XMLWidget_host_status_array, NULL, 1, false);

      if (m_XMLWidget_host_status_array)
      {
         // Set the read mode to changed and the write mode to NO.
         if (m_XMLWidget_host_status_array->FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
         {
            pXMLElement->AddElementValue((LPCTSTR)CStr_navaid_status);    // Set IOS alias name to READ from.

            STRING2STRING_MAP::iterator s2sIt = NULL;
            CString strValue;
            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("READ_MODE")) == true)
            {
               CString strValue;
               strValue.Format("%d",READ_ALWAYS);
               (*s2sIt).second = _FSI_STL::string((LPCTSTR)strValue);
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("WRITE_MODE")) == true)
            {
               (*s2sIt).second = _FSI_STL::string("YES");
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
               (*s2sIt).second = "host_status_array";              // Set the ChangeValue string.
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ARRAY_ELEMENT")) == true)
            {
               (*s2sIt).second = "-1";                             // -1 for whole array.
            }
         }

         pXMLElement = NULL;
         if (m_XMLWidget_host_status_array->FindElement(pXMLElement, pos, _FSI_STL::string("RANGE_DATA")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("NONE"));
         }

         pXMLElement = NULL;
         if (m_XMLWidget_host_status_array->FindElement(pXMLElement, pos, _FSI_STL::string("FORMAT")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("Generic_No_Conversions"));
         }

         m_CommsAction_host_status_array->Initialize(m_XMLWidget_host_status_array, NULL, 1, false);
      }
   }


   pXMLElement                      =  NULL;
   pos                              =  NULL;
   m_XMLWidget_host_idents_array    =  NULL;
   m_CommsAction_host_idents_array  =  new   CNavaidCommsAction;

   if (m_CommsAction_host_idents_array)
   {
      m_CommsAction_host_idents_array->Initialize(m_XMLWidget_host_idents_array, NULL, 1, false);

      if (m_XMLWidget_host_idents_array)
      {
         // Set the read mode to changed and the write mode to NO.
         if (m_XMLWidget_host_idents_array->FindElement(pXMLElement, pos, _FSI_STL::string("VARIABLE")) == true)
         {
            pXMLElement->AddElementValue((LPCTSTR)CStr_navaid_idents);    // Set IOS alias name to READ from.

            STRING2STRING_MAP::iterator s2sIt = NULL;
            CString strValue;
            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("READ_MODE")) == true)
            {
               CString strValue;
               strValue.Format("%d",READ_ALWAYS);
               (*s2sIt).second = _FSI_STL::string((LPCTSTR)strValue);
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("WRITE_MODE")) == true)
            {
               (*s2sIt).second = _FSI_STL::string("YES");
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ELEMENT_VAR")) == true)
            {
               (*s2sIt).second = "host_idents_array";              // Set the ChangeValue string.
            }

            if (pXMLElement->FindAttribute(s2sIt,_FSI_STL::string("ARRAY_ELEMENT")) == true)
            {
               (*s2sIt).second = "-1";                             // -1 for whole array.
            }
         }

         pXMLElement = NULL;
         if (m_XMLWidget_host_idents_array->FindElement(pXMLElement, pos, _FSI_STL::string("RANGE_DATA")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("NONE"));
         }

         pXMLElement = NULL;
         if (m_XMLWidget_host_idents_array->FindElement(pXMLElement, pos, _FSI_STL::string("FORMAT")) == true)
         {
            pXMLElement->AddElementValue(_FSI_STL::string("Generic_No_Conversions"));
         }

         m_CommsAction_host_idents_array->Initialize(m_XMLWidget_host_idents_array, NULL, 1, false);
      }
   }
}


void  CNavaidSystemInterface::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   if (pCV == NULL)
       return;

   CVariant* pVariant = pCV->Variant();

   if (rstrElementVar      == "ac_latitude")
   {
      m_latitude  =  *pVariant;
   }
   else if (rstrElementVar == "ac_longitude")
   {
      m_longitude  =  *pVariant;
   }
   else if (rstrElementVar == "host_status_array")
   {
      int nLength = pVariant->Length();
      if (nLength  <= sizeof(m_long_status)/sizeof(m_long_status[0]))
      {
         if (memcmp(&m_long_status, pVariant->CreateVar(), nLength*sizeof(m_long_status[0])))
         {
            memcpy(&m_long_status, pVariant->CreateVar(), nLength*sizeof(m_long_status[0]));

            // Update user interface to reflect configuration options.
            _FSI_STL::list<CAction*>::iterator lIt    = m_listActions.begin();
            _FSI_STL::list<CAction*>::iterator lendIt = m_listActions.end();
            while (lIt != lendIt)
            {
               (*lIt)->OnUpdate();

               lIt++;
            }
         }
      }
   }
   else if (rstrElementVar == "host_idents_array")
   {
      int   ii;
      int   number_of_strings =  pVariant->Length()   >>  2;

      if (number_of_strings   >  MAX_STATION_KILL)    // Prevent overflow problem.
         number_of_strings    =  MAX_STATION_KILL;

      char  *pData   =  (char *)(pVariant->CreateVar());

      // Upper case every character.
      for (ii = 0; ii < number_of_strings; ii++)
      {
         m_CStr_idents[ii].Format("%.4s", &(pData[ii<<2]));
         m_CStr_idents[ii].MakeUpper();
      }
   }
}
