// iosRadioDB.cpp : implementation file
//
// IOS Runtime Radio Data Base class.
//
// Copyright (c) 1997 by ZCT Systems Group, Inc.
// All Rights Reserved.
//
// Notes :
//
//	This file contains the IOS Runtime Radio Data Base class.  This class is designed to
// perform all functions related to radio data base access.
//

#ifdef   JPATS
   #include "..\core\stdafx.h"
   #include "..\core\DataConversion.h"
   #include "winsock2.h"
#else
   #include "stdafx.h"
#endif

#include "iosRadioDB.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CIosRadioDB

//
// CIosRadioDB::CIosRadioDB
// Constructor
// Entry   : nothing
// Returns : nothing
//

CIosRadioDB::CIosRadioDB()
{
	// Initialize.

	m_pFreq = NULL;
	m_nFreqTableRecords = 0;
	m_pIdent = NULL;
	m_nIdentName = 0;
	m_nIdentPosition = -1;
	m_nIdentTableRecords = 0;
	m_pILS = NULL;
	m_pLatLon = NULL;
	m_nLatLonTableRecords = 0;
	m_pRunway = NULL;
	m_nStation = -1;
	m_pStations = NULL;
}

//
// CIosRadioDB::~CIosRadioDB
// Destructor
// Entry   : nothing
// Returns : nothing
//

CIosRadioDB::~CIosRadioDB()
{
	Delete();
}

bool CIosRadioDB::Delete()
{
	// Free the station data.

	if(m_pStations)
	{
		free(m_pStations);
		m_pStations = NULL;
	}

	// Free the frequency table.

	if(m_pFreq)
	{
		free(m_pFreq);
		m_pFreq = NULL;
	}

	// Free the ident table.

	if(m_pIdent)
	{
		free(m_pIdent);
		m_pIdent = NULL;
	}

	// Free the lat/lon table.

	if(m_pLatLon)
	{
		free(m_pLatLon);
		m_pLatLon = NULL;
	}
	return true;
}


/////////////////////////////////////////////////////////////////////////////
// CIosRadioDB functions.

//
// CIosRadioDB::Load
// Load a FlightSafety International radio data base file
// Entry   : pathname to radio data base file
// Returns : TRUE if successful, FALSE if not
// Notes   :
//
//	The first 256 bytes of the radio data base file contains a file header.  In
// the header, along with other data, are pointers to various tables of interest
// to IosRt.  The offset to the file pointers to the tables of interest to IosRt
// are defined as follows:
//		
//		RADIODB_AIRPT_TABLE_DATA_OFFSET		: offset to pointer to airport data array
//		RADIODB_FREQ_TABLE_DATA_OFFSET		: offset to pointer to frequency array
//		RADIODB_IDENT_TABLE_DATA_OFFSET		: offset to pointer to ident array
//		RADIODB_LATLON_TABLE_DATA_OFFSET	: offset to pointer to lat/lon array
//
//	Immediately following the header is the "stations" data.  The file offset is defined
// as:
//
//		RADIODB_STATIONS_OFFSET				: offset to stations array.
//
// and, since the header is 256 bytes, this offset is defined as 256.  The "real" data
// is located in this table.  However, the elements in this table vary in size, thus
// the table cannot be accessed as an array.  The element definitions are type defined
// in iosRadioDB.h.
//
//	The remaining tables contain pointers into the stations table elements sorted in
// ascending order of the table type.  For example, the ident table contains pointers
// to the elements in the stations table, where the pointers are sorted by ident.
// Since the pointers are a constant size, these tables can be accessed as arrays.
// The tables in the radio data base file following the station data tables, in order,
// are as follows:
//
//			frequency			: frequency
//			ident				: ident
//			lat/lon				: lat/lon
//			airport				: not used
//			airport 3 character : not used
//
// *************** WARNING WARNING WARNING WARNING WARNING WARNING ****************
//
//	Be aware the FlightSafety International radio data base data is in "big-endian"
// or "network" byte ordering format, while the Intel family of processors expects data
// in "little-endian" byte ordering format.  To access any data in any of the tables,
// the "ntoh" or "ntohl" or equivalent byte order reversing functions must be used.
// This function converts all of this data to Intel byte ordering including the frequency,
// ident and lat/lon tables as well as the station data itself.
//

BOOL	CIosRadioDB::Load(const char * chPathName)
{
	// Clean up from any previous Load
	Delete();

	// Attempt to open the radio data base file.

	FILE*	fpFile;

	if(strlen(chPathName))
	{
		if((fpFile = fopen(chPathName, "rb")) == NULL)
		{
			// Error, cannot open file.

			return FALSE;
		}
	}
	else
	{
		// Error, no path name.

		return FALSE;
	}

   char  header[256];
	if ((fread(header, sizeof(header),  1, fpFile)) != 1)
   {
		fclose(fpFile);
		return FALSE;
   }
   m_bool_intel_format  =  header[255] == 1;

	// Allocate memory and load tables for frequency, ident and lat/lon tables
	if (!LoadTable(fpFile, RADIODB_FREQ_TABLE_DATA_OFFSET,   (void **)&m_pFreq,   sizeof(rdbStation *), m_nFreqTableRecords  ) ||
		!LoadTable(fpFile, RADIODB_IDENT_TABLE_DATA_OFFSET,  (void **)&m_pIdent,  sizeof(rdbStation *), m_nIdentTableRecords ) ||
		!LoadTable(fpFile, RADIODB_LATLON_TABLE_DATA_OFFSET, (void **)&m_pLatLon, sizeof(rdbLatLon),    m_nLatLonTableRecords) ||
		!LoadStations(fpFile))
	{
		fclose(fpFile);
		return FALSE;
	}

	// Convert the pointers to Intel byte ordering.

		// station data
      SwapStationData();

		// frequency table
		for(int nCount = 0; nCount < m_nFreqTableRecords; nCount ++)
		{
         if (m_bool_intel_format)
		   	((long*)m_pFreq)[nCount] = (     (((long *)m_pFreq)[nCount]) - RADIODB_STATIONS_OFFSET) + (long)m_pStations;
         else
		   	((long*)m_pFreq)[nCount] = (ntohl(((long *)m_pFreq)[nCount]) - RADIODB_STATIONS_OFFSET) + (long)m_pStations;
		}

		// ident table
		for(nCount = 0; nCount < m_nIdentTableRecords; nCount ++)
		{
			if (m_bool_intel_format)
            ((long*)m_pIdent)[nCount] = (     (((long *)m_pIdent)[nCount]) - RADIODB_STATIONS_OFFSET) + (long)m_pStations;
         else
            ((long*)m_pIdent)[nCount] = (ntohl(((long *)m_pIdent)[nCount]) - RADIODB_STATIONS_OFFSET) + (long)m_pStations;
		}

		// lat/lon table		
		for(nCount = 0; nCount < m_nLatLonTableRecords; nCount ++)
		{
         if (m_bool_intel_format)
         {
			   m_pLatLon[nCount].nLat     = (m_pLatLon[nCount].nLat);
			   m_pLatLon[nCount].nLon     = (m_pLatLon[nCount].nLon);
			   m_pLatLon[nCount].pStation = (rdbStation *)(((long)m_pLatLon[nCount].pStation) - RADIODB_STATIONS_OFFSET + (long)m_pStations);
         }
         else
         {
			   m_pLatLon[nCount].nLat     = ntohl(m_pLatLon[nCount].nLat);
			   m_pLatLon[nCount].nLon     = ntohl(m_pLatLon[nCount].nLon);
			   m_pLatLon[nCount].pStation = (rdbStation *)(ntohl((long)m_pLatLon[nCount].pStation) - RADIODB_STATIONS_OFFSET + (long)m_pStations);
         }
		}
   
	// End.

	fclose(fpFile);


	// Return data base loaded.
	return TRUE;
}

//
// CIosRadioDB::Airport
// Locate first airport
// Entry   : ident
// Returns : pointer to Station if successful, NULL if not
//

rdbStation*	CIosRadioDB::Airport(int nIdent)
{
	rdbStation*	pStation = Ident(nIdent);
	
	while(pStation != NULL)
	{
		// See if this station is an airport.

		if((pStation->nType1 & rdbType1APT) == rdbType1APT)
		{
			// Found airport.

			return	pStation;
		}

		// Not an airport, get the next station.

		pStation = IdentNext();
	}

	// Airport not found.

	return NULL;
}

//
// CIosRadioDB::Ident
// Locate first ident in cluster
// Entry   : ident
// Returns : pointer to Station if successful, NULL if not
// Notes   :
//
//	The stations table contains clusters of elements with the same ident.  This function
// returns the first element in the cluster.
//

rdbStation*	CIosRadioDB::Ident(int nIdent)
{
	if((m_pStations) && (m_pIdent))
	{
		// Have stations and idents, perform a binary search for nIdent;

		int		nCurrent;
		int		nHighest = m_nIdentTableRecords - 1;
		int		nLowest = 0;

		while(nLowest <= nHighest)
		{
			// Look between the current highest and lowest positions.

			nCurrent = (nLowest + nHighest) >> 1;

			// Compare idents.

			int nThisIdent = m_pIdent[nCurrent]->nIdent;

			if(nIdent > nThisIdent)
			{
				// Ident is higher than current, thus the lowest can
				// now be current + 1.

				nLowest = nCurrent + 1;
			}

			else if(nIdent < nThisIdent)
			{
				// Ident is lower than current, thus the highest can
				// now be current - 1.

				nHighest = nCurrent - 1;
			}

			else
			{
				// Found ident, locate first ident in cluster.

				while((nCurrent) && ((int)m_pIdent[nCurrent - 1]->nIdent == nIdent))
					nCurrent --;

				// Record the ident name and position for IdentNext().

				m_nIdentName = nIdent;
				m_nIdentPosition = nCurrent;

				// End.

				return(m_pIdent[nCurrent]);
			}
		}
	}

	// Ident not found.

	m_nIdentName = 0;
	m_nIdentPosition = -1;
	return	NULL;
}

//
// CIosRadioDB::IdentNext
// Locate next ident in cluster
// Entry   : nothing
// Returns : pointer to Station if successful, NULL if not
// Notes   :
//
//	A call to Ident() is required before calling IdentNext().
//

rdbStation*	CIosRadioDB::IdentNext()
{
	if((m_pStations) && (m_pIdent) && (m_nIdentPosition >=0) && (m_nIdentPosition < (m_nIdentTableRecords - 1)))
	{
		// Have a stations array, an ident array, and a valid ident array position.
		// Check the next ident to see if the name matches that of the last Ident()
		// call.

		if(m_nIdentName == m_pIdent[m_nIdentPosition + 1]->nIdent)
		{
			// Name matches, update the position.

			m_nIdentPosition ++;

			// Then return the station pointer.

			return(m_pIdent[m_nIdentPosition]);
		}
	}

	// Cannot find next.

	return NULL;
}

//
// CIosRadioDB::ILS
// Locate first ILS at an airport
// Entry   : airport ident
// Returns : pointer to Station if successful, NULL if not
//

rdbStation*	CIosRadioDB::ILS(int nIdent)
{
	// Locate the airport.

	m_pILS = Airport(nIdent);
	
	// Locate the first ILS at the airport.

	while(m_pILS != NULL)
	{
		// See if this station is an ILS.

		if((m_pILS->nType1 & rdbType1ILS) == rdbType1ILS)
		{
			// Found ILS, typecast it as such.
			rdbILS* pILS = (rdbILS*)m_pILS;

			// If the ILS station is from a different airport than
			// the aiport corresponding to nIdent then return NULL.
			if(pILS->nAptIdent != nIdent)
			{
				pILS = NULL;
				
				//typecast back to generic rdbStation pointer
				m_pILS = (rdbStation*)pILS;
			}

			return	m_pILS;
		}


		// Not an ILS, get the next station.

		m_pILS = StationNext(m_pILS);

	}

	// ILS not found.

	return m_pILS;
}

//
// CIosRadioDB::ILSNext
// Locate next ils
// Entry   : nothing
// Returns : pointer to Station if successful, NULL if not
// Notes   :
//
//	A call to ILS() is required before calling ILSNext().
//

rdbStation*	CIosRadioDB::ILSNext()
{
	// Move to the next station.

	m_pILS = StationNext(m_pILS);

	while(m_pILS != NULL)
	{
		// See if this station is an ILS.

		if((m_pILS->nType1 & rdbType1ILS) == rdbType1ILS)
		{
			// Found ILS.

			return	m_pILS;
		}
		else
		{
			// End of ILS group.

			m_pILS = NULL;
		}
	}

	// ILS not found.

	return m_pILS;
}

//
// CIosRadioDB::Runway
// Locate first runway at an airport
// Entry   : airport ident
// Returns : pointer to Station if successful, NULL if not
//

rdbStation*	CIosRadioDB::Runway(int nIdent)
{
	// Locate the airport.

	m_pRunway = Airport(nIdent);
	
	// Locate the first runway at the airport.

	while(m_pRunway != NULL)
	{
		// See if this station is a runway.

		if((m_pRunway->nType1 & rdbType1RWY) == rdbType1RWY)
		{
			// Found runway.

			return	m_pRunway;
		}

		// Not a runway, get the next station.

		m_pRunway = StationNext(m_pRunway);
	}

	// Runway not found.

	return m_pRunway;
}

//
// CIosRadioDB::RunwayNext
// Locate next runway
// Entry   : nothing
// Returns : pointer to Station if successful, NULL if not
// Notes   :
//
//	A call to Runway() is required before calling RunwayNext().
//

rdbStation*	CIosRadioDB::RunwayNext()
{
	// Move to the next station.

	m_pRunway = StationNext(m_pRunway);

	while(m_pRunway != NULL)
	{
		// See if this station is a runway.

		if((m_pRunway->nType1 & rdbType1RWY) == rdbType1RWY)
		{
			// Found runway.

			return	m_pRunway;
		}
		else
		{
			// End of runway group.

			m_pRunway = NULL;
		}
	}

	// Runway not found.

	return m_pRunway;
}

//
// CIosRadioDB::StationFirst
// Obtain pointer to first station.
// Entry   : none
// Returns : pointer to first Station if successful, NULL if not
// Notes   :
//

rdbStation*	CIosRadioDB::StationFirst()
{
	return (rdbStation *)m_pStations;
}

//
// CIosRadioDB::StationFirstLat
// Obtain pointer to first station within given latitude range
// Entry   : minimum, maximum latitude in degrees
// Returns : pointer to Station if successful, NULL if not
//

rdbStation*	CIosRadioDB::StationFirstLat(double dLatitudeMin, double dLatitudeMax)
{
	if((m_pStations) && (m_pLatLon))
	{
		// Have station and lat/lon tables, perform a binary search for latitude.

		int		nCurrent;
		int		nHighest = m_nLatLonTableRecords - 1;
		int		nLowest = 0;

		while(nLowest <= nHighest)
		{
			// Look between the current highest and lowest positions.

			nCurrent = (nLowest + nHighest) >> 1;

			// Compare latitudes.

			double	dLatitudeCurrent = BamsToDegrees(m_pLatLon[nCurrent].nLat);

			if(dLatitudeCurrent > dLatitudeMin)
			{
				// Latitude is higher than current, thus the lowest can
				// now be current + 1.

				nHighest = nCurrent - 1;
			}

			else if(dLatitudeCurrent < dLatitudeMin)
			{
				// Latitude is lower than current, thus the highest can
				// now be current - 1.

				nLowest = nCurrent + 1;
			}

			else
			{
				// Found latitude, locate first latitude in cluster.

				while(nCurrent && BamsToDegrees(m_pLatLon[nCurrent].nLat) == dLatitudeMin)
					nCurrent --;

				// End.

				m_nStation = nCurrent;
				return	(rdbStation*)m_pLatLon[nCurrent].pStation;
			}
		}

		// Exact latitude match not found.

		while(nCurrent < m_nLatLonTableRecords && BamsToDegrees(m_pLatLon[nCurrent].nLat) < dLatitudeMin)
			nCurrent ++;

		if(nCurrent >= m_nLatLonTableRecords)
		{
			m_nStation = -1;
			return	NULL;
		}

		// Make sure maximum latitude has not been exceeded.

		if(BamsToDegrees(m_pLatLon[nCurrent].nLat) > dLatitudeMax)
		{
			m_nStation = -1;
			return	NULL;
		}

		/* Found station. */

		m_nStation = nCurrent;
		return	(rdbStation*)m_pLatLon[nCurrent].pStation;
	}
	else
	{
		// Database not loaded.

		m_nStation = -1;
		return	NULL;
	}
}

//
// CIosRadioDB::StationNext
// Obtain pointer to the next station
// Entry   : pointer to current station
// Returns : pointer to next station or NULL if end of stations
// Notes   :
//
//	The stations array (m_pStations) is not linear in that it contains records of
// differing sizes.  In order to increment to the next station record, this function
// adds the number of bytes required to store the current station record being pointed
// at by the parameter pStation.
//

rdbStation *	CIosRadioDB::StationNext(rdbStation* pStation)
{
	// Convert station pointer to char pointer for byte add.

	char *		pChar = (char *) pStation;

	// Validate char pointer.

	if(pChar != NULL)
	{
		// Char pointer valid, offset to next station based on current
		// station type.

		switch(pStation->nType1 & 0x7ce0fc00)
		{
			case rdbType1APT:
			{
				pChar += sizeof(rdbAPT);
			}
			break;

			case rdbType1AWM:
			{
				pChar += sizeof(rdbAWM);
			}
			break;

			case rdbType1COM:
			{
				pChar += sizeof(rdbCOM);
			}
			break;

			case rdbType1VOR:
			case rdbType1DME:
			case rdbType1TAC:
			case rdbType1VORDME:
			case rdbType1VORTAC:
			case rdbType1DMETAC:
			case rdbType1VORDMETAC:
			{
				pChar += sizeof(rdbVHF);
			}
			break;

			case rdbType1ENR:
			{
				pChar += sizeof(rdbENR);
			}
			break;

			case rdbType1HP:
			{
				pChar += sizeof(rdbHP);
			}
			break;

			case rdbType1ILS:
			case rdbType1ILSDME:
			{
				pChar += sizeof(rdbILS);
			}
			break;

			case rdbType1MLS:
			{
				pChar += sizeof(rdbMLS);
			}
			break;

			case rdbType1NDB:
			{
				pChar += sizeof(rdbNDB);
			}
			break;

			case rdbType1RWY:
			{
				pChar += sizeof(rdbRWY);
			}
			break;

			case rdbType1WPT:
			case rdbType1TWPT:
			{
				pChar += sizeof(rdbWPT);
			}
			break;

			case rdbType1NULL:
			{
				pChar = NULL;
			}
			break;

			default:
			{
				// if you hit this assertion, then the radio db you're using has a newly defined Type1 code
				//  or a new combination of codes
				ASSERT(0);
				pChar = NULL;
			}
			break;
		}
	}

	// End.

	return (rdbStation *) pChar;
}

//
// CIosRadioDB::StationNextLat
// Obtain pointer to next station within given latitude range
// Entry   : minimum, maximum latitude in degrees
// Returns : pointer to Station if successful, NULL if not
//

rdbStation*	CIosRadioDB::StationNextLat(double dLatitudeMin, double dLatitudeMax)
{
	m_nStation ++;

	if((m_nStation >= 0) && (m_nStation < m_nLatLonTableRecords))
	{
		rdbStation* pStation = (rdbStation*)m_pLatLon[m_nStation].pStation;
		double		dLatitudeCurrent = BamsToDegrees(m_pLatLon[m_nStation].nLat);

		if((dLatitudeCurrent >= dLatitudeMin) && (dLatitudeCurrent <= dLatitudeMax))
		{
			return pStation;
		}
	}

	// No more stations within lat.

	m_nStation = -1;
	return	NULL;
}

//
// CIosRadioDB::LoadTable
// Read in the specified table into allocated memory
// Entry   : input file, location of header info, target buffer, record size
// Returns : true if completely successful
//
bool CIosRadioDB::LoadTable(FILE *fpFile, int nHeaderLoc, void **ppBuffer, int nRecordSize, int &nNumOfRecords)
{
	// Seek the location of the file pointer to the header location.
	if(fseek(fpFile, nHeaderLoc, SEEK_SET))
	{
		return false;		// Error, cannot seek position.
	}

	// Now read the start offset for this table.
	int nStartOffset = 0;
	if((fread((char *)& nStartOffset, sizeof(nStartOffset), 1, fpFile)) != 1)
	{
		return false;		// Error, cannot read file.
	}
   if (!m_bool_intel_format)
	   nStartOffset = ntohl(nStartOffset);

	// Then read the number of records for this table.
	nNumOfRecords = 0;
	if((fread((char *)& nNumOfRecords, sizeof(nNumOfRecords), 1, fpFile)) != 1)
	{
		return false;		// Error, cannot read file.
	}
   if (!m_bool_intel_format)
   	nNumOfRecords = ntohl(nNumOfRecords);

	// calculate size of table
	int nBuffSize = nNumOfRecords * nRecordSize;

	// Allocate the memory.
	if ((*ppBuffer = malloc(nBuffSize)) == NULL)
	{
		return false; // Error, not enough memory.
	}

	// Seek the table data.
	if(fseek(fpFile, nStartOffset, SEEK_SET))
	{
		return false; // Error, cannot seek position.
	}

	// Load the data.
	if((fread(*ppBuffer, nBuffSize, 1, fpFile)) != 1)
	{
		return false; // Error, cannot read file.
	}

	return true;
}

//
// CIosRadioDB::LoadStations
// Read in the Station data from the datafile into allocated memory
// Entry   : input file pointer
// Returns : true if completely successful
//
bool CIosRadioDB::LoadStations(FILE *fpFile)
{
	// Seek the location of the file pointer following the station data.
	if(fseek(fpFile, RADIODB_FREQ_TABLE_DATA_OFFSET, SEEK_SET))
	{
		return false;	// Error, cannot seek position.
	}

	// Now read the start offset for this table.
	int nFreqOffset = 0;
	if((fread((char *)& nFreqOffset, sizeof(nFreqOffset), 1, fpFile)) != 1)
	{
		return false;	// Error, cannot read file.
	}
   if (!m_bool_intel_format)
   	nFreqOffset = ntohl(nFreqOffset);

	// Allocate memory for the station data.
	// Note that this includes space for one extra 'NULL' station to serve as
	// and 'end-of-block' sentinel.
	size_t nStationsSize = nFreqOffset - RADIODB_STATIONS_OFFSET + sizeof(rdbStation);
	if((m_pStations = (rdbStation *)malloc(nStationsSize)) == NULL)
	{
		return false;	// Error, not enough memory.
	}
	memset(m_pStations, '\0', nStationsSize);

	// Seek the station data.

	if(fseek(fpFile, RADIODB_STATIONS_OFFSET, SEEK_SET))
	{
		return false;	// Error, cannot seek position.
	}

	// Load the station data.

	if((fread(m_pStations, nFreqOffset - RADIODB_STATIONS_OFFSET, 1, fpFile)) != 1)
	{
		return false;	// Error, cannot read file.
	}

	return true;
}

//
// CIosRadioDB::SwapStationData
// Perform all byte-swapping for the station records of the radio db
// Entry   : nothing
// Returns : nothing
//
void CIosRadioDB::SwapStationData(void)
{
   rdbStation  *pStation   =  m_pStations;

   while(pStation)
   {
      // convert common header data
      pStation->nIdent     =  ntohl(pStation->nIdent);
      if (!m_bool_intel_format)
      {
         pStation->nICAO   =  ntohl(pStation->nICAO);
         pStation->nType1  =  ntohl(pStation->nType1);
         pStation->nType2  =  ntohl(pStation->nType2);
         pStation->nLat    =  ntohl(pStation->nLat);
         pStation->nLon    =  ntohl(pStation->nLon);
      }

      // convert type-specific data
      switch(pStation->nType1 & 0x7ce0fc00)
      {
         case rdbType1APT:
         {
            if (!m_bool_intel_format)
            {
               rdbAPT   *pAPT    =  (rdbAPT *)pStation;
               pAPT->sMvarX10    =  ntohs(pAPT->sMvarX10);     // short sMvarX10;
               pAPT->sElev       =  ntohs(pAPT->sElev);        // short sElev;
               pAPT->n3Id        =  ntohl(pAPT->n3Id);         // short n3Id;
               pAPT->Spare1      =  ntohs(pAPT->Spare1);       // short Spare1;
            }
         }
         break;

         case rdbType1AWM:
         {
            if (!m_bool_intel_format)
            {
               rdbAWM   *pAWM    =  (rdbAWM *)pStation;
               pAWM->sMvarX10    =  ntohs(pAWM->sMvarX10);     // short sMvarX10;
               pAWM->sElev       =  ntohs(pAWM->sElev);        // short sElev;
               pAWM->sMorseCode  =  ntohs(pAWM->sMorseCode);   // short sMorseCode;
               pAWM->sHdgX10     =  ntohs(pAWM->sHdgX10);      // short sHdgX10;
            }
         }
         break;

         case rdbType1COM:
         {
            if (!m_bool_intel_format)
            {
               rdbCOM   *pCOM    =  (rdbCOM *)pStation;
               pCOM->sMvarX10    =  ntohs(pCOM->sMvarX10);     // short sMvarX10;
               pCOM->sElev       =  ntohs(pCOM->sElev);        // short sElev;
               pCOM->sFreqX100   =  ntohs(pCOM->sFreqX100);    // short sFreqX100;
               pCOM->sSpare      =  ntohs(pCOM->sSpare);       // short sSpare;
            }
         }
         break;

         case rdbType1VOR:
         case rdbType1DME:
         case rdbType1TAC:
         case rdbType1VORDME:
         case rdbType1VORTAC:
         case rdbType1DMETAC:
         case rdbType1VORDMETAC:
         {
            if (!m_bool_intel_format)
            {
               rdbVHF   *pVHF    =  (rdbVHF *)pStation;
               pVHF->sMvarX10    =  ntohs(pVHF->sMvarX10);     // short sMvarX10;
               pVHF->sElev       =  ntohs(pVHF->sElev);        // short sElev;
               pVHF->sFreqX100   =  ntohs(pVHF->sFreqX100);    // short sFreqX100;
               pVHF->sDeclX10    =  ntohs(pVHF->sDeclX10);     // short sDeclX10;
            }
         }
         break;

         case rdbType1ENR:
         {
            if (!m_bool_intel_format)
            {
               rdbENR   *pENR    =  (rdbENR *)pStation;
               // swap the rest of the lat/lons - note that the first was swapped as part of the common header
               for(int i = 1; i < 6; i++)
               {
                  pENR->nLatLon[i][0]  =  ntohl(pENR->nLatLon[i][0]);  // int nLatLon[6][2];
                  pENR->nLatLon[i][1]  =  ntohl(pENR->nLatLon[i][1]);
               }
            }
         }
         break;

         case rdbType1HP:
         {
            if (!m_bool_intel_format)
            {
               rdbHP    *pHP     =  (rdbHP *)pStation;
               pHP->sCourseX10   =  ntohs(pHP->sCourseX10);    // short sCourseX10;
               pHP->sLegLength   =  ntohs(pHP->sLegLength);    // short sLegLength;
               pHP->sSpare       =  ntohs(pHP->sSpare);        // short sSpare;
               pHP->sSpare2      =  ntohs(pHP->sSpare2);       // short sSpare2;
            }
         }
         break;

         case rdbType1ILS:
         case rdbType1ILSDME:
         {
            rdbILS   *pILS       =  (rdbILS *)pStation;
            pILS->nAptIdent      =  ntohl(pILS->nAptIdent);    // int nAptIdent;
            pILS->nRwyIdent      =  ntohl(pILS->nRwyIdent);    // int nRwyIdent;

            if (!m_bool_intel_format)
            {
               pILS->sMvarX10    =  ntohs(pILS->sMvarX10);     // short sMvarX10;
               pILS->sElev       =  ntohs(pILS->sElev);        // short sElev;
               pILS->sFreqX100   =  ntohs(pILS->sFreqX100);    // short sFreqX100;
               pILS->sThdgX10    =  ntohs(pILS->sThdgX10);     // short sThdgX10;
               pILS->sLgd        =  ntohs(pILS->sLgd);         // short sLgd;
               pILS->sLbwX100    =  ntohs(pILS->sLbwX100);     // short sLbwX100;
               pILS->sGsaX100    =  ntohs(pILS->sGsaX100);     // short sGsaX100;
               pILS->sGsw        =  ntohs(pILS->sGsw);         // short sGsw;
               pILS->nDmeLat     =  ntohl(pILS->nDmeLat);      // int nDmeLat;
               pILS->nDmeLon     =  ntohl(pILS->nDmeLon);      // int nDmeLon;
               pILS->nImLat      =  ntohl(pILS->nImLat);       // int nImLat;
               pILS->nImLon      =  ntohl(pILS->nImLon);       // int nImLon;
               pILS->nMmLat      =  ntohl(pILS->nMmLat);       // int nMmLat;
               pILS->nMmLon      =  ntohl(pILS->nMmLon);       // int nMmLon;
               pILS->nOmLat      =  ntohl(pILS->nOmLat);       // int nOmLat;
               pILS->nOmLon      =  ntohl(pILS->nOmLon);       // int nOmLon;
               pILS->sGsDist     =  ntohs(pILS->sGsDist);      // short sGsDist;
               pILS->spare       =  ntohs(pILS->spare);        // short spare;
            }
         }
         break;

         case rdbType1MLS:
         {
            rdbMLS   *pMLS       =  (rdbMLS *)pStation;
            pMLS->nAptIdent      =  ntohl(pMLS->nAptIdent);    // int nAptIdent;
            pMLS->nRwyIdent      =  ntohl(pMLS->nRwyIdent);    // int nRwyIdent;
            
            if (!m_bool_intel_format)
            {
               pMLS->sMvarX10    =  ntohs(pMLS->sMvarX10);     // short sMvarX10;
               pMLS->sElev       =  ntohs(pMLS->sElev);        // short sElev;
               pMLS->sFreqX100   =  ntohs(pMLS->sFreqX100);    // short sFreqX100;
               pMLS->sThdgX10    =  ntohs(pMLS->sThdgX10);     // short sThdgX10;
               pMLS->sAedLgd     =  ntohs(pMLS->sAedLgd);      // short sAedLgd;
               pMLS->sLrApa      =  ntohs(pMLS->sLrApa);       // short sLrApa;
               pMLS->sLrAca      =  ntohs(pMLS->sLrAca);       // short sLrAca;
               pMLS->sEasGsw     =  ntohs(pMLS->sEasGsw);      // short sEasGsw;
               pMLS->sNeaGsa     =  ntohs(pMLS->sNeaGsa);      // short sNeaGsa;
               pMLS->sMinGpa     =  ntohs(pMLS->sMinGpa);      // short sMinGpa;
               pMLS->sBatadLgd   =  ntohs(pMLS->sBatadLgd);    // short sBatadLgd;
               pMLS->sLrBpa      =  ntohs(pMLS->sLrBpa);       // short sLrBpa;
               pMLS->sLrBca      =  ntohs(pMLS->sLrBca);       // short sLrBca;
               pMLS->sTbabX10    =  ntohs(pMLS->sTbabX10);     // short sTbabX10;
               pMLS->sGsd        =  ntohs(pMLS->sGsd);         // short sGsd;
               pMLS->sSpare      =  ntohs(pMLS->sSpare);       // short sSpare;  
            }
         }
         break;

         case rdbType1NDB:
         {
            if (!m_bool_intel_format)
            {
               rdbNDB   *pNDB    =  (rdbNDB *)pStation;
               pNDB->sMvarX10    =  ntohs(pNDB->sMvarX10);     // short sMvarX10;
               pNDB->sElev       =  ntohs(pNDB->sElev);        // short sElev;
               pNDB->sFreqX100   =  ntohs(pNDB->sFreqX100);    // short sFreqX100;
               pNDB->sSpare      =  ntohs(pNDB->sSpare);       // short sSpare;
            }
         }
         break;

         case rdbType1RWY:
         {
            rdbRWY   *pRWY       =  (rdbRWY *)pStation;
            pRWY->nLocIdent      =  ntohl(pRWY->nLocIdent);    // int nLocIdent;
            pRWY->nAptIdent      =  ntohl(pRWY->nAptIdent);    // int nAptIdent;

            if (!m_bool_intel_format)
            {
               pRWY->sThdgx10    =  ntohs(pRWY->sThdgx10);     // short sThdgx10;
               pRWY->sLen        =  ntohs(pRWY->sLen);         // short sLen;
               pRWY->sDispThr    =  ntohs(pRWY->sDispThr);     // short sDispThr;
               pRWY->sWidth      =  ntohs(pRWY->sWidth);       // short sWidth;
            }
         }
         break;

         case rdbType1WPT:
         case rdbType1TWPT:
         {
            rdbWPT   *pWPT    =  (rdbWPT *)pStation;
            pWPT->sMvarX10    =  ntohs(pWPT->sMvarX10);        // short sMvarX10;
            pWPT->sSpare      =  ntohs(pWPT->sSpare);          // short sSpare;
            pWPT->sSpare2     =  ntohs(pWPT->sSpare2);         // short sSpare2;
            pWPT->sSpare3     =  ntohs(pWPT->sSpare3);         // short sSpare3;
         }
         break;

         default:
         {
         // assert(0);  // need to implement all station types
         }
         break;
      }
      pStation =  StationNext(pStation);
   }
}

// some constants for distance calculations
const double dPi = 3.141592654;
const double dDeg2Rad = dPi/180.0;
const double dRad2Deg = 180.0/dPi;
const double dRad2Nm = (180.0*60.0)/dPi;

//
// DeltaLatLonDegrees
// Utility function for calculating the distance between to lat/lon points
// Entry   : two lat/lon pairs (each in decimal degrees)
// Returns : distance (decimal degrees)
//
static inline double DeltaLatLonDegrees(double lat1, double lon1, double lat2, double lon2)
{
	lat1 *= dDeg2Rad;
	lat2 *= dDeg2Rad;
	lon1 *= dDeg2Rad;
	lon2 *= dDeg2Rad;
	return acos(sin(lat1)*sin(lat2)+cos(lat1)*cos(lat2)*cos(lon1-lon2)) * dRad2Deg;
}

//
// IsNavAid
// Utility function for determining if a station is a NavAid
// Entry   : station of interest
// Returns : true/false
//
static inline bool IsNavAid(rdbStation *pStation)
{
	ASSERT(pStation);
	switch(pStation->nType1 & 0xfffffe00u)
	{
		case rdbType1ILSDME:
		case rdbType1ILS:
		case rdbType1NDB:
		case rdbType1VOR:
		case rdbType1DME:
      case rdbType1TAC:
		case rdbType1VORDME:
		case rdbType1VORTAC:
		case rdbType1DMETAC:
		case rdbType1VORDMETAC:
			return true;
		default:
			return false;
	}
	return false;
}

//
// CIosRadioDB::NearestNavAid
// Locate the nearest Ident-specified NavAid station from a specific lat/lon
// Entry   : ident, lat/lon
// Returns : station if found
//
rdbStation *CIosRadioDB::NearestNavAid(int nIdent, double dLat, double dLon, int & nNumDups)
{
	rdbStation *pNearStation = 0;
	nNumDups = 0;
	double dMinDist = 9999.0;

	// locate the first station of this ident
	rdbStation*	pStation = Ident(nIdent);
	while(pStation)
	{
		// we only want NavAid stations
		if (IsNavAid(pStation))
		{
			// found another one
			nNumDups++;

			// check the distance
			double dDistance = DeltaLatLonDegrees(
									dLat,
									dLon,
									BamsToDegrees(pStation->nLat),
									BamsToDegrees(pStation->nLon));
			if (dDistance < dMinDist)
			{
				// found a closer one
				pNearStation = pStation;
				dMinDist = dDistance;
			}
		}
		pStation = IdentNext();
	}

	// return the closest one (or NULL if not found)
	return pNearStation;
}

//
// CIosRadioDB::AdjacentNavAids
// Locate all NavAid stations within some specified distance from a selected station
// Entry   : current station, distance and output vector
// Returns : nothing
//
void CIosRadioDB::AdjacentNavAids(rdbStation *pCurrStation, double dDistance, VecStation & vecpStations)
{
	vecpStations.empty();

	// verify that we're working with a NavAid station
	if (!IsNavAid(pCurrStation))
	{
		return;
	}

	// locate the first station for this ident
	rdbStation*	pStation = Ident(pCurrStation->nIdent);
	while(pStation)
	{
		if (pStation != pCurrStation && IsNavAid(pStation))
		{
			// check the distance from the specified station
			double dThisDistance = DeltaLatLonDegrees(
						BamsToDegrees(pCurrStation->nLat),
						BamsToDegrees(pCurrStation->nLon),
						BamsToDegrees(pStation->nLat),
						BamsToDegrees(pStation->nLon)) * dDeg2Rad * dRad2Nm ;
			if (dThisDistance <= dDistance)
			{
				// found one
				vecpStations.push_back(pStation);
			}
		}
		pStation = IdentNext();
	}
	return;
}

//
// CIosRadioDB::AdjacentStations
// Locate all desired stations within some specified distance from a given lat/lon
// and determine the distance for each located station.  
// Entry   : search radius, lat/lon, station type
// Returns : nothing
//
void CIosRadioDB::AdjacentStations(double dLatitude, double dLongitude,  double dSearchRadius, VecStation & vecpStations, StationDistances & vecStationDistances, int nStationType)
{
	int nCount=0;
	vecpStations.empty();
	vecStationDistances.empty();

	// locate the first station
	rdbStation*	pStation = StationFirst();
	
	while(pStation)
	{
		if((pStation->nType1 & 0x7ce0fc00) == nStationType)
		{
			// check the distance from the specified station
			double dThisDistance = DeltaLatLonDegrees(
											dLatitude,
											dLongitude,
											BamsToDegrees(pStation->nLat),
											BamsToDegrees(pStation->nLon)) * dDeg2Rad * dRad2Nm ;

			if (dThisDistance <= dSearchRadius)
			{
				nCount++;

				// ATS - Need to bring these together with a STL map.
				// Found one.
				vecpStations.push_back(pStation);

				// Keep track how far away it was.
				vecStationDistances.push_back(dThisDistance);
			}
		}
		pStation = StationNext(pStation);
	}
	return;
}
