// AreaMap.cpp: implementation of the CAreaMap class.
//
//////////////////////////////////////////////////////////////////////

#include "..\core\stdafx.h"
#include "AreaMap.h"
#include "..\core\IOAction.h"
#include "..\core\DataConversion.h"
#include "..\comms\CommsShared.h"
#include "CommUtils.h"

typedef  struct
{
   char  chIdent[4];
   char  chICAO[4];
   int   nType1;
   int   nType2;
   int   nLat;
   int   nLon;
   short sMvarX10;
   short sElev;
}  rdbGEN;

//
// Generic Station Data Overlay
//

typedef union
{
   rdbAPT      apt;
   rdbAWM      awm;
   rdbCOM      com;
   rdbENR      enr;
   rdbHP       hp;
   rdbILS      ils;
   rdbMLS      mls;
   rdbNDB      ndb;
   rdbRWY      rwy;
   rdbStation  stn;
   rdbVHF      vhf;
   rdbWPT      wpt;
   rdbGEN      gen;
}  rdbGNRC;


const float SPIDER_WEB_OUTER_RADIUS       =  2.0f;
const float SPIDER_WEB_INNER_RADIUS       =  0.0f;
const int   SPIDER_WEB_NUMBER_OF_RINGS    =  10;
const int   SPIDER_WEB_NUMBER_OF_RADIALS  =  36;

static const unsigned char  VICTOR_AIRWAYS   =  1;
static const unsigned char  JET_AIRWAYS      =  2;
static const unsigned char  ILSDME           =  3;
static const unsigned char  PRACTICE_AREA    =  4;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CAreaMap::CAreaMap():m_symbolInfoList(128), m_symbolLabelInfoList(128), m_identWindowInfoList(128)
,m_bLeftMouseDown(false)
,m_mouse_x_position_lpv(0)
,m_mouse_y_position_lpv(0)
{

   m_exPtUpperLeft      =  CExtentsPoint(CPoint(0,0));
   m_exPtLowerRight     =  CExtentsPoint(CPoint(640,640));

   m_stlStrWidgetName   =  _FSI_STL::string("Area_Map");

   {
      CVariant *pVariant = CCommsShared::FindLocalVariant("area_map_scale");
      ASSERT(pVariant != NULL);
      if (pVariant != NULL)
         m_scale = 7200.0f / (float)(*pVariant);
   }
   m_scale += 0.01f;
   m_scale_cv                       =  m_scale;
   m_spider_web_scale               =  -1.0f;

   m_area_map_spider_web            =  SPIDER_WEB_OFF;
   m_area_map_spider_web_cv         =  m_area_map_spider_web;

   m_sym_disp_lead_ac_track_on      =  false;
   m_sym_disp_lead_ac_track_on_cv   =  m_sym_disp_lead_ac_track_on;

   m_sym_disp_wing_ac_track_on      =  false;
   m_sym_disp_wing_ac_track_on_cv   =  m_sym_disp_wing_ac_track_on ;

   m_sym_disp_ils_on                =  false;
   m_sym_disp_ils_on_cv             =  m_sym_disp_ils_on;

   m_sym_disp_vor_on                =  false;
   m_sym_disp_vor_on_cv             =  m_sym_disp_vor_on;

   m_sym_disp_markers_on            =  false;
   m_sym_disp_markers_on_cv         =  m_sym_disp_markers_on;

   m_sym_disp_ndb_on                =  false;
   m_sym_disp_ndb_on_cv             =  m_sym_disp_ndb_on;

   m_sym_disp_victor_routes_on      =  false;
   m_sym_disp_victor_routes_on_cv   =  m_sym_disp_victor_routes_on;

   m_sym_disp_jet_routes_on         =  false;
   m_sym_disp_jet_routes_on_cv      =  m_sym_disp_jet_routes_on;

   m_sym_disp_waypoints_on          =  false;
   m_sym_disp_waypoints_on_cv       =  m_sym_disp_waypoints_on;

   m_sym_disp_waypoints_text_on     =  true;
   m_sym_disp_waypoints_text_on_cv  =  m_sym_disp_waypoints_text_on;

   m_sym_disp_airports_on           =  false;
   m_sym_disp_airports_on_cv        =  m_sym_disp_airports_on;

   m_sym_disp_ils_text_on           =  false;
   m_sym_disp_ils_text_on_cv        =  m_sym_disp_ils_text_on;

   m_sym_disp_vor_text_on           =  false;
   m_sym_disp_vor_text_on_cv        =  m_sym_disp_vor_text_on;

   m_sym_disp_ndb_text_on           =  false;
   m_sym_disp_ndb_text_on_cv        =  m_sym_disp_ndb_text_on;

   m_sym_disp_airport_text_on       =  false;
   m_sym_disp_airport_text_on_cv    =  m_sym_disp_airport_text_on;

   m_sym_disp_lat_long_grid_on      =  false;
   m_sym_disp_lat_long_grid_on_cv   =  m_sym_disp_lat_long_grid_on;

   m_sym_disp_mouse_lat_long_on     =  false;
   m_sym_disp_mouse_lat_long_on_cv  =  m_sym_disp_mouse_lat_long_on;

   m_sym_disp_mouse_lat_long_decimal_degree     =  false;
   m_sym_disp_mouse_lat_long_decimal_degree_cv  =  m_sym_disp_mouse_lat_long_decimal_degree;

   m_sym_disp_weather_areas_on      =  false;
   m_sym_disp_weather_areas_on_cv   =  m_sym_disp_weather_areas_on;

   m_sym_disp_custom_obstacles_on               =  false;
   m_sym_disp_custom_obstacles_on_cv            =  m_sym_disp_custom_obstacles_on;

   m_sym_disp_custom_obstacles_text_on          =  false;
   m_sym_disp_custom_obstacles_text_on_cv       =  m_sym_disp_custom_obstacles_text_on;

   m_sym_disp_custom_practice_areas_on          =  false;
   m_sym_disp_custom_practice_areas_on_cv       =  m_sym_disp_custom_practice_areas_on;

   m_sym_disp_custom_practice_areas_text_on     =  false;
   m_sym_disp_custom_practice_areas_text_on_cv  =  m_sym_disp_custom_practice_areas_text_on;

   m_sym_disp_custom_lines_on                   =  false;
   m_sym_disp_custom_lines_on_cv                =  m_sym_disp_custom_lines_on;

   m_sym_disp_custom_lines_text_on              =  false;
   m_sym_disp_custom_lines_text_on_cv           =  m_sym_disp_custom_lines_text_on;

   m_sym_disp_storm_on                          =  false;
   m_sym_disp_storm_on_cv                       =  m_sym_disp_storm_on;

   m_sym_disp_storm_text_on                     =  false;
   m_sym_disp_storm_text_on_cv                  =  m_sym_disp_storm_text_on;

   m_spider_web_magvar              =  0.0f;
   m_spider_web_magvar_cv           =  m_spider_web_magvar;

   m_bClearLead                     =  false;
   m_bClearWing                     =  false;

   m_active_airport_defaulted       = false;

   m_mouse_latitude.Value(0.0f);
   m_mouse_longitude.Value(0.0f);
   
   m_symbolInfoList.RemoveAll();
   m_symbolLabelInfoList.RemoveAll();

   m_listGraphicalElementVars.clear();
   m_listGraphicalElementVars.push_back("area_map_scale");
   m_listGraphicalElementVars.push_back("area_map_scale_auto");
   m_listGraphicalElementVars.push_back("area_map_latitude_center");
   m_listGraphicalElementVars.push_back("area_map_longitude_center");
   m_listGraphicalElementVars.push_back("area_map_center");

   m_listGraphicalElementVars.push_back("ac_heading_lead");
   m_listGraphicalElementVars.push_back("ac_latitude_lead");
   m_listGraphicalElementVars.push_back("ac_longitude_lead");
   m_listGraphicalElementVars.push_back("ac_altitude_lead");

   m_listGraphicalElementVars.push_back("ac_heading_wing");
   m_listGraphicalElementVars.push_back("ac_latitude_wing");
   m_listGraphicalElementVars.push_back("ac_longitude_wing");
   m_listGraphicalElementVars.push_back("ac_altitude_wing");

   m_listGraphicalElementVars.push_back("Reposition Select");
   m_listGraphicalElementVars.push_back("Reposition Latitude");
   m_listGraphicalElementVars.push_back("Reposition Longitude");
   m_listGraphicalElementVars.push_back("touch_reposition");

   m_listGraphicalElementVars.push_back("Storm_Location_Select");
   m_listGraphicalElementVars.push_back("Storm_Latitude");
   m_listGraphicalElementVars.push_back("Storm_Longitude");
   m_listGraphicalElementVars.push_back("Storm_Name");
   m_listGraphicalElementVars.push_back("Storm_Length");
   m_listGraphicalElementVars.push_back("Storm_Width");
   m_listGraphicalElementVars.push_back("Storm_Active");
   m_listGraphicalElementVars.push_back("touch_storm_location");
   
   m_listGraphicalElementVars.push_back("clear_track");
   m_listGraphicalElementVars.push_back("area_map_center_airport_id");
   m_listGraphicalElementVars.push_back("area_map_center_airport");
   m_listGraphicalElementVars.push_back("area_map_center_aircraft");
   m_listGraphicalElementVars.push_back("area_map_center_cursor");
   m_listGraphicalElementVars.push_back("area_map_center_auto");

   m_listGraphicalElementVars.push_back("spider_web_center_airport_id");
   m_listGraphicalElementVars.push_back("spider_web_center_navaid_id");
   m_listGraphicalElementVars.push_back("spider_web_latitude_center");
   m_listGraphicalElementVars.push_back("spider_web_longitude_center");
   m_listGraphicalElementVars.push_back("active_airport");
   m_listGraphicalElementVars.push_back("active_runway");
   m_listGraphicalElementVars.push_back("area_map_spider_web");
   m_listGraphicalElementVars.push_back("spider_web_magvar");
   m_listGraphicalElementVars.push_back("Magvar_Latitude");
   m_listGraphicalElementVars.push_back("Magvar_Longitude");
   m_listGraphicalElementVars.push_back("spider_web_center_on_active_airport");

   m_listGraphicalElementVars.push_back("sym_disp_ils_on");
   m_listGraphicalElementVars.push_back("sym_disp_vor_on");
   m_listGraphicalElementVars.push_back("sym_disp_markers_on");
   m_listGraphicalElementVars.push_back("sym_disp_ndb_on");
   m_listGraphicalElementVars.push_back("sym_disp_victor_routes_on");
   m_listGraphicalElementVars.push_back("sym_disp_jet_routes_on");
   m_listGraphicalElementVars.push_back("sym_disp_waypoints_on");
   m_listGraphicalElementVars.push_back("sym_disp_airports_on");
   m_listGraphicalElementVars.push_back("sym_disp_ils_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_vor_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_ndb_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_waypoints_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_airport_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_lead_ac_track_on");
   m_listGraphicalElementVars.push_back("sym_disp_wing_ac_track_on");
   m_listGraphicalElementVars.push_back("sym_disp_lat_long_grid_on");
   m_listGraphicalElementVars.push_back("sym_disp_mouse_lat_long_on");
   m_listGraphicalElementVars.push_back("sym_disp_mouse_lat_long_decimal_degree");
   m_listGraphicalElementVars.push_back("sym_disp_weather_areas_on");
   m_listGraphicalElementVars.push_back("sym_disp_custom_obstacles_on");
   m_listGraphicalElementVars.push_back("sym_disp_custom_obstacles_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_custom_practice_areas_on");
   m_listGraphicalElementVars.push_back("sym_disp_custom_practice_areas_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_custom_lines_on");
   m_listGraphicalElementVars.push_back("sym_disp_custom_lines_text_on");
   m_listGraphicalElementVars.push_back("sym_disp_storm_on");
   m_listGraphicalElementVars.push_back("sym_disp_storm_text_on");

   m_listGraphicalElementVars.push_back("Weather Area 1 latitude");
   m_listGraphicalElementVars.push_back("Weather Area 1 longitude");
   m_listGraphicalElementVars.push_back("Weather Area 1 radius");
   m_listGraphicalElementVars.push_back("Weather Area 1 loaded");
   m_listGraphicalElementVars.push_back("Weather Area 1 active");

   m_listGraphicalElementVars.push_back("Weather Area 2 latitude");
   m_listGraphicalElementVars.push_back("Weather Area 2 longitude");
   m_listGraphicalElementVars.push_back("Weather Area 2 radius");
   m_listGraphicalElementVars.push_back("Weather Area 2 loaded");
   m_listGraphicalElementVars.push_back("Weather Area 2 active");

   m_listGraphicalElementVars.push_back("Weather Area 3 latitude");
   m_listGraphicalElementVars.push_back("Weather Area 3 longitude");
   m_listGraphicalElementVars.push_back("Weather Area 3 radius");
   m_listGraphicalElementVars.push_back("Weather Area 3 loaded");
   m_listGraphicalElementVars.push_back("Weather Area 3 active");

   m_listGraphicalElementVars.push_back("Weather Area 4 latitude");
   m_listGraphicalElementVars.push_back("Weather Area 4 longitude");
   m_listGraphicalElementVars.push_back("Weather Area 4 radius");
   m_listGraphicalElementVars.push_back("Weather Area 4 loaded");
   m_listGraphicalElementVars.push_back("Weather Area 4 active");

   m_listOverlays.push_back("Station_Kill_Overlay");
   m_listOverlays.push_back("Slews_Overlay");

   m_listColorSelectable.push_back("ACTIVE_ILS");
   m_listColorSelectable.push_back("NAVAID NO KILL BACKGROUND");
   m_listColorSelectable.push_back("NAVAID NO KILL FOREGROUND");
   m_listColorSelectable.push_back("NAVAID PARTIAL KILL BACKGROUND");
   m_listColorSelectable.push_back("NAVAID PARTIAL KILL FOREGROUND");
   m_listColorSelectable.push_back("NAVAID TOTAL KILL BACKGROUND");
   m_listColorSelectable.push_back("NAVAID TOTAL KILL FOREGROUND");

   m_touch_reposition_requested        =  false;
   m_touch_storm_location_requested    =  false;
   m_spider_web_at_cursor_requested    =  false;
   m_map_center_at_cursor_requested    =  false;
   m_map_center_at_aircraft_requested  =  true;
   m_area_map_center_on_airport        =  false;

   m_area_map_auto_scale         =  false;
   m_area_map_auto_scale_cv      =  m_area_map_auto_scale;
   m_area_map_auto_center        =  false;     
   m_area_map_auto_center_cv     =  m_area_map_auto_center;     


   m_varDisplayStationKill.Value(false);
   m_varDisplaySlews.Value(false);
   m_varDisplayStorms.Value(false);

   for (int ii =  0; ii < MAX_NUMBER_OF_LISTS; ii++)
   {
      m_int_lists[ii]   =  -1;
   }

}

CAreaMap::~CAreaMap()
{
}

BEGIN_MESSAGE_MAP(CAreaMap, COpenGLWidget)
   //{{AFX_MSG_MAP(CAreaMap)
   ON_WM_MOUSEMOVE()
   ON_WM_LBUTTONUP()
	ON_WM_MOUSEWHEEL()
	ON_WM_LBUTTONDOWN()
	//}}AFX_MSG_MAP
   ON_MESSAGE(WM_MOUSELEAVE,OnMouseLeave)
END_MESSAGE_MAP()

void  CAreaMap::InitAreaMapCenterAirport(void)
{
   CCommUtils::SetVariable( LocalComms, "area_map_center_airport_id", (char *)(LPCTSTR)m_active_airport_id);
}

void  CAreaMap::InitSpiderWebCenterAirport(void)
{
   CCommUtils::SetVariable( LocalComms, "spider_web_center_airport_id", (char *)(LPCTSTR)m_active_airport_id);
}

void  CAreaMap::Setup(void)
{
   COpenGLMap::Initialize(GetSafeHwnd());

   LookupCustomData();

   #ifdef   HAF
      LoadTacanChannelData();
   #endif

   m_webChevron   =  "V";

   // Changed 0,0,0's to 1,1,1's to support reverse on printing
   latitudeLabel.SetForegroundColor(0,    96,   0);   // Green
   latitudeLabel.SetBackgroundColor(true, 1, 1, 1);
   longitudeLabel.SetForegroundColor(0,   96,   0);   // Green
   longitudeLabel.SetBackgroundColor(true, 1, 1, 1);

   // Changed background color to 1,1,1 to support reverse on print
   webLabel.SetForegroundColor(96, 192, 96);
   webLabel.SetBackgroundColor(true, 1, 1, 1);

   mouse_position.SetForegroundColor(255, 255, 0);

   m_circle.ArraySize(360);

   m_aircraftLeadTrack.UseAltitude(false);
   m_aircraftLeadTrack.TypeOfTrack(AreaMapLead);
   m_aircraftLeadTrack.SetColor(0, 255, 255);
   aircraftLead.SetForegroundColor(0, 255, 255);
   m_aircraft_lead.Format("%c", THE_PLANE);

   m_aircraftWingTrack.UseAltitude(false);
   m_aircraftWingTrack.TypeOfTrack(AreaMapWing);
   m_aircraftWingTrack.SetColor(255, 0, 255);
   // Changed 255,255,255 to 254,254,254 to support printing
   aircraftWing.SetForegroundColor(254, 0, 254);
   m_aircraft_wing.Format("%c", THE_PLANE);


   for (int ii =  0; ii < MAX_NUMBER_OF_LISTS; ii++)
   {
      if (m_int_lists[ii]  != -1)
         glDeleteLists(m_int_lists[ii], 1);
   }

   m_int_lists[BIG_UNUSUAL_BOX_OUTLINE]   =  glGenLists(1);
   glNewList(m_int_lists[BIG_UNUSUAL_BOX_OUTLINE], GL_COMPILE);
   {
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      glRGB(1, 1, 1);   // Changed 0,0,0 to 1,1,1 to support printing
      glBegin(GL_POLYGON);
         glVertex2f(-0.45f,    0.025f);
         glVertex2f( 0.45f,    0.025f);
         glVertex2f( 0.45f,   -0.055f);
         glVertex2f( 0.075f,  -0.055f);
         glVertex2f( 0.00f,   -0.10f);
         glVertex2f(-0.075f,  -0.055f);
         glVertex2f(-0.45f,   -0.055f);
      glEnd();
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

      glRGB(255, 0, 0);
      glBegin(GL_LINE_LOOP);
         glVertex2f(-0.45f,    0.025f);
         glVertex2f( 0.45f,    0.025f);
         glVertex2f( 0.45f,   -0.055f);
         glVertex2f( 0.075f,  -0.055f);
         glVertex2f( 0.00f,   -0.10f);
         glVertex2f(-0.075f,  -0.055f);
         glVertex2f(-0.45f,   -0.055f);
      glEnd();
   }
   glEndList();

   m_int_lists[SMALL_UNUSUAL_BOX_OUTLINE]   =  glGenLists(1);
   glNewList(m_int_lists[SMALL_UNUSUAL_BOX_OUTLINE], GL_COMPILE);
   {
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      glRGB(1, 1, 1);   // Changed 0,0,0 to 1,1,1 to support printing
      glBegin(GL_POLYGON);
         glVertex2f(-0.15f,    0.025f);
         glVertex2f( 0.15f,    0.025f);
         glVertex2f( 0.15f,   -0.055f);
         glVertex2f( 0.075f,  -0.055f);
         glVertex2f( 0.00f,   -0.10f);
         glVertex2f(-0.075f,  -0.055f);
         glVertex2f(-0.15f,   -0.055f);
      glEnd();
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

      glRGB(255, 0, 0);
      glBegin(GL_LINE_LOOP);
         glVertex2f(-0.15f,    0.025f);
         glVertex2f( 0.15f,    0.025f);
         glVertex2f( 0.15f,   -0.055f);
         glVertex2f( 0.075f,  -0.055f);
         glVertex2f( 0.00f,   -0.10f);
         glVertex2f(-0.075f,  -0.055f);
         glVertex2f(-0.15f,   -0.055f);
      glEnd();
   }
   glEndList();

   m_int_lists[SPIDER_WEB]    =  glGenLists(1);

}

void  CAreaMap::Render(void)
{

   BeginDrawing();

   // Save these for later use....
   glGetDoublev(GL_PROJECTION_MATRIX, m_projMatrix);
   glGetIntegerv(GL_VIEWPORT, m_viewport);


   glPushMatrix();
   {
      if (m_aspect_ratio  <= 1.0f)
         glScalef(m_aspect_ratio, 1.0f, 1.0f);
      else if (m_aspect_ratio != 0.0f)
         glScalef(1.0f, 1.0f  /  m_aspect_ratio, 1.0f);

      SetupPerspective();
      MoveViewingPosition();
      DrawEarth();

      // Rotate to the map center.
      glRotated(m_area_map_latitude_center,  1.0,  0.0,  0.0);
      glRotated(m_area_map_longitude_center, 0.0,  1.0,  0.0);

      if (m_area_map_spider_web)             // Display the spider web if necessary.
         DrawSpiderWeb();
      else if (m_sym_disp_lat_long_grid_on)  // Display the Latitude/Longitude grid if necessary.
         DrawLatitudeLongitudeGrid();

      if (m_sym_disp_weather_areas_on)
         DrawWeatherAreas();

	  DrawStorm();

      DrawSymbols();

      DrawLeadTrack();

      DrawWingTrack();

      DrawMousePosition();
   }
   glPopMatrix();


   EndDrawing();

   if (( ! m_active_airport_defaulted ) &&
	   ( ! m_active_airport_id.IsEmpty() ) &&
	   ( m_active_airport_id != "    " ))
   {
	   
      bool SortieWasLoaded = false;
	  CCommUtils::GetVariable(LocalComms, "sortie_was_loaded", SortieWasLoaded);
	  if ( ! SortieWasLoaded )
	  {
         InitAreaMapCenterAirport();
         InitSpiderWebCenterAirport();
         m_active_airport_defaulted = true;
	  }
   }
}


CWidget* CAreaMap::CreateObject()
{
   return new CAreaMap();
}

void CAreaMap::ChangeValue(const CString& rstrElementVar, CChangeValue* pCV)
{
   if (pCV == NULL)
   {
       return;
   }
   CVariant* pVariant = pCV->Variant();

   static   int   lead_track_size     =  0;
   static   int   wing_track_size     =  0;
   double   miles_to_move        =  (7200.0  /  m_scale) /  4.0;

   if (rstrElementVar         == "area_map_scale")
   {
      m_scale_cv  =  *pVariant;

      if (m_scale_cv == 0.0f)
         m_scale_cv  =  7200.0f;

      m_scale_cv  =  7200.0f  /  m_scale_cv;
   }
   else if (rstrElementVar    == "area_map_scale_auto")
   {
      m_area_map_auto_scale_cv   =  *pVariant;
   }

   else if (rstrElementVar    == "ac_heading_lead")
   {
      m_ac_heading_lead_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "ac_latitude_lead")
   {
      m_lead_ac_latitude_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "ac_longitude_lead")
   {
      m_lead_ac_longitude_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "ac_altitude_lead")
   {
      m_lead_ac_altitude_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "ac_heading_wing")
   {
      m_ac_heading_wing_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "ac_latitude_wing")
   {
      m_ac_latitude_cv        = *pVariant;
   }
   else if (rstrElementVar    == "ac_longitude_wing")
   {
      m_ac_longitude_cv       = *pVariant;
   }
   else if (rstrElementVar    == "ac_altitude_wing")
   {
      m_ac_altitude_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "Storm_Latitude")
   {
      m_storm_latitude_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "Storm_Longitude")
   {
      m_storm_longitude_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "Storm_Heading")
   {
      m_storm_heading_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "Storm_Name")
   {
      CString  storm_id((char *)((_FSI_STL::string)(*pVariant)).c_str());
      storm_id.MakeUpper();
	  storm_id.MakeReverse();
      m_storm_name_cv  =  storm_id;
   }
   else if (rstrElementVar    == "Storm_Length")
   {
      m_storm_length_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "Storm_Width")
   {
      m_storm_width_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "Storm_Active")
   {
      m_storm_active_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "clear_track")
   {
      if ((_FSI_STL::string)*pVariant        == "Erase L Track")
      {
         m_bClearLead   =  true;
      }
      else if ((_FSI_STL::string)*pVariant   == "Erase W Track")
      {
         m_bClearWing   =  true;
      }
   }

   else if (rstrElementVar    == "area_map_latitude_center")
   {
      m_area_map_latitude_center_cv    =  *pVariant;
      if (m_area_map_latitude_center_cv   <  90.0  && m_area_map_latitude_center_cv >  -90.0)
      {
         m_area_map_latitude_center_cv =  fmod(m_area_map_latitude_center_cv, 90.0);
      }
      else if (m_area_map_latitude_center_cv >  90.0)
      {
         m_area_map_latitude_center_cv =  90.0;
      }
      else if (m_area_map_latitude_center_cv <  -90.0)
      {
         m_area_map_latitude_center_cv =  -90.0;
      }
   }
   else if (rstrElementVar    == "area_map_longitude_center")
   {
      m_area_map_longitude_center_cv   =  *pVariant;

      if (m_area_map_longitude_center_cv  <  180.0 && m_area_map_longitude_center_cv   >  -180.0)
      {
         m_area_map_longitude_center_cv   =  fmod(m_area_map_longitude_center_cv, 180.0);
      }
      else if (m_area_map_longitude_center_cv   >  180.0)
      {
          m_area_map_longitude_center_cv  =  180.0;
      }
      else if (m_area_map_longitude_center_cv   <  -180.0)
      {
          m_area_map_longitude_center_cv  =  -180.0;
      }
      m_area_map_longitude_center_cv   *= -1.0f;
   }
   else if (rstrElementVar    == "area_map_center_aircraft")
   {
      m_map_center_at_aircraft_requested  =  *pVariant;
   }
   else if (rstrElementVar    == "area_map_center_cursor")
   {
      m_map_center_at_cursor_requested    =  *pVariant;
   }
   else if (rstrElementVar    == "area_map_center_auto")
   {
      m_area_map_auto_center_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "area_map_center_airport")
   {
      m_area_map_center_on_airport        =  *pVariant;
   }
   else if (rstrElementVar    == "area_map_center_airport_id")
   {
      m_airport_id   =  (char *)((_FSI_STL::string)(*pVariant)).c_str();
   }
   else if (rstrElementVar    == "area_map_center")
   {
      if ((_FSI_STL::string)*pVariant        == "NW")
      {
         m_area_map_latitude_center_cv    += miles_to_move  /  NM_PER_DEGREE;
         m_area_map_longitude_center_cv   += miles_to_move  /  NM_PER_DEGREE;
      }
      else if ((_FSI_STL::string)*pVariant   == "N")
      {
         m_area_map_latitude_center_cv    += miles_to_move  /  NM_PER_DEGREE;
      }
      else if ((_FSI_STL::string)*pVariant   == "NE")
      {
         m_area_map_latitude_center_cv    += miles_to_move  /  NM_PER_DEGREE;
         m_area_map_longitude_center_cv   -= miles_to_move  /  NM_PER_DEGREE;
      }
      else if ((_FSI_STL::string)*pVariant   == "SW")
      {
         m_area_map_latitude_center_cv    -= miles_to_move  /  NM_PER_DEGREE;
         m_area_map_longitude_center_cv   += miles_to_move  /  NM_PER_DEGREE;
      }
      else if ((_FSI_STL::string)*pVariant   == "S")
      {
         m_area_map_latitude_center_cv    -= miles_to_move  /  NM_PER_DEGREE;
      }
      else if ((_FSI_STL::string)*pVariant   == "SE")
      {
         m_area_map_latitude_center_cv    -= miles_to_move  /  NM_PER_DEGREE;
         m_area_map_longitude_center_cv   -= miles_to_move  /  NM_PER_DEGREE;
      }
      else if ((_FSI_STL::string)*pVariant   == "E")
      {
         m_area_map_longitude_center_cv   -= miles_to_move  /  NM_PER_DEGREE;
      }
      else if ((_FSI_STL::string)*pVariant   == "W")
      {
         m_area_map_longitude_center_cv   += miles_to_move  /  NM_PER_DEGREE;
      }
   }

   else if (rstrElementVar    == "sym_disp_ils_on")
   {
      m_sym_disp_ils_on_cv             =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_vor_on")
   {
      m_sym_disp_vor_on_cv             =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_markers_on")
   {
      m_sym_disp_markers_on_cv         =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_ndb_on")
   {
      m_sym_disp_ndb_on_cv             =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_victor_routes_on")
   {
      m_sym_disp_victor_routes_on_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_jet_routes_on")
   {
      m_sym_disp_jet_routes_on_cv      =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_waypoints_on")
   {
      m_sym_disp_waypoints_on_cv       =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_airports_on")
   {
      m_sym_disp_airports_on_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_ils_text_on")
   {
      m_sym_disp_ils_text_on_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_vor_text_on")
   {
      m_sym_disp_vor_text_on_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_ndb_text_on")
   {
      m_sym_disp_ndb_text_on_cv        =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_airport_text_on")
   {
      m_sym_disp_airport_text_on_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_waypoints_text_on")
   {
      m_sym_disp_waypoints_text_on_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_lead_ac_track_on")
   {
      m_sym_disp_lead_ac_track_on_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_wing_ac_track_on")
   {
      m_sym_disp_wing_ac_track_on_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_lat_long_grid_on")
   {
      m_sym_disp_lat_long_grid_on_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_mouse_lat_long_on")
   {
      m_sym_disp_mouse_lat_long_on_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_mouse_lat_long_decimal_degree")
   {
      m_sym_disp_mouse_lat_long_decimal_degree_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_weather_areas_on")
   {
      m_sym_disp_weather_areas_on_cv   =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_custom_obstacles_on")
   {
      m_sym_disp_custom_obstacles_on_cv            =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_custom_obstacles_text_on")
   {
      m_sym_disp_custom_obstacles_text_on_cv       =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_custom_practice_areas_on")
   {
      m_sym_disp_custom_practice_areas_on_cv       =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_custom_practice_areas_text_on")
   {
      m_sym_disp_custom_practice_areas_text_on_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_custom_lines_on")
   {
      m_sym_disp_custom_lines_on_cv                =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_custom_lines_text_on")
   {
      m_sym_disp_custom_lines_text_on_cv           =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_storm_on")
   {
      m_sym_disp_storm_on_cv                       =  *pVariant;
   }
   else if (rstrElementVar    == "sym_disp_storm_text_on")
   {
      m_sym_disp_storm_text_on_cv                  =  *pVariant;
   }

   else if (rstrElementVar    == "active_airport")
   {
      CString  airport_id((char *)((_FSI_STL::string)(*pVariant)).c_str());
      airport_id.MakeUpper();

      m_active_airport_id_cv  =  airport_id;

      int   length   =  m_active_airport_id_cv.GetLength();
      for (; length < 4; length++)
      {
         m_active_airport_id_cv  += " ";
      }
   }
   else if (rstrElementVar    == "active_runway")
   {
      CString  runway_id((char *)((_FSI_STL::string)(*pVariant)).c_str());
      runway_id.MakeUpper();

      m_active_runway_id_cv   =  runway_id;

      int   length   =  m_active_runway_id_cv.GetLength();
      for (; length < 4; length++)
      {
         m_active_runway_id_cv   += " ";
      }
   }

   else if (rstrElementVar    == "touch_reposition")
   {
      m_touch_reposition_requested  =  *pVariant;
   }

   else if (rstrElementVar    == "touch_storm_location")
   {
      m_touch_storm_location_requested  =  *pVariant;
   }

   else if (rstrElementVar    == "spider_web_center_airport_id")
   {
      m_spider_web_center_airport_id_cv   =  ((char *)((_FSI_STL::string)(*pVariant)).c_str());
   }
   else if (rstrElementVar    == "spider_web_center_navaid_id")
   {
      m_spider_web_center_navaid_id_cv   =  ((char *)((_FSI_STL::string)(*pVariant)).c_str());
   }
   else if (rstrElementVar    == "spider_web_magvar")
   {
      m_spider_web_magvar_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "area_map_spider_web")
   {
      m_area_map_spider_web_cv   =  *pVariant;

      if (m_area_map_spider_web_cv  == SPIDER_WEB_CURSOR)
         m_spider_web_at_cursor_requested =  true;
      else
         m_spider_web_at_cursor_requested =  false;
   }
   else if (rstrElementVar    == "spider_web_latitude_center")
   {
      if (m_area_map_spider_web_cv  == SPIDER_WEB_CURSOR)
         m_spider_web_latitude_center_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "spider_web_longitude_center")
   {
      if (m_area_map_spider_web_cv  == SPIDER_WEB_CURSOR)
         m_spider_web_longitude_center_cv =  *pVariant;
   }

   else if (rstrElementVar    == "Weather Area 1 latitude")
   {
      m_weather_area_1_latitude_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 1 longitude")
   {
      m_weather_area_1_longitude_cv =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 1 radius")
   {
      m_weather_area_1_radius_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 1 loaded")
   {
      m_weather_area_1_loaded_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 1 active")
   {
      m_weather_area_1_active_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 2 latitude")
   {
      m_weather_area_2_latitude_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 2 longitude")
   {
      m_weather_area_2_longitude_cv =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 2 radius")
   {
      m_weather_area_2_radius_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 2 loaded")
   {
      m_weather_area_2_loaded_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 2 active")
   {
      m_weather_area_2_active_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 3 latitude")
   {
      m_weather_area_3_latitude_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 3 longitude")
   {
      m_weather_area_3_longitude_cv =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 3 radius")
   {
      m_weather_area_3_radius_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 3 loaded")
   {
      m_weather_area_3_loaded_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 3 active")
   {
      m_weather_area_3_active_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 4 latitude")
   {
      m_weather_area_4_latitude_cv  =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 4 longitude")
   {
      m_weather_area_4_longitude_cv =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 4 radius")
   {
      m_weather_area_4_radius_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 4 loaded")
   {
      m_weather_area_4_loaded_cv    =  *pVariant;
   }
   else if (rstrElementVar    == "Weather Area 4 active")
   {
      m_weather_area_4_active_cv    =  *pVariant;
   }
   else
      int   x=1;

}

void CAreaMap::OnMouseMove(UINT nFlags, CPoint point) 
{
   m_mouse_x_position  =  point.x;
   m_mouse_y_position  =  point.y;

   int   new_mouse_x_position(m_mouse_x_position);
   int   new_mouse_y_position(m_mouse_y_position);
   if (m_bLeftMouseDown)   // Panning....
   {
      double   SelectedScale(7200.0 / m_scale);
      // Find out how far the mouse has moved...
      int      x_delta(new_mouse_x_position  -  m_mouse_x_position_lpv);   // E is positive X   && W is negative X
      int      y_delta(new_mouse_y_position  -  m_mouse_y_position_lpv);   // N is positive Y   && S is negative Y
      // Convert the deltas into miles...
      double   x_miles_to_move   =  (x_delta *  (SelectedScale /  m_width)  /  m_aspect_ratio);
      double   y_miles_to_move   =  (y_delta *  (SelectedScale /  m_height));
      // Get current map center...
      double   area_map_latitude_center(m_area_map_latitude_center_cv);    // current latitude
      double   area_map_longitude_center(m_area_map_longitude_center_cv);  // current longitude
      // Apply deltas...
      area_map_latitude_center   =  area_map_latitude_center   +  (y_miles_to_move /  NM_PER_DEGREE);   // N+
      double   correction        =  cos((double)area_map_latitude_center   *  DEG_TO_RAD);
      area_map_longitude_center  =  area_map_longitude_center  +  (x_miles_to_move /  NM_PER_DEGREE)  /  correction;   // E+
      // Export new map center...
      m_area_map_latitude_center_cv    =  area_map_latitude_center;        // new latitude
      m_area_map_longitude_center_cv   =  area_map_longitude_center;       // new longitude
   }
   // Save off mouse position...
   m_mouse_x_position_lpv  =  new_mouse_x_position;
   m_mouse_y_position_lpv  =  new_mouse_y_position;

   //Needed to post the WM_MOUSEHOVER and WM_MOUSELEAVE messages.  This will help
   //the window grab focus when the mouse enters the area of the window.
   TRACKMOUSEEVENT   tme;  
   tme.cbSize        =  sizeof(TRACKMOUSEEVENT); 
   tme.dwFlags       =  TME_LEAVE   |  TME_HOVER; 
   tme.dwHoverTime   =  200;
   tme.hwndTrack     =  m_hWnd;  
   TrackMouseEvent(&tme); 
   SetFocus();
}

void  CAreaMap::DrawLatitudeLongitudeGrid(void)
{
   // Draw Longitude and Latitude lines.
   glPushMatrix();
   {
      glScalef(m_scale,   m_scale,   m_scale);
      glRGB(64, 64, 64);

      // Draw Longitude lines (circles - very short cylinders).
      double   ii             =  0.0;
      double   sweep_detail   =  0.0;
      double   start_angle    =  90.0  -  (90.0  /  m_scale)   /  cos(m_area_map_latitude_center   *  DEG_TO_RAD);
      double   end_angle      =  90.0  +  (90.0  /  m_scale)   /  cos(m_area_map_latitude_center   *  DEG_TO_RAD);

      // If within 60 miles of the poles, draw all the lines of longitude.
      if (fabs(m_area_map_latitude_center)   +  (NM_PER_DEGREE /  m_scale) >= 90.0)
      {
         start_angle =  0.0;
         end_angle   =  180.0;
      }

      if (start_angle   <  0.0)
         start_angle    =  0;
      if (end_angle     >  180.0)
         end_angle      =  180.0;

      sweep_detail   =  (end_angle  -  start_angle)  /   36.0;
      start_angle    =  start_angle -  fmod(m_area_map_longitude_center,  sweep_detail);
   
      if (start_angle   <  0.0)
         start_angle    =  0;
      if (end_angle     >  180.0)
         end_angle      =  180.0;

      // Start angle and end angle are on either side of 90 degrees
      // since you are looking down the z axis and 0 degrees would
      // be on the x axis.
      for (ii = start_angle; ii < end_angle; ii += sweep_detail)
      {
         glPushMatrix();
         {
            glRotated(-(ii +  m_area_map_longitude_center), 0.0,  1.0,  0.0);
            glScalef(0.95f, 0.95f, 0.95f);
            m_circle.Draw();
         }
         glPopMatrix();
      }

      // Draw Latitude lines (circles - very short cylinders).
      start_angle    =  90.0  -  (90.0 /  m_scale) +  m_area_map_latitude_center;
      end_angle      =  90.0  +  (90.0 /  m_scale) +  m_area_map_latitude_center;
      sweep_detail   =  5.0   /  m_scale;

      if (start_angle   <  0.0)
         start_angle    =  0;
      if (end_angle     >  180.0)
         end_angle      =  180.0;
   
      start_angle    =  start_angle -  fmod(m_area_map_latitude_center,  sweep_detail);

      if (start_angle   <  0.0)
         start_angle    =  0;
      if (end_angle     >  180.0)
         end_angle      =  180.0;

      for (ii = start_angle; ii < end_angle; ii += sweep_detail)
      {
         glPushMatrix();
         {
            glRotated(90.0,   1.0,  0.0,  0.0);                // So we see the edge of the circle not the circle itself.
            double   move_z   =  0.95  *  cos(ii   *  DEG_TO_RAD);
            double   radius   =  0.95  *  fabs(sin(ii *  DEG_TO_RAD));
            glTranslated(0.0, 0.0,  move_z);                   // Move up or down to the appropriate latitude.
            glScalef(radius, radius, 1.f);
            m_circle.Draw();
         }
         glPopMatrix();
      }
   }
   glPopMatrix();


   CVariant varDegrees;
   CVariant varDMS;

   // Draw Longitude labels.
   glPushMatrix();
   {
      double   ii             =  0.0;
      double   sweep_detail   =  0.0;
      double   start_angle    =  90.0  -  (90.0  /  m_scale)   /  cos(m_area_map_latitude_center   *  DEG_TO_RAD);
      double   end_angle      =  90.0  +  (90.0  /  m_scale)   /  cos(m_area_map_latitude_center   *  DEG_TO_RAD);

      // If within 60 miles of the poles, draw all the lines of longitude.
      if (fabs(m_area_map_latitude_center)   +  (NM_PER_DEGREE /  m_scale) >= 90.0)
      {
         start_angle =  0.0;
         end_angle   =  180.0;
      }

      if (start_angle   <  0.0)
         start_angle    =  0;
      if (end_angle     >  180.0)
         end_angle      =  180.0;

      sweep_detail   =  (end_angle  -  start_angle)  /   36.0;
      start_angle    =  start_angle -  fmod(m_area_map_longitude_center,  sweep_detail);
   
      if (start_angle   <  0.0)
         start_angle    =  0;
      if (end_angle     >  180.0)
         end_angle      =  180.0;

      for (ii = start_angle; ii < end_angle; ii += sweep_detail)
      {
         glPushMatrix();
         {
            glRGB(32, 32, 32);
            glRotated(90.0 -  ii -  m_area_map_longitude_center, 0.0,  1.0,  0.0);
            glRotated(-m_area_map_latitude_center,  1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95   *  m_scale);

            double   decimal_value  =  0.0;
            double   real_angle  =  90.0 -  ii -  m_area_map_longitude_center;

            if (fabs(real_angle  -  m_area_map_longitude_center)  <  0.001)
            {
               m_longitude.Empty();
               longitudeLabel.Draw(m_longitude, true, 0.0f,    0.4f, 1.0f);
            }
            else
            {
                int nQuadrant = (int)(real_angle / 180.0)  % 2 ;
                double dThrowAway;
                if (real_angle > 180.0)
                {
                    if (nQuadrant <  1)
                       decimal_value  =  modf(real_angle / 1800.0, &dThrowAway) * 180;
                    else 
                       decimal_value  =  -180.0  +  modf(real_angle / 180.0, &dThrowAway) * 180;
                }
                else if ( real_angle < -180.0)
                {
                    if (nQuadrant >  -1)
                       decimal_value  =  modf(real_angle / 180.0, &dThrowAway) * 180;
                    else 
                       decimal_value  =   180.0  +  modf(real_angle / 180.0, &dThrowAway) * 180;
                }
                else
                {
                    decimal_value = real_angle;
                }

                varDegrees.Value(decimal_value);
                varDMS  =  CDataConversion::Convert(varDegrees, Decimal_Degrees, Degrees_Minutes_Seconds_Longitude);

                m_longitude = ((_FSI_STL::string)varDMS).c_str();
                longitudeLabel.Draw(m_longitude,   true, -90.0f,    0.4f, 1.0f);
            }
         }
         glPopMatrix();
      }
   }
   glPopMatrix();


   // Draw Latitude labels.
   glPushMatrix();
   {
      double   start_angle    =  -(90.0   /  m_scale) +  m_area_map_latitude_center;
      double   end_angle      =   (90.0   /  m_scale) +  m_area_map_latitude_center;
      double   sweep_detail   =  5.0   /  m_scale;
      double   ii             =  0.0;
      double   decimal_value  =  0.0;

      start_angle    =  start_angle -  fmod(m_area_map_latitude_center,  sweep_detail);

      for (ii = start_angle; ii < end_angle; ii += sweep_detail)
      {
         glPushMatrix();
         {
            glRotated(-m_area_map_longitude_center,   0.0,  1.0,  0.0);
            glRotated(-ii, 1.0,  0.0,  0.0);
            glTranslated(0.0, 0.0,  0.95  *  m_scale);

            if (fabs(ii -  m_area_map_latitude_center)   <  0.001)
            {
               m_latitude.Empty();
               latitudeLabel.Draw(m_latitude,   true, 0.0f, 0.4f, 1.0f);
            }
            else
            {
                int nQuadrant = (int)(ii / 90.0)  % 4 ;
                double dThrowAway;
                if (ii > 90.0)
                {
                    if (nQuadrant <  1)
                       decimal_value  =  modf(ii / 90.0, &dThrowAway) * 90;
                    else if (nQuadrant  <  2)
                       decimal_value  =  90.0  -  modf(ii / 90.0, &dThrowAway) * 90;
                    else if (nQuadrant < 3)
                       decimal_value  =  -modf(ii / 90.0, &dThrowAway) * 90;
                    else 
                       decimal_value  =  -90.0  + modf(ii / 90.0, &dThrowAway) * 90;
                }
                else if ( ii < -90.0)
                {
                    if (nQuadrant >  -1)
                       decimal_value  =  modf(ii / 90.0, &dThrowAway) * 90;
                    else if (nQuadrant  >  -2)
                       decimal_value  =  -90.0  -  modf(ii / 90.0, &dThrowAway) * 90;
                    else if (nQuadrant > -3)
                       decimal_value  =  -modf(ii / 90.0, &dThrowAway) * 90;
                    else 
                       decimal_value  =  90.0  + modf(ii / 90.0, &dThrowAway) * 90;
                }
                else
                {
                    decimal_value = ii;
                }

                varDegrees.Value(decimal_value);
                varDMS  =  CDataConversion::Convert(varDegrees, Decimal_Degrees, Degrees_Minutes_Seconds_Latitude);

                m_latitude = ((_FSI_STL::string)varDMS).c_str();
                latitudeLabel.Draw(m_latitude,   true, 0.0f,    0.4f, 1.0f);
            }
         }
         glPopMatrix();
      }
   }
   glPopMatrix();
}

void  CAreaMap::DrawMousePosition(void)
{
   if (m_bEditing)   // If in the editor, then we don't need to DrawMousePosition.
      return;

   // If one of the following isn't true then there's no need to do anything.
   if (!m_touch_storm_location_requested && !m_touch_reposition_requested   && !m_spider_web_at_cursor_requested   && !m_map_center_at_cursor_requested   && !m_sym_disp_mouse_lat_long_on)
      return;

   // Display the mouse position.
   glPushMatrix();
   {
      static   int   lpv_x_position =  0;
      static   int   lpv_y_position =  0;
/*
      float x_distance_from_center  =  0.0f;
      float y_distance_from_center  =  0.0f;

      if (m_scale > 1.0f   /  0.95f)
      {
         float y_limit  =  0.95f *  m_height *  m_scale;
         if (m_mouse_y_position >  y_limit)
            m_mouse_y_position  =  y_limit;
         if (m_mouse_y_position <  m_height -  y_limit)
            m_mouse_y_position  =  m_height -  y_limit;

         y_distance_from_center  =  ((2.0f *  y_limit  -  m_height)   /  2.0f  -  m_mouse_y_position) /  ((2.0f *  y_limit  -  m_height)   /  2.0f);

         float x_limit  =  0.95f *  m_width *  m_scale;
         if (m_mouse_x_position >  x_limit)
            m_mouse_x_position  =  x_limit;
         if (m_mouse_x_position <  m_width  -  x_limit)
            m_mouse_x_position  =  m_width  -  x_limit;

         x_distance_from_center  =  ((2.0f *  x_limit  -  m_width)   /  2.0f  -  m_mouse_x_position) /  ((2.0f *  x_limit  -  m_width)   /  2.0f);
      }
      else
      {
         x_distance_from_center  =  (m_width    /  2.0f  -  m_mouse_x_position) /  (m_width    /  2.0f);
         y_distance_from_center  =  (m_height   /  2.0f  -  m_mouse_y_position) /  (m_height   /  2.0f);
      }


      double   offset_from_latitude_center;

    //  if (lpv_y_position   != m_mouse_y_position)
      {
         lpv_y_position =  m_mouse_y_position;

         offset_from_latitude_center   =  asin(y_distance_from_center   /  (0.95  *  m_scale))  *  RAD_TO_DEG;
//         m_mouse_latitude  =  -(m_area_map_latitude_center  +  offset_from_latitude_center);
         m_mouse_latitude.Value(-offset_from_latitude_center);
      }

//      if (lpv_x_position   != m_mouse_x_position)
      {
         lpv_x_position =  m_mouse_x_position;

         double   offset_from_longitude_center   =  asin(x_distance_from_center   /  (0.95   *  m_scale  *  cos((double)m_mouse_latitude *  DEG_TO_RAD)))  *  RAD_TO_DEG;

//         m_mouse_longitude =  -(m_area_map_longitude_center +  offset_from_longitude_center);
         m_mouse_longitude.Value(-offset_from_longitude_center);
//         m_mouse_latitude.Value(-(m_area_map_latitude_center *  cos((double)m_mouse_longitude   *  DEG_TO_RAD) +  offset_from_latitude_center));
         m_mouse_latitude.Value((m_area_map_latitude_center *  cos((double)m_mouse_longitude   *  DEG_TO_RAD) +  offset_from_latitude_center));
      }
*/


      if (lpv_y_position   != m_mouse_y_position)
      {
         lpv_y_position =  m_mouse_y_position;
         double   sweep_angle    =  120.0 /  m_scale;

         m_mouse_latitude.Value(-(sweep_angle /  m_height *  m_mouse_y_position  -  sweep_angle /  2.0 -  m_area_map_latitude_center));
      }

      if (lpv_x_position   != m_mouse_x_position)
      {
         lpv_x_position =  m_mouse_x_position;
         double   sweep_angle    =  (120.0   /  m_scale  /  cos((double)m_mouse_latitude *  DEG_TO_RAD) /  m_aspect_ratio);

         m_mouse_longitude.Value(sweep_angle /  m_width  *  m_mouse_x_position  -  sweep_angle /  2.0   -  m_area_map_longitude_center);
      }


      CString  mouse_longitude;
      CString  mouse_latitude;
      CVariant varDegrees;
      CVariant varDMS;

      double   decimal_value;

      if (fabs(m_mouse_longitude)   >  180.0)
         decimal_value  =  360.0  -  fabs(m_mouse_longitude);
      else
         decimal_value = m_mouse_longitude;

      if (m_sym_disp_mouse_lat_long_decimal_degree)
         mouse_longitude.Format("%lf", decimal_value);
      else
      {
         varDegrees.Value(decimal_value);
         varDMS   =  CDataConversion::Convert(varDegrees, Decimal_Degrees, Degrees_Minutes_Seconds_Longitude);
         mouse_longitude = ((_FSI_STL::string)varDMS).c_str();
      }

      if (fabs(m_mouse_latitude)   >  90.0)
         decimal_value  =  180.0  -  fabs(m_mouse_latitude);
      else
         decimal_value = m_mouse_latitude;

      if (m_sym_disp_mouse_lat_long_decimal_degree)
         mouse_latitude.Format("%lf", decimal_value);
      else
      {
         varDegrees.Value(decimal_value);
         varDMS   =  CDataConversion::Convert(varDegrees, Decimal_Degrees, Degrees_Minutes_Seconds_Latitude);
         mouse_latitude = ((_FSI_STL::string)varDMS).c_str();
      }

      if (m_sym_disp_mouse_lat_long_decimal_degree)
      {
         m_theDataToCopy.Format("\r\nLatitude=%s\r\nLongitude=%s\r\n\r\n", mouse_latitude, mouse_longitude);
      }


#ifdef   FIXED_POSITION
      glRotated(-m_area_map_longitude_center, 0.0,  1.0,  0.0);
      glRotated(-m_area_map_latitude_center,  1.0,  0.0,  0.0);
      glTranslated(0.0, -0.95,  0.95  *  m_scale);
      mouse_position.SetForegroundColor(192, 192, 192);
      m_cursor_position.Format("MOUSE POSITION %s  %s", mouse_latitude, mouse_longitude);
      mouse_position.Draw(m_cursor_position, true, 0.0f, 0.45f,   1.0f, true);
#else
      glRotated(m_mouse_longitude, 0.0,  1.0,  0.0);
      glRotated(-(double)m_mouse_latitude,  1.0,  0.0,  0.0);

      if (m_touch_reposition_requested)
      {
         glTranslated(0.0, 0.1025,  0.95  *  m_scale);
         glCallList(m_int_lists[BIG_UNUSUAL_BOX_OUTLINE]);
         m_cursor_position.Format("CLICK ANYWHERE ON THE MAP TO REPOSITION\nTHE AIRCRAFT TO THE BELOW POINT");
         mouse_position.SetBackgroundColor(true, 1, 1, 1);  // Changed 0,0,0 to 1,1,1 to support printing
         mouse_position.Draw(m_cursor_position, true, 0.0f, 0.45f,   1.0f, false);
      }

	  else if (m_touch_storm_location_requested)
      {
         glTranslated(0.0, 0.1025,  0.95  *  m_scale);
         glCallList(m_int_lists[BIG_UNUSUAL_BOX_OUTLINE]);
         m_cursor_position.Format("CLICK ANYWHERE ON THE MAP TO POSITION\nSTORM CELL AT THE BELOW POINT");
         mouse_position.SetBackgroundColor(true, 1, 1, 1);
         mouse_position.Draw(m_cursor_position, true, 0.0f, 0.45f,   1.0f, false);
      }

      else if (m_spider_web_at_cursor_requested)
      {
         glTranslated(0.0, 0.1025,  0.95  *  m_scale);
         glCallList(m_int_lists[BIG_UNUSUAL_BOX_OUTLINE]);
         m_cursor_position.Format("CLICK ANYWHERE ON THE MAP TO REPOSITION\nTHE SPIDER WEB TO THE BELOW POINT");
         mouse_position.SetBackgroundColor(true, 1, 1, 1);  // Changed 0,0,0 to 1,1,1 to support printing
         mouse_position.Draw(m_cursor_position, true, 0.0f, 0.45f,   1.0f, false);
      }
      else if (m_map_center_at_cursor_requested)
      {
         glTranslated(0.0, 0.1025,  0.95  *  m_scale);
         glCallList(m_int_lists[BIG_UNUSUAL_BOX_OUTLINE]);
         m_cursor_position.Format("CLICK ANYWHERE ON THE MAP TO\nRE-CENTER THE MAP TO THE BELOW POINT");
         mouse_position.SetBackgroundColor(true, 1, 1, 1);  // Changed 0,0,0 to 1,1,1 to support printing
         mouse_position.Draw(m_cursor_position, true, 0.0f, 0.45f,   1.0f, false);
      }
      else if (m_sym_disp_mouse_lat_long_on)
      {
         glTranslated(0.0, 0.1025,  0.95  *  m_scale);
         glCallList(m_int_lists[SMALL_UNUSUAL_BOX_OUTLINE]);
         m_cursor_position.Format("% 11s\n% 11s", mouse_latitude, mouse_longitude);
         mouse_position.SetBackgroundColor(false, 1, 1, 1); // Changed 0,0,0 to 1,1,1 to support printing
         mouse_position.Draw(m_cursor_position, true, 0.0f, 0.45f,   1.0f, false);
      }
#endif
   }
   glPopMatrix();
}


bool CAreaMap::UpdateRenderVariables()
{
   bool  bRetVal  =  CWidget::UpdateRenderVariables();

   if (bRetVal == true)
   {
      m_area_map_auto_scale         =  m_area_map_auto_scale_cv;
      m_area_map_auto_center        =  m_area_map_auto_center_cv;     
      m_ac_heading_lead             =  m_ac_heading_lead_cv;
      m_ac_heading_wing             =  m_ac_heading_wing_cv;
      m_ac_altitude                 =  m_ac_altitude_cv;

      m_spider_web_magvar           =  m_spider_web_magvar_cv;
      m_ac_latitude                 =  m_ac_latitude_cv;
      m_ac_longitude                =  m_ac_longitude_cv;
      m_lead_ac_latitude            =  m_lead_ac_latitude_cv;
      m_lead_ac_longitude           =  m_lead_ac_longitude_cv;
      m_lead_ac_altitude            =  m_lead_ac_altitude_cv;

	  m_storm_active                = m_storm_active_cv;
      m_storm_latitude              = m_storm_latitude_cv;
      m_storm_longitude             = m_storm_longitude_cv;
      m_storm_length                = m_storm_length_cv;
      m_storm_width                 = m_storm_width_cv;
      m_storm_heading               = m_storm_heading_cv;
      m_storm_name                  = m_storm_name_cv;
	  m_sym_disp_storm_on           = m_sym_disp_storm_on_cv;

      m_sym_disp_lead_ac_track_on               =  m_sym_disp_lead_ac_track_on_cv;
      m_sym_disp_wing_ac_track_on               =  m_sym_disp_wing_ac_track_on_cv;
      m_sym_disp_lat_long_grid_on               =  m_sym_disp_lat_long_grid_on_cv;
      m_sym_disp_mouse_lat_long_on              =  m_sym_disp_mouse_lat_long_on_cv;
      m_sym_disp_mouse_lat_long_decimal_degree  =  m_sym_disp_mouse_lat_long_decimal_degree_cv;
      m_sym_disp_weather_areas_on               =  m_sym_disp_weather_areas_on_cv;

      m_weather_area_1_latitude     =  m_weather_area_1_latitude_cv;
      m_weather_area_1_longitude    =  m_weather_area_1_longitude_cv;
      m_weather_area_1_radius       =  m_weather_area_1_radius_cv;
      m_weather_area_1_loaded       =  m_weather_area_1_loaded_cv;
      m_weather_area_1_active       =  m_weather_area_1_active_cv;

      m_weather_area_2_latitude     =  m_weather_area_2_latitude_cv;
      m_weather_area_2_longitude    =  m_weather_area_2_longitude_cv;
      m_weather_area_2_radius       =  m_weather_area_2_radius_cv;
      m_weather_area_2_loaded       =  m_weather_area_2_loaded_cv;
      m_weather_area_2_active       =  m_weather_area_2_active_cv;

      m_weather_area_3_latitude     =  m_weather_area_3_latitude_cv;
      m_weather_area_3_longitude    =  m_weather_area_3_longitude_cv;
      m_weather_area_3_radius       =  m_weather_area_3_radius_cv;
      m_weather_area_3_loaded       =  m_weather_area_3_loaded_cv;
      m_weather_area_3_active       =  m_weather_area_3_active_cv;

      m_weather_area_4_latitude     =  m_weather_area_4_latitude_cv;
      m_weather_area_4_longitude    =  m_weather_area_4_longitude_cv;
      m_weather_area_4_radius       =  m_weather_area_4_radius_cv;
      m_weather_area_4_loaded       =  m_weather_area_4_loaded_cv;
      m_weather_area_4_active       =  m_weather_area_4_active_cv;



      if (m_area_map_spider_web           != m_area_map_spider_web_cv            ||
          m_spider_web_center_airport_id  != m_spider_web_center_airport_id_cv   ||
          m_spider_web_center_navaid_id   != m_spider_web_center_navaid_id_cv    ||
          m_active_airport_id             != m_active_airport_id_cv)
      {
         if (m_area_map_spider_web_cv        == SPIDER_WEB_AIRPORT)
         {
            CenterSpiderWebOnAirport(m_spider_web_center_airport_id_cv);
            m_bool_new_magvar_needed         =  true;
            m_CVar_magvar_latitude.Value(m_spider_web_latitude_center_cv);
            m_CVar_magvar_longitude.Value(m_spider_web_longitude_center_cv);
         }
         else if (m_area_map_spider_web_cv   == SPIDER_WEB_ACTIVE_AIRPORT)
         {
            CenterSpiderWebOnAirport(m_active_airport_id_cv);
            m_bool_new_magvar_needed         =  true;
            m_CVar_magvar_latitude.Value(m_spider_web_latitude_center_cv);
            m_CVar_magvar_longitude.Value(m_spider_web_longitude_center_cv);
         }
         else if (m_area_map_spider_web_cv   == SPIDER_WEB_NAVAID)
         {
            CenterSpiderWebOnNavaid(m_spider_web_center_navaid_id_cv);
         }
         m_spider_web_center_airport_id   =  m_spider_web_center_airport_id_cv;
         m_spider_web_center_navaid_id    =  m_spider_web_center_navaid_id_cv;
         m_area_map_spider_web            =  m_area_map_spider_web_cv;
      }
      else if (m_area_map_spider_web_cv   == SPIDER_WEB_AIRCRAFT)
      {
         m_bool_new_magvar_needed         =  true;
         m_CVar_magvar_latitude.Value(m_ac_latitude);
         m_CVar_magvar_longitude.Value(m_ac_longitude);

         m_spider_web_latitude_center_cv  =  m_ac_latitude;
         m_spider_web_longitude_center_cv =  m_ac_longitude;
      }
      m_spider_web_latitude_center        =  m_spider_web_latitude_center_cv;
      m_spider_web_longitude_center       =  m_spider_web_longitude_center_cv;


      if (m_area_map_auto_center || m_area_map_auto_scale)
      {
         AdjustMapCenterOrScale();
      }


      if (m_map_center_at_aircraft_requested)
      {
         m_area_map_latitude_center_cv       =   m_ac_latitude_cv;
         m_area_map_longitude_center_cv      =  -m_ac_longitude_cv;
         m_map_center_at_aircraft_requested  =  false;

         // Update local comms to turn off area_map_center_aircraft.  The last known latitude and longitude of the center of the 
         // map is saved so we don't need to remember how it got centered at that point.
         CVariant var;
         long  old_type;
         _FSI_STL::string  stlStrVariable;
         stlStrVariable = (LPCTSTR)"area_map_center_aircraft";
         CCommsShared::GetLocalValue(var, stlStrVariable);
         old_type =  var.Type();
         var.Value(m_map_center_at_aircraft_requested);
         var.ChangeType(old_type);
         CCommsShared::SetLocalValue(var, stlStrVariable, 0);
      }

      if (m_area_map_center_on_airport)
      {
         bool  valid =  GetAirportLatLon(m_airport_id, m_area_map_latitude_center_cv, m_area_map_longitude_center_cv);

         if (valid)
            m_area_map_longitude_center_cv   *= -1.0;

         m_area_map_center_on_airport  =  false;

         // Update local comms to turn off area_map_center_airport.  The last known latitude and longitude of the center of the 
         // map is saved so we don't need to remember how it got centered at that point.
         CVariant var;
         long  old_type;
         _FSI_STL::string  stlStrVariable;
         stlStrVariable = (LPCTSTR)"area_map_center_airport";
         CCommsShared::GetLocalValue(var, stlStrVariable);
         old_type =  var.Type();
         var.Value(m_area_map_center_on_airport);
         var.ChangeType(old_type);
         CCommsShared::SetLocalValue(var, stlStrVariable, 0);
      }


      if (m_aircraftLeadTrack.Hwnd() == NULL)
         m_aircraftLeadTrack.Hwnd(reinterpret_cast<HWND>(1));

      if (m_bClearLead == true && m_aircraftLeadTrack.VerticesAdded() > 0)
      {
         m_aircraftLeadTrack.Clear();
         m_bClearLead   =  false;
      }
      else
      {
         m_bClearLead   =  false;
      }

      if (m_scale != m_scale_cv)
      {
         m_aircraftLeadTrack.Scale(m_scale_cv);
      }

      m_aircraftLeadTrack.UpdateData();

      if (m_aircraftWingTrack.Hwnd() == NULL)
         m_aircraftWingTrack.Hwnd(reinterpret_cast<HWND>(1));

      if (m_bClearWing == true && m_aircraftWingTrack.VerticesAdded() > 0)
      {
         m_aircraftWingTrack.Clear();
         m_bClearWing   =  false;
      }
      else
      {
         m_bClearWing   =  false;
      }

      if (m_scale != m_scale_cv)
      {
         m_aircraftWingTrack.Scale(m_scale_cv);
      }

      m_aircraftWingTrack.UpdateData();

      if (m_bool_new_magvar_needed)
      {
         // Update other widgets...
         if (m_bEditing == false)
         {
            _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
            for (; lIt != m_listActions.end(); lIt++)
            {
               CIOAction* pIOAction = dynamic_cast<CIOAction*>(*lIt);
               if (pIOAction != NULL)
               {
                  CString str(pIOAction->GraphicalElementName().c_str());
                  if (str   == "Magvar_Latitude" || str   == "Magvar_Longitude")
                  {
                     // Force a "click", which in turn will cause GetValue() to be called, which in turn will update the above variables.
                     pIOAction->OnLButtonUp();
                  }
               }
            }
         }
      }

      // Check to see if the symbols need to be refreshed.  This should
      // only need to happen if the scale has changed or the map center
      // has changed.  Render will take care of drawing only what needs
      // to be drawn.
      if (m_scale                                  != m_scale_cv                             ||
          m_area_map_latitude_center               != m_area_map_latitude_center_cv          ||
          m_area_map_longitude_center              != m_area_map_longitude_center_cv         ||
          m_sym_disp_airport_text_on               != m_sym_disp_airport_text_on_cv          ||
          m_sym_disp_airports_on                   != m_sym_disp_airports_on_cv              ||
          m_sym_disp_ils_text_on                   != m_sym_disp_ils_text_on_cv              ||
          m_sym_disp_ils_on                        != m_sym_disp_ils_on_cv                   ||
          m_sym_disp_ndb_text_on                   != m_sym_disp_ndb_text_on_cv              ||  
          m_sym_disp_ndb_on                        != m_sym_disp_ndb_on_cv                   ||
          m_sym_disp_vor_text_on                   != m_sym_disp_vor_text_on_cv              ||  
          m_sym_disp_vor_on                        != m_sym_disp_vor_on_cv                   ||
          m_active_airport_id                      != m_active_airport_id_cv                 ||
          m_active_runway_id                       != m_active_runway_id_cv                  ||

          m_sym_disp_markers_on                    != m_sym_disp_markers_on_cv               ||
          m_sym_disp_waypoints_on                  != m_sym_disp_waypoints_on_cv             ||
          m_sym_disp_waypoints_text_on             != m_sym_disp_waypoints_text_on_cv        ||

          m_sym_disp_victor_routes_on              != m_sym_disp_victor_routes_on_cv         || 
          m_sym_disp_jet_routes_on                 != m_sym_disp_jet_routes_on_cv            ||
          m_sym_disp_custom_lines_on               != m_sym_disp_custom_lines_on_cv          ||
          m_sym_disp_custom_lines_text_on          != m_sym_disp_custom_lines_text_on_cv     ||
		  m_sym_disp_storm_on                      != m_sym_disp_storm_on_cv                 ||
          m_sym_disp_storm_text_on                 != m_sym_disp_storm_text_on_cv            ||
          m_sym_disp_custom_obstacles_on           != m_sym_disp_custom_obstacles_on_cv      ||
          m_sym_disp_custom_obstacles_text_on      != m_sym_disp_custom_obstacles_text_on_cv ||
          m_sym_disp_custom_practice_areas_on      != m_sym_disp_custom_practice_areas_on_cv ||
          m_sym_disp_custom_practice_areas_text_on != m_sym_disp_custom_practice_areas_text_on_cv
         )
      {
         // These assignments aren't needed outside of this if statement because if these lines aren't reached, then none
         // of the values changed.
         m_scale                                   =  m_scale_cv;
         m_area_map_latitude_center                =  m_area_map_latitude_center_cv;
         m_area_map_longitude_center               =  m_area_map_longitude_center_cv;
         m_sym_disp_airports_on                    =  m_sym_disp_airports_on_cv;
         m_sym_disp_airport_text_on                =  m_sym_disp_airport_text_on_cv;
         m_sym_disp_ils_text_on                    =  m_sym_disp_ils_text_on_cv;
         m_sym_disp_ils_on                         =  m_sym_disp_ils_on_cv;
         m_sym_disp_ndb_text_on                    =  m_sym_disp_ndb_text_on_cv;
         m_sym_disp_ndb_on                         =  m_sym_disp_ndb_on_cv;
         m_sym_disp_vor_text_on                    =  m_sym_disp_vor_text_on_cv;
         m_sym_disp_vor_on                         =  m_sym_disp_vor_on_cv;
         m_sym_disp_markers_on                     =  m_sym_disp_markers_on_cv;
         m_sym_disp_waypoints_on                   =  m_sym_disp_waypoints_on_cv;
         m_sym_disp_waypoints_text_on              =  m_sym_disp_waypoints_text_on_cv;
         m_active_airport_id                       =  m_active_airport_id_cv;
         m_active_runway_id                        =  m_active_runway_id_cv;
         m_sym_disp_victor_routes_on               =  m_sym_disp_victor_routes_on_cv;
         m_sym_disp_jet_routes_on                  =  m_sym_disp_jet_routes_on_cv;
         m_sym_disp_custom_lines_on                =  m_sym_disp_custom_lines_on_cv;
         m_sym_disp_custom_lines_text_on           =  m_sym_disp_custom_lines_text_on_cv;
		 m_sym_disp_storm_on                       =  m_sym_disp_storm_on_cv;
         m_sym_disp_storm_text_on                  =  m_sym_disp_storm_text_on_cv;
         m_sym_disp_custom_obstacles_on            =  m_sym_disp_custom_obstacles_on_cv;
         m_sym_disp_custom_obstacles_text_on       =  m_sym_disp_custom_obstacles_text_on_cv;
         m_sym_disp_custom_practice_areas_on       =  m_sym_disp_custom_practice_areas_on_cv;
         m_sym_disp_custom_practice_areas_text_on  =  m_sym_disp_custom_practice_areas_text_on_cv;


         // Update other widgets...
         if (m_bEditing == false)
         {
            CVariant var;
            long  old_type;
            _FSI_STL::string  stlStrVariable;

            // Save off the latitude and longitude of the center of the map into local comms variables.  Do this so that the next 
            // time the area map page is loaded it will be centered at the same place it was when the page was removed.
            stlStrVariable = (LPCTSTR)"area_map_latitude_center";
            CCommsShared::GetLocalValue(var, stlStrVariable);
            old_type =  var.Type();
            var.Value(m_area_map_latitude_center);
            var.ChangeType(old_type);
            CCommsShared::SetLocalValue(var, stlStrVariable, 0);

            stlStrVariable = (LPCTSTR)"area_map_longitude_center";
            CCommsShared::GetLocalValue(var, stlStrVariable);
            old_type =  var.Type();
            var.Value(-m_area_map_longitude_center);
            var.ChangeType(old_type);
            CCommsShared::SetLocalValue(var, stlStrVariable, 0);


            _FSI_STL::list<CAction*>::iterator lIt = m_listActions.begin();
            for (; lIt != m_listActions.end(); lIt++)
            {
               CIOAction* pIOAction = dynamic_cast<CIOAction*>(*lIt);
               if (pIOAction != NULL)
               {
                  CString str(pIOAction->GraphicalElementName().c_str());
                  if (str  == "area_map_scale")
                  {
                     // Force a "click", which in turn will cause GetValue() to be called, which in turn will update the above variables.
                     pIOAction->OnLButtonUp();
                  }
               }
            }
         }

         m_symbolInfoList.RemoveAll();
         m_symbolLabelInfoList.RemoveAll();

         if (m_scale >= 15.0)    // Scale must be less than 480 miles to do lookup.
         {
            symbol.ComputeBoxSize(CString("XXX.XX YYYY"),  true, m_symbolLabelInfo.width,      m_symbolLabelInfo.height);

            // Calculate the number of degrees shown on the map (vertically, or in the latitude direction).
            double   sweep_angle =  7200.0f  /  m_scale  /  NM_PER_DEGREE;
            sweep_angle =  __max(sweep_angle, 0.08333333);
            // Adjust the sweep_angle to compensate for the aspect ratio.
            if (m_aspect_ratio  > 1.0f)
               sweep_angle *=  m_aspect_ratio;
            else if (m_aspect_ratio != 0.0f)
               sweep_angle /=  m_aspect_ratio;
            // Adjust the sweep_angle to compensate for longitude convergence.
            if (m_area_map_latitude_center   >  0.0f)
            {
               if ((m_area_map_latitude_center  +  (sweep_angle   /  2.0f) != 90.0))
                  sweep_angle =  sweep_angle /  cos((m_area_map_latitude_center  +  (sweep_angle   /  2.0f)) *  DEG_TO_RAD);
               else
                  sweep_angle =  sweep_angle /  0.0001;
            }
            else
            {
               if ((m_area_map_latitude_center  -  (sweep_angle   /  2.0f) != 90.0))
                  sweep_angle =  sweep_angle /  cos((m_area_map_latitude_center  -  (sweep_angle   /  2.0f)) *  DEG_TO_RAD);
               else
                  sweep_angle =  sweep_angle /  0.0001;
            }

            double   min_longitude  =  -m_area_map_longitude_center   -  sweep_angle /  2.0;
            double   max_longitude  =  -m_area_map_longitude_center   +  sweep_angle /  2.0;

            // Calculate the number of degrees shown on the map (vertically, or in the latitude direction).
            sweep_angle    =  7200.0f  /  m_scale  /  NM_PER_DEGREE;
            sweep_angle    =  __max(sweep_angle, 0.08333333);
            double   min_latitude   =  m_area_map_latitude_center -  sweep_angle /  2.0;
            double   max_latitude   =  m_area_map_latitude_center +  sweep_angle /  2.0;

            CVariant var;
            var.Value((double)min_longitude);
            long  min_longitude_in_BAMS   =  (long)CDataConversion::Convert(var, Decimal_Degrees, BAMS);

            var.Value((double)max_longitude);
            long  max_longitude_in_BAMS   =  (long)CDataConversion::Convert(var, Decimal_Degrees, BAMS);


            // Get pointer to first Station in lat-lon range.
            rdbGNRC  *pGnrc;
            pGnrc    =  (rdbGNRC *)rdbRadioDB.StationFirstLat(min_latitude, max_latitude);

            while (pGnrc)
            {
               // CIOSRadioDB class no longer limits searches longitudenally so limit it here.
               if (pGnrc->stn.nLon  <  min_longitude_in_BAMS   || pGnrc->stn.nLon   >  max_longitude_in_BAMS)
               {
                  pGnrc   =  (rdbGNRC *)rdbRadioDB.StationNextLat(min_latitude, max_latitude);
                  continue;
               }

               var.Value((long)pGnrc->stn.nLat);
               m_symbolInfo.position.m_latitude             =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

               var.Value((long)pGnrc->stn.nLon);
               m_symbolInfo.position.m_longitude            =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);


               m_symbolInfo.heading                         =  0.0;
               m_symbolInfo.type                            =  0;
               m_symbolInfo.active_airport_and_runway       =  false;
               m_symbolLabelInfo.heading                    =  0.0;
               m_symbolLabelInfo.active_airport_and_runway  =  false;

               DWORD dwTypePri   =  pGnrc->stn.nType1;
               DWORD dwTypeSec   =  pGnrc->stn.nType2;

               dwTypePri   &= 0xfffffe00u;   // Mask-out range bits.


               switch (dwTypePri)
               {
                  case rdbType1APT:
                  {
                     m_symbolInfo.type =  AIRPORT;
                     m_symbolInfo.symbol.Format("%c", AIRPORT);
                     m_symbolInfoList.AddTail(m_symbolInfo);
                  
                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Empty();
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  AIRPORT;

                     if (m_active_airport_id  == m_symbolLabelInfo.symbol_label)
                        m_symbolLabelInfo.active_airport_and_runway  =  true;

                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                  }
                  break;
      
                  case rdbType1DME:
                  {
                     m_symbolInfo.type =  DME;
                     m_symbolInfo.symbol.Format("%c", DME);
                     m_symbolInfoList.AddTail(m_symbolInfo);
                  }
                  break;

                  case rdbType1ILS:
                  {
                     m_symbolInfo.heading =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;

                     m_symbolInfo.type    =  ILS_FAN;
                     m_symbolInfo.symbol.Format("%c", ILS_FAN);


                     CString  airport_id;
                     airport_id  =  (char)((pGnrc->ils.nAptIdent  &  0xff000000) >> 24);
                     airport_id  += (char)((pGnrc->ils.nAptIdent  &  0x00ff0000) >> 16);
                     airport_id  += (char)((pGnrc->ils.nAptIdent  &  0x0000ff00) >> 8);
                     airport_id  += (char)((pGnrc->ils.nAptIdent  &  0x000000ff));
                     airport_id  += "\0";

                     if (m_active_airport_id  == airport_id)
                     {
                        CString  runway_id;
                        runway_id   =  (char)((pGnrc->ils.nRwyIdent  &  0xff000000) >> 24);
                        runway_id   += (char)((pGnrc->ils.nRwyIdent  &  0x00ff0000) >> 16);
                        runway_id   += (char)((pGnrc->ils.nRwyIdent  &  0x0000ff00) >> 8);
                        runway_id   += (char)((pGnrc->ils.nRwyIdent  &  0x000000ff));
                        runway_id   += "\0";

                        if (m_active_runway_id   == runway_id)
                        {
                           m_symbolInfo.active_airport_and_runway       =  true;
                           m_symbolLabelInfo.active_airport_and_runway  =  true;
                        }
                     }
                     m_symbolInfoList.AddTail(m_symbolInfo); 

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Format("%3.2f ", (float)(unsigned short)pGnrc->ils.sFreqX100 /  100.0);
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  ILS_FAN;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);

                     if (pGnrc->ils.nImLat   && pGnrc->ils.nImLon)
                     {
                        // INNER marker
                        CVariant var;
                        var.Value((long)pGnrc->ils.nImLat);
                        m_symbolInfo.position.m_latitude       =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        var.Value((long)pGnrc->ils.nImLon);
                        m_symbolInfo.position.m_longitude      =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        m_symbolInfo.heading                   =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                        m_symbolInfo.type                      =  INNER_MARKER;
                        m_symbolInfo.symbol.Format("%c", INNER_MARKER);
                        m_symbolInfoList.AddTail(m_symbolInfo);
                     }

                     if (pGnrc->ils.nMmLat   && pGnrc->ils.nMmLon)
                     {
                        // MIDDLE marker
                        CVariant var;
                        var.Value((long)pGnrc->ils.nMmLat);
                        m_symbolInfo.position.m_latitude       =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        var.Value((long)pGnrc->ils.nMmLon);
                        m_symbolInfo.position.m_longitude      =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        m_symbolInfo.heading                   =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                        m_symbolInfo.type                      =  MIDDLE_MARKER;
                        m_symbolInfo.symbol.Format("%c", MIDDLE_MARKER);
                        m_symbolInfoList.AddTail(m_symbolInfo);
                     }

                     if (pGnrc->ils.nOmLat   && pGnrc->ils.nOmLon)
                     {
                        // OUTER marker
                        CVariant var;
                        var.Value((long)pGnrc->ils.nOmLat);
                        m_symbolInfo.position.m_latitude       =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        var.Value((long)pGnrc->ils.nOmLon);
                        m_symbolInfo.position.m_longitude      =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        m_symbolInfo.heading                   =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                        m_symbolInfo.type                      =  OUTER_MARKER;
                        m_symbolInfo.symbol.Format("%c", OUTER_MARKER);
                        m_symbolInfoList.AddTail(m_symbolInfo);
                     }
                  }
                  break;

                  case rdbType1ILSDME:
                  {
                     m_symbolInfo.heading =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                     m_symbolInfo.type    =  ILS_FAN;
                     m_symbolInfo.symbol.Format("%c", ILS_FAN);

                     CString  airport_id;
                     airport_id  =  (char)((pGnrc->ils.nAptIdent  &  0xff000000) >> 24);
                     airport_id  += (char)((pGnrc->ils.nAptIdent  &  0x00ff0000) >> 16);
                     airport_id  += (char)((pGnrc->ils.nAptIdent  &  0x0000ff00) >> 8);
                     airport_id  += (char)((pGnrc->ils.nAptIdent  &  0x000000ff));
                     airport_id  += "\0";

                     if (m_active_airport_id  == airport_id)
                     {
                        CString  runway_id;
                        runway_id   =  (char)((pGnrc->ils.nRwyIdent  &  0xff000000) >> 24);
                        runway_id   += (char)((pGnrc->ils.nRwyIdent  &  0x00ff0000) >> 16);
                        runway_id   += (char)((pGnrc->ils.nRwyIdent  &  0x0000ff00) >> 8);
                        runway_id   += (char)((pGnrc->ils.nRwyIdent  &  0x000000ff));
                        runway_id   += "\0";

                        if (m_active_runway_id   == runway_id)
                        {
                           m_symbolInfo.active_airport_and_runway       =  true;
                           m_symbolLabelInfo.active_airport_and_runway  =  true;
                        }
                     }
                     m_symbolInfoList.AddTail(m_symbolInfo);

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Format("%3.2f ", (float)(unsigned short)pGnrc->ils.sFreqX100 /  100.0);
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  ILS_FAN;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);

                     if (pGnrc->ils.nDmeLat  && pGnrc->ils.nDmeLon)
                     {
                        CVariant var;
                        var.Value((long)pGnrc->ils.nDmeLat);
                        m_symbolInfo.position.m_latitude       =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        var.Value((long)pGnrc->ils.nDmeLon);
                        m_symbolInfo.position.m_longitude      =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        m_symbolInfo.heading                   =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                        m_symbolInfo.type                      =  ILSDME;
                        m_symbolInfo.symbol.Format("%c", DME);
                        m_symbolInfoList.AddTail(m_symbolInfo);
                     }

                     if (pGnrc->ils.nImLat   && pGnrc->ils.nImLon)
                     {
                        // INNER marker
                        CVariant var;
                        var.Value((long)pGnrc->ils.nImLat);
                        m_symbolInfo.position.m_latitude       =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        var.Value((long)pGnrc->ils.nImLon);
                        m_symbolInfo.position.m_longitude      =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        m_symbolInfo.heading                   =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                        m_symbolInfo.type                      =  INNER_MARKER;
                        m_symbolInfo.symbol.Format("%c", INNER_MARKER);
                        m_symbolInfoList.AddTail(m_symbolInfo);
                     }

                     if (pGnrc->ils.nImLat   && pGnrc->ils.nImLon)
                     {
                        // MIDDLE marker
                        CVariant var;
                        var.Value((long)pGnrc->ils.nMmLat);
                        m_symbolInfo.position.m_latitude       =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        var.Value((long)pGnrc->ils.nMmLon);
                        m_symbolInfo.position.m_longitude      =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        m_symbolInfo.heading                   =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                        m_symbolInfo.type                      =  MIDDLE_MARKER;
                        m_symbolInfo.symbol.Format("%c", MIDDLE_MARKER);
                        m_symbolInfoList.AddTail(m_symbolInfo);
                     }

                     if (pGnrc->ils.nImLat   && pGnrc->ils.nImLon)
                     {
                        // OUTER marker
                        CVariant var;
                        var.Value((long)pGnrc->ils.nOmLat);
                        m_symbolInfo.position.m_latitude       =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        var.Value((long)pGnrc->ils.nOmLon);
                        m_symbolInfo.position.m_longitude      =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                        m_symbolInfo.heading                   =  (double)(int)pGnrc->ils.sThdgX10   /  10.0;
                        m_symbolInfo.type                      =  OUTER_MARKER;
                        m_symbolInfo.symbol.Format("%c", OUTER_MARKER);
                        m_symbolInfoList.AddTail(m_symbolInfo);
                     }
                  }
                  break;

                  case rdbType1NDB:
                  {
                     m_symbolInfo.heading =  (double)(int)pGnrc->ndb.sMvarX10   /  10.0;
                     m_symbolInfo.type    =  NDB;
                     m_symbolInfo.symbol.Format("%c", NDB);
                     m_symbolInfoList.AddTail(m_symbolInfo);

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Format("%3.2f ", (float)(unsigned short)pGnrc->ndb.sFreqX100 /  10.0);
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  NDB;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                  }
                  break;

                  case rdbType1RWY:
                  {
                     m_symbolInfo.heading =  (double)(int)pGnrc->rwy.sThdgx10   /  10.0;
                     m_symbolInfo.type    =  RUNWAY;
                     m_symbolInfo.symbol.Format("%c", RUNWAY);
                     double   length      =  (double)pGnrc->rwy.sLen;
                     double   width       =  (double)pGnrc->rwy.sWidth;
                     m_symbolInfo.length  =  (length  /  NMI_FT)  *  (m_scale /  3600.0f);
                     m_symbolInfo.width   =  (width   /  NMI_FT)  *  (m_scale /  3600.0f);
                     m_symbolInfoList.AddTail(m_symbolInfo);

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  m_symbolInfo.heading;
                     m_symbolLabelInfo.symbol_label.Empty();
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  RUNWAY;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                  }
                  break;

                  case rdbType1VOR:
                  {
                     m_symbolInfo.heading =  (double)(int)pGnrc->vhf.sMvarX10   /  10.0;
                     m_symbolInfo.type    =  VOR;
                     m_symbolInfo.symbol.Format("%c", VOR);
                     m_symbolInfoList.AddTail(m_symbolInfo);

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Format("%3.2f ", (float)(unsigned short)pGnrc->vhf.sFreqX100 /  100.0);
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  VOR;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                  }
                  break;

                  case rdbType1VORDME:
                  {
                     m_symbolInfo.heading =  (double)(int)pGnrc->vhf.sMvarX10   /  10.0;
                     m_symbolInfo.type    =  VORDME;
                     m_symbolInfo.symbol.Format("%c", VORDME);
                     m_symbolInfoList.AddTail(m_symbolInfo);

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Format("%3.2f ", (float)(unsigned short)pGnrc->vhf.sFreqX100 /  100.0);
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  VORDME;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                  }
                  break;

                  case rdbType1VORTAC:
                  {
                     m_symbolInfo.heading =  (double)(int)pGnrc->vhf.sMvarX10   /  10.0;
                     m_symbolInfo.type    =  VORTAC;
                     m_symbolInfo.symbol.Format("%c", VORTAC);
                     m_symbolInfoList.AddTail(m_symbolInfo);

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Format("%3.2f ", (float)(unsigned short)pGnrc->vhf.sFreqX100 /  100.0);
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];

                     #ifdef   HAF
                        CString  frequency, channel;
                        frequency.Format("%3.2f", (float)(unsigned short)pGnrc->vhf.sFreqX100 /  100.0);
                        BOOL  bExists  =  m_TacanChannel.Lookup(frequency, channel);
                        if (bExists && !channel.IsEmpty())
                        {
                           m_symbolLabelInfo.symbol_label         += " CH " + channel;
                        }
                     #endif

                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  VORTAC;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                  }
                  break;

                  case  rdbType1TAC:
                  {
                     #ifdef   HAF
                        m_symbolInfo.heading =  (double)(int)pGnrc->vhf.sMvarX10   /  10.0;
                        m_symbolInfo.type    =  TACAN;
                        m_symbolInfo.symbol.Format("%c", TACAN);
                        m_symbolInfoList.AddTail(m_symbolInfo);

                        m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                        m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                        m_symbolLabelInfo.heading              =  0.0;
                        m_symbolLabelInfo.symbol_label.Format(" ");
                        m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                        m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                        m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                        m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];

                        CString  frequency, channel;
                        frequency.Format("%3.2f", (float)(unsigned short)pGnrc->vhf.sFreqX100 /  100.0);
                        BOOL  bExists  =  m_TacanChannel.Lookup(frequency, channel);
                        if (bExists && !channel.IsEmpty())
                        {
                           m_symbolLabelInfo.symbol_label         += " CH " + channel;
                        }

                        m_symbolLabelInfo.symbol_label         += '\0';
                        m_symbolLabelInfo.type                 =  TACAN;
                        m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                     #endif
                  }
                  break;

//                  case rdbType1WPT:    // Enroute Waypoints
                  case rdbType1TWPT:   // Terminal Waypoints
                  {
                     m_symbolInfo.heading =  0.0;
                     m_symbolInfo.type    =  WPT;
                     m_symbolInfo.symbol.Format("%c", WPT);
                     m_symbolInfoList.AddTail(m_symbolInfo);

                     m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
                     m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
                     m_symbolLabelInfo.heading              =  0.0;
                     m_symbolLabelInfo.symbol_label.Empty();
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                     m_symbolLabelInfo.symbol_label         += pGnrc->gen.nType2;
                     m_symbolLabelInfo.symbol_label         += '\0';
                     m_symbolLabelInfo.type                 =  WPT;
                     m_symbolLabelInfo.width                =  0.75f;
                     m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
                  }
                  break;

                  default:
                     break;
               }
               pGnrc   =  (rdbGNRC *)rdbRadioDB.StationNextLat(min_latitude, max_latitude);
            }


            // Search for JET and VICTOR routes only if they are to be displayed.
            if (m_sym_disp_victor_routes_on  || m_sym_disp_jet_routes_on)
            {
               min_longitude  =  -m_area_map_longitude_center   -  sweep_angle /  2.0;
               max_longitude  =  -m_area_map_longitude_center   +  sweep_angle /  2.0;

               // Calculate the number of degrees shown on the map (vertically, or in the latitude direction).
               sweep_angle    =  7200.0f  /  m_scale  /  NM_PER_DEGREE;
               double   min_latitude   =  m_area_map_latitude_center -  sweep_angle /  2.0;
               double   max_latitude   =  m_area_map_latitude_center +  sweep_angle /  2.0;

               // Get pointer to first Station in lat-lon range.
               rdbGNRC  *pGnrc;
               pGnrc    =  (rdbGNRC *)rdbRadioDB.StationFirstLat(min_latitude, max_latitude);

               while (pGnrc)
               {
                  CVariant var;
                  var.Value((long)pGnrc->stn.nLat);
                  m_symbolInfo.position.m_latitude             =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                  var.Value((long)pGnrc->stn.nLon);
                  m_symbolInfo.position.m_longitude            =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                  // Don't limit longitude here.

                  m_symbolInfo.heading                         =  0.0;
                  m_symbolInfo.type                            =  0;
                  m_symbolInfo.active_airport_and_runway       =  false;
                  m_symbolLabelInfo.heading                    =  0.0;
                  m_symbolLabelInfo.active_airport_and_runway  =  false;

                  DWORD dwTypePri   =  pGnrc->stn.nType1;
                  DWORD dwTypeSec   =  pGnrc->stn.nType2;

                  dwTypePri   &= 0xfffffe00u;   // Mask-out range bits.

                  switch (dwTypePri)
                  {
                     case rdbType1ENR:
                     {
                        bool  process_it  =  false;
                        // If we want to display victor routes and this is a victor route, 
                        if (pGnrc->gen.chIdent[3]  == 'V')
                        {
                           process_it              =  true;       // process this record.
                           m_symbolLabelInfo.type  =  VICTOR_AIRWAYS;
                           m_symbolInfo.type       =  VICTOR_AIRWAYS;
                        }
                        // If we want to display jet routes and this is a jet route, 
                        if (pGnrc->gen.chIdent[3]  == 'J')
                        {
                           process_it              =  true;       // process this record.
                           m_symbolLabelInfo.type  =  JET_AIRWAYS;
                           m_symbolInfo.type       =  JET_AIRWAYS;
                        }
                        if (! process_it)                         // Either we aren't diplaying this victor or jet route, or
                           break;                                 // this isn't a victor or jet route.

                        CVariant var;
                        for (int ii = 0; ii < 5; ii++)
                        {
                           if ((pGnrc->enr.nLatLon[ii][0]   == 0x0)  || (pGnrc->enr.nLatLon[ii][1] == 0x0))
                              break;
                           if ((pGnrc->enr.nLatLon[ii+1][0] == 0x0)  || (pGnrc->enr.nLatLon[ii+1][1]  == 0x0))
                              break;

                           var.Value((long)pGnrc->enr.nLatLon[ii][0]);
                           double   dLat        =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);
                           var.Value((long)pGnrc->enr.nLatLon[ii][1]);
                           double   dLon        =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);
                           var.Value((long)pGnrc->enr.nLatLon[ii+1][0]);
                           double   next_dLat   =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);
                           var.Value((long)pGnrc->enr.nLatLon[ii+1][1]);
                           double   next_dLon   =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

                           if (dLon >  max_longitude  && next_dLon   >  max_longitude)
                              continue;
                           if (dLon <  min_longitude  && next_dLon   <  min_longitude)
                              continue;
                           if (dLat >  max_latitude   && next_dLat   >  max_latitude)
                              continue;
                           if (dLat <  min_latitude   && next_dLat   <  min_latitude)
                              continue;

                           
                           // Calculate the two positions for each route label one on each side of a junction.
                           // Symbol Label Information that is common for the two labels.
                           m_symbolLabelInfo.symbol_label.Empty();
                           m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[3];
                           m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[2];
                           m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[1];
                           m_symbolLabelInfo.symbol_label         += pGnrc->gen.chIdent[0];
                           m_symbolLabelInfo.symbol_label         += '\0';

                           double   x_distance  =  (next_dLat  -  dLat) *  NM_PER_DEGREE;
                           double   y_distance  =  ((next_dLon -  dLon) *  NM_PER_DEGREE) *  cos(((next_dLat   +  dLat)  /  2.0)  *  DEG_TO_RAD);

                           m_symbolLabelInfo.position.m_latitude  =  dLat;       // Position one.
                           m_symbolLabelInfo.position.m_longitude =  dLon;
                           m_symbolLabelInfo.heading              =  270.0 -  atan2(y_distance, x_distance) *  RAD_TO_DEG;
                           m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);     // Add label for position one.

                           m_symbolLabelInfo.position.m_latitude  =  next_dLat;  // Position two.
                           m_symbolLabelInfo.position.m_longitude =  next_dLon;
                           m_symbolLabelInfo.heading              =  270.0 -  atan2(-y_distance, -x_distance) *  RAD_TO_DEG;
                           m_symbolLabelInfoList.AddTail(m_symbolLabelInfo);     // Add label for position two.

                           // Symbol information.
                           m_symbolInfo.position.m_latitude       =  dLat;  
                           m_symbolInfo.position.m_longitude      =  dLon;
                           m_symbolInfo.position_next.m_latitude  =  next_dLat;  
                           m_symbolInfo.position_next.m_longitude =  next_dLon;
                           m_symbolInfo.heading                   =  0.0;
                           m_symbolInfo.length                    =  0.0f;
                           m_symbolInfo.width                     =  0.0f;
                           m_symbolInfoList.AddTail(m_symbolInfo);
                        }
                     }
                     break;
                  }
                  pGnrc   =  (rdbGNRC *)rdbRadioDB.StationNextLat(min_latitude, max_latitude);
               }
            }

            if (m_sym_disp_custom_lines_on   || m_sym_disp_custom_obstacles_on   || m_sym_disp_custom_practice_areas_on)
            {
               // Calculate the number of degrees shown on the map (vertically, or in the latitude direction).
               sweep_angle =  7200.0f  /  m_scale  /  NM_PER_DEGREE;
               sweep_angle =  __max(sweep_angle, 0.08333333);
               // Adjust the sweep_angle to compensate for the aspect ratio.
               if (m_aspect_ratio  > 1.0f)
                  sweep_angle *=  m_aspect_ratio;
               else if (m_aspect_ratio != 0.0f)
                  sweep_angle /=  m_aspect_ratio;
               // Adjust the sweep_angle to compensate for longitude convergence.
               if (m_area_map_latitude_center   >  0.0f)
               {
                  if ((m_area_map_latitude_center  +  (sweep_angle   /  2.0f) != 90.0))
                     sweep_angle =  sweep_angle /  cos((m_area_map_latitude_center  +  (sweep_angle   /  2.0f)) *  DEG_TO_RAD);
                  else
                     sweep_angle =  sweep_angle /  0.0001;
               }
               else
               {
                  if ((m_area_map_latitude_center  -  (sweep_angle   /  2.0f) != 90.0))
                     sweep_angle =  sweep_angle /  cos((m_area_map_latitude_center  -  (sweep_angle   /  2.0f)) *  DEG_TO_RAD);
                  else
                     sweep_angle =  sweep_angle /  0.0001;
               }

               min_longitude  =  -m_area_map_longitude_center   -  sweep_angle /  2.0;
               max_longitude  =  -m_area_map_longitude_center   +  sweep_angle /  2.0;

               // Calculate the number of degrees shown on the map (vertically, or in the latitude direction).
               sweep_angle    =  7200.0f  /  m_scale  /  NM_PER_DEGREE;
               double   min_latitude   =  m_area_map_latitude_center -  sweep_angle /  2.0;
               double   max_latitude   =  m_area_map_latitude_center +  sweep_angle /  2.0;

               POSITION    pos   =  m_custom_symbolInfoList.GetHeadPosition();
               SymbolInfo           symbolInfo;

               while (pos)
               {
                  symbolInfo           =  m_custom_symbolInfoList.GetNext(pos);

                  if (symbolInfo.position.m_longitude >  max_longitude  && symbolInfo.position_next.m_longitude   >  max_longitude)
                     continue;
                  if (symbolInfo.position.m_longitude <  min_longitude  && symbolInfo.position_next.m_longitude   <  min_longitude)
                     continue;
                  if (symbolInfo.position.m_latitude  >  max_latitude   && symbolInfo.position_next.m_latitude    >  max_latitude)
                     continue;
                  if (symbolInfo.position.m_latitude  <  min_latitude   && symbolInfo.position_next.m_latitude    <  min_latitude)
                     continue;

                  m_symbolInfoList.AddTail(symbolInfo);
               }

               if (m_sym_disp_custom_lines_text_on || m_sym_disp_custom_obstacles_text_on || m_sym_disp_custom_practice_areas_text_on)
               {
                  pos   =  m_custom_symbolLabelInfoList.GetHeadPosition();
                  SymbolLabelInfo   symbolLabelInfo;

                  while (pos)
                  {
                     symbolLabelInfo   =  m_custom_symbolLabelInfoList.GetNext(pos);

                     if (symbolLabelInfo.position.m_longitude >  max_longitude)
                        continue;
                     if (symbolLabelInfo.position.m_longitude <  min_longitude)
                        continue;
                     if (symbolLabelInfo.position.m_latitude  >  max_latitude)
                        continue;
                     if (symbolLabelInfo.position.m_latitude  <  min_latitude)
                        continue;

                     m_symbolLabelInfoList.AddTail(symbolLabelInfo);
                  }
               }  // if text is on for the custom symbols
            }  // if custom symbols are on
         }  // check against scale
      }
   }

   return bRetVal;
}


void  CAreaMap::CreateNewSpiderWeb()
{
   glNewList(m_int_lists[SPIDER_WEB], GL_COMPILE);
   {
      glPushMatrix();
      {
         COpenGLCircle  circle;
         circle.ArraySize(SPIDER_WEB_NUMBER_OF_RADIALS);
         glRGB(64, 64, 64);
         float scale =  1.0f;
         for (int kk = 0; kk <= SPIDER_WEB_NUMBER_OF_RINGS; kk++)
         {
            glPushMatrix();
               glScalef(1.0f - scale, 1.0f - scale, 1.0f);
               circle.Draw();
            glPopMatrix();
            scale -= 0.2f;
         }
         for (kk = 0; kk < SPIDER_WEB_NUMBER_OF_RADIALS  /  4; kk++)
         {
            glRotatef(360.0   /  SPIDER_WEB_NUMBER_OF_RADIALS, 0.0f, 0.0f, 1.0f);
            glBegin(GL_LINES);
               glVertex2f(-2.0f, 0.0f);
               glVertex2f( 2.0f, 0.0f);
            glEnd();
            glBegin(GL_LINES);
               glVertex2f(0.0f, -2.0f);
               glVertex2f(0.0f,  2.0f);
            glEnd();
         }
      }
      glPopMatrix();


      float label_radius   =  (SPIDER_WEB_OUTER_RADIUS   -  SPIDER_WEB_INNER_RADIUS) /  SPIDER_WEB_NUMBER_OF_RINGS;
      for (int ii =  0; ii < SPIDER_WEB_NUMBER_OF_RINGS; ii+=2)
      {
         glPushMatrix();
            glRotatef(90.0f,  0.0f, 0.0f, 1.0f);
            glTranslatef(SPIDER_WEB_INNER_RADIUS   +  label_radius   *  (ii   +  1),   0.0f, 0.0f);
            webLabel.Draw(m_webChevron, true, -270.0f, 0.45f);
         glPopMatrix();

         m_webLabel.Format("%2.1fNM",  (3600.0  /  m_scale) *  label_radius   *  (ii   +  1));
         glPushMatrix();
            glRotatef(45.0f,  0.0f, 0.0f, 1.0f);
            glTranslatef(SPIDER_WEB_INNER_RADIUS   +  label_radius   *  (ii   +  1),   0.0f, 0.0f);
            webLabel.Draw(m_webLabel, true, -45.0f    +  m_spider_web_magvar, 0.45f);
         glPopMatrix();
         glPushMatrix();
            glRotatef(225.0f,  0.0f, 0.0f, 1.0f);
            glTranslatef(SPIDER_WEB_INNER_RADIUS   +  label_radius   *  (ii   +  1),   0.0f, 0.0f);
            webLabel.Draw(m_webLabel, true, -225.0f   +  m_spider_web_magvar, 0.45f);
         glPopMatrix();

         m_webLabel.Format("%2.1fNM",  (3600.0  /  m_scale) *  label_radius   *  (ii   +  2));
         glPushMatrix();
            glRotatef(135.0f,  0.0f, 0.0f, 1.0f);
            glTranslatef(SPIDER_WEB_INNER_RADIUS   +  label_radius   *  (ii   +  2),   0.0f, 0.0f);
            webLabel.Draw(m_webLabel, true, -135.0f   +  m_spider_web_magvar, 0.45f);
         glPopMatrix();
         glPushMatrix();
            glRotatef(315.0f,  0.0f, 0.0f, 1.0f);
            glTranslatef(SPIDER_WEB_INNER_RADIUS   +  label_radius   *  (ii   +  2),   0.0f, 0.0f);
            webLabel.Draw(m_webLabel, true, -315.0f   +  m_spider_web_magvar, 0.45f);
         glPopMatrix();
      }

      for (int jj =  0; jj <  SPIDER_WEB_NUMBER_OF_RADIALS; jj+=3)
      {
         float angle =  (360.0f  /  SPIDER_WEB_NUMBER_OF_RADIALS) *  jj;
         m_webLabel.Format("%.0f%c", angle, DEGREES);
         glPushMatrix();
            glRotatef(90.0f   -  angle,  0.0f, 0.0f, 1.0f);
            glTranslatef(SPIDER_WEB_INNER_RADIUS   +  label_radius   *  4.5f, 0.0f, 0.0f);
            webLabel.Draw(m_webLabel, true, (jj >  SPIDER_WEB_NUMBER_OF_RADIALS  /  2) ?  180.0f   :  0.0f, 0.45f);
         glPopMatrix();
      }
   }
   glEndList();
}


void  CAreaMap::DrawSpiderWeb()
{
   glPushMatrix();
   {
      glRotated(m_spider_web_longitude_center,  0.0,  1.0,  0.0);
      glRotated(-m_spider_web_latitude_center,  1.0,  0.0,  0.0);
      glRotated(-m_spider_web_magvar,  0.0,  0.0,  1.0);
      glTranslated(0.0, 0.0,  0.95 *  m_scale);

      if (m_spider_web_scale  != m_scale)
      {
         CreateNewSpiderWeb();   // Create the spider web list.
         m_spider_web_scale   =  m_scale;
      }

      // Draw the actual spider web.
      glCallList(m_int_lists[SPIDER_WEB]);
   }
   glPopMatrix();
}


CVariant* CAreaMap::GetValue(const CString &rstrValue)
{

   if (m_touch_storm_location_requested)
   {
	  static   bool  storm_latitude_sent  =  false;
      static   bool  storm_longitude_sent =  false;
      if (rstrValue        == "Storm_Latitude")
      {
		 storm_latitude_sent    =  true;
         return   &m_mouse_latitude;
      }
      else if (rstrValue   == "Storm_Longitude")
      {
		 storm_longitude_sent    =  true;
         return   &m_mouse_longitude;
      }
      else if (rstrValue   == "Storm_Location_Select")
      {
         if (storm_latitude_sent && storm_longitude_sent)
         {
            m_touch_storm_location_requested  =  false;
            storm_latitude_sent                 =  false;
            storm_longitude_sent                =  false;
            m_storm_location_select.Value(true);
         }
         else
            m_storm_location_select.Value(false);

         return   &m_storm_location_select;
      }
   }
   else if (rstrValue == "touch_storm_location")
   {
      m_variant.Value(false);
      return   &m_variant;
   }

   if (m_touch_reposition_requested)
   {
      static   bool  latitude_sent  =  false;
      static   bool  longitude_sent =  false;
      if (rstrValue        == "Reposition Latitude")
      {
         latitude_sent     =  true;
         return   &m_mouse_latitude;
      }
      else if (rstrValue   == "Reposition Longitude")
      {
         longitude_sent    =  true;
         return   &m_mouse_longitude;
      }
      else if (rstrValue   == "Reposition Select")
      {
         if (latitude_sent && longitude_sent)
         {
            m_touch_reposition_requested  =  false;
            latitude_sent                 =  false;
            longitude_sent                =  false;
            m_reposition_select.Value((long)20);
         }
         else
            m_reposition_select.Value((long)0);

         return   &m_reposition_select;
      }
   }
   else if (rstrValue == "touch_reposition")
   {
      m_variant.Value(false);
      return   &m_variant;
   }



   if (m_spider_web_at_cursor_requested)
   {
      static   bool  reset_reposition  =  false;

      if (rstrValue == "spider_web_latitude_center")
      {
         if (reset_reposition)
         {
            m_spider_web_at_cursor_requested =  false;
            reset_reposition                 =  false;
         }
         else
            reset_reposition                 =  true;

         m_bool_new_magvar_needed            =  true;
         m_CVar_magvar_latitude              =  m_mouse_latitude;
         m_CVar_magvar_longitude             =  m_mouse_longitude;
         return   &m_mouse_latitude;
      }
      else if (rstrValue == "spider_web_longitude_center")
      {
         if (reset_reposition)
         {
            m_spider_web_at_cursor_requested =  false;
            reset_reposition                 =  false;
         }
         else
            reset_reposition                 =  true;

         m_bool_new_magvar_needed            =  true;
         m_CVar_magvar_latitude              =  m_mouse_latitude;
         m_CVar_magvar_longitude             =  m_mouse_longitude;
         return   &m_mouse_longitude;
      }
   }

   if (m_map_center_at_cursor_requested)
   {
      static   bool  reset_reposition  =  false;

      if (rstrValue == "area_map_latitude_center")
      {
         if (reset_reposition)
         {
            m_map_center_at_cursor_requested =  false;
            reset_reposition                 =  false;
            // Update local comms to turn off area_map_center_cursor.  The last known latitude and longitude of the center of the 
            // map is saved so we don't need to remember how it got centered at that point.
            CVariant var;
            long  old_type;
            _FSI_STL::string  stlStrVariable;
            stlStrVariable = (LPCTSTR)"area_map_center_cursor";
            CCommsShared::GetLocalValue(var, stlStrVariable);
            old_type =  var.Type();
            var.Value(m_map_center_at_cursor_requested);
            var.ChangeType(old_type);
            CCommsShared::SetLocalValue(var, stlStrVariable, 0);
         }
         else
            reset_reposition                 =  true;

         return   &m_mouse_latitude;
      }
      else if (rstrValue == "area_map_longitude_center")
      {
         if (reset_reposition)
         {
            m_map_center_at_cursor_requested =  false;
            reset_reposition                 =  false;
            // Update local comms to turn off area_map_center_cursor.  The last known latitude and longitude of the center of the 
            // map is saved so we don't need to remember how it got centered at that point.
            CVariant var;
            long  old_type;
            _FSI_STL::string  stlStrVariable;
            stlStrVariable = (LPCTSTR)"area_map_center_cursor";
            CCommsShared::GetLocalValue(var, stlStrVariable);
            old_type =  var.Type();
            var.Value(m_map_center_at_cursor_requested);
            var.ChangeType(old_type);
            CCommsShared::SetLocalValue(var, stlStrVariable, 0);
         }
         else
            reset_reposition                 =  true;

         return   &m_mouse_longitude;
      }
   }

   if (rstrValue == "Station_Kill_Overlay")
   {
      if ((bool)m_varDisplayStationKill  == true)
      {
         m_varDisplayStationKill.Value(false);
         return   &m_varDisplayStationKill;
      }
      else
      {
         return   NULL;
      }
   }

   if (rstrValue == "Slews_Overlay")
   {
      if ((bool)m_varDisplaySlews  == true)
      {
         m_varDisplaySlews.Value(false);
         return   &m_varDisplaySlews;
      }
      else
      {
         return   NULL;
      }
   }

   if (rstrValue == "Storms_Overlay")
   {
      if ((bool)m_varDisplayStorms  == true)
      {
         m_varDisplayStorms.Value(false);
         return   &m_varDisplayStorms;
      }
      else
      {
         return   NULL;
      }
   }

   if (m_new_scale_value)
   {
      if (rstrValue == "area_map_scale")
      {
         m_new_scale_value =  false;
         m_scale_value.Value((float)(7200.0f /  m_scale));
         return   &m_scale_value;
      }
   }

   if (m_bool_new_magvar_needed  && !m_spider_web_at_cursor_requested)
   {
      static   bool  latitude_sent  =  false;
      static   bool  longitude_sent =  false;

      if (rstrValue        == "Magvar_Latitude")
      {
         latitude_sent     =  true;
         if (longitude_sent)
         {
            m_bool_new_magvar_needed   =  false;
            latitude_sent              =  false;
            longitude_sent             =  false;
         }
         return   &m_CVar_magvar_latitude;
      }
      else if (rstrValue   == "Magvar_Longitude")
      {
         longitude_sent    =  true;
         if (latitude_sent)
         {
            m_bool_new_magvar_needed   =  false;
            latitude_sent              =  false;
            longitude_sent             =  false;
         }
         return   &m_CVar_magvar_longitude;
      }
   }

   return   NULL;
}


void CAreaMap::OnLButtonUp(UINT nFlags, CPoint point) 
{
   if (m_bEditing == true)
   {
   }
   else
   {
      if (!m_touch_storm_location_requested && !m_touch_reposition_requested   && !m_spider_web_at_cursor_requested   && !m_map_center_at_cursor_requested   && !m_sym_disp_mouse_lat_long_decimal_degree)
      {
         IdentWindowInfo   identWindowInfo;

         POSITION current_label_list_position   =  m_identWindowInfoList.GetHeadPosition();
         while (current_label_list_position)
         {
            identWindowInfo   =  m_identWindowInfoList.GetNext(current_label_list_position);

            if (identWindowInfo.rectangle.PtInRect(point))
            {
               m_navaid.SetCurrentIdent(_FSI_STL::string((LPCTSTR)identWindowInfo.ident));
               m_varDisplayStationKill.Value(true);
               break;
            }
         }
      }
      else if (m_touch_reposition_requested)
      {
         m_varDisplaySlews.Value(true);
      }
	  else if (m_touch_storm_location_requested)
      {
         m_varDisplayStorms.Value(true);
      }
      else if (m_sym_disp_mouse_lat_long_decimal_degree)
      {
         ::OpenClipboard(NULL);
         ::EmptyClipboard();

         LPTSTR   lptstrCopy; 
         HGLOBAL  hglbCopy; 

         int   length   =  m_theDataToCopy.GetLength();

         // Allocate a global memory object for the text. 
         hglbCopy =  GlobalAlloc(GMEM_MOVEABLE, (length + 1) * sizeof(TCHAR)); 
         if (hglbCopy) 
         {
            // Lock the handle and copy the text to the buffer. 
            lptstrCopy  =  (LPTSTR)GlobalLock(hglbCopy); 
            memcpy(lptstrCopy, (LPCTSTR)m_theDataToCopy, length * sizeof(TCHAR)); 
            lptstrCopy[length]   =  (TCHAR)0;   // null character 
            GlobalUnlock(hglbCopy); 
            // Place the handle on the clipboard. 
            SetClipboardData(CF_TEXT, hglbCopy); 
         }
         CloseClipboard();
      }
      m_bLeftMouseDown     =  false;

      COpenGLWidget::OnLButtonUp(nFlags, point);
   }
}

void CAreaMap::AdjustMapCenterOrScale()
{
   // The xgas term is the north-south distance from the aircraft to the center of the map, in nautical miles.  
   double   xgas        =  (m_area_map_latitude_center_cv   -  m_ac_latitude_cv)    *  NM_PER_DEGREE;
   // The ygas term is the east-west distance from the aircraft to the center of the map, in nautical miles.  
   double   ygas        =  (-m_area_map_longitude_center_cv -  m_ac_longitude_cv)   *  NM_PER_DEGREE  *  cos(((m_area_map_latitude_center_cv +  m_ac_latitude_cv) /  2.0)  *  DEG_TO_RAD);
   // Calculate the distance to the map center, in nautical miles.
   float    ac_to_map_center_distance  =  sqrt(xgas   *  xgas  +  ygas  *  ygas);

   float    half_the_x_scale =  (7200.0f /  m_scale_cv) /  2.0f;
   float    half_the_y_scale =  (7200.0f /  m_scale_cv) /  2.0f;

   // Adjust half_the_y_scale to compensate for the aspect ratio.
   if (m_aspect_ratio  > 1.0f)
      half_the_y_scale  *=  m_aspect_ratio;
   else if (m_aspect_ratio != 0.0f)
      half_the_y_scale  /=  m_aspect_ratio;

   float scale_in_NM =  (7200.0f /  m_scale_cv);

   if (m_area_map_auto_center)
   {
      // If the distance from the center of the map to the aircraft is greater than the distance from the center of the map to the
      // edge of the map, re-center the map on the aircraft.
      if ((fabs(xgas)   >  half_the_x_scale)   || (fabs(ygas)   >  half_the_y_scale))
      {
         m_area_map_latitude_center_cv    =  m_ac_latitude_cv;
         m_area_map_longitude_center_cv   =  -m_ac_longitude_cv;
      }
   }
   else if (m_area_map_auto_scale)
   {
      // If we are more than 0.75 of the distance from the center of the map to the edge of the map then increase the scale (zoom out).
      if ((ac_to_map_center_distance   >  half_the_x_scale  *  0.75f)   || (ac_to_map_center_distance >  half_the_y_scale  *  0.75f))
      {
         if (scale_in_NM   <  50.0f)   // Only double the scale if it isn't above 50 yet.
         {
            m_new_scale_value =  true;
            m_scale_cv  /= 2.0f;       // Double the scale.
         }
      }
      // If we are less than 0.25 of the distance from the center of the map to the edge of the map then decrease the scale (zoom in).
      else if ((ac_to_map_center_distance <  half_the_x_scale  *  0.25f)   || (ac_to_map_center_distance <  half_the_y_scale  *  0.25f))
      {
         if (scale_in_NM   > 5.0f)     // Only halve the scale if it isn't below 5 yet.
         {
            m_new_scale_value =  true;
            m_scale_cv  *= 2.0f;       // Halve the scale.
         }
      }
   }
}

void CAreaMap::BeginDrawing()
{
   BeginDraw(1,1,1);    // Changed () to (1,1,1) to indicate "special" black which will be changed to white on print
}

void CAreaMap::DrawLeadAircraft()
{
   glRotated(m_lead_ac_longitude,  0.0,  1.0,  0.0);
   glRotated(-m_lead_ac_latitude,  1.0,  0.0,  0.0);
   glTranslated(0.0, 0.0, 0.95 * m_scale);

   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   aircraftLead.Draw(m_aircraft_lead, true, -m_ac_heading_lead, 1.0f);
   glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
}

void CAreaMap::DrawWingAircraft()
{
   glRotated(m_ac_longitude,  0.0,  1.0,  0.0);
   glRotated(-m_ac_latitude,  1.0,  0.0,  0.0);
   glTranslated(0.0, 0.0, 0.95 * m_scale);

   glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
   aircraftWing.Draw(m_aircraft_wing, true, -m_ac_heading_wing, 1.0f);
   glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
}

void CAreaMap::DrawStorm()
{

   // If there is an active storm cell and display storms is turned on
   if (m_storm_active && m_sym_disp_storm_on)
   {
	  // Convert storm length and width to nautical miles
      float storm_length  =  (m_storm_length  /  NMI_FT)  *  (m_scale /  3600.0f);
      float storm_width   =  (m_storm_width   /  NMI_FT)  *  (m_scale /  3600.0f);

      glPushMatrix();
	  {
		 // Set drawing origin to storm center
         glRotated(m_storm_longitude,  0.0,  1.0,  0.0);
         glRotated(-m_storm_latitude,  1.0,  0.0,  0.0);
         glTranslated(0.0, 0.0,  0.95  *  m_scale);

		 // Draw the storm name if display storm text is turned on
		 if (m_sym_disp_storm_text_on)
		 {
			CString temp;
			temp.Format("STORM\"%s\"",m_storm_name);
			stormLabel.SetForegroundColor(255, 0, 128);
            stormLabel.Draw(temp, true, 0.0f, 0.45f);
         }

		 // Rotate the storm to specified heading
         glRotated(270.0 - m_storm_heading,  0.0,  0.0,  1.0);

		 // Set the storm boundary color
         glRGB(255, 0, 128);

		 // Set storm boundary line width
         glLineWidth(1.0);

		 // Set polygon mode to line (wireframe)
         glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

		 // Draw storm boundary
         glBegin(GL_QUADS);
            glVertex2f(-storm_length / 2.0, -storm_width  /  2.0);  // top left
            glVertex2f(-storm_length / 2.0,  storm_width  /  2.0);  // bottom left
            glVertex2f(storm_length / 2.0,  storm_width  /  2.0); // bottom right
            glVertex2f(storm_length / 2.0, -storm_width  /  2.0); // top right
         glEnd();
	  }
      glPopMatrix();
   }
}

void CAreaMap::DrawSymbols(void)
{
   double   modelMatrix[16];
   double   buffer[6];

   // Symbol Labels
   if (m_scale >= 15.0)    // 480 miles
   {
      // Symbols
      glPushMatrix();
      {
         POSITION    pos   =  m_symbolInfoList.GetHeadPosition();
         SymbolInfo           symbolInfo;

         while (pos)
         {
            symbolInfo  =  m_symbolInfoList.GetNext(pos);
      
            glPushMatrix();
            {
               glRotated(symbolInfo.position.m_longitude,  0.0,  1.0,  0.0);
               glRotated(-symbolInfo.position.m_latitude,  1.0,  0.0,  0.0);
               glTranslated(0.0, 0.0,  0.95  *  m_scale);

               glRotated(270.0 - symbolInfo.heading,  0.0,  0.0,  1.0);

               switch   (symbolInfo.type)
               {
                  case  JET_AIRWAYS:
                  case  VICTOR_AIRWAYS:
                  case  PRACTICE_AREA:
                  case  LINE:
                  {
                     if (!m_sym_disp_victor_routes_on          && symbolInfo.type == VICTOR_AIRWAYS)
                         break;

                     if (!m_sym_disp_jet_routes_on             && symbolInfo.type == JET_AIRWAYS)
                         break;

                     if (!m_sym_disp_custom_practice_areas_on  && symbolInfo.type == PRACTICE_AREA)
                         break;

                     if (!m_sym_disp_custom_lines_on           && symbolInfo.type == LINE)
                         break;

                     glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
                     gluProject(0.0f, 0.0f, 0.0f, modelMatrix, m_projMatrix, m_viewport, &buffer[0], &buffer[1], &buffer[2]);


                     glPopMatrix();
                     glPushMatrix();
                     {
                        glRotated(symbolInfo.position_next.m_longitude,  0.0,  1.0,  0.0);
                        glRotated(-symbolInfo.position_next.m_latitude,  1.0,  0.0,  0.0);
                        glTranslated(0.0, 0.0,  0.95  *  m_scale);
                        glRotated(270.0 - symbolInfo.heading,  0.0,  0.0,  1.0);

                        glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);

                        gluProject(0.0f, 0.0f, 0.0f, modelMatrix, m_projMatrix, m_viewport, &buffer[3], &buffer[4], &buffer[5]);
                     }
                     glPopMatrix();
                     glPushMatrix();

                     glLoadIdentity();

                     double objx, objy, objz;
                     double next_objx, next_objy, next_objz;

                     glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);

                     gluUnProject(buffer[0], buffer[1], 0.0, modelMatrix, m_projMatrix, m_viewport, &objx,      &objy,      &objz);
                     gluUnProject(buffer[3], buffer[4], 0.0, modelMatrix, m_projMatrix, m_viewport, &next_objx, &next_objy, &next_objz);

                     switch   (symbolInfo.type)
                     {
                        case  JET_AIRWAYS:
                           glRGB(96,  96,  128);
                           glBegin(GL_LINES);
                              glVertex3f(objx, objy, 0.0f);
                              glVertex3f(next_objx, next_objy, 0.0f);
                           glEnd();
                           break;

                        case  VICTOR_AIRWAYS:
                           glRGB(192, 192, 255);
                           glBegin(GL_LINES);
                              glVertex3f(objx, objy, 0.0f);
                              glVertex3f(next_objx, next_objy, 0.0f);
                           glEnd();
                           break;
						
						case  PRACTICE_AREA:
                           glRGB(128, 64,  0);
                           glLineWidth(symbolInfo.width);
                           glEnable(GL_LINE_STIPPLE);
                           // The following is not a DEBUG vs. RELEASE issue.  It is a video card driver issue.  
                           // Using #ifdef _DEBUG is just a convenient way to destinguish on which machine (video card) the code
                           // is running.  The NVIDIA RIVA TNT2/TNT2 Pro Driver 6.5 crashes when stipple is anything but 0xFFFF and
                           // the application is minimized.  It runs fine in maximized mode.  This is the card on Charles' 
                           // development machine, which runs the _DEBUG version of the code mostly.  No problems have been seen 
                           // with the Oxygen GVX1 cards, which always are running in Release mode.
                           #ifdef _DEBUG                 
                              glLineStipple(2, 0xFFFF);
                           #else                         
                              glLineStipple(2, 0x2222);
                           #endif
                           // End of issue discussed above.
                           glBegin(GL_LINES);
                              glVertex3f(objx, objy, 0.0f);
                              glVertex3f(next_objx, next_objy, 0.0f);
                           glEnd();
                           glDisable(GL_LINE_STIPPLE);
                           glLineWidth(1.0);
                           break;

                        case  LINE:
                           glRGB(128, 128, 128);
                           glLineWidth(symbolInfo.width);
                           glBegin(GL_LINES);
                              glVertex3f(objx, objy, 0.0f);
                              glVertex3f(next_objx, next_objy, 0.0f);
                           glEnd();
                           glLineWidth(1.0);
                           break;
                     }
                  }
                  break;

                  case  ILS_FAN:
                  {
                     if (m_sym_disp_ils_on)
                     {
                        if (symbolInfo.active_airport_and_runway)
                           symbol.SetForegroundColor(m_mapColorSelectables["ACTIVE_ILS"].s_color.Red(), m_mapColorSelectables["ACTIVE_ILS"].s_color.Green(), m_mapColorSelectables["ACTIVE_ILS"].s_color.Blue());
                        else
                           symbol.SetForegroundColor(0, 128, 0);
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 110.0f * (m_scale /  3600.0f));
                     }
                  }
                  break;

                  case  AIRPORT:
                  {
                     if (m_sym_disp_airports_on)
                     {
                        symbol.SetForegroundColor(0, 192, 192);
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 1.5f);
                     }
                  }
                  break;

                  case  NDB:
                  {
                     if (m_sym_disp_ndb_on)
                     {
                        symbol.SetForegroundColor(255, 255, 0);
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 1.5f);
                     }
                  }
                  break;

                  case  DME:
                  {
                     symbol.SetForegroundColor(255, 255, 0);
                     symbol.Draw(symbolInfo.symbol);
                  }
                  break;

                  case  ILSDME:
                  {
                     if (m_sym_disp_ils_on)
                     {
                        symbol.SetForegroundColor(255, 255, 0);
                        symbol.Draw(symbolInfo.symbol);
                     }
                  }
                  break;

                  case  INNER_MARKER:
                  {
                     if (m_sym_disp_markers_on)
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 1.5f);
                  }
                  break;

                  case  MIDDLE_MARKER:
                  {
                     if (m_sym_disp_markers_on)
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 1.5f);
                  }
                  break;

                  case  OUTER_MARKER:
                  {
                     if (m_sym_disp_markers_on)
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 1.5f);
                  }
                  break;

                  case  VOR:
                  {
                     if (m_sym_disp_vor_on)
                     {
                        symbol.SetForegroundColor(192, 192, 0);
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 1.5f);
                     }
                  }
                  break;

                  case  VORDME:
                  {
                     if (m_sym_disp_vor_on)
                     {
                        symbol.SetForegroundColor(192, 192, 0);
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 1.5f);
                     }
                  }
                  break;

                  case  VORTAC:
                  case  TACAN:
                  {
                     if (m_sym_disp_vor_on)
                     {
                        symbol.SetForegroundColor(192, 192, 0);
                        symbol.Draw(symbolInfo.symbol);
                     }
                  }
                  break;

                  case  WPT:
                  {
                     if (m_sym_disp_waypoints_on)
                     {
                        symbol.SetForegroundColor(0, 255, 255);
                        symbol.Draw(symbolInfo.symbol);
                     }
                  }
                  break;

                  case  RUNWAY:
                  {
                     glPushMatrix();
                     {
                        glTranslated(-symbolInfo.length  /  2.0,  0.0,  0.0);
                        glScaled(symbolInfo.length,   symbolInfo.width, 1.0);
                        symbol.SetForegroundColor(96, 96, 96);
                        symbol.Draw(symbolInfo.symbol);
                     }
                     glPopMatrix();
                  }
                  break;

                  case  OBSTACLE:
                  case  GROUP_OBSTACLE:
                  case  SPOT_ELEVATION:
                  {
                     if (m_sym_disp_custom_obstacles_on)
                     {
                        symbol.SetForegroundColor(192, 192, 192);
                        symbol.Draw(symbolInfo.symbol, true, 90.0f, 1.0f);
                     }
                  }
                  break;

                  case  DELTA:
                  {
                     if (m_sym_disp_custom_lines_on)
                     {
                        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
                        symbol.SetForegroundColor(192, 192, 192);
                        symbol.Draw(symbolInfo.symbol, true, 0.0f, 0.5f);
                        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
                     }
                  }
                  break;

                  case  CIRCLE:
                  {
//                     if (m_sym_disp_airports_on)
                     {
                        glRGB(128, 128, 128);
                        float scale_in_NM =  (3600.0f /  m_scale);
                        float radius      =  symbolInfo.length /  scale_in_NM;
                        glRotated(-90.0-(270.0 - symbolInfo.heading),  0.0,  0.0,  1.0);
                        glScalef(-radius, radius, 1.0f);
                        glLineWidth(3.0);
                        m_circle.DrawArc(symbolInfo.heading, symbolInfo.width);
                        glLineWidth(1.0);
                     }
                  }
                  break;
               }
            }
            glPopMatrix();
         }
      }
      glPopMatrix();

      glPushMatrix();
      {
         POSITION          current_label_list_position;
         SymbolLabelInfo   symbolLabelInfo;

         bool  something_moved   =  true;
         while (something_moved)
         {
            something_moved      =  false;
            current_label_list_position   =  m_symbolLabelInfoList.GetHeadPosition();
            while (current_label_list_position)
            {
               symbolLabelInfo      =  m_symbolLabelInfoList.GetAt(current_label_list_position);

               if (// Declutter everything except for airway and runway labels.
                   (symbolLabelInfo.type   == AIRPORT && m_sym_disp_airport_text_on    && m_sym_disp_airports_on)    ||
                   (symbolLabelInfo.type   == ILS_FAN && m_sym_disp_ils_text_on        && m_sym_disp_ils_on)         ||
                   (symbolLabelInfo.type   == NDB     && m_sym_disp_ndb_text_on        && m_sym_disp_ndb_on)         ||
                   (symbolLabelInfo.type   == VOR     && m_sym_disp_vor_text_on        && m_sym_disp_vor_on)         ||
                   (symbolLabelInfo.type   == VORTAC  && m_sym_disp_vor_text_on        && m_sym_disp_vor_on)         ||
                   (symbolLabelInfo.type   == TACAN   && m_sym_disp_vor_text_on        && m_sym_disp_vor_on)         ||
                   (symbolLabelInfo.type   == VORDME  && m_sym_disp_vor_text_on        && m_sym_disp_vor_on))
               {
                  double   latitude    =  symbolLabelInfo.position.m_latitude;
                  double   longitude   =  symbolLabelInfo.position.m_longitude;

                  POSITION          other_label_list_position   =  m_symbolLabelInfoList.GetHeadPosition();
                  SymbolLabelInfo   other_symbolLabelInfo;
                  while (other_label_list_position != current_label_list_position)
                  {
                     other_symbolLabelInfo      =  m_symbolLabelInfoList.GetAt(other_label_list_position);

                     if (// Declutter everything except for airway and runway labels.
                         (other_symbolLabelInfo.type  == AIRPORT  && m_sym_disp_airport_text_on    && m_sym_disp_airports_on)    ||
                         (other_symbolLabelInfo.type  == ILS_FAN  && m_sym_disp_ils_text_on        && m_sym_disp_ils_on)         ||
                         (other_symbolLabelInfo.type  == NDB      && m_sym_disp_ndb_text_on        && m_sym_disp_ndb_on)         ||
                         (other_symbolLabelInfo.type  == VOR      && m_sym_disp_vor_text_on        && m_sym_disp_vor_on)         ||
                         (other_symbolLabelInfo.type  == VORTAC   && m_sym_disp_vor_text_on        && m_sym_disp_vor_on)         ||
                         (other_symbolLabelInfo.type  == TACAN    && m_sym_disp_vor_text_on        && m_sym_disp_vor_on)         ||
                         (other_symbolLabelInfo.type  == VORDME   && m_sym_disp_vor_text_on        && m_sym_disp_vor_on))
                     {
                        double   other_latitude    =  other_symbolLabelInfo.position.m_latitude;
                        double   other_longitude   =  other_symbolLabelInfo.position.m_longitude;

                        double   delta_longitude   =  longitude -  other_longitude;
                        double   x_offset          =  0.95  *  m_scale  *  sin(fabs(delta_longitude)  *  DEG_TO_RAD);

                        double   delta_latitude    =  latitude  -  other_latitude;
                        double   y_offset          =  0.95  *  m_scale  *  sin(fabs(delta_latitude)   *  DEG_TO_RAD);

                        if ((fabs(x_offset)  <= other_symbolLabelInfo.width)  && (fabs(y_offset)   <= other_symbolLabelInfo.height  /  2.0))
                        {
                           double   angle_encompassing_height  =  asin(other_symbolLabelInfo.height   /  (0.95 *  m_scale))   *  RAD_TO_DEG;
                           double   latitude_angle             =  fabs(angle_encompassing_height   /  2.0   -  fabs(delta_latitude));

                           latitude_angle    *= 1.05;          // Move it a tad more than necessary so it looks cleaner.
                           symbolLabelInfo.position.m_latitude += latitude_angle;
                           latitude          =  symbolLabelInfo.position.m_latitude;

                           m_symbolLabelInfoList.SetAt(current_label_list_position, symbolLabelInfo);
                           something_moved   =  true;
                        }
                     }
                     other_symbolLabelInfo   =  m_symbolLabelInfoList.GetNext(other_label_list_position);
                  }
               }
               symbolLabelInfo   =  m_symbolLabelInfoList.GetNext(current_label_list_position);
            }
         }

         // Empty the list of visible idents.
         m_identWindowInfoList.RemoveAll();


         // Go through the list of all the symbol label information and just pick out the WPTs and draw their text.  
         // Other labels will be drawn in the next section of code - which means that the other labels may overlap or overwrite the WPT labels.
         current_label_list_position   =  m_symbolLabelInfoList.GetHeadPosition();
         while (current_label_list_position)
         {
            if ((symbolLabelInfo.type  == WPT   && m_sym_disp_waypoints_text_on  && m_sym_disp_waypoints_on))
            {
               glPushMatrix();
               {
                  glRotated(symbolLabelInfo.position.m_longitude,   0.0,  1.0,  0.0);
                  glRotated(-symbolLabelInfo.position.m_latitude,   1.0,  0.0,  0.0);
                  const   float symbol_scale   =  0.45f;

                  glRotated(symbolLabelInfo.heading,  0.0,  0.0,  1.0);
                  int   middle_of_label   =  symbolLabelInfo.symbol_label.GetLength()  /  2;
                  glTranslated(0.025   +  (0.015   *  middle_of_label), 0.0225,  0.95  *  m_scale);

                  switch   (symbolLabelInfo.type)
                  {
                     case  WPT:
                        symbol.SetForegroundColor(0, 255, 255);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale   *  symbolLabelInfo.width, 1.0f, false);
                        break;
                  }
               }
               glPopMatrix();
            }
            symbolLabelInfo   =  m_symbolLabelInfoList.GetNext(current_label_list_position);
         }


         // Go through the list of all the symbol label information and draw their text - excluding WPTs.  
         // WPT labels were drawn in the above section of code.
         current_label_list_position   =  m_symbolLabelInfoList.GetHeadPosition();
         while (current_label_list_position)
         {
            if ((symbolLabelInfo.type  == RUNWAY            && m_scale  >= 7200.0f  /  5.0f                                                     )  ||
                (symbolLabelInfo.type  == AIRPORT           && m_sym_disp_airport_text_on                && m_sym_disp_airports_on              )  ||
                (symbolLabelInfo.type  == ILS_FAN           && m_sym_disp_ils_text_on                    && m_sym_disp_ils_on                   )  ||
                (symbolLabelInfo.type  == NDB               && m_sym_disp_ndb_text_on                    && m_sym_disp_ndb_on                   )  ||
                (symbolLabelInfo.type  == VOR               && m_sym_disp_vor_text_on                    && m_sym_disp_vor_on                   )  ||
                (symbolLabelInfo.type  == VORTAC            && m_sym_disp_vor_text_on                    && m_sym_disp_vor_on                   )  ||
                (symbolLabelInfo.type  == TACAN             && m_sym_disp_vor_text_on                    && m_sym_disp_vor_on                   )  ||
                (symbolLabelInfo.type  == VORDME            && m_sym_disp_vor_text_on                    && m_sym_disp_vor_on                   )  ||
                (symbolLabelInfo.type  == VICTOR_AIRWAYS    && m_sym_disp_victor_routes_on                                                      )  ||
                (symbolLabelInfo.type  == JET_AIRWAYS       && m_sym_disp_jet_routes_on                                                         )  ||
                (symbolLabelInfo.type  == PRACTICE_AREA     && m_sym_disp_custom_practice_areas_text_on  && m_sym_disp_custom_practice_areas_on )  ||
                (symbolLabelInfo.type  == OBSTACLE          && m_sym_disp_custom_obstacles_text_on       && m_sym_disp_custom_obstacles_on      )  ||
                (symbolLabelInfo.type  == GROUP_OBSTACLE    && m_sym_disp_custom_obstacles_text_on       && m_sym_disp_custom_obstacles_on      )  ||
                (symbolLabelInfo.type  == SPOT_ELEVATION    && m_sym_disp_custom_obstacles_text_on       && m_sym_disp_custom_obstacles_on      )  ||
                (symbolLabelInfo.type  == LINE              && m_sym_disp_custom_lines_text_on           && m_sym_disp_custom_lines_on          )
               )
            {
               glPushMatrix();
               {
                  glRotated(symbolLabelInfo.position.m_longitude,   0.0,  1.0,  0.0);
                  glRotated(-symbolLabelInfo.position.m_latitude,   1.0,  0.0,  0.0);
                  const   float symbol_scale   =  0.45f;

                  if (symbolLabelInfo.type   == VICTOR_AIRWAYS || symbolLabelInfo.type == JET_AIRWAYS)
                  {
                     glRotated(symbolLabelInfo.heading,  0.0,  0.0,  1.0);
                     glTranslated(-0.25,  0.0,  0.95  *  m_scale);
                  }
                  else if (symbolLabelInfo.type == OBSTACLE          || 
                           symbolLabelInfo.type == GROUP_OBSTACLE    ||
                           symbolLabelInfo.type == SPOT_ELEVATION    ||
                           symbolLabelInfo.type == WPT)
                  {
                     glRotated(symbolLabelInfo.heading,  0.0,  0.0,  1.0);
                     int   middle_of_label   =  symbolLabelInfo.symbol_label.GetLength()  /  2;
                     glTranslated(0.025   +  (0.015   *  middle_of_label), 0.0125,  0.95  *  m_scale);
                  }
                  else  // Generate a list of coordinates for the rectangles used to select stations to kill.
                  {
                     int   middle_of_label   =  symbolLabelInfo.symbol_label.GetLength()  /  2;
                     glTranslated(0.075   +  (0.015   *  middle_of_label), -0.025,  0.95  *  m_scale);
                     glRotated(symbolLabelInfo.heading,  0.0,  0.0,  1.0);
                     glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
                     glGetDoublev(GL_PROJECTION_MATRIX, m_projMatrix);

                     float   upper_left_x   =  -symbolLabelInfo.width  /  2.0f  *  symbol_scale;
                     float   upper_left_y   =  symbolLabelInfo.height  /  2.0f  *  symbol_scale;
                     float   lower_right_x  =  symbolLabelInfo.width   /  2.0f  *  symbol_scale;
                     float   lower_right_y  =  -symbolLabelInfo.height /  2.0f  *  symbol_scale;

                     gluProject(upper_left_x,   upper_left_y,  0.0f, modelMatrix, m_projMatrix, m_viewport, &buffer[0], &buffer[1], &buffer[2]);
                     gluProject(lower_right_x,  lower_right_y, 0.0f, modelMatrix, m_projMatrix, m_viewport, &buffer[3], &buffer[4], &buffer[5]);

                     if (symbolLabelInfo.type   == AIRPORT   || symbolLabelInfo.type == RUNWAY  || symbolLabelInfo.type == NDB   || symbolLabelInfo.type == WPT)
                     {
                        // Keep AIRPORTs, RUNWAYs, NDBs and WPTs from being selectable...
                     }
                     else if (m_pBaseWidget->Wnd())
                     {
                        CWnd  *pWnd =  m_pBaseWidget->Wnd()->GetParent();
                        if (pWnd)
                        {
                           m_identWindowInfo.ident       =  symbolLabelInfo.symbol_label.Mid(symbolLabelInfo.symbol_label.Find(" ") + 1, 4);
                           m_identWindowInfo.rectangle   =  CRect(CPoint(buffer[0], m_height - buffer[1]), CPoint(buffer[3], m_height - buffer[4]));
                           pWnd->ClientToScreen(&m_identWindowInfo.rectangle);
                           m_pWnd->ScreenToClient(&m_identWindowInfo.rectangle);
                           m_identWindowInfoList.AddTail(m_identWindowInfo);
                        }
                     }
                  }

                  long     navaid_status  =  0;
                  long     navaid_config  =  0;

                  switch   (symbolLabelInfo.type)
                  {
                     case  ILS_FAN:
                     case  NDB:
                     case  ILSDME:
                     case  VOR:
                     case  VORDME:
                     case  VORTAC:
                     case  TACAN:
                     {
                        CString  temp  =  symbolLabelInfo.symbol_label.Mid(symbolLabelInfo.symbol_label.Find(" ") + 1, 4);
                        navaid_status  =  m_navaid.GetIdentStatus((LPCTSTR)temp);
                        if (navaid_status)
                        {
                           navaid_config  =  m_navaid.GetIdentConfig((LPCTSTR)temp);

                           if (navaid_status == navaid_config)             // NAVAID Total Kill
                           {
                              symbol.SetBoxFillColor(    m_mapColorSelectables["NAVAID TOTAL KILL BACKGROUND"].s_color.Red(),
                                                         m_mapColorSelectables["NAVAID TOTAL KILL BACKGROUND"].s_color.Green(),
                                                         m_mapColorSelectables["NAVAID TOTAL KILL BACKGROUND"].s_color.Blue());
                           
                              symbol.SetForegroundColor( m_mapColorSelectables["NAVAID TOTAL KILL FOREGROUND"].s_color.Red(),
                                                         m_mapColorSelectables["NAVAID TOTAL KILL FOREGROUND"].s_color.Green(),
                                                         m_mapColorSelectables["NAVAID TOTAL KILL FOREGROUND"].s_color.Blue());
                           }
                           else                                            // NAVAID Partial Kill
                           {
                              symbol.SetBoxFillColor(    m_mapColorSelectables["NAVAID PARTIAL KILL BACKGROUND"].s_color.Red(),
                                                         m_mapColorSelectables["NAVAID PARTIAL KILL BACKGROUND"].s_color.Green(),
                                                         m_mapColorSelectables["NAVAID PARTIAL KILL BACKGROUND"].s_color.Blue());
                           
                              symbol.SetForegroundColor( m_mapColorSelectables["NAVAID PARTIAL KILL FOREGROUND"].s_color.Red(),
                                                         m_mapColorSelectables["NAVAID PARTIAL KILL FOREGROUND"].s_color.Green(),
                                                         m_mapColorSelectables["NAVAID PARTIAL KILL FOREGROUND"].s_color.Blue());
                           }
                        }
                        else                                               // NAVAID No Kill
                        {
                           symbol.SetBoxFillColor(    m_mapColorSelectables["NAVAID NO KILL BACKGROUND"].s_color.Red(),
                                                      m_mapColorSelectables["NAVAID NO KILL BACKGROUND"].s_color.Green(),
                                                      m_mapColorSelectables["NAVAID NO KILL BACKGROUND"].s_color.Blue());
                           
                           symbol.SetForegroundColor( m_mapColorSelectables["NAVAID NO KILL FOREGROUND"].s_color.Red(),
                                                      m_mapColorSelectables["NAVAID NO KILL FOREGROUND"].s_color.Green(),
                                                      m_mapColorSelectables["NAVAID NO KILL FOREGROUND"].s_color.Blue());
                        }
                     }
                     break;

                     default:
                     {
                        navaid_status  =  0;
                        navaid_config  =  0;
                        symbol.SetBoxFillColor(    m_mapColorSelectables["NAVAID NO KILL BACKGROUND"].s_color.Red(),
                                                   m_mapColorSelectables["NAVAID NO KILL BACKGROUND"].s_color.Green(),
                                                   m_mapColorSelectables["NAVAID NO KILL BACKGROUND"].s_color.Blue());
                        
                        symbol.SetForegroundColor( m_mapColorSelectables["NAVAID NO KILL FOREGROUND"].s_color.Red(),
                                                   m_mapColorSelectables["NAVAID NO KILL FOREGROUND"].s_color.Green(),
                                                   m_mapColorSelectables["NAVAID NO KILL FOREGROUND"].s_color.Blue());
                     }
                     break;
                  }

                  switch   (symbolLabelInfo.type)
                  {
                     case  ILS_FAN:
                        if (symbolLabelInfo.active_airport_and_runway)
                        {
                           symbol.SetBoxOutlineColor(m_mapColorSelectables["ACTIVE_ILS"].s_color.Red(),  m_mapColorSelectables["ACTIVE_ILS"].s_color.Green(),  m_mapColorSelectables["ACTIVE_ILS"].s_color.Blue());
                           if (navaid_status == 0)
                           {
                              symbol.SetForegroundColor(255, 255, 255);
                              symbol.SetBoxFillColor(m_mapColorSelectables["ACTIVE_ILS"].s_color.Red()/4,  m_mapColorSelectables["ACTIVE_ILS"].s_color.Green()/4,  m_mapColorSelectables["ACTIVE_ILS"].s_color.Blue()/4);
                           }
                        }
                        else
                        {
                           symbol.SetBoxOutlineColor(0, 128, 0);
                        }
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale, 1.0f, true);
                        break;

                     case  AIRPORT:
                        symbol.SetBoxOutlineColor(0, 192, 192);
                        if (symbolLabelInfo.active_airport_and_runway)
                        {
                           symbol.SetForegroundColor(255, 255, 255);
                           symbol.SetBoxFillColor(0/4,  192/4,  192/4);
                        }
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale, 1.0f, true);
                        break;

                     case  NDB:
                        symbol.SetBoxOutlineColor(255, 255, 0);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale, 1.0f, true);
                        break;

                     case  ILSDME:
                        symbol.SetBoxOutlineColor(255, 255, 0);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale, 1.0f, true);
                        break;

                     case  VOR:
                        symbol.SetBoxOutlineColor(192, 192, 0);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale, 1.0f, true);
                        break;

                     case  VORDME:
                        symbol.SetBoxOutlineColor(192, 192, 0);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale, 1.0f, true);
                        break;

                     case  VORTAC:
                     case  TACAN:
                        symbol.SetBoxOutlineColor(192, 192, 0);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale, 1.0f, true);
                        break;

                     case  RUNWAY:
                        symbol.SetForegroundColor(192, 192, 192);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, -symbolLabelInfo.heading, symbol_scale, 1.0f, false);
                        break;

                     case  JET_AIRWAYS:
                     case  VICTOR_AIRWAYS:
                        symbol.Draw(symbolLabelInfo.symbol_label, true, -symbolLabelInfo.heading, symbol_scale, 1.0f, false);
                        break;

                     case  PRACTICE_AREA:
                        symbol.SetForegroundColor(128, 64,  0);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, -symbolLabelInfo.heading, symbol_scale * symbolLabelInfo.width, 1.0f, false);
                        break;

                     case  LINE:
                        {
                           symbol.SetForegroundColor(128, 128, 128);
                           symbol.SetBackgroundColor(true, 1, 1, 1);
                           CString  modified_label(" "   +  symbolLabelInfo.symbol_label  +  " ");
                           symbol.Draw(modified_label, true, 0.0f, symbol_scale * symbolLabelInfo.width, 1.0f, false);
                           symbol.SetBackgroundColor(false, 1, 1, 1);
                        }
                        break;

                     case  OBSTACLE:
                     case  GROUP_OBSTACLE:
                     case  SPOT_ELEVATION:
                        symbol.SetForegroundColor(192, 192, 192);
                        symbol.Draw(symbolLabelInfo.symbol_label, true, 0.0f, symbol_scale   *  symbolLabelInfo.width);
                        break;

                     case  DME:
                     case  INNER_MARKER:
                     case  MIDDLE_MARKER:
                     case  OUTER_MARKER:
                        // No text to display for these types.
                        break;
                  }
               }
               glPopMatrix();
            }
            symbolLabelInfo   =  m_symbolLabelInfoList.GetNext(current_label_list_position);
         }


      }
      glPopMatrix();
   }
}


void CAreaMap::DrawLeadTrack(void)
{
   if (m_sym_disp_lead_ac_track_on)
   {
      // Aircraft Track information display.
      glPushMatrix();
         m_aircraftLeadTrack.Draw();
      glPopMatrix();

      glPushMatrix();
         DrawLeadAircraft();
      glPopMatrix();
   }
}

void CAreaMap::DrawWingTrack(void)
{
   if (m_sym_disp_wing_ac_track_on)
   {
      // Aircraft Track information display.
      glPushMatrix();
         m_aircraftWingTrack.Draw();
      glPopMatrix();

      glPushMatrix();
         DrawWingAircraft();
      glPopMatrix();
   }
}


void CAreaMap::SetupPerspective()
{
}

void CAreaMap::MoveViewingPosition()
{
}

void CAreaMap::DrawEarth()
{
}

void CAreaMap::EndDrawing()
{
    EndDraw();
}


void  CAreaMap::CenterSpiderWebOnAirport(CString &id)
{
   bool  valid =  GetAirportLatLon(id, m_spider_web_latitude_center_cv, m_spider_web_longitude_center_cv);

   if (valid)
   {
      spider_web_airport_center_latitude  =  m_spider_web_latitude_center_cv;
      spider_web_airport_center_longitude =  m_spider_web_longitude_center_cv;
   }
}


void  CAreaMap::CenterSpiderWebOnNavaid(CString &id)
{
   if (m_spider_web_latitude_center_cv < 1.0f)  // Set initial lat/lon to North America
   {
       m_spider_web_latitude_center_cv = 39.0f;
       m_spider_web_longitude_center_cv = -104.0f;
   }

   bool  valid =  GetNavaidLatLon(id, m_spider_web_latitude_center_cv, m_spider_web_longitude_center_cv, m_spider_web_magvar_cv);

   if (valid)
   {
      spider_web_navaid_center_latitude  =  m_spider_web_latitude_center_cv;
      spider_web_navaid_center_longitude =  m_spider_web_longitude_center_cv;
      m_spider_web_magvar                =  m_spider_web_magvar_cv;
   }
}


void  CAreaMap::DrawWeatherAreas(void)
{
   static   CString  temp;
   float    scale_in_NM =  (3600.0f /  m_scale);

   if (m_weather_area_1_loaded)
   {
      glPushMatrix();
      {
         glRotated(m_weather_area_1_longitude,  0.0,  1.0,  0.0);
         glRotated(-m_weather_area_1_latitude,  1.0,  0.0,  0.0);
         glTranslated(0.0, 0.0,  0.95 *  m_scale);

         if (m_weather_area_1_active)
         {
            glRGB(255, 0, 255);
            weatherAreaLabel.SetForegroundColor(255, 0, 255);
         }
         else
         {
            glRGB(128, 0, 128);
            weatherAreaLabel.SetForegroundColor(128, 0, 128);
         }

         float radius   =  m_weather_area_1_radius /  scale_in_NM;
         glPushMatrix();
            glTranslatef(radius, 0.0f, 0.0f);
            temp =  "WX 1";
            weatherAreaLabel.Draw(temp, true, 0.0f, 0.45f);
         glPopMatrix();

         glScalef(radius, radius, 1.0f);
         m_circle.Draw();
      }
      glPopMatrix();
   }

   if (m_weather_area_2_loaded)
   {
      glPushMatrix();
      {
         glRotated(m_weather_area_2_longitude,  0.0,  1.0,  0.0);
         glRotated(-m_weather_area_2_latitude,  1.0,  0.0,  0.0);
         glTranslated(0.0, 0.0,  0.95 *  m_scale);

         if (m_weather_area_2_active)
         {
            glRGB(255, 0, 255);
            weatherAreaLabel.SetForegroundColor(255, 0, 255);
         }
         else
         {
            glRGB(128, 0, 128);
            weatherAreaLabel.SetForegroundColor(128, 0, 128);
         }

         float radius   =  m_weather_area_2_radius /  scale_in_NM;
         glPushMatrix();
            glTranslatef(radius, 0.0f, 0.0f);
            temp =  "WX 2";
            weatherAreaLabel.Draw(temp, true, 0.0f, 0.45f);
         glPopMatrix();

         glScalef(radius, radius, 1.0f);
         m_circle.Draw();
      }
      glPopMatrix();
   }

   if (m_weather_area_3_loaded)
   {
      glPushMatrix();
      {
         glRotated(m_weather_area_3_longitude,  0.0,  1.0,  0.0);
         glRotated(-m_weather_area_3_latitude,  1.0,  0.0,  0.0);
         glTranslated(0.0, 0.0,  0.95 *  m_scale);

         if (m_weather_area_3_active)
         {
            glRGB(255, 0, 255);
            weatherAreaLabel.SetForegroundColor(255, 0, 255);
         }
         else
         {
            glRGB(128, 0, 128);
            weatherAreaLabel.SetForegroundColor(128, 0, 128);
         }

         float radius   =  m_weather_area_3_radius /  scale_in_NM;
         glPushMatrix();
            glTranslatef(radius, 0.0f, 0.0f);
            temp =  "WX 3";
            weatherAreaLabel.Draw(temp, true, 0.0f, 0.45f);
         glPopMatrix();

         glScalef(radius, radius, 1.0f);
         m_circle.Draw();
      }
      glPopMatrix();
   }

   if (m_weather_area_4_loaded)
   {
      glPushMatrix();
      {
         glRotated(m_weather_area_4_longitude,  0.0,  1.0,  0.0);
         glRotated(-m_weather_area_4_latitude,  1.0,  0.0,  0.0);
         glTranslated(0.0, 0.0,  0.95 *  m_scale);

         if (m_weather_area_4_active)
         {
            glRGB(255, 0, 255);
            weatherAreaLabel.SetForegroundColor(255, 0, 255);
         }
         else
         {
            glRGB(128, 0, 128);
            weatherAreaLabel.SetForegroundColor(128, 0, 128);
         }

         float radius   =  m_weather_area_4_radius /  scale_in_NM;
         glPushMatrix();
            glTranslatef(radius, 0.0f, 0.0f);
            temp =  "WX 4";
            weatherAreaLabel.Draw(temp, true, 0.0f, 0.45f);
         glPopMatrix();

         glScalef(radius, radius, 1.0f);
         m_circle.Draw();
      }
      glPopMatrix();
   }
}


bool  GetAirportLatLon(CString &id, double &latitude, double &longitude)
{
   id.MakeUpper();
   int   length   =  id.GetLength();
   for (; length < 4; length++)
   {
      id += " ";
   }

   CString  temp(id);
   temp.MakeReverse();
   int   int_ident   =  (int)*(int *)LPCTSTR(temp);
   rdbStation  *pStation   =  COpenGLMap::rdbRadioDB.Airport(int_ident);

   if (pStation)
   {
      CVariant var;
      var.Value((long)pStation->nLat);
      latitude    =   (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

      var.Value((long)pStation->nLon);
      longitude   =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);
      return   true;
   }
   else
   {
      CString  message;
      message.Format("Sorry, no airport with an ident of  \"%s\" could be found.", id);
      COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
      COpenGLMap::m_stlStrCaption   =  "Invalid Ident";
      AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
      return   false;
   }

   return   false;
}


bool  GetNavaidLatLon(CString &id, double &latitude, double &longitude, float &magvar)
{
   id.MakeUpper();
   CString  temp(id);

   int   length   =  temp.GetLength();
   for (; length < 4; length++)
   {
      temp += " ";
   }

   temp.MakeReverse();
   int   int_ident         =  (int)*(int *)LPCTSTR(temp);
   int   nNumNavAids       =  0;
   rdbStation  *pStation   =  COpenGLMap::rdbRadioDB.NearestNavAid(int_ident, latitude, longitude, nNumNavAids);

   if (pStation)
   {
      if((pStation->nType1 & rdbType1ILS)    == rdbType1ILS     ||
         (pStation->nType1 & rdbType1ILSDME) == rdbType1ILSDME  ||
         (pStation->nType1 & rdbType1NDB)    == rdbType1NDB)
      {
         CString  message;
         message.Format("Sorry, NAVAID \"%s\" is not a VOR, DME or TACAN.", id);
         COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
         COpenGLMap::m_stlStrCaption   =  "Invalid Ident";
         AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
         return   false;
      }
      else
      {
         CVariant var;
         rdbVHF *pVHF = (rdbVHF *)pStation;

         var.Value((long)pVHF->nLat);
         latitude    =   (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

         var.Value((long)pVHF->nLon);
         longitude   =  (double)CDataConversion::Convert(var, BAMS, Decimal_Degrees);

         var.Value((short)pVHF->sDeclX10);
         magvar =  (float)var / 10.0;

         return   true;
      }
   }
   else
   {
      CString  message;
      message.Format("Sorry, no NAVAID with an ident of  \"%s\" could be found.", id);
      COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
      COpenGLMap::m_stlStrCaption   =  "Invalid Ident";
      AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
      return   false;
   }

   return   false;
}


void  CAreaMap::LookupCustomData(void)
{
   CString  fullProfilePath("areamap.ini");
   long     number_of_practice_areas   =  0;
   long     number_of_obstacles        =  0;
   long     number_of_group_obstacles  =  0;
   long     number_of_spot_elevations  =  0;
   long     number_of_lines            =  0;
   long     number_of_arcs             =  0;
   float    relative_font_size         =  1.0f;
   int      count =  0;
   char     buffer[2048];
   int      ii, jj, kk, zz;
   char     delimeters[]   =  "=,";
   char     *token;

   m_custom_symbolInfoList.RemoveAll();
   m_custom_symbolLabelInfoList.RemoveAll();

   // Open the registry.
   HKEY  hKey;
   RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\FlightSafety\\FSISuite\\Radio Database\\"), 0, KEY_READ, &hKey);
   if (hKey != NULL)
   {
      CString  strValue;
      DWORD    dwType,  dwCount;

      LONG  lResult  =  RegQueryValueEx(hKey, _T("AreaMapCustomDataFile"), NULL, &dwType, NULL, &dwCount);
      if (lResult == ERROR_SUCCESS)
      {
         ASSERT(dwType  == REG_SZ);
         lResult  =  RegQueryValueEx(hKey, _T("AreaMapCustomDataFile"), NULL, &dwType, (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)), &dwCount);
         strValue.ReleaseBuffer();
      }
      fullProfilePath   =  strValue;

      RegCloseKey(hKey);
   }


   count =  GetPrivateProfileSection("HEADER",  buffer,  sizeof(buffer), fullProfilePath);
   for (zz = 0; zz < count; zz++)
   {
      if (buffer[zz] == 0)
         buffer[zz]  =  ',';
   }

   token =  strtok(buffer, delimeters);                                 // Get the first string.
   while (token)
   {
      if (stricmp(token,      "Number of Practice Areas")   == 0)
      {
         token =  strtok(NULL, delimeters);                             // Get the next string.
         number_of_practice_areas   =  atoi(token);
      }
      else if (stricmp(token, "Number of Obstacles")        == 0)
      {
         token =  strtok(NULL, delimeters);                             // Get the next string.
         number_of_obstacles        =  atoi(token);
      }
      else if (stricmp(token, "Number of Group Obstacles")  == 0)
      {
         token =  strtok(NULL, delimeters);                             // Get the next string.
         number_of_group_obstacles  =  atoi(token);
      }
      else if (stricmp(token, "Number of Spot Elevations")  == 0)
      {
         token =  strtok(NULL, delimeters);                             // Get the next string.
         number_of_spot_elevations  =  atoi(token);
      }
      else if (stricmp(token, "Number of Lines")            == 0)
      {
         token =  strtok(NULL, delimeters);                             // Get the next string.
         number_of_lines            =  atoi(token);
      }
      else if (stricmp(token, "Number of Arcs")             == 0)
      {
         token =  strtok(NULL, delimeters);                             // Get the next string.
         number_of_arcs             =  atoi(token);
      }
      else if (stricmp(token, "Relative Font Size")         == 0)
      {
         token =  strtok(NULL, delimeters);                             // Get the next string.
         relative_font_size         =  atof(token);
      }
      token =  strtok(NULL, delimeters);                                // Get the next string.
   }

   if (number_of_obstacles  >  0)
   {
      CString  CStr_obstacles;

      for (ii = 0; ii < number_of_obstacles; ii++)
      {
         CStr_obstacles.Format("Obstacle %d", ii + 1);
         count =  GetPrivateProfileSection(CStr_obstacles,  buffer,  sizeof(buffer),   fullProfilePath);
         for (int zz = 0; zz < count; zz++)
         {
            if (buffer[zz] == 0)
               buffer[zz]  =  ',';
         }

         bool  got_name =  false,   got_latitude   =  false,   got_longitude  =  false;

         token =  strtok(buffer, delimeters);                                 // Get the first string.
         while (token)
         {
            if (stricmp(token, "Name") == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               if (stricmp(token, "none")             != 0)
                  m_symbolLabelInfo.symbol_label.Format("%s", token);
               else
                  m_symbolLabelInfo.symbol_label.Empty();
               got_name                               =  true;
            }
            else if (stricmp(token, "Latitude")       == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_latitude       =  atof(token);
               got_latitude                           =  true;
            }
            else if (stricmp(token, "Longitude")      == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_longitude      =  atof(token);
               got_longitude                          =  true;
            }

            if (got_name   && got_latitude   && got_longitude)
            {
               m_symbolInfo.type                      =  OBSTACLE;
               m_symbolInfo.heading                   =  0.0;
               m_symbolInfo.symbol.Format("%c", OBSTACLE);
               m_custom_symbolInfoList.AddTail(m_symbolInfo);

               m_symbolLabelInfo.heading              =  0.0;
               m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
               m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
               m_symbolLabelInfo.type                 =  OBSTACLE;
               m_symbolLabelInfo.width                =  relative_font_size;  // Relative size of the font.
               m_custom_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
            }
            token =  strtok(NULL, delimeters);                                // Get the next string.
         }
      }
   }


   if (number_of_group_obstacles  >  0)
   {
      CString  CStr_group_obstacles;

      for (ii = 0; ii < number_of_group_obstacles; ii++)
      {
         CStr_group_obstacles.Format("Group Obstacle %d", ii + 1);
         count =  GetPrivateProfileSection(CStr_group_obstacles,  buffer,  sizeof(buffer),   fullProfilePath);
         for (int zz = 0; zz < count; zz++)
         {
            if (buffer[zz] == 0)
               buffer[zz]  =  ',';
         }

         bool  got_name =  false,   got_latitude   =  false,   got_longitude  =  false;

         token =  strtok(buffer, delimeters);                                 // Get the first string.
         while (token)
         {
            if (stricmp(token, "Name") == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               if (stricmp(token, "none")             != 0)
                  m_symbolLabelInfo.symbol_label.Format("%s", token);
               else
                  m_symbolLabelInfo.symbol_label.Empty();
               got_name                               =  true;
            }
            else if (stricmp(token, "Latitude")       == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_latitude       =  atof(token);
               got_latitude                           =  true;
            }
            else if (stricmp(token, "Longitude")      == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_longitude      =  atof(token);
               got_longitude                          =  true;
            }

            if (got_name   && got_latitude   && got_longitude)
            {
               m_symbolInfo.type                      =  GROUP_OBSTACLE;
               m_symbolInfo.heading                   =  0.0;
               m_symbolInfo.symbol.Format("%c", GROUP_OBSTACLE);
               m_custom_symbolInfoList.AddTail(m_symbolInfo);

               m_symbolLabelInfo.heading              =  0.0;
               m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
               m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
               m_symbolLabelInfo.type                 =  GROUP_OBSTACLE;
               m_symbolLabelInfo.width                =  relative_font_size;  // Relative size of the font.
               m_custom_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
            }
            token =  strtok(NULL, delimeters);                                // Get the next string.
         }
      }
   }


   if (number_of_spot_elevations  >  0)
   {
      CString  CStr_spot_elevations;

      for (ii = 0; ii < number_of_spot_elevations; ii++)
      {
         CStr_spot_elevations.Format("Spot Elevation %d", ii + 1);
         count =  GetPrivateProfileSection(CStr_spot_elevations,  buffer,  sizeof(buffer),   fullProfilePath);
         for (int zz = 0; zz < count; zz++)
         {
            if (buffer[zz] == 0)
               buffer[zz]  =  ',';
         }

         bool  got_name =  false,   got_latitude   =  false,   got_longitude  =  false;

         token =  strtok(buffer, delimeters);                                 // Get the first string.
         while (token)
         {
            if (stricmp(token, "Name") == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               if (stricmp(token, "none")             != 0)
                  m_symbolLabelInfo.symbol_label.Format("%s", token);
               else
                  m_symbolLabelInfo.symbol_label.Empty();
               got_name                               =  true;
            }
            else if (stricmp(token, "Latitude")       == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_latitude       =  atof(token);
               got_latitude                           =  true;
            }
            else if (stricmp(token, "Longitude")      == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_longitude      =  atof(token);
               got_longitude                          =  true;
            }

            if (got_name   && got_latitude   && got_longitude)
            {
               m_symbolInfo.type                      =  SPOT_ELEVATION;
               m_symbolInfo.heading                   =  0.0;
               m_symbolInfo.symbol.Format("%c", SPOT_ELEVATION);
               m_custom_symbolInfoList.AddTail(m_symbolInfo);

               m_symbolLabelInfo.heading              =  0.0;
               m_symbolLabelInfo.position.m_latitude  =  m_symbolInfo.position.m_latitude;
               m_symbolLabelInfo.position.m_longitude =  m_symbolInfo.position.m_longitude;
               m_symbolLabelInfo.type                 =  SPOT_ELEVATION;
               m_symbolLabelInfo.width                =  relative_font_size;  // Relative size of the font.
               m_custom_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
            }
            token =  strtok(NULL, delimeters);                                // Get the next string.
         }
      }
   }


   if (number_of_lines  >  0)
   {
      CString  CStr_lines;

      for (ii = 0; ii < number_of_lines; ii++)
      {
         CStr_lines.Format("Line %d", ii + 1);
         count =  GetPrivateProfileSection(CStr_lines,   buffer,  sizeof(buffer),   fullProfilePath);
         for (int zz = 0; zz < count; zz++)
         {
            if (buffer[zz] == 0)
               buffer[zz]  =  ',';
         }

         bool  got_name          =  false;
         bool  got_arrow_1       =  false,   got_arrow_2       =  false;
         bool  got_latitude_1    =  false,   got_latitude_2    =  false;
         bool  got_longitude_1   =  false,   got_longitude_2   =  false;
         bool  need_arrow_1      =  false,   need_arrow_2      =  false;

         token =  strtok(buffer, delimeters);                                 // Get the first string.
         while (token)
         {
            if (stricmp(token, "Name")                == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               if (stricmp(token, "none")             != 0)
                  m_symbolLabelInfo.symbol_label.Format("%s", token);
               else
                  m_symbolLabelInfo.symbol_label.Empty();
               got_name                               =  true;
            }
            else if (stricmp(token, "Arrow 1")        == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               if (stricmp(token,   "Yes")            == 0)
                  need_arrow_1                        =  true;
               got_arrow_1                            =  true;
            }
            else if (stricmp(token, "Arrow 2")        == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               if (stricmp(token,   "Yes")            == 0)
                  need_arrow_2                        =  true;
               got_arrow_2                            =  true;
            }
            else if (stricmp(token, "Latitude 1")     == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_latitude       =  atof(token);
               got_latitude_1                         =  true;
            }
            else if (stricmp(token, "Longitude 1")    == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_longitude      =  atof(token);
               got_longitude_1                        =  true;
            }
            else if (stricmp(token, "Latitude 2")     == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position_next.m_latitude  =  atof(token);
               got_latitude_2                         =  true;
            }
            else if (stricmp(token, "Longitude 2")    == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position_next.m_longitude =  atof(token);
               got_longitude_2                        =  true;
            }

            if (got_name         && got_arrow_1       && got_arrow_2       &&
                got_latitude_1   && got_latitude_2    && got_longitude_1   && got_longitude_2)
            {
               double   avg_latitude   =  (m_symbolInfo.position.m_latitude         +  m_symbolInfo.position_next.m_latitude)    /  2.0;
               double   avg_longitude  =  (m_symbolInfo.position.m_longitude        +  m_symbolInfo.position_next.m_longitude)   /  2.0;
               double   x_distance     =  (m_symbolInfo.position_next.m_latitude    -  m_symbolInfo.position.m_latitude)   *  NM_PER_DEGREE;
               double   y_distance     =  ((m_symbolInfo.position_next.m_longitude  -  m_symbolInfo.position.m_longitude)  *  NM_PER_DEGREE) *  cos(avg_latitude  *  DEG_TO_RAD);
               double   heading        =  270.0 -  atan2(-y_distance, -x_distance)  *  RAD_TO_DEG;
               heading  =  fmod(heading, 360.0);

               m_symbolInfo.length                          =  0.0f;
               m_symbolInfo.width                           =  3.0f;          // Width of the line segments.
               m_symbolInfo.active_airport_and_runway       =  false;
               m_symbolInfo.heading                         =  0.0;
               m_symbolInfo.type                            =  LINE;
               m_custom_symbolInfoList.AddTail(m_symbolInfo);

               m_symbolLabelInfo.position.m_latitude        =  avg_latitude;
               m_symbolLabelInfo.position.m_longitude       =  avg_longitude;
               m_symbolLabelInfo.type                       =  LINE;
               m_symbolLabelInfo.heading                    =  heading;
               m_symbolLabelInfo.active_airport_and_runway  =  false;
               m_symbolLabelInfo.width                      =  relative_font_size;  // Relative size of the font.
               m_custom_symbolLabelInfoList.AddTail(m_symbolLabelInfo);

               if (need_arrow_1)
               {
                  m_symbolInfo.heading                      =  -heading  -  180.0;
                  m_symbolInfo.type                         =  DELTA;
                  m_symbolInfo.symbol.Format("%c", DELTA);
                  m_custom_symbolInfoList.AddTail(m_symbolInfo);
               }

               if (need_arrow_2)
               {
                  m_symbolInfo.heading                      =  -heading;
                  m_symbolInfo.position.m_latitude          =  m_symbolInfo.position_next.m_latitude;
                  m_symbolInfo.position.m_longitude         =  m_symbolInfo.position_next.m_longitude;
                  m_symbolInfo.type                         =  DELTA;
                  m_symbolInfo.symbol.Format("%c", DELTA);
                  m_custom_symbolInfoList.AddTail(m_symbolInfo);
               }
            }
            token =  strtok(NULL, delimeters);                                // Get the next string.
         }
      }
   }


   if (number_of_arcs  >  0)
   {
      CString  CStr_arcs;

      for (ii = 0; ii < number_of_arcs; ii++)
      {
         CStr_arcs.Format("Arc %d", ii + 1);
         count =  GetPrivateProfileSection(CStr_arcs, buffer,  sizeof(buffer),   fullProfilePath);
         for (int zz = 0; zz < count; zz++)
         {
            if (buffer[zz] == 0)
               buffer[zz]  =  ',';
         }

         bool  got_radius        =  false,   got_latitude      =  false,   got_longitude  =  false;
         bool  got_start_heading =  false,   got_end_heading   =  false;

         token =  strtok(buffer, delimeters);                                 // Get the first string.
         while (token)
         {
            if (stricmp(token, "Start Heading")       == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.heading                   =  atof(token);
               got_start_heading                      =  true;
            }
            else if (stricmp(token, "End Heading")    == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.width                     =  atof(token);
               got_end_heading                        =  true;
            }
            else if (stricmp(token, "Radius")         == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.length                    =  atof(token);
               got_radius                             =  true;
            }
            else if (stricmp(token, "Latitude")       == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_latitude       =  atof(token);
               got_latitude                           =  true;
            }
            else if (stricmp(token, "Longitude")      == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolInfo.position.m_longitude      =  atof(token);
               got_longitude                          =  true;
            }

            if (got_start_heading   && got_end_heading   && got_radius  && got_latitude   && got_longitude)
            {
               m_symbolInfo.type                      =  CIRCLE;
               m_symbolInfo.symbol.Format("%c", CIRCLE);
               m_custom_symbolInfoList.AddTail(m_symbolInfo);
            }
            token =  strtok(NULL, delimeters);                                // Get the next string.
         }
      }
   }


   if (number_of_practice_areas  >  0)
   {
      CString        practice_area;
      unsigned long  number_of_points  =  0;

      for (ii = 0; ii < number_of_practice_areas; ii++)
      {
         practice_area.Format("Practice Area %d", ii + 1);

         count =  GetPrivateProfileSection(practice_area,   buffer,  sizeof(buffer), fullProfilePath);
         for (int zz = 0; zz < count; zz++)
         {
            if (buffer[zz] == 0)
               buffer[zz]  =  ',';
         }

         bool  got_name    =  false,   got_name_latitude =  false,   got_name_longitude   =  false,   got_number_of_points =  false;
         bool  error_found =  false;

         token =  strtok(buffer, delimeters);                                 // Get the first string.
         while (token   && !error_found)
         {
            if (stricmp(token, "Name") == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               if (stricmp(token, "none")             != 0)
                  m_symbolLabelInfo.symbol_label.Format("%s", token);
               else
                  m_symbolLabelInfo.symbol_label.Empty();
               got_name                                  =  true;
            }
            else if (stricmp(token, "Name Latitude")     == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolLabelInfo.position.m_latitude     =  atof(token);
               got_name_latitude                         =  true;
            }
            else if (stricmp(token, "Name Longitude")    == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               m_symbolLabelInfo.position.m_longitude    =  atof(token);
               got_name_longitude                        =  true;
            }
            else if (stricmp(token, "Number of Points")  == 0)
            {
               token =  strtok(NULL, delimeters);                             // Get the next string.
               number_of_points                          =  atoi(token);
               got_number_of_points                      =  true;
            }

            if (got_name   && got_name_latitude && got_name_longitude   && got_number_of_points)
            {
               double   *latitude   =  new   double[number_of_points];
               double   *longitude  =  new   double[number_of_points];
               char     latitude_string[16];
               char     longitude_string[16];
               memset(latitude,  0, number_of_points   *  sizeof(double));
               memset(longitude, 0, number_of_points   *  sizeof(double));

               token =  strtok(NULL, delimeters);                             // Get the next string.
               for (jj = 0; jj < number_of_points && token; jj++)
               {
                  sprintf(latitude_string,   "Latitude %d",  jj + 1);
                  if (stricmp(token, latitude_string)    == 0)
                  {
                     token =  strtok(NULL, delimeters);                       // Get the next string.
                     latitude[jj]   =  atof(token);
                  }
                  else
                  {
                     CString  message;
                     message.Format("%s not found in [%s].", latitude_string, practice_area);
                     COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
                     COpenGLMap::m_stlStrCaption   =  "Invalid Custom Areamap Data";
                     AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
                     number_of_points  =  jj;      // Change number_of_points to be number of valid points
                     error_found       =  true;
                     break;
                  }
                  token =  strtok(NULL, delimeters);                          // Get the next string.

                  sprintf(longitude_string,  "Longitude %d", jj + 1);
                  if (stricmp(token, longitude_string)   == 0)
                  {
                     token =  strtok(NULL, delimeters);                       // Get the next string.
                     longitude[jj]  =  atof(token);
                  }
                  else
                  {
                     CString  message;
                     message.Format("%s not found in [%s].", longitude_string, practice_area);
                     COpenGLMap::m_stlStrError     =  (LPCTSTR)message;
                     COpenGLMap::m_stlStrCaption   =  "Invalid Custom Areamap Data";
                     AfxBeginThread(COpenGLMap::MsgThread, (LPVOID)0);
                     number_of_points  =  jj;      // Change number_of_points to be number of valid points
                     error_found       =  true;
                     break;
                  }
                  token =  strtok(NULL, delimeters);                          // Get the next string.

                  if (jj   == number_of_points - 1)                           // About to exit loop so
                  {
                     while (token)                                            // Get the next string but
                        token =  strtok(NULL, delimeters);                    // just discard it.
                  }
               }

               m_symbolInfo.heading                         =  0.0;
               m_symbolInfo.length                          =  0.0f;
               m_symbolInfo.width                           =  8.0f;          // Width of the line segments.
               m_symbolInfo.active_airport_and_runway       =  false;
               m_symbolInfo.type                            =  PRACTICE_AREA;

               for (kk = 0; kk < number_of_points - 1; kk++)
               {
                  if ((latitude[kk]    == 0.0)  || (longitude[kk]    == 0.0))
                     break;
                  if ((latitude[kk+1]  == 0.0)  || (longitude[kk+1]  == 0.0))
                     break;

                  // Symbol information.
                  m_symbolInfo.position.m_latitude       =  latitude[kk];  
                  m_symbolInfo.position.m_longitude      =  longitude[kk];
                  m_symbolInfo.position_next.m_latitude  =  latitude[kk    +  1];
                  m_symbolInfo.position_next.m_longitude =  longitude[kk   +  1];
                  m_custom_symbolInfoList.AddTail(m_symbolInfo);
               }

               delete   [] latitude;
               delete   [] longitude;
            }

            token =  strtok(NULL, delimeters);                                // Get the next string.
         }  // while tokenizing the buffer

         m_symbolLabelInfo.type                       =  PRACTICE_AREA;
         m_symbolLabelInfo.heading                    =  0.0;
         m_symbolLabelInfo.active_airport_and_runway  =  false;
         m_symbolLabelInfo.width                      =  relative_font_size;  // Relative size of the font.
         m_custom_symbolLabelInfoList.AddTail(m_symbolLabelInfo);
      }  // for loop through all the practice areas
   }  // if there are any practice areas

}

void  CAreaMap::LoadTacanChannelData(void)
{
   CString  fullProfilePath("TacanChannels.csv");

   // Open the registry.
   HKEY  hKey;
   RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Software\\FlightSafety\\FSISuite\\Radio Database\\"), 0, KEY_READ, &hKey);
   if (hKey != NULL)
   {
      CString  strValue;
      DWORD    dwType,  dwCount;

      LONG  lResult  =  RegQueryValueEx(hKey, _T("TacanChannelDataFile"), NULL, &dwType, NULL, &dwCount);
      if (lResult == ERROR_SUCCESS)
      {
         ASSERT(dwType  == REG_SZ);
         lResult  =  RegQueryValueEx(hKey, _T("TacanChannelDataFile"), NULL, &dwType, (LPBYTE)strValue.GetBuffer(dwCount/sizeof(TCHAR)), &dwCount);
         strValue.ReleaseBuffer();
      }
      fullProfilePath   =  strValue;

      RegCloseKey(hKey);
   }

   FILE     *stream;
   char     data[100];

   if ((stream =  fopen(fullProfilePath,  "r")) != NULL)
   {
      char  *pdata   =  fgets(data, 100, stream);

      while (pdata)
      {
         CString  line(data);
         int   first_comma =  line.Find(",");
         int   new_line    =  line.Find("\n");
         CString  channel(line.Left(first_comma));
         CString  frequency(line.Mid(first_comma + 1), new_line - first_comma - 1);

         m_TacanChannel.SetAt(frequency, channel);

         pdata =  fgets(data, 100, stream);
      }

      fclose(stream);
   }
}


BOOL CAreaMap::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
   // Get the latitude and longitude of the current mouse position
   double   lat_sweep_angle(120.0   /  m_scale_cv);
   double   before_lat(-(lat_sweep_angle  /  m_height *  m_mouse_y_position   -  lat_sweep_angle   /  2.0   -  m_area_map_latitude_center));

   double   lon_sweep_angle((120.0  /  m_scale_cv  /  cos(before_lat *  DEG_TO_RAD) /  m_aspect_ratio));
   double   before_lon(lon_sweep_angle /  m_width  *  m_mouse_x_position   -  lon_sweep_angle   /  2.0   -  m_area_map_longitude_center);

   // Change the scale
   double   SelectedScale(7200.0 /  m_scale_cv);
   if (zDelta < 0)                  // zoom in 
      SelectedScale -= 1.0;
   else if (zDelta > 0)             // zoom out
      SelectedScale += 1.0;

   // Limit scaling from 1 to 100 NM.
   if (SelectedScale >  100.0)
      SelectedScale =   100.0;
   if (SelectedScale <  1.0)
      SelectedScale =   1.0;

   {
      CVariant var;
      CCommsShared::GetLocalValue(var, _FSI_STL::string("area_map_scale"));
      float old_type =  var.Type();
      var.Value(SelectedScale);
      var.ChangeType(old_type);
      CCommsShared::SetLocalValue(var, _FSI_STL::string("area_map_scale"), 0);
   }

   m_scale_cv  =  (7200.0  /  SelectedScale);

   // Determine latitude and longitude of the same mouse position - but at the new scale
   lat_sweep_angle   =  120.0 /  m_scale_cv;
   double   after_lat(-(lat_sweep_angle   /  m_height *  m_mouse_y_position   -  lat_sweep_angle   /  2.0   -  m_area_map_latitude_center));
   lon_sweep_angle   =  ((120.0  /  m_scale_cv  /  cos(after_lat  *  DEG_TO_RAD) /  m_aspect_ratio));
   double   after_lon(lon_sweep_angle  /  m_width  *  m_mouse_x_position   -  lon_sweep_angle   /  2.0   -  m_area_map_longitude_center);

   // Calculate how much the latitude and longitude changed and apply those offsets to the map center
   // so that the map symbology under the mouse appears to stay at the same location.
   double   lat_diff(before_lat  -  after_lat);
   double   lon_diff(before_lon  -  after_lon);
   m_area_map_latitude_center_cv    += lat_diff;
   m_area_map_longitude_center_cv   -= lon_diff;

	return 0;
}


void CAreaMap::OnLButtonDown(UINT nFlags, CPoint pt) 
{
   if (m_bEditing == true)
   {
   }
   else
   {
      NMHDR nmhdr;
      nmhdr.idFrom = GetDlgCtrlID();
      nmhdr.hwndFrom = GetSafeHwnd();
      nmhdr.code = WM_LBUTTONDOWN;
      GetParent()->SendMessage(WM_NOTIFY,GetDlgCtrlID(),(long)&nmhdr);

      m_bLeftMouseDown  =  true;
   }
	
	COpenGLWidget::OnLButtonDown(nFlags, pt);
}


LRESULT CAreaMap::OnMouseLeave(WPARAM,LPARAM)
{
   m_bLeftMouseDown     =  false;
   return 0;
}
